import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
/*
 * Data.java
 *
 * Created on June 18, 2007, 8:53 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

/**
 * 
 * @author Siddnat
 */
public class Data 
{
    //Derived from the LTL condition
    // G(((activeReaders > 0) -> !(activeWriters > 0)) 
    //  & ((activeWriters > 0) -> !(activeReaders >0))
    //  & (activeWriters <= 1) )
				
    //@ invariant !(activeReaders > 0 && activeWriters > 0);
    //@ invariant !(activeWriters > 1);
    
    public static final int MIN = 0;
    public static final int MAX = 24000;
    /**
     * This is the shared data. It's a simple integer between
     * 0 and 24000. Any attempt to write something larger than
     * 24000 is modded with 24000, and any negative value is made
     * positive
     */
    private int data;
    
    /**
     * A simple mutex lock
     */
    private Lock lock;

    /**
     * A condition variable to indicate when it is okay to read
     */
    private Condition okToRead;
    
    /**
     * A condition variable to indicate when it is okay to write
     */
    private Condition okToWrite;
    
    int activeReaders;
    int activeWriters;
    int waitingReaders;
    int waitingWriters;
    
    /**
     * Indicates whether the data has been read after being written for the
     * first time or not
     */
    boolean read;
    
    /** Creates a new instance of Data */
    public Data() 
    {
        //Set negative to ensure that the first time around, data is written
        //into the buffer before anyone can read from it.
       data = -1;
       lock = new ReentrantLock ();
       this.okToRead = lock.newCondition();
       this.okToWrite = lock.newCondition();
       
       //Set it to true in the beginning so that the
       //the writers have permission to write the data
       //in the very beginning
       read = true;
       
       activeReaders = 0;
       activeWriters = 0;
       waitingReaders = 0;
       waitingWriters = 0;
    }
    
    //Post Conditions derived from LTL conditions
    // G ( ((waitingReaders > 0) & (waitingWriters > 0) & (read = true)) -> X( (activeWriters>0) )
    // G ( (activeWriters > 0) -> X (activeReaders > 0) )
    /*@ behavior
      @     assignable waitingReaders, activeReaders
      @     ensures activeReaders>=1 && ((\old(activeWriters) + \old(waitingWriters) == 0 || !(\old(read)))&&(data>=0)) ==> \old(activeReaders) + 1 == activeReaders
     @*/
    public void startRead ()
    {
        try
        {
            lock.lock();
            
            //Case where something is being written to the buffer/data
            //We'll also have to wait if the buffer/data is empty
            while ((activeWriters + waitingWriters > 0 && read) || data < 0 )
            {
                //System.out.println ("Read waiting");
                ++waitingReaders;
                okToRead.awaitUninterruptibly();
                --waitingReaders;
            }
            
            ++activeReaders;
        }
        finally
        {
            lock.unlock();
        }
    }
    
    public int read() 
    {
        startRead ();
        int k = data;
        endRead ();
        return k;
    }

    //Post Conditions derived from LTL conditions
    // G ( (activeWriters > 0) -> X (activeReaders > 0) )
    /*@ behavior
      @     assignable activeReaders, read
      @     ensures \old(activeReaders) + 1 == activeReaders && read == true
      @*/
    public void endRead ()
    {
        try
        {
            lock.lock();
            
            --activeReaders;
            
            //Now to wake up all the waiting writers
            if (activeReaders == 0 && waitingWriters>0)
                okToWrite.signalAll();
            
            //To clear out the buffer since the message has been read
            read = true;
        }
        finally
        {
            lock.unlock();
        }
    }
    
    public void write(int w) 
    {
        startWrite ();
        data = (Math.abs(w))%MAX;
        endWrite ();
    }
    
    //Post Conditions derived from LTL conditions
    // G ( ((waitingReaders > 0) & (waitingWriters > 0) & (read = true)) -> X( (activeWriters>0) )
    // G ( (activeWriters > 0) -> X (activeReaders > 0) )
    // G (  ((activeReaders > 0) -> !(activeWriters > 0)) 
    //      & ((activeWriters > 0) -> !(activeReaders >0))
    //      & (activeWriters <= 1) )

    /*@ behavior
      @     assignable waitingWriters, activeWriters
      @     ensures activeWriters==1 && (\old(activeWriters)+\old(activeReaders)==0 && read) ==> \old(activeWriters)+1 == activeWriters
      @*/
    public void startWrite ()
    {
        try
        {
            lock.lock();
            
            //Now, we can only write if no one is already writing or reading
            //Also, the buffer/data must be empty for the writer to write to it
            while (activeWriters + activeReaders > 0 || !read)
            {
                //System.out.println ("write waiting");
                ++waitingWriters;
                okToWrite.awaitUninterruptibly();
                --waitingWriters;
            }
            
            ++activeWriters;
        }
        finally
        {
            lock.unlock();
        }
    }
    

    //Post Conditions derived from LTL conditions
    // G ( (activeWriters > 0) -> X (activeReaders > 0) )
    /*@ behavior
      @     assignable activeWriters, read
      @     ensures activeWriters==0 && !read
      @*/
    public void endWrite ()
    {
        try
        {
            lock.lock();
            
            --activeWriters;
            if (waitingWriters > 0)
                okToWrite.signalAll();
            if (waitingReaders > 0)
                okToRead.signalAll();
            read = false;
        }
        finally
        {
            lock.unlock();
        }
    }
}
