#include <stdlib.h>
#include <stdio.h>
#include <sys/procfs.h>
#include <sys/syscall.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "ExecHook.h"
#include "fetchstr.h"

ExecHook::ExecHook(FILE *out_, int what):Hook(out_, what)
{
}


/*
 *------------------------------------------------------------------
 *
 * Enter --
 *
 *          Gets called on every SYS_exec and SYS_execve.
 *          Need to tell log data structure the name of this
 *          new program. 
 *
 *          If the program is suid or the execuable file is not 
 *          readable by me, then we need to detach
 *          from it since otherwise the job will be killed
 *          for security reasons (cannot trace a suid job).
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
void
ExecHook::Enter(const struct prstatus *p, int procfd)
{
  char *childName;

  Hook::Enter(p, procfd);

  childName = fetchstr(procfd, (char *)p->pr_sysarg[0]);
  printArgs(p, procfd, childName);
  if(!allowedToTrace(childName)){
    detach(p, procfd, childName);
  }
  free(childName);
}

/*
 *------------------------------------------------------------------
 *
 * allowedToTrace --
 *
 *          Return false if the specified file has its suid or sgid
 *          flag set. Or if it is not readable by me. True otherwise.
 * 
 *          If any of these conditions hold true, we are not allowed
 *          to trace this file via /proc.
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
int
ExecHook::allowedToTrace(char *childName)
{
  struct stat statBuf;
  char buf[80];
  /*
   * Note: stat automatically follows symlinks, so don't
   * need to worry about that.
   */
  if(stat(childName, &statBuf)){
    return 1;  /* probably a bad path or something. Not our problem.  */
  }

  /*
   * Not allowed to trace if executable is setuid or setgid
   */
  if((statBuf.st_mode & S_ISUID) || (statBuf.st_mode & S_ISGID)){
    fprintf(out, "\nNOTE: Setuid %s\n",childName);
    return 0;
  }
  /*
   * Not allowed to trace if file is unreadable by me.
   */
  if(!(statBuf.st_mode & S_IROTH)  
     && (statBuf.st_mode & S_IRGRP || statBuf.st_gid != getegid())
     && (statBuf.st_mode & S_IRUSR || statBuf.st_uid != geteuid())){
    return 0 ;
  }
  return 1;
}

/*
 *------------------------------------------------------------------
 *
 * detach --
 *    Report that we will not be allowed to trace the specified process.
 *
 *------------------------------------------------------------------
 */
void
ExecHook::detach(const struct prstatus *p, int procfd, char *childName)
{
  fprintf(out, "NOTE: Forced detach from %s\n", childName?childName:"<unknown chldName>");
  return;
}


/*
 *------------------------------------------------------------------
 *
 * printArgs --
 *
 *          description.
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
void
ExecHook::printArgs(const struct prstatus *p, int procfd, char *childName)
{
  char **argv;
  // char **envp;
  int ii;
  char buf[256];
  argv = fetchArrayOfStrings(procfd, (const char **)p->pr_sysarg[1]);
  //  envp = fetchArrayOfStrings(procfd, (const char **)p->pr_sysarg[2]);
  
  fprintf(out, "NOTE: SYS_exec by %d  %s \t",p->pr_pid,childName);
  for(ii = 0; argv[ii] != NULL; ii++){
    fprintf(out, " %s\t", argv[ii]);
    free(argv[ii]);
  }
  fprintf(out, "\n");
  free(argv);
  return;
}




