/*
 * Copyright (c) 1993 The Regents of the University of California.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/procfs.h>

#include "fetchstr.h"
/*
 * fetch bytes from the app's address space, up to a '\0'
 * 'str' is a pointer valid in the *child's* address space
 * acts like a strdup(): don't forget to free() the pointer it returns!
 */
char  *
fetchstr(const int procfd, const char *str)
{
#define BLOCKSIZE   1024 /* important only for performance */
    char    *buf = (char *)malloc(BLOCKSIZE);
    int	    validbytes = 0, bufsize = BLOCKSIZE, rv;

    lseek(procfd, (int) str, SEEK_SET);
    for (;;) {
	if (validbytes == bufsize)
	    buf = (char *)realloc(buf, bufsize += BLOCKSIZE);
    	rv = read(procfd, buf+validbytes, bufsize-validbytes);
	if (rv <= 0) {
	    free(buf);
	    return(0);
	}
	if (memchr(buf+validbytes, '\0', rv))
	    return(buf);
	validbytes += rv;
    }
}



/*
 * Fetch an array of strings (e.g. char **argv or char **envp)
 * from the traced address space into our address space.
 * First fetch the array of pointers, up to a NULL pointer.
 * Then fetchstr each of the entries in the array.
 * 'strArray' is a pointer in the application's address space
 * acts like a strdup(): don't forget to free() the pointer it returns!
 */
char **
fetchArrayOfStrings(const int procfd, const char** strArray)
{
  char **array = (char **)malloc(BLOCKSIZE * sizeof(char *));
  int nStrings = 0;
  int nBuf = BLOCKSIZE;
  int rv;

  lseek(procfd, (int)strArray, SEEK_SET);
  while(1){
    rv = read(procfd, array + nStrings, BLOCKSIZE * sizeof(char *));
    if(rv <= 0){
      free(array);
      return 0;
    }
    nBuf = nStrings + rv;
    for(; nStrings < nBuf; nStrings++){
      if(array[nStrings] != NULL){
	array[nStrings] = fetchstr(procfd, array[nStrings]);
      }
      else{
	return array;
      }
    }
    if(nStrings == nBuf){
      array = (char **)realloc(array, nBuf + BLOCKSIZE);
      nBuf += BLOCKSIZE;
    }
  }    
}

