/*
 * Copyright (c) 1993 The Regents of the University of California.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "module.h"
#include "version.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <sys/stropts.h>
#include <sys/tiuser.h>
#include <sys/sockmod.h>
#include <sys/filio.h>
#include <sys/termios.h>


typedef struct s_pid_node
{
    pid_t pid;
    struct s_pid_node *next;
} pid_node;

typedef struct
{
    pid_node *head;
    pid_node **tail;
} state_t;

int	verbose = 0;

action	kill_hook(const struct prstatus *p, int procfd, void *st)
{
    state_t *state = st;
    pid_t killpid = p->pr_sysarg[0];
    pid_node *ptr;



    /* Is it killing itself? */
    if (p->pr_pid == killpid) return(ALLOW);

    /*
     * MDD: Is it killing its own group? (Apache needs this)
     * MDD: From kill(2)
     * MDD: If pid is negative but not (pid_t)-1, sig will  be  sent  to
     * MDD: all  processes  whose process group ID is equal to the abso-
     * MDD: lute value of pid and for which the process  has  permission
     * MDD: to send a signal.
     * MDD: If pid is 0, sig will be sent  to  all  processes  excluding
     * MDD: special  processes  (see intro(2)) whose process group ID is
     * MDD: equal to the process group ID of the sender.
     */
    if ((p->pr_pid == -killpid) || killpid == 0){
      return (ALLOW);
    }

    /* Check if killpid is in the list */
    for(ptr=state->head; ptr; ptr=ptr->next)
    {
	if (ptr->pid == killpid) return(ALLOW);
    }
    return(DENY);
}

action	wait_hook(const struct prstatus *p, int procfd, void *st)
{
    state_t *state = st;
    pid_t oldpid = p->pr_reg[R_O0];

    if (oldpid > 0)
    {
	/* Remove oldpid from the list */
	pid_node **ptr = &state->head;
	while(*ptr)
	{
	    if ((*ptr)->pid == oldpid)
	    {
		/* Remove this node */
		pid_node *savepid = *ptr;
		*ptr = (*ptr)->next;
		free(savepid);
		if (*ptr == NULL)
		{
		    state->tail = ptr;
		}
	    }
	    else
	    {
		ptr = &((*ptr)->next);
	    }
	}
    }

    return(NO_COMMENT);
}

action	fork_hook(const struct prstatus *p, int procfd, void *st)
{
    state_t *state = st;
    pid_t newpid = p->pr_reg[R_O0];

    if (newpid > 0)
    {
	/* Add newpid to the list */
	pid_node *newnode = (pid_node *)malloc(sizeof(pid_node));
	if (newnode == NULL)
	{
	    fprintf(stderr, "fork_hook: Out of memory!\n");
	    exit(1);
	}
	newnode->pid = newpid;
	newnode->next = NULL;
	*(state->tail) = newnode;
	state->tail = &(newnode->next);
    }

    return(NO_COMMENT);
}

/* syscalls I don't understand; thus deny & wait & see if anything breaks */
#define UNKNOWN DENY

/* what to do when you see a dup(): several syscalls let you do a dup */
#define DUP_ACTION ALLOW

action    fcntl_hook(const struct prstatus *p, int procfd, void *unused)
{
    switch(p->pr_sysarg[1]) {
	case F_SETFL: if ((mode_t) p->pr_sysarg[2] & (O_RDONLY|O_RDWR|O_WRONLY))
			return(DENY); /* bad mean nasty */
		    return(ALLOW);
	case F_DUPFD: return(DUP_ACTION);
	case F_GETFL: return(ALLOW);

	/* set/get fd-close-on-exec flag is fine */
	/* set/get advisory locks are ok */

	/*
	 * BUG: should explicitly enumerate the allowed actions,
	 * and make the default be DENY.
	 */
	default: return(ALLOW);
    }
}

/*
 * This is a pain in the ass to get right; ioctls are like yet
 * another set of very fine-grained low-level miscellaneous system calls.
 * This is tricky.  I'm worried about it.  Oh well.
 * 
 * The connect.c module also enables ioctls which are relevant to
 * net connections and the STREAMS TLI TCP/UDP interfaces.
 */
action    ioctl_hook(const struct prstatus *p, int procfd, void *unused)
{
    unsigned int which = p->pr_sysarg[1];

    switch(which) {
	case TIOCGPGRP: case TIOCGSID:
	case FIONBIO: case FIONREAD:
	case I_SETCLTIME: case I_SWROPT:
	    return(ALLOW);
	case TIOCSTI:
	    return(DENY);
    }
    switch(which & 0xFF00) {
	case _TIOC: return(ALLOW);
    }
    return(NO_COMMENT);
}

action	unnamed_hook(const struct prstatus *p, int procfd, void *unused)
{
    if (verbose)
	fprintf(stderr, "DEBUG: unnamed syscall %d. Denied.\n", p->pr_what);
    return(DENY);
}

void *	init(const char *conf_line)
{
    state_t *state = malloc(sizeof(state_t));
    if (state == NULL) return (INIT_FAIL);

    state->head = NULL;
    state->tail = &(state->head);

    if (conf_line && conf_line[0] == 'v')
	verbose = 1;

    return state;
}

#define MDD 
#ifdef MDD
#define MDD_ALLOW_LWP ALLOW /* apache instrumentation needs threads */
#else
#define MDD_ALLOW_LWP DENY
#endif

const syscall_entry     entries[] = {

/*****PERMISSIONS group********/

    {SYS_chmod, DENY, 0},
    {SYS_chown, DENY, 0},
    {SYS_getuid, ALLOW, 0},
    {SYS_getgid, ALLOW, 0},
    {SYS_umask, DENY, 0}, 
    {SYS_getgroups, ALLOW, 0},
    {SYS_fchmod, DENY, 0},
    {SYS_fchown, DENY, 0},
    {SYS_lchown, DENY, 0},
    {SYS_setegid, DENY, 0},
    {SYS_seteuid, DENY, 0},
    {SYS_vhangup, DENY, 0},

/*****SIGNALS group************/

    {SYS_alarm, ALLOW, 0}, 
    {SYS_pause, ALLOW, 0}, 
    {SYS_signal, ALLOW, 0}, 
    {SYS_sigprocmask, ALLOW, 0}, 
    {SYS_sigsuspend, ALLOW, 0}, 
    {SYS_sigaltstack, ALLOW, 0}, 
    {SYS_sigaction, ALLOW, 0}, 
    {SYS_sigpending, ALLOW, 0}, 
    {SYS_sigsendsys, ALLOW, 0}, 
    {SYS_sigqueue, ALLOW, 0}, 

/*****MEMORY group*************/

    {SYS_brk, ALLOW, 0}, 
    {SYS_mincore, ALLOW, 0}, 
    {SYS_mmap, ALLOW, 0}, 
    {SYS_mprotect, ALLOW, 0}, 
    {SYS_munmap, ALLOW, 0}, 
    {SYS_memcntl, ALLOW, 0},
    
/*****SHARED MEMORY group******/
    
    {SYS_shmsys, DENY, 0},
    {SYS_semsys, DENY, 0},     
    
    
/*****THREADS group************/    
    
    {SYS_context, ALLOW, 0},		/* allow LWP for netscape */
    {SYS_priocntlsys, DENY, 0},     
#ifdef MDD_ALLOW_LWP
    {SYS_fork1, EXIT_FUNC, fork_hook}, 
    {SYS_fork1, ALLOW, 0},
#else
    {SYS_fork1, DENY, 0},     
#endif
    {SYS_sigtimedwait, MDD_ALLOW_LWP, 0},
    {SYS_yield, DENY, 0},     
    {SYS_lwp_sema_wait, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_sema_post, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_create, MDD_ALLOW_LWP, 0},
    {SYS_lwp_exit, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_suspend, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_continue, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_kill, DENY, 0},     
    {SYS_lwp_self, MDD_ALLOW_LWP, 0},
    {SYS_lwp_setprivate, DENY, 0},     
    {SYS_lwp_getprivate, DENY, 0},     
    {SYS_lwp_wait, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_mutex_unlock, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_mutex_lock, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_cond_wait, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_cond_signal, MDD_ALLOW_LWP, 0},     
    {SYS_lwp_cond_broadcast, MDD_ALLOW_LWP, 0},     
    {SYS_processor_bind, DENY, 0},     
    {SYS_nanosleep, DENY, 0},     
    
/*****INFO*TIME group***********/    

    {SYS_time, ALLOW, 0}, 
    {SYS_times, ALLOW, 0}, 
    {SYS_profil, ALLOW, 0}, 
    {SYS_uname, ALLOW, 0}, 
    {SYS_gettimeofday, ALLOW, 0}, 
    {SYS_getitimer, ALLOW, 0}, 
    {SYS_setitimer, ALLOW, 0}, 
    {SYS_processor_info, ALLOW, 0}, 
    {SYS_clock_gettime, ALLOW, 0}, 
    {SYS_clock_getres, ALLOW, 0}, 
    {SYS_timer_create, ALLOW, 0}, 
    {SYS_timer_delete, ALLOW, 0}, 
    {SYS_timer_settime, ALLOW, 0}, 
    {SYS_timer_gettime, ALLOW, 0}, 
    {SYS_timer_getoverrun, ALLOW, 0}, 
    
/*****UNABLE TO GROUP group*******/
    
    {SYS_syscall, DENY, 0}, 
    {SYS_stty, DENY, 0}, 
    {SYS_gtty, DENY, 0}, 
    {SYS_xenix, DENY, 0}, 
    {SYS_msgsys, DENY, 0}, 
    {SYS_syssun, DENY, 0}, 
    {SYS_utssys, DENY, 0}, 
    {SYS_evsys, DENY, 0}, 
    {SYS_evtrapret, DENY, 0}, 
    {SYS_nfssys, DENY, 0}, 
    {SYS_waitsys, DENY, 0}, 
    {SYS_hrtsys, DENY, 0}, 
    {SYS_acancel, DENY, 0}, 
    {SYS_async, DENY, 0}, 
    {SYS_xstat, DENY, 0}, 
    {SYS_lxstat, DENY, 0}, 
    {SYS_fxstat, DENY, 0}, 
    {SYS_xmknod, DENY, 0}, 
    {SYS_clocal, DENY, 0}, 
    {SYS_sysconfig, ALLOW, 0}, 
    {SYS_systeminfo, DENY, 0}, 
    {SYS_vtrace, DENY, 0}, 
    {SYS_lwp_info, DENY, 0}, 
    {SYS_modctl, DENY, 0}, 
    {SYS_inst_sync, DENY, 0}, 
    {SYS_acl, DENY, 0},  
    {SYS_auditsys, DENY, 0}, 
    {SYS_acl, DENY, 0},
    {SYS_auditsys, DENY, 0}, 
    {SYS_facl, DENY, 0},

/*****PRIVILEGED group*******/

    {SYS_mount, DENY, 0},
    {SYS_umount, DENY, 0},
    {SYS_setuid, DENY, 0},
    {SYS_stime, DENY, 0},
    {SYS_plock, DENY, 0},
    {SYS_setgid, DENY, 0},
    {SYS_acct, DENY, 0},
    {SYS_uadmin, DENY, 0},
    {SYS_chroot, DENY, 0},
    {SYS_setgroups, DENY, 0},
    {SYS_adjtime, DENY, 0},
    {SYS_fchroot, DENY, 0},
    {SYS_p_online, DENY, 0},
    {SYS_clock_settime, DENY, 0},

/*****PROCESSES group*******/

    {SYS_exit, ALLOW, 0},
    {SYS_fork, EXIT_FUNC, fork_hook},
    {SYS_fork, ALLOW, 0},
    {SYS_kill, FUNC, kill_hook},
    {SYS_wait, ALLOW, 0},
    {SYS_waitsys, ALLOW, 0},
    {SYS_getpid, ALLOW, 0},
    {SYS_ptrace, DENY, 0},
    {SYS_nice, ALLOW, 0},
    {SYS_pgrpsys, ALLOW, 0},
    {SYS_ulimit, ALLOW, 0},
    {SYS_vfork, EXIT_FUNC, fork_hook},
    {SYS_vfork, ALLOW, 0},
    {SYS_setrlimit, DENY, 0},
    {SYS_getrlimit, ALLOW, 0},

/*****FD group*******/

    {SYS_read, ALLOW, 0},
    {SYS_write, ALLOW, 0},
    {SYS_readv, ALLOW, 0},
    {SYS_writev, ALLOW, 0},
    {SYS_pread, ALLOW, 0},
    {SYS_pwrite, ALLOW, 0},
    {SYS_lseek, ALLOW, 0},
    {SYS_llseek, ALLOW, 0},
    {SYS_fstat, ALLOW, 0},
    {SYS_fstatfs, UNKNOWN, 0},
    {SYS_dup, DUP_ACTION, 0},
    {SYS_pipe, ALLOW, 0},
    {SYS_fdsync, ALLOW, 0},	/* for netscape's bookmarks file */
    {SYS_fcntl, FUNC, fcntl_hook},
    {SYS_getdents, ALLOW, 0},
    {SYS_getmsg, ALLOW, 0},
    {SYS_getpmsg, ALLOW, 0},
    {SYS_putmsg, DENY, 0}, /* must be handled by tcpconnect */
    {SYS_putpmsg, DENY, 0}, /* must be handled by tcpconnect */
    {SYS_poll, ALLOW, 0},
    {SYS_fstatvfs, UNKNOWN, 0},
    {SYS_fpathconf, UNKNOWN, 0},
    {SYS_fchdir, DENY, 0},
    {SYS_ioctl, FUNC, ioctl_hook},


/*****DOORS *******/
    {SYS_door, ALLOW, 0},

/*****UNNAMED group*******/

    {56, FUNC, unnamed_hook}, 
    {64, FUNC, unnamed_hook}, 
    {65, FUNC, unnamed_hook}, 
    {66, FUNC, unnamed_hook}, 
    {67, FUNC, unnamed_hook}, 
    {68, FUNC, unnamed_hook}, 
    {69, FUNC, unnamed_hook}, 
    {70, FUNC, unnamed_hook}, 
    {71, FUNC, unnamed_hook}, 
    {72, FUNC, unnamed_hook}, 
    {73, FUNC, unnamed_hook}, 
    {74, FUNC, unnamed_hook}, 
    {75, FUNC, unnamed_hook}, 
    {76, FUNC, unnamed_hook}, 
    {77, FUNC, unnamed_hook}, 
    {78, FUNC, unnamed_hook}, 
    {82, FUNC, unnamed_hook}, 
    {83, FUNC, unnamed_hook}, 
    {105, FUNC, unnamed_hook}, 
    {140, FUNC, unnamed_hook}, 
    {149, FUNC, unnamed_hook}, 
    {150, FUNC, unnamed_hook}, 
    {151, FUNC, unnamed_hook}, 
    {177, FUNC, unnamed_hook}, 
    {178, FUNC, unnamed_hook}, 
    {179, FUNC, unnamed_hook}, 
    {180, FUNC, unnamed_hook}, 
    {181, FUNC, unnamed_hook}, 
    {182, FUNC, unnamed_hook}, 
    {183, FUNC, unnamed_hook}, 
    {184, FUNC, unnamed_hook}, 
    {202, FUNC, unnamed_hook}, 
    {203, FUNC, unnamed_hook}, 
    {204, FUNC, unnamed_hook}, 
    {205, FUNC, unnamed_hook}, 
    {206, FUNC, unnamed_hook}, 
    {207, FUNC, unnamed_hook}, 
    {208, FUNC, unnamed_hook}, 
    {209, FUNC, unnamed_hook}
    };
const int		nentries = sizeof(entries) / sizeof(syscall_entry);
