/*
 * Copyright (c) 1993 The Regents of the University of California.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

/*
 * module for allowing or disallowing file & directory access.
 * 
 * usage: path allow|deny|super-deny mode-list path1 [path2 [...]]
 * 
 * read docs/path for more usage information.
 * 
 * see comments at the end of this file for some unresolved issues, etc.
 */

#include "module.h"
#include "version.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>


extern int	match(const char *, const char *);
extern char    *fetchstr(const int, const char *);



/* I'm going to be very evil and | the flags together.  Don't mind me. */
typedef enum {NONE=0, READ=1, WRITE=2, EXEC=4, ALL=READ|WRITE|EXEC} ac_mode_t;

typedef struct node_s {
	action	upon_match;
	ac_mode_t mode;
	char	*pattern;
	struct node_s	*next;
} node;

/* I'm very evil: this is both an action and an ac_mode_t.  Don't mind me. */
#define ERR -1

node	gl_head_node = {ERR, ERR, (char *) 0, (node *) 0};
node	*gl_list = &gl_head_node;

typedef enum {ACTIVE, PASSIVE} status_t;

/*
 * 'str' may be read, write, exec, r, w, x
 * or it may be a comma-separated list of some combination of the above
 * (no spaces between commas!)
 * in the case of a list of r, w, x, commas can be ommitted.
 * parse this string, and return the access mode.
 * beware, I call strtok.  deal with it.
 */
ac_mode_t    parse_mode(char *str)
{
    ac_mode_t    rv = 0;
    char	*p;

    if (strspn(str, "rwx") == strlen(str)) {
	for (; *str; str++)
	    switch(*str) {
		case 'r':    rv |= READ; break;
		case 'w':    rv |= WRITE; break;
		case 'x':    rv |= EXEC; break;
	    }
	return(rv);
    } 

    for (p = strtok(str, ","); p; p = strtok((char *) 0, ","))
	if (strcmp(p, "read") == 0)
	    rv |= READ;
	else if (strcmp(p, "write") == 0)
	    rv |= WRITE;
	else if (strcmp(p, "exec") == 0)
	    rv |= EXEC;
	else
	    return(ERR);
    return(rv);
}

action    parse_upon_match(char *str)
{
    if (strcmp(str, "super-allow") == 0)
	return(SUPER_ALLOW);
    else if (strcmp(str, "allow") == 0)
	return(ALLOW);
    else if (strcmp(str, "deny") == 0)
	return(DENY);
    else if (strcmp(str, "super-deny") == 0)
	return(SUPER_DENY);
    return(ERR);
}

/* parse the args, save 'em in the linked list, and return a context */
void	*init(const char *conf_line)
{
    char    *p, *arg = 0, *pum, *pmo;
    char    *error_msg = 0;
    node    *np, *nq;
    action  um;
    ac_mode_t mo;
    status_t s;
    static status_t whichcall = ACTIVE;

#define FAIL(err) {error_msg = err; goto fail;}

    if (!conf_line)
	FAIL("no arguments")
    arg = strdup(conf_line);
    if (!arg)
	FAIL("malloc failed")

    if ((p = strchr(arg, '#')) != 0)
	*p = '\0';

    pum = strtok(arg, " \t");
    pmo = strtok((char *) 0, " \t");

    p = strtok((char *) 0, ""); /* save rest of arg [for strtok() reentrance] */

    /* placed here, not earlier, because they may wanna use strtok() */
    if ((um = parse_upon_match(pum)) == ERR)
	FAIL("bad args: no allow/deny")
    if ((mo = parse_mode(pmo)) == ERR)
	FAIL("bad args: no access mode")

    /* go to end of linked list */
    for (np = gl_list; np->next; np = np->next)
	;
    /* append patterns to end of linked list */
    for (p = strtok(p, " \t"); p; p = strtok((char *) 0, " \t")) {
	nq = (node *) malloc(sizeof(node));
	if (!nq)
	    FAIL("malloc failed");

	nq->upon_match = um;
	nq->mode = mo;
	nq->pattern = strdup(p);
	if (!nq->pattern) {
	    free(nq); FAIL("strdup failed")
	}
	nq->next = np->next;
	np->next = nq;
	np = nq;
    }

    free(arg);
    s = whichcall;
    if (whichcall == ACTIVE)
	whichcall = PASSIVE;
    return((void *) s);

#undef FAIL

fail:
    if (error_msg)
	fprintf(stderr, "path error: %s\n", error_msg);
    if (arg)
	free(arg);
    return((void *) INIT_FAIL);
}

/* parses open() flags, translates them into ac_mode_t */
static ac_mode_t    f2am(mode_t o_flags)
{
    ac_mode_t	m = 0;

    if (o_flags & (O_TRUNC|O_CREAT|O_APPEND))
	m |= WRITE;
    switch(o_flags & (O_RDONLY|O_WRONLY|O_RDWR)) {
	case O_RDONLY:  return(m|READ);
	case O_WRONLY:  return(m|WRITE);
	case O_RDWR:    return(m|READ|WRITE);
	default:	fprintf(stderr, "bad open flag %ld\n", o_flags);
			return(ERR); /* paranoia */
    }
}

/*
 * for ONE mode (e.g. READ, WRITE, or EXEC, but NOT a combination!),
 * runs through the whole linked list and verifies access is allowed.
 */
static action	check_one_access(const char *arg, ac_mode_t m)
{
    node *np;
    action todo = NO_COMMENT;

    for (np = gl_list->next; np; np = np->next)
	if ((m & np->mode) && match(np->pattern, arg))
	    switch(np->upon_match) {
		case NO_COMMENT: break;
		case SUPER_DENY: return(DENY);
		case SUPER_ALLOW: return(ALLOW);
		case DENY: todo = DENY; break;
		case ALLOW: todo = ALLOW; break;
		default:	fprintf(stderr, "path err: burp!\n");
				return(DENY); /* should never get here */
	    }
    return(todo);
}



/*
 * the general access checker.
 * takes a path name (arg0ptr), a requested access type (access_flags),
 * and a /proc file descriptor (procfd).
 * returns the action that should be taken on this type of access request.
 * 
 * for each type of access requested, we independently check
 * whether that access is allowed, and then combine the results in stop().
 * we combine in an &&-like stage, where DENY takes precedence over
 * ALLOW/NO_COMMENT, and NO_COMMENT takes precedence over ALLOW.
 * 
 * Usage:
 *   start(procfd)
 *   request(arg0ptr, m)
 *   request(arg1ptr, n)
 *   ...
 *   result = stop()
 */

/* for use only by start(), request(), stop() */
static int cur_fd = -1;
static int cur_allows, cur_nocoms, cur_denies;

void	start(int procfd)
{
    cur_fd = procfd;
    cur_allows = cur_nocoms = cur_denies = 0;
}

/*
 * canonicalize -- take a file pathname and canonicalize it
 * to ease path matching and catch attempts to subvert the
 * security policy (e.g. "otherwise/../../valid/path").
 * 
 * returns the new pointer to the pathname.  the original pathname
 * must have been allocated with malloc.  might modify the argument
 * in place, or just return a new pointer to a canonicalized pathname.
 * 
 * returns 0 upon failure.  the caller must deny the request upon failure.
 * 
 * the following transformations are applied:
 *	deny ".."
 *	fold "//" to "/"
 *	fold "/./" to "/"
 * 	eliminate leading "./"s
 * 	turn "" into "."
 * 
 * apologies for duplicating file handling code from the kernel.
 * it's more-or-less necessary in a user-level tool like this.
 */
char	*canonicalize(char *path)
{
	char	*p, *q;

	/* "/../", ^"../", "/.."$, ^".."$	-->	deny */
	if (strstr(path, "/../") || strstr(path, "../") == path
			|| strcmp(path, "..") == 0
			|| strstr(path, "/..") == path+strlen(path)-3
	) {
		cur_denies++; return(0);
	}

	/* "//"		-->	"/" */
	while ((p = strstr(path, "//"))) {
		/* strcpy(p, p+1); */
		for (q=p; *(q+1); q++)
			*q = *(q+1);
		*q = '\0';
	}

	/* "/./"	-->	"/" */
	while ((p = strstr(path, "/./"))) {
		/* strcpy(p, p+2); */
		for (q=p; *(q+2); q++)
			*q = *(q+2);
		*q = '\0';
	}

	/* ^"./"	-->	"" */
	while ((path[0] == '.' && path[1] == '/')) {
		/* strcpy(path, path+2); */
		for (q=path; *(q+2); q++)
			*q = *(q+2);
		*q = '\0';
	}

	/* ""		-->	"." */
	if (strcmp(path, "") == 0) {
		path = realloc(path, 2);
		if (path)
			strcpy(path, ".");
	}

	if (!path)
		cur_denies++;

	/* printf("XXX janus path: %s\n", path);  */
	return(path);
}


/*
 * hack.  hack.  hack.  for symlink(), who needs to do the fetchstr()
 * itself, and then munge the result, and *then* call request().  sigh.
 */
void	request_local_buf(char *arg, ac_mode_t m)
{
    ac_mode_t modes[3] = {READ, WRITE, EXEC};
    int i;
 
    /*
     * note optimization (because we're implementing the equivalent of `&&'):
     * if cur_denies > 0, can just immediately return.
     */

    if (cur_fd == -1 || !arg || m == ERR || cur_denies > 0) {
	cur_denies++; return;
    }
    arg = canonicalize(arg);
    if (!arg) {
	cur_denies++; return;
    }
    /*    fprintf(stderr, "MDD: janus:path: open(%s)\n", arg); */
    for (i=0; i<3; i++)
	if (modes[i] & m)
	    switch (check_one_access(arg, modes[i])) {
		case DENY: cur_denies++; return;
		case NO_COMMENT: cur_nocoms++; break;
		case ALLOW: cur_allows++; break;
		default: fprintf(stderr, "path err: the impossible, wasn't!\n");
			cur_denies++; return; /* shouldn't happen! */
	    }

}

void	request(const long arg0ptr, ac_mode_t m)
{
	char	*arg;

	if (cur_fd == -1 || !arg0ptr) {
		cur_denies++; return;
	}
	arg = (char *) fetchstr(cur_fd, (char *) arg0ptr);
	request_local_buf(arg, m);
	free(arg);
}


action stop(void)
{
    action a = NO_COMMENT;

    /* order is important */
    if (cur_denies > 0)
	a = DENY;
    else if (cur_nocoms > 0)
	a = NO_COMMENT;
    else if (cur_allows > 0)
	a = ALLOW;
    cur_fd = -1;
    return(a);
}

static void debug(char *syscallname, int procfd, const struct prstatus *p)
{
    char *debug0 = fetchstr(procfd, (char *) p->pr_sysarg[0]);

    fprintf(stderr, "DEBUG %s: arg0 = `%s'.\n", syscallname, debug0);
    free(debug0);
}



/* the same everywhere.  for clarity. */
#define HOOK_ARGS const struct prstatus *p, int procfd, void *v
#define HOOK_START { if ((status_t) v != ACTIVE) return(NO_COMMENT); \
			start(procfd); }
#define HOOK_STOP  return(stop());
#define ARG(i)     (p->pr_sysarg[i]) /* the i-th arg to the syscall */

action	open_hook(HOOK_ARGS)
{
    HOOK_START
    if ((ARG(1) & O_CREAT) /* mode ignored unless O_CREAT is used */
		&& (mode_t) ARG(2) & (S_ISUID|S_ISGID|S_ISVTX))
	return(DENY); /* no setuid, setgid, save-text files allowed */
    request(ARG(0), f2am((mode_t) ARG(1)));
    HOOK_STOP
}

/* equivalent to open(., O_WRONLY|O_CREAT|O_TRUNC, .), says the manual */
action	creat_hook(HOOK_ARGS)
{
    HOOK_START
    if ((mode_t) ARG(1) & (S_ISUID|S_ISGID|S_ISVTX))
	return(DENY); /* no setuid, setgid, save-text files allowed */
    request(ARG(0), f2am((mode_t) O_WRONLY|O_CREAT|O_TRUNC));
    HOOK_STOP
}

/* 
 * paranoia!  require *all* access permissions on the target:
 * don't want app's foiling access checks via links/renaming.
 * 
 * this is highly non-trivial because of the wacky wacky
 * semantics of symlink (sigh).  See comments below.
 * 
 * BUG.  If we have `foo -> bar', we want to make sure that the following
 * situation cannot arise:
 * 	the untrusted process having some access to foo which it doesn't have
 * 	to bar.
 * The current checks below are too strict, and Netscape's lock file
 * gets screwed by it.  Fix it.
 */
action	symlink_hook(HOOK_ARGS)
{
	char	*a0 = NULL, *a1 = NULL, *target = NULL;

	HOOK_START
	a0 = fetchstr(procfd, (char *) ARG(0));
	a1 = fetchstr(procfd, (char *) ARG(1));
	if (!a0 || !a1 || !(target = malloc(strlen(a0) + strlen(a1) + 2)))
		goto fail;
	if (a1[strlen(a1)-1] == '/')
		goto fail; /* sanity check: a1 shouldn't end in '/' */
	if (a0[0] == '/' || !strchr(a1, '/'))
		strcpy(target, a0);
	else {
		/*
		 * Non-trivial issue.
		 * symlink("xxx", "foo/bar") should issue
		 * request("foo/xxx", ALL) and request("foo/bar", WRITE)
		 */
		char	*p;

		strcpy(target, a1);
		p = strrchr(target, '/');
		*++p = '\0';
		strcpy(p, a0);
	}

	/* request(ARG(0), ALL); */ /* failsafe */
	request_local_buf(target, ALL);
	request(ARG(1), WRITE);
	free(a0); free(a1); free(target);
	HOOK_STOP

fail:
	if (a0)
		free(a0);
	if (a1)
		free(a1);
	if (target)
		free(target);
	return(DENY);
}

action	link_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), ALL);
    request(ARG(1), WRITE);
    HOOK_STOP
}
action	rename_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), ALL);
    request(ARG(1), WRITE);
    HOOK_STOP
}

action	unlink_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), WRITE);
    HOOK_STOP
}

action	mknod_hook(HOOK_ARGS)
{
    HOOK_START
    if (((mode_t) ARG(1) & S_IFMT) != S_IFIFO)
	return(DENY);
    if ((mode_t) ARG(1) & (S_ISUID|S_ISGID|S_ISVTX))
	return(DENY); /* no setuid, setgid, save-text files allowed */
    request(ARG(0), WRITE);
    HOOK_STOP
}

action	mkdir_hook(HOOK_ARGS)
{
    HOOK_START
    if ((mode_t) ARG(1) & S_ISUID)
	return(DENY); /* paranoia! setuid dirs are meaningless */
    request(ARG(0), WRITE);
    HOOK_STOP
}

action	rmdir_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), WRITE);
    HOOK_STOP
}

/*
 * Note that we require not only read&exec privileges to exec something,
 * but we also insist it not be setuid or setgid.  The claim is that
 * the way we currently use /proc will ensure that exec()ing a setuid
 * or setgid file will not be possible.
 */

action	exec_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), READ|EXEC);
    HOOK_STOP
}

action	execve_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), READ|EXEC);
    HOOK_STOP
}

action	utimes_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), WRITE);
    HOOK_STOP
}
action	utime_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), WRITE);
    HOOK_STOP
}

action	stat_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), READ);
    HOOK_STOP
}

action	lstat_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), READ);
    HOOK_STOP
}

action	readlink_hook(HOOK_ARGS)
{
    HOOK_START
    request(ARG(0), READ);
    HOOK_STOP
}

/*
 * BUG: against my better judgement, I'm enabling use of access().
 * Some important programs (e.g. /bin/sh) depend on access (!!),
 * so I think this is a necessary evil.
 * Still, that doesn't mean I have to leave it totally unrestricted,
 * and it doesn't mean I have to like it. :-)
 * 
 * (I really hate access(), because it has a race condition, and
 * thus its return value is not trustworthy -- yet every program that
 * ever calls it, trusts its return value.  It's very existence
 * is (IMHO) a bug with security consequences.  Grrr.)
 */
action  access_hook(HOOK_ARGS)
{
    ac_mode_t m;

    HOOK_START
    m = READ; /* we end up verifying the existence of the file, always. :-) */
    m |= (ARG(1) & (R_OK|F_OK)) ? READ : 0;
    m |= (ARG(1) & W_OK) ? WRITE : 0;
    m |= (ARG(1) & X_OK) ? EXEC : 0;
    request(ARG(0), m);
    HOOK_STOP
}

/* makes it the right type for the syscall_entry table */
#define HOOK_T action (*)(const struct prstatus *, int, void *)

/* syscalls I don't understand; thus deny & wait & see if anything breaks */
#define UNKNOWN DENY

/* the syscall_entry table is required */
const syscall_entry	entries[] = {
#if 1 
    {SYS_open, FUNC, (HOOK_T) open_hook},
#else
    {SYS_open, ALLOW, (HOOK_T)0},
#endif
    {SYS_creat, FUNC, (HOOK_T) creat_hook},
    {SYS_symlink, FUNC, (HOOK_T) symlink_hook},
    {SYS_link, FUNC, (HOOK_T) link_hook},
    {SYS_unlink, FUNC, (HOOK_T) unlink_hook},
    {SYS_mknod, FUNC, (HOOK_T) mknod_hook},
    {SYS_mkdir, FUNC, (HOOK_T) mkdir_hook},
    {SYS_rmdir, FUNC, (HOOK_T) rmdir_hook},
    {SYS_rename, FUNC, (HOOK_T) rename_hook},
    {SYS_exec, FUNC, (HOOK_T) exec_hook},
    {SYS_execve, FUNC, (HOOK_T) execve_hook},
    {SYS_pathconf, UNKNOWN, (HOOK_T) 0},
    {SYS_utimes, FUNC, (HOOK_T) utimes_hook},
    {SYS_utime, FUNC, (HOOK_T) utime_hook},
    {SYS_close, ALLOW, (HOOK_T) 0},
    {SYS_sync, ALLOW, (HOOK_T) 0},
#if 1
    {SYS_stat, FUNC, (HOOK_T) stat_hook},
#else
    {SYS_stat, ALLOW, (HOOK_T)0},
#endif
    {SYS_lstat, FUNC, (HOOK_T) lstat_hook},
    {SYS_readlink, FUNC, (HOOK_T) readlink_hook},
    {SYS_statfs, UNKNOWN, (HOOK_T) 0},
    {SYS_statvfs, UNKNOWN, (HOOK_T) 0},
    {SYS_sysfs, UNKNOWN, (HOOK_T) 0},
    {SYS_access, FUNC, (HOOK_T) access_hook},
    {SYS_chdir, DENY, (HOOK_T) 0},
    {SYS_fchdir, DENY, (HOOK_T) 0}
    };
const int		nentries = sizeof(entries) / sizeof(syscall_entry);


/*

MINOR UNRESOLVED ISSUES & BUGS:
    File creation: is governed by access permissions on the filename,
        NOT by those on the parent directory (ala Unix).  Change?
    I enabled access(), since some programs (e.g. /bin/sh) seem to require
        it.  See comments in the source for more info.  Any problems here?
    Should we allow stat()s on `..', etc. so that getcwd() works?
    Specifying `exec' without `read' is pointless.  I could've made the
        `exec' keyword automagically control `read' access too, but I
        figured it's better to force the config file to be explicit, so
        there's no confusion.  Any complaints?

ASSUMPTIONS & INTER-MODULE DEPENDENCIES:
    All fd's closed before helper app is started.  Passing fd's through
        AF_UNIX sockets is disallowed by some other module.
    There's no way to pass fd's through a FIFO -- right?
    Umask is set before helper app is started.
    Core dumps will be disabled by some other module (e.g. ulimit 0?).
    If `read' access to /dev/tcp (/dev/udp?) is allowed (in the config file),
        some other module (e.g. connect) will mediate network connections.
    Access to /proc must be disallowed (explicitly, in the config file).
    If the helper app can read a file, assume that it can leak that
        information out over the network via some covert channel.
        Thus, must control all reads carefully (explicitly, in the config file).
    There must be a guarantee that attempts to exec() a setuid (OR setgid!)
        files will fail -- some other module must handle this.

*/
