/*
 * Copyright (c) 1993 The Regents of the University of California.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include "systable.h"
#include "module.h"
#include "version.h"

#define MAXLINELENGTH 2048

/* Initialize the systable */
void init_table(actionlist systable[NSYSCALL])
{
    int i;

    for(i=0;i<NSYSCALL;++i)
    {
	systable[i].head = NULL;
	systable[i].tail = &(systable[i].head);
    }
}

/* Setup the supplied systable with the given config file.
   conf should be at the beginning of the file.
   systable should be initialized properly. */
void conf_table(FILE *conf, actionlist systable[NSYSCALL])
{
    char buf[MAXLINELENGTH];
    int i;

    while(fgets(buf, MAXLINELENGTH-1, conf))
    {
	char *nl, *sp, *fn;
	char filename[MAXLINELENGTH];
	void *dlhandle;
	void* (*initptr)(const char *);
	void *state;
	int *versptr;
	int *nentries;
	syscall_entry *entries;

	/* Check for a newline */
	nl = strchr(buf, '\n');

	if (!nl)
	{
	    fprintf(stderr, "Line too long in config file! %s...\n", buf);
	    exit(1);
	}

	/* Zap the newline */
	*nl = '\0';

	/* Strip leading whitespace */
	fn = buf;
	while(*fn == ' ' || *fn == '\t') ++fn;

	/* Check for blank lines or comments */
	if (*fn == '#' || *fn == '\0') continue;

	/* Find the first space or tab */
	sp = fn;
	while (*sp != ' ' && *sp != '\t' && *sp != '\0') ++sp;
	if (*sp)
	{
	    *sp = '\0';
	    ++sp;

	    /* Strip leading whitespace from the args */
	    while (*sp == ' ' || *sp == '\t') ++sp;
	}

	/* Construct the filename */
	sprintf(filename, "%s.so", fn);

	/* Open the module */
	dlhandle = dlopen(filename, RTLD_NOW); 
	if (!dlhandle)
	{
#ifdef LIBDIR
	    sprintf(filename, "%s/%s.so", LIBDIR, fn);
	    dlhandle = dlopen(filename, RTLD_NOW);
	    if (!dlhandle)
	    {
#endif /* LIBDIR */
		perror("Cannot open module");
		fprintf(stderr, "%s: %s\n", filename, dlerror());
		exit(1);
#ifdef LIBDIR
	    }
#endif /* LIBDIR */
	}

	/* Check the version number */
	versptr = (int *)dlsym(dlhandle, "module_version");
	if (!versptr || (*versptr) != module_version)
	{
	    /* Bad version */
	    fprintf(stderr, "Module %s has a bad version number.\n", filename);
	    exit(1);
	}

	/* See if there's an init */
	initptr = (void * (*)(const char *))dlsym(dlhandle, "init");
	if (initptr)
	{
	    /* Run the init and check the return code */
	    state = (*initptr)(sp);
	    if (state == INIT_FAIL)
	    {
		fprintf(stderr, "Module %s initialization failed.\n", filename);
		exit(1);
	    }
	}
	else
	{
	    state = NULL;
	}

	/* Get the number of entries in the table */
	nentries = (int *)dlsym(dlhandle, "nentries");
	if (!nentries)
	{
	    fprintf(stderr, "Module %s has no 'nentries' symbol!\n", filename);
	    exit(1);
	}

	if (*nentries)
	{
	    entries = (syscall_entry *)dlsym(dlhandle, "entries");
	    if (!entries)
	    {
		fprintf(stderr, "Module %s has no 'entries' symbol!\n", filename);
		exit(1);
	    }

	    /* Load the entries into the table */
	    for(i=0;i<(*nentries);++i)
	    {
		/* Allocate a new node and link it in */
		actionnode *newNode = (actionnode *)malloc(sizeof(actionnode));
		if (!newNode)
		{
		    fprintf(stderr, "Out of memory allocating actionnode!\n");
		    exit(1);
		}
		newNode->kind = entries[i].kind;
		newNode->hook = entries[i].hook;
		newNode->state = state;
		newNode->next = NULL;
		*(systable[entries[i].which].tail) = newNode;
		systable[entries[i].which].tail = &(newNode->next);
	    }
	}
    }
}

static void delete_node(actionnode **node)
{
    actionnode *tofree;

    /* Be paranoid */
    if (!node || !*node) return;

    tofree = *node;
    *node = (*node)->next;
    free(tofree);
}

static void delete_entries(actionnode **node)
{
    while(node && *node)
    {
	if ((*node)->kind != EXIT_FUNC)
	{
	    delete_node(node);
	}
	else
	{
	    node = &((*node)->next);
	}
    }
}

/* Optimize the systable, and mark which syscalls need to be watched for
    entry and/or exit */
void optimize_table(actionlist systable[NSYSCALL], sysset_t *entryset,
	sysset_t *exitset, int verbose)
{
    int sysc;
    actionnode **previfshort, **current, **nextcurrent;
    int entryhandle, exithandle;
    action justallow;

    for(sysc=0;sysc<NSYSCALL;++sysc)
    {
	entryhandle = exithandle = 0;
	current = &(systable[sysc].head);
	previfshort = NULL;
	justallow = ALLOW;

	/* Go through the list and remove redundant things */
	while(*current)
	{
	    /* What should we do with this? */
	    switch((*current)->kind)
	    {
	    case NO_COMMENT:
		/* This is a useless node; delete it. */
		delete_node(current);
		break;

	    case SUPER_ALLOW:
	    case SUPER_DENY:
		/* Delete all entry-type things after this node */
		delete_entries(&((*current)->next));
		/* FALLTHROUGH */
	    case DENY:
	    case ALLOW:
		entryhandle = 1;
		if (justallow != FUNC) justallow = (*current)->kind;

		/* Remember the next node */
		nextcurrent = &((*current)->next);

		/* If the previous node was a shortcut, delete it,
		    because it would be superceded by this one */
		if (previfshort)
		{
		    delete_node(previfshort);
		}
		else
		{
		    previfshort = current;
		}

		/* This node is a shortcut; advance the current node */
		current = nextcurrent;

		/* Advance the previfshort pointer past EXIT_FUNCs,
		    if necessary */
		while(previfshort && *previfshort &&
		    (*previfshort)->kind == EXIT_FUNC)
		{
		    previfshort = &((*previfshort)->next);
		}
		
		break;

	    case FUNC:
		/* Leave the entry handler alone. */
		entryhandle = 1;
		justallow = FUNC;

		/* This node is not a shortcut; advance the current node */
		previfshort = NULL;
		current = &((*current)->next);
		break;

	    case EXIT_FUNC:
		/* Leave the exit handler alone. */
		exithandle = 1;

		/* Advance the current node */
		current = &((*current)->next);
		break;

	    default:
		/* This is a bad node; delete it. */
		fprintf(stderr, "Error: Bad kind (%d); deleting node.\n",
		    (*current)->kind);
		delete_node(current);
		break;
	    }
	}

	/* Keep the tail of the list right */
	systable[sysc].tail = current;

	if (exithandle)
	{
	    /* Add this syscall to the list of those to trap on exit */
	    praddset(exitset, sysc);
	}

	if (entryhandle)
	{
	    if (justallow != ALLOW && justallow != SUPER_ALLOW)
	    {
		/* Add this syscall to the list of those to trap on entry */
		praddset(entryset, sysc);
	    }
	}
	else if (verbose)
	{
	    fprintf(stderr, "Warning: syscall %d is not handled by any module."
				"It will be denied.\n", sysc);
	}
    }
}
