/*
 * Copyright (c) 1993 The Regents of the University of California.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stropts.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/signal.h>
#include <sys/fault.h>
#include <sys/syscall.h>
#include <sys/procfs.h>
#include <sys/param.h>
#include <thread.h>
#include <synch.h>
#include "sysentry.h"
#include "global.h"

int proc_attach(unsigned short pid);
FILE *openLogFile(char *logFileName, int pid);
void Usage(char *progname);

char **traced_environ;
extern char **environ;

int tracedProgIsMultithreaded = 0;

FILE *log = NULL;

FDClass *fdClassMap;

/*
 * Just in case we're tracing a multithreaded process, put
 * locks around our updates to data structures.
 */
mutex_t mutex; 

int main(int argc, char **argv)
{
    FILE *cf;
    int pipefds[2];
    int tracedfd;
    unsigned short pid = 0;
    struct prstatus prstat;
    struct prrun prrn;
    int debug = 0;
    char *progname = argv[0];
    char *logfileName = NULL;
    char buffer[80];
    mutex_init(&mutex, NULL, NULL);

    log = stderr;

    fdClassMap = new FDClass();

    /* Remove the argv[0] */
    --argc;
    ++argv;


    /* Parse options */
    while (argc > 0 && argv[0][0] == '-')
    {
	if (!strcmp(argv[0], "-v"))
	{
	    debug = 1;
	    --argc;
	    ++argv;
	}
	else if (!strcmp(argv[0], "-p"))
	{
	    if (argc > 1)
	    {
		pid = atoi(argv[1]);
		argc -= 2;
		argv += 2;
	    }
	    else
	    {
		Usage(progname);
	    }
	}
	else if(!strcmp(argv[0], "-l")){

	  if(argc > 1){
	    logfileName = argv[1];
	    argc -= 2;
	    argv += 2;
	  }
	}
	else
	{
	    Usage(progname);
	}
    }

    /* Check args */
    if ((pid == 0 && argc < 1) || (pid != 0 && argc > 0))
    {
	Usage(progname);
    }

    
    sigset(SIGCLD, SIG_IGN);
    
    char buf[10] = { '\n' };
    /* fork a child to exec the command, if necessary */
    pipefds[1] = -1;
    if (!pid)
      {
	/* Make a pipe */
	pipe(pipefds);
	/* Do the fork thing */
	pid = fork();
	if (!pid)
	{
	    /* I'm the child! */
	    close(pipefds[1]);
	    fflush(log);
	    
	    /* Signal the parent to proceed */
	    write(pipefds[0], buf, 1);
	    
	    /* Wait until the parent is ready */
	    read(pipefds[0], buf, 1);
	    close(pipefds[0]);

	    execvp(argv[0], argv);
	    perror("Exec from sandbox failed.");
	    exit(1);
	}
	else
	{
	    close(pipefds[0]);
	    sigset(SIGCLD, SIG_IGN);
	    /* Wait for the child to set its environment */
	    read(pipefds[1], buf, 1);
	}
      }
    
    
    /* Clear the prstat structure */
    memset(&prstat, 0, sizeof(prstat));
    
    /* Attach to the pid */
    tracedfd = proc_attach(pid);
    if(logfileName != NULL){
      fclose(log);
      log = openLogFile(logfileName, pid);
    }

    /*set the faults for which it should stop*/
    prrn.pr_flags=PRSFAULT;
    premptyset(&prrn.pr_fault);
    praddset(&prrn.pr_fault,FLTPAGE);
    /* Set it running */
    if (ioctl(tracedfd, PIOCRUN, &prrn) < 0){ 
      perror("ioctl PIOCRUN"); exit(1); 
    }
    
    /* Wake it up */
    if (pipefds[1] >= 0)
      {
	write(pipefds[1], "\n", 1);
	close(pipefds[1]);
      }
    

    /* Main loop */
    while(1)
      {

	/* Wait for it to do something interesting */
	if(ioctl(tracedfd, PIOCWSTOP, &prstat) < 0) break;
	
	
	/* OK; why did it stop? */
	switch (prstat.pr_why){
	case PR_SYSENTRY: 
	  mutex_lock(&mutex);
	  sysDispatch[prstat.pr_what].hook->Enter(&prstat, tracedfd);
	  mutex_unlock(&mutex);
	  break;
	  
	  
	case PR_SYSEXIT:
	  mutex_lock(&mutex);
	  sysDispatch[prstat.pr_what].hook->Exit(&prstat, tracedfd);
	  mutex_unlock(&mutex);

	  
	  /* See if fork or vfork just exited */
	  if (prstat.pr_why == PR_SYSEXIT &&
	      (prstat.pr_what == SYS_fork 
	       || prstat.pr_what == SYS_vfork
	       || prstat.pr_what == SYS_fork1)){
	    /* The new pid is prstat.pr_reg[R_O0] */
	    int child = fork();
	    if (child == 0){
	      /* I am the child. Trace the new process */
	      close(tracedfd);
	      fflush(log);
	      int parentPid = pid;
	      pid = prstat.pr_reg[R_O0];
	      char msg[80];
	      fprintf(log, "NOTE: Pid %d forked %d (traceproc %d)\n", 
		parentPid, pid, getpid());
	      	      
	      tracedfd = proc_attach(pid);
	      if(logfileName != NULL){
		fclose(log);
		log = openLogFile(logfileName, pid);
	      }
	    }
	    else if (child < 0){
	      perror("fork");
	      return 1;
	    }
	  }
	  
	  break;


	case PR_FAULTED:
	  if(prstat.pr_what==FLTPAGE){
	    mutex_lock(&mutex);
	    SE_pageFaultHook->Enter(&prstat, tracedfd);
	    mutex_unlock(&mutex);
	  }
	  break;

	default:
	  fprintf(stderr, "XXX Unknown case in switch\n");
	  exit(-1);
	}
	
	/* Reap any leftover children */
	while(waitpid(-1, NULL, WNOHANG) > 0);
	
	/* Set the process running again. */
	if(ioctl(tracedfd, PIOCRUN, &prrn) < 0) { perror("ioctl"); exit(1); }
    }
    SE_report();
    SE_pageFaultHook->Report();
}

int proc_attach(unsigned short pid)
{
    char procfname[20];
    int procfd;
    long flg;
    sysset_t sysentryset, sysexitset;
    int sysc;

    /* For now, trace all calls */
    for(sysc=0;sysc<NSYSCALL;++sysc){
      praddset(&sysexitset, sysc);
      praddset(&sysentryset, sysc);
    }
    
    sprintf(procfname, "/proc/%05d", pid);
    procfd = open(procfname, O_RDWR|O_EXCL);
    if (procfd < 0)
      {
	fprintf(log, "proc_attach to %hu failed\n", pid);
	exit(1);

      }
    
    /* Stop the target process (just in case) */
    if (ioctl(procfd, PIOCSTOP, 0) < 0) { perror("ioctl"); exit(1); }

    /* Set keep on fork and kill on last close */
    flg = PR_FORK|PR_RLC|PR_ASYNC;
    if (ioctl(procfd, PIOCSET, &flg) < 0) { perror("ioctl"); exit(1);}


    /* Mark the syscalls to be traced on entry */
    if (ioctl(procfd, PIOCSENTRY, &sysentryset) < 0)
      { perror("ioctl PIOCSENTRY"); exit(1); }
    
    
    /* Mark the syscalls to be traced on exit */
    if (ioctl(procfd, PIOCSEXIT, &sysexitset) < 0)
	{ perror("ioctl PIOCSEXIT"); exit(1); }

    SE_init(log);
    tracedProgIsMultithreaded = 0;

    return procfd;
}

FILE *openLogFile(char *logFileName, int pid)
{
  FILE *log;
  char *fileName;
  assert(logFileName != NULL);
  fileName = (char *)malloc((strlen(logFileName) + 10) * sizeof(char));
  if(!fileName){
    fprintf(log, "Cannot change log file (nomem)\n");
    exit(-1);
  }
  sprintf(fileName, "%s-%05d", logFileName, pid);
  log = fopen(fileName, "w");
  if(!log){
    log = stderr;
    fprintf(log, "Cannot change log file to %s\n", 
	    fileName);
    Usage("xyz");
  }
  return log;
}


void Usage(char *progname)
{
  fprintf(log, "Usage: %s [-v] [-f configfile] [-l logfile] command | -p pid\n",progname);
    exit(1);
}

void traceChild()
{





}
