/*
 *------------------------------------------------------------------
 *
 * ProcHook --
 *
 *          Hooks for handling process-management syscalls
 *
 *------------------------------------------------------------------
 */
#include <assert.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/syscall.h>
#include <sys/procfs.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <sys/statvfs.h>
#include <unistd.h>
#include <ustat.h>

#include "fetchstr.h"
#include "global.h"
#include "log.h"

static int allowedToTrace(char *childName);
static void detach(const struct prstatus *p, int procfd, char *childName);
static void printArgs(const struct prstatus *p, int procfd, char *childName);
static void waitForDetached(int pid);


/*-----------------------------------------------------------------
*/

/*
 *------------------------------------------------------------------
 *
 * OpenFileHookEntry --
 *
 *          Gets called on every SYS_open.
 *          Used to tell log data structure the time spent in 
 *          open of this file system. 
 *
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
void
OpenFileHookEntry(const struct prstatus *p, int procfd)
{
  log->openStart=gethrtime();
}

/*
 *------------------------------------------------------------------
 *
 * OpenFileHookExit --
 *
 *          Gets called on every SYS_open.
 *          Used to tell log data structure the time spent in 
 *          open of this file system. 
 *
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
void
OpenFileHookExit(const struct prstatus *p, int procfd)
{
  char *childName;
  struct statvfs stvfs;
  hrtime_t end;
  end= gethrtime();
  if (log->openStart == 0) return;
  childName = fetchstr(procfd, (char *)p->pr_sysarg[0]);
  statvfs(childName,&stvfs);
  if(strcmp("ufs",stvfs.f_basetype)==0){
    log->opentime[UFS]+=end - log->openStart;
    log->opencount[UFS]++;
  }
  else  if(strcmp("nfs",stvfs.f_basetype)==0){
    log->opentime[NFS]+=end - log->openStart;
    log->opencount[NFS]++;
  }
  else  if(strcmp("proc",stvfs.f_basetype)==0){
    log->opentime[PROC]+=end - log->openStart;
    log->opencount[PROC]++;
  }
  else  if(strcmp("swap",stvfs.f_basetype)==0){
    log->opentime[SWAP]+=end - log->openStart;
    log->opencount[SWAP]++;
  }
  else if(strcmp("fd",stvfs.f_basetype)==0){
    log->opentime[FD]+=end - log->openStart;
    log->opencount[FD]++;
  }
  else  if(strcmp("tmpfs",stvfs.f_basetype)==0){
    log->opentime[TMPFS]+=end - log->openStart;
    log->opencount[TMPFS]++;
  }
  else{
    log->opentime[UNKNOWNFS]+=end - log->openStart;
    log->opencount[UNKNOWNFS]++;
  }
  
  free(childName);
  log->openStart=0;
}

/*                
 *------------------------------------------------------------------
 *                 
 * CloseFileHookEntry --
 *
 *          Gets called on every SYS_close.
 *          Need to tell log data structure              
 *          the time spent in this file system.                                 *                                   
 *
 * Arguments:
 *      None.      
 *
 * Results:                                              
 *      None.                                               
 *          
 * Side effects:
 *      None.                                                  
 *          
 *------------------------------------------------------------------
 */
void
CloseFileHookEntry(const struct prstatus *p, int procfd)
{
  log->closeStart=gethrtime(); 
}




/*                
 *------------------------------------------------------------------
 *                 
 * CloseFileHookExit --
 *
 *          Gets called on every SYS_close.
 *          Need to tell log data structure              
 *          the time spent in this file system.                                 *                                   
 *
 * Arguments:
 *      None.      
 *
 * Results:                                              
 *      None.                                               
 *          
 * Side effects:
 *      None.                                                  
 *          
 *------------------------------------------------------------------
 */
void
CloseFileHookExit(const struct prstatus *p, int procfd)
{
 
  int readfd;
  struct statvfs stvfs;
  hrtime_t end;
  end= gethrtime();
  if (log->closeStart == 0) return;
  readfd= p->pr_sysarg[0];
  fstatvfs(readfd,&stvfs);
  
  if(strcmp("ufs",stvfs.f_basetype)==0){
    log->closetime[UFS]+=end - log->closeStart;
    log->closecount[UFS]++;
  }
  else  if(strcmp("nfs",stvfs.f_basetype)==0){
    log->closetime[NFS]+=end - log->closeStart;
    log->closecount[NFS]++;
  }
  else  if(strcmp("proc",stvfs.f_basetype)==0){
    log->closetime[PROC]+=end - log->closeStart;
    log->closecount[PROC]++;
  }
  else  if(strcmp("swap",stvfs.f_basetype)==0){
    log->closetime[SWAP]+=end - log->closeStart;
    log->closecount[SWAP]++;
  }
  else if(strcmp("fd",stvfs.f_basetype)==0){
    log->closetime[FD]+=end - log->closeStart;
    log->closecount[FD]++;
  }
  else  if(strcmp("tmpfs",stvfs.f_basetype)==0){
    log->closetime[TMPFS]+=end - log->closeStart;
    log->closecount[TMPFS]++;
  }
  else{
    log->closetime[UNKNOWNFS]+=end - log->closeStart;
      log->closecount[UNKNOWNFS]++;
  }
  log->closeStart=0;
}



/*                
 *------------------------------------------------------------------
 *                 
 * MmapHookExit --
 *
 *          Gets called on every SYS_mmap.
 *          Need to tell log data structure              
 *          about the mapping.                                 
 *                                   
 *
 * Arguments:
 *      None.      
 *
 * Results:                                              
 *      None.                                               
 *          
 * Side effects:
 *      None.                                                  
 *          
 *------------------------------------------------------------------
 */
void
MmapHookExit(const struct prstatus *p, int procfd)
{
  unsigned long address=*(p->pr_reg+8);
  if(address<0){
    fprintf(errLog,"\nMmap failed");
    return;
  }
  int len=p->pr_sysarg[1];
    
 }



/*                
 *------------------------------------------------------------------
 *                 
 * MunmapHookExit --
 *
 *          Gets called on every SYS_munmap.
 *          Need to tell log data structure              
 *          to remove the mapping info.
 *                                   
 *
 * Arguments:
 *      None.      
 *
 * Results:                                              
 *      None.                                               
 *          
 * Side effects:
 *      None.                                                  
 *          
 *------------------------------------------------------------------
 */
void
MunmapHookExit(const struct prstatus *p, int procfd)
{

}

