; -*- Mode: Lisp; Syntax: Common-Lisp; Package: KM; Base: 10 -*-

(in-package 'km)



;;;;				 KM Extensions
;;;;				 -------------



;;; GET+  -- the universal get function with lots of options
;;; ----
;;;
;;; purpose: to get the filler (i.e. set of slot values) for a slot
;;;
;;; arguments
;;;  required
;;;    unit:  unit to be accessed
;;;    slot:  slot to be accessed on unit
;;;  optional
;;;    levels-of-kb: the levels of the knowledge base to be accessed
;;;                when embedded units are returned (i.e. bare-value
;;;                is nil).
;;;                A list of one or more specifications, each of which
;;;                is of the form:
;;;                     1) a positive integer
;;;                     2) a range of levels specified by the list:
;;;                        (m n), where m and n are positive integers
;;;                        and n>m.
;;;                The specifications are interpreted conjunctively.
;;;                For example, ((1 10) (30 85) 4) retrieves information
;;;                on levels 1-10, 30-85, AND level 4 of the knowledge base.
;;;                To determine the information stored on each level of the
;;;                knowledge base, see the LEVEL-OF-KB facet on each slot,
;;;                or the summary of this information in the file:
;;;                         levels-in-kb in the BKB directory
;;;                (default: (1 100), i.e. domain knowledge only)
;;;    bare-value: t if the bare value is to be returned;
;;;                nil if the value annotations (if any) are to be returned
;;;                (default: nil)
;;;    use-inheritance-always: t if inheritance should be used to augment
;;;                any local values on unit/slot
;;;                (default: nil)
;;;    use-inheritance-if-novalue: t if inheritance should be used when
;;;                there is no local value on unit/slot
;;;                (default: nil)
;;;    listify: if listify=t then values returned are listified
;;;                (default: t)
;;;    nullify: if nullify=t then nil is returned when unit/slot has no
;;;                no value;
;;;                otherwise, :novalue is returned when unit/slot has no 
;;;                value (default: nil)

(defun get+ (unit slot
		  &key 
		  (levels-of-kb '(1 100))
		  (bare-value nil)
		  (use-inheritance-always nil)
		  (use-inheritance-if-novalue nil)
		  (listify t)
		  (nullify nil))
  (let ((vals nil))
    (when use-inheritance-always
      (if bare-value
	  (setf vals (get-global-without-annotation unit slot))
	  (setf vals (get-global-with-annotation 
		      unit slot :levels-of-kb levels-of-kb))))
    (when (not use-inheritance-always)
      (if bare-value
	  (setf vals (get-local-without-annotation unit slot))
	  (setf vals (get-local-with-annotation unit slot 
						:levels-of-kb levels-of-kb)))
      (when (and (equal vals ':no-value)
		 use-inheritance-if-novalue)
	(if bare-value
	    (setf vals (get-global-without-annotation))
	    (setf vals (get-global-with-annotation 
			unit slot :levels-of-kb levels-of-kb)))))
    (when (and listify (atom vals))
      (setf vals (list vals)))
    (when (and nullify (not (tvalue? vals)))
      (setf vals nil))
    vals))


;;; GET-UNIT+ 
;;; ----------
;;;
;;; purpose: to get the contents of a unit (top-level or embedded)
;;;
;;; arguments
;;;  required
;;;    unit:  unit to be accessed 
;;;  optional
;;;    levels-of-kb: the levels of the knowledge base to be accessed
;;;                A list of one or more specifications, each of which
;;;                is of the form:
;;;                     1) a positive integer
;;;                     2) a range of levels specified by the list:
;;;                        (m n), where m and n are positive integers
;;;                        and n>m.
;;;                The specifications are interpreted conjunctively.
;;;                For example, ((1 10) (30 85) 4) retrieves information
;;;                on levels 1-10, 30-85, AND level 4 of the knowledge base.
;;;                To determine the information stored on each level of the
;;;                knowledge base, see the LEVEL-OF-KB facet on each slot,
;;;                or the summary of this information in the file:
;;;                         levels-in-kb in the BKB directory
;;;                (default: (1 100), i.e. domain knowledge only)
;;;    bare-unit:  t if the unit is to be returned "bare", i.e. without
;;;                any substructures on slots and values within the unit.
;;;                nil if the substructures are to be returned.
;;;                (default: nil)
;;;    get-slots-using-inheritance: t if inheritance should be used to
;;;                get all the legal slots (and their fillers) for unit.
;;;                (note: this option does not use inheritance to get
;;;                all the values for slots on the unit.  For this, use
;;;                get+ with the use-inheritance features).
;;;                nil if only unit's explicit slots should be returned
;;;                (default: nil)


;(defun get-unit (unit 
;		 &key 
;		 (levels-of-kb '(1 100))
;		 (bare-unit nil)
;		 (get-slots-using-inheritance nil))
;
;)





;;; Specialized Selector Functions
;;; ------------------------------

;;; finds values of get-global call
(defun get-global-vals (addr)
  (first (get-global addr)))


;;; retrieves an unannotated value from an annotated value
(defun bareify (annot-value)
  (first annot-value))


;;; returns the source unit of the address, where source means
;;; the origin, and source may be either an top-level or embedded unit
(defun start-of-path (addr)
  (if (listp addr)
      (first addr)
      addr))




;;; Get-Global-with-Annotation Functions
;;; ------------------------------------

;;; get-global-with-annotation returns those values that are
;;; locally placed on the unit at the slot (unit and slot are the
;;; passed parameters), plus those that can be inherited 
;;; to the address.  The only tricky part of the function is pruning 
;;; values that are replaced by more specific ones.
;;; (note: unit may be an implicit unit.)
;;;
;;; Since unit might not have any explicit generalizations, 
;;; the function might "guess" what generalizations it might have.
;;; This occurs only when unit is an embedded unit which has not yet
;;; been created.  For example, the caller might ask for the filler of:
;;;   - unit = (flower parts)
;;;   - slot =  ordinality
;;; even though the unit does not exist.  By "guessing", get-global
;;; starts the "ancestry-thread" for the unit at the slot "parts"
;;;
;;; Important Note: we use ancestry* to perform a breadth-first search
;;; --------------  upward through the KB; this is accomplished, but
;;;                 it is not true that the order of the values
;;;                 returned by ancestry* have the property that they
;;;                 are in increasing order of generality; in fact,
;;;                 one value may be more specific than its predecessor
;;;                 in the list of values returned; this has far-reaching
;;;                 and (yet unforseen) consequences

(defun get-global-with-annotation (unit slot 
					&key 
					(levels-of-kb '(1 100)))

  (let* ((global-vals (get-local-with-annotation unit slot
						 :levels-of-kb levels-of-kb))
	 (replacements (slot-values-replaced unit slot global-vals))
	 (ancestors (ancestry-with-guessing+ unit)))
    (dolist (ancestor ancestors)
      (let ((vals-on-ancestor (get-local-with-annotation 
			       ancestor slot
			       :levels-of-kb levels-of-kb)))
	(dolist (val-on-ancestor vals-on-ancestor)
	  (let ((extended-addr-with-value
		 (extend-address-indefinitely ancestor
					      slot
					      (first val-on-ancestor))))
					; recall that val-on-ancestor is of 
					;the form: (value annotation)

	    ;;if extended-addr-with-value has been replaced by a more
	    ;;specific value, then remove it from the list of replacements
	    ;;(an efficiency hack; controls the length of the list).
	    ;;Else, add an explanation to the annotation on val-on-ancestor
	    ;;and add val-on-ancestor to global-vals

;	    (format *terminal-io* "extended-addr-with-value: ~a~%"
;		    extended-addr-with-value)
;	    (format *terminal-io* "replacements:~a~%" replacements)
	    (if (member extended-addr-with-value replacements 
			:test #'equal)
		(setf replacements
		      (delete extended-addr-with-value replacements
			      :test #'equal))
		(setf global-vals 
		      (push 

		       (cons
			(first val-on-ancestor)
			(list (cons (list 'inherited-from 
					  (extend-address-indefinitely
					   ancestor slot
					   (first val-on-ancestor)))
				    (second val-on-ancestor))))

		       global-vals)))))

	;;destructive operation rationale: arguments built up from copies
	(setf replacements 
	      (nconc 
	       (slot-values-replaced ancestor slot vals-on-ancestor) 
	       replacements))))

    ;;reverse the order of global-vals so that the they are increasing
    ;;generality (local values on (unit slot) are first).
    ;;destructive operation rationale: arguments built up from copies

    (nreverse global-vals)))
  

;;; return the ancestry of a unit, "guessing" if necessary.
;;; first determines if the unit has explicit generalizations
;;; in the KB, thereby giving it ancestry;
;;; if so, they are returned;
;;; otherwise "guesses" a generalization for the first
;;; link (as defined below), starting the "ancestry-thread",
;;; and then performs the traditional ancestry* operation
(defun ancestry-with-guessing (unit)
  (let ((ancestors (ancestry* unit)))
    (if (length-of-1 ancestors)
	(ancestry* (first (guess-generalization unit)))
	ancestors)))
   
 

;; return the list of addresses that are replaced by the values
;; at (unit slot)
(defun slot-values-replaced (unit slot values)
  (mapcan #'(lambda (val)
	     (let ((replaces-for-val (get-replaces+ unit slot val)))
	       (if replaces-for-val
		   (copy-list replaces-for-val))))
	  values))

;;; returns slot values at address ``(unit slot)'', excluding any
;;; annotation on those values.

(defun get-local-without-annotation (unit slot)
   (let* ((slot-filler (get-slot-object (extend-address unit slot)))
	 (values+annots (cdr slot-filler)))
     (first values+annots)))


;;; returns slot values at address ``(unit slot)'', including the
;;; annotation on each value. 
;;;
;;; returns a list of the form:
;;;  ((v1 annot1) (v2 annot2) ... )
;;;  where annot-n is a list of slots and values
;;; example:
;;;    (get-local-with-annotation 'embryo-sac-formation 'developee)
;;;    returns the single value (embryo-sac) with its annotation:
;;;        ((EMBRYO-SAC
;;;         ((I-GENLS (EMBRYO-SAC))
;;;          (BEFORE-STATE (MEGASPORE-MOTHER-CELL))
;;;          (AFTER-STATE (EMBRYO-SAC)))))

(defun get-local-with-annotation (unit slot &key (levels-of-kb '(1 100)))
   (let* ((slot-filler (get-slot-object (extend-address unit slot)))
	 (values+annots (cdr slot-filler))
	 (values (listifyn (first values+annots))) ;use listify in case the
	 (annots (rest values+annots))             ;slot value is :novalue
	 (return-list nil))
    (dolist (value values)
      (setf return-list
	    (push
	     (cons value 
		   (get-local-with-annotation-aux value annots levels-of-kb))
	     return-list)))
    return-list))


;;; return the annotation (from the list annots) that corresponds to value
(defun get-local-with-annotation-aux (value annots levels-of-kb)
  (dolist (annot annots)
    (when (equal value (first annot))
      (return-from get-local-with-annotation-aux 
	(list (cddr annot)))))
  nil)

(defun filter-by-level (unit levels-of-kb)
  t)

(defun my-getobj (unit levels-of-kb)
  (filter-by-level (getobj unit) levels-of-kb))

;;; Much like get-global-with-annotation, but leaves off the annotation.
;;; returns the values stored locally on (unit slot) plus those
;;; values inherited to (unit slot).  
(defun get-global-without-annotation (unit slot)
  (let ((vals-with-annotation (get-global-with-annotation unit slot)))
    (mapcar #'(lambda (val)
		(first val))
	    vals-with-annotation)))


;;; Put-Local-with-annotation
;;; -------------------------
(defun put-local-with-annotation (unit slot vals)
  (dolist (val vals)
    (add-val-with-annotation unit slot val)))



;;; Kludged Add-val-with-annotation
;;; -------------------------------

;;; adding the annotation is a KLUDGE (it only works for 2 levels)
;;; see note on FIX-IT list
;;; ALSO, it installs the value locally, without installing inverses
;;; contains a kludge in the atom test below; we've got to get the
;;; new low-level access fn's corrected!
(defun add-val-with-annotation (unit slot value)
  (let ((bare-value (first value))
	(annotation (second value)))
    (add-val-local (extend-address unit slot) bare-value)
    (when annotation
      (dolist (annot annotation)
	(let ((annot-slot (first annot))
	      (annot-fillers (second annot)))
	  (if (atom annot-fillers)
	    (add-val-local (extend-address-indefinitely unit slot bare-value
						  annot-slot)
		   annot-fillers)
	    (dolist (val annot-fillers)
	      (add-val-local (extend-address-indefinitely unit slot bare-value
							  annot-slot)
			     val))))))))


;;; Add-Val (Our own version)
;;; -------------------------

;;; this version does not do any get-globals for rules
;;; or constraints; it does, however, compute and
;;; install inverses in accord with the global inverse
;;; variable setting 

(defun my-add-val (addr val)
  (when (tvalue? val)
    (let ((oldval (get-local-internal addr)))
      (unless (member val oldval :test #'equal)
	(put-local-internal addr (cons-end val oldval)))))
  (let ((inverse-pairs (compute-inverse addr val)))
    (dolist (inv-pair inverse-pairs)
      (put-local (first inv-pair) (second inv-pair)))))


