; -*- Mode: Lisp; Syntax: Common-Lisp; Package: KM; Base: 10 -*-


(in-package 'km)

#||  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

The following are examples of view retriever invocations:

  (retrieve-view '(embryo-sac-formation dimension (black-box)))
  (retrieve-view '(photosynthesis dimension (black-box)))
  (retrieve-view '(photosynthesis dimension (event-structural)))
  (retrieve-view '(photosynthesis dimension (black-box event-structural)))
  (retrieve-view '(photosynthesis as-kind-of conversion))
  (retrieve-view '(photosynthesis as-kind-of production))
  (retrieve-view '(photosynthesis as-kind-of production))
  (retrieve-view '(flower dimension (spatial-structural))) --> vp1
  (retrieve-view '(angiosperm-sexual-reproduction dimension 
                  (event-structural)))                     --> vp2
  (retrieve-view '(composite vp1 vp2 (actor-in)))
      where: vp1 and vp2 are bound above
  (retrieve-view '(embryo-sac-formation dimension 
                 (event-structural)))               --> vp1
  (retrieve-view '(embryo-sac dimension (spatial-structural))) --> vp2
  (retrieve-view '(composite vp2 vp1 (actor-in)))
      where: vp1 and vp2 are bound above
  (retrieve-view '(composite (flower dimension (spatial-structural))
                             (angiosperm-sexual-reproduction 
                              dimension (event-structural))
                             (actor-in)))

||#


;;;;			      The View Retriever
;;;;			      ------------------


;;;; Organization of File
;;;; --------------------
;;;; 1. The Driver
;;;; 2. Basic Dimension Constructor
;;;; 3. As-Kind-Of Constructor
;;;; 4. Composite View Constructor
;;;; 5. View Retriever Error Signaler
;;;; 6. Test Call Functions


;;;; To Fix
;;;; ------
;;;; Caching of viewpoints: storing & checking at retrieve time
;;;; (need to check if above works)
;;;;
;;;; filtering out general values (this isn't replacements; its
;;;; layers)
;;;;
;;;; make sure that put-local puts the annotation that have been
;;;; carefully get-global'ed (need to find KM function that will
;;;; serve as the basis for this)
;;;;
;;;; test get-global-with-annotation more thoroughly; in particular,
;;;; try it with embedded units and with replacement chains of length
;;;; greater than 1
;;;;
;;;; get-local needs to handle deeply embedded annotations
;;;;
;;;; we really need to ask Erik to rewrite all-explicit-slots with 
;;;; an optional argument &include-notation? that defaults to nil
;;;; (more generally, we need to provide a layering mechanism for
;;;; the KB)
;;;;
;;;; fix MAJOR kludge in add-val-with-annotation so it really adds
;;;; annotations to KB; make certain that it works when given implicit
;;;; units; also, make sure that deeply-nested annotations are also
;;;; added properly; (note that get-local-with-annotation doesn't currently
;;;; properly reformat deeply nested annotations, but it needs to be
;;;; fixed to do so)
;;;;
;;;; add efficiency hack to get-global: when the slot's ordinality is
;;;; atom, halt when encounter first value in upward search
;;;;
;;;; in construct-basic-dimension-vp-2 there is a call in the let binding-list
;;;; to get-explicit-slots (previously, it was get-legal-slots); what we
;;;; really want is something "between" these two: it should get all explicitly
;;;; filled slots on ancestry* of the unit, but does not get values on slots
;;;; that are genl-slots of the slots that appear on the unit
;;;;
;;;; the view retriever needs to set the *interactive?* global variable to nil
;;;; so that we achieve a menu-less call to add-val and put-local
;;;;
;;;; get cardinality and ordinality straightened out; currently
;;;; they are confused;
;;;; moreover, we need to distinguish two properties: ordered? and ordinality?
;;;;
;;;; fix get-contentful-slots so it doesn't compute progeny* of several
;;;; units EVERYTIME it's called
;;;;
;;;; consider using only local puts in viewpoint code so that inverses from 
;;;; domain KB units to viewpoints are not installed.  Remember that in the
;;;; code for building a composite view, add-val is called (instead of put-
;;;; local) because it has the side-effect of installing the intermediaries
;;;; used in paths).  
;;;;
;;;; Viewpoint code (and all other applications) should never use put-local.
;;;; Such code only gets what it deserves.... a dirty KB
;;;;
;;;; Constraints:
;;;;      - create slot: put-constraints
;;;;      - add following values to (New-slot-top put-constraints):
;;;;             (domain-constraint range-constraint ordinality-constraint)
;;;;
;;;; Fix up generalizations-ok sweep to also include a call like:
;;;;     (set-difference *kb-objects* (progeny* 'entity))
;;;; Otherwise, the sweep is incomplete.
;;;;
;;;; Fix up my-add-val (see file my-add-val-error)
;;;; This fn is called by the composite view function (and no others)


;;;; Sweeps
;;;; ------
;;;;
;;;; install inverses and replaces
;;;;
;;;; write function to be run periodically that searches
;;;; universally for generalizations of embedded units
;;;;
;;;; after run replaces sweep in KB, change construct-basic-dimension-vp-2
;;;; so that it does a get-global-with-annotation rather than the (kludged)
;;;; get-local-with-annotation
;;;;
;;;; sweep for all x such that there is a path through only genls back to x
;;;; (or a path through only specs back to x) (x includes embedded units also)



;;;;--------------------------------------------------------------------------
;;;;
;;;;	         	  VIEW RETRIEVER
;;;;			  --------------
;;;; 
;;;; given a specification for a viewpoint, return the name of the viewpoint
;;;; unit created (or retrieved) that satisfies the specification, or an 
;;;; error code (see below).
;;;;
;;;; The syntax for the specification depends on the type of viewpoint:
;;;; I.  Basic Dimension Viewpoint: 
;;;;              (<concept> dimension <dimension-types>)
;;;;  - concept may be either a top-level unit or an embedded unit
;;;;  - dimension-types may be one or more of spatial-structural, 
;;;;    temporal-structural etc.; see the unit basic-dimension-value 
;;;;    in the KB and examine its specializations to determine all 
;;;;    dimension-types
;;;;
;;;; II. As-Kind-Of Viewpoint:
;;;;             (<concept> as-kind-of <reference concept>)
;;;; Note that the specification might contain an auxillary basic-dimension
;;;; viewpoint specification in place of a <reference concept>,
;;;; i.e. a recursive call to the view retriever, such as: 
;;;;          (X as-kind-of (Y basic-dimension (spatial-structural)))
;;;;
;;;; III. Composite Viewpoint:
;;;;
;;;; IV. As-Having Viewpoint:
;;;;          not currently implemented.
;;;;
;;;; ERRORS: This function returns the following error codes under the
;;;; conditions specified below:
;;;;
;;;;   '(error invalid-view-spec)       
;;;;        The syntax of the viewpoint specification is wrong.
;;;;   '(error unknown-type-of-vp)
;;;;        The specification asked for an unknown type of viewpoint.
;;;;   '(error unknown-concept)
;;;;        The concept for which the viewpoint is requested is not in the KB.
;;;;   '(error unknown-type-of-basic-dimension)
;;;;        The basic-dimension requested is unknown.
;;;;   '(error ref-concept-not-genl-of-coi)
;;;;        In the specification for an as-kind-of viewpoint, 
;;;;        the reference concept is not an all-generalization of the
;;;;        concept of interest, and it must be.
;;;;   '(error ill-formed-AS-KIND-OF-viewpoint)
;;;;        The syntax of the specification for an as-kind-of viewpoint
;;;;        is wrong.


(defun retrieve-view (vspec)
  (reset);; make sure KM stack is clear
  (with-no-inverses
      (cond ((or (atom vspec)           
		 (< (length vspec) 2))  
	     (signal-vr-error 'invalid-view-spec))
	    ((eq (second vspec) 'dimension)
	     (construct-basic-dimension-vp vspec))
	    ((eq (second vspec) 'as-kind-of)
	     (construct-as-kind-of-vp vspec))
	   ;((eq (second vspec) 'as-having)
	   ; (construct-as-having-vp vspec))
	    ((eq (car vspec) 'composite)
	     (decompose-composite-vp-spec
	      (second vspec)
	      (third vspec)
	      (fourth vspec)))
	    (t (signal-vr-error 'unknown-type-of-vp)))))


;;;; decompose the specification for a composite viewpoint.  
;;;; Unlike other types of viewpoint specifications, these
;;;; can involve recursive calls to the view retriever, e.g.
;;;; the following structural-functional composite view of flower:
;;;;  (retrieve-view '(composite 
;;;;     (retrieve-view '(flower dimension (spatial-structural)))
;;;;     (retrieve-view '(angiosperm-sexual-reproduction dimension 
;;;;                      (event-structural)))
;;;;                     (actor-in)))
;;;;
(defun decompose-composite-vp-spec (concept-or-spec-1 
				    concept-or-spec-2 
				    correspondence-slots)
  (let ((concept1 (if (object? concept-or-spec-1)
		      concept-or-spec-1 
		      (retrieve-view concept-or-spec-1)))
	(concept2 (if (object? concept-or-spec-2)
		      concept-or-spec-2
		      (retrieve-view concept-or-spec-2))))
    (construct-composite-vp concept1 concept2 correspondence-slots)))



;;;;--------------------------------------------------------------------------
;;;;
;;;;
;;;;			  Basic Dimension Constructor
;;;;			  ---------------------------


;;;; construct-basic-dimension returns a viewpoint of a concept
;;;; created along a basic dimension
;;;;
;;;; viewpoint specification form:
;;;;      (<concept> dimension <dimension-type>)
;;;;
;;;;  - concept may be either a top-level unit or an embedded unit
;;;;  - dimension-type may be one of spatial-structural, temporal-structural
;;;;    etc.; see the unit basic-dimension-value in the KB and examine its
;;;;    specializations for all dimension-types
;;;;  - viewpoints created for some types of basic dimensions have 
;;;;    interconnecting relations.  For example, 
;;;;    basic dimensions have interconnecting relations. For example,
;;;;    the spatial-structural type includes relations such as parts,
;;;;    surrounds, encloses, etc; the black-box type has no relations.
;;;;    For find (or change) the interconnecting relations for a 
;;;;    particular type of basic-dimension, look in the KB at 
;;;;    (<type-of-basic-dimension> interconnection-dimension) or
;;;;    call the function: 
;;;;        (get-interconnect-slots '(<type-of-basic-dimension>)).
;;;;
;;;;  - an error is returned if the concept of interest is not in the KB.


(defun construct-basic-dimension-vp (vspec)
  (let* ((concept (first vspec))
	 (dimensions (third vspec))
	 (concept-in-kb? (object? concept)))
    (cond ((not (listp dimensions))
	   (signal-vr-error 'dimension-list-not-a-list))
	  ((error-in-vr? concept-in-kb?)
	   (signal-vr-error 'object-in-view-specification-not-in-kb))
	  (t (construct-basic-dimension-vp-2 concept dimensions)))))


;;; return a viewpoint constructed along basic-dimensions,
;;; a passed parameter. 
;;; First, get those legal slots of concept that pertain to basic-dimensions
;;; if there are any pertinent slots, create a viewpoint using them,
;;; otherwise return with an error.  A viewpoint is created by:
;;;      1.  initializing it with its type(s)
;;;      2.  installing inherited values on all pertinent slots
;;;      3.  installing facets on pertinent slots, if there are any.
;;;      4.  installing interconnections.  Note that this is a no-op
;;;          for viewpoints created along basic dimensions that do not
;;;          have any interconnections.

(defun construct-basic-dimension-vp-2 (concept basic-dimensions)
  (let* ((slots-on-concept (get-explicit-slots concept));need another fn here
	                                                ;see FIX IT list
	 (pertinent-slots (slots-in-basic-dimensions slots-on-concept
						     basic-dimensions)))
    (cond ((not (subsetp basic-dimensions
			 ;; JL changed following line 8-29-93 so will get
			 ;; active-functional-dimension
			 (progeny* 'basic-dimension-value)
			 :test #'equal))
	   (signal-vr-error 'unknown-type-of-basic-dimension))
	  ((null slots-on-concept)
	   (signal-vr-error 'no-slots-on-given-concept))
	  ((null pertinent-slots)
	   (signal-vr-error 'no-slots-in-basic-dimension-on-concept))
	  (t (let ((viewpoint (init-basic-dim-vp concept basic-dimensions)))
	       (dolist (slot pertinent-slots)
		 ;;see FIX IT line above
		 (let ((vals (get-local (list concept slot)))
		       ;(vals (get-local-with-annotation concept slot))
		       ) 
		   (when vals                                           
		     (dolist (val vals)
		       (add-val-local (list viewpoint slot) val)
		       (copy-values-only (list concept slot val)
					 (list viewpoint slot val)))
		     ;(put-local-with-annotation viewpoint slot vals)
		     ;   to return to old version, sub (put-local-with ...)
		     ;   for (copy-values-only ...)
		     )))
					;fix:(put-facets viewpoint
	                                ;                slot
	                                ;           (get-facets concept slot))
	       (add-interconnects viewpoint basic-dimensions pertinent-slots)
	       viewpoint)))))



;;; return the subset of candidate-slots that belong to basic-dimensions.
;;; This is done by checking the (possibly inherited) value for
;;; each slot's "slot-dimension" facet.
(defun slots-in-basic-dimensions (candidate-slots basic-dimensions)
  (let ((result nil))
    (dolist (slot candidate-slots)
      (if (intersection (get-global-without-annotation slot 'slot-dimension)
			basic-dimensions)
	  (push slot result)))
    result))


;;; create and return a new viewpoint, one taken of concept along
;;; basic-dimensions.
(defun init-basic-dim-vp (concept basic-dimensions)
  (let ((new-vp (make-new-view concept)))
    (put-local (extend-address new-vp 'basic-dimensions) basic-dimensions)
    new-vp))


;--------------------------------------------------------------------------
;;; original version

(defun make-new-view (concept)
  (let ((view (gentemp "VIEWPOINT-")))
    (put-local (list view 'instance-of) '(viewpoint))
    (add-val-local '(viewpoint instances) view)
    (put-local (list view 'viewpoint-of)
	       (list concept))
    (put-local (list view 'generalizations)  
	       (list concept))
    (put-local (list view 'author) '("view retriever"))
    view))

;;;
;;;
;;;			    View Shell Constructor Version
;;;                         ------------------------------


;;; Because the old view shell constructor was so slow, with almost
;;; half of the time consumed by the construction of the unit itself,
;;; we will now keep a collection of view-shells around and build
;;; a new batch of them on-the-fly when the collection runs dry
;;;
;;; Algorithm
;;; ---------
;;; shell <-- get-next-viewpoint-shell
;;; if (null shell) then
;;;    shell-list <-- make 50 viewpoint shells
;;;    shell <-- get-next-viewpoint-shell
;;;    remove shell from shell-list
;;; else remove shell from shell-list
;;; 
;;; Representation
;;; --------------
;;; Unit: VP-Shell (on the instances)
;;; Slot on Knight-Global-State: Create-VP-Shells-Dynamically-Batch
;;;         - defaults to TRUE
;;;
;;; Speed-Up
;;; --------
;;; timing statistics indicate that this new version takes 25%
;;; less time than the old version
;
;
;;; return an "empty" viewpoint of concept, one with a unique name,
;;; which is interned by gentemp into the current package.
;(defun make-new-view (concept)
;  (let ((view (get-vp-shell)))
;    (put-local (list view 'viewpoint-of)
;	       ;; changed by J.L. so works for embeded units 9-5-93
;	       (if (listp concept)
;		   (list concept)
;		   concept))
;    (put-local (list view 'generalizations)  
;	       (if (listp concept)
;		   (list concept)
;		   concept))
;    (put-local (list view 'author) '("view retriever"))
;    view))
;
;
;;; obtains new shell and creates 50 news ones if necessary
;(defun get-vp-shell ()
;
;  ;; create new shells if necessary
;  (when (null (get-local '(vp-shell instances)))
;    (cond ((create-vp-shells-batch?)
;
;	   ;; create a batch of shells
;	   (format t "Creating 50 new viewpoint shells.~%~%")
;	   (make-multiple-vp-shells 10))
;
;	  (t
;	   ;; create only one new shell (because batch not wanted)
;	   (make-vp-shell))))
;
;  ;; grab a shell
;  (grab-first-vp-shell))
;
;
;;; switches a shell from being a shell to being a viewpoint,
;;; and returns the shell
;;;
;;; assumes a shell exists
;(defun grab-first-vp-shell ()
;  (let ((new-shell (first (get-local '(vp-shell instances)))))
;    ;(format t "New-shell: ~a~%" new-shell)
;    (remove-val '(vp-shell instances)
;		new-shell)
;    new-shell))
;
;
;;; creates n new viewpoint shells
;(defun make-multiple-vp-shells (n)
;  (dotimes (i n)
;    (make-vp-shell)))
;
;
;;; creates a new viewpoint shell
;(defun make-vp-shell ()
;  (let ((shell (gentemp "VIEWPOINT-")))
;    (put-local (list shell 'instance-of) '(vp-shell viewpoint))
;    (add-val-local '(vp-shell instances) shell)
;    (add-val-local '(viewpoint instances)
;		   shell)
;    shell))
;
;
;(defun create-vp-shells-batch? ()
;  (equal (get-only-val '(knight-global-state create-vp-shells-batch?))
;	 'true))
;
;
;--------------------------------------------------------------------------


;;; for testing the make-new-view-function
;;;(defun test-maker ()
;;;  (time
;;;   (dotimes (i 10)
;;;     (test-make-new-view 'embryo-sac))))


;;; add interconnection slots to a viewpoint constructed along
;;; a basic dimension.
;;;
;;; parameters:
;;;    view: the viewpoint to which interconnections 
;;;          will be added (must be a basic-dimension viewpoint)
;;;    dimensions: a list of the types of basic dimensions for which
;;;                view was constructed
;;;    pertinent-slots: a list of slots on view that are candidates
;;;                     for interconnections
;;;
;;; Algorithm is as follows: Collect the set of values appearing
;;; locally on all pertinent-slots of view.  Examine the units
;;; representing each of these values.  If a unit contains relevant
;;; interconnection information, copy that information to the 
;;; viewpoint.  Interconnection information is considered "relevant"
;;; when it refers to another value appearing locally on the view.
;;; 
;;; Canonical Example:
;;;       
;;;                 KB before interconnections are added
;;; 
;;; viewpoint007                             Calyx
;;; ------------                             -----
;;; instance-of: viewpoint                   surrounded-by: corolla, stem
;;; viewpoint-of: flower                     attached-to: perianth
;;; basic-dimension: spatial-structural
;;; parts: calyx, corolla, perianth
;;; 
;;;                 KB after interconnections are added
;;; 
;;; viewpoint007
;;; ------------
;;;    ...
;;; parts: calyx, corolla, perianth
;;;   annotation on calyx:           <-- added because (calyx surrounded-by)
;;;      surrounded-by: corolla          contains corolla, and corolla is
;;;                                      a value (along with calyx) on the vp.
;;;                                      Note that stem is NOT included.
;;;      attached-to: perianth       <-- added for similar reasons.
;;; 

(defun add-interconnects (view dimensions pertinent-slots)
  (let ((values-on-view (get-values-on-selected-slots 
			 view pertinent-slots))
	(all-interconnect-slots (get-interconnect-slots dimensions)))
    (dolist (pertinent-slot pertinent-slots)
      (dolist (val (get-local (extend-address view pertinent-slot)))
	(dolist (interconnect-slot (intersection
				    (get-contentful-slots val)
				    all-interconnect-slots
				    :test #'equal))
	  ;; copy an interconnection:
	  ;;   - FROM the unit referred to in the viewpoint,
	  ;;     e.g. "calyx" in the example above, and an
	  ;;     interconnection slot, e.g. surrounded-by or attached-to
	  ;;     in the example above.
	  ;;   - TO the viewpoint unit as annotation, e.g. on the address
	  ;;     (viewpoint007 parts calyx) in the example above.
	  ;;   - UNLESS that interconnection refers to a concept that
	  ;;     that is not used in the viewpoint -- that is, it is not
	  ;;     "interconnecting."  In the example above, "stem" is omitted.
	  (copy-selected-values
	   val				; origin unit
	   interconnect-slot		; origin slot
	   (extend-address-indefinitely ; destination unit
	    view                        
	    pertinent-slot
	    val)
	   interconnect-slot		; destination slot
	   #'(lambda (value)		; value-pruning predicate
	       (not (member (first value) values-on-view
			    :test #'equal)))))))))


;;; copies those values on (origin-unit origin-slot) to
;;; (destination-unit destination-slot) that do not satisfy
;;; value-pruning-predicate
;;;
;;; note that the value-pruning-predicate is applied to the
;;; tuple (value annotation) - just as it is returned by
;;; get-local-with-annotation - and not merely the bare value
(defun copy-selected-values (origin-unit
			     origin-slot
			     destination-unit
			     destination-slot
			     value-pruning-predicate)
  (dolist (val (get-local-with-annotation origin-unit origin-slot))
    (unless (funcall value-pruning-predicate val)
      (add-val-with-annotation destination-unit destination-slot val))))


;;; follows the interconnection-dimension slot from each basic-dimension
;;; to one of: connections-spatial, connections-temporal,
;;; connections-subevents, or connections-specialization.  From there,
;;; it returns the fillers of dimension-of-slots of the
;;; interconnection-dimension and their progeny.


(defun get-interconnect-slots (basic-dimensions)
  (mapcan #'(lambda (dimension)
	      (let* ((inter-connect-dim
		      (get-only-val (list dimension
					  'interconnection-dimension)))
		     (inter-conn-slots			     
		      (get-local (list inter-connect-dim
				       'dimension-of-slots))))
		(copy-list 
		 (mapcan #'progeny*	      ;WARNING: destructive operation
			 inter-conn-slots)))) ;depends on progeny* making a
	  basic-dimensions))	              ;copy of the objects it returns

;; the old way - less efficient
;(defun get-interconnect-slots (basic-dimensions)
;  (reduce #'append
;	  (mapcar #'(lambda (dimension)
;		      (let* ((inter-connect-dim
;			      (get-only-val (list dimension
;						 'interconnection-dimension)))
;			     (inter-conn-slots			     
;			      (get-local (list inter-connect-dim
;					       'dimension-of-slots))))
;			(reduce #'append
;				(mapcar #'progeny*
;					inter-conn-slots))))
;		  basic-dimensions)))
;



;;;;--------------------------------------------------------------------------
;;;;
;;;;			    As-Kind-Of Constructor
;;;;			    ----------------------


;;;; construct an AS-KIND-OF viewpoint
;;;; 
;;;; decompose the viewpoint specification
;;;;
;;;; Note that it might contain an auxillary viewpoint specification in 
;;;; place of a reference concept, i.e. a recursive call to the
;;;; view retriever, such as: "X as-kind-of 'Y basic-dimension Structural'".
;;;; This recursive call is not handled here, instead its specification
;;;; is simply extracted and handled by a recursive call to retrieve-view.

(defun construct-as-kind-of-vp (vspec)
  (let* ((concept-of-interest (first vspec))
	 (reference-concept-or-spec (third vspec))
	 (concept-in-kb? (object? concept-of-interest)))
    (if (error-in-vr? concept-in-kb?)
	concept-in-kb?
	(if (not (member (get-ref-for-path reference-concept-or-spec)
			 (ancestry* concept-of-interest)
			 :test #'equal))

	    ;; reference-concept-or-spec may be a nested view specification
	    (let ((secondary-view (retrieve-view reference-concept-or-spec)))
	      (if (error-in-vr? secondary-view)

		  ;; either the reference-concept is an ill-formed
		  ;; view specification or a non-existant concept
		  (signal-vr-error (second secondary-view))
		  (construct-as-kind-of-vp-2 concept-of-interest
					     secondary-view)))

	    ;; reference-concept-or-spec is a reference-concept
	    (construct-as-kind-of-vp-2 concept-of-interest 
				       reference-concept-or-spec)))))


;;; coi = concept of interest
;;; reference = reference concept.  Note that reference might be
;;;             a viewpoint in the event that the VR was called with
;;;             a recursive as-kind-of specification.
;;; assumption: reference may be a viewpoint constructed along a basic
;;; dimension or an as-kind-of viewpoint, but not a composite viewpoint.
;;;
;;; old algorithm: used only slots whose domain was the reference
;;; problem: produced *very* few values on viewpoint
;;; modification: allow all slots that appear on reference, whether
;;;               they originate there or not
;;; result: filters out slots that are inherited from another ancestor
;;;         in the taxonomy other than via reference
;;;
;;; another modification: using ref-for-path in the case of nested
;;;                       view retrieval calls results in a too
;;;                       inclusive set of slots; it includes slots
;;;                       that the first (innermost) view tried to
;;;                       filter out; instead, maybe just use reference
(defun construct-as-kind-of-vp-2 (coi reference)
  ;; in the event that reference is a viewpoint, ref-for-path is the concept
  ;; of which that viewpoint is taken; otherwise, ref-for-path is reference.
  ;; This function finds the path (through taxonomic slots) that connects 
  ;; coi to ref-for-path.  Then it copies slots/values from coi to a new
  ;; as-kind-of viewpoint whenever the slot is in the domain-of ref-for-path
  ;; OR the value replaces a filler on ref-for-path.
  (let* ((ref-for-path (get-ref-for-path reference)) 
	 (connecting-path (find-ref-conc-connecting-path coi ref-for-path)))
    (if connecting-path
	(let (;(domain-of-slots (get-local (list ref-for-path 'domain-of)))
	      (filled-ref-slots (get-explicit-slots  reference))
				                     ;ref-for-path
	      (filled-coi-slots (get-explicit-slots coi)) 
	      (new-viewpoint (init-ako-vp coi reference)))
	  ;(format t "~%filled-ref-slots: ~a~%~%"
	  ;	  filled-ref-slots)
	  ;(format t "~%filled-coi-slots: ~a~%~%"
	  ;	  filled-coi-slots)
	  (dolist (coi-slot filled-coi-slots)
	    (dolist (coi-value (get-local-with-annotation coi coi-slot))
	      (if (or (ultimately-replaces                           
		       coi                                           
		       coi-slot                                      
       		       (bareify coi-value)                           
		       reference    ; note: this is reference, not
		                    ; ref-for-path,
                                    ; because, if reference is a viewpoint, it
		                    ; resulted from useful filtering
		                    ; of the facts
		                    ; on ref-for-path
		       connecting-path)
		      (specializes-one-of? coi-slot filled-ref-slots))
					     ;domain-of-slots
	       (add-val-with-annotation new-viewpoint             
					coi-slot
					coi-value))))
	  new-viewpoint)
	(signal-vr-error 'ref-concept-not-genl-of-coi))))


;;; if ref is not a viewpoint, return ref
;;; if ref is a viewpoint, trace viewpoint-of links to non-viewpoint
(defun get-ref-for-path (ref)
  (if (member 'viewpoint (get-local (extend-address ref 'instance-of)))
      (get-ref-for-path (get-only-val (extend-address ref 'viewpoint-of)))
      ref))



;;; FIX: Consider adding the inverse for the put-local below when
;;       reference is a viewpoint (but not otherwise)
;;; create and return the "skeleton" of a new as-kind-of viewpoint
(defun init-ako-vp (coi reference)
   (let ((new-vp (make-new-view coi)))
     (put-local (extend-address new-vp 'as-kind-of) (listify reference))
     ;; copy the basic dimensions if the reference concept is
     ;; a viewpoint to the new viewpoint
     (copy-all-values (extend-address reference 'basic-dimensions)
		      (list new-vp 'basic-dimensions))
     new-vp))


;;; Parameters:
;;;    required:
;;;       origin-unit: a unit in the KB from which the search will
;;;                    begin
;;;       slot: a slot located on origin-unit
;;;       value: a value on (origin-unit slot)
;;;       target-unit: a unit that is the target of the search,
;;;                    i.e., where the string of replaces terminates
;;;                    when this function succeeds
;;;    optional:
;;;       connecting-path: a path connecting origin-unit to 
;;;                        target-unit through taxonomic slots.
;;;
;;; return T iff (origin-unit slot value) is connected through a 
;;; series of replaces slots to a filler on target-unit.
;;; If a connecting-path is given, then the series of replaces slots
;;; may not deviate from units on that path.
(defun ultimately-replaces (origin-unit slot value 
					target-unit 
					&optional connecting-path)
  (let* ((success-function (make-success-function-ako-vr target-unit))
	 (pruning-function (if connecting-path
			       (make-pruning-function-for-ako-vr 
				connecting-path)))
	 (kb-search-results
	  (kb-search (list (extend-address-indefinitely origin-unit 
							slot value))
		     (list 'replaces)
		     :terminate-with-success-criteria success-function
		     :pruning-function pruning-function
		     :control-strategy 'breadth-first
		     :collect-path? nil
		     :loop-elimination? t)))
    (equal (first kb-search-results) 'success)))


;;; this function when called generates a function that
;;; takes 1 parameter, which is the unit being considered
;;; by the search function, i.e., current
;;;
;;; the reference concept is built into this generated function
(defun make-success-function-ako-vr (reference)
  #'(lambda (unit)
      (equal (start-of-path unit)
	     reference)))


;;; this function when called generates a function that
;;; takes 1 parameter, which is the unit being considered
;;; by the search function, i.e., current
;;;
;;; the path is built into this generated function
(defun make-pruning-function-for-ako-vr (path)
  #'(lambda (unit)
      (not (member (start-of-path unit) path
		   :test #'equal))))


;;; search for a path from the concept of interest (coi) to the
;;; reference concept through the slot generalizations and its
;;; progeny.
;;; returns list with elements (coi u1 u2 ... u3 ref-conc)
;;; if no path exists returns nil
(defun find-ref-conc-connecting-path (coi reference)
  (let ((kb-search-result (kb-search 
			   (list coi)
			   '(generalizations)
			   :search-progeny? t
			   :goal-units (list reference)
			   :collect-path? t
			   :control-strategy 'breadth-first
			   :loop-elimination? t)))
    (if (equal (car kb-search-result) 'success)
	(third kb-search-result))))


;;;;-------------------------------------------------------------------------- 
;;;;
;;;;			  Composite View Constructor
;;;;			  --------------------------

;;; given:
;;;    concept1: a concept in the KB
;;;    concept2: a concept in the KB
;;;    correspondence-slots: slots to connect concept1 to concept2
;;;
;;; method:
;;;    for each "contentful" slot s on concept1
;;;       for each value v on s
;;;          find connection from v to a value on a "contentful" slot 
;;;               on concept2
;;;          record this connection in the new viewpoint
;;;
;;; returns:
;;;         New-Correspondence-VP
;;;         ---------------------
;;;         viewpoint-of: concept1
;;;                       --------
;;;                       (optional: viewpoint-of: <concept1 viewpoint-of>)
;;;         correspondence-type: correspondence-slots
;;;         correspondence-with: concept2 
;;;                              --------
;;;                              (optional: viewpoint-of: 
;;;                                              <concept2 viewpoint-of>)
;;;         slot-i

;	     (slots-to-search (remove-duplicates 
;			       (reduce #'append
;				       (mapcar #'(lambda (slot)
;						   (progeny* slot))
;					       correspondence-slots))
;			       :test #'equal)))

(defun construct-composite-vp (concept1 concept2 correspondence-slots)
  (if (not (legal-correspondence-slots correspondence-slots))
      (signal-vr-error 'bad-correspondence-slot-to-composite-viewpoint-request)

      (let* ((new-view (init-composite-vp 
			concept1 concept2 correspondence-slots))
	     (slots-on-concept1 (get-contentful-slots concept1))
	     (slots-on-concept2 (get-contentful-slots concept2))
	     (vals-on-concept2 (get-values-on-selected-slots
				concept2 slots-on-concept2))
	     ; WARNING: the following destructive operation depends on
	     ;          the fact that progeny* makes a copy of the
	     ;          objects it returns
	     (slots-to-search (remove-duplicates 
			       (mapcan #'progeny* correspondence-slots)
			       :test #'equal)))


	(dolist (slot-on-concept1 slots-on-concept1)
	  (let ((vals-on-concept1 (get-local+ concept1 slot-on-concept1)))
	    (dolist (val-on-concept1 vals-on-concept1)
	      (dolist (val-on-concept2 vals-on-concept2)

		(let ((search-result
		       (kb-search (list val-on-concept1)
				  slots-to-search
				  :search-progeny? nil
				  :goal-units (list val-on-concept2)
				  :collect-path? t)))

		  (when (equal (first search-result) 'success)
		    (let* ((search-path (third search-result))
			   (slot-that-begins-path (second search-path))
			   (rest-of-path (cddr search-path)))
		      (add-val-local
		       (extend-address-indefinitely new-view
						    slot-on-concept1
						    val-on-concept1
						    slot-that-begins-path)
		       (if (equal (length rest-of-path) 1)
			   (first rest-of-path)
			   rest-of-path)))))))))
	new-view)))


;;; returns t iff correspondence-slots is non-null, it is a list, and
;;; each member is a slot
(defun legal-correspondence-slots (correspondence-slots)
  (and correspondence-slots
       (listp correspondence-slots)
       (every #'slot? correspondence-slots)))


;;; create the "shell" for a composite viewpoint.  It's form is:
;;;
;;;      new-correspondence-vp
;;;      ---------------------
;;;        viewpoint-of: concept1
;;;                      --------
;;;                      (optional: viewpoint-of: <concept1 viewpoint-of>)
;;;        correspondence-type: correspondence-slots
;;;        correspondence-with: concept2 
;;;                             ----------
;;;                             (optional: viewpoint-of: <concept2 viewpoint-of>)
;;;
;;; the optional information above is added when concept1 and/or concept2
;;; are viewpoints
;;;
(defun init-composite-vp (concept1 concept2 correspondence-slots)
  (let ((new-vp (make-new-view concept1)))
    (put-local (extend-address new-vp 'viewpoint-of) concept1)
    (put-local (extend-address new-vp 'correspondence-type) correspondence-slots)
    (put-local (extend-address new-vp 'correspondence-with) concept2)
    (when (viewpoint? concept1)
      (put-local (extend-address-indefinitely new-vp 'viewpoint-of
					      concept1 'viewpoint-of)
		 (get-local+ concept1 'viewpoint-of)))
    (when (viewpoint? concept2)
      (put-local (extend-address-indefinitely new-vp 'correspondence-with
					      concept2 'viewpoint-of)
		 (get-local+ concept2 'viewpoint-of)))
    new-vp))


;;; returns t iff unit is a viewpoint
(defun viewpoint? (unit)
  (a-kind-of? unit 'viewpoint))


;;;;---------------------------------------------------------------------------- 
;;;;
;;;;			  Signal View-Retriever Error
;;;;			  ---------------------------


;;; signal a view-retriever error if concept is not in the KB.
(defun check-concept (concept)
  (when (not (object? concept))
    (signal-vr-error 'unknown-concept)))


;;; return an error from the view retriever
(defun signal-vr-error (errortype)
  (list 'error errortype))  	


(defun error-in-vr? (possibly-an-error-code)
  (and (listp possibly-an-error-code)
       (equal (car possibly-an-error-code) 'error)))



;;;;---------------------------------------------------------------------------- 
;;;;
;;;;			      Test Call Functions
;;;;			      -------------------

(defun is-calyx (value)
  (equal (first value) 'calyx))
