; -*- Mode: Lisp; Syntax: Common-Lisp; Package: KM; Base: 10 -*-
(in-package 'km)


;;;				   KB Search
;;;				   ---------

;;; generalized KB search
;;; searches through KB using criteria provided by user
;;;
;;; given: - initial-unit-list: list of units that serve as initial
;;;                             source(s) in the search
;;;
;;;        - slots            : list of slots through which search
;;;                             may pass 
;;;
;;;        - search-progeny?  : if given as t, then searches both
;;;                             slots and their progeny
;;;                             (default: nil)
;;;
;;;        - goal-units       : list of units that serve as final
;;;                             success states in the search
;;;
;;;        - terminate-with-success-criteria
;;;                           : predicate on units that returns t
;;;                             if the unit to which it is applied
;;;                             is to be considered a goal
;;;
;;;        - terminate-with-failure-criteria
;;;                           : predicate on units that returns t
;;;                             if the unit to which it is applied
;;;                             is to be considered a failure
;;;
;;;        - pruning-function : predicate on units that returns t
;;;                             if the unit and all of its successors
;;;                             should be excluded from the search
;;;
;;;        - control-strategy : depth-first or breadth-first
;;;                             (default: depth-first)
;;;
;;;        - collect-path?    : if given as t, then collects and
;;;                             returns successful search path
;;;                             (default: nil)
;;;
;;;        - loop-elimination?: if given as t, then checks for 
;;;                             and eliminates loops during search
;;;                             (default: t)
;;;
;;; use of arguments:
;;;      - user must provide initial-unit-list and slots
;;;      - user must provide at least one of
;;;        terminate-with-success-criteria
;;;        or goal-units; he may provide both
;;;      - all other arguments are optional
;;;
;;; returns one of:
;;;      - (fail no-goal-found)
;;;      - (fail terminate-with-failure <failure-unit>)
;;;      - (success <goal-unit> <path>)
;;;
;;; if <failure-unit> is non-null, its value is the unit
;;; at which the terminate-with-failure-criteria returned t
;;;
;;; if <goal-unit> is non-null, its value is either the unit
;;; at which the terminate-with-success-criteria returned t
;;; or is a member of goal-units
;;;
;;; Notes on removing duplicates from open-list during search:
;;;    1) in the case when a path will not be returned from the search,
;;;       duplicates are removed.  Reason: the code doesn't, and can't,
;;;       exploit information about the path to each node on the
;;;       open-list; therefore, different paths to the same node
;;;       are equivalent.
;;;    2) in the case when a path will be returned from the search:
;;;       a) the current implementation of this search program does
;;;          not permit the user to apply predicates on paths (e.g. to 
;;;          measure the length of a path, or to order nodes on open, 
;;;          best-first search, based on characteristics of their paths).
;;;          These are options that might be added later.
;;;       b) so, duplicates are removed from the open-list in this
;;;          implementation.  However, we've tried to document the
;;;          parts of the code that might be modified later.
;;;       c) interestingly, the method for removing duplicates does not
;;;          depend on control strategy.  For both breadth-first and
;;;          depth-first search, duplicates are removed from the 
;;;          end of the open-list.  For depth-first search, this insures
;;;          that newly placed nodes on open overshadow older copies;
;;;          for breadth-first search, this insures that older copies
;;;          overshadow newly placed ones.

;;; Organization of File
;;; --------------------
;;; 1. General KB-Search function
;;; 2. Path-Finding Search functions
;;; 3. Non-Path-Finding Search functions
;;; 4. Auxiliary functions used by both (2) and (3)
;;; 5. Test function calls


;;; this top-level function simply collects all the slots that will
;;; searched (by collecting the progeny of slots, a passed parameter,
;;; when search-progeny?=T) and calling one of two search programs,
;;; based on whether paths are being collected.
(defun kb-search (initial-unit-list slots
				    &key search-progeny?
				    goal-units
				    terminate-with-success-criteria
				    terminate-with-failure-criteria
				    pruning-function
				    (control-strategy 'depth-first)
				    collect-path?
				    (loop-elimination? t))
  (if (not (or goal-units terminate-with-success-criteria))
      '(fail no-goal-requested)
      (let ((slots-to-search (if search-progeny?
				 (remove-duplicates 
				  (reduce #'append
					  (mapcar #'(lambda (slot)
						      (progeny* slot))
						  slots))
				  :test #'equal)
				 slots)))
	(if collect-path?
	    (kb-search-for-path initial-unit-list
				slots-to-search
				goal-units
				terminate-with-success-criteria
				terminate-with-failure-criteria
				pruning-function
				control-strategy
				loop-elimination?)
	    (kb-search-for-goal initial-unit-list
				slots-to-search
				goal-units
				terminate-with-success-criteria
				terminate-with-failure-criteria
				pruning-function
				control-strategy
				loop-elimination?)))))


;;;-------------------------------------------------------------------
;;; Path-finding functions (return a solution path)
;;;
;;; a solution path is of the form:
;;;      (<initial-unit> s1 u1 s2 u2 ... sn <goal-unit>)
;;; where <initial-unit> occurs in the initial-unit-list
;;; and s1, s2,..., sn are slots that occur in slots-to-search
;;;
;;; the variable Open is a list of pairs of the form
;;;      (<path from initial-unit to unit-i> <unit-i>)
;;; where initial-unit is in the initial-unit-list and
;;; unit-i is reachable through slots-to-search


(defun kb-search-for-path (initial-unit-list
			   slots-to-search
		           goal-units
			   terminate-with-success-criteria
			   terminate-with-failure-criteria
			   pruning-function
			   control-strategy
			   loop-elimination?)
  (do ((open (initialize-open initial-unit-list))
       (closed nil)
       (terminated? nil))
      (terminated?
       terminated?)
    (if (null open)
	(setf terminated? '(fail no-goal-found))
	(let ((current-unit (second (first open)))
	      (current-path (first (first open))))
	  (setf terminated?
		(check-termination-with-path
		 current-unit
		 current-path
		 goal-units
		 terminate-with-success-criteria
		 terminate-with-failure-criteria))
;	  (format t "Current-unit: ~a~%" current-unit)
;	  (format t "Open:")
;	  (pprint open)
;	  (format t "~%")
	  ;(format t "Terminated?: ~a~%~%" terminated?)
	  ;(print-abbrev-open open)
	  (when (not terminated?)
	    (when loop-elimination?
	      (setf closed (cons current-unit closed)))
	      ;(format t "Closed:")
	      ;(pprint closed)
	      ;(format t "~%")
	    (setf open
		  (update-open-with-path (rest open)
					 current-unit
					 current-path
					 closed
					 loop-elimination?
					 slots-to-search
					 control-strategy
					 pruning-function)))))))


;;; initializes the variable Open to be a list of pairs of the
;;; form ((initial-unit) initial-unit)
(defun initialize-open (initial-unit-list)
  (mapcar #'(lambda (unit)            
	      (list (list unit) unit))
	  initial-unit-list))


;(defun print-abbrev-open (open)
;  (format t "Open:")
;  (let ((open-abbrev 
;	 (mapcar #'(lambda (element)
;		     (second element))
;		 open)))
;    (pprint open-abbrev))
;  (format t "~%"))




;;; returns the open list updated by the successors of the current unit,
;;; with duplicates removed
(defun update-open-with-path (open current-unit current-path
				   closed loop-elimination? slots
				   control-strategy pruning-function)
  (let ((additions-to-open
	 (generate-path-successors-of-unit
	  current-unit
	  current-path
	  slots
	  loop-elimination?
	  closed
	  pruning-function)))
    (remove-duplicates
     (case control-strategy
       (depth-first
	(append additions-to-open
		open))
       (breadth-first
	(append open
		additions-to-open)))
     :key #'(lambda (element-on-open)
	      (second element-on-open))
     :test #'equal
     :from-end t)))


;;; finds all successors of current that are connected via slots
;;; if no loop-elimination, returns all successors; otherwise
;;; returns successors that are not on closed and that are not pruned
;;; each successor is a pair of the form (<path> <successor-unit>)
;;; the last step in the process is to append the partial-paths to
;;;    each successor
(defun generate-path-successors-of-unit (current-unit current-path
					 slots loop-elimination?
					 closed pruning-function)
  (let* ((successors
	  (generate-unpruned-unit-successors-with-slots current-unit
							slots))
	 (non-dupe-successors
	  (if loop-elimination?
	      (set-difference successors closed
			      :test #'(lambda (successor-item closed-item)
					(equal (second successor-item)
					       closed-item)))
	      successors))
	 (non-pruned-successors
	  (if pruning-function
	      (apply-pruning-function-with-slots pruning-function
						 non-dupe-successors)
	      non-dupe-successors)))
    (add-path non-pruned-successors current-path)))


;;; CONSIDER THIS EFFICIENCY HACK:
;;;    REDUCE THE LIST OF SLOTS BY INTERSECTING IT WITH THE EXPLICIT
;;;    SLOTS ON CURRENT-UNIT, THEN GET-LOCAL'ing ONLY THOSE

;;; finds all successors of current that are connected via slots
;;; current is a unit
;;; returns list of the form:
;;;        ( ... (slot successor) ...)
(defun generate-unpruned-unit-successors-with-slots (current-unit slots)
  (reduce #'append
	  (mapcar #'(lambda (slot)
		      (mapcar #'(lambda (value)
				       (list slot value))
			      (get-local (extend-address current-unit
							 slot))))
		  slots)))


;;; returns subset of candidates that do not satisfy the pruning function
;;; candidates is a list of pairs of the form (slot unit)
(defun apply-pruning-function-with-slots (pruning-function candidates)
  (reduce #'append
	  (mapcar #'(lambda (candidate)
		      (if (funcall pruning-function (second candidate))
			  nil
			  (list candidate)))
		  candidates)))


;;; extends the path of each successor by adding current-path
;;; to the front
;;;
;;; example:
;;;     successors: ((x y) (z w))
;;;     current-path: (a b c)
;;;     result: (((a b c x y) y) ((a b c z w) w))
(defun add-path (successors current-path)
  (mapcar #'(lambda (successor)
	      (list (append current-path successor)
		    (second successor)))
	  successors))


(defun check-termination-with-path (current-unit current-path
				    goal-units
				    terminate-with-success-criteria
				    terminate-with-failure-criteria)
  (cond ((member current-unit goal-units :test #'equal)
	 (list 'success current-unit current-path))
	((and terminate-with-success-criteria
	      (funcall terminate-with-success-criteria current-unit))
	 (list 'success current-unit current-path))
	((and terminate-with-failure-criteria
	      (funcall terminate-with-failure-criteria current-unit))
	 (list 'fail current-unit))
	(t nil)))


;;;-------------------------------------------------------------------
;;; Goal-searching functions (does not return a solution path)

(defun kb-search-for-goal (initial-unit-list
			   slots
		           goal-units
			   terminate-with-success-criteria
			   terminate-with-failure-criteria
			   pruning-function
			   control-strategy
			   loop-elimination?)
  (do ((open initial-unit-list)
       (closed nil)
       (terminated? nil))
      (terminated?
       terminated?)
    (if (null open)
	(setf terminated? '(fail no-goal-found))
	(let ((current (first open)))
	  (setf terminated? (check-termination current
					       goal-units
					       terminate-with-success-criteria
					       terminate-with-failure-criteria))
	  ;(format t "Current: ~a~%" current)
;	  (format t "Open:")
;	  (pprint open)
;	  (format t "~%")
	  ;(format t "Terminated?: ~a~%~%" terminated?)
	  (when (not terminated?)
	    (when loop-elimination?
	      (setf closed (cons current closed)))
	                                ;(format t "Closed:")
					;(pprint closed)
					;(format t "~%")
	    (setf open (update-open (rest open) current closed
				    loop-elimination?
				    slots
				    control-strategy
				    pruning-function)))))))


;;; returns the open list updated by the successors of the current unit,
;;; with duplicates removed
(defun update-open (open current closed loop-elimination?
                    slots control-strategy pruning-function)
  (let ((additions-to-open
	 (generate-successors-of-unit current slots
				      loop-elimination?
				      closed
				      pruning-function)))
    (remove-duplicates
     (case control-strategy
       (depth-first
	(append additions-to-open
		open))
       (breadth-first
	(append open
		additions-to-open)))
     :test #'equal
     :from-end t)))



;;; finds all successors of current that are connected via slots
;;; if no loop-elimination, returns all successors; otherwise
;;; returns successors that are not on closed and that are not pruned
(defun generate-successors-of-unit (current slots loop-elimination?
					    closed pruning-function)
  (let* ((successors (generate-unpruned-unit-successors current
							slots))
	 (non-dupe-successors (if loop-elimination?
				  (set-difference successors closed
						  :test #'equal)
				  successors)))
    (if pruning-function
	(apply-pruning-function pruning-function non-dupe-successors)
	non-dupe-successors)))
    


;;; finds all successors of current that are connected via slots
(defun generate-unpruned-unit-successors (current slots)
  (reduce #'append
	  (mapcar #'(lambda (slot)
		      (get-local (extend-address current slot)))
		  slots)))


;;; returns subset of candidates that do not satisfy the pruning function
(defun apply-pruning-function (pruning-function candidates)
  (reduce #'append
	  (mapcar #'(lambda (candidate)
		      (if (funcall pruning-function candidate)
			  nil
			  (list candidate)))
		  candidates)))


(defun check-termination (current goal-units
				  terminate-with-success-criteria
				  terminate-with-failure-criteria)
  (cond ((member current goal-units :test #'equal)
	 (list 'success current nil))
	((and terminate-with-success-criteria
	      (funcall terminate-with-success-criteria current))
	 (list 'success current nil))
	((and terminate-with-failure-criteria
	      (funcall terminate-with-failure-criteria current))
	 (list 'fail current))
	(t nil)))


;;;-----------------------------------------------------------------------
;;; sample test functions

;;; prunes all flowers and their offspring
;;; returns t iff unit satisfies test
(defun flower-picker (unit)
  (if (member unit (progeny* 'flower)
	      :test #'equal)
      t))


;;; termination test functions

(defun is-sepal? (unit)
  (equal unit 'sepal))

(defun is-embryo-sac-formation? (unit)
  (equal unit 'embryo-sac-formation))

(defun is-corolla? (unit)
  (equal unit 'corolla))

(defun is-perianth? (unit)
  (equal unit 'perianth))

;;; sample kb-search test call

(defun test-kb-search ()
    (kb-search '(flower) '(actor-in occurs-after temporal-ordering) 
	       :goal-units '(embryo-sac-formation)
	       :search-progeny? t
	       :loop-elimination? t 
               :collect-path? t
	       :control-strategy 'breadth-first))
