;;; -*- Mode:Lisp; Package:User; Base:10 -*-


;;;;		 The Auxiliary Content Determination Functions
;;;;		 ---------------------------------------------


;;;; These are the functions used by the EDPs for content determination.
;;;;
;;;;
;;;; What's Returned
;;;; ---------------
;;;; Either a unit or a list is returned; if the content is to be
;;;; used by another function, only a list will be returned; on
;;;; the other hand, if the content is to be part of an explantion
;;;; plan, then a viewpoint unit will be returned (just like the
;;;; the View Retriever does)
;;;;
;;;;
;;;; Error Handling
;;;; --------------
;;;; They include methods for reporting errors encountered during
;;;; execution.  In general, during content determination, if an
;;;; error is found, the function returns a pair of the form:
;;;;          (error <error-type>)
;;;; Typically, the Applier will omit the intended content.
;;;;
;;;;
;;;; User Modeling Issues
;;;; --------------------
;;;; Some functions exploit the user model.  When the user modeling
;;;; facility is off, these functions behave differently.  They
;;;; take the approach of doing whatever is the ``right thing''
;;;; without regard to what might have been in the user model, e.g.,
;;;; when finding a reference concept, the function merely finds
;;;; an immediate genl of the given concept, rather than the most
;;;; specific concept that was familiar to the user.  (See the header
;;;; of the User Model functions file for additional information.)

;;;;-------------------------------------------------------------------
;;;;		     Top-Level Content Determination Calls
;;;;-------------------------------------------------------------------
;;;;
;;;; Function                           Arguments
;;;; --------                           ---------
;;;;
;;;; connect-to-core                    (origin)
;;;;
;;;; connect-to-core-rev                (origin)
;;;;
;;;; find-partonomic-connection         (object)
;;;;
;;;; make-as-kind-of-perceptual-view    (concept ref-conc)
;;;;
;;;; make-auxiliary-process-view        (process view-type)
;;;;
;;;; make-bare-ako-view                 (concept ref-conc)
;;;;
;;;; make-black-box-view                (process reference-process)
;;;;
;;;; make-functional-view               (actor-slot black-box-view)
;;;;
;;;; make-subevent-list                 (process)
;;;;
;;;; make-sub-structural-view           (object part-slot)
;;;;
;;;; make-temporal-step-of-view         (process)
;;;;
;;;;-------------------------------------------------------------------


(in-package 'km)


;-----------------------------------------------------------------------
;			   Concept-Has-Value-On-Slot
;-----------------------------------------------------------------------

;;; arguments: - Concept
;;;            - Slot
;;;            - Value
;;;
;;; returns non-null if Value appears locally on Concept.Slot

(defun concept-has-value-on-slot (concept slot value)
  (member value (get-local (list concept slot))
	  :test #'equal))

			   
;-----------------------------------------------------------------------
;				Collect-Actors
;-----------------------------------------------------------------------

;;; arguments: Content Node with Black-box-view
;;;
;;; searches through the Black box view to find the actor slots
;;; 
;;; seaches through the values on these slots to return the actors
;;; 
;;; returns a list of actors
;;;
;;; errors:
;;;             - condition: no content node exists
;;;             - returns: (error no-content-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;                          with black-box view
;;;             - returns: (error error-in-node-with-black-box-view)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint-in-content-node)
;;;
;;;		- condition: viewpoint exists, but no actors
;;;		- returns: (error no-actor-slots-on-viewpoint)

(defun collect-actors (content-node)
  (cond ((null content-node)
	 '(error no-content-node))
	((content-node-has-error? content-node)
	 '(error error-in-node-with-black-box-view))
	(t (let ((viewpoint
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null viewpoint)
		 '(error no-viewpoint-in-content-node)
		 (let ((actor-slots (collect-actor-slots viewpoint)))
		   (if (null actor-slots)
		       '(error no-actor-slots-on-viewpoint)
		       (collect-all-values-on-slots viewpoint
						    actor-slots))))))))


(defun collect-actor-slots (viewpoint)
  (let ((slots-with-values (all-explicit-slots viewpoint)))
    (remove-if-not #'is-actor-slot-p
		   slots-with-values)))


(defun is-actor-slot-p (slot)
  (member 'actors (ancestry* slot)))


;;; returns list of all values that appear on some slot in
;;; slot-list on frame
;;;
;;; Important note: mapcan is *not* used below and should
;;;                 not be used anywhere in the code because
;;;                 it nconc's some structures in the KB,
;;;                 sometimes creating circular structures
;;;                 that cause KM to go into infinite recursion

(defun collect-all-values-on-slots (frame slot-list)
  (remove-duplicates
   (reduce #'append
	   (mapcar #'(lambda (slot) (get-local (extend-address frame slot)))
		   slot-list))))


;-----------------------------------------------------------------------
;			 Collect-Actors-From-Viewpoint
;-----------------------------------------------------------------------

;;; arguments: an Output-Actor Viewpoint of a Process
;;;
;;; searches through the Black box view to find the actor slots
;;; 
;;; seaches through the values on these slots to return the actors
;;; 
;;; returns a list of actors
;;;
;;; like function collect-actors, but a viewpoint is given directly
;;;
;;; errors:
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint-in-content-node)
;;;
;;;		- condition: viewpoint exists, but no actors
;;;		- returns: (error no-actor-slots-on-viewpoint)

(defun collect-actors-from-viewpoint (viewpoint)
  (if (not (viewpoint? viewpoint))
      '(error no-viewpoint-in-content-node)
      (let ((actor-slots (collect-actor-slots viewpoint)))
	(if (null actor-slots)
	    '(error no-actor-slots-on-viewpoint)
	    (collect-all-values-on-slots viewpoint
					 actor-slots)))))


;-----------------------------------------------------------------------
;			 Collect-Functional-Processes
;-----------------------------------------------------------------------

;;; arguments: Active-Functional-View
;;;
;;; goes through Active-Functional-View to find processes which are
;;; values of actor-in slots
;;;
;;; returns (a list of a) list of processes
;;;
;;; errors:
;;;		- condition: no active-functional-view exist
;;;		- returns: (error no-view)
;;;
;;;		- condition: viewpoint exists, but no processes
;;;		- returns: (error no-processes)

(defun collect-functional-processes (active-functional-view)
  (if (null active-functional-view)
      '(error no-view)
      (let* ((slot-list (all-explicit-slots active-functional-view))
	     (actor-in-slots (reduce
			      #'append
			      (mapcar #'(lambda (slot)
					  (if (member 'actor-in
						      (ancestry* slot))
					      (list slot)))
				      slot-list))))
	(get-processes-on-slots active-functional-view
				actor-in-slots))))


;-----------------------------------------------------------------------
;		  Collect-Funct-Processes-From-Content-Nodes
;-----------------------------------------------------------------------

;;; arguments: Object-Signif-Topic-Node
;;;
;;; goes through children of Object-Significance-Topic-Node to examine
;;; the kb-subgraph associated with each one, and then collects the
;;; the process that is the function of each one
;;;
;;; returns list of processes
;;;
;;; errors:
;;;		- condition: no Object-Significance-Topic-Node exists
;;;		- returns: (error no-object-signif-topic-node)

(defun collect-funct-processes-from-content-nodes (object-signif-topic-node)
  (if object-signif-topic-node
      (let ((content-nodes (get-local (list object-signif-topic-node
					    'child-nodes-ordered))))
	(remove-duplicates
	 (reduce #'append
		 (mapcar #'(lambda (content-node)
			     (let ((viewpoint
				    (get-only-val (list content-node
							'kb-subgraph))))
			       (if viewpoint
				   (get-local (list viewpoint
						    'viewpoint-of)))))
			      content-nodes))))))

	 
;-----------------------------------------------------------------------
;			 Collect-Modulatory-Processes
;-----------------------------------------------------------------------

;;; arguments: Content-Node with a Causal-Agent-Viewpoint,
;;;            a Causal-Facilitating-Viewpoint, or
;;;            a Causal-Inhibiting-Viewpoint
;;;
;;; can test with:
;;;                Enables: <No non-embedded processes>
;;;                Facilitates: <None>
;;;                Inhibits: <None>
;;;                Causes: Guard-Cell-Collapse, Plant-Water-Loss
;;;                Caused-by: Plant-water-stress, Stoma-closing-sequence
;;;
;;; returns list of processes
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint)
;;;
;;;		- condition: viewpoint exists, but error
;;;		- returns: (error problem-viewpoint)
;;;
;;;		- condition: viewpoint exists, but no processes
;;;		- returns: (error no-processes)

(defun collect-modulatory-processes (content-node)
  (if (null content-node)
      '(error no-content-node)
      (let ((the-viewpoint (get-only-val (list content-node
					       'kb-subgraph))))
	(cond ((null the-viewpoint)
	       '(error no-viewpoint))
	      ((content-node-has-error? content-node)
	       '(error problem-viewpoint))
	      (t (let* ((all-slots (all-explicit-slots the-viewpoint))
			(interesting-slots
			 (set-difference all-slots
					 '(last-edit-time
					   last-editor
					   specification-type 
					   generalizations
					   viewpoint-of
					   instance-of
					   author
					   creation-time)
					 :test #'equal))
			(processes (get-processes-on-slots the-viewpoint
							   interesting-slots)))
		   (if (null processes)
		       '(error no-processes)
		       processes)))))))


;;; returns all values on unit that appear on some slot in slot-list
;;; and are specs of process
(defun get-processes-on-slots (unit slot-list)
  (remove-duplicates
   (reduce #'append
	   (mapcar #'(lambda (slot)
		       (let ((values (get-local (extend-address unit slot))))
			 (reduce #'append
				 (mapcar #'(lambda (value)
					     (if (process-p value)
						 (list value)))
					 values))))
		   slot-list))
   :test 'equal))


;-----------------------------------------------------------------------
;				Connect-To-Core
;-----------------------------------------------------------------------

;;; arguments: Origin
;;;
;;; called from Process-Significance
;;;
;;; finds connection from Origin to either (1) a process that 
;;; is marked ``core'' in the knowledge base, or (2) a process 
;;; that is known to the user and is on the search path to a
;;; core process, or (3) the unit ``Process'' is reached, which
;;; is an error
;;;
;;; first climbs up superevents and then climbs up generalizations
;;;
;;; assumes if user knows a process, he knows the relation of that
;;; process to a core process (if at the begining of a session
;;; the user is told about all core processes, or if it is
;;; known that he knows about all core processes, then this
;;; assumption is true by induction)
;;;
;;; if a process has more than one superevent or more than one
;;; generalization, the first one is chosen (arbitrarily)
;;;
;;; errors:
;;;             - condition: given parameter is not a process
;;;             - returns: (error origin-not-a-process)
;;;
;;;		- condition: cannot find a genl
;;;		- returns: (error no-known-or-core-genl-available)
;;;
;;; returns a list: the first item in the list is the
;;; unit at which the search terminated, the next 
;;; item in the list is the slot that connects the 
;;; units in the search path, the next item is the 
;;; penultimate unit in the search, etc., until the 
;;; final item in the list which is the origin
;;;
;;; all of this is found in a viewpoint with the following template:
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           specification-type: Core-Connection
;;;                           viewpoint-of: <Given-Process>
;;;                           connection-to-core: (<Core-Process> <link-1>
;;;                                               <Linking-Process-1> <link-2>
;;;                                               <Linking-Process-2> <link-3>
;;;                                               ... <link-3>
;;;                                                   <Given-Process>)
;;;                           first-item-is-core?: <True or False>


(defun connect-to-core (origin)
  (let ((connect-to-core-result 
	 (if (not (process-p origin))
	     '(error origin-not-a-process)

	     ;; first climb superevents
	     (let* ((superevent-search-result (climb-superevents origin))
		    (top-superevent (first superevent-search-result)))
	       (if (or (is-known-p top-superevent)
		       (is-core-p top-superevent))
		   superevent-search-result

		   ;; then climb genl-events
		   (let ((genl-event-search-result
			  (climb-genl-events top-superevent)))
		     (if genl-event-search-result
			 (append (all-but-last genl-event-search-result)
				 superevent-search-result)
			 '(error no-known-or-core-genl-available)))))))

	(new-view-unit (make-new-view origin)))

    (put-local (list new-view-unit 'specification-type)
	       '(core-connection))
    (cond ((contains-error connect-to-core-result)
	   (signal-kb-access-error new-view-unit
				   (second connect-to-core-result)))
	  (t (put-local (list new-view-unit 'connection-to-core)
			(list connect-to-core-result))
	     (put-local (list new-view-unit 'first-item-is-core?)
			(if (is-core-p (first connect-to-core-result))
			    '(true)
			    '(false)))))
    new-view-unit))


(defun climb-superevents (origin)
  (substitute 'subevents
	      'superevents
	      (reverse (third
			(kb-search (list origin)
				   (list 'superevents)
				   :terminate-with-success-criteria
				   #'(lambda (process)
				       (null (get-local
					      (extend-address process
							      'superevents))))
				   :collect-path? t
				   :loop-elimination? t)))))

	
(defun climb-genl-events (process)
  (let ((search-result
	 (kb-search (list process)
		    (list 'generalizations 'stage-of 'i-genls)
		    :terminate-with-success-criteria
		    #'(lambda (current-process)
			(or (is-known-p current-process)
			    (is-core-p current-process)))
		    :collect-path? t
		    :loop-elimination? t)))
    (if (not (equal (first search-result) 'fail))
	(mapcar #'(lambda (path-element)
		    (case path-element
		      ((generalizations i-genls) 'specializations)
		      (stage-of                  'stages)
		      (t                         path-element)))
		(reverse (third search-result))))))


;-----------------------------------------------------------------------
;			      Connect-To-Core-Rev
;-----------------------------------------------------------------------

;;; arguments: Origin
;;;
;;; called from Process-Significance
;;;
;;; If user modeling ON:
;;; --------------------
;;; finds connection from Origin to either (1) a process that 
;;; is marked ``core'' in the knowledge base, or (2) a process 
;;; that is known to the user and is on the search path to a
;;; core process, or (3) the unit ``Process'' is reached, which
;;; is an error
;;;
;;; If user modeling OFF:
;;; ---------------------
;;; climbs up a superevent if it exists
;;;
;;; returns an error if Origin is elementary (so no need to
;;;                                           make a viewpoint)
;;;
;;; exactly like Connect-to-Core (except for elementary check), but
;;; path is presented in order from Origin to terminal process
;;;
;;; errors:
;;;             - condition: given parameter is not a process
;;;             - returns: (error origin-not-a-process)
;;;
;;;             - condition: given parameter is elementary
;;;             - returns: (error origin-is-elementary)
;;;
;;;		- condition: cannot find a genl
;;;		- returns: (error no-known-or-core-genl-available)
;;;
;;;             - condition: no superevent available
;;;             - returns: (error no-superevent-available)
;;;
;;; all of this is found in a viewpoint with the following template:
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           specification-type: Core-Connection-Rev
;;;                           viewpoint-of: <Given-Process>
;;;                           connection-to-core: (Given-Process> <link-1>
;;;                                               <Linking-Process-1> <link-2>
;;;                                               <Linking-Process-2> <link-3>
;;;                                               ... <link-3> <Core-Process>)

(defun connect-to-core-rev (origin)
  (let ((connect-to-core-result 
	 (if (not (process-p origin))
	     '(error origin-not-a-process)

	     ;; check for elementary
	     (if (is-elementary-p origin)
		 '(error origin-is-elementary)

		 (if (not (user-modeling-on?))

		     ;; general case of user modeling off
                     ;; ---------------------------------
		     (let ((super-event
				(get-only-val (list origin 'superevents))))
		       (if super-event
			   (list origin 'superevents super-event)
			   '(error no-superevent-available)))
		     
		     ;; general case of user modeling on
		     ;; --------------------------------

		     ;; first climb superevents
		     (let* ((superevent-search-result (climb-superevents-rev
						       origin))
			    (top-superevent (first
					     (last
					      superevent-search-result))))
		       (if (or (is-known-p top-superevent)
			       (is-core-p top-superevent))
			   superevent-search-result

			   ;; then climb genl-events
			   (let ((genl-event-search-result
				  (climb-genl-events-rev top-superevent)))
			     (if genl-event-search-result
				 (append superevent-search-result
					 (rest genl-event-search-result))
				 '(error no-known-or-core-genl-available)))))))))

	(new-view-unit (make-new-view origin)))

    (put-local (list new-view-unit 'specification-type)
	       '(core-connection-rev))
    (cond ((contains-error connect-to-core-result)
	   (signal-kb-access-error new-view-unit
				   (second connect-to-core-result)))
	  (t (put-local (list new-view-unit 'connection-to-core)
			(list connect-to-core-result))))
    new-view-unit))


(defun climb-superevents-rev (origin)
  (third
   (kb-search (list origin)
	      (list 'superevents)
	      :terminate-with-success-criteria
	      #'(lambda (process)
		  (null (get-local
			 (extend-address process
					 'superevents))))
	      :collect-path? t
	      :loop-elimination? t)))


(defun climb-genl-events-rev (process)
  (let ((search-result
	 (kb-search (list process)
		    (list 'generalizations 'stage-of 'i-genls)
		    :terminate-with-success-criteria
		    #'(lambda (current-process)
			(or (is-known-p current-process)
			    (is-core-p current-process)))
		    :collect-path? t
		    :loop-elimination? t)))
    (if (not (equal (first search-result) 'fail))
	(third search-result))))


;-----------------------------------------------------------------------
;			   Find-Destination-Process
;-----------------------------------------------------------------------

;;; arguments: Core-Content-Node
;;;
;;; called from Process-Significance
;;;
;;; looks in content node resulting from Connect-to-core call
;;;
;;; returns first element of the list, which is the destination
;;; of the search that was conducted in Connect-to-core
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;                          with connect-to-core path
;;;             - returns: (error error-in-node-with-connect-to-core)
;;;
;;;             - condition: an error was returned by Connect-to-core
;;;             - returns: (error connect-to-core-failed)

(defun find-destination-process (core-content-node)
  (cond ((null core-content-node)
	 '(error no-content-node))
	((content-node-has-error? core-content-node)
	 '(error error-in-node-with-connect-to-core))
	(t (let* ((connection-view (get-only-val (list core-content-node
						       'kb-subgraph)))
		  (connection-path (get-only-val (list connection-view
						       'connection-to-core))))
	     (if (or (null connection-path)
		     (equal (get-only-val (list core-content-node
						'error-occurred))
			    'true))
		 '(error connect-to-core-failed)
		 (first connection-path))))))


;-----------------------------------------------------------------------
;				 Find-Inverse
;-----------------------------------------------------------------------

;;; arguments: Slot
;;;
;;; looks up inverse
;;;
;;; note: differs from KM's get-inverse in that actual slot
;;;       is returned
;;;
;;; returns inverse of given slot
;;;
;;; errors:
;;;		- condition: no inverse can be found
;;;		- returns: (error no-inverse-available)
;;;

(defun find-inverse (slot)
  (let ((listed-inverse (get-inverse slot)))
    (if (consp listed-inverse)
	(first listed-inverse)
	'(error no-inverse-available))))


;-----------------------------------------------------------------------
;		   Find-Object-Functions-from-Content-Nodes
;-----------------------------------------------------------------------

;;; arguments: Content-Node-List: a list of content nodes
;;;                               of type ``Object-Functions''
;;;
;;; called from Sub-Struct-Function
;;;
;;; for each content node in list, first finds the viewpoint of
;;; of that content node (with the kb-subgraph slot), and then
;;; finds the process that the view is taken of
;;;
;;; returns a list of processes (functions) which are the functions
;;; discussed in these content nodes
;;;
;;; errors: none

(defun find-object-functions-from-content-nodes (obj-funct-content-nodes)
  (remove-duplicates
   (mapcar #'(lambda (content-node)
	       (let ((viewpoint (get-only-val (list content-node
						    'kb-subgraph))))
		 (if viewpoint
		     (get-only-val (list viewpoint 'viewpoint-of)))))
	   obj-funct-content-nodes)
   :test #'equal))


;-----------------------------------------------------------------------
;			  Find-Partonomic-Connection
;-----------------------------------------------------------------------

;;; arguments: Object
;;;
;;; searches up the partonomy from Object to find concept with
;;; which user is familiar
;;;
;;; considers slots other than ``part-of'' that are similar,
;;; e.g., contained-in
;;;
;;; If user modeling is ON:
;;; -----------------------
;;; search continues until encounter either (1) a known concept 
;;; (uses function is-known-p) or (2) an elementary concept 
;;; (is-known-p uses function is-elementary-p), or (3) no
;;; familiar concept is encountered and cannot climb higher
;;; in partonomy (in which case the highest super-part is returned)
;;;
;;; if user-modeling is not on, returns (1) error if Object has
;;; no super-parts, or (2) highest super-part encountered
;;;
;;; performs a breadth-first search up the partonomy (needs to
;;; be breadth-first to ensure that most specific concept in
;;; partonomy is found)
;;;
;;;     template:          Specification
;;;                        -------------
;;;                        specification-type: Super-Structural-Connection
;;;                        viewpoint-of: <Given-Object>
;;;                        highest-superpart: <Super-Part>
;;;                        super-parts-chain: (<Given-Object> <part-link-1>
;;;                                            <Intermediate-Object-1>
;;;                                            <part-link-2> 
;;;                                            <Intermediate-Object-2> ...
;;;                                            <part-link-n> <Super-Part>)
;;;
;;; If user modeling is OFF:
;;; ------------------------
;;; gets ALL of an object's superparts (if it exists)
;;; 
;;;
;;;                          Specification-i
;;;                          ---------------
;;;                          specification-type: Super-Structural-Flat
;;;                          viewpoint-of: <Given-Object>
;;;                          [attribute-1: (<Value-1 ... <Value-p>)]
;;;                          [attribute-2: (<Value-1 ... <Value-q>)]
;;;                                .
;;;                                .
;;;                                .
;;;                          [attribute-n: (<Value-1 ... <Value-r>)]
;;;
;;;
;;; errors:
;;;             - condition: parameter has no superparts
;;;             - returns: (error has-no-superparts)

(defun find-partonomic-connection (object)
  (if (not (user-modeling-on?))
      (find-partonomic-connection-flat object)
      (find-partonomic-connection-chain object)))


(defun find-partonomic-connection-flat (object)
  (let ((new-view (retrieve-view (list object
				       'dimension
				       '(spatial-superstructural)))))
    (when (viewpoint? new-view)
      (put-local (list new-view 'specification-type)
		 '(super-structural-flat))
      (remove-all-values (list new-view 'basic-dimensions)))
    new-view))


(defun find-partonomic-connection-chain (object)
  (let ((connection-result
	 (if (or (not (object-p object))
		 (has-no-superparts object))
	     '(error has-no-superparts)
	     (find-partonomic-connection-aux object)))
	(new-view-unit (make-new-view object)))

    (put-local (list new-view-unit 'specification-type)
	       '(super-structural-connection))
    (cond ((contains-error connection-result)
	   (signal-kb-access-error new-view-unit
				   (second connection-result)))
	  (t (put-local (list new-view-unit 'super-parts-chain)
			(list connection-result))
	     (put-local (list new-view-unit 'highest-superpart)
			(list (first (last connection-result))))))

    new-view-unit))


(defun find-partonomic-connection-aux (object)
  (let ((initial-unit-list (list object))
	(part-slots (get-local '(part-slot-list
				 part-slots-used-by-knight))))
    (if (not (user-modeling-on?))

	;; case 1: user modeling is off
	;; ----------------------------
	(let ((super-part-slot-value-pair
	       (get-immediate-super-part-slot-value-pair object)))
	  (if super-part-slot-value-pair
	      (cons object super-part-slot-value-pair)
	      '(error has-no-superparts)))

	;; case 2: user modeling is on
	;; ---------------------------
	(let ((search-result 
	       (kb-search initial-unit-list
			  part-slots
			  :terminate-with-success-criteria
			  #'(lambda (object)
			      (or (is-known-p object)
				  (has-no-superparts object)))
			  :control-strategy 'breadth-first
			  :collect-path? t
			  :loop-elimination? t)))
	  (if (equal (first search-result) 'fail)

	      ;; error
	      '(error has-no-superparts)

	      ;; terminated with success, so return path
	      (third search-result))))))


;;; if object has a superpart, return a pair:
;;;              (superpart-slot superpart)
;;; otherwise nil
;;;
;;; if there multiple ones, arbitrarily chooses a single slot and value 
(defun get-immediate-super-part-slot-value-pair (object)
  (let ((part-slots (get-local '(part-slot-list
				 part-slots-used-by-knight)))
	(slot-value-pair nil))
    ;; note: part slots are reversed to correspond to default search order
    (dolist (part-slot (reverse part-slots))
      (let ((superpart (get-only-val (list object part-slot))))
	(when superpart
	  (setf slot-value-pair (list part-slot superpart)))))
    slot-value-pair))


;;; returns t iff object has no super-parts
(defun has-no-superparts (object)
  (null (get-immediate-super-parts object)))


;;; finds immediate super-parts of an object
(defun get-immediate-super-parts (object)
  (let ((part-slots (get-local '(part-slot-list
				 part-slots-used-by-knight))))
    (reduce #'append   
	    (mapcar #'(lambda (slot)
			(get-local (extend-address object slot)))
		    part-slots))))


;-----------------------------------------------------------------------
;				 Find-Ref-Conc
;-----------------------------------------------------------------------

;;; arguments: Origin
;;;
;;; searches up the taxonomy from Origin to find concept with
;;; which user is familiar
;;; 
;;; considers slots ``generalizations'', ``stage-of'', and ``i-genls''
;;;
;;; search continues until encounter either (1) a known concept 
;;; (uses function Is-known-p) or (2) an elementary concept 
;;; is-known-p uses function is-elementary-p), or (3) no
;;; familiar concept is encountered (this is an error)
;;;
;;; performs a breadth-first search up the taxonomy (needs to
;;; be breadth-first to ensure that most specific concept in
;;; partonomy is found) via the KM function ``ancestry*''
;;;
;;; slight complication: because of the structure of the taxonomy
;;;                      it may be the case that the breadth-first
;;;                      search will return an elementary concept
;;;                      that is encountered sooner but is in fact
;;;                      very general, e.g., ``object'' or ``process''
;;;
;;; solution: if finds an elementary concept first, and this concept is
;;;           either ``process'' or ``object,'' attempts to find a
;;;           more specific elementary concept; if fails to find a more
;;;           specific elementary concept, then returns either ``process''
;;;           or ``object''
;;;
;;; if user-modeling is not on, returns (1) error if Origin has
;;; no genl's, or (2) first genl encountered
;;;
;;; returns that concept at which the search halted
;;; 
;;; Note: only will undertake search if origin is a top-level unit
;;; 
;;; errors:
;;;             - condition: origin is not a top-level unit
;;;             - returns: (error origin-not-top-level-unit)
;;;
;;;		- condition: user modeling is off and origin has
;;;                          no genls
;;;		- returns: (error no-user-modeling-and-no-genl)
;;;
;;;		- condition: no familiar concept is encountered and
;;;                          cannot climb higher in taxonomy
;;;		- returns: (error no-familiar-generalizations)

(defun find-ref-conc (origin)
  (if (or (not (atom origin))
	  (not (member 'entity (ancestry* origin) :test #'equal)))
      '(error origin-not-top-level-unit)
      (let ((initial-unit-list (list origin))
	    (genl-slots (list 'generalizations 'stage-of 'i-genls)))
	(if (user-modeling-on?)

	    ;; find reference concept with user modeling on
	    (let ((search-result 
		   (kb-search initial-unit-list
			      genl-slots
			      :terminate-with-success-criteria #'is-known-p
			      :control-strategy 'breadth-first
			      :loop-elimination? t)))
	      (if (equal (first search-result) 'fail)

		  ;; no familiar concept found
		  '(error no-familiar-generalizations)

		  ;; terminated with success, so determine if result
		  ;; was elementary or not; if so, and if the elementary
		  ;; concept was either ``object'' or ``process'' then
		  ;; attempts to find a more specific elementary concept
		  (let ((termination-concept (second search-result)))
		    (if (and (is-elementary-p termination-concept)
			     (or (equal termination-concept 'object)
				 (equal termination-concept 'process)))
			(find-more-specific-elementary-concept
			 origin
			 termination-concept)
			termination-concept))))

	    ;; user-modeling is off, so check to see if object has
	    ;; generalizations, if yes, return one; if not, return error
	    (let ((immediate-generalizations
		   (reduce #'append   
			   (mapcar #'(lambda (slot)
				       (get-local (extend-address origin slot)))
				   genl-slots))))
	      (if (null immediate-generalizations)
		  '(error no-user-modeling-and-no-genl)

		  ;;otherwise arbitrarily choose one
		  (first immediate-generalizations)))))))


;;; given:
;;;   - origin: concept at which the first search begun
;;;   - termination-concept: concept at which the first search ended
;;;
;;; assumptions: 
;;;   - termination-concept is elementary
;;;
;;; returns: if can find an elementary concept that is in the ancestry of
;;;          origin and is not ``object'' or ``process'' then returns
;;;          that concept; otherwise returns termination concept

(defun find-more-specific-elementary-concept (origin termination-concept)
  (let ((search-result 
	 (kb-search (list origin)
		    (list 'generalizations 'stage-of 'i-genls)
		    :terminate-with-success-criteria
		    #'(lambda (concept)
			(and (is-elementary-p concept)
			     (not (equal concept 'process))
			     (not (equal concept 'object))))
		    :control-strategy 'breadth-first
		    :loop-elimination? t)))
    (if (equal (first search-result) 'fail)
	termination-concept
	(second search-result))))


;-----------------------------------------------------------------------
;			      Find-Subevent-Order
;-----------------------------------------------------------------------

;;; arguments: Event
;;;
;;; simply returns list of subevents
;;;
;;; *** later: should examine the list in slot: ordered-subevents
;;;
;;; errors:
;;;		- condition: event is not a process
;;;             - returns: (error non-process-argument)

(defun find-subevent-order (event)
  (if (process-p event)
      (get-local (list event 'subevents))
      '(error non-process-argument)))

;-----------------------------------------------------------------------
;				Find-Subevents
;-----------------------------------------------------------------------

;;; arguments: - Event-List: a list of events
;;;
;;; simply returns list of all subevents of events in Event-List
;;; (with dupes removed)
;;;

(defun find-subevents (event-list)
  (if (listp event-list)
      (remove-duplicates
       (reduce #'append
	       (mapcar #'(lambda (event)
			   (if (process-p event)
			       (get-local (list event 'subevents))))
		       event-list))
       :test #'equal)))


;-----------------------------------------------------------------------
;			    Find-Undiscussed-Actors
;-----------------------------------------------------------------------

;;; arguments: Actor-List
;;;
;;; for each actor in Actor-list, places actor in result
;;; list if actor hasn't been discussed
;;;
;;; uses ``is-known-p'' function
;;;
;;; returns subset of Actor-list where the actors haven't yet been discussed
;;;
;;; if user modeling not on, returns entire actor-list
;;;
;;; errors:
;;;		- condition: list of actors is empty
;;;             - returns: (error no-actors)

(defun find-undiscussed-actors (actor-list)
  (cond	((null actor-list)
	 '(error no-actors))
	((not (user-modeling-on?))
	 actor-list)
	(t (remove-if #'(lambda (x) (is-known-p x))
		      actor-list))))


;-----------------------------------------------------------------------
;			     Get-Function-Process
;-----------------------------------------------------------------------

;;; arguments: Object-significance-content-node
;;;
;;; retrieves process which was used in functional description
;;; in this content node
;;; 
;;; currently finds first process on the agent-in slot
;;; of the viewpoint
;;;
;;; later may want to modify to return multiple processes 
;;; (for objects with multiple functions)
;;;
;;; returns the process
;;;
;;; errors:
;;;		- condition: node is empty
;;;		- returns: (error no-significance-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;             - returns: (error error-in-object-significance-node)
;;;
;;;             - condition: no viewpoint exists in content node
;;;             - returns: (error no-viewpoint-in-significance-node)

(defun get-function-process (content-node)
  (cond ((null content-node)
	 '(error no-significance-node))
	((content-node-has-error? content-node)
	 '(error error-in-object-significance-node))
	(t (let ((viewpoint
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null viewpoint)
		 '(error no-viewpoint-in-significance-node)
		 (get-function-process-aux viewpoint))))))


(defun get-function-process-aux (viewpoint)
  (let* ((actual-slots (all-explicit-slots viewpoint))
	 (chosen-slot (choose-function-process-slot actual-slots)))
    (get-only-val (extend-address viewpoint chosen-slot))))


;;; the function to change to more intelligently choose 
;;; a function slot; currently just picks the first
(defun choose-function-process-slot (slot-list)
  (first slot-list))

	
;-----------------------------------------------------------------------
;			   Get-Functional-Processes
;-----------------------------------------------------------------------

;;; arguments: Content Node with Struct-Funct-Viewpoint
;;;
;;; traverses structural-functional viewpoint to obtain list
;;; of processes that are mentioned in the viewpoint
;;;
;;; returns list of processes
;;;
;;; errors:
;;;		- condition: node is empty
;;;		- returns: (error no-black-box-functional-iteration-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;             - returns: (error error-in-black-box-functional-iteration-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-struct-funct-viewpoint)
;;;
;;;		- condition: struct-funct viewpoint has no top-level view
;;;		- returns: (error struct-funct-viewpoint-malformed)

(defun get-functional-processes (content-node)
  (cond ((null content-node)
	 '(error no-black-box-functional-iteration-node))
	((content-node-has-error? content-node)
	 '(error error-in-black-box-functional-iteration-node))
	(t (let ((struct-funct-viewpoint 
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null struct-funct-viewpoint)
		 '(error no-struct-funct-viewpoint)
		 (let ((viewpoint-of-top-event
			(get-viewpoint-of-top-event struct-funct-viewpoint)))
		   (if (null viewpoint-of-top-event)
		       '(error struct-funct-viewpoint-malformed)
		       (get-local (list viewpoint-of-top-event
					      'subevents)))))))))


(defun get-viewpoint-of-top-event (struct-funct-viewpoint)
  (let ((top-event
	 (get-only-val (list struct-funct-viewpoint 'correspondence-with))))
    (get-only-val (list struct-funct-viewpoint
			'correspondence-with
			top-event
			'viewpoint-of))))


;-----------------------------------------------------------------------
;			       Get-Location-Slot
;-----------------------------------------------------------------------

;;; arguments: Locational-Viewpoint
;;;
;;; searches through the viewpoint to find the location slot
;;;
;;; assumes:
;;;          - there is only one location slot (if there is more
;;;            than one, it is ignored)
;;;          - the location slot has only one value
;;;       
;;; returns the location slot
;;;
;;; errors:
;;;             - condition: viewpoint is nil
;;;               returns:   nil
;;;
;;;             - condition: finds no location slot
;;;               returns:   nil

(defun get-location-slot (locational-viewpoint)
  (let* ((slots-with-values (all-explicit-slots locational-viewpoint))
	 (all-location-slots (remove-if-not #'is-location-slot-p
					    slots-with-values)))
    (if all-location-slots
	(first all-location-slots))))


(defun is-location-slot-p (slot)
  (member 'location (ancestry* slot)))


;-----------------------------------------------------------------------
;			       Get-Output-From-Slot
;-----------------------------------------------------------------------

;;; arguments: an Output-Actor-From-Viewpoint
;;;
;;; searches through the viewpoint to find the output-actor-from slots
;;;
;;; arbitrarily chooses one slot and returns it
;;;
;;; errors:
;;;             - condition: viewpoint is nil
;;;               returns:   nil
;;;
;;;             - condition: finds no output-actor-from-slots
;;;               returns:   nil

(defun get-output-from-slot (output-actor-from-viewpoint)
  (let* ((slots-with-values (all-explicit-slots output-actor-from-viewpoint))
	 (all-output-from-slots (remove-if-not #'is-output-from-slot-p
					       slots-with-values)))
    (if all-output-from-slots
	(first all-output-from-slots))))


(defun is-output-from-slot-p (slot)
  (let ((slot-ancestors (ancestry* slot)))
    (or (member 'output-from slot-ancestors)
	(member 'developee-in slot-ancestors))))

	   
;-----------------------------------------------------------------------
;			  Get-Part-Slots-With-Values
;-----------------------------------------------------------------------

;;; arguments: Object
;;;
;;; Art also asked for the following algorithm (6-28-94)
;;;
;;; procedure: get all slots
;;;            if any slots then
;;;                 if have non-composed-of slots AND composed-of slot
;;;                 keep only non-composed-of slots
;;;             elseif have only non-composed-of slots
;;;                 keep them
;;;             elseif have composed-of slots
;;;                 keep them
;;;             else NONE
;;;
;;; returns list of part-slots that have at least one value on Object

(defun get-part-slots-with-values (object)
  (if (object-p object)
      (let* ((all-slots (all-explicit-slots object))
	     (part-slots (get-local
			  (list 'part-slot-list
				'all-part-slots-substructures-used-by-knight)))
	     (part-slots-on-object
	      (remove-if-not
	       #'(lambda (slot)
		   (member slot part-slots :test #'equal))
	       all-slots)))
	(if (member 'composed-of part-slots-on-object)
	    (if (> (length part-slots-on-object) 1)
		(remove 'composed-of part-slots-on-object
			:test #'equal)
		part-slots-on-object)
	    part-slots-on-object))))
	

;-----------------------------------------------------------------------
;			      Get-Slot-For-Focus
;-----------------------------------------------------------------------

;;; arguments: Black-Box-Viewpoint
;;;            Object
;;;
;;; assumes either (a) Object is a value of some slot on Black-Box-Viewpoint
;;;         or     (b) Object is a value on annotation of one the slots
;;;
;;;         note: case (b) needed to find
;;;                            Leaf-Initiation
;;;                            ---------------
;;;                            generated: Leaf-Primordium
;;;                                          origin: Shoot-Apical-Meristem
;;;
;;; returns: slot on Black-Box-Viewpoint that has Object as one
;;;          of its values
;;;
;;;        note: in our example above, the function will return ``origin''
;;;
;;; note: extra code for looking at annotations had to be written quickly,
;;;       so it's very inelegant (sorry!)
;;;
(defun get-slot-for-focus (black-box-viewpoint object)
  (let* ((filled-slots (get-explicit-slots black-box-viewpoint))
	 (object-found 
	  (get-slot-for-focus-aux black-box-viewpoint
				  filled-slots
				  object)))
    (if object-found
	object-found
	(get-slot-for-focus-aux-2 black-box-viewpoint
				   filled-slots
				   object))))


(defun get-slot-for-focus-aux (black-box-viewpoint black-box-slots object)
  (if black-box-slots
      (let ((curr-slot (first black-box-slots)))
	(if (member object (get-local (list black-box-viewpoint curr-slot)))
	    curr-slot
	    (get-slot-for-focus-aux black-box-viewpoint
				    (rest black-box-slots)
				    object)))))


;;; searches top-most annotations for the object
(defun get-slot-for-focus-aux-2 (black-box-viewpoint black-box-slots object)
  (if black-box-slots
      (let* ((candidate-slot (first black-box-slots))
	     (slot (get-slot-for-focus-from-annotations black-box-viewpoint
							candidate-slot
							object)))
	(if slot
	    slot
	    (get-slot-for-focus-aux-2 black-box-viewpoint
				      (rest black-box-slots)
				      object)))))


;;; searches top-most annotations on values of
;;;         (viewpoint candidate-slot)
;;; for the object
(defun get-slot-for-focus-from-annotations (black-box-viewpoint
					    candidate-slot
					    object)
  (let ((candidate-values (get-local (list black-box-viewpoint
					   candidate-slot))))
    (get-slot-for-focus-for-annotations-aux black-box-viewpoint
					    candidate-slot
					    candidate-values
					    object)))


;;; searches each value on 
;;;   (viewpoint cand-slot)
;;; for an annotation whose value is object
(defun get-slot-for-focus-for-annotations-aux (black-box-viewpoint
					       candidate-slot
					       candidate-values
					       object)
  (if candidate-values
      (let* ((annotation-slots
	      (get-explicit-slots (list black-box-viewpoint
					candidate-slot
					(first candidate-values))))
	     (slot
	      (get-slot-for-focus-for-annotations-aux-2 black-box-viewpoint
							candidate-slot
							(first
							 candidate-values)
							annotation-slots
							object)))
	(if slot
	    slot
	    (get-slot-for-focus-for-annotations-aux black-box-viewpoint
						    candidate-slot
						    (rest candidate-values)
						    object)))))


;;; searches each annotation on 
;;;   (viewpoint cand-slot cand-value)
;;; for a value that is object
(defun get-slot-for-focus-for-annotations-aux-2 (black-box-viewpoint
						 candidate-slot
						 candidate-value
						 annotation-slots
						 object)
  (if annotation-slots
      (if (member object (get-local (list black-box-viewpoint
					  candidate-slot
					  candidate-value
					  (first annotation-slots))))
	  (first annotation-slots)
	  (get-slot-for-focus-for-annotations-aux-2 black-box-viewpoint
						     candidate-slot
						     candidate-value
						     (rest
						      annotation-slots)
						     object))))


;-----------------------------------------------------------------------
;			    Has-Common-Super-Event?
;-----------------------------------------------------------------------
;;; arguments: Proc1 Proc2
;;;
;;; returns t iff Proc1 and Proc2 have the same superevent
;;; otherwise nil
;;;
;;; used in output actor fates to ensure that a particular fate (function)
;;; of an actor is worth reporting
;;;    art's heuristic: if a candidate fate and the primary concept process
;;;                     are subevents of the same process, then it is
;;;                     worthwhile to report the fate; otherwise, it
;;;                     should be cut

(defun has-common-super-event? (proc-1 proc-2)
  (if (and (process-p proc-1)
	   (process-p proc-2))
      (let ((super-for-proc-1 (get-only-val (extend-address proc-1
							    'superevents)))
	    (super-for-proc-2 (get-only-val (extend-address proc-2
							    'superevents))))
	(and super-for-proc-1
	     super-for-proc-2
	     (equal super-for-proc-1 super-for-proc-2)))))


;-----------------------------------------------------------------------
;			    Is-Conversion-Process?
;-----------------------------------------------------------------------
;;; arguments: - Concept
;;;
;;; returns t iff concept is a conversion process
;;; otherwise nil

(defun is-conversion-process? (concept)
  (if (and concept
	   (member 'conversion (ancestry* concept)))
      t))


;-----------------------------------------------------------------------
;			Make-As-Kind-Of-Perceptual-View
;-----------------------------------------------------------------------

;;; arguments: a concept and a reference concept
;;;
;;; creates an as-kind-of viewpoint
;;;
;;; returns a viewpoint of the form
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           generalizations: <COI>
;;;                           instance-of: Viewpoint
;;;                           as-kind-of: <Ref-Conc>
;;;                           specification-type: as-kind-of
;;;                           viewpoint-of: <COI>
;;;                          [attribute list]              
;;; errors: as-kind-of errors

(defun make-as-kind-of-perceptual-view (concept ref-conc)
  (let ((new-ako-vp (retrieve-view (list concept 'as-kind-of
				     (list ref-conc
					   'dimension
					   (list 'perceptual))))))
    (when (viewpoint? new-ako-vp)
      (put-local (list new-ako-vp 'as-kind-of) (list ref-conc))
      (put-local (list new-ako-vp 'specification-type) '(as-kind-of))
      (remove-all-values (list new-ako-vp 'basic-dimensions))
      (if (has-perceptual-slots? new-ako-vp)
	  new-ako-vp
	  '(error no-perceptual-slots)))))


;;; returns non-null iff viewpoint has perceptual slots
(defun has-perceptual-slots? (viewpoint)
  (let* ((interesting-slots (get-domain-slots viewpoint))
	 (perceptual-slots (set-difference
			    interesting-slots
			    '(SPECIFICATION-TYPE
			      AS-KIND-OF
			      GENERALIZATIONS
			      VIEWPOINT-OF
			      INSTANCE-OF))))
    perceptual-slots))


;-----------------------------------------------------------------------
;			  Make-Auxiliary-Process-View
;-----------------------------------------------------------------------

;;; arguments: Process, View-Type
;;;
;;; used for the following view types:
;;;                                     - Connections-Temporal
;;;                                     - Modulatory-Facilitators
;;;                                     - Modulatory-Inhibitors
;;;                                     - Causal-Agent
;;;                                     - Temporal-Attributes
;;;                                     - Locational
;;;                                       
;;; returns a viewpoint unit of the form
;;;
;;;                          Specification-i
;;;                          ---------------
;;;                          specification-type: auxiliary-process-description
;;;                          viewpoint-of: <Process-1>
;;;                          [attribute-1: (<Value-1 ... <Value-p>)]
;;;                          [attribute-2: (<Value-1 ... <Value-q>)]
;;;                                .
;;;                                .
;;;                                .
;;;                          [attribute-n: (<Value-1 ... <Value-r>)]
;;;
;;; errors: none

(defun make-auxiliary-process-view (process view-type)
  (let ((new-view (retrieve-view (list process
				       'dimension
				       (list view-type)))))
    (when (viewpoint? new-view)
      (put-local (list new-view 'specification-type)
		 '(auxiliary-process-description))
      (remove-all-values (list new-view 'basic-dimensions)))
    new-view))


;-----------------------------------------------------------------------
;			      Make-Bare-AKO-View
;-----------------------------------------------------------------------

;;; arguments: a concept and a reference concept
;;;
;;; creates an as-kind-of viewpoint and strips out all of the slots on it
;;; except for: as-kind-of, generalizations, viewpoint-of, instance-of
;;;
;;; returns a stripped out viewpoint of the form
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           generalizations: <COI>
;;;                           instance-of: Viewpoint
;;;                           as-kind-of: <Ref-Conc>
;;;                           as-kind-of-relation: <stage-of OR generalization>
;;;                           specification-type: as-kind-of
;;;                           viewpoint-of: <COI>
;;; errors: none

(defun make-bare-ako-view (concept ref-conc)
  (let ((new-ako-vp (make-new-view concept)))
    (put-local (list new-ako-vp 'as-kind-of) (list ref-conc))
    (put-local (list new-ako-vp 'specification-type) '(as-kind-of))
    (put-local (list new-ako-vp 'as-kind-of-relation)
	       (list (find-as-kind-of-relation concept ref-conc)))
    new-ako-vp))


;;; fills in the as-kind-of-relation for as-kind-of-view
;;;
;;; if there is a stage-of link in the chain, returns ``stage-of''
;;; otherwise, returns ``generalizations''
;;;
;;; assumes a path exists between them

(defun find-as-kind-of-relation (concept ref-conc)
    (let* ((search-result 
	   (kb-search (list concept)
		      (list 'generalizations 'stage-of 'i-genls)
		      :terminate-with-success-criteria
		      #'(lambda (concept)
			  (equal concept ref-conc))
		      :control-strategy 'breadth-first
		      :collect-path? t
		      :loop-elimination? t))
	   (path (third search-result)))
      ;(format t "~%Path: ~a~%"
      ;        path)
      (if (member 'stage-of path)
	  'stage-of
	  'generalizations)))


;-----------------------------------------------------------------------
;			      Make-Black-Box-View
;-----------------------------------------------------------------------

;;; arguments: a concept and a reference concept
;;;
;;; creates a black-box view
;;;
;;; the reference concept will be used by the PDT mechanism
;;;
;;; hard-during? is an optional argument
;;; if its has a value (typically t) then the realizer will include the
;;; during clause, even if it violates redundancy checks
;;;
;;; if an error is found by the view retriever, it is reported
;;; 
;;; errors:
;;;             - condition process is nil
;;;             - returns (error null-process)
;;;
;;;             - condition: process is elementary
;;;             - returns: (error elementary-process)

(defun make-black-box-view (process reference-process &key hard-during?)
  (if (null process)
      '(error null-process)
      (if (is-elementary-p process)
	  '(error elementary-process)
	  (let ((new-vp (retrieve-view (list process
					     'dimension
					     (list 'black-box)))))
	    (when (viewpoint? new-vp)
	      (put-local (list new-vp 'specification-type)
			 '(black-box-process-description))
	      (put-local (list new-vp 'reference-concept)
			 (list reference-process))
	      (put-local (list new-vp 'include-during-clause?)
			 '(true))
	      (when hard-during? 
		(put-local (list new-vp 'include-during-clause-hard?)
			   '(true)))
	      (remove-all-values (list new-vp 'basic-dimensions)))
	    new-vp))))


;-----------------------------------------------------------------------
;			     Make-Functional-View
;-----------------------------------------------------------------------

;;; arguments: Black-Box-View: the black-box view that has been taken
;;;                            of the process
;;;            Actor-Slot:  the actor slot on the given process (which
;;;                         is the function of the object) that has
;;;                         the object as its value
;;;
;;; method: uses Black-Box view as basis for functional view, and so
;;;         the correct emphasis will be placed on the Object, exploits
;;;         FUF's focus mechanism
;;;
;;; returns: a viewpoint unit that is identical to Black-Box-View
;;;          except the focused-concept slot has been assigned the value
;;;          Actor-Slot
;;;
;;; errors:
;;;             - condition: no actor slot was given
;;;             - returns: (error no-actor-slot)
;;;
;;;             - condition: no black-box view was given
;;;             - returns: (error no-black-box-view)
;;;
;;;		- condition: black-box view has an error
;;;		- returns: (error erroneous-black-box-view)

(defun make-functional-view (actor-slot black-box-view)
  (cond ((null actor-slot)
	 '(error no-actor-slot))
	((null black-box-view)
	 '(error no-black-box-view))
	((contains-error black-box-view)
	 '(error erroneous-black-box-view))
	(t (put-local (list black-box-view 'focused-concept)
		      (list actor-slot))
	   (put-local (list black-box-view
			    'include-during-clause?)
		      '(true))
	   black-box-view)))

;-----------------------------------------------------------------------
;			      Make-Subevent-List
;-----------------------------------------------------------------------

;;; arguments: Process
;;;
;;; makes ordered subevent list
;;;
;;; errors:
;;;		- condition: Process is not an actual process
;;;		- returns: (error non-process-has-no-subevents)
;;;
;;; returns a viewpoint unit of the form
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           specification-type: Subevent-List
;;;                           viewpoint-of: <Parent-Process>
;;;                           ordered-subevents: (<Process-1> <Process-2>
;;;                                               ... <Process-n>)

(defun make-subevent-list (process)
  (let* ((new-view-unit (make-new-view process))
	 (subevents (get-local (extend-address process 'ordered-subevents))))
    (put-local (list new-view-unit 'specification-type)
	       '(subevent-list))
    (cond ((not (process-p process))
	   (signal-kb-access-error new-view-unit
				   'non-process-has-no-subevents))
	  ((null subevents)
	   (signal-kb-access-error new-view-unit
				   'process-has-no-subevents))
	  (t (put-local (list new-view-unit 'ordered-subevents)
			subevents)))
  new-view-unit))


;-----------------------------------------------------------------------
;			    Make-Sub-Structural-View
;-----------------------------------------------------------------------

;;; arguments: - Object
;;;            - Part-Slot
;;;
;;; returns a viewpoint of the form
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           specification-type: Sub-Structural-Description
;;;                           viewpoint-of: <Object>
;;;                           part-slot-for-view: <Part-Slot>
;;;                           part-values: <Values>
;;;
(defun make-sub-structural-view (object part-slot)
  (let ((new-view-unit (make-new-view object))
	(vals
	 ;(get-local-with-annotation object part-slot)
	 (get-local (extend-address object part-slot)))
	)

    (put-local (list new-view-unit 'specification-type)
	       '(sub-structural-description))
    (put-local (list new-view-unit 'part-slot-for-view)
	       (list part-slot))
    (when vals
      (put-local (list new-view-unit 'part-values) vals)
      (dolist (val vals)
	(copy-values-only (list object part-slot val)
			  (list new-view-unit 'part-values val)))
      )
    ;(put-local-with-annotation new-view-unit 'part-values vals)
    new-view-unit))


;-----------------------------------------------------------------------
;		 Is-Function-Of-Interest-Wrt-Object-Functions?
;-----------------------------------------------------------------------

;;; arguments: Function: the function of a sub-structure being considered
;;;            Object-Functions  : the main object
;;;
;;; returns t iff Function is a subevent of Object-Functions
;;;         otherwise nil
;;; errors: none

(defun is-function-of-interest-wrt-object-functions? (function
						      object-functions)
  (if (and function
	   object-functions
	   (listp object-functions))
      (let ((all-subevents (reduce #'append
				   (mapcar #'(lambda (process)
					       (get-local (list process
								'subevents)))
					   object-functions))))
	(if (member function all-subevents :test #'equal)
	    t))))


;-----------------------------------------------------------------------
;			  Make-Temporal-Step-Of-View
;-----------------------------------------------------------------------

;;; arguments: Process: the process being described
;;;
;;; returns a viewpoint of the following form:
;;;
;;;                           Specification-i
;;;                           ---------------
;;;                           specification-type: Core-Connection-Rev
;;;                           viewpoint-of: <Given-Process>
;;;                           connection-to-core: (Given-Process> superevents
;;;                                                <Known-Superevent>)
;;;
;;; assumes: a process has at most one superevent
;;;
;;; errors: 
;;;		- condition: Process is not an actual process
;;;		- returns: (error origin-not-a-process)
;;;
;;;		- condition: Process has no superevents
;;;		- returns: (error process-has-no-subevents)
;;;
;;;             - condition: Superevent is not known
;;;             - returns: (error superevent-not-known)

(defun make-temporal-step-of-view (origin)
  (let ((step-of-result 
	 (if (not (process-p origin))
	     '(error origin-not-a-process)
	     (let ((superevent (get-only-val (list origin 'superevents))))
	       (if (null superevent)
		   '(error process-has-no-superevents)
		   superevent))))
	(new-view-unit (make-new-view origin)))

    (cond ((contains-error step-of-result)
	   (signal-kb-access-error new-view-unit
				   (second step-of-result)))
	  ((and (user-modeling-on?)
		(not (is-known-p step-of-result)))
	   (signal-kb-access-error new-view-unit
				   '(error superevent-not-known)))
	  (t
	   (put-local (list new-view-unit 'specification-type)
		      '(core-connection-rev))
	   (put-local (list new-view-unit 'connection-to-core)
		      (list (list origin
				  'superevents
				  step-of-result)))))
    new-view-unit))
   

