;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       
				       
;;;;			     The Error Predicates
;;;;			     --------------------


;;;; This file contains predicates for determining if an error
;;;; has occurred.
;;;;
;;;; Function Directory:
;;;;
;;;;    (Function)                         (Arguments)    (Comments)
;;;;
;;;;     Error Predicates
;;;;     -----------------
;;;;     contains-error                     expression     for templates
;;;;     content-node-has-error?            content-node
;;;;     viewpoint-contains-error?          viewpoint
;;;;     empty-view-p                       viewpoint
;;;;
;;;;     Error Signaling Functions
;;;;     -------------------------
;;;;     signal-empty-viewpoint-warning     content-node
;;;;     signal-kb-access-error             content-node,


(in-package 'km)


;-----------------------------------------------------------------------
;			       Error Predicates
;-----------------------------------------------------------------------

;;; determines if the keyword ``error'' occurs anyplace
;;; in the nested expression

(defun contains-error (expression)
  (cond ((null expression)
	 nil)
	((and (atom expression)
	      (equal expression 'error))
	 t)
	((atom expression) nil)
	(t (or (contains-error (first expression))
	       (contains-error (cdr expression))))))


;;; determines if an error is shown by a content-node

(defun content-node-has-error? (content-node)
  (equal (get-only-val (list content-node 'error-occurred?))
	 'true))


;;; determines if a viewpoint has an error
;;;
;;; returns t iff viewpoint is not a true viewpoint or
;;;               when viewpoint has an error

(defun viewpoint-contains-error? (viewpoint)
  (or (not (viewpoint? viewpoint))
      (equal (get-only-val (list viewpoint 'error-occurred?))
	     'true)))


;;; determines if a view returned by the view retriever is empty

(defun empty-view-p (view)
  (let* ((slots-with-values (all-explicit-slots view))
	 (uninteresting-slots (get-local '(non-domain-slot-list
					   non-domain-slots)))
	 (view-slots '(viewpoint-of instance-of basic-dimensions
		       kb-subgraph-of generalizations
		       specification-type focused-concept
		       reference-concept object-focus
		       part-slot-for-view
		       first-item-is-core?
		       highest-superpart))
	 (all-uninteresting-slots (append view-slots uninteresting-slots))
	 (interesting-slots (set-difference slots-with-values
					    all-uninteresting-slots)))
    ;(format t "Interesting-slots: ~a~%" interesting-slots)
    (null interesting-slots)))


;-----------------------------------------------------------------------
;			   Error Signaling Functions
;-----------------------------------------------------------------------

;;; signals an empty viewpoint error

(defun signal-empty-viewpoint-warning (content-node)
  (when (trace-kb-access?)
    (format t "KB Access Warning: Empty viewpoint at ~a.~%~%"
	    content-node))
  (put-local (list content-node 'error-occurred?)
	     '(true))
  (put-local (list content-node 'error-type)	     
	     '(empty-viewpoint)))

    
;;; records KB Access error in content-node of Explanation Plan
;;; when trace is turned on, prints error message

(defun signal-kb-access-error (content-node error-type)
  (when (trace-kb-access?)	       
    (format t "An error has occurred at ~a.~%"
	    content-node)
    (format t "Error type: ~a~%~%"
	    error-type))
  (put-local (list content-node 'error-occurred?)
	     '(true))
  (put-local (list content-node 'error-type)	     
	     (list error-type)))


;;; signals that an iteration list is empty

(defun signal-empty-iteration-list-error (topic-node instantiated-template)
  (format t "An error has occurred at ~a.~%"
	  topic-node)
  (format t "Error type: Null iteration list.~%")
  (format t "Instantiated iteration template:")
    (pprint instantiated-template t)
    (format t "~%~%"))

