;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       
				       
;;;;			   The Evaluation Functions
;;;;			   ------------------------
				       

;;;; This file contains functions for running evaluations.


;-----------------------------------------------------------------------
;			 Evaluation Function Directory
;-----------------------------------------------------------------------
;;;;
;;;;     (choose-random-concepts number-to-choose
;;;;                             concept-type
;;;;		                 &optional (include-embedded-units? nil)
;;;;      		                   (include-elementary? nil))
;;;;
;;;;     (collect-vps-to-test exposition-node)
;;;;
;;;;     (make-nlg-test-specs-report)


(in-package 'km)


;-----------------------------------------------------------------------
;			   Concept Choice Functions
;-----------------------------------------------------------------------
;;;
;;; These functions randomly select a specified number of object
;;; and/or processes from the knowledge base.
;;;
;;; The user may specify whether embedded units and elementary concepts
;;; should be included.


;;; defines which object concepts in the KB whose progeny should
;;; be considered for choice
;;;
;;; Old list: (plant plant-part)
;;;
;;; Art's List:  (not consistent and not correct)
;;;                          hypocotyl
;;;          	             biota
;;;		             plant-association
;;;                          population
;;;                          accessory-cells
;;;                          epidermal-cells
;;;                          virus
;;;                          cell-walls
;;;                          protoplasm
;;;                          cytoplasm
;;;                          meristem
;;;		             organ
;;;       		     organism
;;;		             cell
;;;		             tissue
;;;                          botanical-system
;;;                          seed
;;;		             organelle
;;;                          protoplasm
;;;			     virus))
;;;
;;; Trouble: (BOTANICAL-ORGAN PLANT-CELL-LEVEL-STRUCTURE PLANT-PART
;;; STOMA-CONTAINING-STRUCTURE)
;;;
;;; must avoid these
;;; generalizations of Plant: (MULTI-CELLULAR-ORGANISM PHOTOSYNTHETIC-ORGAN
;;;                            PLANT-ORGAN-LEVEL-STRUCTURE)
;;; these run only into trouble

(defparameter *object-tops* '(
                          subcellular-region
                          plant-tissue
                          haploid-cell
			  diploid-cell
                          botanical-system
                          diploid-phase-plant
                          haploid-phase-plant
                          mature-plant
                          zygote
                          embryo
                          seedling
                          immature-plant
                          seed
			  plant-supercellular-region
                          seed-part
                          plant-tissue-part
			  stomatal-structure
			  stomatal-structures
                          epidermis
			  hypocotyl
			  vascular-system-part
			  shoot-system-part
			  vascular-tissue
			  phloem-network
			  plant-organ-part
			  intercellular-spaces
			  fiber-strand
			  dermal-system-part
			  branches
			  shoots
			  crown
			  thorn
			  prickle
			  nectary
			  perforation-plate
			  thecum
			  flower
			  root
			  leaf
			  stem
			  plant-subcellular-part
			      ))

;;; defines objects which if a candidate genl's up to it, the
;;; candidate should be thrown away -- even if the candidate
;;; goes up to other ``good'' stuff, e.g., moth_pollinated_flower
(defparameter *bad-object-tops* '(flower))


;;; defines which process concepts in the KB whose progeny should
;;; be considered for choice
(defparameter *process-tops* '(acquisition biosynthesis chemical-reaction
			       degeneration destruction development
			       energy-transduction formation generation 
			       growth maturation removal reproduction
			       reproductive-fertilization storage
			       transportation))



;;; defines processes which if a candidate genl's up to it, the
;;; candidate should be thrown away -- even if the candidate
;;; goes up to other ``good'' stuff, e.g., cell-cycle should
;;; be tossed because genl's up to complex-event
(defparameter *bad-process-tops* '(complex-event))


;-----------------------------------------------------------------------
;			   Concept Choice Functions
;-----------------------------------------------------------------------
;;; randomly chooses a collection of concepts from the KB
;;;
;;; given:
;;;       - number-to-choose        : number of concepts to choose
;;;       - concept-type            : one of: object, process
;;;       - include-embedded-units? : flag for whether embedded units
;;;                                   should be included; default is nil
;;;       - include-elementary?     : flag for whether elementary units
;;;                                   should be included; default is nil
(defun choose-random-concepts (number-to-choose
			       concept-type
			       &optional
			       (include-embedded-units? nil)
			       (include-elementary? nil))
  (when (legal-evaluation-arguments number-to-choose concept-type)
    (let ((concept-list nil)
	  (concept-list-length 0))
      (do* ((new-concept (choose-concept concept-type
					 include-embedded-units?
					 include-elementary?)
			 (choose-concept concept-type
					 include-embedded-units?
					 include-elementary?)))
	   ((equal concept-list-length number-to-choose)
	    concept-list)
	(when (and new-concept
		   (not (member new-concept concept-list))
		   ;; this line special for generating more new concepts
		   (not (member new-concept *existing-process-list*)))
	  (push new-concept concept-list)
	  (setf concept-list-length (1+ concept-list-length))
	  (format t "Adding ~a to concept-list.~%" new-concept)))
      concept-list)))


(defun legal-evaluation-arguments (number-to-choose concept-type)
  (cond ((and (member concept-type '(object process))
	      (numberp number-to-choose)
	      (< number-to-choose 500))
	 t)
	(t (format t "Illegal arguments for evaluation.~%")
	   (values))))


;;; a candidate passes the test if
;;;     (1) it genl's up to something in TOPS
;;;     (2) either
;;;                (a) it does not genl up to something in BAD
;;;                (b) or if it does, it is actually in BAD
;;; note: BAD is itself not bad; only it progeny are bad,
;;;       e.g., flower, which is OK, but Moth_Pollinated_Flower isn't ok
(defun choose-concept (concept-type include-embedded-units? include-elementary?)
  (let* ((candidate-concept (random-mem *kb-objects*))
	 (candidate-passes-type-test?
	  (case concept-type
	    ((object) (and
		       (intersection (ancestry* candidate-concept)
				     *object-tops*)
		       (or (null (intersection (ancestry* candidate-concept)
					       *bad-object-tops*))
			   (member candidate-concept
				   *bad-object-tops*))))
	    ((process) (and
			(intersection (ancestry* candidate-concept)
				      *process-tops*)
			(or (null (intersection (ancestry* candidate-concept)
						*bad-process-tops*))
			    (member candidate-concept
				    *bad-process-tops*)))))))
    (cond ;;checks to ensure of correct type
          ((not candidate-passes-type-test?)
	   nil)
	  ;; checks for embedded (path) concepts
	  ((and (not include-embedded-units?)
		(listp candidate-concept))
	   nil)
	  ;; checks for elementary concepts
	  ((and (not include-elementary?)
		(elementary-at-all? candidate-concept))
	   nil)
	  ;; checks for empty concept
	  ((empty-concept candidate-concept)
	   nil)
	  ;; checks for viewpoints
	  ((is-a-viewpoint-p candidate-concept)
	   nil)
	  ;; checks for processes that are too high
	  ((and (equal concept-type 'process)
		(member candidate-concept *process-tops*))
	   nil)
	  (t candidate-concept))))

    
;;; randomly chooses a member of a list
;;;
;;; function definition appears in Norvig's book, p. 322

(defun random-mem (given-list)
  (nth (random (length (the list given-list)))
       given-list))


;;; returns t iff concept is elementary 
;;;           or if the concept has an elementary all-spec
;;;
;;; the second condition includes concepts that are higher
;;; up in the KB but might not have been marked explicitly as elementary
;;;
;;; rationale: if a concept has a spec that is elementary, then it is
;;;            sufficiently high up in the KB to be considered elementary

(defun elementary-at-all? (concept)
  (or (is-elementary-p concept)
      (has-elementary-specs concept)))


;;; returns t iff the concept has an elementary concept in its progeny
;;;
;;; KB problem with leaf (can't take its progeny*) ---> 
;;;                                   that why the leaf hack is there
(defun has-elementary-specs (concept)
  (let* ((progeny (if (equal concept 'leaf)
		      '(leaf)
		      (progeny* concept)))
	 (elementary-specs (find-if #'(lambda (x)
					(and x
					     (is-elementary-p x)))
				       progeny)))
    (if elementary-specs
	t)))


;;; returns non-nil if candidate concept is completely lacking 
;;;         interesting facts
(defun empty-concept (concept)
  (let* ((slots-with-values (all-explicit-slots concept))
	 (uninteresting-slots (get-local '(non-domain-slot-list
					   non-domain-slots)))
	 (other-uninteresting-slots '(generalizations))
	 (all-uninteresting-slots (append uninteresting-slots
					  other-uninteresting-slots))
	 (interesting-slots (set-difference slots-with-values
					    all-uninteresting-slots)))
    ;(format t "Interesting-slots: ~a~%" interesting-slots)
    (null interesting-slots)))


;;; returns non-nil if candidate concept is a viewpoint
;;;
;;; note: since some viewpoints are (for some reason) not
;;; instances of viewpoint, we need to use this grosser
;;; test
(defun is-a-viewpoint-p (concept)
  (get-local (list concept 'viewpoint-of)))


;-----------------------------------------------------------------------
;			      Collect-Vps-To-Test
;-----------------------------------------------------------------------

;;; given: an exposition node
;;;
;;; adds all of the filtered linearized viewpoints to nlg-test-specs


(defun collect-vps-to-test (exposition-node)
  (let ((linearized-leaves (get-local (list exposition-node
					    'linearized-leaves))))
    (dolist (paragraph linearized-leaves)
      (when (listp paragraph)
	(dolist (viewpoint paragraph)
	  (when (viewpoint? viewpoint)
	    (add-val-local '(nlg-test-specs instances)
			   viewpoint)))))))


;-----------------------------------------------------------------------
;			  Make-NLG-Test-Specs-Report
;-----------------------------------------------------------------------

;;; prints report on specifications (viewpoints) in NLG-Test-Specs
(defun make-nlg-test-specs-report ()
  (collect-domain-relations-to-represent)
  (collect-spec-types-and-examples)
  (values))


;;; prints out domain relations to be represented in lexicon
(defun collect-domain-relations-to-represent ()
  (let* ((relations-list (collect-domain-relations-to-represent-aux))
	 (actor-relations (remove-if-not #'actor-slot-p
					 relations-list))
	 (non-actor-relations (set-difference relations-list
					      actor-relations)))
    (format t "~%~%Relation Lists for NLG-Test-Specs~%")
    (format t "---------------------------------~%~%")
    (format t "Relations for PDTs: ")
    (pprint actor-relations t)
    (format t "~%~%Relations for FD-Skeletons and Make-Np: ")
    (pprint non-actor-relations t)
    (values)))


;;; goes through nlg-test-specs to collect list of domain relations
(defun collect-domain-relations-to-represent-aux ()
  (let ((specs-to-test (get-local '(nlg-test-specs instances)))
	(relations-list nil))
    (dolist (spec specs-to-test)
      (setf relations-list (append (collect-all-domain-relations spec)
				   relations-list)))
    (remove-duplicates relations-list)))


;;; goes through a unit (including embedded units) to find all
;;; domain relations
(defun collect-all-domain-relations (unit)
  (let* ((all-slots (all-explicit-slots unit))
	 (vp-slots (get-local '(viewpoint-slots specializations)))
	 (non-domain-slots (get-local '(non-domain-slot-list
					non-domain-slots)))
	 (other-slots-to-avoid '(generalizations instance-of i-genls))
	 (good-slots (set-difference all-slots
				     (append vp-slots
					     non-domain-slots
					     other-slots-to-avoid)))
	 (relations-list (if (viewpoint? unit)
			     (collect-relations-on-vp-slots unit))))
    (dolist (curr-slot good-slots)
      (setf relations-list (cons curr-slot relations-list))
      (dolist (curr-value (get-local (extend-address unit curr-slot)))
	(let ((nested-address
	       (extend-address-indefinitely unit
					    curr-slot
					    curr-value)))
	  (when (get-substructure nested-address)
	    ;; address has annotations, so collected the nested relations
	    (setf relations-list
		  (append (collect-all-domain-relations nested-address)
			  relations-list))))))
    (remove-duplicates relations-list)))


;;; collects relations from viewpoint slots on a viewpoint
;;;
;;; assumes given unit is a viewpoint
(defun collect-relations-on-vp-slots (unit)
  (let ((spec-type (get-only-val (list unit 'specification-type))))
    (case spec-type
      ((core-connection core-connection-rev super-structural-connection)
       (collect-relations-on-chain unit spec-type))
      (t
       (get-local (list unit 'part-slot-for-view))))))


;;; collects relations that occur on a chain
(defun collect-relations-on-chain (viewpoint spec-type)
  (let* ((slot-of-interest
	  (case spec-type
	    ((core-connection core-connection-rev) 'connection-to-core)
	    ((super-structural-connection)         'super-parts-chain)))
	 (chain (get-only-val (list viewpoint slot-of-interest))))
    (remove-duplicates
     (remove 'generalizations
	     (remove-if-not #'slot-p chain)))))


;;; prints out specification types and examples of each that
;;; reside on nlg-test-specs
(defun collect-spec-types-and-examples ()
  (let* ((spec-types (collect-all-spec-types))
	 (spec-type-spec-list-pairs
	  (collect-spec-type-example-list spec-types)))
    (format t "~%~%Specification Types and Examples on NLG-Test-Specs~%")
    (format t "--------------------------------------------------~%~%")
    (dolist (spec-type-spec-list-pair spec-type-spec-list-pairs)
      (let ((spec-type (first spec-type-spec-list-pair))
	    (spec-list (second spec-type-spec-list-pair)))
	(format t "~a: " spec-type)
	(pprint spec-list t)
	(format t "~%~%")))
    (values)))


;;; collects all specification types that are represented by some
;;; specification on nlg-test-specs
(defun collect-all-spec-types ()
  (let ((viewpoints (get-local '(viewpoint instances)))
	(result-list nil))
    (dolist (vp viewpoints)
      (let ((spec-type (get-only-val (list vp 'specification-type))))
	(when (not (member spec-type result-list))
	  (setf result-list (cons spec-type result-list)))))
    result-list))
    

;;; constructs a list of pairs of the following format from the
;;; specifiations found on nlg-test-specs:
;;;
;;;  (<spec-type> <specification-list>)
;;;
;;; where <specification-list> is a list of all specifications
;;; of type <spec-type> that occur on nlg-test-specs
;;;
(defun collect-spec-type-example-list (spec-types)
  (let ((viewpoints (get-local '(viewpoint instances)))
	(result-list nil))
    (dolist (spec-type spec-types)
      (let* ((specialized-specification-list
	      (remove-if-not
	       #'(lambda (spec)
		   (let ((specification-type
			  (get-only-val (list spec 'specification-type))))
		     (equal specification-type spec-type)))
	       viewpoints))
	     (result-pair (list spec-type specialized-specification-list)))
	(setf result-list (cons result-pair result-list))))
    result-list))



;-----------------------------------------------------------------------
;				  Counts VPs
;-----------------------------------------------------------------------

(defun count-good-vps ()
  (let ((count 0))
    (dolist (vp (get-local '(viewpoint instances)))
      (when (not (viewpoint-contains-error? vp))
	(setf count (1+ count))))
    count))

;-----------------------------------------------------------------------
;			       Test for Loop-It
;-----------------------------------------------------------------------

;;; when Charles' loopit function is run, this code should be inside it
;;; reason: filters out lots of erroneous viewpoints which won't
;;;         be passed to the realizer anyway
;;;
;;; returns: if legit viewpoint        -->   viewpoint
;;;          if problematic viewpoint  -->   nil

(defun viewpoint-looks-good-for-loopit
    (viewpoint)
  (let ((content-spec-node (get-only-val (list viewpoint 'kb-subgraph-of))))
    (if (and content-spec-node
	     (null (content-node-has-error? content-spec-node))
	     (null (viewpoint-contains-error? viewpoint)))
	viewpoint)))


;-----------------------------------------------------------------------
;			   Concept n Random Concepts
;-----------------------------------------------------------------------
;;; randomly chooses n concepts from a given list
;;;
;;; given:
;;;       - number-to-choose        : number of concepts to choose
;;;       - given-concept-list      : list of concepts
;;;
(defun choose-n-random-concepts (number-to-choose
                                 given-concept-list)
  (if (> number-to-choose
	 (length given-concept-list))
      (progn (format t "~%The specified number of concepts is ~
                          to large for the given list.~%")
	     (values))
      (let ((result-concept-list nil)
	    (result-concept-list-length 0))
	(do* ((new-concept (random-mem given-concept-list)
			   (random-mem given-concept-list)))
	     ((equal result-concept-list-length number-to-choose)
	      result-concept-list)
	  (when (not (member new-concept result-concept-list))
	    (push new-concept result-concept-list)
	    (setf result-concept-list-length (1+ result-concept-list-length))
	    (format t "Adding ~a to result list.~%" new-concept)))
	result-concept-list)))
