;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;		     The Template Instantiation Functions
;;;;		     ------------------------------------


;;;; This file contains the template instantiation functions.
;;;; It has 7 sections of functions:
;;;;
;;;;                 - Content Specification Instantiation Function
;;;;                 - Iterate-Over Instantiation Function
;;;;                 - Iteration Condition Instantiation Function
;;;;                 - Focus Condition Instantiation Function
;;;;                 - Generic Template Instantiation Function
;;;;                 - Trace Switches and Accessors
;;;;		     - Instantiation Error Functions


(in-package 'km)


;-----------------------------------------------------------------------
;		      The Content Specification Function
;-----------------------------------------------------------------------


;;; on every EDP node with a template slot will be a slot 
;;;   ``local-variables'' which will be a list of variables that are 
;;;   mentioned in the template slot, along with their associated means 
;;;   of computation via a to-compute
;;;
;;;		- in a template, variables to be instantiated will
;;;               have as the first character of their names a ``?''
;;;
;;;		- for example, a content specification will look like:
;;;
;;;                  content-specifications: (C1 ... Ci ... Cn)
;;;
;;;                         Ci:
;;;                           local-variables: (?v1 ... ?vj ... ?vm)
;;;
;;;                                 ?vj:
;;;                                    to-compute: <function-call>
;;;
;;;                           content-specification-template: (
;;;                                                 <template-expression>)
;;;
;;;		- special variables, i.e., variables which are not
;;;               local, are 
;;;
;;;			- ?primary-concept         : the primary concept
;;;                                                  of in an EDP call
;;;
;;;			- ?current-exposition-node : the exposition node in  
;;;                                                  the explanation plan
;;;                                                  that this EDP is
;;;                                                  being used to construct
;;;
;;;			- ?current-topic-node      : the topic node in the 
;;;                                                  explanation plan
;;;                                                  that this topic in this
;;;                                                  EDP is being used to 
;;;                                                  construct
;;;
;;;			- ?current-T-content-node  : the content node in the 
;;;                                                  explanation plan
;;;                                                  that the current topic
;;;                                                  has generated
;;;
;;;			- ?current-E-content-node  : the content node in the 
;;;                                                  explanation plan
;;;                                                  that the current 
;;;                                                  elaboration has generated
;;;
;;;                     - ?current-elaboration-node: the elaboration node
;;;                                                  that is currently (or
;;;                                                  was last) under
;;;                                                  construction

(defun instantiate-specification-template (content-specification
					   content-node
					   &key loop-var-bindings)
  (let ((template (get-only-val (list content-specification
				      'content-specification-template))))
    (instantiate-template template
			  content-specification
			  content-node
			  loop-var-bindings)))


;-----------------------------------------------------------------------
;		    The Iterate-Over Instantiation Function
;-----------------------------------------------------------------------


;;; instantiates the iterate-over template to produce an expression
;;;   which, when evaluated, will produce a list of elements to
;;;   iterate over
;;;
;;; the Loop-Var-Bindings are a list of dotted pairs of Variable-
;;;   Value bindings, which may be nil; if they are not nil, they
;;;   were passed to this function from an outer loop
;;;
;;; at the time that this function is called, a Content Node has not
;;;   yet been created in the Explanation Plan Tree, so the generic
;;;   instantiation functions called by this node cannot store away
;;;   local variable values if they are computed
;;;
;;; the nil in the instantiate-template function call below indicates
;;;   that there is  no Plan-Tree-Node; more specifically, there
;;;   is no Content Node on which computed values of local variables
;;;   may be found or stored

(defun instantiate-iterate-over-template (content-specification
					  loop-var-bindings)
  (let ((iterate-over-template (get-only-val (list content-specification
						   'iterate-over-template))))
    (instantiate-template iterate-over-template
			  content-specification
			  nil
			  loop-var-bindings)))


;-----------------------------------------------------------------------
;		The Iteration Condition Instantiation Function
;-----------------------------------------------------------------------


;;; instantiates the iteration condition of a conditional-iterative
;;;   content specification
;;;
;;; Loop-Var-Bindings are bindings of loop variables in outer loops;
;;;   this list may be nil
;;;
;;; at the time that this function is called, a Content Node has not
;;;   yet been created in the Explanation Plan Tree, so the generic
;;;   instantiation functions called by this node cannot store away
;;;   local variable values if they are computed
;;;
;;; the first nil in the function call below indicates that there is 
;;;   no loop variable binding list
;;;
;;; the second nil in the function call below indicates that there is 
;;;   no Plan-Tree-Node; more specifically, there is no Content Node
;;;   on which computed values of local variables may be found or stored

(defun instantiate-iteration-condition (iteration-condition
					loop-var-bindings
					content-specification)
  (instantiate-template iteration-condition
			content-specification
			nil
			loop-var-bindings))


;-----------------------------------------------------------------------
;		  The Focus Condition Instantiation Function
;-----------------------------------------------------------------------


;;; instantiates Focus-Condition
(defun instantiate-focus-condition (focus-condition
				    edp-unit
				    plan-tree-node)
  (instantiate-template focus-condition
			edp-unit
			plan-tree-node))


;-----------------------------------------------------------------------
;		  The Generic Template Instantiation Function
;-----------------------------------------------------------------------


;;; Given: a template to instantiate
;;;        an edp-unit
;;;        a list of loop variable bindings (may be nil)
;;;        a plan tree node (may be nil)
;;;
;;; if Plan-Tree-Node is nil, then the results of computing
;;;   local variables will not be searched for (nor will
;;;   the resulting bindings be cached on it)
;;;
;;; Searching variables of other nodes:
;;;   the system will sometimes attempt to instantiate a variable
;;;   with a binding that has been made to it at a higher level
;;;   in the Explanation Plan Tree; it will do this *only* in the 
;;;   following situation:
;;;
;;;        (1) it has ruled out all special variables, and
;;;        (2) the variable wasn't found among the loop variables, and
;;;        (3) the Plan-Tree-Node is not nil
;;;
;;;   if it fails to find the variable, it will continue searching 
;;;   upward in the tree until it either finds the variable or it 
;;;   runs out of nodes to examine in the tree (indicated by 
;;;   encountering a nil value for child-of-node, meaning the top of 
;;;   the Explanation Plan Tree has been reached)
;;;
;;;   note that the system currently does not invoke to-computes
;;;   of variables that occur at higher nodes; it simply examines
;;;   the nodes in the explanation plan to determine if a value
;;;   for the variable of interest has been previously computed
;;;
;;;   the system substitutes t for true and nil for false to
;;;   aid the evaluation of the statement later
;;;
;;; the real work is done by the function instantiate-template-aux;
;;;    the top-level function instantiate-template is used to return 
;;;    nil (as opposed to a partially instantiated template) in the
;;;    case of an instantiation error

(defun instantiate-template (template edp-unit 
		             &optional plan-tree-node loop-var-bindings)
  "Generic template instantiation function"

  (when (trace-instantiation?)
    (format t "Attempting to instantiate template")
    (pprint template t)
    (format t "~%in edp-unit ~a.~%~%"
	    edp-unit)
    (when (not (null plan-tree-node))
      (format t "The instantiator was given plan tree node ~a.~%~%"
	      plan-tree-node))
    (when (not (null loop-var-bindings))
      (format t "It was given loop variable bindings:~% ~a.~%~%"
	      loop-var-bindings)))
  (let ((result (instantiate-template-aux template
					  edp-unit 
					  plan-tree-node
					  loop-var-bindings)))
    (when (trace-instantiation?)
      (format t "The resulting instantiated template is")
      (pprint result t)
      (format t "~%~%"))
    (cond ((null (instantiation-error?)) result)
	  (t (switch-off-instantiation-error)))))


(defun instantiate-template-aux (template edp-unit 
 			         &optional plan-tree-node loop-var-bindings)
  (cond ((atom template)
	 (cond ((and (symbolp template)
		     (char= (schar (symbol-name template) 0)
			    #\?))
		(instantiate-template-var template
					  edp-unit  
					  plan-tree-node
					  loop-var-bindings))
	       ((and (symbolp template)
		     (equal template 'true))
		t)
	       ((and (symbolp template)
		     (equal template 'false))
		nil)
	       (t template)))
	((null template) nil)
	(t (cons (instantiate-template-aux (first template)
					   edp-unit
					   plan-tree-node
					   loop-var-bindings)
		 (instantiate-template-aux (rest template)
					   edp-unit
					   plan-tree-node
					   loop-var-bindings)))))
	     

;;; Template Variable Instantiation Method:
;;;
;;;			1. Determine if it is a special variable 
;;;
;;;			2. if it is a special variable, instantiate it
;;;                        with the appropriate value
;;;
;;;			3. if it is not a special variable, then
;;;                        determine if it is a loop variable as
;;;                        follows:
;;;
;;;                             a. assoc of it on Loop-Var-Bindings
;;;
;;;                             b. return cdr of assoc
;;;
;;;                     4. if it was not a loop variable either, then
;;;                        check if it it is a local variable:
;;;
;;;                             a. first check if its value has
;;;                                already been computed by
;;;				   examining list of
;;;                                local-variable-bindings on
;;;                                Plan-Tree-Node (if node was given)
;;;
;;;                             b. if not, check if there is a
;;;                                to-compute for it on the EDP unit
;;;
;;;                             c. if there is a to-compute, apply it
;;;
;;;                     5. if it is not a local variable on the EDP,
;;;                        search upward in the Explanation Plan
;;;                        Tree until either find the variable
;;;                        or run out of nodes to examine in the
;;;                        tree (indicated by encountering a nil 
;;;                        value for child-of-node, meaning the top
;;;                        of the Explanation Plan Tree has been
;;;                        reached)           
;;;
;;;                        (assumes function ``search-upward-for-non-local''
;;;                         returns nil if it is unable to find a value)
;;;
;;; error results: returns nil if fails to instantiate variable

(defun instantiate-template-var (template-var edp-unit
				 &optional plan-tree-node loop-var-bindings)
  "Instantiates a template variable"

  (when (trace-instantiation-details?)
    (format t "Attempting to instantiate ~a ~%in EDP node ~a.~%~%"
	    template-var
	    edp-unit))

  ;; first check if variable is special
  (let ((special-var-binding (check-special-vars template-var)))
    (if (not (null special-var-binding)) special-var-binding

	;; then check if variable is a loop variable
	(let ((possible-loop-binding
	       (assoc template-var loop-var-bindings)))
	  ;(format t "Just checked possible loop bindings.~%~%")
	  (if (not (null possible-loop-binding))
	      (progn (when (trace-instantiation-details?)
		       (format t "The variable ~a is a loop variable with~%~
                                  with value ~a.~%~%"
			       template-var
			       (cdr possible-loop-binding)))
		     (cdr possible-loop-binding))

	      ;; then check if binding exists on plan-tree-node
	      (if (not (null plan-tree-node))
		  (let ((local-binding
			 (assoc template-var
				(get-local
				 (list plan-tree-node
				       'local-variable-bindings)))))
		    ;(format t "Just checked local variable bindings~%~%")
		    (if (not (null local-binding))
			(progn (when (trace-instantiation-details?)
				 (format t "The variable ~a is an existing ~
                                            local variable~%on ~a ~
                                            with value ~a.~%~%"
					 template-var
                                         plan-tree-node
					 (cdr local-binding)))
			       (cdr local-binding))

			;; then check if there is a to-compute for it
			(let ((to-compute-template
			       (get-to-compute-for-local-var template-var
							     edp-unit)))
			  (if (not (null to-compute-template))
			      (evaluate-computable-local-var
			                               to-compute-template
			                               template-var
						       edp-unit
						       plan-tree-node
						       loop-var-bindings)

			      ;; otherwise search upward in Plan
			      (let ((value-of-var 
				     (search-upward-for-non-local
				      template-var plan-tree-node)))
				(if (not (null value-of-var))
				    value-of-var
				    (signal-instantiation-error
				           template-var
				           edp-unit
				           plan-tree-node)))))))

		  ;; if no plan tree node, check for to-computes
		  (let ((to-compute-template
			 (get-to-compute-for-local-var template-var
						       edp-unit)))

		    (when (and (trace-instantiation-details?)
			       (not (null to-compute-template)))
		      (format t "The to-compute template for ~
                                 template variable ~a is:~%"
			      template-var)
		      (pprint to-compute-template t)
		      (format t "~%~%"))

		    (if (not (null to-compute-template))
			(evaluate-computable-local-var
			 to-compute-template
			 template-var
			 edp-unit
			 plan-tree-node
			 loop-var-bindings)
			
			;; otherwise search upward in Plan
			(let ((value-of-var 
			       (search-upward-for-non-local
				template-var plan-tree-node)))
			  (if (not (null value-of-var))
			      value-of-var
			      (signal-instantiation-error
			       template-var
			       edp-unit
			       plan-tree-node)))))))))))


	
;;; checks if template variable is a special variable
;;; if so, the value of the special variable is returned
;;; otherwise returns nil

(defun check-special-vars (template-var)
  "Looks through special variables for value of template variable"
  (let ((result 
	 (case template-var
	   (?primary-concept
	    (get-only-val (list (current-exposition-node)
				'primary-concept)))
	   (?current-exposition-node (current-exposition-node))
	   (?current-topic-node (current-topic-node))
	   (?current-t-content-node (current-t-content-node))
	   (?current-e-content-node (current-e-content-node))
	   (?current-elaboration-node (current-elaboration-node)))))
    (when (trace-instantiation-details?)
      (if (null result)
	  (format t "The variable ~a is not a special variable.~%~%"
		  template-var)
	  (format t "The variable ~a is a special variable ~%~
                     with value ~a.~%~%"
		  template-var
		  result)))
    result))


;;; searches for to-compute expression for template-var on edp-unit
;;; returns expression if exists
;;; otherwise returns nil
;;;
;;; assumes that if a variable appears on the local variable list of
;;;    an edp-unit, then there is a to-compute expression associated 
;;;    with it

(defun get-to-compute-for-local-var (template-var edp-unit)
  "Searches for to-compute template for local variable"
  (let* ((local-vars (get-local (list edp-unit 'local-variables)))
	 (result (if (member template-var local-vars)
		     (get-only-val (list edp-unit
					 'local-variables
					 template-var
					 'to-compute)))))
    (when (and (trace-instantiation-details?)
	       (null result))
      (format t "The variable ~a has no to-compute expression~%~
                 on ~a.~%~%"
	      template-var
	      edp-unit)
      (format t "The to-compute expression for ~a~%on ~a is:"
	      template-var
	      edp-unit)
      (pprint result t)
      (format t "~%~%"))
    result))


;;; evaluates a local variable template-var with to-compute-template
;;;      in edp-unit for building plan-tree-node
;;;
;;; method:
;;;
;;;     (1) instantiates to-compute template
;;;
;;;     (2) evaluates it
;;;
;;;     (3) if Plan-Tree-Node is not nil, then the value is stored
;;;
;;;     (4) returns value or, if failed to instantiate variable, returns nil
;;;
;;; instantiating to-compute-template may involve evaluating
;;;    the to-computes of local variables; these to-computes must 
;;;    be interpreted by KNIGHT, not by KM, which doesn't know how 
;;;    to correctly instantiate special variables mentioned in the 
;;;    to-compute, such as ``?current-topic-node''; this may involve
;;;    calling this function recursively; the recursion bottoms out 
;;;    when either
;;;         (1) a function only mentions special variables, or
;;;         (2) a function has values for all local variables
;;;                already computed

(defun evaluate-computable-local-var (to-compute-template
				      template-var
				      edp-unit
				      plan-tree-node
				      loop-var-bindings)
  "Evaluates local variable with to-compute"
  (let ((to-compute-instantiation
	 (instantiate-template-aux to-compute-template 
				   edp-unit
				   plan-tree-node
				   loop-var-bindings)))
    (when (trace-instantiation-details?)
      (format t "The instantiation of the to-compute expression for ~a~%~
                 on ~a is:"
	      template-var
	      edp-unit)
      (pprint to-compute-instantiation t)
      (format t "~%~%"))
    (if (not (null to-compute-instantiation))
	(let ((value-of-var (eval to-compute-instantiation)))

	  (when (trace-instantiation-details?)
	    (format t "The value of the instantiated expression for ~a~%~
                       on ~a is: ~a.~%~%"
		    template-var
		    edp-unit
		    value-of-var))
	  ;; install computed value in plan-tree-node		      
	  (if (not (null plan-tree-node))
	      (add-val (list plan-tree-node 'local-variable-bindings)
		       (list template-var value-of-var)))
	  value-of-var)

	;; found to-compute, but couldn't instantiate it
	(signal-instantiation-error template-var
				    edp-unit
				    plan-tree-node
				    'to-compute-problem))))


;;; searches upwards for value of V in Explanation Plan Tree 
;;;
;;; search begins at Plan-Tree-Node
;;;
;;; calls self recursively on parent of Plan-Tree-Node if
;;;   cannot find it in current Plan-Tree-Node
;;;
;;; if fails returns nil

(defun search-upward-for-non-local (v plan-tree-node)
  "Searches upward in Explanation Plan Tree for value of local variable"
  (if (not (null plan-tree-node))
      (let ((possible-binding (assoc v (get-local
					(list plan-tree-node
					      'local-variable-bindings)))))
	(when (trace-instantiation-details?)
	  (format t "Considering bindings on node ~a~%~%" plan-tree-node)
	  (format t "Bindings are ~a~%~%" (get-local
					   (list plan-tree-node
						 'local-variable-bindings))))
	(if (not (null possible-binding))
	    (cdr possible-binding)
	    (search-upward-for-non-local v (get-only-val
					    (list plan-tree-node
						  'child-of-node)))))))


;-----------------------------------------------------------------------
;		  The Instantiation Error Functions
;-----------------------------------------------------------------------

;;; instantiation-error? is a slot on the KB frame Knight-Global-State


(defun switch-on-instantiation-error ()
  (put-local '(knight-global-state instantiation-error?)
	     'true))


(defun switch-off-instantiation-error ()
  (put-local '(knight-global-state instantiation-error?)
	     'false))


(defun instantiation-error? ()
  (equal (get-only-val '(knight-global-state instantiation-error?))
	 'true))


;;; prints instantiation error message
(defun signal-instantiation-error (template-var edp-unit plan-tree-node
				   &optional to-compute?)
  (switch-on-instantiation-error)
  (when (trace-instantiation?)
    (format t "~%Could not instantiate variable ~a~%~
                 in EDP node ~a~%~
                 when constructing Explanation Plan node ~a.~%~%"
	  template-var
	  edp-unit
	  plan-tree-node)
    (if (equal to-compute? 'to-compute-problem)
	(format t "Reason: Could not properly instantiate ~
                 ~a's to-compute.~%~%" template-var)))
  nil)




