;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       
				       
;;;;		    The Knowledge Base Filtering Functions
;;;;		    --------------------------------------



;;;; given: a content-node and its associated viewpoint
;;;;
;;;; filters content node in three ways, then checks if the viewpoint
;;;; has become emptied
;;;;
;;;; (1) filters (signals errors on) viewpoints whose concept of
;;;;     interest is a path
;;;;
;;;; if an error was not signalled on the viewpoint, then
;;;;
;;;;     (2) filters triples with concepts which are marked as unspeakable
;;;;
;;;;     (3) if user modeling is on, filters known concepts from the
;;;;         viewpoint and adds triples that will be communicated to
;;;;         the user model
;;;;
;;;;     finally,
;;;;
;;;;     (4) checks if the filtering process has removed the final
;;;;         bits of information from the viewpoint, leaving only
;;;;         a ``shell;'' if so, signals an error in the content node
;;;;
;;;; returns: the main result is the side-effects of filtering the
;;;;          given viewpoint


(in-package 'km)


;-----------------------------------------------------------------------
;			 Top-Level Filtering Function
;-----------------------------------------------------------------------

(defun filter-content (content-node)
  (when (not (content-node-has-error? content-node))

    (let ((viewpoint (get-only-val (list content-node 'kb-subgraph))))

      (when (not (viewpoint-contains-error? viewpoint))

	(cond

	  ;; check if filtering paths from viewpoints
	  ((and (filter-path-viewpoints?)
		(concept-of-interest-is-path? viewpoint))

	   ;; filter one if encountered
	   (signal-kb-access-error content-node
				   'concept-of-interest-is-path)
	   )

	  (t ;; otherwise do all the other filtering


	   ; function below calls code the Charles' code takes care of
	   ;(filter-overly-general-concepts viewpoint)

	   (when (user-modeling-on?)
	     (filter-known-concepts-and-update-user-model viewpoint))

	   (when (filter-path-values?)
	     (filter-path-values viewpoint))

	   (when (empty-view-p viewpoint)
	     (signal-kb-access-error
	      content-node
	      'kb-filtering-removed-essential-information)))
	  ))
      )))


(defun filter-path-viewpoints? ()
  (equal (get-only-val '(knight-global-state filter-path-viewpoints?))
	 'true))

(defun concept-of-interest-is-path? (viewpoint)
  (let ((coi (get-only-val (list viewpoint 'viewpoint-of))))
    (and coi (listp coi))))


;-----------------------------------------------------------------------
;		       Overly General Concept Filtering
;-----------------------------------------------------------------------
;;;
;;; filters concepts that are too general and should not be included
;;; in a natural language translation of the viewpoint
;;;
;;; the with-no-prompts call is used to prevent KM from calling
;;; KnEd and issuing a window prompt when remove-val is called
;;; on a value that has annotations
;;;
;;; some slots should not be filtered on particular kinds of viewpoints
;;; these are all included in the ``slots-special-not-to-filter''
;;; variable, e.g., 
;;;
;;;     if have as-kind-of viewpoint
;;;        if concept of interest is an object
;;;           do not filter values of as-kind-of slot
;;;               --> put as-kind-of in slots-special-not-to-filter
;;;        if concept of interest is a process
;;;           do filter values of as-kind-of slot


(defun filter-overly-general-concepts (viewpoint)
  (with-no-prompts
      (let* ((spec-type
	      (get-only-val (list viewpoint 'specification-type)))

	     (slots-always-not-to-filter
	      '(generalizations kb-subgraph-of viewpoint-of))

	     (slots-special-not-to-filter
	      (case spec-type
		(black-box-process-description '(reference-concept))
		(t                             nil                 )
		))

	     (all-slots-not-to-filter (append slots-always-not-to-filter
					      slots-special-not-to-filter))
	     )
	(case spec-type
	  ((core-connection core-connection-rev super-structural-connection)
	   (filter-overly-general-concepts-chain viewpoint spec-type))
	  (t (filter-overly-general-concepts-aux viewpoint
						 all-slots-not-to-filter))))))


(defun filter-overly-general-concepts-chain (viewpoint spec-type)
  (let* ((slot-of-interest
	  (case spec-type
	    ((core-connection core-connection-rev) 'connection-to-core)
	    ((super-structural-connection)         'super-parts-chain)))
	 (chain (get-only-val (list viewpoint slot-of-interest)))
	 (overly-general-concept-found? nil))
    (dolist (concept chain)
      (when (is-unspeakable-p concept)
	(setf overly-general-concept-found? t)))
    (when overly-general-concept-found?
      (when (trace-viewpoint-filtering?)
	(format t "Viewpoint filtering:~%Filtered ~a~%~
                   of ~a.~%"
		slot-of-interest
		viewpoint)
	(format t "Reason: Unspeakable concept occurred on chain.~%~%"))

      (put-local (list viewpoint slot-of-interest)
		 (list (remove-chain-bits chain
					  #'is-unspeakable-p))))))


(defun filter-overly-general-concepts-aux (unit all-slots-not-to-filter)
  (let ((slots-to-filter (set-difference (get-explicit-slots unit)
					 all-slots-not-to-filter)))
    
      (dolist (curr-slot slots-to-filter)
	(dolist (curr-value (get-local (extend-address unit
						       curr-slot)))
	  (cond ((is-unspeakable-p curr-value)
		 (when (trace-viewpoint-filtering?)
		   (format t "Viewpoint filtering:~%Filtered ~a~%~
                              on ~a of ~a.~%"
			   curr-value
			   curr-slot
			   unit)
		   (format t "Reason: Unspeakable concept.~%~%"))
		 (remove-val (extend-address unit curr-slot)
			     curr-value))
		(t (let ((nested-address
			  (extend-address-indefinitely unit
						       curr-slot
						       curr-value)))
		     (when (get-substructure nested-address)
		       ;; address has annotations, so filter them too
		       (filter-overly-general-concepts-aux
			nested-address
			all-slots-not-to-filter)))))))))


;-----------------------------------------------------------------------
;			     User Model Filtering
;-----------------------------------------------------------------------
;;;
;;; There are two primary functions which need to be accomplished:
;;;
;;;        - Filtering: Given a viewpoint, remove those triples
;;;                     (and their annotations) that have
;;;                     already been discussed
;;;
;;;        - Updating:  Given a viewpoint, update the UM to indicate
;;;                     that the triples in it have been discussed
;;; given: a viewpoint
;;;
;;; if the viewpoint is a standard viewpoint, then the viewpoint
;;; is filtered for triples that contain known concepts, and the
;;; user model is updated with concepts that will be discussed
;;;
;;; if viewpoint is one whose main content is a specially constructed
;;; list structure, e.g. connect-to-core, then the user model
;;; is updated with each of the triples that compose the chain

(defun filter-known-concepts-and-update-user-model (viewpoint)
  (with-no-prompts
      (let ((spec-type (get-only-val (list viewpoint 'specification-type))))
	(case spec-type
	  ((core-connection core-connection-rev super-structural-connection)
	   (update-um-with-list-structure viewpoint spec-type))
	  ((sub-structural-description)
	   (update-um-with-sub-structural-view viewpoint))
	  (t
	   (filter-known-concepts-and-update-um-standard viewpoint))))))


(defun update-um-with-list-structure (viewpoint spec-type)
  (let* ((slot-of-interest
	  (case spec-type
	    ((core-connection core-connection-rev) 'connection-to-core)
	    ((super-structural-connection)         'super-parts-chain)))
	 (chain (get-only-val (list viewpoint slot-of-interest)))
	 (triples-list (break-chain-into-triples chain)))
    (dolist (triple triples-list)
      (add-um-triple (list (first triple)
			   (second triple)
			   (third triple))))))


;;; breaks a chain into its composite triples
;;;
;;; example: (a b c d e) --> ((a b c) (c d e))
	     
(defun break-chain-into-triples (chain)
  (if (and chain
	   (not (equal (length chain) 1)))
      (cons (list (first chain)
		  (second chain)
		  (third chain))
	    (break-chain-into-triples (rest (rest chain))))))


(defun update-um-with-sub-structural-view (viewpoint)
  (let* ((concept-of-interest (get-only-val (list viewpoint 'viewpoint-of)))
	 (part-slot (get-only-val (list viewpoint 'part-slot-for-view)))
	 (parts (get-local (list viewpoint 'part-values))))
    (dolist (part parts)
      (add-um-triple (list concept-of-interest
			   part-slot
			   part)))))

      
(defun filter-known-concepts-and-update-um-standard (viewpoint)
  (let ((concept-of-interest (get-only-val (list viewpoint 'viewpoint-of))))

    (dolist (curr-slot (set-difference
			(get-explicit-slots viewpoint)
			(append (get-local '(viewpoint-slots specializations))
				'(generalizations kb-subgraph-of
				  instance-of focused-concept))))
      (dolist (curr-value (get-local (extend-address viewpoint
						     curr-slot)))
	(cond ((is-known-p (list concept-of-interest
				 curr-slot
				 curr-value))

	       ;; triple is known so remove it
	       (when (trace-viewpoint-filtering?)
		 (format t "Viewpoint filtering:~%Filtered ~a~%on ~a.~%"
			 (list (get-only-val (list viewpoint 'viewpoint-of))
			       curr-slot
			       curr-value)
			 viewpoint)
		 (format t "Reason: Known triple.~%~%"))
	       (remove-val (extend-address viewpoint curr-slot)
			   curr-value))
		
	      ;; triple is now known so do not remove it
              ;; but include it in viewpoint and then
	      ;; add it to the user model
	      (t (add-um-triple (list concept-of-interest
				      curr-slot
				      curr-value))))))))


;-----------------------------------------------------------------------
;		       Filters Embedded Units
;-----------------------------------------------------------------------

;;; For some cases in testing, especially until we get the NLG system
;;; to work for paths, we need a mechanism for filtering embedded units.
;;;
;;; The predicate and filter in this section performs these functions.

;;; predicate for testing whether embedded unit filtering is on

(defun filter-path-values? ()
  (equal (get-only-val '(knight-global-state filter-path-values?))
	 'true))


;;;filters embedded units that occur as values of slots on viewpoints
;;;
;;; the with-no-prompts call is used to prevent KM from calling
;;; KnEd and issuing a window prompt when remove-val is called
;;; on a value that has annotations
;;;
;;; below we are careful not to filter values of viewpoint slots
;;; which are special cases; these slots have values that are not
;;; paths in the traditional sense; instead they are values which
;;; are ordered lists, such as the results of a connect-to-core
;;; specification; these types of units need special treatment

(defun filter-path-values (viewpoint)
  (with-no-prompts
      (let* ((spec-type (get-only-val (list viewpoint 'specification-type))))
	(case spec-type
	  
	  ((core-connection
	    core-connection-rev
	    super-structural-connection) (filter-path-values-special
					  spec-type
					  viewpoint))

	  (t                             (filter-path-values-standard
					  viewpoint))))))


;;; recursively works through substructures of unit to filter paths
(defun filter-path-values-standard (unit)
  (let* ((slots-to-filter (set-difference (get-explicit-slots unit)
					  '(generalizations
					    i-genls
					    i-specs
					    kb-subgraph-of
					    viewpoint-of))))
      (dolist (curr-slot slots-to-filter)
	(dolist (curr-value (get-local (extend-address unit
						       curr-slot)))
	  (cond ((listp curr-value)
		 (when (trace-viewpoint-filtering?)
		   (format t "Viewpoint filtering:~%Filtered ~a~%~
                              on ~a of ~a.~%"
			   curr-value
			   curr-slot
			   unit)
		   (format t "Reason: Path value.~%~%"))
		 (remove-val (extend-address unit curr-slot)
			     curr-value))
		(t (let ((nested-address
			  (extend-address-indefinitely unit
						       curr-slot
						       curr-value)))
		     (when (get-substructure nested-address)
		       ;; address has annotations, so filter them too
		       (filter-path-values-standard
			nested-address)))))))))


;;; works through list value of special viewpoints such as
;;; substructures of unit to filter paths
;;;
;;; after finding the chain, filters out all slots from the chain
;;; and then checks the units that remain
;;;
;;; if one or more of the units in the chain are paths, the
;;; entire chain is deleted from the slot on the viewpoint, thereby
;;; effectively leaving the viewpoint empty
(defun filter-path-values-special (spec-type viewpoint)
  (let* ((slot-of-interest
	  (case spec-type
	    ((core-connection core-connection-rev) 'connection-to-core)
	    ((super-structural-connection)         'super-parts-chain)))
	 (chain (get-only-val (list viewpoint slot-of-interest)))
	 (units-in-chain (remove-if #'slot-p chain))
	 (path-unit-found-in-chain? nil))

    ;(format t "Units-in-chain: ~a~%" units-in-chain)

    (dolist (unit units-in-chain)
      (when (listp unit)
	(setf path-unit-found-in-chain? t)))

    (when path-unit-found-in-chain?

      (when (trace-viewpoint-filtering?)
	(format t "Viewpoint filtering:~%Filtered ~a~%~
                   of ~a.~%"
		slot-of-interest
		viewpoint)
	(format t "Reason: Path value occurred on chain.~%~%"))

      (put-local (list viewpoint slot-of-interest)
		 (list (remove-chain-bits chain
					  #'listp))))))


;-----------------------------------------------------------------------
;		       Removing Bits from Chains Utility
;-----------------------------------------------------------------------
;;; given: a chain (a b c d e) of length at least 3
;;;        a predicate
;;;  
;;;
;;; if don't find member of chain that satisfies predicate then
;;; return chain
;;;
;;; for each member that satisfies the predicate then
;;;    if offending member is in the first position then
;;;       return nil else:
;;;              alter chain to remove offender and its predecessor (a slot)
;;;              if length of resulting chain is less than 3 then
;;;                 return nil else
;;;                 return chain
;;;
;;; example: (remove-chain-bits '(a b c d e f g)
;;;                             #'(lambda (x) (equal x 'd)))
;;;          --> (a b e f g)
;;;
;;; note: VERY unelegant, but had to write it quickly
;;;
;;; use: call remove-chain-bits with chain #'unspeakable 

(defun remove-chain-bits (chain predicate)
  (let ((altered-chain (remove-chain-bits-aux chain predicate)))
    (cond ((null altered-chain)
	   nil)
	  ((equal chain altered-chain)
	   chain)
	  (t (remove-chain-bits altered-chain predicate)))))


(defun remove-chain-bits-aux (chain predicate)
  (let ((chain-length (length chain)))
    (let ((bad-bit-position (position-if #'(lambda (x)
					     (funcall predicate x))
					 chain)))
      (if (numberp bad-bit-position)
	  (if (and (not (zerop bad-bit-position))
		   (not (equal bad-bit-position 1)))
	      (let* ((slot-before-bad-bit-position (1- bad-bit-position))
		     (altered-chain
		      (append
		       (subseq chain
			       0
			       slot-before-bad-bit-position)
		       (subseq chain
			       (1+ bad-bit-position)
			       chain-length))))
		(if (>= (length altered-chain) 3)
		    altered-chain)))
	  chain))))


