;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;			   The Linearizing Functions
;;;;			   -------------------------

;;; This file contains functions that linearize leaves of
;;; of an explanation plan.  It is divided into 5 sections:
;;; 
;;;       - Top-level Linearization Functions
;;;       - Explanation Plan Traversal
;;;       - Paragraph Grouping
;;;       - Paragraph Flattening
;;;       - EDP Organization Checking


;-----------------------------------------------------------------------
;		       Top-Level Linearization Functions
;-----------------------------------------------------------------------


;;; Traversal
;;; ---------
;;; traverses an explanation plan to produce a list of viewpoints
;;; that reside at the leaves of the explanation plan
;;;
;;; Filtering
;;; ---------
;;; filters those viewpoints that are erroneous or incomplete
;;;
;;; if a content node contains an error, neither its associated
;;; viewpoint or its elaboration nodes will be considered for
;;; inclusion
;;;
;;; Paragraphing
;;; ------------
;;; rather than returning a flat list, the list defines the paragraph
;;; structure of the explanation
;;;
;;; for example,
;;;               ((A B) (C) (D E F))
;;; means that viewpoints A and B will be expressed in a single
;;; paragraph, C will be expressed in a paragraph by itself, and
;;; D, E, and F will grouped together into a single paragraph
;;;
;;; method: (1) performs depth-first traversal of explanation plan; at
;;;             each node examines the paragraph-structure slot on the
;;;             EDP-unit that is the ``type'' of that node
;;;
;;;         (2) performs second pass to return embedded list with only
;;;             a single level of embedding (takes care of case where
;;;             a parent EDP unit begins a new paragraph, then a
;;;             descendant of that unit in the EDP begins yet another
;;;             new paragraph; this results in multiple levels of
;;;             embeddings, which has to be reduced to one level of
;;;             embedding)
;;;
;;;          (Optional step) may later add post-processing routines
;;;             that split up large paragraphs into smaller ones; of
;;;             course, this will need to be done very carefully so
;;;             that the first sentence of each new paragraph can serve
;;;             as a topic sentence
;;;
;;; representation of paragraph structure on EDP units:
;;;
;;;      - uses slots ``paragraph-structure'' and
;;;                   ``paragraphize-children?''
;;;    
;;;      - if both slots are empty, then no new paragraph should be begun,
;;;        all of the subtopics should be included in the same paragraph
;;;        as their supertopic
;;;
;;;      - embedded list on ``paragraph-structure'': new paragraph(s) begun
;;;                       example: ((A) (B C)) indicates that a new
;;;                                paragraph should be begun for A,
;;;                                and B and C should be included together
;;;                                in yet another new paragraph
;;;
;;;      - advice:
;;;               - to cause children of node that are produced by
;;;                 iteration to each be in their own paragraph,
;;;                 give the parent node the value ``True'' for
;;;                 the slot ``paragraphize-children?''
;;;
;;;               - to put into separate paragraphs a content
;;;                 specification's viewpoint and its elaborations,
;;;                 add a paragraph-structure value that names the
;;;                 elaborations
;;;
;;;      - rules: (1) the root of every EDP must have a value for either the
;;;                   paragraph-structure slot or the paragraphize-children?
;;;                   slot
;;;
;;;               (2) any unit in the EDP that has subtopics may
;;;                   have a value for the paragraph-structure slot
;;;
;;;               (3) any other unit in the EDP (except for
;;;                   content-specifications) that has subtopics
;;;                   may have a value for the paragraphize-children? slot
;;;    
;;;               (4) any unit that has a value for the paragraph-structure
;;;                   slot must satisfy the following criteria:
;;;                       the order of the units in ``flattened''
;;;                       paragraph-structure list must be the same
;;;                       as the order of the units in the organization
;;;                       specification in that unit, the units mentioned
;;;                       must be exactly the same in 3 lists:
;;;                            - the ``topic-list'' slot,
;;;                            - the ``organization'' slot
;;;                            - the paragraph-structure slot
;;;
;;;               (5) any unit that has non-null values for both its
;;;                   paragraph-structure and paragraphize-children?
;;;                   slots will be assigned the structure specified by
;;;                   the paragraph-structure slot
;;;
;;;       - rule checking: to see if an EDP satisfies all of these
;;;                        run the function ``check-organization''
;;;    
;;; Result
;;; ------
;;; asserts list of viewpoints as value of filtered, paragraphed
;;; ``linearized-leaves'' to slot of given exposition node


(in-package 'km)


(defun linearize-exposition-node (exposition-node)
  (let* ((ordered-explanation-plan-leaves
	  (find-leaves-of-exposition-node exposition-node))
	 (properly-paragraphed-leaves
	  (parag-flatten ordered-explanation-plan-leaves)))
    (put-local (list exposition-node 'linearized-leaves)
	       properly-paragraphed-leaves)))


;-----------------------------------------------------------------------
;			``Explanation Plan Traversal''
;-----------------------------------------------------------------------
;;; algorithm:
;;;              - visit node N in the explanation plan tree
;;;              - at N, get:
;;;                           Child-List          : a list of N's children
;;;              - if Child-List is empty then
;;;                   - return (N)
;;;                else
;;;                   - get: 
;;;                           EDP-Unit            : the corresponding EDP unit
;;;                           Paragraph-Structure : parag-struct of EDP-Unit
;;;                   - if Paragraph-Structure is empty
;;;                        and paragraphize-children? is empty then
;;;                        - return <ordered recursive results of applying
;;;                                  function to each element of Child-List>
;;;                     else if paragraph-structure is non-null then
;;;                        - return <ordered recursive results of applying
;;;                                  function to each element of Child-List,
;;;                                  where results are grouped by 
;;;                                  Paragraph-Structure>
;;;                     else (paragraphize-children? is non-null)
;;;                        - return <ordered recursive results of applying
;;;                                  function to each element of Child-List,
;;;                                  where each Child is assigned its own
;;;                                  paragraph
;;;
;;; assumes: exposition-node has either a paragraph-structure value
;;;                              or     a paragraphize-children? value
;;;          exposition-node has topic-nodes          
;;;
(defun find-leaves-of-exposition-node (exposition-node)
  (let* ((topic-nodes (get-local (list exposition-node
				       'child-nodes-ordered)))
	 (edp-unit (get-only-val (list exposition-node
				       'node-type)))
	 (paragraph-structure (get-local (list edp-unit
					       'paragraph-structure))))

    (if paragraph-structure
	(let ((grouped-topic-nodes (group-nodes topic-nodes
						paragraph-structure)))
	  ;(format t "~%Grouped topic nodes: ~a~%"
	  ;	  grouped-topic-nodes)
	  (remove nil
		  (mapcar #'find-leaves-of-topic-node-group
			  grouped-topic-nodes)))
	;;assumes should paragraphize children, i.e., each child should
	;;be in its own paragraph 	  
	(mapcar #'find-leaves-of-topic-node
		topic-nodes))))


(defun find-leaves-of-topic-node-group (topic-node-group)
  (reduce #'append
	  (mapcar #'find-leaves-of-topic-node
		  topic-node-group)))


(defun find-leaves-of-topic-node (topic-node)
  (let ((content-nodes (get-local (list topic-node
					'child-nodes-ordered))))

    (if (null content-nodes)
	(list topic-node)
	(let* ((edp-unit (get-only-val (list topic-node
					     'node-type)))
	       (paragraph-structure (get-local (list edp-unit
						     'paragraph-structure)))
	       (paragraphize-children (get-only-val
				       (list edp-unit
					     'paragraphize-children?))))
	  (cond ((and (null paragraph-structure)
		      (null paragraphize-children))
		 (find-leaves-of-content-node-group content-nodes))
		(paragraph-structure
		 (let ((grouped-content-nodes
			(group-nodes content-nodes paragraph-structure)))
		   (mapcar #'find-leaves-of-content-node-group
			   grouped-content-nodes)))
		(t
		 ;;paragraphize-children is TRUE
		 (remove-if #'null
			    (mapcar #'find-leaves-of-content-node
				       content-nodes))))))))


(defun find-leaves-of-content-node-group (content-node-group)
  (reduce #'append
	  (mapcar #'find-leaves-of-content-node
		  content-node-group)))


;;; filters those content nodes which have an error recorded in them
;;;                             or which have an erroneous viewpoint
;;;
;;; if an error has occurred at a content node (or its associated
;;; viewpoint), no elaboration nodes below are included
(defun find-leaves-of-content-node (content-node)
  (if (and (not (content-node-has-error? content-node))
	   (not (viewpoint-contains-error? (get-only-val
					    (list content-node
						  'kb-subgraph)))))
      (let ((elaboration-nodes (get-local (list content-node
						'child-nodes-ordered))))
	(append

	 ;;viewpoints at this level
	 (get-local (list content-node 'kb-subgraph))

	 ;;results of recursing into elaboration nodes
	 (if elaboration-nodes
	     (let* ((edp-unit (get-only-val (list content-node
						  'node-type)))
		    (paragraph-structure (get-local
					  (list edp-unit
						'paragraph-structure))))
	       (if (null paragraph-structure)
		   (find-leaves-of-elaboration-node-group elaboration-nodes)
		   (let ((grouped-elaboration-nodes
			  (group-nodes elaboration-nodes
				       paragraph-structure)))
		     (mapcar #'find-leaves-of-elaboration-node-group
			     grouped-elaboration-nodes)))))))))


(defun find-leaves-of-elaboration-node-group (elaboration-node-group)
  (reduce #'append
	  (mapcar #'find-leaves-of-elaboration-node
		  elaboration-node-group)))


(defun find-leaves-of-elaboration-node (elaboration-node)
  (let ((content-nodes (get-local (list elaboration-node
					'child-nodes-ordered))))
    (if (null content-nodes)
	(list elaboration-node)
	(let* ((edp-unit (get-only-val (list elaboration-node
					     'node-type)))
	       (paragraph-structure (get-local (list edp-unit
						     'paragraph-structure)))
	       (paragraphize-children (get-only-val
				       (list edp-unit
					     'paragraphize-children?))))
	  (cond ((and (null paragraph-structure)
		      (null paragraphize-children))
		 (find-leaves-of-content-node-group content-nodes))
		(paragraph-structure
		 (let ((grouped-content-nodes
			(group-nodes content-nodes paragraph-structure)))
		   (mapcar #'find-leaves-of-content-node-group
			   grouped-content-nodes)))
		(t
		 ;;paragraphize-children it TRUE
		 (mapcar #'find-leaves-of-content-node
			 content-nodes)))))))


;-----------------------------------------------------------------------
;			   ``Paragraph Grouping''
;-----------------------------------------------------------------------


;;; groups nodes into paragraph structure
;;;
;;; example:
;;;          plan-nodes: (a c d)
;;;          paragraph-structure: ( (A B C) (D E) )
;;;          result: ( (a c) (d) )
;;;
(defun group-nodes (plan-nodes paragraph-structure)
  (if (null paragraph-structure)
      nil
      (let* ((first-parag-list (first paragraph-structure))
	     (matching-nodes (retrieve-matching-nodes plan-nodes
						      first-parag-list))
	     (remaining-nodes (retrieve-remaining-nodes plan-nodes
							matching-nodes)))
	(cons matching-nodes
	      (group-nodes remaining-nodes (rest paragraph-structure))))))


;;; finds nodes that correspond to units in paragraph-structure
;;; and preserves order of topic-nodes list 
;;;
;;; example:
;;;          plan-nodes: (a c d)
;;;          paragraph-structure: (A B C)
;;;          result: (a c)
;;;
;;; must consider: nodes that don't match directly but match 
;;;                modulo iteration
;;;
(defun retrieve-matching-nodes (plan-nodes parag-list)
  (let ((first-node (first plan-nodes)))
    (if (plan-node-matches-some-unit? first-node parag-list)
	(cons first-node
	      (retrieve-matching-nodes (rest plan-nodes)
				       parag-list)))))


;;; returns the (first) edp-unit that matches plan-node
;;; if none, returns nil
;;;
(defun plan-node-matches-some-unit? (plan-node paragraph-structure)
  (or (plan-node-matches-some-unit-directly? plan-node
					     paragraph-structure)
      (plan-node-matches-some-unit-with-iteration? plan-node
						   paragraph-structure)))


;;; returns non-nil if finds an edp-unit that matches plan-node directly
;;; if none, returns nil
;;;
(defun plan-node-matches-some-unit-directly? (plan-node
					      paragraph-structure)
  (let ((node-type (get-only-val (list plan-node
				       'node-type))))
    (member node-type paragraph-structure)))


;;; returns the (first) edp-unit that matches plan-node modulo iteration
;;; if none, returns nil
;;;
;;;
(defun plan-node-matches-some-unit-with-iteration? (plan-node
						    paragraph-structure)
  (if (produced-by-iteration? plan-node)
      (let* ((node-type (get-only-val (list plan-node
					    'node-type)))
	     (iterative-node-types (collect-all-content-specs-of node-type)))
	(intersection iterative-node-types paragraph-structure))))


;;; returns t iff plan node was produced by iteration
(defun produced-by-iteration? (plan-node)
  (equal (get-only-val (list plan-node 'produced-by-iteration?))
	 'true))


;;; climbs up EDP units until finds a non-iterative unit
;;;
;;; returns list of all EDP units encountered
;;;
(defun collect-all-content-specs-of (node-type)
  (if node-type
      (let ((parent (get-only-val (list node-type
					'content-specification-of))))
	(cons node-type (collect-all-content-specs-of parent)))))


;;; finds nodes that haven't yet been matched and returns them in order
;;;
;;; example:
;;;          topic-nodes: (a c d)
;;;          matching-nodes: (a c)
;;;          result: (d)
;;;
;;; assumes: topic-nodes and matching-nodes match at beginning
;;; 
;;; method: works by peeling off matching-nodes from topic-nodes
;;;
(defun retrieve-remaining-nodes (topic-nodes matching-nodes)
  (cond ((null topic-nodes)
	 nil)
	((null matching-nodes)
	 topic-nodes)
	(t (retrieve-remaining-nodes (rest topic-nodes)
				     (rest matching-nodes)))))


;-----------------------------------------------------------------------
;			   ``Paragraph Flattening''
;-----------------------------------------------------------------------
;;; converts a possibly deeply-embedded list to a single level
;;; of embedding
;;;
;;; assumes: parag-list is in fact a list
;;;
;;; example: 
;;;  (((A)) (B K  (C D  (E)  F)  G H) ((((I)))) ((J)))) -->
;;;  ( (A)  (B K) (C D) (E) (F) (G H)    (I)     (J)  )
;;;
;;; rationale: even though B, K, G, and H are at the same levels
;;;            in the text tree, because paragraphs were created
;;;            between K and G, G must begin a new paragraph
;;;
;;; also removes non-viewpoint elements

(defun parag-flatten (parag-list)
  (cond ((null parag-list) nil)
	((flat-list-p parag-list)
	 (let ((list-with-only-views (remove-if-not #'viewpoint-p parag-list)))
	   (if list-with-only-views
	       (list list-with-only-views))))
	(t
	 ;;parag-list has at least one embedding
	 (let ((up-to-embedding (get-up-to-embedded-list parag-list))
	       (first-embedding (get-first-embedded-list parag-list)) 
	       (after-embedding (get-after-embedded-list parag-list)))
	   (append (if up-to-embedding
		       (list up-to-embedding))
		   (parag-flatten first-embedding)
		   (parag-flatten after-embedding))))))


;;; returns t iff the-list has no embedded sub-lists
;;; assumes the-list is a list
(defun flat-list-p (view-list)
  (notany #'listp
	  view-list))


;;; given: a non-null list that includes at least one embedded list
;;;
;;; returns: list of atomic elements up to but not including
;;;          the first embedded-list
;;;
;;; example: (a b c (d e (f g)) h i) --> (a b c)
;;;
;;; also removes non-viewpoint elements
;;;
(defun get-up-to-embedded-list (list-with-embedding)
  (if list-with-embedding   ; J.L. added this check 6-24-94
      (let ((first-elem (first list-with-embedding)))
	(if (atom first-elem)
	    (if (viewpoint-p first-elem)
		(cons first-elem
		      (get-up-to-embedded-list (rest list-with-embedding)))
		(get-up-to-embedded-list (rest list-with-embedding)))))))


;;; given: a non-null list that includes at least one embedded list
;;;
;;; returns: first embedded-list
;;;
;;; example: (a b c (d e (f g)) h i) --> (d e (f g))
;;;
(defun get-first-embedded-list (list-with-embedding)
  (let ((first-element (first list-with-embedding)))
    (if (listp first-element)
	first-element
	(get-first-embedded-list (rest list-with-embedding)))))


;;; given: a non-null list that includes at least one embedded list
;;;
;;; returns: list of elements beginning with embedded list
;;;
;;; example: (a b c (d e (f g)) h i) --> (h i)
;;;
(defun get-after-embedded-list (list-with-embedding)
  (let ((first-element (first list-with-embedding)))
    (if (listp first-element)
	(rest list-with-embedding)
	(get-after-embedded-list (rest list-with-embedding)))))


;-----------------------------------------------------------------------
;			EDP Organization Checking
;-----------------------------------------------------------------------

;;;      - rules: (1) the root of every EDP must have a value for either the
;;;                   paragraph-structure slot or the paragraphize-children?
;;;                   slot
;;;
;;;               (2) any unit in the EDP that has subtopics may
;;;                   have a value for the paragraph-structure slot
;;;
;;;               (3) any other unit in the EDP (except for
;;;                   content-specifications) that has subtopics
;;;                   may have a value for the paragraphize-children? slot
;;;    
;;;               (4) any unit that has a value for the paragraph-structure
;;;                   slot must satisfy the following criteria:
;;;                       the order of the units in ``flattened''
;;;                       paragraph-structure list must be the same
;;;                       as the order of the units in the organization
;;;                       specification in that unit, the units mentioned
;;;                       must be exactly the same in 3 lists:
;;;                            - the ``topic-list'' slot,
;;;                            - the ``organization'' slot
;;;                            - the paragraph-structure slot
;;;
;;;     - advice: if encounter error, then after fix it, need to rerun to
;;;               check for other errors


(defun check-organization (edp)
  (let ((top-unit-problem? (check-top-unit edp)))
    (if top-unit-problem?
	top-unit-problem?
	(check-topic-units (get-local (list edp 'topic-list))))))


;-----------------
;;; traverses EDPs
;-----------------


;;; returns (problem <edp-unit> <type>) if there is a problem
;;; if no problem, then returns nil
;;;
(defun check-top-unit (edp-unit)
  (let ((organization-problem?
	 (check-org-spec edp-unit
			 'topic-list
			 'inter-topic-organization-specification))
	(paragraphize-children? 
	 (get-local (list edp-unit 'paragraphize-children?)))
	(paragraph-structure 
	 (get-local (list edp-unit 'paragraph-structure))))
    (cond (organization-problem?
	   (signal-org-error edp-unit 'illegal-organization-spec))
	  ((and (null paragraphize-children?)
		(null paragraph-structure))
	   (signal-org-error edp-unit 'no-paragraph-info))
	  (paragraph-structure
	   (let ((parag-results
		  (check-paragraph-structure
		   edp-unit
		   'inter-topic-organization-specification)))
	     (if parag-results
		 (signal-org-error edp-unit 'illegal-parag-info))))
	  (t nil))))


(defun check-topic-units (topic-units)
  (if topic-units
      (let* ((first-unit (first topic-units))
	     (topic-unit-problem? (check-topic first-unit)))
	(if topic-unit-problem?
	    topic-unit-problem?
	    (check-topic-units (rest topic-units))))))


;;; checks topic-unit locally
;;; if there is a problem reports it
;;; otherwise, checks children of topic-unit
;;; returns (problem <problem-unit> <problem-type>) if there is a problem
;;;
(defun check-topic (topic-unit)
  (let* ((organization-problem?
	  (check-org-spec topic-unit
			  'content-specifications
			  'intra-topic-organization-specification))
	 (paragraph-structure 
	  (get-local (list topic-unit 'paragraph-structure)))
	 (parag-problem?
	  (if paragraph-structure
	      (check-paragraph-structure
	       topic-unit
	       'intra-topic-organization-specification))))
    (cond (organization-problem?
	   (signal-org-error topic-unit 'illegal-organization-spec))
	  ((and paragraph-structure
		parag-problem?)
	   (signal-org-error topic-unit 'illegal-parag-info))	   
	  (t (check-content-specification-units
	      (get-local (list topic-unit 'content-specifications)))))))


(defun check-content-specification-units (content-specifications)
  (if content-specifications
      (let* ((first-unit (first content-specifications))
	     (content-specification-problem? (check-content-spec first-unit)))
	(if content-specification-problem?
	    content-specification-problem?
	    (check-content-specification-units
	     (rest content-specifications))))))


;;; note: a content-specification may have 2 kinds of children:
;;;                 - content-specifications (for iteration)
;;;                 - elaborations
;;;       but it may have only one of these two types because
;;;       if it's iterative, it will only have content-specifications,
;;;       and if it's not iterative, it will either have elaborations
;;;       or no value on either slot; if it's iterative, no need to
;;;       check organization-specification
;;;
(defun check-content-spec (content-specification)
  (let* ((has-nested-specifications?
	  (has-content-specifications-p content-specification))
	 (elaborations
	  (retrieve-elaborations content-specification))
	 (organization-problem?
	  (if elaborations
	      (check-org-spec content-specification
			      'elaborations
			      'inter-elaboration-organization-specification)))
	 (paragraph-structure 
	  (if elaborations
	      (get-local (list content-specification 'paragraph-structure))))
	 (parag-problem?
	  (if paragraph-structure
	      (check-paragraph-structure
	       content-specification
	       'inter-elaboration-organization-specification))))
    (cond (organization-problem?
	   (signal-org-error content-specification
			     'illegal-organization-spec))
	  ((and paragraph-structure
		parag-problem?)
	   (signal-org-error content-specification
			     'illegal-parag-info))
	  (has-nested-specifications?
	   (check-content-spec has-nested-specifications?))
	  (elaborations
	   (check-elaboration-units elaborations))
	  (t nil))))


;;; returns nested spec iff content-specification has nested
;;; otherwise nil
;;; assumes at most one nested spec 
(defun has-content-specifications-p (content-specification)
  (get-only-val (list content-specification 'content-specifications)))


;;; returns elaborations iff content-specification has elaborations
;;; otherwise nil
(defun retrieve-elaborations (content-specification)
  (get-local (list content-specification 'elaborations)))


(defun check-elaboration-units (elaboration-units)
  (if elaboration-units
      (let* ((first-unit (first elaboration-units))
	     (elab-unit-problem? (check-elaboration first-unit)))
	(if elab-unit-problem?
	    elab-unit-problem?
	    (check-elaboration-units (rest elaboration-units))))))


;;; checks elaboration unit locally
;;; if there is a problem reports it
;;; otherwise, checks children of elaboration
;;; returns (problem <problem-unit> <problem-type>) if there is a problem
;;;
(defun check-elaboration (elaboration-unit)
  (let* ((organization-problem?
	  (check-org-spec elaboration-unit
			  'content-specifications
			  'intra-elaboration-organization-specification))
	 (paragraph-structure 
	  (get-local (list elaboration-unit 'paragraph-structure)))
	 (parag-problem?
	  (if paragraph-structure
	      (check-paragraph-structure
	       elaboration-unit
	       'intra-elaboration-organization-specification))))
    (cond (organization-problem?
	   (signal-org-error elaboration-unit 'illegal-organization-spec))
	  ((and paragraph-structure
		parag-problem?)
	   (signal-org-error elaboration-unit 'illegal-parag-info))	   
	  (t (check-content-specification-units
	      (get-local (list elaboration-unit 'content-specifications)))))))


;---------------------------------------------------
;;; checks organization and paragraph specifications
;---------------------------------------------------

;;; returns t if there is a problem
;;; otherwise returns nil
;;;
;;; an EDP unit must satisfy the following criteria:
;;; the units in two lists must be the same
;;;                            - the ``organization'' slot
;;;                            - the paragraph-structure slot
;;;
(defun check-org-spec (edp-unit child-slot organization-slot)
  (let* ((child-list
	  (get-local (list edp-unit child-slot)))
	 (organization-list 
	  (get-local (list edp-unit organization-slot))))
    (not (sets-are-equal child-list organization-list))))


;;; returns t if there is a problem
;;; otherwise returns nil
;;;
;;; any unit that has a value for the paragraph-structure slot must
;;; satisfy the following criteria:
;;;                       the order of the units in ``flattened''
;;;                       paragraph-structure list must be the same
;;;                       as the order of the units in the organization
;;;                       specification in that unit, the units mentioned
;;;                       must be exactly the same in 2 lists:
;;;                            - the ``organization'' slot
;;;                            - the paragraph-structure slot
;;;
(defun check-paragraph-structure (edp-unit organization-slot)
  (let* ((organization-list 
	  (get-local (list edp-unit organization-slot)))
	 (paragraph-structure 
	  (get-local (list edp-unit 'paragraph-structure)))
	 (flattened-paragraph-structure
	  (flatten paragraph-structure)))
    (not (equal organization-list
		flattened-paragraph-structure))))


(defun sets-are-equal (set1 set2)
  (and (subsetp set1 set2 :test #'equal)
       (subsetp set2 set1 :test #'equal)))


(defun signal-org-error (edp-unit error-type)
  (list 'problem edp-unit error-type))

