;;; -*- Mode:Lisp; Package:User; Base:10 -*-


;;;;		 The Auxiliary Content Determination Functions
;;;;		 ---------------------------------------------


;;;; These are the functions used by the EDPs for content determination.
;;;;
;;;;
;;;; What's Returned
;;;; ---------------
;;;; Often a list is returned; for compatibility with KM, this is
;;;; first turned into a list of a list before it is returned; hence,
;;;; the entire desired list will be put as a single value on the
;;;; kb-subgraph slot of a content node
;;;;
;;;;
;;;; Error Handling
;;;; --------------
;;;; They include methods for reporting errors encountered during
;;;; execution.  In general, during content determination, if an
;;;; error is found, the function returns a pair of the form:
;;;;          (error <error-type>)
;;;;
;;;; The error handler of the EDP Applier will record the error type
;;;; in the Explanation Plan and act appropriately.  Typically, the
;;;; Applier will omit the intended content.
;;;;
;;;;
;;;; User Modeling Issues
;;;; --------------------
;;;; Some functions exploit the user model.  When the user modeling
;;;; facility is off, these functions behave differently.  They
;;;; take the approach of doing whatever is the ``right thing''
;;;; without regard to what might have been in the user model, e.g.,
;;;; when finding a reference concept, the function merely finds
;;;; an immediate genl of the given concept, rather than the most
;;;; specific concept that was familiar to the user.  (See the header
;;;; of the User Model functions file for additional information.)
;;;;
;;;;
;;;; Consider Doing Later: Dealing with Implicit Units in Code
;;;; ---------------------------------------------------------
;;;; Consider revising all content determination functions to deal with
;;;; values that are addresses, e.g., when climbing up in Connect-to-Core
;;;; the code treats all values though they are units (like asking
;;;; is-known-p, is-core-p, but actually values might be addresses;
;;;; one superevent of Stem-Cortex-Generation is
;;;;             (Development subevents Generation)


(in-package 'km)


;-----------------------------------------------------------------------
;			Collect-Causal-Agent-Processes
;-----------------------------------------------------------------------

;;; arguments: Content-Node with Causal-Agent-Viewpoint
;;;
;;; goes through causal agent viewpoint to find processes
;;; which the current process affects
;;;
;;; basic-dimension: Causal-Agent
;;;
;;; can test with:
;;;                Enables: <No non-embedded processes>
;;;                Facilitates: <None>
;;;                Inhibits: <None>
;;;                Causes: Guard-Cell-Collapse, Plant-Water-Loss
;;;
;;; returns (a list of a) list of processes
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint)
;;;
;;;		- condition: viewpoint exists, but error
;;;		- returns: (error problem-viewpoint)
;;;
;;;		- condition: viewpoint exists, but no processes
;;;		- returns: (error no-processes)

(defun collect-causal-agent-processes (content-node)
  (if (null content-node)
      '((error no-content-node))
      (let ((the-viewpoint (get-only-val (list content-node
					       'kb-subgraph))))
	(cond ((null the-viewpoint)
	       '((error no-viewpoint)))
	      ((content-node-has-error? content-node)
	       '((error problem-viewpoint)))
	      (t (let* ((slot-list (get-local '(causal-agent
						dimension-of-slots)))
			(processes (get-processes-on-slots the-viewpoint
							   slot-list)))
		   (if (null processes)
		       '((error no-processes))
		       (list processes))))))))


;;; returns all values on unit that appear on some slot in slot-list
;;; and are specs of process
(defun get-processes-on-slots (unit slot-list)
  (remove-duplicates
   (reduce #'append
	   (mapcar #'(lambda (slot)
		       (let ((values (get-local (list unit slot))))
			 (reduce #'append
				 (mapcar #'(lambda (value)
					     (if (member 'process
							 (ancestry* value))
						 (list value)))
					 values))))
		   slot-list))
   :test 'equal))


;-----------------------------------------------------------------------
;		     Collect-Causal-Facilitating-Processes
;-----------------------------------------------------------------------

;;; arguments: Content-Node with Causal-Facilitating-Viewpoint
;;;
;;; goes through causal facilitating viewpoint to find processes
;;; which facilitate or cause the current process
;;;
;;; basic-dimension: Modulatory-Facilitators
;;;
;;; can test with:
;;;                Caused-by: Plant-water-stress, Plant-water-loss,
;;;                           Stoma-closing-sequence, Guard-Cell-Collapse
;;;                Facilitated-by: <None>
;;;
;;; returns (a list of a) list of processes
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint)
;;;
;;;		- condition: viewpoint exists, but error
;;;		- returns: (error problem-viewpoint)
;;;
;;;		- condition: viewpoint exists, but no processes
;;;		- returns: (error no-processes)

(defun collect-causal-facilitating-processes (content-node)
  (if (null content-node)
      '((error no-content-node))
      (let ((the-viewpoint (get-only-val (list content-node
					       'kb-subgraph))))
	(cond ((null the-viewpoint)
	       '((error no-viewpoint)))
	      ((content-node-has-error? content-node)
	       '((error problem-viewpoint)))
	      (t (let* ((slot-list (get-local '(modulatory-facilitators
						dimension-of-slots)))
			(processes (get-processes-on-slots the-viewpoint
							   slot-list)))
		   (if (null processes)
		       '((error no-processes))
		       (list processes))))))))


;-----------------------------------------------------------------------
;			 Collect-Inhibiting-Processes
;-----------------------------------------------------------------------

;;; arguments: Content-Node with Causal-Inhibiting-Viewpoint
;;;
;;; goes through causal inhbiting viewpoint to find processes
;;; which inhibits the current process
;;;
;;; basic-dimension: Modulatory-Inhibitors
;;;
;;; can test with:
;;;                Inhibited-By: <none>
;;;
;;; returns (a list of a) list of processes
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint)
;;;
;;;		- condition: viewpoint exists, but error
;;;		- returns: (error problem-viewpoint)
;;;
;;;		- condition: viewpoint exists, but no processes
;;;		- returns: (error no-processes)

(defun collect-inhibiting-processes (content-node)
  (if (null content-node)
      '((error no-content-node))
      (let ((the-viewpoint (get-only-val (list content-node
					       'kb-subgraph))))
	(cond ((null the-viewpoint)
	       '((error no-viewpoint)))
	      ((content-node-has-error? content-node)
	       '((error problem-viewpoint)))
	      (t (let* ((slot-list (get-local '(modulatory-inhibitors
						dimension-of-slots)))
			(processes (get-processes-on-slots the-viewpoint
							   slot-list)))
		   (if (null processes)
		       '((error no-processes))
		       (list processes))))))))


;-----------------------------------------------------------------------
;				Collect-Actors
;-----------------------------------------------------------------------

;;; arguments: Content Node with Black-box-view
;;;
;;; searches through the Black box view to find the actor slots
;;; 
;;; seaches through the values on these slots to return the actors
;;; 
;;; returns a list of actors
;;;
;;; errors:
;;;             - condition: no content node exists
;;;             - returns: (error no-content-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;                          with black-box view
;;;             - returns: (error error-in-node-with-black-box-view)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-viewpoint-in-content-node)
;;;
;;;		- condition: viewpoint exists, but no actors
;;;		- returns: (error no-actor-slots-on-viewpoint)

(defun collect-actors (content-node)
  (cond ((null content-node)
	 '((error no-content-node)))
	((content-node-has-error? content-node)
	 '((error error-in-node-with-black-box-view)))
	(t (let ((viewpoint
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null viewpoint)
		 '((error no-viewpoint-in-content-node))
		 (let ((actor-slots (collect-actor-slots viewpoint)))
		   (if (null actor-slots)
		       '((error no-actor-slots-on-viewpoint))
		       (collect-all-values-on-slots viewpoint
						    actor-slots))))))))


(defun collect-actor-slots (viewpoint)
  (let ((slots-with-values (all-explicit-slots viewpoint)))
    (remove-if-not #'is-actor-slot-p
		   slots-with-values)))


(defun is-actor-slot-p (slot)
  (member 'actors (ancestry* slot)))


;;; returns list of all values that appear on some slot in
;;; slot-list on frame
;;;
;;; Important note: mapcan is *not* used below and should
;;;                 not be used anywhere in the code because
;;;                 it nconc's some structures in the KB,
;;;                 sometimes creating circular structures
;;;                 that cause KM to go into infinite recursion

(defun collect-all-values-on-slots (frame slot-list)
  (remove-duplicates
   (reduce #'append
	   (mapcar #'(lambda (slot) (get-local (list frame slot)))
		   slot-list))))


;-----------------------------------------------------------------------
;				Connect-To-Core
;-----------------------------------------------------------------------

;;; arguments: Origin
;;;
;;; called from Process-Significance
;;;
;;; finds connection from Origin to either (1) a process that 
;;; is marked ``core'' in the knowledge base, or (2) a process 
;;; that is known to the user and is on the search path to a
;;; core process, or (3) the unit ``Process'' is reached, which
;;; is an error
;;;
;;; assumes that if unit has a value on core? then the unit is
;;; a core process
;;;
;;; first climbs up superevents and then climbs up generalizations
;;;
;;; assumes if user knows a process, he knows the relation of that
;;; process to a core process (if at the begining of a session
;;; the user is told about all core processes, or if it is
;;; known that he knows about all core processes, then this
;;; assumption is true by induction)
;;;
;;; if a process has more than one superevent or more than one
;;; generalization, the first one is chosen (arbitrarily)
;;;
;;; errors:
;;;		- note: all errors involve failing to find either
;;;                     a core process or a familiar process;
;;;                     there are 2 basic cases: during the
;;;                     superevent ascent and during the genl
;;;                     ascent
;;;
;;;		- condition: climb up superevents yields deadend,
;;;                          i.e., superevents lead up to unit 
;;;                          ``Process''
;;;		- returns: (error super-event-search-dead-ended)
;;;
;;;		- condition: cannot find a genl
;;;		- returns: (error no-known-or-core-genl-available)
;;;
;;;	- Note: the system will later need to decide whether to
;;;       talk about the ``destination'' process, i.e, the unit
;;;       at which the search terminated; 
;;;
;;;		- one way of making this decision would be for
;;;               the kb-access function to return a pair, where
;;;               the first item in the pair is either the atom
;;;               ``core'' to indicate that the search terminated 
;;;               at a core process, or ``known'' to indicate that the 
;;;               search terminated at a process that is familiar to the
;;;               user; the second item in the pair would have been
;;;               the list
;;;
;;;			- problem: this information about whether the
;;;                       process is known or core is needed to make
;;;                       the decision about whether to have an
;;;                       elaboration or not about the ``destination'';
;;;                       however, things will be much cleaner if the
;;;                       focus conditions of the elaboration look this
;;;                       up themselves in the KB
;;;
;;;		        - principle: don't return anything in KB-access
;;;                                  except the content itself
;;;
;;; Some cases:
;;;      (1)  known-or-core process is origin
;;;      (2)  known-or-core process is superevent
;;;      (2') superevent climb dead-ends
;;;      (3)  known-or-core process is genl
;;;      (4)  known-or-core process is an all genl of an all-superevent
;;;      (4') genl climb dead-ends
;;;
;;; returns (a list of) a list: the first item in the list is the
;;; unit at which the search terminated, the next 
;;; item in the list is the slot that connects the 
;;; units in the search path, the next item is the 
;;; penultimate unit in the search, etc., until the 
;;; final item in the list which is the origin


(defun connect-to-core (origin)
  (let* ((super-event-result (climb-super-events origin))
	 (destination-type (first super-event-result))
	 (sub-event-path (second super-event-result)))
    ;(format t "Sub-event-path: ~a~%" sub-event-path)
    (case destination-type
      (error '((error super-event-search-dead-ended)))
      (known-or-core (list sub-event-path))
      (not-known-or-core
       (let* ((genls-result (climb-genls (first sub-event-path)))
	      (final-destination-type (first genls-result))
	      (genls-path (second genls-result)))
	 ;(format t "Genls-path: ~a~%" genls-path)
	 (case final-destination-type
	   (error '((error no-known-or-core-genl-available)))
	   (path-found (list (append genls-path
				     (cdr sub-event-path))))))))))


;;; climbs super-events until encounters either (1) a process that
;;; is marked ``core'' in the knowledge base or (2) a process 
;;; that is known to the user
;;;
;;; if user modeling is off, only considers core processes
;;;
;;; returns a list of the form
;;; where <destination-type> is either 'known-or-core or 'not-known-or-core    
;;; or 'error, and <path> is a list of the form
;;;    (destination 'sub-event p(n-1) 'sub-event ... 'sub-event origin)
;;; or nil if <destination-type> is 'error

(defun climb-super-events (origin)
  (let* ((result (climb-super-events-aux origin))
	 (last-of-result (first (last result))))
    (cond ((equal last-of-result 'error)
	   (list 'error nil))
	  ((or (and (user-modeling-on?) (is-known-p last-of-result))
	       (is-core-p last-of-result))
	   (list 'known-or-core (reverse result)))
	  (t (list 'not-known-or-core (reverse result))))))


;;; climbs up super-events
;;;
;;; assumes an event has at most 1 (direct) super-event (in fact,
;;; this is not always true, and code may someday need to be revised
;;; to reflect this
;;;
;;; returns a list of the form
;;;    (origin 'sub-event p1 'sub-event ... 'sub-event destination)
;;; where destination is either the destination process or 'error

(defun climb-super-events-aux (origin)
  ;(format t "Considering in climb-super-events-aux ~a~%" origin)
  (cond ((or (and (user-modeling-on?) (is-known-p origin))
	     (is-core-p origin))
	 (list origin))
	((equal origin 'process)
	 (list 'error))
	(t (let ((superevent (first (get-local (list origin 'superevents)))))
	     (if (null superevent)
		 (list origin)
		 (cons origin
		       (cons 'subevent
			     (climb-super-events-aux superevent))))))))


;;; climbs genl's until encounters either (1) a process that
;;; is marked ``core'' in the knowledge base or (2) a process 
;;; that is known to the user, or (3) no known or core process
;;; is encountered
;;;
;;; if user modeling is off, only considers core processes
;;;
;;; returns a pair of the form: (<destination-type> <path>)
;;; where <destination-type> is either 'error or 'path-found,
;;; and <path> is a list of the form
;;; (pn 'spec p(n-1) 'spec ... 'spec origin)
;;; or nil if <destination-type> is error

(defun climb-genls (origin)
  (let ((result (climb-genls-aux origin nil)))
    (if (null result)
	(list 'error nil)
	(list 'path-found (reverse result)))))


;;; performs a depth-first up the taxonomy
;;; 
;;; deals with multiple genl's
;;;
;;; returns list of form:
;;;      (origin 'specialization p1 'specialization p2 'specialization ...
;;;              'specialization destination)
;;; or nil

(defun climb-genls-aux (origin genls-already-considered)
  ;(format t "Considering in climb-genls-aux ~a~%" origin)
  (cond ((or (and (user-modeling-on?) (is-known-p origin))
	     (is-core-p origin))
	 (list origin))
	((equal origin 'content-frame)
	 nil)
	(t (let* ((new-genls (get-local (list origin 'generalizations)))
		  (actual-genls (set-difference new-genls
						genls-already-considered)))
	     (if (not (null actual-genls))
		 (do* ((remaining-genls actual-genls
					(cdr remaining-genls))
		       (first-remaining-genl (first remaining-genls)
					     (first remaining-genls))
		       (path (climb-genls-aux
			      first-remaining-genl
			      (cons origin
				    genls-already-considered))
			     (climb-genls-aux
			      first-remaining-genl
			      (cons origin
				    genls-already-considered))))
		      ((or (not (null path))
			   (null remaining-genls))
		       (if (not (null path))
			   (cons origin
				 (cons 'specialization
				       path))))))))))


;-----------------------------------------------------------------------
;			   Find-Destination-Process
;-----------------------------------------------------------------------

;;; arguments: Core-Content-Node
;;;
;;; called from Process-Significance
;;;
;;; looks in content node resulting from Connect-to-core call
;;;
;;; returns first element of the list, which is the destination
;;; of the search that was conducted in Connect-to-core
;;;
;;; errors:
;;;		- condition: no content node exists
;;;		- returns: (error no-content-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;                          with connect-to-core path
;;;             - returns: (error error-in-node-with-connect-to-core)
;;;
;;;             - condition: an error was returned by Connect-to-core
;;;             - returns: (error connect-to-core-failed)

(defun find-destination-process (core-content-node)
  (cond ((null core-content-node)
	 '((error no-content-node)))
	((content-node-has-error? core-content-node)
	 '((error error-in-node-with-connect-to-core)))
	(t (let ((connection-path (get-only-val (list core-content-node
						      'kb-subgraph))))
	     (if (or (null connection-path)
		     (equal (first connection-path) 'error))
		 '((error connect-to-core-failed))
		 (first connection-path))))))

	     
;-----------------------------------------------------------------------
;			  Find-Partonomic-Connection
;-----------------------------------------------------------------------

;;; arguments: Object
;;;
;;; searches up the partonomy from Object to find concept with
;;; which user is familiar
;;;
;;; considers slots other than ``part-of'' that are similar,
;;; e.g., contained-in
;;;
;;; search continues until encounter either (1) a known concept 
;;; (uses function is-known-p) or (2) an elementary concept 
;;; (is-known-p uses function is-elementary-p), or (3) no
;;; familiar concept is encountered and cannot climb higher
;;; in partonomy (an error)
;;;
;;; if user-modeling is not on, returns (1) error if Object has
;;; no super-parts, or (2) first super-part encountered
;;;
;;; performs a breadth-first search up the partonomy (needs to
;;; be breadth-first to ensure that most specific concept in
;;; partonomy is found)
;;;
;;; efficiency addition: keeps ``closed'' list so doesn't exam same
;;;                      node twice
;;;
;;; extra benefit of efficiency addition: no infinite loop on if encounter
;;;                                       circularities in partonomy
;;;
;;; returns that concept at which the search halted
;;;
;;; errors:
;;;		- condition: user modeling is off and object has
;;;                          no super-parts
;;;		- returns: (error no-user-modeling-and-no-super-part)
;;;
;;;		- condition: no familiar concept is encountered and
;;;                          cannot climb higher in partonomy
;;;		- returns: (error no-familiar-super-part)

(defun find-partonomic-connection (object)
  (if (not (user-modeling-on?))
      (let ((super-parts (find-super-parts object)))
	(if (null super-parts)
	    '((error no-user-modeling-and-no-super-part))
	    (first super-parts)))
      (find-part-conn-aux (list object) nil)))


(defun find-part-conn-aux (part-list parts-already-considered)
  (if (null part-list)
      nil
      (let ((first-part (first part-list)))
	;(format t "Considering ~a~%" first-part)
	;(format t "Part list ~a~%" part-list)
	;(format t "Parts already considered ~a~%" parts-already-considered)
	(if (is-known-p first-part)
	    first-part
	    (let* ((super-parts (find-super-parts first-part))
		   (actual-super-parts
		    (set-difference super-parts parts-already-considered)))
	      (find-part-conn-aux (append (cdr part-list)
					  actual-super-parts)
				  (cons first-part
					parts-already-considered)))))))


;;; finds all super-parts of given object; if none, returns nil
;;;
;;; list of parts-slots is stored in:
;;;          (Part-slot-list part-slots-used-by-knight)

(defun find-super-parts (object)
  (let ((parts
	 (do* ((remaining-parts-slots (get-local '(part-slot-list
						   part-slots-used-by-knight))
				      (cdr remaining-parts-slots))
	       (current-slot (first remaining-parts-slots) 
			     (first remaining-parts-slots))
	       (current-values (get-local (list object current-slot))
			       (get-local (list object current-slot)))
	       (super-parts (if (not (null current-values))
				current-values)
			    (if (not (null current-values))
				(append super-parts current-values)
				super-parts)))
	      ((null remaining-parts-slots)
	       super-parts)
	   ;(format t "Remaining parts slots: ~a~%" remaining-parts-slots)
	   ;(format t "Current slot: ~a~%" current-slot)
	   ;(format t "Current values: ~a~%" current-values)
	   ;(format t "Super-parts: ~a~%" super-parts)
	   )))
    (remove-duplicates parts)))


;-----------------------------------------------------------------------
;				 Find-Ref-Conc
;-----------------------------------------------------------------------

;;; arguments: Origin
;;;
;;; searches up the taxonomy from Origin to find concept with
;;; which user is familiar
;;; 
;;; considers slots ``generalizations'', ``instance-of'' and ``stage-of''
;;;
;;; search continues until encounter either (1) a known concept 
;;; (uses function Is-known-p) or (2) an elementary concept 
;;; is-known-p uses function is-elementary-p), or (3) no
;;; familiar concept is encountered (this is an error)
;;;
;;; performs a breadth-first search up the taxonomy (needs to
;;; be breadth-first to ensure that most specific concept in
;;; partonomy is found) via the KM function ``ancestry*''
;;;
;;; if user-modeling is not on, returns (1) error if Origin has
;;; no genl's, or (2) first genl encountered
;;;
;;; returns that concept at which the search halted
;;; 
;;; errors:
;;;		- condition: user modeling is off and origin has
;;;                          no genls
;;;		- returns: (error no-user-modeling-and-no-genl)
;;;
;;;		- condition: no familiar concept is encountered and
;;;                          cannot climb higher in taxonomy
;;;		- returns: (error no-familiar-generalizations)

(defun find-ref-conc (origin)
  (if (not (user-modeling-on?))
      (let ((genls (get-local (list origin 'generalizations))))
	(if (null genls)
	    '((error no-user-modeling-and-no-genl))
	    (first genls)))
      (do* ((remaining-ancestors (ancestry* origin)
				 (cdr remaining-ancestors))
	    (current-concept (first remaining-ancestors)
			     (first remaining-ancestors))
	    (current-concept-known? (is-known-p current-concept)
				    (is-known-p current-concept)))
	   ((or current-concept-known?
		(null remaining-ancestors))
	    (if current-concept-known?
		current-concept
		;; search has exhausted remaining actors
		'((error no-familiar-generalizations))))
	;(format t "Considering concept: ~a~%" current-concept)
	)))


;-----------------------------------------------------------------------
;			      Find-Subevent-Order
;-----------------------------------------------------------------------

;;; arguments: Process
;;;
;;; obtains list of Process.subevents
;;;
;;; examines temporal constaints such as occurs-before and
;;; occurs-after to find the temporal order of the subevents
;;;
;;; assumes that knowledge engineer put in as specific info as possible
;;;
;;; if temporal information is missing, orders subevents arbitrarily
;;;
;;; Ordering-relation slots in KB: (slots considered in algorithm are *'d)
;;;        - starts-before             *
;;;        - starts-after              *
;;;        - occurs-after              *
;;;        - occurs-before             *
;;;        - contiguous-with           (too weak)
;;;             - contiguous-before    *
;;;             - contiguous-after     *
;;;        - overlaps-start            *
;;;        - overlaps-end              *
;;;        - coincidental-with         *
;;;        - co-terminal-with          (no clue about which starts first)
;;;        - contains-temporal         *
;;;        - co-originating-with       *
;;;        - occurs-within             *
;;;
;;; this algorithm is *not* intended to be a sound and complete temporal
;;; reasoner; its goal is merely to provide a general guide for ordering
;;; subevents in a discourse, so unsound reasoning is allowed, e.g.,
;;; if we have (i starts-before j), we conclude (i before j); though
;;; we do not know that i ends before j begins, it makes more sense
;;; to discuss i before j; in general, the heuristic is to place event-i
;;; before event-j if event-i begins before event-j begins
;;;
;;; Algorithm
;;; ---------
;;;
;;;   (1) Find all ordering relations between all subevents
;;;
;;;   (2) Map kb temporal relations to discourse temporal relations
;;;
;;;   (3) Condense discourse temporal relations to list of form
;;;
;;;            (event-item-1 event-item-2 ... event-item-n)
;;;
;;;       which indicates order of event beginnings;
;;;
;;;       An event-item is a pair of the form 
;;;
;;;            ((same-as event-i event-j ...) (contains event-k event-l ...))
;;;
;;;       where the same-as component can contain one or more events and
;;;       the contains component may contain be empty except for the
;;;       keyword `contains.''
;;;      
;;;       Note that only one level of containment can be accounted for
;;;       by this scheme.  It is assumed that if there were multiple levels,
;;;       the events wouldn't have all been made subevents of the same
;;;       event in the first place.
;;;
;;;       Special conditions:
;;;         - condition:  elements appear in list twice
;;;           action:     remove second occurrence
;;;
;;;         - condition:  some elements left over
;;;           action:     place (arbitrarily) at end
;;;
;;;         - if neither of special conditions hold, and all elements
;;;           in the ordered-list constructed in (3) are related to
;;;           one another, then a total order has been achieved
;;;
;;;   (4) Flatten structured list, i.e., construct an unnested list
;;;       by reading left-to-right in structured list
;;;
;;; returns (a list of a) the flat list described above
;;;
;;; errors:
;;;		- condition: process has no subevents
;;;		- returns: (error no-subevents)

(defun find-subevent-order (process)
  (let ((subevent-list (get-local (list process 'subevents))))
    (cond ((null subevent-list) '(error no-subevents))
	  ((equal (length subevent-list) 1) (list subevent-list))
	  (t
	   ;; at least 2 subevents
	   (let* ((kb-triples
		   (get-temporal-relation-triples subevent-list))
		  (discourse-triples
		   (map-to-discourse-triples kb-triples))
		  (structured-ordered-elements-results
		   (map-to-ordered-list discourse-triples))
		  (total-order?
		   (first structured-ordered-elements-results))
		  (structured-ordered-elements
		   (second structured-ordered-elements-results))
		  (flattened-ordered-elements
		   (flatten-temporal-structure structured-ordered-elements))
		  (exist-dupe-events? (exist-duplicate-elements?
				       flattened-ordered-elements))
		  (exist-uncovd-events? (exist-uncovered-events?
					 flattened-ordered-elements
					 subevent-list))
		  (final-total-order? (and (equal total-order? 'total-order)
					   (not exist-dupe-events?)
					   (not exist-uncovd-events?))))
	     ;; tests for special conditions
	     ;(format t "Total-order?: ~a~%" total-order?)
	     ;(format t "Discourse-triples:~%")
	     ;(pprint discourse-triples)
	     ;(format t "~%Structured-ordered-elements:~%")
	     ;(pprint structured-ordered-elements-results)
	     ;(format t "~%Flattend-ordered-elements: ~a~%"
	     ;	     flattened-ordered-elements)
	     ;(format t "Dupe-events?: ~a~%" exist-dupe-events?)
	     ;(format t "Uncovd-events?: ~a~%" exist-uncovd-events?)
	     ;(format t "Final-total-order?: ~a~%" final-total-order?)
	     (cond (final-total-order?
                    ;total order					
		    (list flattened-ordered-elements))

		   (exist-dupe-events?
		    ;no total order
		    (list (remove-duplicates flattened-ordered-elements
					     :from-end t)))
		   
		   (exist-uncovd-events?
		    ;no total order
		    (list
		     (append flattened-ordered-elements
			     exist-uncovd-events?)))

		   (t
		    ;no total order
		    (list flattened-ordered-elements))))))))


;;; given a list of events, finds all ordering-relation triples
;;; that relate pairs of events
;;;
;;; for each temporal-relation finds all pairs of events that
;;; are related by that relation
;;;
;;; returns list of form ( ... (event-i temp-relation-p event-j) ... )

(defun get-temporal-relation-triples (event-list)
  (let ((temporal-relation-list
	 (list 'starts-before 'starts-after 'occurs-after 'occurs-before
	       'contigous-before 'contiguous-after
	       'overlaps-start 'overlaps-end 'coincidental-with
	       'contains-temporal 'co-originating-with
	       'occurs-within))
	(triple-list nil))
    (dolist (temporal-relation temporal-relation-list)
      (dolist (event event-list)
	(dolist (value (get-local (list event temporal-relation)))
	  (if (member value event-list)
	      (setf triple-list
		    (push (list event
				temporal-relation
				value)
			  triple-list))))))
    ;(pprint triple-list)
    triple-list))


;;; maps kb temporal relations to discourse temporal relations
;;;
;;;   Mapping
;;;   -------
;;;   KB-Temporal-Relation                  --> Discourse-Temporal-Relation 
;;;
;;;   (event-i starts-before event-j)       --> (event-i before event-j)
;;;   (event-j starts-after event-i)        --> (event-i before event-j)
;;;
;;;   (event-i contiguous-before event-j)   --> (event-i before event-j)
;;;   (event-j contiguous-after event-i)    --> (event-i before event-j)
;;;
;;;   (event-i occurs-before event-j)       --> (event-i before event-j)
;;;   (event-j occurs-after event-i)        --> (event-i before event-j)
;;;
;;;   (event-i overlaps-start event-j)      --> (event-i before event-j)
;;;   (event-j overlaps-end event-i)        --> (event-i before event-j)
;;;
;;;   (event-i contains-temporal event-j)   --> (event-i contains event-j)
;;;   (event-j occurs-within event-i)       --> (event-i contains event-j)
;;;
;;;   (event-i coincidental-with event-j)   --> (event-i same-as event-j)
;;;   (event-i co-originating-with event-j) --> (event-i same-as event-j)

(defun map-to-discourse-triples (kb-triples)
  (mapcar
   #'(lambda (triple) 
       (let ((unit (first triple))
	     (slot (second triple))
	     (value (third triple)))
	 (case slot
	   (starts-before       (list unit  'before   value))
	   (starts-after        (list value 'before   unit))
	   (contiguous-before   (list unit  'before   value))
	   (contiguous-after    (list value 'before   unit))
	   (occurs-before       (list unit  'before   value))
	   (occurs-after        (list value 'before   unit))
	   (overlaps-start      (list unit  'before   value))
	   (overlaps-end        (list value 'before   unit))
	   (contains-temporal   (list unit  'contains value))
	   (occurs-within       (list value 'contains unit))
	   (coincidental-with   (list unit  'same-as  value))
	   (co-originating-with (list unit  'same-as  value)))))
   kb-triples))


;;; condenses discourse temporal relations to list of form
;;;
;;;       (event-item-1 event-item-2 ... event-item-n)
;;;
;;; which indicates order of event beginnings;
;;;
;;; an event-item is a dotted pair of the form 
;;;
;;;       ((same-as event-i event-j ...) . (contains event-k event-l ...))
;;;
;;; where the ``same-as'' component can contain one or more events and the
;;; ``contains'' component may be empty except for the keyword `contains.''
;;;      
;;;       Mapping
;;;       -------
;;;       Discourse-Temporal-Relation --> Ordering
;;;
;;;       (event-i before event-j)  -->
;;;                                  (((same-as event-i)  (contains))
;;;                                   ((same-as event-j)  (contains)))
;;;
;;;       (event-i same-as event-j)  -->
;;;                                  (((same-as event-i event-j) (contains)))
;;;
;;;       (event-i contains event-j) -->
;;;                                  (((same-as event-i) (contains event-j)))
;;;
;;;       Method
;;;       ------
;;;       (a) Eliminate duplicates in discourse temporal relations
;;;       (b) Find all of before-triples, same-as-triples, and contains-triples
;;;       (c) Order before's (considering the same-as information)
;;;       (d) Incorporate the same-as and strip out ``before'' keywords
;;;       (e) Incorporate the contains information

(defun map-to-ordered-list (discourse-triples)
  (let* ((non-redundant-triples (remove-duplicates discourse-triples
						   :test #'equal))
	 (before-triples (find-before-triples non-redundant-triples))
	 (same-as-triples (find-same-as-triples non-redundant-triples))
	 (contains-triples (find-contains-triples non-redundant-triples))
	 (same-as-groups (find-same-as-groups same-as-triples))
	 (ordered-befores-results (impose-before-order before-triples
						       same-as-groups))
	 (total-order? (first ordered-befores-results))
	 (ordered-befores (second ordered-befores-results))
	 (ordered-befores-and-same-as (impose-same-as-order ordered-befores
							    same-as-groups))
	 (ordered-contains (impose-contains-order ordered-befores-and-same-as
						  contains-triples)))
    ;(format t "Before-triples: ~a~%" before-triples)
    ;(format t "Same-as-groups: ~a~%" same-as-groups)
    ;(format t "Ordered-befores:~%")
    ;(pprint ordered-befores)
    ;(format t "~%")
    (list total-order? ordered-contains)))


;;; finds triples in triple-list whose slot is 'before
(defun find-before-triples (triple-list)
  (reduce #'append
	  (mapcar #'(lambda (triple) (if (equal (second triple)
						'before)
					 (list triple)))
		  triple-list)))


;;; finds triples in triple-list whose slot is 'same-as
(defun find-same-as-triples (triple-list)
  (reduce #'append
	  (mapcar #'(lambda (triple) (if (equal (second triple)
						'same-as)
					 (list triple)))
		  triple-list)))


;;; finds triples in triple-list whose slot is 'contains
(defun find-contains-triples (triple-list)
  (reduce #'append
	  (mapcar #'(lambda (triple) (if (equal (second triple)
						'contains)
					 (list triple)))
		  triple-list)))


;;; finds all groups of same-as-elements
;;; returns list of groups
;;; example: original-form: ((a same-as b) (a same-as c) (d same-as e))
;;;          reduced-form:  ((a b) (a c) (d e)
;;;          final-form:    ((a b c) (d e))
(defun find-same-as-groups (same-as-triples)
  (let ((reduced-form-of-triples
	 (mapcar #'(lambda (triple) (list (first triple)
					  (third triple)))
		 same-as-triples)))
    (find-same-as-groups-aux reduced-form-of-triples)))


;;; converts from reduced form of same-as-triples to merged groups
(defun find-same-as-groups-aux (reduced-triples-list)
  (if (null reduced-triples-list)
      nil
      (let ((first-group (first reduced-triples-list))
	    (remaining-groups (rest reduced-triples-list))
	    (remaining-groups-for-iteration (rest reduced-triples-list))
	    (merged? nil))
	(dolist (group remaining-groups-for-iteration)
	  (let ((common-elements (intersection first-group group)))
	    (when (not (null common-elements))
	      (setf first-group (remove-duplicates (append first-group group)))
	      (setf remaining-groups (remove group remaining-groups
					     :test #'equal))
	      (setf merged? t))))
	(if merged?
	    (find-same-as-groups-aux (cons first-group remaining-groups))
	    (cons first-group (find-same-as-groups-aux remaining-groups))))))


;;; imposes order on list of ``before'' triples
;;; considers same-as groups
;;;
;;; examples:
;;;           before-triples: ((a before b)) 
;;;                         --> (a before b)
;;;
;;;           before-triples: ((a before b) (c before d))
;;;                         --> ((a before b) (cd before d))
;;;
;;;           before-triples: ((a before b) (j before c))
;;;           same-as-groups: ((b j))
;;;                         --> ((a before b) (j before c))
;;;
;;; returns pair (<ordering> ordered-list)
;;; where <ordering> is either 'total-order or 'no-total-order

(defun impose-before-order (before-triples same-as-groups)
  (let* ((ordered-triples (sort-triples-and-same-as before-triples
						    same-as-groups))
	 (totally-ordered? (triples-have-total-before-order ordered-triples
							    same-as-groups)))
    (if totally-ordered?
	(list 'total-order ordered-triples)
	(list 'no-total-order ordered-triples))))


;;; sorts before-triples considering same-as-groups
;;;
;;; sort is *not* destructive
;;;
;;; can't just use built in function ``sort'' because not every pair
;;; of triples are directly related to one another by the ``less-than''
;;; ordering
(defun sort-triples-and-same-as (before-triples same-as-groups)
  (if (null before-triples)
      nil
      (let ((least-element (find-least-triple before-triples same-as-groups)))
	(cons least-element
	      (sort-triples-and-same-as (remove least-element
						before-triples)
					same-as-groups)))))


;;; auxilliary function for sort-triples-and-same-as
;;;
;;; note that because there may not necessarily be a total order
;;; possible on the triple-list, there may be several least elements;
;;; one is chosen arbitrarily
;;;
(defun find-least-triple (triple-list same-as-groups)
  (if (equal (length triple-list) 1)
      (first triple-list)
      (let* ((first-triple (first triple-list))
	     (lesser-triple (find-first-earlier-triple first-triple
						       (rest triple-list)
						       same-as-groups)))
	(if (null lesser-triple)
	    first-triple
	    (find-least-triple (rest triple-list) same-as-groups)))))


;;; finds first triple in triple list that is ``less than'' current triple
;;; example: (a before b) is ``less than'' (b before c)
;;;
;;; also considers same-as group info
;;; example: (a before b) is ``less than'' (j before c) if
;;;          (j b) is in the same-as-groups
(defun find-first-earlier-triple (given-triple triple-list same-as-groups)
  (if (null triple-list)
      nil
      (let* ((given-triple-unit (first given-triple))
	     (given-triple-unit-equivs (get-same-as-equivs
					given-triple-unit
					same-as-groups))
	     (current-triple (first triple-list))
	     (current-triple-value (third current-triple))
	     (current-triple-value-equivs (get-same-as-equivs
					   current-triple-value
					   same-as-groups)))
	(if (not (null (intersection given-triple-unit-equivs
				     current-triple-value-equivs)))
	    current-triple
	    (find-first-earlier-triple given-triple
				       (rest triple-list)
				       same-as-groups)))))


;;; determines if element is in one of same-as-groups
;;; if so, returns the group; otherwise returns a list of the element
(defun get-same-as-equivs (element same-as-groups)
  (cond ((null same-as-groups)
	 (list element))
	((member element (first same-as-groups))
	 (first same-as-groups))
	(t (get-same-as-equivs element (rest same-as-groups)))))


;;; determines if triples have total before order
;;; method: checks if every pair of adjacent triples is of form
;;;         (x before y) followed by (y before z) or
;;;         (x before y) followed by (w before z) where
;;;         y and w are in the same same-as group
(defun triples-have-total-before-order (ordered-triples same-as-groups)
  (cond ((null ordered-triples) t)
	((equal (length ordered-triples) 1) t)
	;; have at least 2 triples in list
	(t (let* ((triple1 (first ordered-triples))
		  (value1 (third triple1))
		  (value1-equivs (get-same-as-equivs value1 same-as-groups))
		  (triple2 (second ordered-triples))
		  (unit2 (first triple2))
		  (unit2-equivs (get-same-as-equivs unit2 same-as-groups)))
	     (if (not (null (intersection value1-equivs unit2-equivs)))
		 (triples-have-total-before-order (rest ordered-triples)
						  same-as-groups)
		 nil)))))


;;; incorporates same-as information from same-as triples
;;; into ordered-list of before-triples and strips out ``before'' keywords
;;;
;;; example: converts ((a before b) (j before c) and ((j b)) to
;;;          ((same-as a) (same-as b d e) (same-as c))
(defun impose-same-as-order (ordered-befores same-as-groups)
  (remove-duplicates 
   (reduce #'append
	   (mapcar #'(lambda (before-triple)
		       (let* ((unit (first before-triple))
			      (unit-equivs
                               (cons 'same-as
				     (get-same-as-equivs unit
							 same-as-groups)))
			      (value (third before-triple))
			      (value-equivs
			       (cons 'same-as
				     (get-same-as-equivs value
							 same-as-groups))))
			 (list unit-equivs value-equivs)))
		   ordered-befores))
   :test #'equal
   :from-end t))


;;; incorporates ``contains'' information from ``contains'' triples
;;; into ordered-list with same-as's
;;;
;;; example: ((same-as a) (same-as b d e) (same-as c)) and
;;;          ((c contains g) (d contains f)) to
;;;          (((same-as a) (contains))
;;;           ((same-as b d e) (contains f))
;;;           ((same-as c) (contains g)))

(defun impose-contains-order (ordered-befores-and-same-as contains-triples)
  (mapcar #'(lambda (same-as-element)
	      (let* ((same-as-events (rest same-as-element))
		     (contains-events
		      (find-contains-events same-as-events
					    contains-triples)))
		(if (null contains-events)
		    (list same-as-element
			  (list 'contains))
		    (list same-as-element
			  (cons 'contains
				contains-events)))))
	  ordered-befores-and-same-as))


;;; finds all events in contains-triples that are contained by
;;; some event in same-as-events
(defun find-contains-events (same-as-events contains-triples)
  (reduce #'append
	  (mapcar #'(lambda (event)
		      (let ((contained-events nil))
			(dolist (triple contains-triples)
			  (if (equal event (first triple))
			      (setf contained-events
				    (push (third triple) contained-events))))
			contained-events))
		  same-as-events)))


;;; flattens out ordered list
;;;
;;; converts from structured list to flat list
;;;
;;; example: (((same-as a) (contains))
;;;           ((same-as b d e) (contains f))
;;;           ((same-as c) (contains g))) 
;;;         to
;;;          (a b d e f c g)

(defun flatten-temporal-structure (structured-ordered-elements)
  (reduce #'append
	  (mapcar #'(lambda (element)
		      (let ((same-as-elements (rest (first element)))
			    (contains-elements (rest (second element))))
			(append same-as-elements contains-elements)))
		  structured-ordered-elements)))


;;; returns t if at least one pair of duplicates exists; otherwise nil
(defun exist-duplicate-elements? (flattened-ordered-elements)
  (cond ((null flattened-ordered-elements) nil)
	((member (first flattened-ordered-elements)
		 (rest flattened-ordered-elements))
	 t)
	(t (exist-duplicate-elements? (rest flattened-ordered-elements)))))


;;; determine if some elements are in original-subevent-list that
;;; weren't included in the list flattened-ordered-elements
;;;
;;; if so, returns those elements; otherwise nil

(defun exist-uncovered-events? (flattened-ordered-elements
				original-subevent-list)
  (set-difference original-subevent-list
		  flattened-ordered-elements))


;-----------------------------------------------------------------------
;			    Find-Undiscussed-Actors
;-----------------------------------------------------------------------

;;; arguments: Actor-List
;;;
;;; for each actor in Actor-list, places actor in result
;;; list if actor hasn't been discussed
;;;
;;; uses ``is-known-p'' function
;;;
;;; returns (a list of the) subset of Actor-list where the actors haven't
;;;         yet been discussed
;;;
;;; if user modeling not on, returns entire actor-list
;;;
;;; errors:
;;;		- condition: list of actors is empty
;;;             - returns: (error no-actors)

(defun find-undiscussed-actors (actor-list)
  (cond	((null actor-list)
	 '((error no-actors)))
	((not (user-modeling-on?))
	 (list actor-list))
	(t (list (remove-if #'(lambda (x) (is-known-p x))
			    actor-list)))))


;-----------------------------------------------------------------------
;			     Get-Function-Process
;-----------------------------------------------------------------------

;;; arguments: Object-significance-content-node
;;;
;;; retrieves process which was used in functional description
;;; in this content node
;;; 
;;; currently finds first process on the agent-in slot
;;; of the viewpoint
;;;
;;; later may want to modify to return multiple processes 
;;; (for objects with multiple functions)
;;;
;;; returns the process
;;;
;;; errors:
;;;		- condition: node is empty
;;;		- returns: (error no-significance-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;             - returns: (error error-in-object-significance-node)
;;;
;;;             - condition: no viewpoint exists in content node
;;;             - returns: (error no-viewpoint-in-significance-node)

(defun get-function-process (content-node)
  (cond ((null content-node)
	 '((error no-significance-node)))
	((content-node-has-error? content-node)
	 '((error error-in-object-significance-node)))
	(t (let ((viewpoint
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null viewpoint)
		 '((error no-viewpoint-in-significance-node))
		 (get-only-val (list viewpoint 'agent-in)))))))

	
;-----------------------------------------------------------------------
;			   Get-Functional-Processes
;-----------------------------------------------------------------------

;;; arguments: Content Node with Struct-Funct-Viewpoint
;;;
;;; traverses structural-functional viewpoint to obtain list
;;; of processes that are mentioned in the viewpoint
;;;
;;; returns (a list of) list of processes
;;;
;;; errors:
;;;		- condition: node is empty
;;;		- returns: (error no-black-box-functional-iteration-node)
;;;
;;;             - condition: an error has occurred in content
;;;                          determination at content node
;;;             - returns: (error error-in-black-box-functional-iteration-node)
;;;
;;;		- condition: no viewpoint
;;;		- returns: (error no-struct-funct-viewpoint)
;;;
;;;		- condition: struct-funct viewpoint has no top-level view
;;;		- returns: (error struct-funct-viewpoint-malformed)

(defun get-functional-processes (content-node)
  (cond ((null content-node)
	 '((error no-black-box-functional-iteration-node)))
	((content-node-has-error? content-node)
	 '((error error-in-black-box-functional-iteration-node)))
	(t (let ((struct-funct-viewpoint 
		  (get-only-val (list content-node 'kb-subgraph))))
	     (if (null struct-funct-viewpoint)
		 '((error no-struct-funct-viewpoint))
		 (let ((viewpoint-of-top-event
			(get-viewpoint-of-top-event struct-funct-viewpoint)))
		   (if (null viewpoint-of-top-event)
		       '((error struct-funct-viewpoint-malformed))
		       (list (get-local (list viewpoint-of-top-event
					      'subevents))))))))))


(defun get-viewpoint-of-top-event (struct-funct-viewpoint)
  (let ((top-event
	 (get-only-val (list struct-funct-viewpoint 'correspondence-with))))
    (get-only-val (list struct-funct-viewpoint
			'correspondence-with
			top-event
			'has-viewpoint))))


;-----------------------------------------------------------------------
;			Get-Temporal-Ordering-Relations
;-----------------------------------------------------------------------

;;; arguments: None
;;;
;;; looks up ordering relations in KB
;;;
;;; returns (a list of) a list of the ordering relations
;;;
;;; errors: none

(defun get-temporal-ordering-relations ()
  (list
   (get-local '(temporal-ordering specializations))))


;-----------------------------------------------------------------------
;			Get-Temporal-Relation-Values
;-----------------------------------------------------------------------

;;; arguments: Process, Temporal-Relation
;;;
;;; looks up values on Process.Temporal-Relation
;;;
;;; returns (a list of) a list of the ordering relations
;;;
;;; errors: none

(defun get-temporal-relation-values (process temporal-relation)
  (list
   (get-local (list process temporal-relation))))


;-----------------------------------------------------------------------
;				  Make-Triple
;-----------------------------------------------------------------------

;;; arguments: Unit, Slot, Value
;;;
;;; appends arguments to construct a triple
;;;
;;; returns (a list of) the triple
;;;
;;; errors: none

(defun make-triple (unit slot value)
  (list
   (list unit slot value)))

