;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;		     The Auxilliary Organization Functions
;;;;		     -------------------------------------


;;;; orders actual nodes by matching up to nodes in specification
;;;;     list and placing in that order
;;;;
;;;; because some actual nodes may have failed a focus function
;;;;     test, first checks to see if actual node has an
;;;;     instantiated focus condition; if so checks to see if
;;;;     it evaluated to True
;;;;
;;;; then installs previous and next-node links
;;;;
;;;; assumes focus-condition-evaluation assigns value 
;;;; that is ``real boolean'', i.e., True or False
;;;;
;;;; if specification-list is null, issues a warning when
;;;; trace-organization? is on, but returns original list of actual
;;;; nodes; this is not technically an error because there are
;;;; cases when an organization specification cannot be given in
;;;; an EDP unit, e.g., ordering the content-nodes that result from
;;;; determining content of the processes that are modulatory facilitators
;;;; of some process
;;;;
;;;; another special case is when iteration is involved:
;;;;
;;;; EDP example:
;;;;              A
;;;;             / \
;;;;  iterative X   Y non-iterative
;;;;            |    
;;;;            Z   
;;;;
;;;; Explanation Plan:
;;;;             a
;;;;           / / \
;;;;         z1  z2 y
;;;;
;;;; all the instances of X/Z need to come before all instances of Y
;;;;
;;;; this latter case is covered by the legal-with-iteration? clause


(in-package 'km)


(defun order-subtopics (actual-nodes specification-list)
  (let ((error? (illegal-ordering-call-p actual-nodes specification-list)))
    (cond (error?
	   (cond ((equal (second (first error?))
			 'null-specification-list)
		  (describe-organization actual-nodes)
		  actual-nodes)
		 ((equal (second (first error?))
			 'actual-node-not-in-specification-list)
		  (let ((iterative-interpretation
			 (legal-with-iteration? actual-nodes
						specification-list)))
		    (cond ((not (null iterative-interpretation))
			   (describe-organization iterative-interpretation)
			   iterative-interpretation)
			  (t 	 (when (trace-organization?)
				   (format t "Organization error: ~
                                              actual-node not in ~
                                              specification-list~%~%"))
				 error?))))
		 (t error?)))
	  (t (let* ((nodes-to-include (remove-if-not #'include-subtopic-p
						     actual-nodes))
		    (ordered-nodes
		     (sort nodes-to-include
			   #'<
			   :key #'(lambda (x)
				    (position (get-only-val
					       (list x 'node-type))
					      specification-list)))))
	       (install-previous-and-next ordered-nodes)
	       (describe-organization ordered-nodes)
	       ordered-nodes)))))


(defun illegal-ordering-call-p (actual-nodes specification-list)
  (cond ((null actual-nodes)
	 (when (trace-organization?)
	   (format t "Organization error: Null actual-node list~%~%"))
	 '((error null-actual-nodes)))

	((not (listp actual-nodes))
	 (when (trace-organization?)
	   (format t "Organization error: Actual-nodes not a list~%~%"))
	 '((error actual-nodes-not-a-list)))

	((null specification-list)
	 (when (trace-organization?)
	   (format t "Organization warning: Null specification-list~%~%"))
	 '((error null-specification-list)))

	((not (listp specification-list))
	 (when (trace-organization?)
	   (format t "Organization error: Specification-list not a list~%~%"))
	 '((error specification-list-not-a-list)))

	((not (all-plan-tree-nodes? actual-nodes))
	 (when (trace-organization?)
	   (format t "Organization error:~%")
	   (format t "Illegal actual-node list: ~a.~%~%"
		   actual-nodes))
	 '((error actual-nodes-not-plan-tree-nodes)))

	((not (all-edp-units? specification-list))
	 (when (trace-organization?)
	   (format t "Organization error:~%")	   
	   (format t "Illegal specification list: ~a.~%~%"
		   specification-list))
	 '((error specification-list-not-edp-units)))

	((not (all-plan-nodes-in-spec-list? actual-nodes
					    specification-list))
	 '((error actual-node-not-in-specification-list)))

	(t nil)))


;;; returns t if each node in actual-nodes is a plan tree node
;;; otherwise returns nil

(defun all-plan-tree-nodes? (actual-nodes)
  (cond ((null actual-nodes) t)
	((not (member 'explanation-plan-node
		      (ancestry* (first actual-nodes))))
	 nil)
	(t (all-plan-tree-nodes? (rest actual-nodes)))))


;;; returns t if each unit in edp-list is an edp-unit
;;; otherwise returns nil

(defun all-edp-units? (edp-list)
  (cond ((null edp-list) t)
	((not (member 'edp-unit
		      (ancestry* (first edp-list))))
	 nil)
	(t (all-edp-units? (rest edp-list)))))


;;; returns t if each node in actual-nodes is of a type that
;;;    appears in specification-list
;;; otherwise returns nil

(defun all-plan-nodes-in-spec-list? (actual-nodes specification-list)
  (if (null actual-nodes)
      t
      (let ((node-type (get-only-val (list (first actual-nodes)
					   'node-type))))
	(if (not (member node-type specification-list))
	    nil
	    (all-plan-nodes-in-spec-list? (rest actual-nodes)
					  specification-list)))))


;;; determines if all actual nodes are either
;;;          (1) the type of some nested iteration specification, or
;;;          (2) included in the standard type
;;; function called only when some actual nodes are not in specification list
;;;
;;; if all actual nodes can be accomodated, returns ordered list
;;; otherwise returns nil

(defun legal-with-iteration? (actual-nodes specification-list)
  (when (trace-organization?)
    (format t "Attempting iterative interpretation of~%~
               organization specification list.~%~%"))
  (let* ((actual-nodes-covered-pairings (find-covered-nodes
					 actual-nodes
					 specification-list))
	 (actual-nodes-not-covered (set-difference
				    actual-nodes
				    (get-covered-nodes
				     actual-nodes-covered-pairings)))
	 (iterative-nodes-covered-pairings (find-iterative-covering
					    actual-nodes-not-covered
					    specification-list)))
    (when (trace-organization?)
      (format t "Actual-nodes-covered pairings:")
      (pprint actual-nodes-covered-pairings t)
      (format t "~%~%")
      (format t "Actual nodes not covered:")
      (pprint actual-nodes-not-covered t)
      (format t "~%~%")
      (format t "Iterative-nodes-covered pairings:")
      (pprint iterative-nodes-covered-pairings t)
      (format t "~%~%"))
    (if (null iterative-nodes-covered-pairings)
	nil
	(let ((ordered-nodes
	       (impose-iterative-ordering actual-nodes
					  actual-nodes-covered-pairings
					  iterative-nodes-covered-pairings
					  specification-list)))
	  (install-previous-and-next ordered-nodes)
	  ordered-nodes))))


;;; finds all actual nodes and their pairing with items in
;;; specification list
;;;
;;; returns association list of form
;;;          (... (actual-node-i . node-type-i) ...)

(defun find-covered-nodes (actual-nodes specification-list)
  (if (null actual-nodes)
      nil
      (let ((node-type (get-only-val (list (first actual-nodes)
					   'node-type))))
	(if (not (member node-type specification-list))
	    (find-covered-nodes (cdr actual-nodes)
				specification-list)
	    (acons (first actual-nodes)
		   node-type
		   (find-covered-nodes (cdr actual-nodes)
				       specification-list))))))


;;; takes actual node pairings generated in find-covered-nodes
;;;
;;; returns list of all covered nodes, i.e., all the keys in the list

(defun get-covered-nodes (actual-nodes-covered-pairings)
  (mapcar #'(lambda (pairing)
	      (first pairing))
	  actual-nodes-covered-pairings))


;;; takes all nodes not covered in ``normal'' manner
;;;
;;; first checks if each actual node is a content node (only
;;; content nodes are involved in iteration, so if node isn't
;;; a content node, it can't be covered with an iterative
;;; interpretation)
;;;
;;; determines if each actual node can be covered by tracing up
;;; nested iterations in EDP to find element in specification list
;;;
;;; if each actual node in actual-nodes-not-covered can be covered
;;; in this manner, returns association list of pairings
;;;          (... (actual-node-i . iterative-node-type-i) ...)
;;;
;;; otherwise returns nil

(defun find-iterative-covering (actual-nodes-not-covered specification-list)
  (if (not (all-actual-nodes-content-nodes? actual-nodes-not-covered))
      nil
      (let ((iterative-covering
	     (find-iterative-covering-aux actual-nodes-not-covered
					  specification-list)))
	(if (not (member 'error iterative-covering))
	    iterative-covering
	    nil))))


;;; returns t if all actual nodes are of type content-specification
;;; otherwise nil

(defun all-actual-nodes-content-nodes? (actual-nodes-not-covered)
  (cond ((null actual-nodes-not-covered) t)
	((member 'content-node
		 (get-local (list (first actual-nodes-not-covered)
				  'generalizations)))
	 (all-actual-nodes-content-nodes? (cdr actual-nodes-not-covered)))
	(t nil)))


(defun find-iterative-covering-aux (actual-nodes-not-covered
				    specification-list)
  (if (null actual-nodes-not-covered)
      nil
      (let ((first-node-covering (get-iterative-covering-for-single-node
				  (first actual-nodes-not-covered))))
	(if (or (null first-node-covering)
		(not (member first-node-covering specification-list)))
	    '(error)
	    (acons (first actual-nodes-not-covered)
		   first-node-covering
		   (find-iterative-covering-aux (rest actual-nodes-not-covered)
						specification-list))))))


;;; climbs up edp-units to get to top of nested iteration
;;;
;;; assumes actual node is a content node
;;;
;;; returns type of actual node if it was generated from an
;;; iterative content specification
;;;
;;; otherwise returns nil

(defun get-iterative-covering-for-single-node (actual-node)
  (do* ((edp-unit (get-only-val (list actual-node 'node-type))
		  parent-of-edp-unit)
	(parent-of-edp-unit (get-only-val (list edp-unit
						'content-specification-of))
			    (get-only-val (list edp-unit
						'content-specification-of))))
	((or (member (get-only-val (list parent-of-edp-unit
					 'generalizations))
		     '(topic elaboration))
	     (null edp-unit))
	 edp-unit)))

		   
;;; assumes all nodes are covered in either
;;;        (1) actual-nodes-covered-pairings, or
;;;        (2) iterative-nodes-covered-pairings
;;;
;;; returns list of ordered nodes based on specification-list
;;; and the pairings

(defun impose-iterative-ordering (actual-nodes
				  actual-nodes-covered-pairings
				  iterative-nodes-covered-pairings
				  specification-list)
  (sort actual-nodes
	#'<
	:key #'(lambda (x)
		 (position
		  (let ((possible-node-type
			 (assoc x actual-nodes-covered-pairings)))
		    (if (null possible-node-type)
			(cdr (assoc x iterative-nodes-covered-pairings))
			(cdr possible-node-type)))
		  specification-list))))


;;; determines if a plan-node should be included
;;; returns T if either (1) node's focus condition evaluated to True
;;;               or    (2) node has no focus condition
;;; returns nil if node's focus condition evaluated to False

(defun include-subtopic-p (plan-node)
  "Predicate for determining whether a plan node should be included"
  (let ((value (get-only-val (list plan-node
				  'focus-condition-evaluation))))
    (or (null value)
	(equal value 'true))))


;;; installs previous and next node links

(defun install-previous-and-next (ordered-node-list)
  "Installs previous and next node links"
  (mapl #'(lambda (x)
	    (when (not (null (rest x)))
	      (let ((current (first x))
		    (next (second x)))
		(connect-to-next current next)		  
		(connect-to-previous next current))))
	ordered-node-list))


(defun connect-to-previous (current-node previous-node)
  "Connects one plan node to `previous' plan node"
  (put-local (list current-node 'previous-node) (list previous-node)))


(defun connect-to-next (current-node next-node)
  "Connects one plan node to `next' plan node"
  (put-local (list current-node 'next-node) (list next-node)))


;;; trace of result of organization

(defun describe-organization (ordered-nodes)
  (when (trace-organization?)
    (format t "Resulting organization:")
    (pprint ordered-nodes t)
    (format t "~%~%")))