;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;		       The Plan-Node Operation Functions
;;;;		       ---------------------------------


;;;; This file contains the functions for creating, removing, and
;;;; locating nodes in Explanation Plan Trees.  It has 4 sections
;;;; of functions:
;;;;
;;;;                 - Node Creation Functions
;;;;                 - Node Creation Test Calls
;;;;                 - Node Removal Functions
;;;;                 - Node Location Functions
;;;;
;-----------------------------------------------------------------------
;			 Plan Node Function Directory
;-----------------------------------------------------------------------
;;;;
;;;; Node Creation Functions
;;;; -----------------------
;;;; (construct-explanation-plan-header verbosity)
;;;; (construct-exposition-node edp-type verbosity primary-concept)
;;;; (construct-topic-node topic-type parent-exposition-node)
;;;; (construct-content-node content-type parent-node kind-of-parent-node)
;;;; (construct-elaboration-node elaboration-type parent-content-node)
;;;;
;;;; Node Creation Test Calls
;;;; ------------------------
;;;; (test-node-creation-fns)
;;;;
;;;; Node Removal Function
;;;; ---------------------
;;;; (clear-out-plan-nodes)
;;;;
;;;; Node Location Functions
;;;; -----------------------
;;;; (current-explanation-plan-header)
;;;; (current-exposition-node)
;;;; (current-topic-node)
;;;; (current-t-content-node)
;;;; (current-e-content-node)
;;;; (current-elaboration-node)
;;;; (find-node &key parent child-slot node-type)
;;;;
;;;;
;-----------------------------------------------------------------------
;			  The Node Creation Functions
;-----------------------------------------------------------------------


;;;; There are 5 functions for creating nodes in an Explanation Plan:
;;;;
;;;;        - construct-explanation-plan-header
;;;;
;;;;        - construct-exposition-node
;;;;
;;;;        - construct-topic-node
;;;;
;;;;        - construct-content-node
;;;;
;;;;        - construct-elaboration-node
;;;;
;;;; There is also a test function, test-node-creation-fns, that
;;;; calls each of these functions to create a node of each type.


(in-package 'km)


(defun construct-explanation-plan-header (verbosity)
  "Constructs new Explanation Plan header"
  (let ((new-header (gentemp "EXPLANATION-PLAN-HEADER-")))
    (when (trace-node-creation?)
      (format t "Creating new explanation plan header: ~a.~%~%"
	      new-header))
    (add-val-local '(explanation-plan-header specializations)
		   new-header
		   :location :start)
    (put-local (list new-header 'generalizations)
	       '(explanation-plan-header))
    (put-local (list new-header 'verbosity-setting)
	       (list verbosity))
    (put-local (list new-header 'number-of-content-nodes)
	       (list 0))
    (put-local (list 'knight-global-state
		     'current-explanation-plan-header)
	       (list new-header))
    new-header))


;;; constructs new exposition node
;;;
;;; assumes the new exposition node will be the subject of the
;;; explanation plan header; this will need to be changed when
;;; edps start calling other edps

(defun construct-exposition-node (edp-type verbosity primary-concept)
  "Constructs new Exposition node"
  (let ((new-exposition-node (gentemp "EXPOSITION-NODE-")))
    (when (trace-node-creation?)
      (format t "~%~%Creating new exposition node: ~a.~%"
	      new-exposition-node)
      (format t "Exposition node type: ~a.~%~%"
	      edp-type))
    (add-val-local '(exposition-node specializations) 
		   new-exposition-node
		   :location :start)
    (put-local (list new-exposition-node 'generalizations)
	       '(exposition-node))
    (put-local (list new-exposition-node 'subject-of-header)
	       (list (current-explanation-plan-header)))
    (put-local (list (current-explanation-plan-header) 'explanation-subject)
	       (list new-exposition-node))
    (put-local (list new-exposition-node 'node-type)
	       (list edp-type))
    (put-local (list new-exposition-node 'primary-concept)
	       (list primary-concept))
    (put-local (list new-exposition-node 'verbosity-setting)
	       (list verbosity))
    (put-local (list new-exposition-node 'topics-considered)
	       (get-local (list edp-type 'topic-list)))
    (put-local (list 'knight-global-state
		     'current-exposition-node)
	       (list new-exposition-node))
    new-exposition-node))


(defun construct-topic-node (topic-type parent-exposition-node)
  "Constructs new Topic node"
  (let ((new-topic-node (make-node-name topic-type)))
    (when (trace-node-creation?)
      (format t "Creating new topic node: ~a.~%"
	      new-topic-node)
      (format t "Topic node type: ~a.~%~%"
	      topic-type))
    (add-val-local '(topic-node specializations)
		   new-topic-node
		   :location :start)
    (put-local (list new-topic-node 'generalizations)
	       '(topic-node))
    (put-local (list new-topic-node 'node-type)
	       (list topic-type))
    (put-local (list new-topic-node 'child-of-node)
	       (list parent-exposition-node))
    (add-val-local (list parent-exposition-node 'topic-nodes)
		   new-topic-node
		   :location :start)
    (put-local (list new-topic-node 'verbosity-setting)
	       (list (get-only-val (list parent-exposition-node
					 'verbosity-setting))))
    (put-local (list new-topic-node 'centrality)
	       (list (let ((centrality
			    (get-only-val (list topic-type
						'centrality))))
		       (if centrality
			   centrality
			   'medium)))) ;centrality defaults to medium
    (put-local (list new-topic-node 'focus-condition)
	       (list 
		;; default is true
		(let ((focus-condition 
		       (get-only-val (list topic-type 'focus-condition))))
		  (if focus-condition
		      focus-condition
		      'true))))
    (put-local (list 'knight-global-state
		     'current-topic-node)
	       (list new-topic-node))
    new-topic-node))


;;; note: kind-of-parent-node must be either Topic or Elaboration
(defun construct-content-node (content-type parent-node kind-of-parent-node)
  "Constructs new Content node"
  (let ((new-content-node (make-node-name content-type)))
    (when (trace-node-creation?)
      (format t "Creating new content node: ~a.~%"
	      new-content-node)
      (format t "Content node type: ~a.~%~%"
	      content-type))
    (add-val-local '(content-node specializations)
		   new-content-node
		   :location :start)
    (put-local (list new-content-node 'generalizations)
	       '(content-node))
    (put-local (list new-content-node 'node-type)
	       (list content-type))
    (put-local (list new-content-node 'child-of-node)
	       (list parent-node))
    (put-local (list new-content-node 'verbosity-setting)
	       (list (get-only-val (list parent-node 'verbosity-setting))))
    (add-val-local (list parent-node 'content-nodes)
		   new-content-node
		   :location :end) ;; J.L. made end for correct order 6-4-94
    (put-local (list new-content-node 'content-specification-template)
	       (get-local (list content-type
				'content-specification-template)))
    (let ((elabs-considered (get-local (list content-type
					     'elaborations))))
      (when elabs-considered
	(put-local (list new-content-node 'elaborations-considered)
		   elabs-considered)))
    (copy-all-values (list content-type 'local-variables)
    		     (list new-content-node 'local-variables))
    (put-local (list (current-explanation-plan-header)
		     'number-of-content-nodes)
	       (list (1+ (get-only-val (list (get-only-val
					      '(knight-global-state
						current-explanation-plan-header))
					     'number-of-content-nodes)))))
    (case kind-of-parent-node
      ((topic) (put-local '(knight-global-state	current-t-content-node)
			  (list new-content-node)))
      ((elaboration) (put-local '(knight-global-state current-e-content-node)
				(list new-content-node))))
    new-content-node))
     

(defun construct-elaboration-node (elaboration-type parent-content-node)
  "Constructs new Elaboration node"
  (let ((new-elaboration-node (make-node-name elaboration-type)))
    (when (trace-node-creation?)
      (format t "Creating new elaboration node: ~a.~%"
	      new-elaboration-node)
      (format t "Elaboration node type: ~a.~%~%"
	      elaboration-type))
    (add-val-local '(elaboration-node specializations)
		   new-elaboration-node
		   :location :start)	;places the new value in
					;the first position
    (put-local (list new-elaboration-node 'generalizations)
	       '(elaboration-node))
    (put-local (list new-elaboration-node 'node-type)
	       (list elaboration-type))
    (put-local (list new-elaboration-node 'child-of-node)
	       (list parent-content-node))
    (add-val-local (list parent-content-node 'elaboration-nodes)
		   new-elaboration-node
		   :location :start)
    (put-local (list new-elaboration-node 'verbosity-setting)
	       (list (get-only-val (list parent-content-node
					 'verbosity-setting))))
    (put-local (list new-elaboration-node 'centrality)
	       (list (let ((centrality
			    (get-only-val (list elaboration-type
						'centrality))))
		       (if centrality
			   centrality
			   'medium)))) ;centrality defaults to medium
    (put-local (list new-elaboration-node 'focus-condition)
	       (list
		;; default is true
		(let ((focus-condition
		       (get-only-val (list elaboration-type
					   'focus-condition))))
		  (if focus-condition
		      focus-condition
		      'true))))
    (put-local '(knight-global-state current-elaboration-node)
	       (list new-elaboration-node))
    new-elaboration-node))


;;; gensym's a name of the given node type
(defun make-node-name (node-type)
  (gentemp (concatenate 'string
			(string node-type)
			"-")))


;---------------------------------------------------------------------
;			   Node Creation Test Calls
;---------------------------------------------------------------------


;(defun test-node-creation-fns ()
;  "Tests out node creation functions"
;  (let* ((my-header (construct-explanation-plan-header 'medium))
;	 (my-expo-node (construct-exposition-node 'process-edp
;						  'medium
;						  'growth))
;	 (my-topic-node (construct-topic-node 'process-significance
;					      my-expo-node))
;	 (my-content-node (construct-content-node 'core-connection
;						  my-topic-node
;						  'topic))
;	 (my-elaboration-node (construct-elaboration-node 'core-elaboration
;							  my-content-node)))))


;-----------------------------------------------------------------------
;			  The Node Removal Functions
;-----------------------------------------------------------------------


;;; Clears out all plan nodes but viewpoints
(defun clear-out-plan-nodes ()
  "Removes all plan tree nodes from KB, including viewpoints"
  (remove-all-children 'explanation-plan-header)
  (remove-all-children 'exposition-node)
  (remove-all-children 'topic-node)
  (remove-all-children 'content-node)
  (remove-all-children 'elaboration-node)
  (remove-all-children 'viewpoint))
		       


;;; Clears out viewpoints (use this one carefully!)
(defun clear-out-view-points ()
  "Severs all viewpoints from Viewpoint frame."
  (put-local '(viewpoint instances)
	     nil)
  (put-local '(vp-shell instances)
	     nil))


;-----------------------------------------------------------------------
;			  The Node Location Functions
;-----------------------------------------------------------------------


;;; finds nodes in current explanation plan tree that are
;;; pointed to by global state variables


(defun current-explanation-plan-header ()
  (get-only-val '(knight-global-state current-explanation-plan-header)))


(defun current-exposition-node ()
  (get-only-val '(knight-global-state current-exposition-node)))


(defun current-topic-node ()
  (get-only-val '(knight-global-state current-topic-node)))


(defun current-t-content-node ()
  (get-only-val '(knight-global-state current-t-content-node)))


(defun current-e-content-node ()
  (get-only-val '(knight-global-state current-e-content-node)))


(defun current-elaboration-node ()
  (get-only-val '(knight-global-state current-elaboration-node)))


;;; selects a node from the Explanation Plan Tree that 
;;;
;;;   (1) has Parent as its parent in the tree, and
;;;
;;;   (2) is connected to Parent via Child-Slot, 
;;;       e.g., ``content-nodes'', and
;;;
;;;   (3) is of type Node-Type, e.g., a content node
;;;       is of type ``Core-Connection'' if it was
;;;       constructed by a content specification named
;;;       Core-Connection
;;;
;;; assumes parent has only one child node of the given node-type

(defun find-node (&key parent child-slot node-type)
  "Finds requested node in Explanation Plan Tree"
  (let ((candidate-nodes (get-local (list parent child-slot))))
    (find node-type candidate-nodes
	  :key #'(lambda (x) (get-only-val (list x 'node-type))))))
	  
