;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;			   The Realization Functions
;;;;			   -------------------------

;;;; This file contains functions that invoke the Natural Language
;;;; Generator.


(in-package 'km)


;;; given: a paragraph-structure: a list of lists of viewpoints,
;;;        i.e., ((VP1 VP2) (VP3) (VP4 VP5))
;;;
;;; assumptions: - each viewpoint in the structure is error-free
;;;
;;;              - each embedded list denotes a paragraph
;;;
;;; returns a list of lists of sentences

(defun realize-explanation (paragraph-structure &key (string nil))
  (let* ((parag-sentence-list
	  (remove-if #'null
		     (mapcar #'realize-paragraph
			     paragraph-structure)))
	 (parag-sentence-list-revised
	  (revise-paragraph-grouping-v2 parag-sentence-list))
	 (parag-sentence-list-revised-no-dupes
	  (remove-duplicate-sentences parag-sentence-list-revised))
	 (parag-sentence-list-revised-no-dupes-with-pronouns
	  (pronominalize-explanation parag-sentence-list-revised-no-dupes)))
	   
    ;(format t "~%parag-sentence-list: ~a~%"
    ;	    parag-sentence-list)
    ;(format t "~%parag-sentence-list-revised: ~a~%"
    ;	     parag-sentence-list-revised)
    ;(format t "~%parag-sentence-list-revised-no-dupes: ~a~%"
    ;        parag-sentence-list-revised-no-dupes)
    ;(format t "~%parag-sentence-list-no-dupes-with-pronouns: ~a~%"
    ;        parag-sentence-list-revised-no-dupes-with-pronouns)

    (put-local (list (current-exposition-node)
		     'explanation)
	       parag-sentence-list-revised-no-dupes-with-pronouns)

    (if (not string) (progn (format t "~%")

    (mapcar #'(lambda (single-paragraph-string-list)
		(make-paragraph-text single-paragraph-string-list))
	    parag-sentence-list-revised-no-dupes-with-pronouns)
    (values))

    (eval `(string-append ,@(mapcar #'add-blanks-to-sent (uncanonize
          parag-sentence-list-revised-no-dupes-with-pronouns)))))))


(defun realize-paragraph (viewpoint-list)
  (let ((strings-and-nils-list (mapcar #'realize-viewpoint
				       viewpoint-list)))
    (if (not (member nil strings-and-nils-list))
	(reduce #'append strings-and-nils-list)
	(cascade-linguistic-pruning viewpoint-list
				    strings-and-nils-list))))


;;; given: a viewpoint
;;;
;;; returns *only* a list of sentences produced from that viewpoint
;;;
;;; if a unification failure occurs for some FD, nil is *not*
;;; returned by that failure
;;;
;;; only if every FD fails should nil be returned
;;;
;;; diagrammatic example:
;;;
;;;    input: viewpoint1
;;;
;;;           viewpoint1 --> FD1,          FD2,       FD3
;;;                           |             |          |
;;;                           |             |          |
;;;                           |             |          |
;;;                           |             |          |
;;;                           v             v          v
;;;                  Sent1, Sent2, Sent3   Fail   Sent4, Sent5
;;;
;;;    output: (Sent1, Sent2, Sent3, Sent4, Sent5)

(defun realize-viewpoint (viewpoint)
  (remove "<fail>"
	  (mapcar #'(lambda (x)
		      (make-sentence (uni-string (first x) :limit 2500
					    :non-interactive t)
				     (second x)))
		  (2fd viewpoint))
	  :test #'string-equal))


(defun make-sentence (string1 string2)
    (cond ((null string2)  string1)
	  ((not (null string1))
	   (concatenate 'string (substitute #\space
					    #\.
					    string1)
			string2 "."))
	  (t  nil)))


;--------------------------------------------------------------------
;			  Cascade-Linguistic-Pruning
;--------------------------------------------------------------------
;;;
;;; this function deals with the problem of the realization component
;;; pruning a viewpoint Vi that would have generated a sentence Si,
;;; and there is a subsequent viewpoint Vj that generates a sentence
;;; Sj, but now that Si is not there, Sj makes no sense out of context.
;;;
;;; Solution:
;;;   each time a viewpoint is pruned, prune all sentences produced
;;;   from viewpoints that are elaborations of the viewpoint whose
;;;   sentence was pruned
;;;
;;; Technique:
;;;   
;;;    given: - viewpoint-list
;;;           - string-and-nil-list (result of realization)
;;;
;;;    if a nil appears in string-and-nil-list then
;;;       find viewpoint V that generated it
;;;       find content-node C associated with V in explanation plan
;;;       if C has elaborations (assumed to be in this paragraph)
;;;          for each elaboration E
;;;              for each content node C-elab of E
;;;                  for each viewpoint of V-elab of C-elab
;;;                     remove sentence produced by V-elab from list
;;;
;;; Note: this function had to be implemented at the last minute
;;;       so it's very inelegant (lots of unneccessary work done by code)
;;;
;;;       the way this should be implemented is to store the strings
;;;       produced by a viewpoint in the explantion plan (since we
;;;       don't do that now, we have to regenerate the mappings
;;;       from viewpoint-list to strings-and-nils-list)
;;;
;;;       currently not general:
;;;            - only implemented for one level of elaboration,
;;;              i.e., top-level content nodes and their elaborations
;;;
(defun cascade-linguistic-pruning (viewpoint-list strings-and-nils-list)
  (let* ((vp-list-to-strings-mapping
	  (make-vp-list-to-strings-mapping viewpoint-list
					   strings-and-nils-list))
	 (new-list-to-strings-mapping
	  (cascade-linguist-pruning-aux vp-list-to-strings-mapping))
	 (new-strings-and-nils-list
	  (return-to-strings-and-nils-list new-list-to-strings-mapping)))
    (reduce #'append
	    new-strings-and-nils-list)))


(defun cascade-linguist-pruning-aux (vp-list-to-strings-mapping)
  (let ((first-failed-viewpoint (get-first-failed-vp
				 vp-list-to-strings-mapping))
	(new-vp-list-to-strings-mapping vp-list-to-strings-mapping))
    (cond (first-failed-viewpoint

	   ;; remove the intial viewpoint
	   (setf new-vp-list-to-strings-mapping
		 (remove-element-from-vp-list-to-strings-mapping
		  first-failed-viewpoint
		  new-vp-list-to-strings-mapping))

	   ;; remove elaborative viewpoints
	   (let* ((content-node-of-failure
		   (get-only-val (list first-failed-viewpoint
				       'kb-subgraph-of)))
		  (elab-nodes-of-failed-content-node
		   (get-local (list content-node-of-failure
				    'elaboration-nodes))))

	     ;(format t "~%Elab nodes of failed content node: ~a~%"
	     ;        elab-nodes-of-failed-content-node)

	     ;; if there are elaborative nodes, remove their viewpoints
	     (when elab-nodes-of-failed-content-node
	       (dolist (elab-node elab-nodes-of-failed-content-node)
		 (dolist (content-node
			   (get-local (list elab-node 'content-nodes)))
		   (let ((elab-viewpoint
			  (get-only-val (list content-node 'kb-subgraph))))

		     ;(format t "~%Elab viewpoint: ~a~%"
		     ;	     elab-viewpoint)

		     ;; elab-viewpoint is a viewpoint that was generated
		     ;; from a content node of an elaboration of the
		     ;; content node that produced first-failed-viewpoint
		     ;;
		     ;; so: remove the (elab-viewpoint sentence-list)
		     ;;     from the vp-list-to-strings-mapping
		 
		     (setf new-vp-list-to-strings-mapping
			   (remove-element-from-vp-list-to-strings-mapping
			    elab-viewpoint
			    new-vp-list-to-strings-mapping)))))))

	   (cascade-linguist-pruning-aux new-vp-list-to-strings-mapping))

	  (t new-vp-list-to-strings-mapping))))


;;; for each V in viewpoint-list, pairs it to the
;;; element in strings-and-nils-list that 
(defun make-vp-list-to-strings-mapping (viewpoint-list
					strings-and-nils-list)
  (pair-up-lists viewpoint-list	strings-and-nils-list))


;;; pairs up elements of lists L1 and L2
;;; assumes: their number of elements is equal
(defun pair-up-lists (L1 L2)
  (if L1
      (cons (list (first L1)
		  (first L2))
	    (pair-up-lists (rest L1) (rest L2)))))


(defun get-first-failed-vp (vp-list-to-strings-mapping)
  (if vp-list-to-strings-mapping
      (let ((first-elem (first vp-list-to-strings-mapping)))
	(if (null (second first-elem))
	    (first first-elem)
	    (get-first-failed-vp (rest vp-list-to-strings-mapping))))))


(defun remove-element-from-vp-list-to-strings-mapping
    (elab-viewpoint vp-list-to-strings-mapping)
  (if vp-list-to-strings-mapping
      (let ((first-elem (first vp-list-to-strings-mapping)))
	(if (equal (first first-elem) elab-viewpoint)
	    (rest vp-list-to-strings-mapping)
	    (cons first-elem
		  (remove-element-from-vp-list-to-strings-mapping
		   elab-viewpoint
		   (rest vp-list-to-strings-mapping)))))))


;;; given:   a list of the form ((a 1) (b 2) (c 3))
;;;
;;; returns: a list of the form (1 2 3)
(defun return-to-strings-and-nils-list (list-to-strings-mapping)
  (if list-to-strings-mapping
      (let ((first-elem (first list-to-strings-mapping)))
	(cons (second first-elem)
	      (return-to-strings-and-nils-list
	       (rest list-to-strings-mapping))))))

	    
;--------------------------------------------------------------------
;			   Revise-Paragraph-Grouping
;--------------------------------------------------------------------
;				   Version 1
;--------------------------------------------------------------------
;;;
;;; this function attempts to improve a paragraph clustering
;;;
;;; given: a list of paragraph cluster of sentences, e.g., 
;;;
;;;        ((s1 s2) (s3) (s4 s5))
;;;
;;; improvements: eliminates paragraphs that have only one sentence
;;;               by merging all 1-sentence paragraphs with
;;;               the paragraphs above them
;;;
;;;               if the first paragraph has only one sentence,
;;;               it is moved down
;;;
;;; deficiency: given ((A 1) (B) (C) (D) (E) (F G))
;;;             the system will dumbly return
;;;                   ((A 1 B C D E) (F G))
;;;             rather than leaving (A 1) alone and merging
;;;             (B), (C), (D), and (E) into a new one
;;;
;;; note: Art didn't like the results of this version, so we
;;;       wrote version 2 below

(defun revise-paragraph-grouping-v1 (paragraph-cluster-list)
  (if (and paragraph-cluster-list
	   (> (length paragraph-cluster-list) 1))
      (let ((cluster-1 (first paragraph-cluster-list))
	    (cluster-2 (second paragraph-cluster-list)))
	(if (equal (length cluster-1) 1)
	    (let ((new-cluster (append cluster-1 cluster-2)))
	      (revise-paragraph-grouping-aux
	       (cons new-cluster
		     (rest (rest paragraph-cluster-list)))))
	    (revise-paragraph-grouping-aux paragraph-cluster-list)))
      paragraph-cluster-list))
	     

(defun revise-paragraph-grouping-aux (paragraph-cluster-list)
  (if (and paragraph-cluster-list
	   (> (length paragraph-cluster-list) 1))
      (let ((cluster-1 (first paragraph-cluster-list))
	    (cluster-2 (second paragraph-cluster-list)))
	(if (equal (length cluster-2) 1)
	    (let ((new-cluster (append cluster-1 cluster-2)))
	      (revise-paragraph-grouping-aux
	       (cons new-cluster
		     (rest (rest paragraph-cluster-list)))))
	    (cons cluster-1
		  (revise-paragraph-grouping-aux
		   (rest paragraph-cluster-list)))))
      paragraph-cluster-list))

	    
;--------------------------------------------------------------------
;			   Revise-Paragraph-Grouping
;--------------------------------------------------------------------
;				   Version 2
;--------------------------------------------------------------------
;;;
;;; this function attempts to improve a paragraph clustering
;;; given: a list of paragraph cluster of sentences, e.g., 
;;;
;;;        ((s1 s2) (s3) (s4 s5))
;;;
;;; improvements: - eliminates a first paragraph that has only
;;;                 one sentence
;;;               - if there are 4 or fewer sentences, merges them
;;;                 all into one paragraph

(defun revise-paragraph-grouping-v2 (paragraph-cluster-list)
  (if paragraph-cluster-list
      (let ((flat-sentence-list (flatten paragraph-cluster-list)))
	(if (<= (length flat-sentence-list) 4)
	    (list flat-sentence-list)
	    (if (and (equal (length (first paragraph-cluster-list))
			    1)
		     (>= (length paragraph-cluster-list)
			 2))
		(let ((new-cluster (append (first paragraph-cluster-list)
					   (second paragraph-cluster-list)))
		      (other-clusters (rest (rest paragraph-cluster-list))))
		  (cons new-cluster
			other-clusters))
		paragraph-cluster-list)))))


;--------------------------------------------------------------------
;			  Remove-Duplicate-Sentences
;--------------------------------------------------------------------
;;;
;;; given: a list of lists of sentences, e.g., 
;;;            ((s1 s2) (s3) (s4 s2))
;;;
;;; returns: a list of lists of sentences in which duplicate sentences
;;;          have been removed but with the original structure
;;;          still intact      
;;;
;;;          always removes all latter appearances and maintains the
;;;          first
;;;
;;; example above returns: 
;;;            ((s1 s2) (s3) (s4))
;;;
(defun remove-duplicate-sentences (paragraph-cluster-list)
  (if (and paragraph-cluster-list
	   (listp paragraph-cluster-list)
	   (listp (first paragraph-cluster-list)))
      (let* ((cluster-1 (first paragraph-cluster-list))
	     (cluster-1-rev
	      (remove-duplicate-sentences-in-cluster cluster-1))
	     (revised-cluster-list-rest (rest paragraph-cluster-list)))
	(dolist (sent cluster-1-rev)
	  (setf revised-cluster-list-rest
		(remove-sent-from-cluster-list sent
					       revised-cluster-list-rest)))
	(cons cluster-1-rev
	      (remove-duplicate-sentences revised-cluster-list-rest))))) 


;;; given: - a sentence
;;;        - a list of cluster
;;; 
;;; for each appearance of sentence in any of the clusters in
;;; cluster-list, removes that sentence
;;;
;;; if removing a sentence causes a cluster to become nil, then
;;; that cluster is removed
;;;
(defun remove-sent-from-cluster-list (sent cluster-list)
  (remove nil
	  (mapcar #'(lambda (cluster)
		      (remove-all-sents-from-cluster sent cluster))
		  cluster-list)))


;;; given: - a sentence
;;;        - a list of sentences
;;;
;;; returns: list of sentences with all instances of sentences
;;;          removed from it
(defun remove-all-sents-from-cluster (sent parag-cluster)
  (remove sent parag-cluster :test #'string-equal))

  
;;; given: a list of sentences
;;;
;;; returns: list of sentences with all but first instance of each
;;;          sentence removed from list
(defun remove-duplicate-sentences-in-cluster (paragraph-cluster)
  (remove-duplicates paragraph-cluster
		     :from-end t
		     :test #'string-equal))


;--------------------------------------------------------------------
;			   Pronominalize Explanation
;--------------------------------------------------------------------
;;; given: a list of sentences
;;;
;;; returns: the list with the following changes:
;;;          for each sentence except the first
;;;              IF the first word is the string associated with
;;;                            primary concept
;;;                                            AND
;;;               (the previous sentence begins with the string OR
;;;                 the previous sentence has already been pronominalized)
;;;                                            AND
;;;               the following character is not a comma
;;;                      (to rule out: ``It, which contains carbon, is ...'')
;;;              THEN
;;;                  the first word is pronominalized by ...
;;;    the pronominalization method:
;;;          for a singular concept: substitute ``It''
;;;          for a plural concept:  substitute ``They''
;;;
(defun pronominalize-explanation (list-of-paragraphs)
  (mapcar #'(lambda (paragraph)
	      (pronominalize-paragraph paragraph))
	  list-of-paragraphs))


(defun pronominalize-paragraph (paragraph)
  (let* ((primary-concept
	  (get-only-val (list (current-exposition-node)
			      'primary-concept)))
	 (pronoun (compute-pronoun primary-concept))
	 (concept-strings (compute-concept-strings primary-concept)))
    (pronominalize-paragraph-aux paragraph concept-strings pronoun)))


;;; given: a concept
;;; returns: a string representing the capitalized pronoun for
;;;          that concept
;;;
;;;          (either "It" or "They")
;;;
(defun compute-pronoun (concept)
  (let ((number (compute-number concept)))
    (if number
	(if (equal number 'singular)
	    "It"
	    "They"))))


;;; if top-level-frame is a concept that is singular, then
;;; returns singular
;;; otherwise returns plural
;;;
(defun compute-number (top-level-frame)
  (if (getobj top-level-frame)
      (if (member2 '(number plural)
		   (first (get-local
			   `(,top-level-frame
			     lexical-info
			     li-primary lex-fd))))
	  'plural
	  'singular)))


;;; given: a concept
;;;
;;; returns: 3 strings representing that concept might appear
;;;          in a sentence 
;;;             (1) "Concept"
;;;             (2) "A concept"
;;;             (3) "The concept"
;;;
(defun compute-concept-strings (concept)
  (let* ((concept-fd (make-np concept))
	 (countable-no-fd (any-to-count-no concept-fd))
         (definite-no-fd (any-to-def-no concept-fd))
         (definite-yes-fd (any-to-def-yes concept-fd)))
    (mapcar #'remove-period
	    (mapcar #'(lambda (fd)
			(uni-string fd
				    :limit 1200
				    :non-interactive t))
		    (list countable-no-fd
			  definite-no-fd
			  definite-yes-fd)))))


;;; given: a string ending in "."
;;;
;;; returns: the string without the "."
;;;
(defun remove-period (string-with-terminal-period)
  (let ((strng-length (length string-with-terminal-period)))
    (subseq string-with-terminal-period 0 (1- strng-length))))


(defun pronominalize-paragraph-aux (paragraph concept-strings pronoun)
  (if (> (length paragraph) 1)
      (let* ((sent-1 (first paragraph))
	     (sent-2 (second paragraph))
	     (pronominalized-sent-2
	      (pronominalize-sent concept-strings pronoun sent-2 sent-1)))
	(cons sent-1
	      (pronominalize-paragraph-aux (cons pronominalized-sent-2
						 (rest (rest paragraph)))
					   concept-strings
					   pronoun)))
      paragraph))


;;; substitutes pronoun for string representing concept if
;;;         the string appears first in the sentence       AND
;;;         the previous sentence begins with the concept,
;;;             which may itself have been pronominalized
;;;         AND the subsequent character is not a comma
;;;         AND the subsequent character is a blank
;;;         AND the sentence begins with the same string as
;;;             the previous sentence (unless the previous sentence
;;;             begins with pronoun)
(defun pronominalize-sent (concept-strings pronoun sent previous-sent)
  (let ((pronominalized-result
	 (pronominalize-sent-aux-1 concept-strings
				   pronoun
				   sent))
	(prev-sent-beginning
	 (previous-sent-begins-with-concept concept-strings
					    pronoun
					    previous-sent)))
    (if (and pronominalized-result
	     prev-sent-beginning
	     (or (equal prev-sent-beginning pronoun)
		 (sent-begins-with-string prev-sent-beginning
					  (length prev-sent-beginning)
					  sent
					  (length sent))))
	pronominalized-result
	sent)))


;;; for each possible-lead concept string, checks if substitution
;;; can be made
;;;
;;; if encounters one that can, returns it
;;; otherwise, checks others
;;;
(defun pronominalize-sent-aux-1 (possible-lead-concept-strings
				 pronoun sent)
  (if possible-lead-concept-strings
      (let* ((first-string (first possible-lead-concept-strings))
	     (result (pronominalize-sent-aux-2 first-string pronoun sent)))
	(if result
	    result
	    (pronominalize-sent-aux-1 (rest possible-lead-concept-strings)
				      pronoun
				      sent)))))


;;; given: - a concept string candidate, e.g., the string representing
;;;                 the possible leading substring
;;;        - the pronoun to replace it with
;;;        - the sentence
;;;
;;; if concept-string-candidate is an initial substring of sent,
;;;         then substitute pronoun in sentence
;;; return: nil                 - if concept-string-candidate was
;;;                               not an initial subseq OR comma test failed
;;;         the revised senence - if concept-string-candidate was
;;;                               an initial subseq
;;;
(defun pronominalize-sent-aux-2 (concept-string-candidate pronoun sent)
  (let ((concept-string-length (length concept-string-candidate))
	(sent-length (length sent)))
    (if (sent-begins-with-string concept-string-candidate
				 concept-string-length
				 sent
				 sent-length)
	(let ((sent-without-leading-concept-string
	       (subseq sent
		       concept-string-length
		       sent-length)))     
	  (if (and
	       (not (begins-with-comma sent-without-leading-concept-string))
	       (begins-with-blank sent-without-leading-concept-string))
	      (concatenate 'string
			   pronoun
			   sent-without-leading-concept-string))))))


;;; returns t iff sent begins with string
;;; otherwise nil    
;;;
(defun sent-begins-with-string (given-string given-string-length
					     sent sent-length)
  (if (> sent-length given-string-length)
      (let ((initial-sent-subseq 
	     (subseq sent 0 given-string-length)))
	(string-equal given-string
		      initial-sent-subseq))))


;;; given: - a list of concept strings
;;;        - a pronoun
;;;        - a sentence
;;;
;;; returns: if prev-sent starts with strng-1, strng-2, or strng-3,
;;;          then returns that string
;;;          otherwise returns nil
;;;
(defun previous-sent-begins-with-concept (concept-strings pronoun prev-sent)
  (let* ((strng-1         (first concept-strings))
	 (strng-2         (second concept-strings))
	 (strng-3         (third concept-strings))
	 (strng-1-length  (length strng-1))
	 (strng-2-length  (length strng-2))	 
	 (strng-3-length  (length strng-3))
	 (pronoun-length  (length pronoun))
	 (sent-length     (length prev-sent)))
    (cond ((sent-begins-with-string pronoun pronoun-length
				    prev-sent sent-length)
	   pronoun)
	  ((sent-begins-with-string strng-1 strng-1-length
				 prev-sent sent-length)
	   strng-1)
	  ((sent-begins-with-string strng-2 strng-2-length
				    prev-sent sent-length)
	   strng-2)
	  ((sent-begins-with-string strng-3 strng-3-length
				    prev-sent sent-length)
	   strng-3)
	  (t nil))))


(defun begins-with-comma (strng)
  (and (stringp strng)
       (> (length strng) 0)
       (equal (elt strng 0)
	      #\,)))


(defun begins-with-blank (strng)
  (and (stringp strng)
       (> (length strng) 0)
       (is-blank-p (elt strng 0))))


;;;; --------------------------------------------------------------------
;;;; formats a list of sentence strings
;;;; --------------------------------------------------------------------

;;; assumes longest word is shorter than value of line-length parameter

(defparameter *line-length* 70)


;;; given: a list of strings, where each string is sentence ending
;;;        in a period
;;;
;;; returns: a formatted string with line breaks
(defun make-paragraph-text (parag-sentence-list)
  (let ((parag-string (make-parag-with-blanks parag-sentence-list)))
    (make-paragraph-text-aux parag-string)
    (format t "~%"))
  (values))


;;; given: a string of sentences with blanks at end of sentences
;;;
;;; returns: a formatted string with line breaks
(defun make-paragraph-text-aux (parag-string)
  (let ((strng-length (length parag-string)))
    (if (not (empty-string-p parag-string strng-length))
	(let* ((end-of-first-line (get-end-of-first-line parag-string))
	       (first-line (subseq parag-string 0 end-of-first-line)))
	  (format t first-line)
	  (format t "~%")
	  (make-paragraph-text-aux
	   (strip-leading-blanks 
	    (subseq parag-string end-of-first-line strng-length)))))))
					 


;;; returns t iff strng is empty
;;;    
;;; a string is empty if it is of 0 length or if each element
;;; is a blank    
(defun empty-string-p (strng strng-length)
  (or (zerop strng-length)
      (and (is-blank-p (elt strng 0))
	   (empty-string-p (subseq strng 1 strng-length)
			   (1- strng-length)))))


;;; given: a string of sentences with blanks at end of sentences
;;;
;;; returns: the position at which the first line should end
(defun get-end-of-first-line (parag-string)
  (if (<= (length parag-string) *line-length*)
      (length parag-string)
      (do ((counter *line-length* (- counter 1)))
	  ((or (is-blank-p (elt parag-string counter))
	       (zerop counter))
	   counter))))


;;; given: a list of sentences
;;; returns: a string of all the sentences with blanks after
;;;          each sentence	 
(defun make-parag-with-blanks (parag-sentence-list)
  (apply #'concatenate
	 'string
	 (mapcar #'add-blanks-to-sent
		 parag-sentence-list)))


;;; addes two spaces to end of sentence
(defun add-blanks-to-sent (sent)
  (concatenate 'string sent "  "))


;;; returns string with leading blanks removed
;;;
(defun strip-leading-blanks (strng)
  (let ((strng-length (length strng)))
    (if (zerop strng-length)
	strng
	(do ((counter 0 (1+ counter)))
	    ((or (not (is-blank-p (elt strng counter)))
		 (equal counter (1- strng-length)))
	     (if (is-blank-p (elt strng counter))
		 ""
		 (subseq strng counter strng-length)))))))


(defun is-blank-p (ch)
  (equal ch '#\Space))


