;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       
				       
;;;;			The Report Generation Functions
;;;;			-------------------------------


;;;; Functions for testing if reports should be generated and
;;;; for generating reports


(in-package 'km)


;-----------------------------------------------------------------------
;		       The Top-Level Reporting Functions
;-----------------------------------------------------------------------

;;; makes a report about problematic content nodes and the
;;; length of the final linearized plan

(defun make-report (question-type)
  (print-report-header)
  (report-inputs question-type)
  (report-problems)
  (report-length)
  (report-top-most-plan-node)
  (report-paragraphized-leaves)
  (format t "~%")
  (format t "~%")
  (values))


;;; sets the make-report parameter, which is recorded in Knight-Global-State
;;;
;;; defaults to false
(defun set-make-report? (&optional (rept-value 'false))
  (case rept-value
    ((false true) (put-local '(knight-global-state make-report?)
			     rept-value))
    (t                 (format t "Illegal report setting.~%")
		       (values))))


;;; retrieves make-report setting
;;;
;;; assumes it is non-null

(defun make-report? ()
  (let ((result (get-only-val '(knight-global-state make-report?))))
    (if (equal result 'true)
	t)))


;-----------------------------------------------------------------------
;				 Print Header
;-----------------------------------------------------------------------

(defun print-report-header ()
  (format t "~%~%-------------------------------------------------------------~
             ------------~%")
  (format t "                         Explanation Generation Results~%")
  (format t "----------------------------------------------------------------~
             ---------~%~%"))


;-----------------------------------------------------------------------
;			       Inputs Reporting
;-----------------------------------------------------------------------

(defun report-inputs (question-type)
  (format t "Question type: ~a~%~%"
	  question-type)
  (format t "Concept of Interest: ~a~%~%"
	  (get-only-val (list (current-exposition-node) 'primary-concept)))
  (format t "Amount of Detail: ~a~%~%"
	  (get-verbosity))
  (format t "Discourse History Setting: ~a~%~%"
	  (get-only-val '(knight-global-state user-modeling?))))


;-----------------------------------------------------------------------
;		   Top-Most Explanation Plan Node Reporting
;-----------------------------------------------------------------------

;;; reports the topmost explanation plan node

(defun report-top-most-plan-node ()
  (when (current-exposition-node)
    (format t "Top-most explanation plan node: ~a~%~%"
	    (current-exposition-node)))
  (values))


;-----------------------------------------------------------------------
;				Error Reporting
;-----------------------------------------------------------------------

;;; reports content-nodes of current explanation plan with errors
;;;
;;; a content node has an error if 
;;;         (1) an error has occurred at the content node
;;;         (2) an error has occurred in the viewpoint associated
;;;             with the content node
;;;
;;; note: this is *not* an efficient implementation;
;;;       it traverses all content nodes and reports only
;;;       those that are in the current explanation plan
;;;       rather than traversing the explanation plan itself;
;;;       the implemented algorithm was just a little easier
;;;       to code

(defun report-problems ()
  (format t "Problematic content nodes:~%~%")
  (dolist (problem-content-node (find-erroneous-content-nodes))
    (let ((problem-type (find-problem-type problem-content-node)))
      (format t "      ~a : ~a~%" problem-content-node problem-type)))
  (format t "~%")
  (values))


(defun find-erroneous-content-nodes ()
  (let* ((all-content-nodes
	  (get-local '(content-node specializations)))
	 (current-content-nodes
	  (remove-if-not #'content-node-of-current-explanation-plan-p
			 all-content-nodes)))
    (remove-if-not #'problem-content-node-p
		   current-content-nodes)))


(defun content-node-of-current-explanation-plan-p (content-node)
  (let* ((current-exposition-node
	  (get-only-val '(knight-global-state
			  current-exposition-node)))
	 (search-result
	  (kb-search (list content-node)
		     (list 'child-of-node)
		    :terminate-with-success-criteria
		    #'(lambda (curr-plan-node)
			(equal curr-plan-node
			       current-exposition-node))
		    :collect-path? nil
		    :loop-elimination? t)))
    (not (equal (first search-result)
		'fail))))

			       
(defun problem-content-node-p (content-node)
  (or (content-node-has-error? content-node)
      (viewpoint-contains-error? (get-only-val (list content-node
						     'kb-subgraph)))))


(defun find-problem-type (problem-content-node)
  (let ((error-on-content-node
	 (get-only-val (list problem-content-node 'error-type))))
    (if error-on-content-node
	error-on-content-node
	(let ((viewpoint (get-only-val (list problem-content-node
					     'kb-subgraph))))
	  (if viewpoint
	      (let ((viewpoint-error (get-only-val (list viewpoint
							 'error-type))))
		(if viewpoint-error
		    viewpoint-error
		    'viewpoint-error))
	      'no-viewpoint)))))


;-----------------------------------------------------------------------
;			       Length Reporting
;-----------------------------------------------------------------------

;;; reports the number of filtered, paragraphed viewpoints in the
;;; linearized plan

(defun report-length ()
  (let* ((leaves (get-local (list (current-exposition-node)
				 'linearized-leaves)))
	 (size (length (flatten leaves))))
    (if (equal size 0)
	(format t "There are no viewpoints in the plan.~%~%")
	(format t "Number of viewpoints: ~a~%~%"
		size)))
  (values))


;-----------------------------------------------------------------------
;			Paragraphized Leaves Reporting
;-----------------------------------------------------------------------

(defun report-paragraphized-leaves ()
  (let ((leaves (get-local (list (current-exposition-node)
			   'linearized-leaves))))
    (when leaves
      (format t "Linearized viewpoints (grouped by paragraph):~%")
      (pprint leaves t))))


;-----------------------------------------------------------------------
;			      Verbosity Functions
;-----------------------------------------------------------------------

;;; sets the verbosity level, which is recorded in Knight-Global-State
;;;
;;; defaults to medium
(defun set-verbosity (&optional (verbosity-level 'medium))
  (case verbosity-level
    ((low medium high) (put-local '(knight-global-state verbosity-setting)
				  verbosity-level))
    (t                 (format t "Illegal verbosity setting.~%")
		       (values))))


;;; retrieves verbosity level
;;;
;;; assumes it is non-null

(defun get-verbosity ()
  (get-only-val '(knight-global-state verbosity-setting)))
