;;; -*- Mode:Lisp; Package:User; Base:10 -*-
    				       

;;;;			  Sweeping Functions
;;;;			  ------------------


;;;; This file contains functions that are used to sweep the KB.


(in-package 'km)


;;;-------------------------------------------------------------------
;;; finds units with particular numbers of hyphens in their names
;;;-------------------------------------------------------------------

(defun find-units-with-more-than-2-hyphens ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/more-than-two-hyphens"
			     :direction :output
			     :if-does-not-exist :create
			     :if-exists :append)
      (dolist (unit (append (progeny* 'object) (progeny* 'process)))
	(when (not (listp unit))
	  (let* ((string-name (string unit))
		 (hyphen-count (hyphens string-name 0 0)))
	    (if (> hyphen-count 2)
		(format outstream "~a~%" unit)))))))
	  

(defun find-units-with-2-hyphens ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/two-hyphens"
			     :direction :output
			     :if-does-not-exist :create
			     :if-exists :append)
      (dolist (unit (append (progeny* 'object) (progeny* 'process)))
	(when (not (listp unit))
	  (let* ((string-name (string unit))
		 (hyphen-count (hyphens string-name 0 0)))
	    (if (= hyphen-count 2)
		(format outstream "~a~%" unit)))))))


;;;-------------------------------------------------------------------
;;; fix domains of most viewpoint units
;;;-------------------------------------------------------------------

(defun fix-viewslots ()
  (let ((slots (get-local '(viewpoint-slots specializations))))
    (dolist (slot slots)
      (put-local (list slot 'domain)
		 'viewpoint))))


;;;-------------------------------------------------------------------
;;; find units with periods
;;;-------------------------------------------------------------------

(defun find-units-with-periods ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/periods"
			     :direction :output
			     :if-does-not-exist :create
			     :if-exists :append)
      (dolist (unit (append (progeny* 'object) (progeny* 'process)))
	(when (not (listp unit))
	  (if (unit-has-period-in-name? unit)
		(format outstream "~a~%" unit))))))


(defun unit-has-period-in-name? (unit)
  (let ((string-name (string unit)))
    (> (period string-name 0 0) 0)))
  

(defun period (unit pos count)
  (if (/= pos (array-total-size unit))
      (if (eq #\. (char unit pos))
	  (period unit (1+ pos) (1+ count))
	  (period unit (1+ pos) count))
      count))


;;;-------------------------------------------------------------------
;;; removes inappropriate lexical slot values
;;;-------------------------------------------------------------------

(defun remove-lexical-slot ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-all-values (list unit 'lexical-slot)))))

(defun remove-lex-item ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-all-values (list unit 'lex-item)))))


(defun remove-plural-lex-item ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-all-values (list unit 'plural-lex-item)))))


(defun remove-name-singular ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-all-values (list unit 'name-singular)))))


(defun remove-name-plural ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-all-values (list unit 'name-plural)))))


;;;-------------------------------------------------------------------
;;; generates reports about missing Process-Description Template info
;;;
;;; also generates reports about missing modulatory and temporal info
;;;-------------------------------------------------------------------

;;; traverses the progeny of a process that has a PDT associated with
;;; it to determine if all the necessary information is present about
;;; the processes to ensure that the PDT can be applied

;;; Note: this version only checks top-level units


(defvar *pdt-required-report-file*
  "/v/sage/v0/brewery/Lex-Info/pdt-required-report")


(defparameter *pdt-list-required*
;;; format: a list of pairs of the form
;;;            (process var-list)
;;;         where process is the name of the process
;;;                 with which the PDT is associated, and, consequently,
;;;                 the place where a search will begin, and
;;;               var-list is a list of variables (slot-names) which
;;;                 must have at least one value on every descendant
;;;                 of process (or there must be a spec-slot
;;;                 of each variable in var-list on every descendant of
;;;                 process)
  '(;Process                      Variables (Slots)
    ;-------                      -----------------
    (acquisition                  (acquired acquirer))
    (assimilation                 (assimilated))
    (attraction                   (attractant attracted))
    (biosynthesis                 (raw-materials products))
    (chemical-reaction            (reactants products))
    (destruction                  (destroyed))
    (development                  (developing-entity))
    (disintegration               (disintegrated))
    (energy-transduction          (input-energy-form output-energy-form))
    (formation                    (formed))
    (generation                   (generated))
    (growth                       (growing-entities))
    (maturation                   (matures))
    (removal                      (transported source))
    (reproduction                 (ancestors))
    (reproductive-fertilization   (fertilizer fertilizee products))
    (size-decrease-transformation (transformed-entity))
    (size-increase-transformation (transformed-entity))
    (storage                      (stored))
    (transportation               (transported))
    ))


(defun report-missing-required-pdt-info ()
  (with-open-file (outstream
		   *pdt-required-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Information for ~
                       Process-Description Templates~%")
      (format outstream "            ------------------------~
                       -----------------------------~%~%")
    (dolist (pdt-pair *pdt-list-required*)
      (let ((pdt-process (first pdt-pair))
	    (var-list (second pdt-pair)))
	(format outstream "~%~%~%Checking processes for ~a PDT~%" pdt-process)
	(format outstream "------------------------------------------------~%")
	(dolist (curr-process (progeny* pdt-process))
	  (when (not (listp curr-process))
	    (check-process-with-pdt curr-process var-list outstream)))))))


(defvar *pdt-optional-report-file*
  "/v/sage/v0/brewery/Lex-Info/pdt-optional-report")


(defparameter *pdt-list-optional*
;;; format: a list of pairs of the form
;;;            (process var-list)
;;;         where process is the name of the process
;;;                 with which the PDT is associated, and, consequently,
;;;                 the place where a search will begin, and
;;;               var-list is a list of variables (slot-names) which
;;;                 might have a value on a descendant of process
;;;                 (or there must be a spec-slot of each variable
;;;                 in var-list on every descendant of process)
  '(;Process             Variables (Slots)
    ;-------             -----------------
    (acquisition         (location))
    (biosynthesis        (location))
    (chemical-reaction   (location))
    (degeneration        (location))
    (destruction         (location))
    (development         (location))
    (energy-transduction (energy-acceptor location))
    (reproductive-fertilization (location))
    (generation          (location))
    (growth              (location))
    (maturation          (location))
    (removal             (destination))
    (reproduction        (offspring location))
    (storage             (location))
    (transportation      (source conduit destination location))
    ))


(defun report-missing-optional-pdt-info ()
  (with-open-file (outstream
		   *Pdt-optional-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Optional Information for ~
                       Process-Description Templates~%")
      (format outstream "            ---------------------------------~
                       -----------------------------~%~%")
    (dolist (pdt-pair *pdt-list-optional*)
      (let ((pdt-process (first pdt-pair))
	    (var-list (second pdt-pair)))
	(format outstream "~%~%~%Checking processes for ~a PDT~%" pdt-process)
	(format outstream "------------------------------------------------~%")
	(dolist (curr-process (progeny* pdt-process))
	  (when (not (listp curr-process))
	    (check-process-with-pdt curr-process var-list outstream)))))))


(defvar *modulatory-report-file*
  "/v/sage/v0/brewery/Lex-Info/modulatory-report")


(defparameter *modulatory-list*
;;; format: a list of pairs of the form
;;;            (process var-list)
;;;         where process is the name of the process
;;;                 with which the PDT is associated, and, consequently,
;;;                 the place where a search will begin, and
;;;               var-list is a list of variables (slot-names) which
;;;                 must have at least one value on every descendant
;;;                 of process (or there must be a spec-slot
;;;                 of each variable in var-list on every descendant of
;;;                 process)
  '(;Process             Variables (Slots)
    ;-------             -----------------
    (process             (inhibits enables causes initiates))
    ))


(defun report-missing-modulatory-info ()
  (with-open-file (outstream
		   *modulatory-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Modulatory Information~%")
    (format outstream "            ------------------------------~%~%")
    (dolist (pdt-pair *modulatory-list*)
      (let ((pdt-process (first pdt-pair))
	    (var-list (second pdt-pair)))
	(dolist (curr-process (progeny* pdt-process))
	  (when (not (listp curr-process))
	    (check-process-with-pdt curr-process var-list outstream)))))))


(defvar *transportation-pdt-required-report-file*
  "/v/sage/v0/brewery/Lex-Info/transportation-pdt-required-report")


(defun report-missing-required-pdt-info-for-transportation ()
  (with-open-file (outstream
		   *transportation-pdt-required-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Information for ~
                       Transportation Process-Description Templates~%")
      (format outstream "            ------------------------~
                       --------------------------------------------~%~%")
      (dolist (curr-process (progeny* 'transportation))
	(when (not (listp curr-process))
	  (when (and (not (slot-or-spec-slot-has-value-on 'source
							  curr-process))
		     (not (slot-or-spec-slot-has-value-on 'destination
							  curr-process)))
	    (format outstream "~a has no value for source or destination.~%~%"
		    curr-process))))))


(defvar *acquistion-pdt-required-report-file*
  "/v/sage/v0/brewery/Lex-Info/acquistion-pdt-required-report")


(defun report-missing-required-pdt-info-for-acquistion ()
  (with-open-file (outstream
		   *acquistion-pdt-required-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Information for ~
                       Acquisition Process-Description Templates~%")
      (format outstream "            ------------------------~
                       -----------------------------------------~%~%")
      (dolist (curr-process (progeny* 'acquisition))
	(when (not (listp curr-process))
	  (when (not (slot-or-spec-slot-has-value-on 'acquirer
						     curr-process))
	    (format outstream "~a has no value for acquirer.~%~%"
		    curr-process))))))


(defvar *development-pdt-required-report-file*
  "/v/sage/v0/brewery/Lex-Info/development-pdt-required-report")


(defun report-missing-required-pdt-info-for-development ()
  (with-open-file (outstream
		   *development-pdt-required-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "            Missing Information for ~
                       Development Process-Description Templates~%")
      (format outstream "            ------------------------~
                       -----------------------------------------~%~%")
      (dolist (curr-process (progeny* 'development))
	(when (not (listp curr-process))
	  (let ((developee (get-only-val (list curr-process
					       'developing-entity))))
	    (if (null developee)
		(format outstream "~a has no value for developing-entity.~%~%"
		    curr-process)
		(let ((before-state (get-only-val (list curr-process
							'developing-entity
							developee
							'before-state)))
		      (after-state (get-only-val (list curr-process
							'developing-entity
							developee
							'after-state))))
		  (when (null before-state)
		    (format outstream
			    "~a has no value for before-state annotation.~%~%"
			    curr-process))
		  (when (null after-state)
		    (format outstream
			    "~a has no value for after-state annotation.~%~%"
			    curr-process)))))))))


(defvar *temporal-report-file*
  "/v/sage/v0/brewery/Lex-Info/temporal-report")


(defparameter *temporal-list*
;;; format: a list of pairs of the form
;;;            (process var-list)
;;;         where process is the name of the process
;;;                 with which the PDT is associated, and, consequently,
;;;                 the place where a search will begin, and
;;;               var-list is a list of variables (slot-names) which
;;;                 must have at least one value on every descendant
;;;                 of process (or there must be a spec-slot
;;;                 of each variable in var-list on every descendant of
;;;                 process)
  '(;Process             Variables (Slots)
    ;-------             -----------------
    (process             (occurs-before occurs-after))
    ))


(defun report-missing-temporal-info ()
  (with-open-file (outstream
		   *temporal-report-file*
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "               Missing Temporal Information~%")
    (format outstream "               ----------------------------~%~%")
    (dolist (pdt-pair *temporal-list*)
      (let ((pdt-process (first pdt-pair))
	    (var-list (second pdt-pair)))
	(dolist (curr-process (progeny* pdt-process))
	  (when (not (listp curr-process))
	    (check-process-with-pdt curr-process var-list outstream)))))))


(defun check-process-with-pdt (curr-process var-list outstream)
  (dolist (var var-list)
    (when (not (slot-or-spec-slot-has-value-on var curr-process))
      (format outstream "~a has no value for ~a.~%~%" curr-process var))))


;;; returns t iff there is a value on unit.slot or there
;;; there is a value on unit.spec-slot, where spec-slot is a
;;; spec-slot of slot
;;;
;;; note: this version requires (probably unnessarily) that unit
;;;       have an explicit value on slot (or one of its specs);
;;;       we may change this to permit inherited values in the future
(defun slot-or-spec-slot-has-value-on (slot unit)
  (let ((slots-on-unit (get-explicit-slots unit))
	(all-spec-slots (progeny* slot)))
    (intersection slots-on-unit all-spec-slots)))


;;;-------------------------------------------------------------------
;;; find processes to which no PDT can be applied
;;;-------------------------------------------------------------------

(defun sweep-for-pdts ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/process-without-pdts"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "                  Processes without PDTs~%")
      (format outstream "                  ----------------------~%~%~%~%")
      (let ((processes (progeny* 'process)))
	(dolist (proc processes)
	  (when (equal (find-pdt proc) 'none)
	    (format outstream "~a~%~%" proc))))))


(defun find-pdt (concept)
  (if (null concept)
      'none
      (let* ((initial-unit-list (list concept))
	     (genl-slots (list 'generalizations 'stage-of 'i-genls))
	     (search-result 
	      (kb-search initial-unit-list
			 genl-slots
			 :terminate-with-success-criteria #'has-pdt?
			 :control-strategy 'breadth-first
			 :loop-elimination? t)))
	(if (equal (first search-result) 'fail)
	    'none
	    (second search-result)))))


(defun has-pdt? (concept)
  (get-local (list concept
		   'lexical-info
		   'li-primary
		   'lex-process-description-template)))


;;;-------------------------------------------------------------------
;;; prints out non-implicit unit Processes and Objects and Relations
;;;-------------------------------------------------------------------


;;; prints out all non-implicit units that are processes in the KB
(defun print-out-processes ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/process-list"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (dolist (curr-process (progeny* 'process))
	(when (not (listp curr-process))
	  (format outstream "~a~%" curr-process)))))


;;; prints out all non-implicit units that are object in the KB
(defun print-out-objects ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/object-list"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (dolist (curr-object (progeny* 'object))
	(when (not (listp curr-object))
	  (format outstream "~a~%" curr-object)))))


;;;-------------------------------------------------------------------
;;; prints out results of different kinds of ``relations'' sweeps
;;;-------------------------------------------------------------------


;;; prints out all ``decent'' relations in the KB
;;; omits all relations that are spec-relations of actor and actor-in
;;;
;;; for a given relation prints out relation
;;;
;;; if the relation has an inverse, prints out the inverse on the
;;; following line in parens
(defun print-out-relations ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/relations-inverse-list"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (let ((relations-seen-so-far nil))
	(dolist (curr-relation (progeny* 'new-slot-top))
	  (when (and (not (listp curr-relation))
		     (not (member 'actor-in (ancestry* curr-relation )))
		     (not (member 'actors (ancestry* curr-relation)))
		     (not (member curr-relation relations-seen-so-far)))
	    (let ((curr-relation-inverse (get-local (list curr-relation
							  'inverse))))
	      (cond (curr-relation-inverse
		     (format outstream "~a ~a~%"
			     curr-relation
			     curr-relation-inverse)
		     (push curr-relation relations-seen-so-far)		
		     (push (first curr-relation-inverse)
			   relations-seen-so-far))
		    (t (format outstream "~a (No inverse dude!)~%"
			       curr-relation)
		       (push curr-relation relations-seen-so-far)))))))))


;;; prints out all relations with no value for the english slot
(defun print-out-relations-with-no-english ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/relations-with-no-english"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (let ((relations-seen-so-far nil))
	(dolist (curr-relation (progeny* 'new-slot-top))
	  (when (and (not (listp curr-relation))
		     (not (member 'actor-in (ancestry* curr-relation )))
		     (not (member 'actors (ancestry* curr-relation)))
		     (not (get-local (list curr-relation 'english))))
	    (format outstream "~a~%" curr-relation))))))


;;; prints out relations and the English Art has given them
;;; format: relation (inverse) English
;;;
;;; for a given relation prints out relation
;;;
;;; if the relation has an inverse, prints out the inverse on the
;;; following line in parens
(defun print-out-relations-and-english ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/relations-with-english"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :append)
      (format outstream "                             Relations~%")
      (format outstream "                             ---------~%~%~%~%")
      (format outstream "Format: <Slot> (<Inverse>)~%<english for slot>~%~%~%")
      (dolist (curr-relation (progeny* 'new-slot-top))
	(when (and (not (listp curr-relation))
		   (not (member 'actor-in (ancestry* curr-relation )))
		   (not (member 'actors (ancestry* curr-relation))))
	  (let ((curr-relation-inverse (get-only-val (list curr-relation
							   'inverse)))
		(curr-relation-english (get-only-val (list curr-relation
							   'english))))
	    (cond (curr-relation-inverse
		   (format outstream "~a (~a)~% ~a~%~%"
			   curr-relation
			   curr-relation-inverse
			   curr-relation-english))
		  (t
 		   (format outstream "~a (no inverse)~% ~a~%~%"
			   curr-relation
			   curr-relation-english))))))))


;;;-------------------------------------------------------------------
;;; prints out results of  ``elementary'' sweeps
;;;-------------------------------------------------------------------

(defun elementary-sweep ()
  (with-open-file (outstream
		   "/v/sage/v0/brewery/Lex-Info/elementary-results"
		   :direction :output
		   :if-does-not-exist :create
		   :if-exists :supersede)
      (format outstream "                       Elementary Info~%")
      (format outstream "                       ---------------~%~%~%~%")
      (dolist (concept (progeny* 'process))
	(let ((elementary-ancestor (find-elementary-ancestor concept)))
	  (when (equal elementary-ancestor 'process)
	    (format outstream "~a~%~%" concept))))))

(defun find-elementary-ancestor (concept)
  (let* ((initial-unit-list (list concept))
	 (genl-slots (list 'generalizations 'stage-of 'i-genls))
	 (search-result 
	  (kb-search initial-unit-list
		     genl-slots
		     :terminate-with-success-criteria #'is-elementary-p
		     :control-strategy 'breadth-first
		     :loop-elimination? t)))
    (if (equal (first search-result) 'fail)

	;; no elementary concept found
	'none

	;; terminated with success, so return elementary concept
	(second search-result))))

;;;-------------------------------------------------------------------
;;; modify lex items of processes: definite = no --> countable = no
;;;-------------------------------------------------------------------

;;; fixes bug in all top-level processes
;;;
;;; known exceptions:  - gradient-driven-process
;;;                    - concentration-driven-process
;;;
;;; place to check it: - plant-reproductive-fertilization


(defun sweep-process-for-lex-definite-no ()
  (let ((processes (progeny* 'process)))
    (dolist (proc processes)
      (when (not (listp proc))
	(fix-process-lex proc)))))


(defun fix-process-lex (process)
  (let* ((fd-bit (get-fd-bit process))
	 (new-fd-bit (make-new-process-fd-bit fd-bit)))
    (put-fd-bit process new-fd-bit)))


(defun put-fd-bit (concept fd-bit)
    (put-local (list concept 'lexical-info 'li-primary 'lex-fd)
	       (list fd-bit)))


(defun get-fd-bit (concept)
  (get-local (list concept 'lexical-info 'li-primary 'lex-fd)))


(defun make-new-process-fd-bit (fd-bit)
  (mapcar #'(lambda (x)
	      (if (equal x '(definite
			     no))
		  '(countable no)   x))
	  (car fd-bit)))

;;;-------------------------------------------------------------------
;;; modify various aspects of lex items
;;;-------------------------------------------------------------------

(defun sweep-lex-items ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-nil-fd unit))))


(defun lex-check-for-empty (unit)
  (let ((lex-fd (get-local (list unit 'lexical-info
				 'li-primary 'lex-fd)))
	(lex-noun (get-local (list unit 'lexical-info
				   'li-primary 'lex-noun))))
    (when (and (not  lex-fd) (not lex-noun))
      (format t "~%Problem lex item on ~a: ~a~%"
	      unit
	      (if lex-fd
		  "no lex-noun"
		  "no lex-fd")))))


;(fix-mass-or-count unit)
;(fix-verb-participle unit)
;(fix-verb-particle unit)
;(fix-lex-type-noun-phrase unit)
;(fix-lex-noun unit)


(defun remove-nil-fd (unit)
  (let ((lex-fd (get-local (list unit 'lexical-info 'li-primary 'lex-fd))))
    (when (equal lex-fd '(nil))
      (remove-all-values (list unit 'lexical-info 'li-primary 'lex-fd)))))


(defun fix-mass-or-count (unit)
  (let ((lex-type (get-local (list unit 'lexical-info 'li-primary 'lex-type))))
    (when (not (or (equal lex-type '(noun))
		   (null lex-type)))
      (remove-all-values (list unit
			       'lexical-info
			       'li-primary
			       'mass-or-count?)))))


(defun fix-verb-participle(unit)
  (let ((participle (get-local (list unit
				   'lexical-info
				   'li-primary
				   'lex-verb-participle))))
    (when participle
      (remove-all-values (list unit
			       'lexical-info
			       'li-primary
			       'lex-verb-participle)))))


(defun fix-verb-particle (unit)
  (let ((particle (get-local (list unit
				   'lexical-info
				   'li-primary
				   'lex-verb-particle))))
    (when particle
      (remove-all-values (list unit
			       'lexical-info
			       'li-primary
			       'lex-verb-particle)))))


(defun fix-lex-type-noun-phrase (unit)
  (let ((lex-type (get-local (list unit 'lexical-info 'li-primary 'lex-type))))
    (when (equal lex-type '(noun-phrase))
      (remove-all-values (list unit
			       'lexical-info
			       'li-primary
			       'lex-noun)))))


(defun fix-lex-noun (unit)
  (let ((noun (get-local (list unit
			       'lexical-info
			       'li-primary
			       'lex-noun))))
    (when noun
      (put-local (list unit
		       'lexical-info
		       'li-primary
		       'lex-noun)
		 noun))))
     

(defun sweep-empty-lex-slots ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(fix-goofy-lex-slot unit 'lex-mass-or-count)
	(fix-goofy-lex-slot unit 'process-description-template)	
	(fix-goofy-lex-slot unit 'lex-verb-type)
	(fix-goofy-lex-slot unit 'lex-verb-mode)
	(fix-goofy-lex-slot unit 'lex-fd-params)
	(fix-goofy-lex-slot unit 'lex-verb-subtype)
	(fix-goofy-lex-slot unit 'lex-verb-voice))))


(defun fix-goofy-lex-slot (unit goofy-lex-slot)
  (when (get-local (list unit 'lexical-info 'li-primary goofy-lex-slot))
    (remove-all-values (list unit
			     'lexical-info
			     'li-primary
			     goofy-lex-slot))))


;;;-------------------------------------------------------------------
;;; Add elementary and unspeakable concepts (5-24-94)
;;;-------------------------------------------------------------------

(add-elementary-concepts
   '(ORGANISM-LEVEL-STRUCTURE
     PHOTOSYNTHETIC-TISSUE-LEVEL-STRUCTURE
     PLANT-TISSUE-LEVEL-STRUCTURE
     PLANT-DIVISION
     ORGANISM-DIVISION
     PLANT-CELL-LEVEL-STRUCTURE
     STOMA-CONTAINING-STRUCTURE
     STARCH-STORAGE
     PHLOEM-UNLOADING
     CAPTURE
     PRIMARY-GROWTH
     PLANT-SEXUAL-REPRODUCTION
     PERIPHERAL-ZONE
     ))


(add-unspeakable-concepts
   '(ORGANISM-LEVEL-STRUCTURE
     PHOTOSYNTHETIC-TISSUE-LEVEL-STRUCTURE
     PLANT-TISSUE-LEVEL-STRUCTURE
     PLANT-DIVISION
     ORGANISM-DIVISION
     PLANT-CELL-LEVEL-STRUCTURE
     STOMA-CONTAINING-STRUCTURE
     CAPTURE
     ))

;;;-------------------------------------------------------------------
;;; Remove all viewpoints that aren't current
;;;-------------------------------------------------------------------

(defun remove-non-current-viewpoints ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(when (is-a-viewpoint-but-not-current? unit)
	  (remove-frame unit)))))


(defun is-a-viewpoint-but-not-current? (unit)
  (and (get-local (list unit 'viewpoint-of))
       (not (equal (get-local (list unit 'instance-of))
		   '(viewpoint)))))


;;;-------------------------------------------------------------------
;;;				  Count Slots
;;;-------------------------------------------------------------------

(defun count-top-level-slots ()
  (length
   (remove-if #'listp
	      (progeny* 'slot))))

