;;; -*- Mode:Lisp; Package:User; Base:10 -*-
				       

;;;;			  The User Modeling Functions
;;;;			  ---------------------------


;-----------------------------------------------------------------------
;			 User Model Function Directory
;-----------------------------------------------------------------------
;;;; This file contains functions that model the user's knowledge
;;;; of the domain.  Here is a function directory:
;;;;
;;;;
;;;;     Initialization
;;;;     --------------
;;;;     (switch-on-um)
;;;;     (switch-off-um)
;;;;     (initialize-user-model &key name concept-list)
;;;;     (clear-um &optional user)
;;;;     (remove-all-known-by-annotations)
;;;;
;;;;     Access
;;;;     ------
;;;;     (user-modeling-on?)
;;;;     (current-user)
;;;;     (is-known-p concept-or-triple &optional user-name)
;;;;     (list-known-concepts &optional user-name)
;;;;
;;;;     Updating
;;;;     --------
;;;;     (add-um-concept concept &optional user-name)
;;;;     (add-um-triple concept &optional user-name)
;;;;     (remove-um-concept concept &optional user-name)
;;;;     (remove-um-triple concept &optional user-name)
;;;;
;;;;     Elementary Concept Functions
;;;;     ----------------------------
;;;;     (is-elementary-p concept)
;;;;     (add-elementary-concepts concept-list)
;;;;     (list-elementary-concepts)
;;;;
;;;;     Unspeakable Concept Functions
;;;;     ----------------------------
;;;;     (is-unspeakable-p concept)
;;;;     (add-unspeakable-concepts concept-list)
;;;;     (list-unspeakable-concepts)
;;;;
;;;;     Core Concept Functions
;;;;     ----------------------
;;;;     (is-core-p concept)
;;;;     (list-core-concepts)
;;;;
;;;;


(in-package 'km)


;-----------------------------------------------------------------------
;		    User Model Initialization and Clearing
;-----------------------------------------------------------------------


(defun switch-on-um ()
  (put-local '(knight-global-state user-modeling?)
	     'true))
	       

(defun switch-off-um ()
  (put-local '(knight-global-state user-modeling?)
	     'false))


(defun initialize-user-model (&key name concept-list)
  (let ((user-name (if (null name)
		       (gentemp "USER-")
		       name)))
    (put-local '(knight-global-state current-user)
	       user-name)
    (dolist (concept concept-list)
      (if (listp concept)

	  ;; add triple to user model
	  (add-um-triple concept user-name)

	  ;; add concept to user model
	  (add-um-concept concept user-name))))
  (values))  ;so nil is not returned


(defun clear-um (&optional user)
  (if (null user)

      ;; clear entire user model
      (let ((concepts-to-clear (get-known-concepts)))   
	(format t "Removing all concepts from all user models.~%")
	(format t "These concepts and the user's associated with them are:~%") 
	(dolist (address-value-pair concepts-to-clear)
	  (let ((address (first address-value-pair))
		(value-list (second address-value-pair)))
	    (dolist (single-value value-list)
	      (cond ((equal (length address) 2)
		     ;; address is a concept
		     (format t "   ~a~55T: ~a~%"
			     (first address)
			     single-value)
		     ;; remove all users from the concept
		     (let ((user-list (get-local address)))
		       (dolist (this-user user-list)
			 (remove-um-concept (first address)
					    this-user))))
		    (t
		     ;; address is a triple
		     (format t "   ~a~55T: ~a~%"
			     (all-but-last address)
			     single-value)
		     (let ((user-list (get-local address)))
		       (dolist (this-user user-list)
			 (remove-um-triple (all-but-last address)
					   this-user
					   t))))))))) ;don't remove inverse

      ;; clear user model for given user
      (let ((concepts-to-clear (get-known-concepts user)))
	(format t "Removing all concepts known by ~a.~%"
		user)
	(format t "These concepts are:~%")       
	(dolist (address concepts-to-clear)
	  (cond ((equal (length address) 2)
		 ;; address is a concept
		 (format t "   ~a~%" (first address))
		 (remove-um-concept (first address) user))
		(t
		 ;; address is a triple
		 (format t "   ~a~%" (all-but-last address))
		 ;; removes triple but not its inverse 
		 ;; (the inverse will be removed later)
		 (remove-um-triple (all-but-last address)
				   user
				   t)))))) ;don't remove inverse

  (values))  ;so nil isn't printed


(defun all-but-last (list)
  "Returns list that contains all but last element"
  (parent list)) ; parent is a km function that does this


;;; removes all um annotations on high-level concepts and their
;;; top level slots

(defun remove-all-known-by-annotations ()
    (dolist (unit *kb-objects*)
      (when (not (listp unit))
	(remove-known-by-triples unit)
	(when (get-local (list unit 'known-by))
	  (remove-all-values (list unit 'known-by))))))


(defun remove-known-by-triples (unit)
  (let ((slots-to-check (get-explicit-slots unit)))
    (dolist (slot slots-to-check)
      (dolist (value (get-local (list unit slot)))
	(let ((known-by-who (get-local (list unit slot value 'known-by))))
	  (when known-by-who
	    (remove-um-triple (list unit slot value))))))))


;;; finds concepts and triples marked as known in the KB

(defun get-known-concepts (&optional user)
  (if (null user)

      ;; finding all concepts known in all user models
      (units-having-function3
       'content-frame  ; the place to begin search
       'known-by       ; the slot to search for
       nil)            ; no value to search for

      ;; finding all concepts known by specified user
      (units-having-function3
       'content-frame  ; the place to begin search
       'known-by       ; the slot to search for
       user)))         ; the value to search for


;-----------------------------------------------------------------------
;			       User Model Access
;-----------------------------------------------------------------------
;;;;
;;;; To determine if a user knows a concept, the system
;;;;
;;;;       (1) checks if the concept is marked as elementary?; 
;;;;           if so, the concept is known
;;;;
;;;;       (2) checks if the concept is marked as known-by the user; 
;;;;           if so, the concept is known
;;;;
;;;;       (3) checks if at least half of the triples associated
;;;;           with the concept are marked as known (only important
;;;;           slots are considered in this computation, e.g.,
;;;;           bookeeping slots are not considered);
;;;;           if at least 50% of the triples are known, the concept
;;;;           is known, and it is marked as known
;;;;
;;;;           (values that are known, but are inherited from above,
;;;;            are not considered; also, for now anyway, only
;;;;            top-level triples are considered for this calculation,
;;;;            i.e., if a deeply nested triple was marked as known,
;;;;            this will have no effect on whether the top-level
;;;;            concept is known)
;;;;
;;;; To determine if a user knows a triple, the system checks if
;;;; the triple is marked as known with a known-by annotation
;;;; (known-by is spec of slot+value-slots)
;;;;
;;;; The functions assume user modeling is monotonic, i.e., once a user 
;;;; knows a concept, he doesn't forget it.


(defun user-modeling-on? ()
  (equal (get-only-val '(knight-global-state user-modeling?))
	 'true))


(defun current-user ()
  (get-only-val '(knight-global-state current-user)))


(defun is-known-p (concept-or-triple &optional user-name)
  (let ((user (if (null user-name)
		  (current-user)
		  user-name)))
    (cond ((listp concept-or-triple)

	   ;; concept-or-triple is a triple
	   (member user
		   (get-local (append concept-or-triple '(known-by)))))

	  (t
	   ;; concept-or-triple is a concept
	   (or (is-elementary-p concept-or-triple)
	       (member user
		       (get-local (list concept-or-triple 'known-by)))
	       (let* ((domain-slots (get-domain-slots concept-or-triple))
		      (known-slots (get-known-slots concept-or-triple
						    domain-slots))
		      (ratio (if (null domain-slots)
				 0
				 (/ (length known-slots)
				    (length domain-slots)))))
		 ;(format t "Domain slots are: ~a.~%" domain-slots)
		 ;(format t "Known slots are: ~a.~%" known-slots)
		 ;(format t "Ratio is ~a.~%" ratio)
		 (when (>= ratio 0.5)
		   ;; adds concept to user model because over half its
		   ;; triples are known
		   (add-um-concept concept-or-triple user)
		   t)))))))


;;; finds slots on concept that satisfy several criteria
;;;      (1) have a value
;;;      (2) are not bookkeeping slots
;;;      (3) are not discourse slots
;;;      (4) are not constraint-slots, e.g., range-of
;;;      (5) are not notation-slots, e.g., english
;;;
;;; uses slot on Non-domain-slot-list called non-domain-slots
;;;
;;; this list of slots is computed when initialize-knight is called

(defun get-domain-slots (concept)
  (let ((slots-with-values (all-explicit-slots concept))
	(uninteresting-slots (get-local '(non-domain-slot-list
					  non-domain-slots))))
    (set-difference slots-with-values uninteresting-slots)))

    
;;; finds slots in slot-list that are known to the current user
;;; (a slot is known if at least one value on it is marked as known)
;;;
;;; the reduce function is used to eliminate nil's
;;;
;;; Important note: mapcan is *not* used below and should
;;;                 not be used anywhere in the code because
;;;                 it nconc's some structures in the KB,
;;;                 sometimes creating circular structures
;;;                 that cause KM to go into infinite recursion

(defun get-known-slots (concept slot-list)
  (reduce #'append
	  (mapcar #'(lambda (slot) (if (exists-a-known-value-on concept slot)
				       (list slot)))
		  slot-list)))


(defun exists-a-known-value-on (concept slot)
  (do* ((rest-of-value-list (get-local (extend-address concept slot))
			    (cdr rest-of-value-list))
	(current-value (first rest-of-value-list)
		       (first rest-of-value-list))
	(current-value-known? (member (current-user)
				      (get-local
				       (extend-address-indefinitely
					concept
					slot
					current-value
					'known-by)))
			      (member (current-user)
				      (get-local
				       (extend-address-indefinitely
					concept
					slot
					current-value
					'known-by)))))
       ((or (null rest-of-value-list)
	    (not (null current-value-known?)))
	current-value-known?)))


;;; lists concepts known to user
;;; may optionally specify a user
;;; default is to list concepts known by current user

(defun list-known-concepts (&optional user-name)
  (let ((result-list
	 (units-having-function3
  		  'content-frame              ; the place to begin search
		  'known-by                   ; the slot to search for
		  (if (not (null user-name))  ; the value to search for
		      user-name
		      (current-user)))))
    (cond ((null result-list)
	   (format t "No concepts are known by ~a.~%~%"
		   (if (not (null user-name))
		       user-name
		       (current-user))))
	  (t 
	   (format t "The following concepts are known by ~a:~%"
		   (if (not (null user-name))
		       user-name
		       (current-user)))
	   (dolist (concept-or-triple result-list)
	     (if (equal (length concept-or-triple) 2)
		 ;; concept-or-triple is a concept
		 (format t "   ~a~%" (first concept-or-triple))
		 ;; concept-or-triple is a concept
		 (format t "   ~a~%" (all-but-last concept-or-triple))))))
    (format t "Note that elementary concepts are ~
               assumed to be known by the user.~%")
    (values)))   ;so nil isn't printed


;-----------------------------------------------------------------------
;			      User Model Updating
;-----------------------------------------------------------------------


;;; adds a concept to a user model

(defun add-um-concept (concept &optional user-name)
  (add-val (extend-address concept 'known-by)
	   (if (not (null user-name))
	       user-name
	       (current-user))))


;;; adds a triple to a user model
;;;
;;; if the triple is actually a triple and not a long address,
;;; and if the inverse of the triple exists, adds the inverse
;;; to the user model too

(defun add-um-triple (triple &optional user-name)
  (let ((user (if (not (null user-name))
		  user-name
		  (current-user))))

    (add-val-local (append triple '(known-by))
		   user)

    (when (equal (length triple) 3)
      (let* ((unit (first triple))
	     (slot (second triple))
	     (value (third triple))
	     (inverse-slot (get-only-val (list slot 'inverse))))
	(when (not (null inverse-slot))
	  (add-val-local (extend-address-indefinitely value inverse-slot
						      unit 'known-by)
			 user))))))


(defun remove-um-concept (concept &optional user-name)
  (remove-val (extend-address concept 'known-by)
	      (if (not (null user-name))
		  user-name
		  (current-user))))


(defun remove-um-triple (triple &optional user-name do-not-remove-inverse?)
  (remove-val (append triple '(known-by))
	      (if (not (null user-name))
		  user-name
		  (current-user)))

  ;; if do-not-remove-inverse? is nil, and triple is a true triple, i.e.,
  ;; it has 3 elements, then remove inverse annotation if it exists
  (unless do-not-remove-inverse?
    (when (equal (length triple) 3)
      (let* ((unit (first triple))
	     (slot (second triple))
	     (value (third triple))
	     (inverse-slot (get-only-val (list slot 'inverse))))
	(when inverse-slot
	  (remove-val (extend-address-indefinitely value inverse-slot
						   unit 'known-by)
		      (if (not (null user-name))
			  user-name
			  (current-user))))))))


;-----------------------------------------------------------------------
;			   Elementary Concept Access
;-----------------------------------------------------------------------


(defun is-elementary-p (concept)
  (equal (get-only-val (extend-address concept 'elementary?))
	 'true))
  

(defun add-elementary-concepts (concept-list)
  (dolist (concept concept-list)
    (put-local (extend-address concept 'elementary?)
	       '(true))))


(defun list-elementary-concepts ()
  (let ((result-list
	 (units-having-function3
	          'content-frame              ; the place to begin search
		  'elementary?                ; the slot to search for
		  'true)))                    ; the value to search for
    (format t "~%The following concepts are considered elementary~%~
                 and are assumed to be known by the user:~%")
    (dolist (concept result-list)
      (format t "   ~a~%" (first concept)))
    (values)))   ;so nil isn't printed


;-----------------------------------------------------------------------
;			  Unspeakable Concept Access
;-----------------------------------------------------------------------
;;; Unspeakable concepts are a subset of elementary concepts
;;; an unspeakable concept is a concept that should never be
;;; included in an explanation's text, e.g., SPATIALLY-EXTENDED-ENTITY
;;; however, most elementary concepts should not be the main topic
;;; of an explanation, e.g., REPRODUCING-STRUCTURE, but it is fine
;;; to mention them as part of an explanation (these should not be
;;; filtered during filtering of overly-general concepts


(defun is-unspeakable-p (concept)
  (equal (get-only-val (extend-address concept 'unspeakable?))
	 'true))
  

(defun add-unspeakable-concepts (concept-list)
  (dolist (concept concept-list)
    (put-local (extend-address concept 'unspeakable?)
	       '(true))))


(defun list-unspeakable-concepts ()
  (let ((result-list
	 (units-having-function3
	          'content-frame              ; the place to begin search
		  'unspeakable?               ; the slot to search for
		  'true)))                    ; the value to search for
    (format t "~%The following concepts are considered unspeakable~%~
                 and are assumed to be known by the user:~%")
    (dolist (concept result-list)
      (format t "   ~a~%" (first concept)))
    (values)))   ;so nil isn't printed


;-----------------------------------------------------------------------
;			      Core Concept Access
;-----------------------------------------------------------------------


(defun is-core-p (concept)
  (equal (get-only-val (extend-address concept 'core?))
	 'true))

  
(defun list-core-concepts ()
  (let ((result-list
	 (units-having-function3
	          'content-frame              ; the place to begin search
		  'core?                      ; the slot to search for
		  'true)))                    ; the value to search for
    (format t "~%The following concepts are considered core:~%")
    (dolist (concept result-list)
      (format t "   ~a~%" (first concept)))
    (values)))   ;so nil isn't printed


;-----------------------------------------------------------------------
;			     User Model Test Calls
;-----------------------------------------------------------------------


(defun test-um-initialization1 ()
  (initialize-user-model :name 'herman
			 :concept-list '(plant
					 embryo-sac
					 leaf
					 (root
					  absorber-in
					  root-water-absorption))))


(defun test-um-initialization2 ()
  (initialize-user-model :name 'ralph
			 :concept-list '(plant
					 stem
					 (root
					  absorber-in
					  root-water-absorption))))

