header {* Basic Modelling *}

theory Base
imports PreList
begin


subsection {* Milawa Objects *}

text {* We model Milawa objects with the following datatype.  Objects are either
  symbols, integers, or (recursively) pairs of other objects. *}

datatype object = Symbol nat 
                | Integer int
                | Cons object object


text{* We do not know a good way to model symbols, so we just use tagged natural
  numbers.  That is, we imagine @{term "Symbol 0"}, @{term "Symbol 1"}, etc., as
  corresponding somehow to the actual symbols of Milawa.  We model integers
  using Isabelle's built-in @{typ int} type, and conses as tagged pairs of @{typ
  object}s. *}



subsection {* Abbreviations for Constants *}

(* BOZO.  Tobias Nipkow says the development version of Isabelle has a new
   "abbreviations" facility that is much slicker than Syntax.  When it is
   released we should consider changing these to use it. *)

text {* In Milawa symbols are named, but here we are modelling them as tagged
  integers.  It would be unfortunate if we had to remember that @{term "Symbol
  0"} was supposed to stand for ``nil'', etc., so we set up certain syntactic
  abbreviations for commonly used symbols (and numbers). *}

syntax
  nil :: "object"
  t :: "object"
  zero :: "object"
  one :: "object"
  negone :: "object"


text {* Our assignment of numbers to these symbols is arbitrary and does not
  accurately reflect the symbol order in Milawa.  This means this model is not
  quite right, but we do not think this difference matters for anything we are
  about to do. *}

translations
  "nil" == "Symbol 0"
  "t" == "Symbol (Suc 0)"  -- {* Untranslates better than @{term "Symbol 1"} *}
  "zero" == "Integer 0"
  "one" == "Integer 1"
  "negone" == "Integer -1"


text {* We will sometimes find it useful to check if a deconstructed symbol is
  @{term nil} or @{term t}.  To facilitate this, we define numbers
  corresponding to our assignment. *}

consts nil_number :: "nat"
       t_number :: "nat"

defs nil_number [simp]: "nil_number \<equiv> 0"
     t_number [simp]: "t_number \<equiv> 1"

text {* It is important to ensure that the definitions of, e.g., @{term
  nil_number} and @{term nil} are kept in sync.  We write some theorems to 
  ensure that this is the case. *}

theorem "check nil_number agreement" []:
  "Symbol nil_number = nil"
  by(auto)

theorem "check t_number agreement" []:
  "Symbol t_number = t"
  by(auto)





subsection {* HOL Utilities *}

text {* We add a few special helper functions that do not exist in Milawa, but which
  are useful for interfacing with HOL. *}

consts 
  hol_rank :: "object \<Rightarrow> nat"
  hol_nfix :: "object \<Rightarrow> nat"
  hol_ifix :: "object \<Rightarrow> int"
  hol_bfix :: "object \<Rightarrow> bool"


text {* @{term hol_rank} is like Milawa's rank function, but returns a HOL
  @{typ nat} instead of an @{typ object}.  This allows us to use @{term
  hol_rank} as a measure in recursive definitions. *}

primrec
  "hol_rank (Symbol s) = 0"
  "hol_rank (Integer i) = 0"
  "hol_rank (Cons x y) = 1 + hol_rank x + hol_rank y"


text {* @{term hol_nfix} is like Milawa's nfix function, but returns a HOL
  @{typ nat} instead of an @{typ object}.  Like @{term hol_rank}, this allows
  @{term hol_nfix} to be used as a measure for recursive definitions.  Note
  that the built-in Isabelle function, @{term "nat i"}, properly returns 
  the natural @{term 0} if @{term i} is negative, or @{term i} as a natural
  otherwise. *}

primrec 
  "hol_nfix (Symbol s) = 0"
  "hol_nfix (Integer i) = nat i"
  "hol_nfix (Cons x y) = 0"


text {* @{term hol_ifix} can be used to interpret an @{typ object} as a HOL
  @{typ int}.  Integer-typed @{typ object}s are interpreted as themselves, and
  all other objects are interpreted as zero: *}

primrec
  "hol_ifix (Symbol s) = 0"
  "hol_ifix (Integer i) = i"
  "hol_ifix (Cons x y) = 0"


text {* Finally, @{term hol_bfix} can be used to interpret an @{typ object} as
  a HOL @{typ bool}.  Like Milawa's {\em if} function, we treat @{term nil} as
  @{term False} and any other object as @{term True}. *}

defs hol_bfix [simp]: "hol_bfix x \<equiv> (x \<noteq> nil)"




subsection {* Modelling the Base Functions *}

consts
  ite :: "object \<Rightarrow> object \<Rightarrow> object \<Rightarrow> object"
  equal :: "object \<Rightarrow> object \<Rightarrow> object"
  consp :: "object \<Rightarrow> object"
  integerp :: "object \<Rightarrow> object"
  symbolp :: "object \<Rightarrow> object"
  car :: "object \<Rightarrow> object"
  cdr :: "object \<Rightarrow> object"
  lessp :: "object \<Rightarrow> object \<Rightarrow> object"
  plus :: "object \<Rightarrow> object \<Rightarrow> object"
  minus :: "object \<Rightarrow> object"

text {* Our Isabelle versions of the Milawa base functions can mostly be given
  the same names.  However, we use @{term ite} rather than {\em if}, since {\em
  if} is already defined in Isabelle.  The definitions are straightforward: *}


defs ite_def [simp]:
  "ite x y z \<equiv> if x \<noteq> nil then y else z"

defs equal_def [simp]:
  "equal x y \<equiv> if x = y then t else nil"

primrec
  "consp (Symbol s) = nil"
  "consp (Integer i) = nil"
  "consp (Cons x y) = t"

primrec
  "integerp (Symbol s) = nil"
  "integerp (Integer i) = t"
  "integerp (Cons x y) = nil"

primrec
  "symbolp (Symbol s) = t"
  "symbolp (Integer i) = nil"
  "symbolp (Cons x y) = nil"

primrec 
  "car (Symbol s) = nil"
  "car (Integer i) = nil"
  "car (Cons x y) = x"

primrec
  "cdr (Symbol s) = nil"
  "cdr (Integer i) = nil"
  "cdr (Cons x y) = y"

defs lessp_def: 
  "lessp x y \<equiv> if hol_ifix x < hol_ifix y then t else nil"

defs plus_def: 
  "plus x y \<equiv> Integer (hol_ifix x + hol_ifix y)"

defs minus_def:
  "minus x \<equiv> Integer (- (hol_ifix x))"




subsection {* Macro Substitutions *}

text {* It is easy to recreate the simple macros such as {\em first} and {\em
  second} as syntactic translations. *}

syntax
  first :: "object \<Rightarrow> object"
  second :: "object \<Rightarrow> object"
  third :: "object \<Rightarrow> object"
  fourth :: "object \<Rightarrow> object"
  fifth :: "object \<Rightarrow> object"

translations
  "first x" == "car x"
  "second x" == "first  (cdr x)"
  "third x" == "second (cdr x)"
  "fourth x" == "third  (cdr x)"
  "fifth x" == "fourth (cdr x)"


text {* The {\em and} and {\em or} macros are also heavily used in Milawa
  definitions, but we cannot define a variable-arity function in Isabelle.  As
  a compromise, we can introduce infix versions of {\em and} and {\em or} which
  can easily be nested. *}

consts
  and_fn :: "object \<Rightarrow> object \<Rightarrow> object"   (infixr "and" 60)
  or_fn  :: "object \<Rightarrow> object \<Rightarrow> object"   (infixr "or" 60)

defs and_fn_def [simp]: 
  "A and B \<equiv> ite A B nil"

defs or_fn_def [simp]: 
  "A or B \<equiv> ite A A B"


text {* We can see that these macros work as we would expect, and the following
  lemmas illustrate how the translation is done. *}

lemma "and test" []:
  "A and B and C and D and E = 
   (ite A (ite B (ite C (ite D (ite E E nil) nil) nil) nil) nil)"
  by(auto)

lemma "or test" []:
  "A or B or C or D or E =
   (ite A A (ite B B (ite C C (ite D D (ite E E nil)))))"
  by(auto)





subsection {* Some Trivial Functions *}

consts  
  not :: "object \<Rightarrow> object"
  zp :: "object \<Rightarrow> object" 
  natp :: "object \<Rightarrow> object"
  nfix :: "object \<Rightarrow> object"
  ifix :: "object \<Rightarrow> object"
  booleanp :: "object \<Rightarrow> object"


text {* Most of these functions can simply be left enabled, and we will just
  let the simplifier break apart the cases. *}
       
defs not_def [simp]: 
  "not x \<equiv> ite x nil t"

defs zp_def [simp]: 
  "zp x \<equiv> ite (integerp x) (not (lessp zero x)) t"

defs natp_def [simp]:
  "natp x \<equiv> ite (integerp x) (not (lessp x zero)) nil"

defs nfix_def [simp]:
  "nfix x \<equiv> ite (natp x) x zero"

defs ifix_def [simp]:
  "ifix x \<equiv> ite (integerp x) x zero"


text {* On the other hand, we leave @{term booleanp} disabled, and prove
  several basic theorems about it. *}

defs booleanp_def:
  "booleanp x \<equiv> ite (equal x t) t (equal x nil)"

theorem "booleanp of Symbol" [simp]:
  "booleanp (Symbol x) = (if (x = nil_number) 
                             then t 
                             else if (x = t_number) 
                                     then t
                                     else nil)"
  by(simp add: booleanp_def)

theorem "booleanp of Integer" [simp]:
  "booleanp (Integer x) = nil"
  by(simp add: booleanp_def)

theorem "booleanp of Cons" [simp]:
  "booleanp (Cons x y) = nil"
  by(simp add: booleanp_def)

theorem "booleanp of equal" [simp]:
  "booleanp (equal x y) = t"
  by(auto)

theorem "booleanp of booleanp" [simp]:
  "booleanp (booleanp x) = t"
  by(case_tac x, auto)

theorem "booleanp of consp" [simp]:
  "booleanp (consp x) = t"
  by(case_tac x, auto)

theorem "booleanp of integerp" [simp]:
  "booleanp (integerp x) = t"
  by(case_tac x, auto)

theorem "booleanp of symbolp" [simp]:
  "booleanp (symbolp x) = t"
  by(case_tac x, auto)

theorem "booleanp of lessp" [simp]:
  "booleanp (lessp x y) = t"
  by(simp add: lessp_def)

theorem "booleanp of natp" [simp]:  (* BOZO why bother? *)
  "booleanp (natp x) = t"
  by(auto)

theorem "booleanp of zp" [simp]:    (* BOZO why bother? *)
  "booleanp (zp x) = t"
  by(auto)



text {* The following rules normalize inequalities with @{term t} or @{term
  nil} when the argument is known to be boolean.  These are very useful rules
  that help to bridge the gap between our @{typ object}s and HOL's @{typ
  bool}s. *}

theorem "booleanp normalizer (true)" [simp]:
  "\<lbrakk> booleanp x = t \<rbrakk> \<Longrightarrow> (x \<noteq> nil) = (x = t)"
  apply(case_tac x, auto)
  apply(split split_if_asm, auto)
done

theorem "booleanp normalizer (false)" [simp]:
  "\<lbrakk> booleanp x = t \<rbrakk> \<Longrightarrow> (x \<noteq> t) = (x = nil)"
  apply(case_tac x, auto)
  apply(split split_if_asm, auto)
done




subsection {* Rules for HOL Utilities *}

text {* We provide a few rewrite rules to help deal with our HOL-specific 
  utility functions.  *}

theorem "hol_ifix when not integerp" [simp]:
  "\<lbrakk> integerp x = nil \<rbrakk> \<Longrightarrow> hol_ifix x = 0"
  by(case_tac x, auto)

theorem "hol_ifix of integer when integerp" [simp]:
  "\<lbrakk> integerp x = t \<rbrakk> \<Longrightarrow> Integer (hol_ifix x) = x"
  by(case_tac x, auto)

theorem "hol_rank when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> hol_rank x = 0"
  by(case_tac x, auto)

theorem "hol_rank of car" [simp]:
  "(hol_rank (car x) < hol_rank x) = (consp x = t)"
  by(case_tac x, auto)

theorem "hol_rank of car (weak)" [simp]:
  "(hol_rank x < hol_rank (car x)) = False"
  by(case_tac x, auto)
  
theorem "hol_rank of cdr" [simp]:
  "(hol_rank (cdr x) < hol_rank x) = (consp x = t)"
  by(case_tac x, auto)

theorem "hol_rank of cdr (weak)" [simp]:
  "(hol_rank x < hol_rank (cdr x)) = False"
  by(case_tac x, auto)

theorem "hol_nfix of plus negone when nonzero" [simp]:
  "\<lbrakk> lessp zero n = t \<rbrakk> \<Longrightarrow> hol_nfix (plus n negone) < hol_nfix n"
  apply(case_tac n)
  apply(auto simp add: lessp_def plus_def split: split_if_asm)
done

theorem "car when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> (car x) = nil"
  by(case_tac x, auto)

theorem "cdr when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> (cdr x) = nil"
  by(case_tac x, auto)


end
