header {* Simplification procedures *}

theory Cheap
imports Main
begin


subsection {* The failure simproc *}


text {* Isabelle runs atop an implementation of Standard ML and follows the
  LCF tradition of theorems as an abstract datatype.  Since ML is a type-safe
  language, we can soundly write ML code that creates and manipulates theorems.
  
  Rather than writing a bunch of ad-hoc ML code to construct theorems, we can
  extend the Isabelle simplifier with custom ``simplification procedures'', 
  called simprocs.  When the simplifier sees terms of a certain pattern, it 
  will invoke our custom ML function, which can either return a rewrite rule 
  to apply to the term, or can fail. *}


text {* To begin, we will write a simproc that always fails, but prints out a 
  message when it is triggered.  This is not useful for proving theorems, but 
  shows how to set up a simplification procedure.

  Our first task is to write the actual ML code to simplify terms.  This is a 
  function of three arguments.  The first, @{text "thy"}, is an Isabelle 
  ``signature'' that has some information about our current context.  The 
  second, @{text "sset"}, is an Isabelle ``simplification set'' that contains
  the current assumptions, currently-active rewrite rules, and even the 
  currently enabled simprocs.  Finally, the last argument, @{text "redex"}, 
  is the term that we are supposed to simplify.

  The return type of our function is @{term "thm option"}, i.e., it can be
  @{text "NONE"} if we fail, or @{text "SOME thm"} on success, where 
  @{text "thm"} is the object to return.

  Our first simproc will just print a message that it has been activated,
  and then it will fail.  We escape to ML using the special keyword 
  @{text "ML"}. *}


ML {*

fun failure_proc thy assms redex = 
  let
    val msg = writeln "The failure proc is about to fail"
  in
    NONE
  end;

*}


text {* Now that we have written our simplification function, we need to 
  build a special simproc structure which is used to hook our function into
  the simplifier.  This is done by calling @{text "Simplifier.simproc"}, 
  which takes four arguments.

  The first argument is the context to use, and we will always use 
  @{text "the_context ()"}.  The second argument is just a name for this
  new simplification procedure.  The third argument is a list of trigger
  patterns which our function should be called on.  Finally, the fourth
  argument is the function we wrote. *}

ML {*

val failure_simproc = Simplifier.simproc
                       (the_context ())
                       "failure_simproc"    (* name of the procedure *)
                       [ "Suc 0" ]          (* terms to trigger on *)
                       failure_proc;

*}


text {* Finally, we need to tell the simplifier to use our hook.  We do
  this as follows. *}

ML {*

Addsimprocs [failure_simproc];

*}


text {* When we submit the following theorem, we will now see our simproc
  in action, printing "The failure proc is about to fail" once for each 
  side of the equality. *}

theorem "Suc 0 = 1" 
  by(auto)

text {* If we later decide that we want to turn off this simproc, we can do
  so as follows. *}

ML {*

Delsimprocs [failure_simproc];

*}

text {* Indeed, if we re-submit our test theorem, we will no longer see the
  messages. *}

theorem "Suc 0 = 1" 
  by(auto)





subsection {* A simproc to emulate backchain limits *}

text {* Sometimes, a group of conditional rewrite rules might trigger loops 
  during backchaining or simply be too expensive to add as enabled simp rules.
  We might be able to use simprocs to still allow these rules to be used, much
  like backchain limits in ACL2. *}

datatype widget = Red nat | White nat | Blue nat

consts is_red :: "widget \<Rightarrow> bool"
       is_white :: "widget \<Rightarrow> bool"
       is_blue :: "widget \<Rightarrow> bool"

text {* Suppose we are have the sum-of-products style datatype, @{term widget},
  defined above, and three recognizer functions, @{term is_red}, 
  @{term is_white}, and @{term is_blue} to decide if a widget is of a certain
  kind: *}

primrec 
  "is_red (Red x) = True"
  "is_red (White x) = False"
  "is_red (Blue x) = False"

primrec 
  "is_white (Red x) = False"
  "is_white (White x) = True"
  "is_white (Blue x) = False"

primrec 
  "is_blue (Red x) = False"
  "is_blue (White x) = False"
  "is_blue (Blue x) = True"

text {* Then, we could prove a number of theorems about these identifying
  functions, e.g., the following rules. *}

theorem "is_red when is_white" []:
  "\<lbrakk> is_white x \<rbrakk> \<Longrightarrow> is_red x = False"
  by(case_tac x, auto)

theorem "is_red when is_blue" []:
  "\<lbrakk> is_blue x \<rbrakk> \<Longrightarrow> is_red x = False"
  by(case_tac x, auto)

theorem "is_red when not white or blue" []:
  "\<lbrakk> ~is_white x ; ~is_blue x \<rbrakk> \<Longrightarrow> is_red x = True"
  by(case_tac x, auto)

theorem "is_white when is_red" []:
  "\<lbrakk> is_red x \<rbrakk> \<Longrightarrow> is_white x = False"
  by(case_tac x, auto)

theorem "is_white when is_blue" []:
  "\<lbrakk> is_blue x \<rbrakk> \<Longrightarrow> is_white x = False"
  by(case_tac x, auto)

theorem "is_white when not red or blue" []:
  "\<lbrakk> ~is_red x ; ~is_blue x \<rbrakk> \<Longrightarrow> is_white x = True"
  by(case_tac x, auto)

theorem "is_blue when is_red" []:
  "\<lbrakk> is_red x \<rbrakk> \<Longrightarrow> is_blue x = False"
  by(case_tac x, auto)

theorem "is_blue when is_white" []:
  "\<lbrakk> is_white x \<rbrakk> \<Longrightarrow> is_blue x = False"
  by(case_tac x, auto)

theorem "is_blue when not red or white" []:
  "\<lbrakk> ~is_red x ; ~is_white x \<rbrakk> \<Longrightarrow> is_blue x = True"
  by(case_tac x, auto)

text {* Unfortunately, these rules will loop with one another quite badly.  The
  following theorem takes a few seconds with a @{text simp_depth_limit} of 6, 
  and the problem compounds very quickly with larger limits. *}

ML {* simp_depth_limit := 1; *}

theorem "loopy theorem" []:
  "is_red x \<or> is_white x \<or> is_blue x"
  by(auto
     simp add: "is_red when is_white"
               "is_red when is_blue"
               "is_red when not white or blue"
               "is_white when is_red"
               "is_white when is_blue"
               "is_white when not red or blue"
               "is_blue when is_red"
               "is_blue when is_white"
               "is_blue when not red or white"
    )

ML {* simp_depth_limit := 100; *}




subsection {* Supporting functions *}

text {* I don't know the proper way to fold "andalso" over a list is, so I just
  write my own @{text "and_list"} function.  Perhaps someone who knows ML well
  will point out a better solution. *}

ML {*

fun and2(x, y) = x andalso y;
fun and_list xs = foldl and2 true xs;

*}

text {* I add some ML aliases for the @{typ bool} and @{typ prop} types, and
  for meta-equality between @{typ prop}s.  I also give names to the boolean 
  @{term True} and @{term False}, and their propositional counterparts.  If 
  any of this is built-in, it would be good to switch to that. *}

ML {*

val bool_type = Type("bool", []);           (* BOZO are there built-ins? *)
val bool_true = Const("True", bool_type);
val bool_false = Const("False", bool_type);

val prop_type = Type("prop", []);
val prop_equal = Const("==", prop_type --> prop_type --> prop_type);

val trueprop = Const("Trueprop", bool_type --> prop_type);
val prop_true = trueprop $ bool_true;
val prop_false = trueprop $ bool_false;

*}

text {* Because I end up doing a couple of low-level proofs about propositions,
  it's useful to be able to call upon a couple of derived rules to coerce these 
  in certain ways.  It might be possible to improve these functions, e.g., I 
  imagine @{text "read_instantiate"} is probably more expensive than 
  @{text "cterm_instantiate"}, but I haven't had much luck with the latter. *}

ML {*

(*     [] ==> A == True
   ------------------------ strip_meta_equal_true
          [] ==> A 

   Derivation.

     1. A == True              Given
     2. True == A              Symmetric; 1
     3. True                   TrueI
     4. A                      ==E; 2,3

   Q.E.D. *)

fun strip_meta_equal_true x = 
  equal_elim (symmetric x) TrueI;


(*       [] ==> A
  ---------------------- insert_meta_equal_true
     [] ==> A == True

  Derivation.

    1.  A                        Given
    2.  P ==> P = True           eqTrueI
    3.  A ==> A = True           Instantiation; 2
    4.  A = True                 ==>E; 3, 1
    5.  x = y ==> x == y         eq_reflection
    6.  A = True ==> A == True   Instantiation; 5
    7.  A == True                ==>E; 6, 4

  Q.E.D. *)

fun insert_meta_equal_true x =
  let
    val sgn = sign_of_thm x;
    val A = string_of_cterm (cterm_of sgn (concl_of x));
    val eq_true_inst = read_instantiate [("P", A)] eqTrueI;
    val hol_equalized = implies_elim eqTrueI x;
    val eq_refl_inst =  read_instantiate [("x", A), ("y", "True")] eq_reflection;
  in
    implies_elim eq_refl_inst hol_equalized                      
  end;

*}

text {* I also have written a couple of substitution routines for terms.  I'd
  be really surprised if Isabelle doesn't have equivalent functions defined 
  somewhere, but I don't know where they would be or what they are called.  In
  the meantime, at least these aren't very complicated. *}

ML {*

fun substitute_free old new (Const(x, T)) = Const(x,T)
  | substitute_free old new (Free(x, T)) = 
       if old = Free(x, T)
          then new
          else Free(x, T)
  | substitute_free old new (Var(x, T)) = Var(x, T)
  | substitute_free old new (Bound n) = Bound(n)
  | substitute_free old new (Abs(a, T, u)) = 
       Abs(a, T, substitute_free old new u)
  | substitute_free old new (t $ u) = 
       (substitute_free old new t) $ (substitute_free old new u);

fun substitute_var old new (Const(x, T)) = Const(x,T)
  | substitute_var old new (Free(x, T)) = Free(x, T)
  | substitute_var old new (Var(x, T)) = 
       if old = Var(x, T)
          then new
          else Var(x, T)
  | substitute_var old new (Bound n) = Bound(n)
  | substitute_var old new (Abs(a, T, u)) = 
       Abs(a, T, substitute_var old new u)
  | substitute_var old new (t $ u) = 
       (substitute_var old new t) $ (substitute_var old new u);

fun substitute_var_list old new [] = []
  | substitute_var_list old new (x::xs) = (substitute_var old new x) :: 
                                          (substitute_var_list old new xs);

*}

text {* Given an input term, @{term x}, Isabelle's simplifier returns a theorem
  of the form @{term "x \<equiv> y"}, where @{term y} is exactly @{term x} on
  failure or is a reduced form of @{term x} otherwise.  The following function
  will decide if the simplifier has been able to completely relieve a
  proposition to True. *}

ML {*

fun concludes_true (equal $ lhs $ rhs) = equal = prop_equal andalso rhs = prop_true
  | concludes_true (_)                 = false;

fun rewrote_to_true x = (prems_of x = nil) andalso concludes_true (concl_of x);

*}

text {* I also have a couple of debugging routines that are just useful for 
  inspecting terms.  Perhaps there are better methods for doing this. *}

ML {*

fun term_type (Const(x, T)) = "const"
  | term_type (Free(x, T)) = "free"
  | term_type (Var(x, T)) = "var"
  | term_type (Bound(n)) = "bound"
  | term_type (Abs(a, T, u)) = "abs"
  | term_type (t $ u) = "app";

*}


subsection {* Writing the simproc *}

text {* It made my head hurt too badly, sorry. *}

end