(* Isabelle Demo 
   Jared Davis
   ACL2 Seminar, October 11, 2006 

Outline.

  0. Preliminary remarks
  1. Introduction to basic Isabelle usage
  2. Working with ACL2-like definitions
  3. ML integration and simplification procedures
  4. Isar proof language


Resources.

  Isabelle Homepage     http://www.cl.cam.ac.uk/Research/HVG/Isabelle/

  Platforms             Linux, Solaris, MacOSX (no Windows)

  Installation          Just extract a few tarballs

  Interface             ProofGeneral (needs X-Emacs 21)
                        The Isabelle program runs emacs for you

  Documentation         Tutorial on Isabelle/HOL (224 pgs)
                        Also other tutorials and manuals online

  Community             Mailing list and "archive of formal proofs"

  Logics                HOL, HOLCF, ZF, FOL, custom
                        "HOL is the best developed"


Interface.
 
   Proof General, an ACL2s-like interface in X-Emacs.

   Manages books ("theories") for you.
     Very little command line interaction.

   Menus, buttons, and key commands to interact with the Isabelle process.

   X-Symbol support gives you fancy \<forall> symbols inside the editor.
     Activate with "Proof General, Options, XSymbol"
     Once activated, symbols are converted as you type them.

*)

theory Demo      (* Should agree with the file name, one theory per file *)
imports PreList  (* Like include-book.  
                      PreList theory: almost everything except lists.
                      Main theory: includes everything. *)
begin

consts my_nat :: "nat"    (* Declare that there will be some constants. *)
       my_int :: int      -- {* You can omit the quotes on simple cases. *}
       my_bool :: bool    

defs my_nat_def:          -- "Define my_nat to be 3, a nat."
     "my_nat \<equiv> 3"

defs my_int_def [simp]:   (* Define my_int to be 3, an int, AND enable it. *)
     "my_int \<equiv> 3"
     my_bool_def:         (* Also define my_bool to be True. *)
     "my_bool \<equiv> True"

theorem "my_int = 3"
  (* Since my_int is enabled, this can be done by the simplification tactic. *)
  apply(simp)
done

theorem "my_nat = 3"
  (* Since my_nat is not enabled, we have to enable it for the simplifier to 
     get it. *)
  apply(simp add: my_nat_def)
done

theorem "my_bool = True"
  (* An alternative is to explicitly insert the definition ourselves. *)
  apply(insert my_bool_def)
  apply(simp)
done

(* You might think we could try to prove the following, but we cannot.
   Equality is only defined between objects of the same type.  If we 
   submit this, we'll get a type error 
   
   theorem "my_nat \<noteq> my_int"
*)

declare my_bool_def [simp add]    (* Enables my_bool from now on. *)
declare my_int_def [simp del]     (* Disables my_int from now on. *)



(* Defining functions. *)

consts double :: "nat \<Rightarrow> nat"       (* Higher order means functions are just constants. *)
       alt_double :: "nat \<Rightarrow> nat"

defs double_def [simp]:             (* Multiplication-based definition of 2n *)
  "double n \<equiv> 2 * n"

primrec                                              (* Successor-based definition of 2n *)
  "alt_double 0 = 0"                                 (* Note the nice rewrite rules *)
  "alt_double (Suc n) = (Suc (Suc (alt_double n)))"

theorem "alt_double n = double n"
  apply(induct n rule: alt_double.induct)    (* Definitions lead to induction schemes *)
  apply(simp_all)                            (* Each subgoal can be proven automatically *)
done



(* Polymorphic tuples *)

theorem "fst (1, (2, True)) = 1"
  by(simp)

theorem "snd (1::nat, 2::int, False) = (2::int, False)"
  by(simp)



(* Aggregates *)

datatype point = Point int int

consts point_x :: "point \<Rightarrow> int"
       point_y :: "point \<Rightarrow> int"

(* Some various ways of defining accessors *)

primrec
  "point_x (Point x y) = x"

defs point_y_def: 
  "point_y p \<equiv> (case p of (Point x y) \<Rightarrow> y)"

theorem "point_y of Point" [simp]:
  "point_y (Point x y) = y"
  apply(simp add: point_y_def)
done

consts add_points :: "point \<Rightarrow> point \<Rightarrow> point"

defs add_points_def: 
  "add_points p1 p2 \<equiv> (Point (point_x p1 + point_x p2)
                             (point_y p1 + point_y p2))"

theorem "add_points of Point and Point" [simp]:
  "add_points (Point x1 y1) (Point x2 y2) = (Point (x1 + x2) (y1 + y2))"
  apply(simp add: add_points_def)
done


(* Polymorphic Types *)

datatype 'a option = None | Some 'a

types intopt = "int option"
      pointopt = "point option"

consts pointopt_x :: "pointopt \<Rightarrow> intopt" 
       pointopt_y :: "pointopt \<Rightarrow> intopt"
       add_pointopts :: "pointopt \<Rightarrow> pointopt \<Rightarrow> pointopt"

primrec
  "pointopt_x None = None"
  "pointopt_x (Some p) = Some (point_x p)"

primrec
  "pointopt_y None = None"
  "pointopt_y (Some p) = Some (point_y p)"

primrec
  "add_pointopts None p = None"
  "add_pointopts (Some p) y = 
    (case y of None \<Rightarrow> None
             | (Some p2) \<Rightarrow> Some (add_points p p2))"

theorem "add_pointopts of None and None" [simp]:
  "add_pointopts None None = None"
  by(simp)

theorem "add_pointopts of None and Point" [simp]:
  "add_pointopts None (Some p) = None"
  by(simp)

theorem "add_pointopts of Point and None" [simp]:
  "add_pointopts (Some p) None = None"
  by(simp)

theorem "add_pointopts of Point and Point" [simp]:
  "add_pointopts (Some p1) (Some p2) = Some (add_points p1 p2)"
  by(simp)


(* Recursive Types *)
       
datatype 'a list = nil | cons "'a" "'a list"

consts len :: "'a list \<Rightarrow> nat"
       app :: "'a list \<Rightarrow> 'a list \<Rightarrow> 'a list"
       rev :: "'a list \<Rightarrow> 'a list"

primrec 
  "len nil = 0"
  "len (cons a x) = 1 + len x"

primrec
  "app nil y = y"
  "app (cons a x) y = cons a (app x y)"

primrec 
  "rev nil = nil"
  "rev (cons a x) = app (rev x) (cons a nil)"

theorem "len of app" [simp]:
  "len (app x y) = len x + len y"
  apply(induct x rule: len.induct)
  apply(simp_all)
done

theorem "app of app" [simp]:
  "app (app x y) z = app x (app y z)"
  apply(induct x rule: len.induct)
  apply(simp_all)
done

theorem "app of nil" [simp]:
  "app x nil = x"
  apply(induct x rule: len.induct)
  apply(simp_all)
done

theorem "rev of app" [simp]:
  "rev (app x y) = app (rev y) (rev x)"
  apply(induct x rule: len.induct)
  apply(simp)
  apply(case_tac x)
  apply(simp_all)
done

theorem "rev of rev" [simp]:
  "rev (rev x) = x"
  apply(induct x rule: len.induct)
  apply(simp)
  apply(case_tac x)
  apply(simp_all)
done

  


(* Quantifiers are sort of neat *)

consts memberp :: "'a \<Rightarrow> 'a list \<Rightarrow> bool"
       subsetp :: "'a list \<Rightarrow> 'a list \<Rightarrow> bool"

primrec
  "memberp a nil = False"
  "memberp a (cons b x) = (a = b \<or> memberp a x)"

defs subsetp_def [simp]:
  "subsetp x y \<equiv> \<forall> a . memberp a x \<longrightarrow> memberp a y"

theorem "memberp of app" [simp]:
  "memberp a (app x y) = (memberp a x \<or> memberp a y)"
  apply(induct_tac x rule: len.induct)
  apply(simp_all)
done

theorem "memberp of rev" [simp]:
  "memberp a (rev x) = memberp a x"
  apply(induct_tac x rule: len.induct)
  apply(auto)
done

theorem "subsetp is reflexive" [simp]:
  "subsetp x x"
  by(simp)

theorem "subsetp of rev and self" [simp]:
  "subsetp (rev x) x"
  by(simp)

theorem "subsetp of x and app" [simp]:
  "subsetp x (app x y)"
  by(simp)

theorem "subsetp of y and app" [simp]:
  "subsetp y (app x y)"
  by(simp)

theorem "subsetp of apps when subsets" [simp]:
  "\<lbrakk> subsetp a b ; subsetp x y \<rbrakk> \<Longrightarrow> subsetp (app a x) (app b y)"
  by(simp)




(* Basic higher-order stuff is really easy. *)

consts map :: "('a \<Rightarrow> 'b) \<Rightarrow> 'a list \<Rightarrow> 'b list"
       filter :: "('a \<Rightarrow> bool) \<Rightarrow> 'a list \<Rightarrow> 'a list"

primrec
  "map f nil = nil"
  "map f (cons a x) = cons (f a) (map f x)"

primrec
  "filter f nil = nil"
  "filter f (cons a x) = (if (f a) 
                             then (cons a (filter f x))
                             else (filter f x))"

theorem "len of map" [simp]:
  "len (map f x) = len x"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "map of app" [simp]: 
  "map f (app x y) = app (map f x) (map f y)"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "map of rev" [simp]:
  "map f (rev x) = rev (map f x)"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "memberp of map" [simp]:
  "memberp a (map f x) = (\<exists> b . memberp b x \<and> (f b) = a)"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "len of filter" [simp]:
  "len (filter f x) <= len x"
  apply(induct x rule: len.induct)
  apply(simp)
  apply(case_tac x)
  apply(auto)
done

theorem "filter of app" [simp]:
  "filter f (app x y) = app (filter f x) (filter f y)"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "filter of rev" [simp]:
  "filter f (rev x) = rev (filter f x)"
  apply(induct x rule: len.induct)
  apply(auto)
done

theorem "memberp of filter" [simp]:
  "memberp a (filter f x) = (memberp a x \<and> f a)"
  apply(induct x rule: len.induct)
  apply(auto)
done



(* Quantifiers can be messy. *)

consts count :: "'a \<Rightarrow> 'a list \<Rightarrow> nat"
       perm  :: "'a list \<Rightarrow> 'a list \<Rightarrow> bool"

primrec 
  "count a nil = 0"
  "count a (cons b x) = (if a = b 
                            then 1 + count a x
                            else count a x)"

defs perm_def [simp]:
  "perm x y \<equiv> \<forall> a . count a x = count a y"

theorem "perm is reflexive" [simp]:
  "perm x x"
  by(auto)

theorem "perm is symmetric" [simp]:
  "perm x y = perm y x"
  by(auto)

theorem "memberp when count is zero" [simp]:
  "\<lbrakk> count a x = 0 \<rbrakk> \<Longrightarrow> memberp a x = False"
  apply(induct x rule: len.induct)
  apply(auto split: split_if_asm)
done

theorem "memberp when count is nonzero" [simp]:
  "\<lbrakk> count a x \<noteq> 0 \<rbrakk> \<Longrightarrow> memberp a x = True"
  apply(induct x rule: len.induct)
  apply(auto split: split_if_asm)
done



(* But they can get really messy if, like me, you don't know what you're doing. *)

(* 
theorem "perm implies subset" [simp]:
  "\<lbrakk> perm x y \<rbrakk> \<Longrightarrow> subsetp x y"
  apply(auto)
  apply(erule allE) 
  apply(case_tac "count a x = 0")
  apply(auto)
  apply(case_tac "count a y = 0")
  apply(auto)
  apply(erule contrapos_pp)
  apply(simp)
  apply(case_tac "count a x = count a y")
  apply(auto)
  -- {* Yuck! *}
done
*)

(* Once you are done with a theory, you can just hit "next" through the end and
   ProofGeneral will handle all the certification for you.  You can then import
   it into your next theory without any trouble.

   You can also do the equivalent of a certify-book from the command line.  First,
   set up a file called ROOT.ML that says which theories to include.  Its
   syntax is:

     use_thy "Theory1";
     use_thy "Theory2";
     \<dots>
     use_thy "TheoryN";

   Then you run the isatool command.  It's syntax is:

      isatool usedir 
                -v true   [for verbose]
                -d pdf    [build a pdf]
                -b        [build an image]
                HOL       [the logic to use]
                name      [the name to use])

   For example, to certify Demo.thy without building a PDF, we can run:

      isatool usedir -v true -b HOL Demo.img

   I don't know what the standard file naming conventions are.  At this point,
   Demo becomes a "registered theory" and Demo.img becomes an Isabelle 
   executable that has Demo preloaded, and can be loaded from ProofGeneral.
*)

end
