header {* List Utility Functions *}

theory ListUtilities
imports Base Arithmetic
begin


text {* We now begin working through the primitive Milawa utility functions
  and setting up a HOL reasoning strategy about them. *}

consts
  cdr_cdr_induct :: "object \<times> object \<Rightarrow> object"
  len :: "object \<Rightarrow> object"
  true_listp :: "object \<Rightarrow> object"
  list_fix :: "object \<Rightarrow> object" 
  memberp :: "object \<Rightarrow> object \<Rightarrow> object" 
  subsetp :: "object \<Rightarrow> object \<Rightarrow> object" 
  app :: "object \<Rightarrow> object \<Rightarrow> object" 
  rev :: "object \<Rightarrow> object" 
  remove_all :: "object \<Rightarrow> object \<Rightarrow> object"
  disjointp :: "object \<Rightarrow> object \<Rightarrow> object" 
  uniquep :: "object \<Rightarrow> object" 
  difference :: "object \<Rightarrow> object \<Rightarrow> object"
  remove_duplicates :: "object \<Rightarrow> object"
  prefixp :: "object \<Rightarrow> object \<Rightarrow> object"



subsection {* @{term cdr_cdr_induct} *}

text {* The @{term cdr_cdr_induct} function doesn't compute anything useful, it
  just generates a useful induction scheme. *}

recdef cdr_cdr_induct 
  "measure (\<lambda>(x, y) . hol_rank x)"
  "cdr_cdr_induct(Symbol x, y) = nil"
  "cdr_cdr_induct(Integer x, y) = nil"
  "cdr_cdr_induct(x, Symbol y) = nil"
  "cdr_cdr_induct(x, Integer y) = nil"
  "cdr_cdr_induct(Cons a x, Cons b y) = cdr_cdr_induct(x, y)"



subsection {* @{term len} *}

text {* Our definition of @{term len} doesn't look much like Milawa's since we
  use pattern matching, which works well with Isabelle's simplifier. *}

primrec
  "len (Symbol x) = zero"
  "len (Integer x) = zero"
  "len (Cons a x) = plus one (len x)"

text {* To address this disconnect, we prove the following theorem, which shows
  the pattern-matching definition lines up with the Lisp definition. *}

theorem "len correspondence" []:
  "len x = (ite (consp x) 
                (plus one (len (cdr x))) 
             zero)"
  by(induct x rule: len.induct, auto)

text {* We will prove similar correspondence theorems for just about every
  function we introduce from now on.  We leave @{thm [source] "len correspondence"} 
  and similar theorems disabled so they do not interfere with rewriting. *}

theorem "len when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> len x = zero"
  by(case_tac x, auto)

theorem "integerp of len" [simp]:
  "integerp (len x) = t"
  by(case_tac x, auto)

theorem "lessp of len and zero" [simp]:
  "lessp (len x) zero = nil"
  by(induct_tac x rule: len.induct, auto)

theorem "lessp of zero and len" [simp]:
  "lessp zero (len x) = (consp x)"
  by(induct_tac x rule: len.induct, auto)

theorem "decrement len when consp" [simp]:
  "\<lbrakk> consp x = t \<rbrakk> \<Longrightarrow> plus (len x) negone = len (cdr x)"
  by(case_tac x, auto)


 
subsection {* @{term true_listp} *}

primrec
  "true_listp (Symbol x) = (if (x = nil_number) then t else nil)"
  "true_listp (Integer x) = nil"
  "true_listp (Cons a x) = true_listp x"

theorem "true_listp correspondence" []:
  "true_listp x = (ite (consp x) 
                       (true_listp (cdr x))
                    (equal x nil))"
  by(induct x rule: len.induct, auto)

theorem "true_listp when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> true_listp x = equal x nil"
  by(case_tac x, auto)

theorem "booleanp of true_listp" [simp]:
  "booleanp (true_listp x) = t"
  by(induct x rule: true_listp.induct, auto)

theorem "len zero when true_listp" [simp]:                (* Added for Isabelle *)
  "\<lbrakk> true_listp x = t \<rbrakk> \<Longrightarrow> (zero = len x) = (nil = x)"
  by(case_tac x, auto split: split_if_asm)

(* BOZO.  In ACL2, we have the following rule with a backchain limit to make
   it cheap.  I don't know how to do a backchain limit here, so it ends up
   being really expensive and slow.  Is there a way to recreate the rule in 
   Isabelle, and is it a useful rule?

   theorem "consp when true_listp" [simp]:
     "\<lbrakk> true_listp x = t \<rbrakk> \<Longrightarrow> (consp x) = (ite x t nil)"
     by(case_tac x, auto split: split_if_asm)
*)



subsection {* @{term list_fix} *}

primrec
  "list_fix (Symbol x) = nil"
  "list_fix (Integer x) = nil"
  "list_fix (Cons a x) = Cons a (list_fix x)"

theorem "list_fix correspondence" []:
  "list_fix x = (ite (consp x)
                     (Cons (car x) (list_fix (cdr x)))
                  nil)"
  by(induct x rule: len.induct, auto)

theorem "list_fix when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> list_fix x = nil"
  by(case_tac x, auto)

theorem "consp of list_fix" [simp]:
  "consp (list_fix x) = consp x"
  by(case_tac x, auto)

theorem "list_fix under iff" [simp]:
  "(list_fix x \<noteq> nil) = ((consp x) = t)"
  by(case_tac x, auto)

theorem "len of list_fix" [simp]:
  "len (list_fix x) = len x"
  by(induct x rule: len.induct, auto)

theorem "true_listp of list_fix" [simp]:
  "true_listp (list_fix x) = t"
  by(induct x rule: len.induct, auto)

theorem "list_fix when true_listp" [simp]:
  "\<lbrakk> true_listp x = t \<rbrakk> \<Longrightarrow> list_fix x = x"
  by(induct x rule: len.induct, auto split: split_if_asm)




subsection {* @{term memberp} *}

primrec
  "memberp a (Symbol x) = nil"
  "memberp a (Integer x) = nil"
  "memberp a (Cons b x) = equal a b or memberp a x"

theorem "memberp correspondence" []:
  "memberp a x = (ite (consp x)
                      (equal a (car x) or memberp a (cdr x))
                    nil)"
  by(induct x rule: len.induct, auto)
  
theorem "member when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> memberp a x = nil"
  by(case_tac x, auto)

theorem "booleanp of memberp" [simp]:
  "booleanp (memberp a x) = t"
  by(induct x rule: len.induct, auto)

theorem "memberp of list_fix" [simp]:
  "memberp a (list_fix x) = memberp a x"
  by(induct x rule: len.induct, auto)

theorem "memberp of car" [simp]:
  "memberp (car x) x = consp x"
  by(case_tac x, auto)

theorem "memberp of second" [simp]:
  "\<lbrakk> consp x = t ; consp (cdr x) = t \<rbrakk> \<Longrightarrow> memberp (second x) x = t"
  by(case_tac x, auto)

theorem "car when memberp and not memberp of cdr" [simp]:
  "\<lbrakk> memberp a x = t ; memberp a (cdr x) = nil \<rbrakk> \<Longrightarrow> car x = a"
  by(case_tac x, auto split: split_if_asm)

theorem "consp when memberp" []: -- {* Expensive *}
  "\<lbrakk> memberp a x = t \<rbrakk> \<Longrightarrow> consp x = t"
  by(case_tac x, auto)

theorem "consp of cdr when memberp not car" [simp]:
  "\<lbrakk> memberp a x = t ; car x \<noteq> a \<rbrakk> \<Longrightarrow> consp (cdr x) = t"
  by(case_tac "cdr x", auto)
  
(* BOZO.  In ACL2, we have the following rule, which is a Pump.  But, if I 
   add this to Isabelle, the simplifier immediately starts looping and getting
   confused.  Is there a way to write a similar rule in Isabelle or to add a 
   backchain limit or something?  How does ACL2 handle pumps?  Heuristics?

   theorem "memberp when memberp of cdr" [simp]:
     "\<lbrakk> memberp a (cdr x) = t \<rbrakk> \<Longrightarrow> memberp a x = t"
     by(case_tac x, auto) 


   Our ACL2 file has the following theorem.  I don't think we need it anymore, 
   since without a backchain limit it is subsumed by the rule afterwards. 

   theorem "car when memberp of singleton list (cheap)" [simp]:
     "\<lbrakk> consp (cdr x) = nil ; memberp a x = t \<rbrakk> \<Longrightarrow> car x = a"
      by(case_tac x, auto split: split_if_asm)

   BOZO consider removing this rule and changing the backchain limit on the
   following one to 1 in the ACL2 sources.

   BOZO consider removing the backchain limit alltogheter in the ACL2 sources
   since the free variable may be sufficiently cheap.

*)



subsection {* @{term subsetp} *}

primrec
  "subsetp (Symbol x) y = t"
  "subsetp (Integer x) y = t"
  "subsetp (Cons a x) y = memberp a y and subsetp x y"

theorem "subsetp correspondence" []:
  "subsetp x y = (ite (consp x)
                      (memberp (car x) y and subsetp (cdr x) y)
                    t)"
  by(induct x rule: len.induct, auto)

theorem "subsetp when not consp (left)" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> subsetp x y = t"
  by(case_tac x, auto)

theorem "subsetp when not consp (right)" [simp]:
  "\<lbrakk> consp y = nil \<rbrakk> \<Longrightarrow> subsetp x y = not (consp x)"
  by(case_tac x, auto)

theorem "subsetp of Symbol (right)" [simp]: -- {* Isabelle style *}
  "subsetp x (Symbol y) = not (consp x)"
  by(case_tac x, auto)

theorem "subsetp of Integer (right)" [simp]: -- {* Isabelle style *}
  "subsetp x (Integer y) = not (consp x)"
  by(case_tac x, auto)

theorem "subsetp of cons (right) when subsetp" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> subsetp x (Cons a y) = t"
  by(induct x rule: len.induct, auto split: split_if split_if_asm)

theorem "booleanp of subsetp" [simp]:
  "booleanp (subsetp x y) = t"
  by(induct x rule: len.induct, auto)

theorem "subsetp of list_fix (left)" [simp]:
  "subsetp (list_fix x) y = subsetp x y"
  by(induct x rule: len.induct, auto)

theorem "subsetp of list_fix (right)" [simp]:
  "subsetp x (list_fix y) = subsetp x y"
  by(induct x rule: len.induct, auto)

theorem "subsetp of cdr when subsetp" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> subsetp (cdr x) y = t"
  by(case_tac x, auto split: split_if_asm)

theorem "memberp when memberp of subsetp" []: -- {* Loops *}
  "\<lbrakk> subsetp x y = t ; memberp a x = t \<rbrakk> \<Longrightarrow> memberp a y = t"
  by(induct x rule: len.induct, auto split: split_if_asm)

theorem "memberp when not memberp of superset" []: -- {* Loops *}
  "\<lbrakk> subsetp x y = t ; memberp a y = nil \<rbrakk> \<Longrightarrow> memberp a x = nil"
  by(induct x rule: len.induct, auto split: split_if_asm)

theorem "subsetp is reflexive" [simp]:
  "subsetp x x = t"
  by(induct x rule: len.induct, auto)

theorem "subsetp is transitive" []: -- {* Loops *}
  "\<lbrakk> subsetp x y = t ; subsetp y z = t \<rbrakk> \<Longrightarrow> subsetp x z = t"
  apply(induct x rule: len.induct)
  apply(auto split: split_if_asm
             simp add: "memberp when memberp of subsetp")
done

theorem "subsetp by membership" []:
  "subsetp x y = (if (\<forall> a . memberp a x = t \<longrightarrow> memberp a y = t)
                       then t else nil)"
  by(induct x rule: len.induct, auto)

(* BOZO in ACL2 we have this rule with a backchain limit, but it is too
   expensive to add to Isabelle.

   theorem "consp when nonempty subsetp" [simp]:
     "\<lbrakk> subsetp(x, y) = t ; consp x = t \<rbrakk> \<Longrightarrow> consp y = t"
     by(case_tac y, auto split: split_if_asm) 
*)



subsection {* @{term app} *}

primrec 
  "app (Symbol x) y = list_fix y"
  "app (Integer x) y = list_fix y"
  "app (Cons a x) y = Cons a (app x y)"

theorem "app when not consp (left)" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> app x y = list_fix y"
  by(case_tac x, auto)

theorem "app of list_fix (left)" [simp]:
  "app (list_fix x) y = app x y"
  by(induct x rule: len.induct, auto)

theorem "app of list_fix (right)" [simp]:
  "app x (list_fix y) = app x y"
  by(induct x rule: len.induct, auto)

theorem "app when not consp (right)" [simp]:
  "\<lbrakk> consp y = nil \<rbrakk> \<Longrightarrow> app x y = list_fix x"
  by(induct x rule: len.induct, auto)

theorem "true_listp of app" [simp]:
  "true_listp (app x y) = t"
  by(induct x rule: len.induct, auto)

theorem "app of app" [simp]:
  "app (app x y) z = app x (app y z)"
  by(induct x rule: len.induct, auto)

theorem "memberp of app" [simp]:
  "memberp a (app x y) = memberp a x or memberp a y"
  by(induct x rule: len.induct, auto)

theorem "consp of app" [simp]:
  "consp (app x y) = consp x or consp y"
  by(induct x rule: len.induct, auto)

theorem "app under iff" [simp]:
  "(app x y \<noteq> nil) = (if consp x = t then True else consp y = t)"
  by(case_tac x, auto)

theorem "len of app" [simp]:
  "len (app x y) = plus (len x) (len y)"
  by(induct x rule: len.induct, auto)

theorem "subsetp of app (left)" [simp]:
  "subsetp (app x y) z = subsetp x z and subsetp y z"
  by(induct x rule: len.induct, auto)

theorem "subsetp of app with self (left)" [simp]:
  "subsetp x (app x y) = t"
  by(induct x rule: len.induct, auto)

theorem "subsetp of app with self (right)" [simp]:
  "subsetp x (app y x) = t"
  apply(insert "subsetp by membership" [of x "app y x"])
  apply(auto split: split_if_asm)
done

theorem "subsetp of apps when subsets" [simp]:
  "\<lbrakk> subsetp x y = t ; subsetp w z = t \<rbrakk> 
     \<Longrightarrow> 
     subsetp (app x w) (app y z) = t"
  apply(insert "subsetp by membership" [of "app x w" "app y z"])
  apply(split split_if_asm)     -- {* Ugly *}
  apply(auto split: split_if_asm
             simp add: "memberp when memberp of subsetp")
done

theorem "subsetp of symmetric apps" [simp]:
  "subsetp (app x y) (app y x) = t"
  apply(insert "subsetp by membership" [of "app x y" "app y x"])
  apply(auto split: split_if_asm)
done

theorem "weirdo rule for subsetp of app (one)" [simp]:
  "subsetp (app (cdr x) (Cons (car x) y)) (app x y) = consp x or memberp nil y"
  by(auto)

theorem "weirdo rule for subsetp of app (two)" [simp]:
  "subsetp (app (cdr x) (Cons (car x) y)) (app y x) = consp x or memberp nil y"
  by(auto)

theorem "cdr of app when x is consp" [simp]:
  "\<lbrakk> consp x = t \<rbrakk> \<Longrightarrow> cdr (app x y) = app (cdr x) y"
  by(case_tac x, auto)

theorem "car of app when x is consp" [simp]:
  "\<lbrakk> consp x = t \<rbrakk> \<Longrightarrow> car (app x y) = car x"
  by(case_tac x, auto)

theorem "memberp of app onto singleton" [simp]:
  "memberp a (app x (Cons b nil)) = memberp a x or equal a b"
  by(auto)

theorem "subsetp of app onto singleton with cons" [simp]:
  "subsetp (app x (Cons a nil)) (Cons a x) = t"
  by(auto)

theorem "subsetp of cons with app onto singleton" [simp]:
  "subsetp (Cons a x) (app x (Cons a nil)) = t"
  by(auto)




subsection {* @{term rev} *}

primrec
  "rev (Symbol x) = nil"
  "rev (Integer x) = nil"
  "rev (Cons a x) = (app (rev x) (Cons a nil))"

theorem "rev correspondence" []:
  "rev x = (ite (consp x)
                (app (rev (cdr x)) (Cons (car x) nil))
              nil)"
  by(induct x rule: len.induct, auto)

theorem "rev when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> rev x = nil"
  by(case_tac x, auto)

theorem "rev of list_fix" [simp]:
  "rev (list_fix x) = rev x"
  by(induct x rule: len.induct, auto)

theorem "true_listp of rev" [simp]:
  "true_listp (rev x) = t"
  by(induct x rule: len.induct, auto)

theorem "len of rev" [simp]:
  "len (rev x) = len x"
  by(induct x rule: len.induct, auto)

theorem "rev under iff" [simp]:
  "(rev x \<noteq> nil) = (if consp x = t then True else False)"
  by(case_tac x, auto)

theorem "memberp of rev" [simp]:
  "memberp a (rev x) = memberp a x"
  by(induct x rule: len.induct, auto)

theorem "subsetp of rev (left)" [simp]:
  "subsetp (rev x) y = subsetp x y"
  apply(insert "subsetp by membership" [of "rev x" y])
  apply(insert "subsetp by membership" [of x y])
  apply(auto)
done

theorem "subsetp of rev (right)" [simp]:
  "subsetp x (rev y) = subsetp x y"
  apply(insert "subsetp by membership" [of x "rev y"])
  apply(insert "subsetp by membership" [of x y])
  apply(auto)
done

lemma "rev of rev lemma" []:
  "rev (app x (Cons a nil)) = Cons a (rev x)"
  by(induct x rule: rev.induct, auto)

theorem "rev of rev" [simp]:
  "rev (rev x) = list_fix x"
  by(induct x rule: rev.induct, auto simp add: "rev of rev lemma")

theorem "rev of app" [simp]:
  "rev (app x y) = app (rev y) (rev x)"
  by(induct x rule: len.induct, auto)



subsection {* @{term remove_all} *}

primrec 
  "remove_all a (Symbol x) = nil"
  "remove_all a (Integer x) = nil"
  "remove_all a (Cons b x) = (if (a = b) 
                                 then remove_all a x
                                 else Cons b (remove_all a x))"

theorem "remove_all correspondence" []:
  "remove_all a x = (ite (consp x)
                         (ite (equal a (car x))
                              (remove_all a (cdr x))
                           (Cons (car x)
                                 (remove_all a (cdr x))))
                      nil)"
  by(induct x rule: len.induct, auto)

theorem "remove_all when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> remove_all a x = nil"
  by(case_tac x, auto)

theorem "remove_all of list_fix" [simp]:
  "remove_all a (list_fix x) = remove_all a x"
  by(induct x rule: len.induct, auto)

theorem "true_listp of remove_all" [simp]:
  "true_listp (remove_all a x) = t"
  by(induct x rule: len.induct, auto)

theorem "memberp of remove_all" [simp]:
  "memberp a (remove_all b x) = memberp a x and not (equal a b)"
  by(induct x rule: len.induct, auto)

theorem "remove_all of app" [simp]:
  "remove_all a (app x y) = app (remove_all a x) (remove_all a y)"
  by(induct x rule: len.induct, auto)

theorem "remove_all of rev" [simp]:
  "remove_all a (rev x) = rev (remove_all a x)"
  by(induct x rule: len.induct, auto)

theorem "subsetp of remove_all with self" [simp]:
  "subsetp (remove_all a x) x = t"
  by(induct x rule: len.induct, auto)

theorem "subsetp of remove_all with remove_all when subsetp" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> subsetp (remove_all a x) (remove_all a y) = t"
  apply(insert "subsetp by membership" [of "remove_all a x" "remove_all a y"])
  apply(auto split: split_if_asm 
             simp add: "memberp when memberp of subsetp")
done

theorem "subsetp of remove_all when subsetp" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> subsetp (remove_all a x) y = t"
  apply(insert "subsetp by membership" [of "remove_all a x" y])
  apply(auto split: split_if_asm
             simp add: "memberp when memberp of subsetp")
done

theorem "remove_all when not memberp" [simp]:
  "\<lbrakk> memberp a x = nil \<rbrakk> \<Longrightarrow> remove_all a x = list_fix x"
  by(induct x rule: len.induct, auto)

theorem "remove_all of remove_all" [simp]:
  "remove_all a (remove_all b x) = remove_all b (remove_all a x)"
  by(induct x rule: len.induct, auto)




subsection {* @{term disjointp} *}

primrec 
  "disjointp (Symbol x) y = t"
  "disjointp (Integer x) y = t"
  "disjointp (Cons a x) y = not (memberp a y) and (disjointp x y)"

theorem "disjointp correspondence" []:
  "disjointp x y = (ite (consp x)
                        ( (not (memberp (car x) y))
                            and
                          (disjointp (cdr x) y) )
                        t)"
  by(induct x rule: len.induct, auto)

theorem "disjointp when not consp (left)" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> disjointp x y = t"
  by(case_tac x, auto)

theorem "disjointp when not consp (right)" [simp]:
  "\<lbrakk> consp y = nil \<rbrakk> \<Longrightarrow> disjointp x y = t"
  by(induct x rule: len.induct, auto)

theorem "booleanp of disjointp" [simp]:
  "booleanp (disjointp x y) = t"
  by(induct x rule: len.induct, auto)

theorem "disjointp of Symbol (right)" [simp]: -- {* Isabelle style *}
  "disjointp x (Symbol y) = t"
  by(induct x rule: len.induct, auto)

theorem "disjointp of Integer (right)" [simp]: -- {* Isabelle style *}
  "disjointp x (Integer y) = t"
  by(induct x rule: len.induct, auto)

theorem "disjointp of cons (right)" [simp]:
  "disjointp x (Cons a y) = not (memberp a x) and disjointp x y"
  by(induct x rule: len.induct, auto)

theorem "symmetry of disjointp" [simp]:
  "disjointp x y = disjointp y x"
  by(induct x rule: len.induct, auto)

theorem "disjointp of list_fix (left)" [simp]:
  "disjointp (list_fix x) y = disjointp x y"
  by(induct x rule: len.induct, auto)

theorem "disjointp of list_fix (right)" [simp]:
  "disjointp x (list_fix y) = disjointp x y"
  by(induct x rule: len.induct, auto)

theorem "disjointp when common member" [simp]:
  "\<lbrakk> memberp a x = t ; memberp a y = t \<rbrakk> \<Longrightarrow> disjointp x y = nil"
  by(induct x rule: len.induct, auto split: split_if_asm)

theorem "disjointp of app (right)" [simp]:
  "disjointp x (app y z) = disjointp x y and disjointp x z"
  by(induct x rule: len.induct, auto)

theorem "disjointp of app (left)" [simp]:
  "disjointp (app x y) z = disjointp x z and disjointp y z"
  by(auto)

theorem "disjointp of rev (right)" [simp]:
  "disjointp x (rev y) = disjointp x y"
  by(induct x rule: len.induct, auto)

theorem "disjointp of rev (left)" [simp]:
  "disjointp (rev x) y = disjointp x y"
  by(auto)

theorem "disjointp when subsetp of disjointp (left)" [simp]:
  "\<lbrakk> subsetp x y = t; disjointp y z = t \<rbrakk> \<Longrightarrow> disjointp x z = t"
  apply(induct x rule: len.induct)
  apply(auto split: split_if_asm) 
  apply(erule contrapos_pp)         -- {* Ugly *}
  apply(auto)
done

theorem "disjointp when subsetp of disjointp (right)" [simp]:
  "\<lbrakk> subsetp x z = t ; disjointp y z = t \<rbrakk> \<Longrightarrow> disjointp x y = t"
  by(auto)

theorem "disjointp when subsetp of other (left)" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> disjointp x y = not (consp x)"
  by(induct x rule: len.induct, auto)

theorem "disjointp when subsetp of other (right)" [simp]:
  "\<lbrakk> subsetp x y = t \<rbrakk> \<Longrightarrow> disjointp y x = not (consp x)"
  by(auto)

theorem "memberp when disjointp (left)" []:  -- {* Loops *}
  "\<lbrakk> disjointp x y = t ; memberp a x = t \<rbrakk> \<Longrightarrow> memberp a y = nil"
  by(induct x rule: len.induct, auto split: split_if_asm)

theorem "memberp when disjointp (right)" []:  -- {* Loops *}
  "\<lbrakk> disjointp x y = t ; memberp a y = t \<rbrakk> \<Longrightarrow> memberp a x = nil"
  apply(insert "memberp when disjointp (left)" [of x y a])
  apply(auto)
  apply(erule contrapos_pp) -- {* Ugly *}
  apply(auto)
done


(* BOZO I think these are redundant.

  theorem "disjointp of singleton (left)" [simp]:
    "disjointp (Cons a nil) x = not (memberp a x)"
    by(auto)

  theorem "disjointp of singleton (right)" [simp]:
    "disjointp x (Cons a nil) = not (memberp a x)"
    by(auto) 
*)


subsection {* @{term uniquep} *}

primrec 
  "uniquep (Symbol x) = t"
  "uniquep (Integer x) = t"
  "uniquep (Cons a x) = not (memberp a x) and uniquep x"

theorem "uniquep correspondence" []:
  "uniquep x = (ite (consp x)
                    ((not (memberp (car x) (cdr x)))
                       and
                     (uniquep (cdr x)))
                  t)"
  by(induct x rule: len.induct, auto)

theorem "uniquep when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> uniquep x = t"
  by(case_tac x, auto)

theorem "uniquep of list_fix" [simp]:
  "uniquep (list_fix x) = uniquep x"
  by(induct x rule: len.induct, auto)

theorem "booleanp of uniquep" [simp]:
  "booleanp (uniquep x) = t"
  by(induct x rule: len.induct, auto)

theorem "uniquep of cdr when uniquep" [simp]:
  "\<lbrakk> uniquep x = t \<rbrakk> \<Longrightarrow> uniquep (cdr x) = t"
  by(case_tac x, auto split: split_if_asm)

theorem "memberp of car in cdr when uniquep" [simp]:
  "\<lbrakk> uniquep x = t \<rbrakk> \<Longrightarrow> memberp (car x) (cdr x) = nil"
  by(case_tac x, auto split: split_if_asm)

theorem "uniquep of app" [simp]:
  "uniquep (app x y) = uniquep x and uniquep y and disjointp x y"
  by(induct x rule: len.induct, auto)

theorem "uniquep of rev" [simp]:
  "uniquep (rev x) = uniquep x"
  by(induct x rule: len.induct, auto)

theorem "uniquep of remove_all when uniquep" [simp]:
  "\<lbrakk> uniquep x = t \<rbrakk> \<Longrightarrow> uniquep (remove_all a x) = t"
  by(induct x rule: len.induct, auto)



subsection {* @{term difference} *}

primrec 
  "difference (Symbol x) y = nil"
  "difference (Integer x) y = nil" 
  "difference (Cons a x) y = (ite (memberp a y)
                                  (difference x y)
                                (Cons a (difference x y)))"

theorem "difference correspondence" []:
  "difference x y = (ite (consp x)
                         (ite (memberp (car x) y)
                              (difference (cdr x) y)
                           (Cons (car x) 
                                 (difference (cdr x) y)))
                        nil)"
  by(induct x rule: len.induct, auto)

theorem "difference when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> difference x y = nil"
  by(case_tac x, auto)

theorem "true_listp of difference" [simp]:
  "true_listp (difference x y) = t"
  by(induct x rule: len.induct, auto)

theorem "difference of list_fix (left)" [simp]:
  "difference (list_fix x) y = difference x y"
  by(induct x rule: len.induct, auto)

theorem "difference of list_fix (right)" [simp]:
  "difference x (list_fix y) = difference x y"
  by(induct x rule: len.induct, auto)

theorem "difference of app (left)" [simp]:
  "difference (app x y) z = app (difference x z) (difference y z)"
  by(induct x rule: len.induct, auto)

theorem "difference of difference (left)" [simp]:
  "difference (difference x y) z = difference x (app y z)"
  by(induct x rule: len.induct, auto)
  
theorem "difference of rev (left)" [simp]:
  "difference (rev x) y = rev (difference x y)"
  by(induct x rule: len.induct, auto)

theorem "difference of rev (right)" [simp]:
  "difference x (rev y) = difference x y"
  by(induct x rule: len.induct, auto)

theorem "memberp of difference" [simp]:
  "memberp a (difference x y) = memberp a x and not (memberp a y)"
  by(induct x rule: len.induct, auto)

theorem "uniquep of difference when uniquep" [simp]:
  "\<lbrakk> uniquep x = t \<rbrakk> \<Longrightarrow> uniquep (difference x y) = t"
  by(induct x rule: len.induct, auto)

theorem "disjointp of difference with rhs" [simp]:
  "disjointp (difference x y) y = t"
  by(induct x rule: len.induct, auto)




subsection {* @{term remove_duplicates} *}

primrec
  "remove_duplicates (Symbol x) = nil"
  "remove_duplicates (Integer x) = nil" 
  "remove_duplicates (Cons a x) = (ite (memberp a x)
                                       (remove_duplicates x)
                                    (Cons a (remove_duplicates x)))"

theorem "remove_duplicates correspondence" []:
  "remove_duplicates x = (ite (consp x)
                              (ite (memberp (car x) (cdr x))
                                   (remove_duplicates (cdr x))
                                (Cons (car x) 
                                      (remove_duplicates (cdr x))))
                           nil)"
  by(induct x rule: len.induct, auto)

theorem "remove_duplicates when not consp" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> remove_duplicates x = nil"
  by(case_tac x, auto)

theorem "true_listp of remove_duplicates" [simp]:
  "true_listp (remove_duplicates x) = t"
  by(induct x rule: len.induct, auto)

theorem "len of remove_duplicates" [simp]:
  "lessp (len x) (len (remove_duplicates x)) = nil"
  by(induct x rule: len.induct, auto)

theorem "remove_duplicates of list_fix" [simp]:
  "remove_duplicates (list_fix x) = remove_duplicates x"
  by(induct x rule: len.induct, auto)

theorem "memberp of remove_duplicates" [simp]:
  "memberp a (remove_duplicates x) = memberp a x"
  by(induct x rule: len.induct, auto)

theorem "uniquep of remove_duplicates" [simp]:
  "uniquep (remove_duplicates x) = t"
  by(induct x rule: len.induct, auto)

theorem "remove_duplicates of difference" [simp]:
  "remove_duplicates (difference x y) = difference (remove_duplicates x) y"
  by(induct x rule: len.induct, auto)

theorem "remove_duplicates when uniquep" [simp]:
  "\<lbrakk> uniquep x = t \<rbrakk> \<Longrightarrow> remove_duplicates x = list_fix x"
  by(induct x rule: len.induct, auto)

theorem "subsetp of remove_duplicates (left)" [simp]:
  "subsetp (remove_duplicates x) y = subsetp x y"
  apply(induct x rule: len.induct)
  apply(auto)
  apply(erule contrapos_pp)  -- {* Ugly *}
  apply(auto simp add: "memberp when memberp of subsetp")
done

theorem "subsetp of remove_duplicates (right)" [simp]:
  "subsetp x (remove_duplicates y) = subsetp x y"
  by(induct x rule: len.induct, auto)

theorem "remove_duplicates of app when uniquep and disjointp" [simp]:
  "\<lbrakk> uniquep y = t ; disjointp x y = t \<rbrakk> 
     \<Longrightarrow> 
     remove_duplicates (app x y) = app (remove_duplicates x) y"
  by(induct x rule: len.induct, auto split: split_if_asm)

theorem "remove_duplicates of remove_all" [simp]:
  "remove_duplicates (remove_all a x) = remove_all a (remove_duplicates x)"
  by(induct x rule: len.induct, auto)




subsection {* @{term prefixp} *}

primrec 
  "prefixp (Symbol x) y = t"
  "prefixp (Integer x) y = t"
  "prefixp (Cons a x) y = (consp y 
                             and equal a (car y)
                             and prefixp x (cdr y))"

theorem "prefixp correspondence" []:
  "prefixp x y = (ite (consp x)
                      (consp y 
                         and equal (car x) (car y) 
                         and prefixp (cdr x) (cdr y))
                     t)"
  by(induct x rule: len.induct, auto)

theorem "prefixp when not consp (left)" [simp]:
  "\<lbrakk> consp x = nil \<rbrakk> \<Longrightarrow> prefixp x y = t"
  by(case_tac x, auto)

theorem "prefixp when not consp (right)" [simp]:
  "\<lbrakk> consp y = nil \<rbrakk> \<Longrightarrow> prefixp x y = not (consp x)"
  by(case_tac x, auto)

theorem "prefixp of Symbol (right)" [simp]: -- {* Isabelle style *}
  "prefixp x (Symbol y) = not (consp x)"
  by(auto)

theorem "prefixp of Integer (right)" [simp]: -- {* Isabelle style *}
  "prefixp x (Integer y) = not (consp x)"
  by(auto)

theorem "prefixp of cons and cons" [simp]:
  "prefixp (Cons a x) (Cons b y) = equal a b and prefixp x y"
  by(auto)

theorem "booleanp of prefixp" [simp]:
  "booleanp (prefixp x y) = t"
  by(induct x y rule: cdr_cdr_induct.induct, auto)

theorem "prefixp of list_fix (left)" [simp]:
  "prefixp (list_fix x) y = prefixp x y"
  by(induct x y rule: cdr_cdr_induct.induct, auto)

theorem "prefixp of list_fix (right)" [simp]:
  "prefixp x (list_fix y) = prefixp x y"
  by(induct x y rule: cdr_cdr_induct.induct, auto)

theorem "same length prefixes equal" []: -- {* Maybe Expensive *}
  "\<lbrakk> prefixp x y = t ; true_listp x = t ; true_listp y = t \<rbrakk> 
     \<Longrightarrow>
     (x = y) = (len x = len y)"
  by(induct x y rule: cdr_cdr_induct.induct, auto split: split_if_asm)

theorem "prefixp when lengths are wrong" [simp]:
  "\<lbrakk> lessp (len y) (len x) = t \<rbrakk> \<Longrightarrow> prefixp x y = nil"
  by(induct x y rule: cdr_cdr_induct.induct, auto split: split_if_asm)

end

