; Efficient Animation of ACL2 Models.

(ccl::egc nil)
(declaim (optimize (speed 3) (safety 0) (space 0)))

; We introduce eobj structures as a wrapper to store any "non-native" objects.
;
; As an invariant to minimize our performance impact, every ordinary cons may
; consist only of native objects.
;
; In this file, we consider only two kinds of extended objects:
;  (1) simple 1-dimensional arrays (type :earray)
;  (2) "extended conses" (type :econs)

(defstruct eobj
  (type)
  (contents))
       
(defstruct econs 
  (car)
  (cdr))

(declaim (inline my-car 
                 my-cdr
                 my-consp
                 my-cons
                 my-equal))

(defun my-car (x)
  (typecase x 
   (cons
    (car x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-car (eobj-contents x)))
      (otherwise (error "Tried to take car of ~a.~%" (eobj-type x)))))
   (t 
    nil)))

(defun my-cdr (x)
  (typecase x 
   (cons
    (cdr x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-cdr (eobj-contents x)))
      (otherwise (error "Tried to take cdr of ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-consp (x)
  (typecase x
   (cons
    t)
   (eobj
    (case (eobj-type x)
      (:econs t)
      (otherwise (error "Tried to consp ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-cons (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (make-eobj :type :econs
                 :contents (make-econs :car x :cdr y))
    (cons x y)))

(defun extended-equal (x y)
  (declare (ignorable x y))
  (error "Implement me"))

(defun my-equal (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (extended-equal x y)
    (equal x y)))


(defun my-len-aux (x acc)
  (if (my-consp x)
      (my-len-aux (my-cdr x) (+ 1 acc))
    acc))

(defun my-len (x)
  (my-len-aux x 0))

(declaim (inline my-make-array
                 my-array-nth
                 my-array-length))

(defun my-make-array (x)
  (let* ((length   (my-len x))
         (contents (if (>= length most-positive-fixnum)
                       (error "make-array: list is too long.")
                     (make-array length))))
    (loop for i fixnum from 1 to length 
          do
          (setf (svref (the simple-vector contents) 
                       (the fixnum i))
                (my-car x))
          (setf x (my-cdr x)))
    (make-eobj :type :earray
               :contents contents)))

(defun my-array-nth (x n)
  (if (and (eobj-p x)
           (eq (eobj-type x) :earray))
      (aref (eobj-contents x) n)
    (let ((type (if (eobj-p x) (eobj-type x) (type-of x))))
      (error "Tried to array-nth ~a.~%" type))))

(defun my-array-length (x)
  (if (and (eobj-p x)
           (eq (eobj-type x) :earray))
      (length (eobj-contents x))
    (let ((type (if (eobj-p x) (eobj-type x) (type-of x))))
      (error "Tried to array-length ~a.~%" type))))




; Investigation into performance impact of these wrappers

(defconstant big-list
  (loop for i from 1 to 100000 collect i))

(defconstant big-array
  (my-make-array big-list))

(defun ordinary-len-aux (x acc)
  (if (consp x)
      (ordinary-len-aux (cdr x) (+ 1 acc))
    acc))

(defun ordinary-len (x)
  (ordinary-len-aux x 0))

(defun ordinary-true-listp (x)
  (if (consp x)
      (ordinary-true-listp (cdr x))
    (not x)))

(defun ordinary-memberp (a x)
  (if (consp x)
      (or (equal a (car x))
          (ordinary-memberp a (cdr x)))
    nil))

(defun my-true-listp (x)
  (if (my-consp x)
      (my-true-listp (cdr x))
    (my-equal x nil)))

(defun my-memberp (a x)
  (if (my-consp x)
      (or (my-equal a (my-car x))
          (my-memberp a (my-cdr x)))
    nil))


; Basic examples of execution.

(ordinary-len big-list)     ;; 100000
(my-len big-list)           ;; 100000
(my-len big-array)          ;; discipline failure
(my-array-length big-array) ;; 100000

; It's okay to cons arrays into other objects

(let* ((x (cons big-array big-list))
       (y (my-cdr x)))
  (equal big-list y))



; LENGTH OF A LIST.
;
; ordinary-len:    12.235 seconds on lhug-3 
; my-len:          12.381 seconds on lhug-3 
; slowdown:        1.19%
(progn 
  (time (loop for i fixnum from 1 to 10000 do (ordinary-len big-list)))
  (time (loop for i fixnum from 1 to 10000 do (my-len big-list))))



; TRUE-LISTP OF A LIST.
;
; ordinary-true-listp:  11.411 seconds on lhug-3
; my-true-listp:        11.675 seconds on lhug-3
; slowdown:             2.31%
(progn 
  (time (loop for i fixnum from 1 to 10000 do (ordinary-true-listp big-list)))
  (time (loop for i fixnum from 1 to 10000 do (my-true-listp big-list))))



; MEMBERP OF A LIST.  Unfortunately, my-equal performs badly.
;
; ordinary-memberp: 12.815 seconds on lhug-3
; my-memberp:       30.403 seconds on lhug-3
; slowdown:         2.37x
(progn
  (time (loop for i from 1 to 10000 do (ordinary-memberp 0 big-list)))
  (time (loop for i from 1 to 10000 do (my-memberp 0 big-list))))

; Of course, the system could provide an optimized version of memberp.

(defun system-memberp (a x)
  (typecase x 
   (cons (ordinary-memberp a x))
   (eobj (my-memberp a x))
   (t    nil)))

; ordinary-memberp: 12.815 seconds on lhug-3 (above)
; system-memberp:   12.567 seconds on lhug-3 
; speedup:          1.9% (noise)
(time (loop for i from 1 to 10000 do (system-memberp 0 big-list)))


       


; NTH using arrays (no surprises here)
;
; list nth:  5.55 seconds
; array nth: .015 seconds
; speedup:   370x 

(time (loop for i fixnum from 1 to 1000 do
            (loop for n fixnum from 1 to 1000 
                  do
                  (nth n big-list))))

(time (loop for i fixnum from 1 to 1000
            do
            (loop for n fixnum from 1 to 1000 
                  do
                  (my-array-nth big-array n))))
