; Efficient Animation of ACL2 Models.

(ccl::egc nil)
(declaim (optimize (speed 3) (safety 0) (space 0)))

; This file is largely the same as animation-1.lisp, but we add some
; destructively modifiable arrays using versions.
;
; An :ECONS is as before, a cons of extended objects.
; An :EARRAY is as before, a read-only array.
; An :EDARRAYPTR is a pointer to a destructible array.
; 

(defstruct eobj
  (type)
  (contents))
       
(defstruct econs 
  (car)
  (cdr))

(defstruct edarray
  (version)
  (contents))

(defstruct edarrayptr
  (version)
  (contents))



(declaim (inline my-car 
                 my-cdr
                 my-consp
                 my-cons
                 my-equal))

(defun my-car (x)
  (typecase x 
   (cons
    (car x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-car (eobj-contents x)))
      (otherwise (error "Tried to take car of ~a.~%" (eobj-type x)))))
   (t 
    nil)))

(defun my-cdr (x)
  (typecase x 
   (cons
    (cdr x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-cdr (eobj-contents x)))
      (otherwise (error "Tried to take cdr of ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-consp (x)
  (typecase x
   (cons
    t)
   (eobj
    (case (eobj-type x)
      (:econs t)
      (otherwise (error "Tried to consp ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-cons (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (make-eobj :type :econs
                 :contents (make-econs :car x :cdr y))
    (cons x y)))

(defun extended-equal (x y)
  (declare (ignorable x y))
  (error "Implement me"))

(defun my-equal (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (extended-equal x y)
    (equal x y)))


(defun my-len-aux (x acc)
  (if (my-consp x)
      (my-len-aux (my-cdr x) (+ 1 acc))
    acc))

(defun my-len (x)
  (my-len-aux x 0))

(declaim (inline my-make-array
                 my-array-nth
                 my-array-length))

(defun my-make-array (x destructivep)
  (let* ((length   (my-len x))
         (contents (if (>= length most-positive-fixnum)
                       (error "make-array: list is too long.")
                     (make-array length))))
    (loop for i fixnum from 1 to length 
          do
          (setf (svref (the simple-vector contents) 
                       (the fixnum i))
                (my-car x))
          (setf x (my-cdr x)))
    (if destructivep
        (let* ((edarray (make-edarray :version 0
                                      :contents contents))
               (edarrayptr (make-edarrayptr :version 0
                                            :contents edarray)))
          (make-eobj :type :edarrayptr
                     :contents edarrayptr))
      (make-eobj :type :earray
                 :contents contents))))

(defun my-array-nth (x n)
  (cond ((not (eobj-p x))
         (error "Tried to array-nth ~a.~%" (type-of x)))
        ((eq (eobj-type x) :earray)
         (aref (eobj-contents x) n))
        ((eq (eobj-type x) :edarrayptr)
         (let* ((ptr     (eobj-contents x))
                (ptr-ver (edarrayptr-version ptr))
                (edarray (edarrayptr-contents ptr))
                (arr-ver (edarray-version edarray)))
           (if (equal ptr-ver arr-ver)
               (aref (edarray-contents edarray) n)
             (error "Tried to array-nth old array.  Ptr version = ~a, Array ver = ~a.~%"
                    ptr-ver arr-ver))))
        (t
         (error "Tried to array-nth ~a.~%" (eobj-type x)))))

(defun my-array-length (x)
  (cond ((not (eobj-p x))
         (error "Tried to array-length ~a.~%" (type-of x)))
        ((eq (eobj-type x) :earray)
         (length (eobj-contents x)))
        ((eq (eobj-type x) :edarrayptr)
         (let* ((ptr     (eobj-contents x))
                (edarray (edarrayptr-contents ptr)))
           (length (edarray-contents edarray))))
        (t
         (error "Tried to array-length ~a.~%" (eobj-type x)))))

(defun my-array-update-nth (x n val)
  (cond ((not (eobj-p x))
         (error "Tried to array-length ~a.~%" (type-of x)))
        ((eq (eobj-type x) :edarrayptr)
         (let* ((ptr     (eobj-contents x))
                (ptr-ver (edarrayptr-version ptr))
                (edarray (edarrayptr-contents ptr))
                (arr-ver (edarrayptr-version edarray)))
           (if (equal ptr-ver arr-ver)
               (progn
                 (incf (edarray-version edarray))
                 (setf (aref (edarray-contents edarray) n) val)
                 (make-eobj :type :edarrayptr
                            :contents (make-edarrayptr :version (+ 1 arr-ver)
                                                       :contents edarray)))
             (error "Array ptr out of date.  Ptr version = ~a, Array ver = ~a.~%"
                    ptr-ver arr-ver))))
        (t
         (error "Tried to array-update-nth ~a.~%" (eobj-type x)))))


; Investigation into performance impact of these wrappers

(defconstant big-list
  (loop for i from 1 to 100000 collect i))

(defconstant big-array
  (my-make-array big-list nil))

; Basic examples of execution.

(ordinary-len big-list)      ;; 100000
(my-len big-list)            ;; 100000
;(my-len big-array)           ;; discipline failure
(my-array-length big-array)  ;; 100000
(my-array-nth big-array 100) ;; 100

(defconstant big-array-2
  (my-make-array big-list t))

;(my-len big-array-2)           ;; discipline failure
(my-array-length big-array-2)  ;; 100000
(my-array-nth big-array-2 100) ;; 100

(defparameter *new-array* (my-array-update-nth big-array-2 100 50))

; (my-array-nth big-array-2 100) ;; discipline error, old array
(my-array-nth *new-array* 100) ;; 50


; Array reads.  
;
; big-array: 1.73 seconds
; new-array: 2.69 seconds
; overhead for pointer chasing: 55%

(time (loop for i fixnum from 1 to 100000
            do 
            (loop for n fixnum from 1 to 1000 
                  do
                  (my-array-nth big-array n))))

(time (loop for i fixnum from 1 to 100000
            do 
            (loop for n fixnum from 1 to 1000 
                  do
                  (my-array-nth *new-array* n))))


(defparameter *big-arr* (my-make-array big-list t))

(defun zero-out-array (arr n)
  (if (= n 0)
      arr
    (zero-out-array (my-array-update-nth arr n 0)
                    (- n 1))))

;; Zero out an array while updating the pointers all over

;; 3.18 seconds, 640 MB allocated
(time (loop for i from 1 to 10000
            do
            (setf *big-arr* (zero-out-array *big-arr* 1000))))


(defparameter *raw-vector* (make-array 100000))

;; Zero out an array in raw lisp
;; .15 seconds, 
(time (loop for i from 1 to 10000
            do
            (loop for n from 1 to 1000
                  do
                  (setf (aref *raw-vector* n) 0))))




