; Efficient Animation of ACL2 Models.

(ccl::egc nil)
(declaim (optimize (speed 3) (safety 0) (space 0)))

; This file is like animation-2.lisp, but we introduce a bottom
; object to alter how arrays are updated.
;
; An :ECONS is as before, a cons of extended objects.
; An :EARRAY is as readable and writable array.
; An :EBOTTOM is a bad object which is no longer valid.
; 

(defstruct eobj
  (type)
  (contents))
       
(defstruct econs 
  (car)
  (cdr))

(declaim (inline my-car 
                 my-cdr
                 my-consp
                 my-cons
                 my-equal))

(defun my-car (x)
  (typecase x 
   (cons
    (car x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-car (eobj-contents x)))
      (otherwise (error "Tried to take car of ~a.~%" (eobj-type x)))))
   (t 
    nil)))

(defun my-cdr (x)
  (typecase x 
   (cons
    (cdr x))
   (eobj
    (case (eobj-type x)
      (:econs (econs-cdr (eobj-contents x)))
      (otherwise (error "Tried to take cdr of ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-consp (x)
  (typecase x
   (cons
    t)
   (eobj
    (case (eobj-type x)
      (:econs t)
      (otherwise (error "Tried to consp ~a.~%" (eobj-type x)))))
   (t
    nil)))

(defun my-cons (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (make-eobj :type :econs
                 :contents (make-econs :car x :cdr y))
    (cons x y)))

(defun extended-equal (x y)
  (declare (ignorable x y))
  (error "Implement me"))

(defun my-equal (x y)
  (if (or (eobj-p x)
          (eobj-p y))
      (extended-equal x y)
    (equal x y)))


(defun my-len-aux (x acc)
  (if (my-consp x)
      (my-len-aux (my-cdr x) (+ 1 acc))
    acc))

(defun my-len (x)
  (my-len-aux x 0))

(declaim (inline my-make-array
                 my-array-nth
                 my-array-update-nth
                 my-array-length))

(defun my-make-array (x)
  (let* ((length   (my-len x))
         (contents (if (>= length most-positive-fixnum)
                       (error "make-array: list is too long.")
                     (make-array length))))
    (loop for i fixnum from 1 to length 
          do
          (setf (svref (the simple-vector contents) 
                       (the fixnum i))
                (my-car x))
          (setf x (my-cdr x)))
    (make-eobj :type :earray
               :contents contents)))

(defun my-array-nth (x n)
  (cond ((not (eobj-p x))
         (error "Tried to array-nth ~a.~%" (type-of x)))
        ((eq (eobj-type x) :earray)
         (aref (eobj-contents x) n))
        (t
         (error "Tried to array-nth ~a.~%" (eobj-type x)))))

(defun my-array-length (x)
  (cond ((not (eobj-p x))
         (error "Tried to array-length ~a.~%" (type-of x)))
        ((eq (eobj-type x) :earray)
         (length (eobj-contents x)))
        (t
         (error "Tried to array-length ~a.~%" (eobj-type x)))))

(defun my-array-update-nth (x n val)
  (cond ((not (eobj-p x))
         (error "Tried to array-length ~a.~%" (type-of x)))
        ((eq (eobj-type x) :earray)
         (progn
           ;; Invalidate the old object
           (setf (eobj-type x) :ebottom)
           ;; Update the array
           (setf (aref (eobj-contents x) n) val)
           ;: Return a new pointer to it.
           (make-eobj :type :earray
                      :contents (eobj-contents x))))
        (t
         (error "Tried to array-update-nth ~a.~%" (eobj-type x)))))


(defconstant big-list
  (loop for i from 1 to 100000 collect i))



(defparameter *arr* (my-make-array big-list))

(my-array-length *arr*)  ;; 100000
(my-array-nth *arr* 100) ;; 100

(defparameter *arr2* (my-array-update-nth *arr* 100 50))
(eobj-type *arr2*) ;; EARRAY
(eobj-type *arr*)  ;; EBOTTOM

; (my-array-nth *arr* 100) ;; Tried to array-nth BOTTOM!
(my-array-nth *arr2* 100)  ;; 50


; Array Reads.  

; my-array-nth: 1.740 seconds
; raw aref:      .228 seconds
; slowdown:     7.63x 

(time (loop for i fixnum from 1 to 100000
            do 
            (loop for n fixnum from 1 to 1000 
                  do
                  (my-array-nth *arr2* n))))

(defparameter *raw-array* (eobj-contents *arr2*))

(time (loop for i fixnum from 1 to 100000
            do 
            (loop for n fixnum from 1 to 1000 
                  do
                  (aref *raw-array* n))))


; Array Writes

(defun zero-out-array (arr n)
  (if (= n 0)
      arr
    (zero-out-array (my-array-update-nth arr n 0)
                    (- n 1))))

; zero-out-array: 1.62 seconds, 320 MB allocated
; raw setf aref:   .15 seconds
; slowdown:       10.7x
(time (loop for i from 1 to 10000
            do
            (setf *arr2* (zero-out-array *arr2* 1000))))

(time (loop for i from 1 to 10000
            do
            (loop for n from 1 to 1000
                  do
                  (setf (aref *raw-array* n) 0))))



; ACL2 stobjs seem to give the same performance as raw setf.

(defstobj foo 
  (bar :type (array t (100000))))

(defun zero-out-array (foo n)
  (declare (xargs :mode :program :stobjs foo))
  (if (= n 0)
      foo
    (let ((foo (update-bari n 0 foo)))
      (zero-out-array foo (- n 1)))))

:q

;; .137 seconds, no garbage
(time (loop for i from 1 to 10000
            do
            (zero-out-array *the-live-foo* 1000)))
