# net.tcl --
#

########################## Modifiable variables ###########################
# Network name
set network net0
# Limit of simulation in ms
set tend 100
# Set tempo of simulation (0 is for fastest, 20 is leisurely, 100 is molasses)
set tempo 0
# Name of log file
set outfile "simuout"
# Interval for plotting potentials - increase to accommodate more events in 
# TracePlot if too many start occurring at the same place
set plotinterval 8
# Width of Main window
set mainWinWidth 600
# Height of Main window
set mainWinHeight 600

########################## Static variables ###########################
# Modify at your own risk

set inpfil ""
set overlap 2.0
set runstop RUN
set refract -90
set spike 70
set trw 500
set trh 250
set timestep 0
set topmargin 10
set bottommargin 50
set boxdim 12
set tcl_precision 4
set numNeurons 0
set routine "run"
set curneuron -1

########################## Display Definition ###########################
set w .net
#catch {destroy $w}
#toplevel $w
#wm title $w "Some Neurons"
#wm iconname $w "Net"
#positionWindow $w
set canvas $w.frame.c

frame $w
frame $w.menu -relief ridge -bd 2
button $w.menu.run -textvar runstop -command Run
button $w.menu.step -text STEP -command Step
button $w.menu.new -text NEW -command "$network new"
button $w.menu.init -text INIT -command "$network init"
button $w.menu.reset -text RESET -command Reset
button $w.menu.quit -text QUIT -command exit
label $w.menu.label -relief ridge -text "Simulation file:"
entry $w.menu.file -relief ridge -width 20 -textvar inpfil
pack $w.menu.run $w.menu.step $w.menu.new $w.menu.init $w.menu.reset \
    $w.menu.quit $w.menu.label $w.menu.file -side left
pack $w.menu -side top -pady 3

frame $w.frame
pack $w.frame -fill both -expand yes

canvas $canvas -width $mainWinWidth -height $mainWinHeight -relief sunken -bd 2
pack $canvas

scale $w.scale -relief ridge -from 0 -to 100 -length $mainWinWidth \
    -variable tempo -orient horizontal -label Tempo -tickinterval 5
pack $w.scale

frame $w.colors -relief ridge
for {set i -90} {$i <= 0} {incr i 10} {
    set name [expr ($i + 90) / 10]
    label $w.colors.$name -text "$i mV" -width 7
    pack $w.colors.$name -side left
}
label $w.colors.spike -text "70 mV" -width 7
pack $w.colors.spike -side left
pack $w.colors

frame $w.stats -relief ridge
frame $w.stats.time -relief groove
label $w.stats.time.lab -width 6 -text "Time:"
label $w.stats.time.step -width 4 -textvar timestep -anchor e
label $w.stats.time.ms1 -width 4 -text "ms"
label $w.stats.time.tendlab -width 6 -text "out of"
label $w.stats.time.tend -width 4 -textvar tend -anchor e
label $w.stats.time.ms2 -width 4 -text "ms"
pack $w.stats.time.lab $w.stats.time.step $w.stats.time.ms1 \
    $w.stats.time.tendlab $w.stats.time.tend $w.stats.time.ms2 -side left
pack $w.stats.time -side left
frame $w.stats.neuron -relief groove
label $w.stats.neuron.nlab -width 6 -text "Neuron:"
label $w.stats.neuron.idx -width 4 -textvar curneuron
pack $w.stats.neuron.nlab $w.stats.neuron.idx -side left 
pack $w.stats.neuron -side right
pack $w.stats -expand true -fill x

pack $w

########################## Net Procedures ###########################

proc Reset {} {
    global canvas network numNeurons routine timestep curneuron

    $network init
    set routine "run"
    for {set i 0} {$i < $numNeurons} {incr i} {
	TraceInit $i
	Display $i
    }
    set timestep 0
}

proc Display {idx} {
    global canvas tempo

    set idx [n$idx id]
    $canvas move $idx 0 0
    update
    after $tempo
}

proc Run {} {
    global runstop

    if {$runstop == "STOP"} {
	set runstop "RUN"
    } else {
	set runstop "STOP"
	set routine "run"
	while {($runstop != "RUN")} {
	    Step
	}
    }
}

proc Step {} {
    global routine canvas network runstop spike refract time \
	curneuron numNeurons timestep

    if {$routine == "run"} {
	if {[$network run]} {
	    set routine "current"
	    Step
	} else {
	    set runstop RUN
	    for {set idx 0} {$idx < $numNeurons} {incr idx} {
		.trace$idx.frame.c xview moveto 1
		Display $idx
	    }
	}
    } elseif {$routine == "current"} {
	if {[$network current]} {
	    set timestep [$network timestep]
	    set routine "active"
	    Step
	} else {
	    set routine "genspikes"
	    Step
	}
    } elseif {$routine == "active"} {
	if {[set cond [$network active]] != -1} {
	    set curneuron [lindex $cond 0]
	    set sidx [lindex $cond 1]
	    SynInput $curneuron $sidx
	    set routine "decaypot"
	} else {
	    set routine "current"
	}
    } elseif {$routine == "decaypot"} {
	set potential [$network decaypot]
	TracePlot $curneuron $potential
	Display $curneuron
	set routine "integratepsp"
    } elseif {$routine == "integratepsp"} {
	set potential [$network integratepsp]
	TracePlot $curneuron $potential
	Display $curneuron
	set routine "current"
    } elseif {$routine == "genspikes"} {
	if {[$network genspikes]} {
	    TracePlot $curneuron $spike
	    SynInput $curneuron -1
	    Display $curneuron
	    set routine "hyperpolarize"
	} else {
	    set routine "run"
	}
    } elseif {$routine == "hyperpolarize"} {
	$network hyperpolarize
	TracePlot $curneuron $refract
	Display $curneuron
	set routine "run"
    }
}

# Condense steps above into a cycle - not interfaced
proc Cycle {} {
    global canvas network runstop spike refract timestep

    if {[$network run]} {
	while {[$network current]} {
	    set timestep [$network timestep]
	    if {[set cond [$network active]] != -1} {
		set nidx [lindex $cond 0]
		set sidx [lindex $cond 1]
		SynInput $nidx $sidx
		set potential [$network decaypot]
		TracePlot $nidx $potential
		Display $nidx
		set potential [$network integratepsp]
		TracePlot $nidx $potential
		Display $nidx
	    }
	}
	if {[$network genspikes]} {
	    TracePlot $nidx $spike
	    Display $nidx
	    $network hyperpolarize
	    TracePlot $nidx $refract
	    Display $nidx
	}
    } else {
	set runstop RUN
	$canvas itemconf neuron -fill red
    }
}

proc positionWindow {w} {
    wm geometry $w +300+300
}

proc TracePotent {poten} {
    global trh spike refract topmargin bottommargin

    set frac [expr ($poten - $refract) / ($spike - $refract)]
    return [expr ($trh - $bottommargin) * (1 - $frac) + $topmargin]
}

proc TracePlot {idx poten} {
    global timestep lasttime lastplot plotinterval width

    set tmp [.trace$idx.frame.hscroll get]
    set first [lindex $tmp 0]
    set last [lindex $tmp 1]
    set first [expr $first * $width]
    set last [expr $last * $width]

    set x $lastplot(x,$idx)
    if {$timestep != $lasttime($idx)} {
	set itv [expr $timestep * $plotinterval]
	set lasttime($idx) $timestep
	set y $lastplot(y,$idx)
	.trace$idx.frame.c create line $x $y $itv $y -fill blue -tag plot$idx
	set x $itv
	set lastplot(x,$idx) $x
    } else {
	incr lastplot(x,$idx)
    }
    # Move scrollbar as graph goes off right end
    if {$lastplot(x,$idx) > $last} {
	set first [expr ($first - $last + $lastplot(x,$idx)) / $width]
	.trace$idx.frame.c xview moveto $first
	
    }
    set y $lastplot(y,$idx)
    set lastplot(y,$idx) [TracePotent $poten]
    .trace$idx.frame.c create line $x $y $lastplot(x,$idx) $lastplot(y,$idx) \
	-fill blue -tag plot$idx
}

proc TraceInit {id} {
    global tend plotinterval trh trw lastplot width topmargin bottommargin

    set w .trace$id
    set can $w.frame.c

    # Refractory Potential Line
    set y [TracePotent -90.0]
    $can create line 0 $y $width $y -fill red

    # Rest Potential Line
    set y [TracePotent -70.0]
    $can create line 0 $y $width $y -fill yellow

    # Firing Threshold Line
    set y [TracePotent -50.0]
    $can create line 0 $y $width $y -fill green

    # Spike Line
    set y [TracePotent 70.0]
    $can create line 0 $y $width $y -fill white

    set y [expr $trh - $topmargin]
    for {set i 0} {$i < $tend} {incr i 5} {
	set x [expr $plotinterval * $i]
	$can create text $x $y -fill orange -text $i
    }

    set first [expr 3.0 / $width]
    set last [expr (3.0 + $trw) / $width]
    $w.frame.c xview moveto 0
    # Figure roughly $plotinterval events per millisecond
    set lastplot(x,$id) 0
    set lastplot(y,$id) [expr $trh - $bottommargin]
    $can delete plot$id
}

proc Trace {id} {
    global tend trw trh width topmargin plotinterval

    set w .trace$id
    catch {destroy $w}
    toplevel $w
    wm title $w "N$id trace"
    set can $w.frame.c

    frame $w.frame
    pack $w.frame -fill both -expand yes

    set width [expr $tend * $plotinterval]

    canvas $can -width $trw -height $trh -scrollregion "0 0 $width $trh" \
	 -relief sunken -bg black -bd 2 -xscrollcommand "$w.frame.hscroll set"
    scrollbar $w.frame.hscroll -orient horiz -command [list $can xview]
    pack $w.frame.hscroll -side bottom -fill x
    pack $can

    TraceInit $id
}

proc DeleteNeuron {idx} {
    global canvas

    $canvas delete neuron$idx
    set from [$canvas find withtag fromNeuron$idx]
    set to [$canvas find withtag toNeuron$idx]
    set ids [concat $from $to]
    foreach id $ids {
	$canvas delete $id
    }
}

proc Neuron {idx numin {a 0.0} {b 0.0}} {
    global canvas overlap network numNeurons nConn lasttime

    # Define a new neuron widget and register it with network
    if {$a == 0.0 && $b == 0.0} {
	set a [expr $a + $overlap]
	set b [expr $b + $overlap]
	set overlap [expr $overlap + 1.0]
    }
    set circle [expr 8.0*atan(1.0)]
    set inner 20
    set outer 50
    set mid 42
    set points ""
    set nump [expr 2*($numin+1)]
    set color blue
    for {set i 0} {$i < $nump} {incr i} {
	if {$i % 2 == 0} {
	    # Synapse -1 is axon
	    set id [expr ($i - 2) / 2]
	    set x [expr $a + $outer*cos($circle*$i/$nump)]
	    set y [expr $b + $outer*sin($circle*$i/$nump)]
	    set u [expr $a + $mid*cos($circle*$i/$nump)]
	    set v [expr $b + $mid*sin($circle*$i/$nump)]
	    if {$id == -1} {     # axon is longer
		set x [expr $x + 100]
		set u [expr $u + 75]
	    }
	    set x1 [expr $u - 4]
	    set x2 [expr $u + 4]
	    set y1 [expr $v - 4]
	    set y2 [expr $v + 4]
	    set oval [$canvas create oval $x1 $y1 $x2 $y2 -fill $color \
			  -tag "neuron$idx synapse$id n${idx}s$id"]
	    set tags [$canvas gettags $oval]
	    set color red
	} else {
	    set x [expr $a + $inner*cos($circle*$i/$nump)]
	    set y [expr $b + $inner*sin($circle*$i/$nump)]
	}
	lappend points $x $y
    }
    set id [eval {$canvas create polygon} $points \
		{-smooth true -tag "neuron neuron$idx"}]

    # Create a neuron
    neuron n$idx -id $id -tag neuron$idx
    # Register it with the network
    $network conf -neuron$idx n$idx
    incr numNeurons
    set nConn($idx) 0
    set lasttime($idx) -1

    $canvas bind neuron$idx <1> "itemStartDrag %x %y"
    $canvas bind neuron$idx <B1-Motion> "itemDrag %x %y $idx"

    Display $idx
    # Create Graph to track neuron's action potentials
    Trace $idx

    return n$idx
}

# Utility procedures for synapses/axons.

$canvas bind current <2> "ConnOrSyn %x %y"
$canvas bind current <3> "StartConn %x %y"
$canvas bind current <B3-Motion> "DrawConn %x %y"
$canvas bind current <ButtonRelease-3> "ConnEnd %x %y"

proc ConnOrSyn {x y} {
    global canvas

    set id [GetSynId $x $y]
    if {$id == ""} {
	set cid [$canvas find closest $x $y]
	if {[$canvas type $cid] == "line"} {
	    set tags [$canvas gettags $cid]
	    set tag [lindex $tags [lsearch $tags fromNeuron*]]
	    scan $tag "fromNeuron%d" nidx
	    set tag [lindex $tags [lsearch $tags conn*]]
	    scan $tag "conn%d" conn
	    Connection $nidx $conn
	}
    } else {
	Synapse $id
    }
}

proc CommandEntry {name label width state} {
    global value

    frame $name -relief groove
    label $name.label -width $width -anchor w -text $label
    entry $name.value -relief sunken -state $state -textvar value($label)
    pack $name.label -side left
    pack $name.value -side right -fill x -expand true
    return $name.value
}

proc GetSynId {x y} {
    global canvas boxdim

    # Return enclosed synapse
    set x1 [expr $x - $boxdim]
    set x2 [expr $x + $boxdim]
    set y1 [expr $y - $boxdim]
    set y2 [expr $y + $boxdim]
    set id [$canvas find enclosed $x1 $y1 $x2 $y2]
    return $id
}

proc NeuronSynapse {id} {
    global canvas

    set tags [$canvas gettags $id]
    set tag [lindex $tags [lsearch $tags neuron*]]
    scan $tag "neuron%d" nidx
    set tag [lindex $tags [lsearch $tags synapse*]]
    scan $tag "synapse%d" sidx
    return "$nidx $sidx"
}

proc SynCoords {id} {
    global canvas

    # Return center of Synapse
    set coords [$canvas coords $id]
    set x1 [lindex $coords 0]
    set x2 [lindex $coords 2]
    set y1 [lindex $coords 1]
    set y2 [lindex $coords 3]
    set x [expr ($x1 + $x2) / 2.0]
    set y [expr ($y1 + $y2) / 2.0]
    return "$x $y"
}

proc SynInput {nidx sidx} {
    global canvas

    set id [$canvas find withtag n${nidx}s$sidx]
    set coords [SynCoords $id]
    set x [lindex $coords 0]
    set y [lindex $coords 1]
    set x1 [expr $x - 10]
    set x2 [expr $x + 10]
    set y1 [expr $y - 10]
    set y2 [expr $y + 10]
    set oval [$canvas create oval $x1 $y1 $x2 $y2 -fill white]
    Display $nidx
    $canvas delete $oval
}

proc Synapse {id} {
    global ok value

    set ns [NeuronSynapse $id]
    set value(nidx) [lindex $ns 0]
    set value(sidx) [lindex $ns 1]
    set values [n$value(nidx) synapse $value(sidx)]
    if {$values == -1} {
	set value(efficacy) ""
	set value(distance) ""
	set value(first) ""
	set value(interval) ""
	set value(connected) ""
    } else {
	set value(efficacy) [lindex $values 0]
	set value(distance) [lindex $values 1]
	set value(first) [lindex $values 2]
	set value(interval) [lindex $values 3]
	set value(connected) [lindex $values 4]
    }

    set w .dialog
    toplevel $w -borderwidth 2
    wm title $w "Synapse Dialog"
    wm iconname $w "synapse"

    frame $w.ent
    CommandEntry $w.ent.neuron nidx 10 disabled
    CommandEntry $w.ent.synapse sidx 10 disabled
    CommandEntry $w.ent.eff efficacy 10 normal
    CommandEntry $w.ent.dist distance 10 normal
    CommandEntry $w.ent.first first 10 normal
    CommandEntry $w.ent.inter interval 10 normal
    CommandEntry $w.ent.conn connected 10 disabled
    pack $w.ent.neuron $w.ent.synapse $w.ent.eff $w.ent.dist $w.ent.first \
	$w.ent.inter $w.ent.conn 
    pack $w.ent

    frame $w.buttons -relief groove
    button $w.buttons.ok -text OK -command {set ok 1}
    button $w.buttons.no -text CANCEL -command {set ok 0}
    pack $w.buttons.ok $w.buttons.no -side left
    pack $w.buttons

    bind $w <Return> {set ok 1}
    bind $w <Control-c> {set ok 0}
    focus $w.ent.eff.value
    tkwait variable ok
    destroy $w
    if {$ok} {
	n$value(nidx) synapse $value(sidx) $value(efficacy) \
	    $value(distance) $value(first) $value(interval)
    }
}

proc Connection {neuron conn {nidx ""} {sidx ""}} {
    global ok value

    set value(neuron) $neuron
    set value(conn) $conn
    set values [n$neuron outcon $conn]
    if {$values == -1} {
	set value(nidx) $nidx
	set value(sidx) $sidx
	set value(delay) ""
    } else {
	set value(nidx) [lindex $values 0]
	set value(sidx) [lindex $values 1]
	set value(delay) [lindex $values 2]
    }

    set w .dialog
    toplevel $w -borderwidth 2
    wm title $w "Connection Dialog"
    wm iconname $w "connection"

    frame $w.ent
    CommandEntry $w.ent.neuron neuron 10 disabled
    CommandEntry $w.ent.conn conn 10 disabled
    CommandEntry $w.ent.nidx nidx 10 disabled
    CommandEntry $w.ent.sidx sidx 10 disabled
    CommandEntry $w.ent.delay delay 10 normal
    pack $w.ent.neuron $w.ent.conn $w.ent.nidx $w.ent.sidx $w.ent.delay
    pack $w.ent

    frame $w.buttons -relief groove
    button $w.buttons.ok -text OK -command {set ok 1}
    button $w.buttons.no -text CANCEL -command {set ok 0}
    pack $w.buttons.ok $w.buttons.no -side left
    pack $w.buttons

    bind $w <Return> {set ok 1}
    bind $w <Control-c> {set ok 0}
    focus $w.ent.nidx.value
    tkwait variable ok
    destroy $w
    if {$ok} {
	n$neuron outcon $conn $value(nidx) $value(sidx) $value(delay)
    }
}

proc MakeConnections {} {
    global canvas numNeurons nConn

    for {set idx 0} {$idx < $numNeurons} {incr idx} {
	set nout [n$idx noutcon]
	for {set i 0} {$i < $nout} {incr i} {
	    set query [n$idx outcon $i]
	    set toNeuron [lindex $query 0]
	    set toSynapse [lindex $query 1]
	    set fromId [$canvas find withtag n${idx}s-1]
	    set toId [$canvas find withtag n${toNeuron}s$toSynapse]
	    Connect $fromId $toId $nConn($idx)
	    incr nConn($idx)
	}
    }
}

proc DrawConn {x y} {
    global canvas axonX axonY

    set x [$canvas canvasx $x]
    set y [$canvas canvasy $y]
#    $c delete conn
    $canvas create line $axonX $axonY $x $y -width 1 -fill green -tag conn
}

proc Connect {fromId toId {conn 0}} {
    global canvas

    set coords [SynCoords $fromId]
    set fromNeuron [lindex [NeuronSynapse $fromId] 0]
    set x1 [lindex $coords 0]
    set y1 [lindex $coords 1]
    set coords [SynCoords $toId]
    set toNeuron [lindex [NeuronSynapse $toId] 0]
    set x2 [lindex $coords 0]
    set y2 [lindex $coords 1]
    set lineId [$canvas create line $x1 $y1 $x2 $y2 -width 2 -fill red]
    $canvas addtag fromSyn$fromId withtag $lineId
    $canvas addtag toSyn$toId withtag $lineId
    $canvas addtag fromNeuron$fromNeuron withtag $lineId
    $canvas addtag toNeuron$toNeuron withtag $lineId
    $canvas addtag conn$conn withtag $lineId
    return $lineId
}

proc StartConn {x y} {
    global axonX axonY axonIdx fromId

    set id [GetSynId $x $y]
    if {$id == ""} {
	return
    }
    set fromId $id
    set conn [NeuronSynapse $id]
    set axonIdx [lindex $conn 0]
    set sidx [lindex $conn 1]
    if {$sidx != -1} {
	puts "Start from the axon"
	return
    }
    set conn [SynCoords $id]
    set axonX [lindex $conn 0]
    set axonY [lindex $conn 1]
}

proc ConnEnd {x y} {
    global canvas axonX axonY axonIdx fromId nConn

    # To-Neuron values
    $canvas delete conn
    set id [GetSynId $x $y]
    if {$id == ""} {
	return
    }
    set toId $id
    set conn [NeuronSynapse $id]
    set nidx [lindex $conn 0]
    set sidx [lindex $conn 1]
    if {$sidx == -1} {
	puts "End on a synapse"
	return
    }
    set conn [SynCoords $id]
    set x [lindex $conn 0]
    set y [lindex $conn 1]
    Connect $fromId $toId nConn($axonIdx)
    Connection $axonIdx $nConn($axonIdx) $nidx $sidx
    incr nConn($axonIdx)
}

proc itemStartDrag {x y} {
    global canvas lastX lastY

    set lastX [$canvas canvasx $x]
    set lastY [$canvas canvasy $y]
}

proc itemDrag {x y idx} {
    global canvas lastX lastY

    set x [$canvas canvasx $x]
    set y [$canvas canvasy $y]
    $canvas move neuron$idx [expr $x-$lastX] [expr $y-$lastY]
    set from [$canvas find withtag fromNeuron$idx]
    set to [$canvas find withtag toNeuron$idx]
    set ids [concat $from $to]
    foreach id $ids {
	set tags [$canvas gettags $id]
	set tag [lindex $tags [lsearch $tags fromSyn*]]
	scan $tag "fromSyn%d" fromId
	set tag [lindex $tags [lsearch $tags toSyn*]]
	scan $tag "toSyn%d" toId
	set tag [lindex $tags [lsearch $tags conn*]]
	scan $tag "conn%d" conn
	Connect $fromId $toId $conn
	$canvas delete $id
    }
    set lastX $x
    set lastY $y
}

########################## Define Net ###########################
# Change network definition here by defining new neurons, synapses, and
# connections

# New network
network $network -tend $tend -decayfact 0.5 -attenfact 0.95 \
    -outfile $outfile -canvas $canvas

# Show potential-color mapping colorbar
for {set i -90} {$i <= 0} {incr i 10} {
    set name [expr ($i + 90) / 10]
    set color [$network color $i]
    $w.colors.$name conf -bg $color -fg orange
}
set color [$network color 70]
$w.colors.spike conf -bg $color -fg orange

# Neurons, synapses, and connections to other neurons
# "Neuron" index, number of synapses, screen x coordinate, screen y coordinate
set name [Neuron $numNeurons 4 100 100]
# Define synapse values
# NeuronName "synapse" index, efficacy(mV),distance(um),first(ms),interval(ms)
$name synapse 0 20 10 4 10
$name synapse 1 -30 20 3 10
$name synapse 2 20 10 2 10
$name synapse 3 20 10 5 5
# Define synapse values
# NeuronName "outcon" index, to_neuron, to_synapse, delay(ms)
$name outcon 0 1 0 5
$name outcon 1 2 1 7

set name [Neuron $numNeurons 4 100 500]
$name synapse 0 30 10 4 10
$name synapse 1 -30 20 3 5
$name synapse 2 25 10 2 6
$name synapse 3 25 10 2 6
$name outcon 0 0 0 5
$name outcon 1 0 1 7

set name [Neuron $numNeurons 5 400 300]
$name synapse 0 20 10 4 7
$name synapse 1 -30 20 3 5
$name synapse 2 15 10 2 8
$name synapse 3 15 10 2 8
$name synapse 4 15 10 2 8
$name outcon 0 1 1 5
$name outcon 1 0 2 7

# Make connections visible
MakeConnections

# Initialize network with above settings
$network init
