#include "network.h"

/**************************** Network Commands *******************************/
int 
NetworkCmd(ClientData clientData, Tcl_Interp *interp,
	     int argc, char *argv[]) {
  Network *netPtr;
  int i;

  netPtr = (Network *) ckalloc(sizeof(Network));
  netPtr->interp = interp;
  interp->result = Tk_GetUid(argv[1]);  /* @#$970517 TK non-window stuff */
  netPtr->objCmd = Tcl_CreateCommand(interp, interp->result, NetworkObjCmd,
				     (ClientData) netPtr,  DeleteNetwork);
  netPtr->nneuron = 0;
  netPtr->timestep = 0;
  netPtr->tend = 0;
  netPtr->decayfact = 0.0;
  netPtr->attenfact = 0.0;
  for (i = 0; i < MAXNEURON; i++ )
    netPtr->neuron[i] = NULL;
  netPtr->queuelen = 0;
  netPtr->headevent = NULL;
  netPtr->prevevent = NULL;
  netPtr->thisevent = NULL;
  netPtr->curevent = NULL;
  netPtr->thisneuron = -1;
  netPtr->thistime = 0.0;
  netPtr->eventcnt = 0;

  NetworkConfigure( netPtr, argc-2, argv+2 );
  return TCL_OK;
}

int 
NetworkObjCmd(ClientData clientData, Tcl_Interp *interp,
	     int argc, char *argv[]) {
  Network *netPtr = (Network *) clientData;
  int i;
  int result = TCL_OK;
  size_t length;
  char c;

  Tcl_Preserve((ClientData) netPtr);
  c = argv[1][0];
  length = strlen(argv[1]);

  /* Create a layer for the network object */
  if ((c == 'r') && (!strncmp(argv[1], "readparms", length))
      && (length >= 4)) {
    /* Read network parameters */

    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		       " readparms File\"", (char *) NULL);
      goto error;
    }

    ReadParms( netPtr, argv[2] );

  } else if ((c == 'd') && (!strncmp(argv[1], "deleteneuron", length))
	     && (length >= 3)) {
    Neuron *nPtr;
    OutconStruct *oPtr;
    int delout, idx, j;

    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		       " deleteneuron Index\"", (char *) NULL);
      goto error;
    }
    idx = atoi( argv[2] );
    DeleteNeuron( netPtr->neuron[idx] );
    /* Fill in empty pointer position */
    for (i = idx; i < netPtr->nneuron; i++ )
      netPtr->neuron[i] = netPtr->neuron[i+1];
    netPtr->nneuron--;

    /* Remove connections to Neuron idx */
    for (i = 0; i < netPtr->nneuron; i++ ) {
      nPtr = netPtr->neuron[i];
      delout = 0;
      /* Fill in gaps */
      for( j = 0; j < nPtr->noutcon; j++ ) {
	oPtr = nPtr->outcon[j];
	if( oPtr->neuronidx == idx ) {
	  ckfree( oPtr );
	  oPtr = nPtr->outcon[j+1];
	  delout++;
	}
      }
      nPtr->noutcon -= delout;
    }

  } else if ((c == 'n') && (!strncmp(argv[1], "new", length))
	     && (length >= 3)) {

    NewNetwork( netPtr );

  } else if ((c == 'i') && (!strncmp(argv[1], "initialize", length))
	     && (length >= 4)) {

    InitNetwork( netPtr );

  } else if ((c == 'g') && (!strncmp(argv[1], "gettend", length))
	     && (length >= 4)) {

    sprintf( netPtr->interp->result, "%d", netPtr->tend);

  } else if ((c == 'q') && (!strncmp(argv[1], "queue", length))
	     && (length >= 4)) {

    printf( "Curevent:   %3d  (next %3d)\n", 
	    netPtr->curevent ? netPtr->curevent->eventidx : -1,
	    (netPtr->curevent && netPtr->curevent->nextevent) ? 
	    netPtr->curevent->nextevent->eventidx : -1 );
    printf( "Prevevent:  %d  (next %d)\n", 
	    netPtr->prevevent ? netPtr->prevevent->eventidx : -1,
	    (netPtr->prevevent && netPtr->prevevent->nextevent) ? 
	    netPtr->prevevent->nextevent->eventidx : -1 );
    printf( "Thisevent:  %d  (next %d)\n", 
	    netPtr->thisevent ? netPtr->thisevent->eventidx : -1,
	    (netPtr->thisevent && netPtr->thisevent->nextevent) ? 
	    netPtr->thisevent->nextevent->eventidx : -1 );
    print_event_queue( stderr, netPtr->headevent );

  } else if ((c == 't') && (!strncmp(argv[1], "timestep", length))
	     && (length >= 4)) {

    /* Same timeslice */
    sprintf( netPtr->interp->result, "%d", netPtr->timestep );

  } else if ((c == 'r') && (!strncmp(argv[1], "run", length))
	     && (length >= 3)) {

    if( !netPtr->headevent ) {
      /* Don't run */
      fprintf( stderr, "Error:  network uninitialized!\n" );
      sprintf( netPtr->interp->result, "0" );
    } else {
      /* Run as long as less than tend */
      sprintf( netPtr->interp->result, "%d", 
	       (netPtr->headevent->nextevent->eventt <= netPtr->tend) );
      /* As side effect, update internal events */
      netPtr->prevevent  = netPtr->headevent;
      netPtr->thisevent  = netPtr->headevent->nextevent;
      netPtr->thisneuron = netPtr->thisevent->neuronidx;
      netPtr->thistime   = netPtr->thisevent->eventt;
    }

  } else if ((c == 'c') && (!strncmp(argv[1], "current", length))
	     && (length >= 4)) {

    /* Same timeslice */
    sprintf( netPtr->interp->result, "%d", 
	     (netPtr->thisevent->eventt < 
	      netPtr->thistime + SIMULTANEOUS_EPSILON) );

  } else if ((c == 'a') && (!strncmp(argv[1], "active", length))
	     && (length >= 4)) {
    int cond, sidx, nidx = 0;

    cond = (netPtr->thisevent->neuronidx == netPtr->thisneuron);
    
    /* return index if same neuron, otherwise -1 */
    /* Side effect:  update internal events */
    if( cond ) {
      nidx = GetCurrentEvent( netPtr );
      sidx = nidx % MAXINPSYN;
      nidx = nidx / MAXINPSYN;
      sprintf( netPtr->interp->result, "%d %d", nidx, sidx );
    } else {
      netPtr->prevevent = netPtr->prevevent->nextevent;
      netPtr->thisevent = netPtr->thisevent->nextevent;
      sprintf( netPtr->interp->result, "-1" );
    }

  } else if ((c == 'd') && (!strncmp(argv[1], "decaypotential", length))
	     && (length >= 4)) {
    
    sprintf( netPtr->interp->result, "%f", DecayPotential( netPtr ) );

  } else if ((c == 'i') && (!strncmp(argv[1], "integratepsp", length))
	     && (length >= 4)) {

    sprintf( netPtr->interp->result, "%f", IntegratePsp( netPtr ) );
    netPtr->thisevent =  netPtr->thisevent->nextevent;

  } else if ((c == 'g') && (!strncmp(argv[1], "genspikes", length))
	     && (length >= 4)) {

    sprintf( netPtr->interp->result, "%d", GenerateSpikeEvents( netPtr ) );

  } else if ((c == 'h') && (!strncmp(argv[1], "hyperpolarize", length))
	     && (length >= 4)) {

    sprintf( netPtr->interp->result, "%d", Hyperpolarize( netPtr ) );

  } else if ((c == 'c') && (!strncmp(argv[1], "color", length))
	     && (length >= 3)) {
    XColor color;
    float potential;

    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
		       " color Potential\"", (char *) NULL);
      goto error;
    }
    potential = atof( argv[2] );
    if( potential < REFRACTORYPOTENTIAL ) potential = REFRACTORYPOTENTIAL;
    if( potential > SPIKEPOTENTIAL ) potential = SPIKEPOTENTIAL;
    color = trans_to_color( atof( argv[2] ), REFRACTORYPOTENTIAL, 
			    SPIKEPOTENTIAL );
    sprintf( netPtr->interp->result, "#%04x%04x%04x", color.red, color.green,
	     color.blue );

  } else if ((c == 'c') && (!strncmp(argv[1], "config", length))
	     && (length >= 4)) {
    Neuron *neuronPtr;
    InpsynStruct *inpsynPtr;
    OutconStruct *outconPtr;
    int j;

    if( argc > 2 )
      NetworkConfigure( netPtr, argc-2, argv+2 );
    else {
      printf( "  netPtr         0x%x\n", netPtr );
      printf( "  interp         0x%x\n", netPtr->interp );
      printf( "  objCmd         0x%x\n", netPtr->objCmd );
      printf( "  nneuron        %d\n", netPtr->nneuron );
      printf( "  tend           %d\n", netPtr->tend );
      printf( "  timestep       %d\n", netPtr->timestep );
      printf( "  decayfact      %f\n", netPtr->decayfact );
      printf( "  attenfact      %f\n", netPtr->attenfact );
      for (i = 0; i < netPtr->nneuron; i++ ) {
	neuronPtr = netPtr->neuron[i];
	printf( "  neuron%d        0x%x\n", i, neuronPtr );
	printf( "    widgetId       %d\n", neuronPtr->widgetId );
	printf( "    widgetTag      %s\n", neuronPtr->widgetTag );
	printf( "    itemPtr        0x%x\n", neuronPtr->itemPtr );
	printf( "    potential      %f\n", neuronPtr->potential );
	printf( "    preveventt     %f\n", neuronPtr->preveventt );
	printf( "    ninpsyn        %d\n", neuronPtr->ninpsyn );
	for( j = 0; j < neuronPtr->ninpsyn; j++ ) {
	  inpsynPtr = neuronPtr->inpsyn[j];
	  printf( "    inpsyn         0x%x\n", inpsynPtr );
	  printf( "      efficacy     %f\n", inpsynPtr->efficacy );
	  printf( "      distance     %f\n", inpsynPtr->distance );
	  printf( "      first        %f\n", inpsynPtr->first );
	  printf( "      interval     %f\n", inpsynPtr->interval );
	  printf( "      connected    %d\n", inpsynPtr->connected );
	}
	printf( "    noutcon        %d\n", neuronPtr->noutcon );
	for( j = 0; j < neuronPtr->noutcon; j++ ) {
	  outconPtr = neuronPtr->outcon[j];
	  printf( "    outcon         0x%x\n", outconPtr );
	  printf( "      to_neuron    %d\n", outconPtr->neuronidx );
	  printf( "      to_synapse   %d\n", outconPtr->inpsynidx );
	  printf( "      delay        %f\n", outconPtr->delay );
	}
      }
      printf( "  thisneuron     %d\n", netPtr->thisneuron );
      printf( "  thistime       %f\n", netPtr->thistime );
      printf( "  headevent      0x%x\n", netPtr->headevent );
      printf( "  curevent       0x%x\n", netPtr->curevent );
      printf( "  thisevent      0x%x\n", netPtr->thisevent );
      printf( "  prevevent      0x%x\n", netPtr->prevevent );
      printf( "  queuelen       %d\n", netPtr->queuelen );
    }

  } else {

    Tcl_AppendResult(interp, "Network: bad option \"", argv[1],
		     "\":  must be config",
		     (char *) NULL);
    goto error;
  }

  Tcl_Release((ClientData) netPtr);
  return result;
  
error:
  Tcl_Release((ClientData) netPtr);
  return TCL_ERROR;
}

void
NetworkConfigure( Network *netPtr, int argc, char **argv ) {
  size_t length;
  int i = 0;
  char c;

  while( i < argc ) {
    if( argv[i][0] != '-' ) {
      fprintf( stderr, 
	       "NetworkConfigure:  Can't process arg%d <%s>; skipping\n", 
	       i, argv[i] );
      i++;
      continue;
    }
    c = argv[i][1];
    length = strlen(argv[i]);

    if ((c == 'n') && (!strncmp(argv[i], "-nneuron", length))
	       && (length >= 4)) {
      netPtr->nneuron = atoi( argv[i+1] );
    } else if ((c == 't') && (!strncmp(argv[i], "-tend", length))
	       && (length >= 4)) {
      netPtr->tend = atoi( argv[i+1] );
    } else if ((c == 'd') && (!strncmp(argv[i], "-decayfact", length))
	       && (length >= 4)) {
      netPtr->decayfact = atof( argv[i+1] );
    } else if ((c == 'a') && (!strncmp(argv[i], "-attenfact", length))
	       && (length >= 4)) {
      netPtr->attenfact = atof( argv[i+1] );
    } else if ((c == 'c') && (!strncmp(argv[i], "-canvas", length))
	       && (length >= 4)) {
      Tcl_CmdInfo infoPtr;

      Tcl_GetCommandInfo( netPtr->interp, argv[i+1], &infoPtr );
      netPtr->canvas = (TkCanvas *) infoPtr.clientData;

    } else if ((c == 'n') && (!strncmp(argv[i], "-neuron", 7)) ) {
      Tcl_CmdInfo infoPtr;
      Neuron *neuronPtr;
      int idx;

      sscanf( argv[i], "-neuron%d", &idx );
      if( !netPtr->neuron[idx] ) {
	Tcl_GetCommandInfo( netPtr->interp, argv[i+1], &infoPtr );
	neuronPtr = infoPtr.clientData;
	netPtr->neuron[idx] = neuronPtr;
	if( idx >= netPtr->nneuron ) netPtr->nneuron++;
      } else {
	fprintf( stderr, "Neuron%d already registered\n", idx );
      }
    } else if ((c == 'o') && (!strncmp(argv[i], "-outfile", length))
	       && (length >= 4)) {

      /* 990919:  Just for testing purposes */
      if( !(netPtr->fp = fopen( argv[i+1], "a" )) ) {
	fprintf( stderr, "Could not open output %s; using stderr\n", 
		 argv[i+1] );
	netPtr->fp = stderr;
      } else {
	fprintf( stderr, "Output written to %s\n", argv[i+1] );
      }
    }
    i += 2;
  }
}

static void 
DeleteNetwork(ClientData clientData) {
  free((char *) clientData);
}

void NewNetwork( Network *netPtr ) {
  Neuron *neuronPtr;
  OutconStruct *outconPtr;
  Event *ptr, *prevptr;
  int i, idx, j;

  /* Free all neurons and subfields */
  for( idx = 0; idx < netPtr->nneuron; idx++ )
    DeleteNeuron( netPtr->neuron[idx] );
  netPtr->nneuron = 0;
  netPtr->timestep = 0;
  netPtr->tend = 0;
  netPtr->decayfact = 0.0;
  netPtr->attenfact = 0.0;
  for (i = 0; i < MAXNEURON; i++ )
    netPtr->neuron[i] = NULL;
  InitNetwork( netPtr );
}

void InitNetwork( Network *netPtr ) {
  /* Initialize queue with network's neurons' first inputs */
  /* Assumes that network is new and defined */
  Neuron *nPtr, *mPtr;
  OutconStruct *oPtr;
  Tk_Item *itemPtr;
  Event *ptr, *prevptr;
  int idx, j;

  /* Initialize queue by freeing all events within except head */
  ptr = netPtr->headevent;
  while( ptr != NULL ) {
    prevptr = ptr;
    ptr = ptr->nextevent;
    FreeEvent(prevptr);
    netPtr->queuelen--;
  }

  /* free the space used by the previous event */
  if( netPtr->curevent ) {
    FreeEvent(netPtr->curevent);
    netPtr->queuelen--;
  }

  netPtr->queuelen = 0;
  netPtr->eventcnt = 0;
  netPtr->headevent = NewEvent( netPtr, 0.0, 0, 0, NULL);
  netPtr->prevevent = NULL;
  netPtr->thisevent = NULL;
  netPtr->curevent = NULL;
  netPtr->timestep = 0;
  netPtr->thisneuron = -1;
  netPtr->thistime = 0.0;

  /* Initialize connections */
  for( idx = 0; idx < netPtr->nneuron; idx++ ) {
    nPtr = netPtr->neuron[idx];
    nPtr->potential = RESTPOTENTIAL;
    for( j = 0; j < nPtr->noutcon; j++ ) {
      oPtr = nPtr->outcon[j];
      mPtr = netPtr->neuron[oPtr->neuronidx];
      mPtr->inpsyn[oPtr->inpsynidx]->connected = 1;
    }
  }

  /* Initialize first inputs to network */
  for( idx = 0; idx < netPtr->nneuron; idx++ ) {
    nPtr = netPtr->neuron[idx];
    for( j = 0; j < nPtr->ninpsyn; j++ ) {
      if( !nPtr->inpsyn[j] ) {
	fprintf( stderr, "Neuron %d synapse %d unallocated\n", idx, j );
      } else if( !nPtr->inpsyn[j]->connected ) {
	if( nPtr->inpsyn[j]->efficacy != 0.0 )
	  AddEvent( netPtr, netPtr->headevent, nPtr->inpsyn[j]->first, 
		    idx, j );
      }
    }
  }

  /* Identify canvas objects with neurons */
  for (idx = 0; idx < netPtr->nneuron; idx++ ) {
    nPtr = netPtr->neuron[idx];
    nPtr->canvas = netPtr->canvas;
    for (itemPtr = netPtr->canvas->firstItemPtr; itemPtr != NULL;
	 itemPtr = itemPtr->nextPtr)
      if (itemPtr->id == nPtr->widgetId)
	nPtr->itemPtr = itemPtr;
  }

  fprintf( netPtr->fp, "\nNetwork initialized for script %s\n\n", 
	   Tcl_GetVar(netPtr->interp, "tcl_rcFileName", TCL_GLOBAL_ONLY) );
}

Event *NewEvent( Network *netPtr, float eventt, int neuronidx, 
		int inpsynidx, Event *nextevent) {
  /* instantiate a new event */
  Event *eventPtr;

  if( !(eventPtr = (Event *) ckalloc(sizeof(Event)) )) {
    printf( "Allocate failed for 0x%x eventt %.2f Nidx %d Sidx %d\n", netPtr,
	    eventt, neuronidx, inpsynidx );
    exit(1);
  }
  eventPtr->eventidx = netPtr->eventcnt++;
  eventPtr->eventt = eventt;
  eventPtr->neuronidx = neuronidx;
  eventPtr->inpsynidx = inpsynidx;
  eventPtr->nextevent = nextevent;
  netPtr->queuelen++;
  return(eventPtr);
}

void FreeEvent( Event *eventPtr ) {

  if( !eventPtr ) return;
  eventPtr->eventt = 0.0;
  eventPtr->neuronidx = 0;
  eventPtr->inpsynidx = 0;
  eventPtr->nextevent = NULL;
  ckfree( eventPtr );
  eventPtr = NULL;
}

void AddEvent( Network *netPtr, Event *ptr, float eventt,
	       int neuronidx, int inpsynidx) {
  /* recursively traverse the queue until the right time slot is found */

  if( ptr->nextevent == NULL ) {    /* add to the tail of the queue */
    ptr->nextevent = 
      NewEvent( netPtr, eventt, neuronidx, inpsynidx, NULL );
  } else if( ptr->nextevent->eventt > eventt) { /* add before next event */
    ptr->nextevent = 
      NewEvent( netPtr, eventt, neuronidx, inpsynidx, ptr->nextevent );
  } else {	        	    /* add further back in the queue */
    AddEvent( netPtr, ptr->nextevent, eventt, neuronidx, inpsynidx);
  }
}

int GetCurrentEvent(Network *netPtr) {
  int nidx, sidx;
  Event *eventPtr;
  InpsynStruct *inpsynPtr;
  
  /* free the space used by the previous event */
  if( netPtr->curevent ) {
    FreeEvent(netPtr->curevent);
    netPtr->queuelen--;
  }

  /* take the next event from the queue */
  eventPtr = netPtr->thisevent;
  netPtr->curevent = eventPtr;
  netPtr->prevevent->nextevent = eventPtr->nextevent;
  nidx = eventPtr->neuronidx;
  sidx = eventPtr->inpsynidx;
  fprintf( netPtr->fp, "At %.2f: input at synapse %d of neuron %d\n",
	   eventPtr->eventt, sidx, nidx );
  fflush( netPtr->fp );
  netPtr->timestep = (int) eventPtr->eventt;

  inpsynPtr = netPtr->neuron[nidx]->inpsyn[sidx];
  /* generate the next event on this same input synapse
     unless the input comes from another neuron in the system */
  if( !inpsynPtr->connected ) {
    AddEvent( netPtr, netPtr->headevent, 
	      eventPtr->eventt + inpsynPtr->interval, nidx, sidx);
  }
  return (nidx*MAXINPSYN + sidx);
}

float DecayPotential(Network *netPtr) {
/* the membrane potential decays exponentially with time:
   a factor of decayfact every millisecond */
  Tk_Window tkwin = netPtr->canvas->tkwin;
  Neuron *nPtr = netPtr->neuron[netPtr->curevent->neuronidx];
  Tk_Item *itemPtr;
  float eventt = netPtr->curevent->eventt;

  nPtr->potential = RESTPOTENTIAL + (nPtr->potential - RESTPOTENTIAL) * 
    pow(netPtr->decayfact,(eventt - nPtr->preveventt));
  fprintf( netPtr->fp, "  membrane potential decayed to %.2f\n", 
	   nPtr->potential);
  fflush( netPtr->fp );
  nPtr->preveventt = eventt;
  NeuronDisplay( nPtr );
  return( nPtr->potential );
}

float IntegratePsp(Network *netPtr) {
  /* the PSP attenuates exponentially with distance:
     a factor of attenfact every micrometer */
  Tk_Window tkwin = netPtr->canvas->tkwin;
  Tk_Item *itemPtr;
  Neuron *nPtr = netPtr->neuron[netPtr->curevent->neuronidx];
  int sidx = netPtr->curevent->inpsynidx;
  float addpotential;

  addpotential =  nPtr->inpsyn[sidx]->efficacy * 
    pow(netPtr->attenfact, nPtr->inpsyn[sidx]->distance);
  fprintf( netPtr->fp, "  %.2f PSP attenuated to %.2f\n", 
	  nPtr->inpsyn[sidx]->efficacy, addpotential);
  fflush( netPtr->fp );
  nPtr->potential += addpotential;
  fprintf( netPtr->fp, "  membrane potential now %.2f\n", nPtr->potential);
  fflush( netPtr->fp );
  NeuronDisplay( nPtr );
  return( nPtr->potential );
}      

int GenerateSpikeEvents(Network *netPtr) {
/* If above threshold, send a spike to other neurons through the 
   output connections */
  Tk_Window tkwin = netPtr->canvas->tkwin;
  Tk_Item *itemPtr;
  Neuron *nPtr = netPtr->neuron[netPtr->curevent->neuronidx];
  OutconStruct *outconPtr;
  int i, cond;

  cond = (nPtr->potential > FIRING_THRESHOLD);
  if( cond ) {
    for(i = 0; i < nPtr->noutcon; i++) {
      outconPtr = nPtr->outcon[i];
      AddEvent( netPtr, netPtr->headevent,
		netPtr->curevent->eventt + outconPtr->delay,
		outconPtr->neuronidx, outconPtr->inpsynidx);
    }
    fprintf( netPtr->fp, "  Neuron %d SPIKES at %.2f!!\n",
	     netPtr->curevent->neuronidx, netPtr->curevent->eventt);
    fflush( netPtr->fp );
    /*    printf( "Neuron %d spikes at %.2f\n",  netPtr->curevent->neuronidx, 
	  netPtr->curevent->eventt); */
    nPtr->potential = SPIKEPOTENTIAL;
    NeuronDisplay( nPtr );
  }
  return( cond );
}

int Hyperpolarize(Network *netPtr) {
  /* a refractory period follows each spike, modeled here by 
     hyperpolarizing the membrane */
  Tk_Window tkwin = netPtr->canvas->tkwin;
  Tk_Item *itemPtr;
  Neuron *nPtr = netPtr->neuron[netPtr->curevent->neuronidx];
  int i, cond;

  cond = (nPtr->potential > FIRING_THRESHOLD);
  if( cond ) {
    nPtr->potential = REFRACTORYPOTENTIAL;
    fprintf( netPtr->fp, "  membrane hyperpolarized to %.2f\n", 
	     nPtr->potential);
    fflush( netPtr->fp );
    NeuronDisplay( nPtr );
  }
  return( cond );
}

static void
ReadParms(Network *netPtr, char *parmfile) {
  /* Read network parameters specified by file */
  Neuron *neuronPtr;
  InpsynStruct *inpsynPtr;
  OutconStruct *outconPtr;
  FILE *fp;
  char file[MAXSTRLEN], s[MAXSTRLEN], cmd[MAXSTRLEN];
  int i, j, neuronidx, inpsynidx, maxneuronidx=0, maxinpsynidx, nread;

  fp = fopen( parmfile, "r" );
  /* read global parameters */
  fscanf(fp, "%f %f %f", &netPtr->tend, &netPtr->decayfact, 
	 &netPtr->attenfact); 
  fgets(s,99,fp);

  /* read parameters for each neuron */
  for( i = 0; fscanf(fp,"%d", &neuronidx)!=EOF; i++) {
    fgets(s,99,fp);
    if(neuronidx > maxneuronidx) maxneuronidx = neuronidx;
    /* read input synapse parameters */
    maxinpsynidx = 0;
    fscanf(fp,"%d", &nread);
    fgets(s,99,fp);
    if( !netPtr->neuron[neuronidx] ) {
      sprintf( cmd, "Neuron %d %d", neuronidx, nread );
      Tcl_Eval( netPtr->interp, cmd );
      fprintf( stderr, "Reading in new neuron%d\n", neuronidx );
    }
    neuronPtr = netPtr->neuron[neuronidx];
    fprintf( stderr, "Neuron%d = 0x%x\n", neuronidx, neuronPtr );
    for( j = 0; j < nread; j++ ) {
      fscanf(fp,"%d", &inpsynidx);
      if(inpsynidx > maxinpsynidx) maxinpsynidx = inpsynidx;
      inpsynPtr = (InpsynStruct *) ckalloc(sizeof(InpsynStruct));
      fscanf( fp, "%f %f %f %f", &inpsynPtr->efficacy, &inpsynPtr->distance,
	     &inpsynPtr->first, &inpsynPtr->interval);
      fgets(s,99,fp);
      neuronPtr->inpsyn[inpsynidx] = inpsynPtr;
      fprintf( stderr, "N%d syn%d = 0x%x\n", neuronidx, inpsynidx,
	       neuronPtr->inpsyn[inpsynidx] );
    }
    neuronPtr->ninpsyn=maxinpsynidx+1;
    /* read output connection parameters */
    fscanf(fp,"%d", &neuronPtr->noutcon);
    fgets(s,99,fp);
    for( j = 0; j < neuronPtr->noutcon; j++ ) {
      outconPtr = (OutconStruct *) ckalloc(sizeof(OutconStruct));
      fscanf( fp,"%d %d %f", &outconPtr->neuronidx, 
	      &outconPtr->inpsynidx, &outconPtr->delay);
      fgets(s,99,fp);
      neuronPtr->outcon[j] = outconPtr;
      fprintf( stderr, "N%d out%d = 0x%x\n", neuronidx, j,
	       neuronPtr->outcon[j] );
    }
  }
  fclose(fp);

  netPtr->nneuron = maxneuronidx+1;
  InitNetwork( netPtr );
}

void print_event_queue(FILE *qp, Event *headevent) {
  Event *ptr;
  int i = 1;
  
  for(ptr=headevent; ptr!=NULL; ptr=ptr->nextevent)
    fprintf( qp, "%4d  %3d  (next %3d):  %.2f (%d %d)\n", i++, ptr->eventidx, 
	     (ptr->nextevent ? ptr->nextevent->eventidx : -1),
	     ptr->eventt, ptr->neuronidx, ptr->inpsynidx );
}
