/* MAD: Software for the Modular Action Description Language
   Copyright (C) 2008, 2009  Selim T. Erdogan

   This file is part of MAD.

   MAD is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   MAD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with MAD.  If not, see <http://www.gnu.org/licenses/>.


   Contact info: selim@cs.utexas.edu
*/


/* File: data-manipulation.c
   This file contains definitions of the functions used in MAD. 
*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "data-types.h"
//#ifndef YACCER_TAB_H_INCLUDED_IN_DATA_MANIPULATION_C
// #define YACCER_TAB_H_INCLUDED_IN_DATA_MANIPULATION_C 1
#include "yaccer.tab.h"
// #endif

/*******************************************/
/*******************************************/
/*******************************************/
/****** Function definitions ***************/
/*******************************************/
/*******************************************/


/* To push/pop input file data, as we recursively encounter
   files in MAD include statements a. */
void push_input_file_name(char * name) {
  if ( input_file_name_stack_pointer >= INPUT_FILE_STACK_SIZE ) {
    print_error_message(stderr, "Stack size %d exceeded for number of \"include\"d files.\n", INPUT_FILE_STACK_SIZE);
    exit(EXIT_FAILURE);
  }
  else {
    input_file_name_stack[input_file_name_stack_pointer++] = name;
  }
}

void push_line_no(int line_number) {
  if ( input_line_number_stack_pointer >= INPUT_FILE_STACK_SIZE ) {
    print_error_message(stderr, "Stack size %d exceeded for number of \"include\"d files.\n", INPUT_FILE_STACK_SIZE);
    exit(EXIT_FAILURE);
  }
  else {
    input_line_number_stack[input_line_number_stack_pointer++] = line_number;
  }
}
char * pop_input_file_name() {
  if ( input_file_name_stack_pointer <= 0 ) {
    print_error_message(stderr, "Internal Error: Trying to pop empty stack of \"include\"d files.\n");
    exit(EXIT_FAILURE);
  }
  else {
    return input_file_name_stack[--input_file_name_stack_pointer];
  }
}

int pop_line_no() {
  if ( input_line_number_stack_pointer <= 0 ) {
    print_error_message(stderr, "Internal Error: Trying to pop empty stack of \"include\"d files.\n");
    exit(EXIT_FAILURE);
  }
  else {
    return input_line_number_stack[--input_line_number_stack_pointer];
  }
}


/* In order to standardize error messages and make changes to
   all error message printing easily, we use this wrapper around
   fprintf(stderr, ...) 

   This is called when errors are encountered during the parsing stage,
   and reports the line number. 

   The implementation is based on print_to_end_of_string() below.
*/
void print_parser_error_message(FILE * error_stream, char * file_name, int line_number, const char *fmt, ...) {

  // Flag error
  parse_error_encountered = 1;

  fprintf(error_stream, "%s, %d:", file_name, line_number);

  va_list ap;
  va_start(ap, fmt);
  vfprintf(error_stream, fmt, ap);
  va_end(ap);
}

/* In order to standardize error messages and make changes to
   all error message printing easily, we use this wrapper around
   fprintf(stderr, ...) 

   This is called for errors that are not related to a specific file 
   and line number.

   The implementation is based on print_to_end_of_string() below.
*/
void print_error_message(FILE * error_stream, const char *fmt, ...) {

  va_list ap;
  va_start(ap, fmt);
  vfprintf(error_stream, fmt, ap);
  va_end(ap);
}


/* Converts a nonnegative integer to a string of characters.
   If the number is negative it returns a NULL string. */
char * integer_to_string(int integer) {
  int num_digits;
  int integer_copy;
  char * integer_string;
  int i;
  int sign = 1;
  
  if ( integer < 0 ) {
    sign = -1;
    //print_error_message(stderr, "Internal error: trying to convert a negative integer to a string?\n");
  }
    
  /* first find out how many digits there are */
  integer_copy = integer;
  num_digits = 1;
  integer = integer / 10;
  while ( integer != 0 ) {
    num_digits = num_digits + 1;
    integer = integer / 10;
  }
    
  /* now allocate that many characters and fill them out */
  integer = integer_copy;

  if ( sign == 1 ) {
    integer_string = (char *) malloc( sizeof(char) * ( num_digits + 1 ) );
  }
  else {
    // if it's negative, allocate an extra character for '-'
    integer_string = (char *) malloc( sizeof(char) * ( num_digits + 2 ) );
    integer_string[num_digits] = '-';
  }
  
  for ( i = num_digits - 1 ; i >= 0 ; i = i - 1 ) {
    integer_string[i] = '0' + integer % 10;
    integer = integer / 10;
  }
  /* Don't forget to tack on a '\0' to indicate the end of the string. */
  integer_string[num_digits]='\0';
  
  return integer_string;
}

/* Given a string s and an integer import_index, returns a new string
   obtained by prepending "I<import_index>." (without < and >) to s. */
char * prepend_import_prefix_to_string(char * s, int import_index) {
  
  char * new_string;

  /* the size of new_string should be long enough to hold 
     the index prefix "In." and the variable name */
  new_string = (char *) malloc( sizeof(char) * ( strlen(s) 
						 + strlen(integer_to_string(import_index))
						 + 3 ) );

  strcpy(new_string, "I");
  strcat(new_string, integer_to_string(import_index));
  strcat(new_string, ".");
  strcat(new_string, s);

  return new_string;
}

/* Given a string s, it checks to see if the string begins with an import
   prefix of the form "I<integer>." */ 
int string_begins_with_an_import_prefix(const char * s) {
  int i;
  if ( (strlen(s) > 1)
       && (s[0] == 'I')
       && ('0' <= s[1]) && (s[1] <= '9') ) {
    // We have a beginning "I<integer>".  Now we look for '.'
    i = 1;
    while ( ('0' <= s[i]) && (s[i] <= '9') ) {
      i++;
    }
    if ( s[i] == '.' ) {
      return 1;
    }
    else {
      return 0;
    }
  }
  else {
    return 0;
  }
  // This next line should be unreachable code
  print_error_message(stderr, "string %s doesn't begin with an import prefix\n", s);
}

/* Given a string s, returns
   a new string obtained by prepending "I0." and appending 
   "_var<new_variable_index>" (without < and >) to s. */
char * make_new_variable_name(char *s) {
  char * new_string;
  
  /* the size of new_string should be long enough to hold 
     the index prefix "In." and the variable name */
  new_string = (char *) malloc( sizeof(char) * ( strlen(s) 
						 + strlen(integer_to_string(new_variable_index))
						 + 8 ) );
  strcpy(new_string, "I0.");
  strcat(new_string, s);
  strcat(new_string, "_var");
  strcat(new_string, integer_to_string(new_variable_index));
  
  /* Increment index to prepare for the next variable creation. */
  new_variable_index = new_variable_index + 1;

  return new_string;
}

/* Takes a string and prepares it to be used as input for CCalc.
   If there are any import prefixes "I<m>." it turns them into
   "i<m>dot" and for the part after any import prefixes, it
   converts all letters to lower case.

   (It assumes the string is in the correct form for MAD, so it just
   replaces "." by "dot" and makes all letters lower case.)

   If the string does not begin with "I", then it just turns it into
   lower case, but doesn't replace "."s  (This is necessary for
   integer ranges which may be of the for "lo..hi" and the dots
   mustn't be replaced.)

   !!!: This doesn't check for any name clashes after conversion
   so, for example, if identifiers agent and Agent exist in
   the MAD input, they will both turn into agent. */
char * prepare_string_for_ccalc(char * s) {

  char * new_string;
  int dot_counter;
  int index;
  int new_index;

  int has_import_beginning = 0;

  if ( s != NULL ) {

    if ( string_begins_with_an_import_prefix(s) ) {
      has_import_beginning = 1;
      /* since we replace '.' with "dot" we need to find out how many dots
	 there are. */
      dot_counter = 0;
      index = 0;
      while ( s[index] != '\0' ) {
	if ( s[index] == '.' ) {
	  dot_counter++;
	}
	index++;
      }
    }
    else {
      dot_counter = 0;
    }
    
    
    /* Allocate enough space to hold the new string, which will have two extra
       characters for every dot. */
    new_string = (char *) malloc( sizeof(char) * ( strlen(s) 
						   + (dot_counter * 2)
						   + 1 ) );

    /* Now go down the string, replacing every dot and making all 
       letters lower-case. */
    if ( ('A' <= s[0]) && (s[0] <= 'Z') ) {
      new_string[0] = (s[0] - 'A') + 'a';
    }
    else {
      new_string[0] = s[0];
    }
    index = new_index = 1;
    while ( s[index] != '\0' ) {
      if ( s[index] == '.' && has_import_beginning ) {
	new_string[new_index++] = 'd';
	new_string[new_index++] = 'o';
	new_string[new_index++] = 't';
      }
      else if ( ('A' <= s[index]) && (s[index] <= 'Z') ) {
	new_string[new_index++] = (s[index] - 'A') + 'a';
      }
      else {
	new_string[new_index++] = s[index];
      }
      index++;
    }

    /* We need to add the end-of-string character explicitly. */
    new_string[new_index] = '\0';

    return new_string;
  }
  else {
    print_error_message(stderr, "A NULL string was given to convert to CCalc format!\n");
    return NULL;
  }
}

/* Takes a character string "message" and prints to the end of 
   that string.  If the string is NULL, it creates it. 
   The resulting string is returned.
   
   This function is based on the example from the man page of sprintf. 
*/
char * print_to_end_of_string(char * message, const char *fmt, ...) {
  /* Guess that we need no more than 100 bytes for the addition. */
  int n, size = 100;
  char *p, *np;
  va_list ap;

  int orig_len; /* length of original message */

  if ( message != NULL ) {
    orig_len = strlen(message);
  }
  else {
    orig_len = 0;
  }

  if ((p = realloc(message, orig_len + size)) == NULL)
    return NULL;
  
  while (1) {
    /* Try to print in the allocated space. */
    va_start(ap, fmt);
    n = vsnprintf ( (p+orig_len), size, fmt, ap);
    va_end(ap);
    /* If that worked, return the string. */
    if (n > -1 && n < size) {
      return p;
    }
    /* Else try again with more space. */
    if (n > -1) {
      size = n + 1; /* precisely what is needed */
    }
    if ((np = realloc (p, size)) == NULL) {
      free(p);
      return NULL;
    } 
    else {
      p = np;
    }
  }
}


/****** Managing lists of numeric symbols **************/

/* initialize the list of numeric_symbols */
void init_numeric_symbols() {
  numeric_symbol_list_first = numeric_symbol_list_last = NULL;
}

/* Looks up a numeric_symbol and returns a pointer to its list entry.
   Returns NULL if not found. */
struct numeric_symbol_list_node * lookup_numeric_symbol(char * lookup_name) {
  struct numeric_symbol_list_node * current;

  current = numeric_symbol_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Called when inserting a new numeric symbol.  Makes a new numeric symbol. */
struct mad_numeric_symbol * make_new_numeric_symbol (char * name, char * integer) {
  struct mad_numeric_symbol * new_numeric_symbol ;

  new_numeric_symbol = (struct mad_numeric_symbol *) malloc( sizeof(struct mad_numeric_symbol) );
  new_numeric_symbol->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_numeric_symbol->name, name );
  new_numeric_symbol->integer = (char *) malloc( sizeof(char) * ( strlen(integer) + 1 ) );
  strcpy( new_numeric_symbol->integer, integer );
  return new_numeric_symbol;
}

/* This inserts a new numeric symbol into the list of numeric symbols */
void insert_numeric_symbol (char * name, char * integer) {
  struct numeric_symbol_list_node * new_node;

  new_node = (struct numeric_symbol_list_node *) malloc( sizeof(struct numeric_symbol_list_node) );
  new_node->node_data = make_new_numeric_symbol(name, integer);
  new_node->next = NULL;

  /* numeric_symbol_list_last is the pointer to the end of the list */
  if ( numeric_symbol_list_last == NULL ) { /* the list is empty */
    numeric_symbol_list_first = new_node;
    numeric_symbol_list_last = new_node;
  }
  else {
  numeric_symbol_list_last->next = new_node;
  numeric_symbol_list_last = new_node;
  }
}

/* for debugging */
void print_current_numeric_symbols(FILE * stream) {
  struct numeric_symbol_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Numeric Symbols:\n");
  current = numeric_symbol_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->name != NULL ) {
	fprintf(stream, "%s", current->node_data->name);
      }
      else {
	fprintf(stream, "(NULL-symbol_name");
      }
      fprintf(stream, " = ");
      if ( current->node_data->integer != NULL ) {
	fprintf(stream, "%s", current->node_data->integer);
      }
      else {
	fprintf(stream, "NULL-integer!");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}


/****** Managing lists of character strings **************/


/* initialize a list of strings */
void init_string_list(struct string_list_node ** first, struct string_list_node ** last) {
  (*first) = (*last) = NULL;
}

/* This inserts a new string into a list of strings */
void insert_string_into_list(struct string_list_node ** first, 
			     struct string_list_node ** last, 
			     char * new_string) {
  struct string_list_node * new_node;

  new_node = (struct string_list_node *) malloc( sizeof(struct string_list_node) );
  new_node->node_data = (char *) malloc( sizeof(char) * ( strlen(new_string) + 1 ) );
  strcpy( new_node->node_data, new_string );
  new_node->next = NULL;

  /* last is the pointer to the end of the list */
  if ( (*last) == NULL ) { /* the list is empty */
    (*first) = new_node;
    (*last) = new_node;
  }
  else {
  (*last)->next = new_node;
  (*last) = new_node;
  }
}

struct string_list_node * lookup_string_in_list(char * lookup_string, struct string_list_node * list) {
  struct string_list_node * current;

  current = list;
  if ( lookup_string != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data, lookup_string) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* This checks whether two lists of strings are the same */
int string_list_same(struct string_list_node * first, 
		     struct string_list_node * second) {

  if ( first != NULL ) {
    if (second != NULL ) { /* both lists are not empty */
      while ( first != NULL && second != NULL) {
	if ( strcmp(first->node_data, second->node_data) != 0 ) {
	  return 0;
	}
	/* the compared strings were the same so we move on to the next pair */
	first = first->next;
	second = second->next;
      }
      if (first == NULL & second == NULL ) {
	/* We got to the end of both lists together */
	return 1;
      }
    }
    else { /* the first list is not empty but the second is */
      return 0;
    }
  }
  else { /* first list is empty */
    if (second != NULL ) {
      return 0;
    }
    else { /* both lists are empty */
      return 1;
    }
  }
}

/* Makes a copy of a given string list */
struct string_list_node * copy_string_list(struct string_list_node * string_list) {
  struct string_list_node * copied_string_list;
  
  if ( string_list != NULL ) {
    /* First copy current node */
    copied_string_list = (struct string_list_node *) malloc( sizeof(struct string_list_node) );
    copied_string_list->node_data = (char *) malloc( sizeof(char) * ( strlen(string_list->node_data) + 1 ) );
    strcpy( copied_string_list->node_data, string_list->node_data);
    /* Then copy the rest */
    if (string_list->next != NULL) {
      copied_string_list->next = copy_string_list(string_list->next);
    }
    else {
      copied_string_list->next = NULL;
    }
  }
  else { // the original string list is empty
    copied_string_list = NULL;
  }

  return copied_string_list;
}

/* Finds the set difference between two given lists of strings and 
   returns it as a list of strings.
   (One use of this is to compare the lists of free variables in two
    formulas.)

   Assumes that the given string lists have no repetitions.
 */
struct string_list_node * set_difference_of_string_lists(struct string_list_node * first, 
							 struct string_list_node * second) {

  struct string_list_node * set_difference_string_list_first;
  struct string_list_node * set_difference_string_list_last;

  /* To go down the two lists */
  struct string_list_node * current_first;
  struct string_list_node * current_second;

  int match_found; /* To signal that we found a matching string in the second list */

  /* Initialize the set difference */
  init_string_list(&set_difference_string_list_first, &set_difference_string_list_last);

  /* Go down the first list, and check whether each element is in the second list. */
  current_first = first;
  while ( current_first != NULL ) {

    /* We will search the second list to see if anything matches the current string from
       the first list. */

    match_found = 0;

    /* We start at the beginning of the second list. */
    current_second = second;
    while ( current_second != NULL ) {
      if ( strcmp(current_first->node_data, current_second->node_data) == 0 ) {
	/* The string is matched in the second list, so we don't need to look further. */
	match_found = 1;
	break;
      }
      current_second = current_second->next;
    }

    if ( match_found == 0) {
      /* Add this to the set difference */
      insert_string_into_list(&set_difference_string_list_first,
			      &set_difference_string_list_last,
			      current_first->node_data);
    }
    
    current_first = current_first->next;
  }
  
  return set_difference_string_list_first;
}

/* Finds the set intersection of two given lists of strings and 
   returns it as a list of strings.
   (One use of this is to find the common free variables in two
    formulas, for standardizing apart before unification.)

   Assumes that the given string lists have no repetitions.
 */
struct string_list_node * set_intersection_of_string_lists(struct string_list_node * first, 
							   struct string_list_node * second) {

  struct string_list_node * set_intersection_string_list_first;
  struct string_list_node * set_intersection_string_list_last;

  /* To go down the two lists */
  struct string_list_node * current_first;
  struct string_list_node * current_second;

  /* Initialize the set intersection */
  init_string_list(&set_intersection_string_list_first, &set_intersection_string_list_last);

  /* Go down the first list, and check whether each element is in the second list. */
  current_first = first;
  while ( current_first != NULL ) {

    /* We will search the second list to see if anything matches the current string from
       the first list. */

    /* We start at the beginning of the second list. */
    current_second = second;
    while ( current_second != NULL ) {
      if ( strcmp(current_first->node_data, current_second->node_data) == 0 ) {
	/* The string is matched in the second list, so we don't need to look further. */
	insert_string_into_list(&set_intersection_string_list_first,
				&set_intersection_string_list_last,
				current_first->node_data);
	break;
      }
      current_second = current_second->next;
    }
    
    current_first = current_first->next;
  }
  
  return set_intersection_string_list_first;
}


/* This adds a string list onto the end of another */
void append_string_list(struct string_list_node ** first, 
			struct string_list_node * second) {

  struct string_list_node * current;

  if ( (*first) == NULL) {
    (*first) = second;
  }
  else {
    current = (*first);
    while ( current->next != NULL ) {
      current = current->next;
    }
    current->next = second;
  }
  
}


/* for debugging */
void print_string_list(FILE * stream, struct string_list_node * string_list) {
  struct string_list_node * current;

  if ( string_list == NULL ) {
    fprintf(stream, "An empty list of strings.\n");
  }
  else {
    fprintf(stream, "String list: (");
    current = string_list;
    fprintf(stream, "%s", current->node_data);
    current = current->next;
    while ( current != NULL ) {
      fprintf(stream, ", %s", current->node_data);
      current = current->next;
    }
    fprintf(stream, ")\n");
  }
}


/****** Managing lists of identifiers **************/


/* initialize the list of identifiers */
void init_identifiers() {
  id_list_first = id_list_last = NULL;
}

/* initialize the list of identifiers to contain the list of global sorts */
void init_identifiers_with_global_sorts() {

  struct sort_list_node * sort_list;

  init_identifiers();

  sort_list = global_sort_list_first;
  while ( sort_list != NULL ) {
    insert_identifier(sort_list->node_data->name);
    sort_list = sort_list->next;
  }
}

/* Looks up an identifier and returns a pointer to its list entry.
   Returns NULL if not found. */
struct id_list_node * lookup_identifier(char * lookup_name) {
  struct id_list_node * current;

  current = id_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->id_name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* This inserts a new identifier into the list of identifiers */
void insert_identifier (char * name) {
  struct id_list_node * new_node;

  new_node = (struct id_list_node *) malloc( sizeof(struct id_list_node) );
  new_node->node_data = make_id_data(name);
  new_node->next = NULL;

  /* id_list_last is the pointer to the end of the list */
  if ( id_list_last == NULL ) { /* the list is empty */
    id_list_first = new_node;
    id_list_last = new_node;
  }
  else {
  id_list_last->next = new_node;
  id_list_last = new_node;
  }
}


/* Called when inserting a new identifier.  Makes a new identifier */
struct id_data * make_id_data (char * name) {
  struct id_data * new_data ;

  new_data = (struct id_data *) malloc( sizeof(struct id_data) );
  new_data->id_name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_data->id_name, name );
  new_data->id_type = UNINITIALIZED_ID;
  return new_data;
}

/* For indicating a type of an identifier.
   The id is found during lexical analysis but we only
   learn it's type during parsing. */
void update_id_type(char * name, int type) {
  struct id_list_node * this_identifier;

  this_identifier = lookup_identifier(name);
  if (this_identifier == NULL) {
    print_error_message(stderr, "Internal error: Trying to update nonexistent identifier %s?\n", name);
  }
  else {
    this_identifier->node_data->id_type = type;
  }
}


/* for debugging */
void print_id_list(FILE * stream) {
  struct id_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Identifiers:\n");
  current = id_list_first;
  while ( current != NULL ) {
    fprintf(stream, "%d: %s, of type", counter, current->node_data->id_name);
    switch (current->node_data->id_type) {
    case UNINITIALIZED_ID:
      fprintf(stream, " Uninitialized\n");
      break;
    case SORT_ID:
      fprintf(stream, " Sort\n");
      break;
    case OBJECT_ID:
      fprintf(stream, " Object\n");
      break;
    case CONSTANT_ID:
      fprintf(stream, " Constant\n");
      break;
    case VARIABLE_ID:
      fprintf(stream, " Variable\n");
      break;
    case MODULE_ID:
      fprintf(stream, " Module\n");
      break;
    }
    counter++;
    current = current->next;
  }
}


/****** Managing lists of sorts **************/


/* initialize the list of sorts */
void init_sorts() {
  sort_list_first = sort_list_last = NULL;
}

/* initialize the global list of sorts */
void init_global_sorts() {
  global_sort_list_first = global_sort_list_last = NULL;
}

/* Looks up a sort and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_list_node * lookup_sort(char * lookup_name) {
  struct sort_list_node * current;

  current = sort_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Looks up a sort in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_list_node * lookup_sort_in_module(char * lookup_name, struct mad_module * module) {
  struct sort_list_node * current;

  current = module->module_sorts;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action". */
int is_a_known_sort(char * name) {
  if ( lookup_sort(name) != NULL
       || strcmp(name, "Boolean") == 0 
       || strcmp(name, "action") == 0 ) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action". */
int is_a_known_sort_in_module(char * name, struct mad_module * module) {
  if ( lookup_sort_in_module(name, module) != NULL
       || strcmp(name, "Boolean") == 0 
       || strcmp(name, "action") == 0 ) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort or the keyword "Boolean". */
int is_a_known_object_sort(char * name) {
  if ( lookup_sort(name) != NULL
       || strcmp(name, "Boolean") == 0) {
    return 1;
  }
  else {
    return 0;
  }
} 

/* Checks if a given string is a declared sort in a particular module,
   or the keyword "Boolean". */
int is_a_known_object_sort_in_module(char * name, struct mad_module * module) {
  if ( lookup_sort_in_module(name, module) != NULL
       || strcmp(name, "Boolean") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort or the keyword "Boolean". */
int is_a_known_constant_domain_sort(char * name) {
  if ( lookup_sort(name) != NULL
       || strcmp(name, "Boolean") == 0) {
    return 1;
  }
  else {
    return 0;
  }
} 

/* Checks if a given string is a declared sort in a particular module,
   or the keyword "Boolean". */
int is_a_known_constant_domain_sort_in_module(char * name, struct mad_module * module) {
  if ( lookup_sort_in_module(name, module) != NULL
       || strcmp(name, "Boolean") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action". */
int is_a_known_constant_argument_sort(char * name) {
  if ( lookup_sort(name) != NULL
       || strcmp(name, "Boolean") == 0
       || strcmp(name, "action") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action". */
int is_a_known_constant_argument_sort_in_module(char * name, struct mad_module * module) {
  if ( lookup_sort_in_module(name, module) != NULL
       || strcmp(name, "Boolean") == 0
       || strcmp(name, "action") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action", "explicitAction". */
int is_a_known_variable_sort(char * name) {
  if ( lookup_sort(name) != NULL
       || strcmp(name, "Boolean") == 0 
       || strcmp(name, "action") == 0 
       || strcmp(name, "explicitAction") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action", "explicitAction". */
int is_a_known_variable_sort_in_module(char * name, struct mad_module * module) {
  if ( lookup_sort_in_module(name, module) != NULL
       || strcmp(name, "Boolean") == 0 
       || strcmp(name, "action") == 0 
       || strcmp(name, "explicitAction") == 0) {
    return 1;
  }
  else {
    return 0;
  }
}

/* Called when inserting a new sort.  Makes a new sort */
struct mad_sort * make_new_sort (char * name) {
  struct mad_sort * new_sort ;

  new_sort = (struct mad_sort *) malloc( sizeof(struct mad_sort) );
  new_sort->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_sort->name, name );

  new_sort->objects = NULL;
  new_sort->num_objects = 0;
  new_sort->objects_built = 0;

  return new_sort;
}


/* This inserts a new sort into the list of sorts */
void insert_sort (char * name) {
  struct sort_list_node * new_node;

  new_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
  new_node->node_data = make_new_sort(name);
  new_node->next = NULL;

  /* sort_list_last is the pointer to the end of the list */
  if ( sort_list_last == NULL ) { /* the list is empty */
    sort_list_first = new_node;
    sort_list_last = new_node;
  }
  else {
  sort_list_last->next = new_node;
  sort_list_last = new_node;
  }

  /* update the list of identifiers to indicate this
     identifier is a sort */
  update_id_type(name, SORT_ID);
}

/* Makes a copy of a given sort list */
struct sort_list_node * copy_sort_list(struct sort_list_node * sort_list) {
  struct sort_list_node * copied_sort_list;
  
  if ( sort_list != NULL ) {
    /* First copy current node */
    copied_sort_list = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
    copied_sort_list->node_data = make_new_sort(sort_list->node_data->name);
    /* Then copy the rest */
    copied_sort_list->next = copy_sort_list(sort_list->next);
  }
  else {
    copied_sort_list = NULL;
  }

  return copied_sort_list;
}

/* Makes the list of sorts a copy of the global list of sorts */ 
void copy_global_sorts() {

  struct sort_list_node * sort_list;

  init_sorts();

  sort_list = global_sort_list_first;
  while ( sort_list != NULL ) {
    insert_sort(sort_list->node_data->name);
    sort_list = sort_list->next;
  }
}

/* Called to merge sorts of an imported module with the current sorts */
void merge_sorts_from_module(struct mad_module * module) {

  struct sort_list_node * current;
  struct id_list_node * id;

  current = module->module_sorts;
  while ( current != NULL ) {
    /* First check if the identifier has been used already */
    id = lookup_identifier(current->node_data->name);
    if ( id != NULL && id->node_data->id_type != SORT_ID ) {
      /* It has been declared previously as something other than a sort */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Trying to merge sort %s (from module %s) into module %s, but this sort was already declared as another identifier.\n",
			  current->node_data->name,
			  module->name,
			  name_of_module_currently_being_parsed);
    }
    else if ( lookup_sort(current->node_data->name) == NULL ) {
      /* It has not been declared previously as something other than a sort 
         and it is not a sort,
         so we know it hasn't been declared */
      insert_identifier(current->node_data->name);
      insert_sort(current->node_data->name);
    }
    else {
      /* The sort has already been declared so there's no need to add it. */
    }
    current = current->next;
  }
}

/* for debugging */
void print_current_sorts(FILE * stream) {
  struct sort_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Sorts:\n");
  current = sort_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->name != NULL ) {
	fprintf(stream, "%s", current->node_data->name);
      }
      else {
	fprintf(stream, "NULL-sort-name");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of sorts (e.g. of a specific module). */
void print_sort_list(FILE * stream, struct sort_list_node * list) {
  struct sort_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Sorts:\n");
  current = list;
  while ( current != NULL ) {
    fprintf(stream, "%d: %s\n", counter, current->node_data->name);
    counter++;
    current = current->next;
  }
}

/* for printing out a sort declaration section to be used as ccalc input */
void print_sort_list_for_ccalc(FILE * stream, struct sort_list_node * list) {
  struct sort_list_node * current;

  // If the sort is an integer range, we need to print it out differently
  char * new_sort_name;
  
  current = list;
  if ( current != NULL ) {
    fprintf(stream, ":- sorts\n");
    if ( is_an_integer_range(current->node_data->name) ) {
      fprintf(stream, 
	      "    %s",
	      convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
    }
    else {
      fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
    }
    current = current->next;
    while ( current != NULL ) {
      if ( is_an_integer_range(current->node_data->name) ) {
	fprintf(stream, ";\n");
	fprintf(stream, 
		"    %s",
		convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
	current = current->next;
      }
      else {
	fprintf(stream, ";\n");
	fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
	current = current->next;
      }
    }
    fprintf(stream, ".\n");
  }
}


/****** Managing integer ranges and arithmetic **************/

// Takes two strings "m", "n" containing integer values and returns
// an integer range string "m..n"
char * make_integer_range_string(char * begin, char * end) {
  char * new_integer_range;
  
  new_integer_range
    = (char *) malloc( sizeof(char) * strlen(begin) + strlen(end) + 2 + 1 );
  strcpy(new_integer_range, begin);
  strcat(new_integer_range, "..");
  strcat(new_integer_range, end);
  
  return new_integer_range;
}

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and makes sure that "lo <= hi"
//
// We assume that it's syntactically valid, having been already
// processed and marked as an integer range in the lexical analyzer
int is_a_valid_integer_range(char * range_string) {

  int lower,upper;
  int old_lower,old_upper;
  int sign;
  int i;

  lower = upper = 0;
  old_lower = old_upper = 0;

  if (range_string == NULL ) {
    print_parser_error_message(stderr,
			       input_file_name, 
			       error_lineno,
			       "Empty integer range!\n");
    return 0;
  }
  
  i=0;

  sign = 1;
  // if the first character is '-' mark the sign and go on
  if ( range_string[0]=='-' ) {
    sign = -1;
    i++;
  }
  while ( range_string[i]!='.') {
    old_lower = lower;
    lower = lower*10 + range_string[i] - '0';
    if ( old_lower > lower ) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "Lower integer of range has too large an absolute value!\n");
      return 0;
    }
    i++;
  }
  lower = lower * sign;

  i=i+2; //skip over ".."

  sign = 1;
  if ( range_string[i]=='-' ) {
    sign = -1;
    i++;
  }
  while ( range_string[i]!='\0') {
    old_upper = upper;
    upper = upper*10 + range_string[i] - '0';
    if ( old_upper > upper ) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "Upper integer of range has too large an absolute value!\n");
      return 0;
    }
    i++;
  }

  upper = upper * sign;

  if ( lower > upper ) {
    print_parser_error_message(stderr,
			       input_file_name, 
			       error_lineno,
			       "Integer range beginning (%d) higher than its end (%d)!\n",
			       lower,
			       upper);
    return 0;
  }

  return 1;
}

// Takes a string and check if it is an integer range 
// in the form "lo..hi" (where lo and hi may be negative too) 
//
// Doesn't check wheter lo<=hi because we assume that has already
// been checked
int is_an_integer_range(char * range_string) {

  int i;

  if (range_string == NULL ) {
    print_error_message(stderr,
			"Internal error: Empty string as integer-range candidate!\n");
    return 0;
  }

  // We will simply attempt to walk through the string, only increasing
  // the index i when the character we see is valid/expected in an integer range
  // If we are able to go through all characters, that means the string is
  // a valid integer range.
  i=0;
  if ( range_string[0]=='-' ) {
    i++;
  }
  while ( ('0' <= range_string[i]) && (range_string[i] <= '9') ) {
      i++;
  }
  if ( range_string[i]=='.' && range_string[i+1]=='.' ) {
    i=i+2;
  }
  if ( range_string[i]=='-' ) {
    i++;
  }
  while ( ('0' <= range_string[i]) && (range_string[i] <= '9') ) {
      i++;
  }

  // Were we able to get to the end?
  if ( i==strlen(range_string) ) {
    return 1;
  }
  else {
    return 0;
  }
}

// Takes a string and check if it is an integer
int is_an_integer(char * s) {

  int i;

  if (s == NULL ) {
    print_error_message(stderr,
			"Internal error: Empty string as integer candidate!\n");
    return 0;
  }

  // We will simply attempt to walk through the string, only increasing
  // the index i when the character we see is valid/expected in an integer
  // If we are able to go through all characters, that means the string is
  // a valid integer.
  i=0;
  if ( s[0]=='-' ) {
    i++;
  }
  while ( ('0' <= s[i]) && (s[i] <= '9') ) {
      i++;
  }

  // Were we able to get to the end?
  if ( i==strlen(s) ) {
    return 1;
  }
  else {
    return 0;
  }
}

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and returns the value of lo
//
// We assume that the string given is syntactically valid, 
// having been already processed and marked as an integer 
// range in the lexical analyzer
int lower_integer_of_range(char * range_string) {

  int lower;
  int sign;
  int i;

  lower = 0;
  sign = 1;

  if (range_string == NULL ) {
    print_error_message(stderr,
			"Internal error: Empty integer-range has no lower bound!\n");
    return 0;
  }
  
  i=0;
  // if the first character is '-' mark the sign and go on
  if ( range_string[0]=='-' ) {
    sign = -1;
    i++;
  }
  while ( range_string[i]!='.') {
    lower = lower*10 + range_string[i] - '0';
    i++;
  }

  return lower * sign;
}

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and returns the value of hi
//
// We assume that the string given is syntactically valid, 
// having been already processed and marked as an integer 
// range in the lexical analyzer
int upper_integer_of_range(char * range_string) {

  int upper;
  int sign;
  int i;

  upper = 0;
  sign = 1;

  if (range_string == NULL ) {
    print_error_message(stderr,
			"Internal error: Empty integer-range has no upper bound!\n");
    return 0;
  }
  
  i=0;
  // if the first character is '-', skip over it
  if ( range_string[0]=='-' ) {
    i++;
  }
  // skip over lower bound "lo"
  while ( range_string[i]!='.') {
    i++;
  }

  i=i+2; //skip over ".."

  if ( range_string[i]=='-' ) {
    sign = -1;
    i++;
  }
  while ( range_string[i]!='\0') {
    upper = upper*10 + range_string[i] - '0';
    i++;
  }

  return upper * sign;
}

// Takes an integer range of the form "lo..hi" and turns it
// returns the string "ccalc_int_range_lotohi"
char * convert_integer_range_to_ccalc_acceptable_form(char * range) {
  
  char * new_sort_name;
  
  new_sort_name
    = (char *) malloc( sizeof(char) * strlen(range) + 16 + 1 );
  strcpy(new_sort_name, "ccalc_int_range_");
  strcat(new_sort_name, integer_to_string(lower_integer_of_range(range)));
  strcat(new_sort_name, "to");
  strcat(new_sort_name, integer_to_string(upper_integer_of_range(range)));
  
  return new_sort_name;
}

/*  If there are built-in integer ranges used in the action description,
    then the integer objects in these need to be declared explicitly.

    This function prints out an object declaration section to with these integers.

 */
void print_integer_range_object_list_for_ccalc(FILE * stream, struct sort_list_node * list) {

  struct sort_list_node * current;

  // If the sort is an integer range, we need to print it out differently
  char * new_sort_name;
  
  int header_printed_already = 0;

  current = list;
  if ( current != NULL ) {
    if ( is_an_integer_range(current->node_data->name) ) {
      if ( !header_printed_already ) {
	fprintf(stream, ":- objects\n");
	fprintf(stream, "    %s", current->node_data->name);
	fprintf(stream, 
		" :: %s",
		convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
	header_printed_already = 1;
      }
      else {
	fprintf(stream, ";\n");
	fprintf(stream, "    %s", current->node_data->name);
	fprintf(stream, 
		" :: %s",
		convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
      }
    }
    current = current->next;
    while ( current != NULL ) {
      if ( is_an_integer_range(current->node_data->name) ) {
	if ( !header_printed_already ) {
	  fprintf(stream, ":- objects\n");
	  fprintf(stream, "    %s", current->node_data->name);
	  fprintf(stream, 
		  " :: %s",
		  convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
	  header_printed_already = 1;
	}
	else {
	  fprintf(stream, ";\n");
	  fprintf(stream, "    %s", current->node_data->name);
	  fprintf(stream, 
		  " :: %s",
		  convert_integer_range_to_ccalc_acceptable_form(current->node_data->name));
	}
      }
      current = current->next;
    }
    if ( header_printed_already ) {
      fprintf(stream, ".\n");
    }
  }
}


// Takes a string and check if it is an arithmetic operator
int is_an_arithmetic_operator(char * s) {

  if ( strcmp(s, "+") == 0 
       || strcmp(s, "*") == 0 ) {
    return 1;
  }
  else {
    return 0;
  }
}

/****** Managing lists of inclusions **************/


/* initialize the list of inclusions */
void init_inclusions() {
  inclusion_list_first = inclusion_list_last = NULL;
}

/* initialize the global list of inclusions */
void init_global_inclusions() {
  global_inclusion_list_first = global_inclusion_list_last = NULL;
}

/* Looks up an inclusion and returns a pointer to its list entry.
   Returns NULL if not found. */
struct inclusion_list_node * lookup_inclusion(char * lookup_supersort, char * lookup_subsort) {
  struct inclusion_list_node * current;

  current = inclusion_list_first;
  if ( lookup_supersort != NULL && lookup_subsort != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->supersort, lookup_supersort) == 0 
	  && strcmp(current->node_data->subsort, lookup_subsort) == 0) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Looks up an inclusion in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct inclusion_list_node * lookup_inclusion_in_module(char * lookup_supersort, 
							char * lookup_subsort, 
							struct mad_module * module) {
  struct inclusion_list_node * current;

  current = module->module_inclusions;
  if ( lookup_supersort != NULL && lookup_subsort != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->supersort, lookup_supersort) == 0 
	  && strcmp(current->node_data->subsort, lookup_subsort) == 0) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* Called when inserting a new inclusion.  Makes a new inclusion */
struct mad_inclusion * make_new_inclusion (char * supersort, char * subsort) {
  struct mad_inclusion * new_inclusion ;

  new_inclusion = (struct mad_inclusion *) malloc( sizeof(struct mad_inclusion) );
  new_inclusion->supersort = (char *) malloc( sizeof(char) * ( strlen(supersort) + 1 ) );
  strcpy( new_inclusion->supersort, supersort );
  new_inclusion->subsort = (char *) malloc( sizeof(char) * ( strlen(subsort) + 1 ) );
  strcpy( new_inclusion->subsort, subsort );
  return new_inclusion;
}


/* This inserts a new inclusion into the list of inclusions */
void insert_inclusion (char * supersort, char * subsort) {
  struct inclusion_list_node * new_node;

  new_node = (struct inclusion_list_node *) malloc( sizeof(struct inclusion_list_node) );
  new_node->node_data = make_new_inclusion(supersort, subsort);
  new_node->next = NULL;

  /* inclusion_list_last is the pointer to the end of the list */
  if ( inclusion_list_last == NULL ) { /* the list is empty */
    inclusion_list_first = new_node;
    inclusion_list_last = new_node;
  }
  else {
  inclusion_list_last->next = new_node;
  inclusion_list_last = new_node;
  }

}


/* Makes a copy of a given inclusion list */
struct inclusion_list_node * copy_inclusion_list(struct inclusion_list_node * inclusion_list) {
  struct inclusion_list_node * copied_inclusion_list;
  
  if ( inclusion_list != NULL ) {
    /* First copy current node */
    copied_inclusion_list = (struct inclusion_list_node *) malloc( sizeof(struct inclusion_list_node) );
    copied_inclusion_list->node_data = make_new_inclusion(inclusion_list->node_data->supersort,
							  inclusion_list->node_data->subsort);
    /* Then copy the rest */
    copied_inclusion_list->next = copy_inclusion_list(inclusion_list->next);
  }
  else {
    copied_inclusion_list = NULL;
  }
  
  return copied_inclusion_list;
}

/* Makes the list of inclusions a copy of the global list of inclusions */ 
void copy_global_inclusions() {

  struct inclusion_list_node * inclusion_list;

  init_inclusions();

  inclusion_list = global_inclusion_list_first;
  while ( inclusion_list != NULL ) {
    insert_inclusion(inclusion_list->node_data->supersort, inclusion_list->node_data->subsort);
    inclusion_list = inclusion_list->next;
  }
}

/* Checks if a given sort is a subsort of another given sort.  
   If the two sorts are the same, then it won't consider them 
   to be subsorts of each other, unless there's a cycle in the
   inclusion graph. (So this can be used as a cycle detector.)

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
 */
int is_a_subsort_of(char * first_sort, char * second_sort) {

  struct inclusion_list_node * current;

  /* Make sure we are given two valid sorts. */
  if ( !is_a_known_variable_sort(first_sort)
       || !is_a_known_variable_sort(second_sort) ) {
    print_error_message(stderr, "Internal error: subsort checking called on non-sorts (%s and %s).\n", first_sort, second_sort);
    return 0;
  }
  
  current = inclusion_list_first;
  while ( current != NULL ) {
    /* Check if current inclusion is about a subsort of second_sort. */
    if ( strcmp(current->node_data->supersort, second_sort) == 0 ) {
      /* Since second_sort has "some" subsort, we need to follow this 
	 in the inclusion graph to see if it leads to first_sort. */

      /* Check if first_sort is a direct subsort */
      if (strcmp(current->node_data->subsort, first_sort) == 0 ) {
	return 1;
      }
      
      /* This inclusion does not say that first_sort is a direct subsort of second_sort,
	 but we need to check if it is an indirect subsort -- by being a subsort of 
	 current->node_data->subsort.
      */
      // We shouldn't have any inclusions with the same supersort and subsort, 
      // but in case of such an error, the check performed below prevents
      // an infinite recursion:
      if ( strcmp(current->node_data->subsort, current->node_data->supersort)== 0) {
	return 0;
      }
      else {
	if ( is_a_subsort_of(first_sort, current->node_data->subsort) ) {
	  return 1;
	}
      }
      //}
    }
    /* This inclusion didn't match, so we continue to the next. */
    current = current->next; 
  }
  /* Either no inclusion matched second_sort as its supersort, or it didn't lead to finding
     first_sort as a direct or indirect subsort. */
  return 0;
}


/* Checks if a given sort is a subsort of another given sort, in a given module.
   If the two sorts are the same, then it won't consider them 
   to be subsorts of each other, unless there's a cycle in the
   inclusion graph. (So this can be used as a cycle detector.) 

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
*/
int is_a_subsort_of_in_module(char * first_sort, 
			      char * second_sort, 
			      struct mad_module * module) {

  struct inclusion_list_node * current;

  if ( !is_a_known_variable_sort_in_module(first_sort,module)
       || !is_a_known_variable_sort_in_module(second_sort, module) ) {
    print_error_message(stderr, "Internal error: subsort checking called on non-sorts (%s and %s).\n", first_sort, second_sort);
    return 0;
  }
  
  current = module->module_inclusions;
  while ( current != NULL ) {
    /* Check if current inclusion is about a subsort of second_sort. */
    if ( strcmp(current->node_data->supersort, second_sort) == 0 ) {
      /* Since second_sort has "some" subsort, we need to follow this 
	 in the inclusion graph to see if it leads to first_sort. */

      /* Check if first_sort is a direct subsort */
      if (strcmp(current->node_data->subsort, first_sort) == 0 ) {
	return 1;
      }
      
      /* This inclusion does not say that first_sort is a direct subsort of second_sort,
	 but we need to check if it is an indirect subsort -- by being a subsort of 
	 current->node_data->subsort. 
      */
      // We shouldn't have any inclusions with the same supersort and subsort, 
      // but in case of such an error, the check performed below prevents
      // an infinite recursion:
      if ( strcmp(current->node_data->subsort, current->node_data->supersort)== 0) {
	return 0;
      }
      else {
	if ( is_a_subsort_of_in_module(first_sort, current->node_data->subsort, module) ) {
	  return 1;
	}
      }
      //}
    }
    /* This inclusion didn't match, so we continue to the next. */
    current = current->next; 
  }
  /* Either no inclusion matched second_sort as its supersort, or it didn't lead to finding
     first_sort as a direct or indirect subsort. */
  return 0;
}


/* Called to merge inclusions of an imported module with the current inclusions */
void merge_inclusions_from_module(struct mad_module * module) {

  struct inclusion_list_node * current;
  struct id_list_node * id;

  current = module->module_inclusions;

  while ( current != NULL ) {
    /* First check if that inclusion has already been declared. */

    if ( lookup_inclusion(current->node_data->supersort, current->node_data->subsort) == NULL ) {
      /* It has not been declared previously. */ 
      
      /* Before adding a new inclusion "a INCLUSION_OPERATOR b", we need to check whether this will lead to a cycle.
	 If, before adding that inclusion, b is a subsort of a, then adding "a INCLUSION_OPERATOR b" will lead to a
	 cycle in the inclusion graph. */
      
      if ( is_a_subsort_of(current->node_data->supersort, current->node_data->subsort) ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Merging \"%s << %s\" (from module %s) into module %s leads to a cycle.\n",
				   current->node_data->subsort,
				   current->node_data->supersort,
				   module->name,
				   name_of_module_currently_being_parsed);
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "   %s is already a subsort of %s.\n",
				   current->node_data->supersort,
				   current->node_data->subsort);
      }
      else {
	insert_inclusion(current->node_data->supersort, current->node_data->subsort);
      }
      //}
    }
    else {
      /* The inclusion has already been declared so there's no need to add it. */
    }
    current = current->next;
  }
}

/* for debugging */
void print_current_inclusions(FILE * stream) {
  struct inclusion_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Inclusions:\n");
  current = inclusion_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->subsort != NULL ) {
	fprintf(stream, "%s", current->node_data->subsort);
      }
      else {
	fprintf(stream, "(NULL-subsort-name");
      }
      fprintf(stream, " << ");
      if ( current->node_data->supersort != NULL ) {
	fprintf(stream, "%s", current->node_data->supersort);
      }
      else {
	fprintf(stream, "NULL-supersort-name!");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of inclusions (e.g. of a specific module). */
void print_inclusion_list(FILE * stream, struct inclusion_list_node * list) {
  struct inclusion_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Inclusions:\n");
  current = list;
  while ( current != NULL ) {
    fprintf(stream, "%d: %s << %s\n", counter, current->node_data->subsort, current->node_data->supersort);
    counter++;
    current = current->next;
  }
}

/* for printing an inclusion declaration section to be used as ccalc input. */
void print_inclusion_list_for_ccalc(FILE * stream,
				    struct inclusion_list_node * inclusion_list) {
  struct inclusion_list_node * current;

  current = inclusion_list;
  if ( current != NULL ) {
    fprintf(stream, ":- sorts\n");
    fprintf(stream, 
	    "    %s >> %s",
	    prepare_string_for_ccalc(current->node_data->supersort),
	    prepare_string_for_ccalc(current->node_data->subsort) );
    current = current->next; 
    while ( current != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream,
	      "    %s >> %s",
	      prepare_string_for_ccalc(current->node_data->supersort),
	      prepare_string_for_ccalc(current->node_data->subsort) );
      current = current->next;
    }
    fprintf(stream, ".\n");
  }
}

/* for printing out a sort declaration section to be used as ccalc input. */
void print_sort_and_inclusion_list_for_ccalc(FILE * stream,
					     struct sort_list_node * sort_list,
					     struct inclusion_list_node * inclusion_list) {
  struct sort_list_node * current_sort;
  struct inclusion_list_node * current_inclusion;

  current_inclusion = inclusion_list;
  current_sort = sort_list;
  if ( (current_inclusion != NULL) || (current_sort != NULL) ) {
    fprintf(stream, ":- sorts\n");
    /* First print inclusions if there are any. */
    if ( current_inclusion != NULL ) {
      fprintf(stream, 
	      "    %s >> %s",
	      prepare_string_for_ccalc(current_inclusion->node_data->supersort),
	      prepare_string_for_ccalc(current_inclusion->node_data->subsort) );
      current_inclusion = current_inclusion->next; 
    }
    while ( current_inclusion != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream, 
	      "    %s >> %s",
	      prepare_string_for_ccalc(current_inclusion->node_data->supersort),
	      prepare_string_for_ccalc(current_inclusion->node_data->subsort) );
      current_inclusion = current_inclusion->next;
    }

    /* Now print sorts */

    /* Check if we have printed any inclusions already. */
    if ( inclusion_list != NULL ) {
      while ( current_sort != NULL ) {
	fprintf(stream, ";\n");
	fprintf(stream, "    %s", prepare_string_for_ccalc(current_sort->node_data->name) );
	current_sort = current_sort->next;
      }
    }
    else {
      if ( current_sort != NULL ) {
	fprintf(stream, "    %s", prepare_string_for_ccalc(current_sort->node_data->name) );
	current_sort = current_sort->next;
      }
      while ( current_sort != NULL ) {
	fprintf(stream, ";\n");
	fprintf(stream, "    %s", prepare_string_for_ccalc(current_sort->node_data->name) );
	current_sort = current_sort->next;
      }
    }

    /* At the very end, finish the sort declarations by adding a dot. */
    fprintf(stream, ".\n");
  }
}



/****** Managing lists of sort dependencies **************/


/* initialize the list of sort dependencies */
void init_sort_dependencies() {
  sort_dependency_list_first = sort_dependency_list_last = NULL;
}

/* initialize the global list of sort dependencies */
void init_global_sort_dependencies() {
  global_sort_dependency_list_first = global_sort_dependency_list_last = NULL;
}

/* Looks up a sort dependency and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_dependency_list_node * lookup_sort_dependency(char * lookup_supersort, 
							  char * lookup_subsort) {
  struct sort_dependency_list_node * current;

  current = sort_dependency_list_first;
  if ( lookup_supersort != NULL && lookup_subsort != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->supersort, lookup_supersort) == 0 
	  && strcmp(current->node_data->subsort, lookup_subsort) == 0) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Looks up a sort dependency in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_dependency_list_node * lookup_sort_dependency_in_module(char * lookup_supersort, 
								    char * lookup_subsort, 
								    struct mad_module * module) {
  struct sort_dependency_list_node * current;

  current = module->module_sort_dependencies;
  if ( lookup_supersort != NULL && lookup_subsort != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->supersort, lookup_supersort) == 0 
	  && strcmp(current->node_data->subsort, lookup_subsort) == 0) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* Called when inserting a new sort_dependency.  Makes a new sort_dependency */
struct mad_sort_dependency * make_new_sort_dependency (char * supersort, char * subsort) {
  struct mad_sort_dependency * new_sort_dependency ;

  new_sort_dependency = (struct mad_sort_dependency *) malloc( sizeof(struct mad_sort_dependency) );
  new_sort_dependency->supersort = (char *) malloc( sizeof(char) * ( strlen(supersort) + 1 ) );
  strcpy( new_sort_dependency->supersort, supersort );
  new_sort_dependency->subsort = (char *) malloc( sizeof(char) * ( strlen(subsort) + 1 ) );
  strcpy( new_sort_dependency->subsort, subsort );
  return new_sort_dependency;
}


/* This inserts a new sort_dependency into the list of sort_dependencies */
void insert_sort_dependency (char * supersort, char * subsort) {
  struct sort_dependency_list_node * new_node;

  new_node = (struct sort_dependency_list_node *) malloc( sizeof(struct sort_dependency_list_node) );
  new_node->node_data = make_new_sort_dependency(supersort, subsort);
  new_node->next = NULL;

  /* sort_dependency_list_last is the pointer to the end of the list */
  if ( sort_dependency_list_last == NULL ) { /* the list is empty */
    sort_dependency_list_first = new_node;
    sort_dependency_list_last = new_node;
  }
  else {
  sort_dependency_list_last->next = new_node;
  sort_dependency_list_last = new_node;
  }

}


/* Makes a copy of a given sort_dependency list */
struct sort_dependency_list_node * copy_sort_dependency_list(struct sort_dependency_list_node * sort_dependency_list) {
  struct sort_dependency_list_node * copied_sort_dependency_list;
  
  if ( sort_dependency_list != NULL ) {
    /* First copy current node */
    copied_sort_dependency_list = (struct sort_dependency_list_node *) malloc( sizeof(struct sort_dependency_list_node) );
    copied_sort_dependency_list->node_data = make_new_sort_dependency(sort_dependency_list->node_data->supersort,
							  sort_dependency_list->node_data->subsort);
    /* Then copy the rest */
    copied_sort_dependency_list->next = copy_sort_dependency_list(sort_dependency_list->next);
  }
  else {
    copied_sort_dependency_list = NULL;
  }
  
  return copied_sort_dependency_list;
}

/* Makes the list of sort dependencies a copy of the global list of sort dependencies */ 
void copy_global_sort_dependencies() {

  struct sort_dependency_list_node * sort_dependency_list;

  init_sort_dependencies();

  sort_dependency_list = global_sort_dependency_list_first;
  while ( sort_dependency_list != NULL ) {
    insert_sort_dependency(sort_dependency_list->node_data->supersort, sort_dependency_list->node_data->subsort);
    sort_dependency_list = sort_dependency_list->next;
  }
}

/* Checks if a given sort is depended upon by another given sort.  
   If the two sorts are the same, then it won't consider them 
   to depend upon each other, unless there's a cycle in the
   sort dependency graph. (So this can be used as a cycle detector.)

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
 */
int is_a_sort_depended_upon_by_another_sort(char * first_sort, char * second_sort) {

  struct sort_dependency_list_node * current;

  /* Make sure we are given two valid sorts. */
  if ( !is_a_known_variable_sort(first_sort)
       || !is_a_known_variable_sort(second_sort) ) {
    print_error_message(stderr, "Internal error: sort dependency checking called on non-sorts (%s and %s).\n", first_sort, second_sort);
    return 0;
  }
  
  current = sort_dependency_list_first;
  while ( current != NULL ) {
    /* Check if current sort dependency is about a subsort of second_sort. */
    if ( strcmp(current->node_data->supersort, second_sort) == 0 ) {
      /* Since second_sort has "some" dependency, we need to follow this 
	 in the sort dependency graph to see if it leads to first_sort. */

      /* Check if first_sort is a direct dependency of the second_sort */
      if (strcmp(current->node_data->subsort, first_sort) == 0 ) {
	return 1;
      }
      
      /* This sort dependency does not say that first_sort is a direct dependencey
	 of second_sort, but we need to check if it is an indirect dependency -- 
	 by being a dependency of current->node_data->subsort.
      */
      // We shouldn't have any sort dependencies with the same supersort and subsort, 
      // but in case of such an error, the check performed below prevents
      // an infinite recursion:
      if ( strcmp(current->node_data->subsort, current->node_data->supersort)== 0) {
	return 0;
      }
      else {
	if (  is_a_sort_depended_upon_by_another_sort(first_sort, 
						      current->node_data->subsort) ) {
	  return 1;
	}
      }
      //}
    }
    /* This sort dependency didn't match, so we continue to the next. */
    current = current->next; 
  }
  /* Either no sort dependency matched second_sort as its supersort, or it didn't
     lead to finding first_sort as a direct or indirect dependency. */
  return 0;
}


/* Checks if a given sort is depended upon by another given sort, 
   in a given module.
   If the two sorts are the same, then it won't consider them 
   to depend upon each other, unless there's a cycle in the
   sort_dependency graph. (So this can be used as a cycle detector.) 

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
*/
int is_a_sort_depended_upon_by_another_sort_in_module(char * first_sort, 
						      char * second_sort, 
						      struct mad_module * module) {

  struct sort_dependency_list_node * current;

  if ( !is_a_known_variable_sort_in_module(first_sort,module)
       || !is_a_known_variable_sort_in_module(second_sort, module) ) {
    print_error_message(stderr, "Internal error: sort dependency checking called on non-sorts (%s and %s).\n", first_sort, second_sort);
    return 0;
  }
  
  current = module->module_sort_dependencies;
  while ( current != NULL ) {
    /* Check if current sort dependency is about a subsort of second_sort. */
    if ( strcmp(current->node_data->supersort, second_sort) == 0 ) {
      /* Since second_sort has "some" dependency, we need to follow this 
	 in the sort dependency graph to see if it leads to first_sort. */

      /* Check if first_sort is a direct dependency */
      if (strcmp(current->node_data->subsort, first_sort) == 0 ) {
	return 1;
      }
      
      /* This sort dependency does not say that first_sort is a direct dependency
	 of second_sort, but we need to check if it is an indirect dependency -- 
	 by being a dependency of current->node_data->subsort. 
      */
      // We shouldn't have any sort dependencies with the same supersort and subsort, 
      // but in case of such an error, the check performed below prevents
      // an infinite recursion:
      if ( strcmp(current->node_data->subsort, current->node_data->supersort)== 0) {
	return 0;
      }
      else {
	if ( is_a_sort_depended_upon_by_another_sort_in_module(first_sort, 
							       current->node_data->subsort, 
							       module) ) {
	  return 1;
	}
      }
      //}
    }
    /* This sort dependency didn't match, so we continue to the next. */
    current = current->next; 
  }
  /* Either no sort dependency matched second_sort as its supersort, or it didn't lead to finding
     first_sort as a direct or indirect dependency. */
  return 0;
}

// Given a list of argument sorts for an object in a string list,
// and the sort of the object,
// this function records sort dependencies between each argument 
// and the object sort
void update_sort_dependencies_according_to_object_declaration(struct string_list_node * args,
							      char * object_sort_name) {
  struct string_list_node * current_arg;

  if ( args == NULL ) {
    // Nothing to update if there are no argument sorts
    return;
  }
  else {
    current_arg = args;
    while ( current_arg != NULL ) {
      // Integer ranges are built-in sorts so they cannot be redefined
      // Therefore they can never occur as the sort of an object
      // so we don't add them as dependencies

      // only record sort dependency if it's not already there
      if ( !is_an_integer_range(current_arg->node_data) 
	   && lookup_sort_dependency(object_sort_name, current_arg->node_data) == NULL ) {

	// We should never add a self-dependency)
	if ( strcmp(object_sort_name, current_arg->node_data) == 0 ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Sort %s cannot be an argument for an object of the same sort.\n",
				     object_sort_name);
	  updating_sort_dependencies_leads_to_an_error = 1;
	  break;
	}
	// Before adding a new dependency "a is depended upon by b", 
	// we need to check whether this will lead to a cycle.
	// If, before adding that dependency, b is a dependency of a, 
	// then adding "a is depended upon by b" will lead to a
	// cycle in the dependency graph.
	else if ( is_a_sort_depended_upon_by_another_sort(object_sort_name,
						     current_arg->node_data) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Sort %s cannot be an argument for an object of sort %s.\n",
				     current_arg->node_data,
				     object_sort_name);
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "  (%s is already depended upon by %s.)\n",
				     object_sort_name,
				     current_arg->node_data);
	  updating_sort_dependencies_leads_to_an_error = 1;
	  break;
	}
	else {
	  insert_sort_dependency(object_sort_name, current_arg->node_data);
	}
      }
      current_arg = current_arg->next;
    }
  }
}

// When we have an inclusion, this means that the supersort of
// the inclusion depends on the dependencies of the subsort
//
// This function updates looks at all inclusions and updates
// the sort dependencies accordingly
void update_sort_dependencies_according_to_inclusions() {

  struct inclusion_list_node * current_inclusion;

  struct sort_dependency_list_node * current_dependency;

  current_inclusion = inclusion_list_first;

  while ( current_inclusion != NULL ) {
    // This inclusion is 
    //    "current_inclusion->node_data->subsort << current_inclusion->node_data->supersort"
    //
    // For each dependency 
    //    "s is depended upon by current_inclusion->node_data->subsort"
    // we need to add a new dependency
    //    "s is depended upon by current_inclusion->node_data->supersort"

    current_dependency = sort_dependency_list_first;
    while ( current_dependency != NULL ) {
      if ( strcmp(current_inclusion->node_data->subsort,
		  current_dependency->node_data->supersort) == 0 ) {
	
	/* First check if the new sort dependency has already been recorded. */
	if ( lookup_sort_dependency(current_inclusion->node_data->supersort,
				    current_dependency->node_data->subsort) 
	     == NULL ) {
	  // It has not been declared previously.
	  
	  // Before adding a new dependency "a is depended upon by b", 
	  // we need to check whether this will lead to a cycle.
	  // If, before adding that dependency, b is a dependency of a, 
	  // then adding "a is depended upon by b" will lead to a
	  // cycle in the dependency graph.
	  
	  if ( is_a_sort_depended_upon_by_another_sort(current_inclusion->node_data->supersort,
						       current_dependency->node_data->subsort) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "When inclusions are taken into account, sorts %s and %s cyclically depend on each other.\n",
				       current_inclusion->node_data->supersort,
				       current_dependency->node_data->subsort);
	    updating_sort_dependencies_leads_to_an_error = 1;
	    break;
	  }
	  else {
	    insert_sort_dependency(current_inclusion->node_data->supersort,
				   current_dependency->node_data->subsort);
	  }
	}
      }
      current_dependency = current_dependency->next;
    }
    current_inclusion = current_inclusion->next;
  }
}


/* Called to merge sort dependencies of an imported module with the current sort dependencies */
void merge_sort_dependencies_from_module(struct mad_module * module) {

  struct sort_dependency_list_node * current;
  struct id_list_node * id;

  current = module->module_sort_dependencies;

  while ( current != NULL ) {
    /* First check if that sort dependency has already been declared. */

    if ( lookup_sort_dependency(current->node_data->supersort, current->node_data->subsort) == NULL ) {
      /* It has not been declared previously. */ 
      
      /* Before adding a new sort_dependency "a is depended upon by b", 
	 we need to check whether this will lead to a cycle.
	 If, before adding that sort_dependency, b is a dependency of a, 
	 then adding "a is depended upon by b" will lead to a
	 cycle in the sort dependency graph. */
      
      if ( is_a_subsort_of(current->node_data->supersort, current->node_data->subsort) ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Merging module %s into module %s leads to a cyclical dependency between objects of sorts %s and %s.\n",
				   module->name,
				   name_of_module_currently_being_parsed,
				   current->node_data->subsort,
				   current->node_data->supersort);
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "   %s already depended on %s before merging.\n",
				   current->node_data->subsort,
				   current->node_data->supersort);
	merging_sort_dependencies_leads_to_error = 1;
      }
      else {
	insert_sort_dependency(current->node_data->supersort, current->node_data->subsort);
      }
      //}
    }
    else {
      /* The sort dependency has already been declared so there's no need to add it. */
    }
    current = current->next;
  }
}

/* for debugging */
void print_current_sort_dependencies(FILE * stream) {
  struct sort_dependency_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Sort dependencies:\n");
  current = sort_dependency_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->subsort != NULL ) {
	fprintf(stream, "%s", current->node_data->subsort);
      }
      else {
	fprintf(stream, "(NULL-subsort-name");
      }
      fprintf(stream, " is depended upon by ");
      if ( current->node_data->supersort != NULL ) {
	fprintf(stream, "%s", current->node_data->supersort);
      }
      else {
	fprintf(stream, "NULL-supersort-name!");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of sort dependencies (e.g. of a specific module). */
void print_sort_dependency_list(FILE * stream, struct sort_dependency_list_node * list) {
  struct sort_dependency_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Sort dependencies:\n");
  current = list;
  while ( current != NULL ) {
    fprintf(stream, "%d: %s is depended upon by %s\n", counter, current->node_data->subsort, current->node_data->supersort);
    counter++;
    current = current->next;
  }
}


/****** Managing lists of objects **************/


/* initialize the list of objects */
void init_objects() {
  object_list_first = object_list_last = NULL;
}

/* initialize a list of objects */
void init_object_list(struct object_list_node ** first, struct object_list_node ** last) {
  (*first) = (*last) = NULL;
}


/* Looks up an object and returns a pointer to its list entry.
   Returns NULL if not found. */
struct object_list_node * lookup_object(char * lookup_name) {
  struct object_list_node * current;

  current = object_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* This inserts a new object into the list of objects */
void insert_object (char * name, int num_arguments, struct string_list_node * arguments) {
  struct object_list_node * new_node;

  new_node = (struct object_list_node *) malloc( sizeof(struct object_list_node) );
  new_node->node_data = make_new_object(name, num_arguments, arguments);
  new_node->next = NULL;

  /* object_list_last is the pointer to the end of the list */
  if ( object_list_last == NULL ) { /* the list is empty */
    object_list_first = new_node;
    object_list_last = new_node;
  }
  else {
    object_list_last->next = new_node;
    object_list_last = new_node;
  }

  /* update the list of identifiers to indicate this
     identifier is a object */
  update_id_type(name, OBJECT_ID);
  
}

/* This inserts a new object into a given list of objects */
void insert_object_into_list (struct object_list_node ** first, 
			      struct object_list_node ** last, 
			      char * name, 
			      int num_arguments, 
			      struct string_list_node * arguments) {
  struct object_list_node * new_node;
  
  new_node = (struct object_list_node *) malloc( sizeof(struct object_list_node) );
  new_node->node_data = make_new_object(name, num_arguments, arguments);
  new_node->next = NULL;
  
  /* last is the pointer to the end of the list */
  if ( (*last) == NULL ) { /* the list is empty */
    (*first) = new_node;
    (*last) = new_node;
  }
  else {
    (*last)->next = new_node;
    (*last) = new_node;
  }
}

/* Called when inserting a new object.  Makes a new object */
struct mad_object * make_new_object (char * name, 
				     int num_arguments,
				     struct string_list_node * arguments) {
  struct mad_object * new_object ;
  
  new_object = (struct mad_object *) malloc( sizeof(struct mad_object) );

  new_object->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_object->name, name );

  new_object->num_arguments = num_arguments;

  new_object->arguments = arguments;
  
  return new_object;
}


/* Add information about sort of a group of objects.
   This is separate from making a new object because we get this
   information later during the declaration. */
void * update_objects (struct string_list_node * current_entry, char * sort) {

  struct object_list_node * this_object;

  while(current_entry != NULL) { 

    this_object = lookup_object(current_entry->node_data);

    if (this_object != NULL) {
      this_object->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(sort) + 1 ) );
      strcpy( this_object->node_data->sort, sort );
    }
    else {
      print_error_message(stderr, "Internal error: trying to update a non-existent object %s!\n",
	     current_entry->node_data);
    }

    current_entry = current_entry->next;
  }
}

/* Checks whether an object is fully instantiated, i.e., has no variables. */
int is_fully_instantiated_object(char * name, 
				 struct mad_constant_argument * arguments) {
  struct object_list_node * given_object;
  
  struct mad_constant_argument * current_argument;

  struct object_list_node * argument_object;

  struct string_list_node * current_declaration_argument;

  given_object = lookup_object(name);
  if ( given_object == NULL ) {
    print_error_message(stderr, 
			"Internal error: checking instantiation of null object!\n");
    return 0;
  }
  else {
    // If there are arguments, go through them and check instantiation for each
    // (and also check that the arguments match the declaration)
    if ( arguments != NULL ) {
      current_argument = arguments;
      current_declaration_argument = given_object->node_data->arguments;
      while ( current_argument != NULL ) {
	if ( is_an_integer(current_argument->name) ) {
	  // check that the argument is an object/integer and that it
	  // agrees with the declaration
	  if ( !(is_an_integer_range(current_declaration_argument->node_data) 
		 && (lower_integer_of_range(current_declaration_argument->node_data)
		     <= atoi(current_argument->name))
		 && (upper_integer_of_range(current_declaration_argument->node_data)
		     >= atoi(current_argument->name)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has integer argument not within range.\n",
				       name);
	    return 0;
	  }
	}
	else {
	  argument_object = lookup_object(current_argument->name);
	  if ( argument_object == NULL ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has a non-instantiated argument %s.\n",
				       name);
	    return 0;
	  }
	  else if ( strcmp(argument_object->node_data->sort, 
			   current_declaration_argument->node_data) != 0 
		    && !(is_a_known_object_sort(argument_object->node_data->sort)
			 && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
			 && is_a_subsort_of(argument_object->node_data->sort,
					    current_declaration_argument->node_data) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has an argument %s of the wrong sort.\n",
				       name,
				       current_argument->name);
	    return 0;
	  }
	}
	// Now check that this argument is fully instantiated
	if ( current_argument->arguments != NULL ) {
	  if ( !is_fully_instantiated_object(current_argument->name, 
					     current_argument->arguments) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s occurring on the LHS of \"is\" not fully instantiated.\n",
				       current_argument->name);
	    return 0;
	  }
	}
	// Go on to next argument
	current_argument = current_argument->next_argument;
      }
    }

    return 1;
  }
}

/* Checks whether an object is fully instantiated, i.e., has no variables.
   when it comes from a given module that is being imported at the moment. */
int is_fully_instantiated_object_in_module_import(char * name, 
						  struct mad_constant_argument * arguments,
						  struct mad_module * module) {
  struct object_list_node * given_object;
  
  struct mad_constant_argument * current_argument;

  struct object_list_node * argument_object;

  struct string_list_node * current_declaration_argument;

  given_object = lookup_object(name);
  if ( given_object == NULL ) {
    print_error_message(stderr, 
			"Internal error: checking instantiation of null object!\n");
    return 0;
  }
  else {
    // If there are arguments, go through them and check instantiation for each
    // (and also check that the arguments match the declaration)
    if ( arguments != NULL ) {
      current_argument = arguments;
      current_declaration_argument = given_object->node_data->arguments;
      while ( current_argument != NULL ) {
	if ( is_an_integer(current_argument->name) ) {
	  // check that the argument is an object/integer and that it
	  // agrees with the declaration
	  if ( !(is_an_integer_range(current_declaration_argument->node_data) 
		 && (lower_integer_of_range(current_declaration_argument->node_data)
		     <= atoi(current_argument->name))
		 && (upper_integer_of_range(current_declaration_argument->node_data)
		     >= atoi(current_argument->name)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has integer argument not within range.\n",
				       name);
	    return 0;
	  }
	}
	else {
	  argument_object = lookup_object(current_argument->name);
	  if ( argument_object == NULL ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has a non-instantiated argument %s.\n",
				       current_argument->name);
	    return 0;
	  }
	  else if ( strcmp(argument_object->node_data->sort, 
			   current_declaration_argument->node_data) != 0 
		    && !(is_a_known_object_sort(argument_object->node_data->sort)
			 && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
			 && is_a_subsort_of(argument_object->node_data->sort,
					    current_declaration_argument->node_data) )
		    && !(is_a_known_object_sort_in_module(argument_object->node_data->sort, module)
			 && is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module)
			 && is_a_subsort_of_in_module(argument_object->node_data->sort,
						      current_declaration_argument->node_data,
						      module) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s has an argument %s of the wrong sort.\n",
				       name,
				       current_argument->name);
	    return 0;
	  }
	}
	// Now check that this argument is fully instantiated
	if ( current_argument->arguments != NULL ) {
	  if ( !is_fully_instantiated_object_in_module_import(current_argument->name, 
							      current_argument->arguments,
							      module) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s occurring on the LHS of \"is\" not fully instantiated.\n",
				       current_argument->name);
	    return 0;
	  }
	}
	// Go on to next argument
	current_argument = current_argument->next_argument;
      }
    }

    return 1;
  }
}

/* Makes a copy of a given object list */
struct object_list_node * copy_object_list(struct object_list_node * object_list) {
  struct object_list_node * copied_object_list;
  
  if (object_list != NULL) {
    /* First copy current node's name and sort */
    copied_object_list = (struct object_list_node *) malloc( sizeof(struct object_list_node) );
    copied_object_list->node_data = make_new_object(object_list->node_data->name,
						    object_list->node_data->num_arguments,
						    copy_string_list(object_list->node_data->arguments));
    copied_object_list->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(object_list->node_data->sort) + 1 ) );
    strcpy( copied_object_list->node_data->sort, object_list->node_data->sort );

    /* Then copy the rest */
    copied_object_list->next = copy_object_list(object_list->next);
  }
  else {
    copied_object_list = NULL;
  }

  return copied_object_list;
}


/* Called to merge objects of an imported module with the current objects */
void merge_objects_from_module(struct mad_module * module) {

  struct object_list_node * current;
  struct id_list_node * id;
  struct object_list_node * existing_object;

  struct string_list_node * current_existing_argument;
  struct string_list_node * current_argument;

  struct string_list_node * current_string_list_first;
  struct string_list_node * current_string_list_last;

  current = module->module_objects;
  while ( current != NULL ) {
    /* First check if the identifier has been used already */
    id = lookup_identifier(current->node_data->name);
    if ( id != NULL && id->node_data->id_type != OBJECT_ID ) {
      /* It has been declared previously as something other than a object */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Trying to merge object %s (from module %s) into module %s, but this object was already declared as another identifier.\n",
			  current->node_data->name,
			  module->name,
			  name_of_module_currently_being_parsed);
    }
    else {
      existing_object = lookup_object(current->node_data->name);
      if ( existing_object == NULL ) {
	/* It has not been declared previously as something other than a object 
	   and it is not a object,
	   so we know it hasn't been declared */
	insert_identifier(current->node_data->name);
	insert_object(current->node_data->name,
		      current->node_data->num_arguments,
		      current->node_data->arguments);
	/* We also need to update its sort */
	init_string_list(&current_string_list_first,
			 &current_string_list_last);
	insert_string_into_list(&current_string_list_first,
				&current_string_list_last,
				current->node_data->name);
	update_objects(current_string_list_first, current->node_data->sort);
      }
      else {
	/* The object has already been declared so there's no need to add it,
	   but we need to check that it was declared to as an object with the
	   same arguments and the same sort. */
	if ( strcmp(existing_object->node_data->sort,
		    current->node_data->sort) 
	     !=
	     0) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Trying to merge object %s (from module %s) into module %s, but this object was already declared as having a different sort.\n",
				     current->node_data->name,
				     module->name,
				     name_of_module_currently_being_parsed);
	}
	else if ( existing_object->node_data->num_arguments
		  != current->node_data->num_arguments ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Trying to merge object %s (from module %s) into module %s, but this object was already declared with a different number of arguments.\n",
				     current->node_data->name,
				     module->name,
				     name_of_module_currently_being_parsed);
	}
	else {
	  current_existing_argument = existing_object->node_data->arguments;
	  current_argument = current->node_data->arguments;
	  while ( current_existing_argument != NULL ) {
	    if ( strcmp( current_existing_argument->node_data,
			 current_argument->node_data) 
		 !=
		 0) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Trying to merge object %s (from module %s) into module %s, but previous declaration has an argument with a different sort.\n",
					 current->node_data->name,
					 module->name,
					 name_of_module_currently_being_parsed);
	    }
	    current_existing_argument = current_existing_argument->next;
	    current_argument = current_argument->next;
	  }
	}
      }
    }
    current = current->next;
  }
}


/* for debugging */
void print_current_objects(FILE * stream) {
  struct object_list_node * current_object;
  int counter;
  struct string_list_node * current_arg;

  counter = 1;
  fprintf(stream, "Objects:\n");
  current_object = object_list_first;
  while ( current_object != NULL ) {
    if ( current_object->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      /* print name first */
      if ( current_object->node_data->name != NULL ) {
	fprintf(stream, "%s", current_object->node_data->name);
      }
      else {
	fprintf(stream, "NULL-object-name");
      }
      
      /* print arguments if there are any.
	 This assumes that all arguments are in a linked
	 list of character strings */
      if (current_object->node_data->num_arguments > 0) {
	fprintf(stream, "(");
	
	/* start with the first argument in the list */
	current_arg = current_object->node_data->arguments;
	if( current_arg != NULL ) {
	  if (current_arg->node_data != NULL) {
	    fprintf(stream, "%s", current_arg->node_data);
	  }
	  else {
	    fprintf(stream, "NULL-argument-name");
	  }
	  current_arg = current_arg->next;
	  while (current_arg != NULL) {
	    fprintf(stream, ",");
	    if (current_arg->node_data != NULL) {
	      fprintf(stream, "%s", current_arg->node_data);
	    }
	    else {
	      fprintf(stream, "NULL-argument-name");
	    }
	    current_arg = current_arg->next;
	  }
	}
	fprintf(stream, ")");
      }
      
      // Print the sort of the object
      fprintf(stream, ", of sort ");
      if ( current_object->node_data->sort != NULL ) {
	fprintf(stream, "%s", current_object->node_data->sort);
      }
      else {
	fprintf(stream, "NULL-sort-name");
      }
      fprintf(stream, "\n");
    }
    counter++;
    current_object = current_object->next;
  }
}


/* for debugging.  
   To print out a specific list of objects (e.g. of a specific module). */
void print_object_list(FILE * stream, struct object_list_node * list) {
  struct object_list_node * current_object;
  int counter;
  struct string_list_node * current_arg;

  counter = 1;
  fprintf(stream, "Objects:\n");
  current_object = list;
  while ( current_object != NULL ) {
    /* print name first */
    fprintf(stream, "%d: %s", counter, current_object->node_data->name);

    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current_object->node_data->num_arguments > 0) {

      fprintf(stream, "(");
      
      /* start with the first argument in the list */
      current_arg = current_object->node_data->arguments;

      fprintf(stream, "%s", current_arg->node_data);
      current_arg = current_arg->next;

      while (current_arg != NULL) {
	fprintf(stream, ",%s", current_arg->node_data);
	current_arg = current_arg->next;
      }
      
      fprintf(stream, ")");
    }

    /* print the sort of object it is */
    fprintf(stream, ", of sort %s\n", current_object->node_data->sort);
    counter++;
    current_object = current_object->next;
  }
}

/* for printing out an object declaration section to be used as ccalc input. */
void print_object_list_for_ccalc(FILE * stream, struct object_list_node * object_list) {
  struct object_list_node * current;
  struct string_list_node * current_arg;

  current = object_list;
  if ( current != NULL ) {
    fprintf(stream, ":- objects\n");
    fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current->node_data->num_arguments > 0) {
      fprintf(stream, "(");
      /* start with the first argument in the list */
      current_arg = current->node_data->arguments;
      if ( is_an_integer_range(current_arg->node_data) ) {
	fprintf(stream, 
		"%s", 
		convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
      }
      else {
	fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
      }
      current_arg = current_arg->next;
      while (current_arg != NULL) {
	if ( is_an_integer_range(current_arg->node_data) ) {
	  fprintf(stream, 
		  "%s", 
		  convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	}
	else {
	  fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	}
	current_arg = current_arg->next;
      }
      fprintf(stream, ")");
    }
    /* print the sort of the object */
    fprintf(stream, " :: %s", prepare_string_for_ccalc(current->node_data->sort) );
    current = current->next; 
    while ( current != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
      /* print arguments if there are any.
	 This assumes that all arguments are in a linked
	 list of character strings */
      if (current->node_data->num_arguments > 0) {
	fprintf(stream, "(");
	/* start with the first argument in the list */
	current_arg = current->node_data->arguments;
	if ( is_an_integer_range(current_arg->node_data) ) {
	  fprintf(stream, 
		  "%s", 
		  convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	}
	else {
	  fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
	}
	current_arg = current_arg->next;
	while (current_arg != NULL) {
	  if ( is_an_integer_range(current_arg->node_data) ) {
	    fprintf(stream, 
		    "%s", 
		    convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	  }
	  else {
	    fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	  }
	  current_arg = current_arg->next;
	}
	fprintf(stream, ")");
      }
      /* print the sort of the object */
      fprintf(stream, " :: %s", prepare_string_for_ccalc(current->node_data->sort) );
      current = current->next;
    }
    fprintf(stream, ".\n");
  }
}


/****** Managing lists of constants **************/


/* initialize the list of constants */
void init_constants() {
  constant_list_first = constant_list_last = NULL;
}

/* initialize a list of constants */
void init_constant_list(struct constant_list_node ** first, struct constant_list_node ** last) {
  (*first) = (*last) = NULL;
}


/* Looks up a constant and returns a pointer to its list entry.
   Returns NULL if not found. */
struct constant_list_node * lookup_constant(char * lookup_name) {
  struct constant_list_node * current;
  
  current = constant_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Looks up a constant in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct constant_list_node * lookup_constant_in_module(char * lookup_name, struct mad_module * module) {
  struct constant_list_node * current;
  
  current = module->module_constants;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* This inserts a new constant into the list of constants */
void insert_constant (char * name, int num_arguments, struct string_list_node * arguments) {
  struct constant_list_node * new_node;

  new_node = (struct constant_list_node *) malloc( sizeof(struct constant_list_node) );
  new_node->node_data = make_new_constant(name, num_arguments, arguments);
  new_node->next = NULL;

  /* constant_list_last is the pointer to the end of the list */
  if ( constant_list_last == NULL ) { /* the list is empty */
    constant_list_first = new_node;
    constant_list_last = new_node;
  }
  else {
    constant_list_last->next = new_node;
    constant_list_last = new_node;
  }

  /* update the list of identifiers to indicate this
     identifier is a constant */
  update_id_type(name, CONSTANT_ID);
  
}

/* This inserts a new constant into a given list of constants */
void insert_constant_into_list (struct constant_list_node ** first, 
				struct constant_list_node ** last, 
				char * name, 
				int num_arguments, 
				struct string_list_node * arguments) {
  struct constant_list_node * new_node;

  new_node = (struct constant_list_node *) malloc( sizeof(struct constant_list_node) );
  new_node->node_data = make_new_constant(name, num_arguments, arguments);
  new_node->next = NULL;

  /* last is the pointer to the end of the list */
  if ( (*last) == NULL ) { /* the list is empty */
    (*first) = new_node;
    (*last) = new_node;
  }
  else {
    (*last)->next = new_node;
    (*last) = new_node;
  }
}


/* Called when inserting a new constant.  Makes a new constant */
struct mad_constant * make_new_constant (char * name, int num_arguments,
                                         struct string_list_node * arguments) {
  struct mad_constant * new_constant ;
  
  new_constant = (struct mad_constant *) malloc( sizeof(struct mad_constant) );
  
  new_constant->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_constant->name, name );

  new_constant->num_arguments = num_arguments;

  new_constant->arguments = arguments;
  
  return new_constant;
}


/* Add information about kind and domain of a group of constants
   This is separate from making a new constant because we get this
   information later during the declaration. */
void update_constants(struct string_list_node * current_entry, int kind, char * domain) {

  struct constant_list_node * this_constant;

  while(current_entry != NULL) { 

    this_constant = lookup_constant(current_entry->node_data);

    if (this_constant != NULL) {
      this_constant->node_data->kind = kind;
      
      this_constant->node_data->domain = (char *) malloc( sizeof(char) * ( strlen(domain) + 1 ) );
      strcpy( this_constant->node_data->domain, domain );
    }
    else {
      print_error_message(stderr, "Internal error: trying to update a non-existent constant %s!\n",
	     current_entry->node_data);
    }

    current_entry = current_entry->next;
  }
}

/* Makes a copy of a given constant list */
struct constant_list_node * copy_constant_list(struct constant_list_node * constant_list) {
  struct constant_list_node * copied_constant_list;
  
  if ( constant_list != NULL ) {
    /* First copy current node's information */
    copied_constant_list = (struct constant_list_node *) malloc( sizeof(struct constant_list_node) );
    copied_constant_list->node_data = make_new_constant(constant_list->node_data->name,
							constant_list->node_data->num_arguments,
							copy_string_list(constant_list->node_data->arguments));
    copied_constant_list->node_data->kind = constant_list->node_data->kind;
    copied_constant_list->node_data->domain = (char *) malloc( sizeof(char) * ( strlen(constant_list->node_data->domain) + 1 ) );
    strcpy( copied_constant_list->node_data->domain, constant_list->node_data->domain );
    
    /* Then copy the rest */
    copied_constant_list->next = copy_constant_list(constant_list->next);
  }
  else {
    copied_constant_list = NULL;
  }

  return copied_constant_list;

}


/* Called to merge constants of an imported module with the current constants */
void merge_constants_from_module(struct mad_module * module) {

  struct constant_list_node * current;
  struct constant_list_node * constant;
  struct id_list_node * id;

  struct string_list_node * current_string_list_first;
  struct string_list_node * current_string_list_last;
  
  current = module->module_constants;
  while ( current != NULL ) {
    constant = lookup_constant(current->node_data->name);
    /* First check if the identifier has been used already */
    id = lookup_identifier(current->node_data->name);
    if ( id != NULL && id->node_data->id_type != CONSTANT_ID ) {
      /* It has been declared previously as something other than a constant */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Trying to merge constant %s (from module %s) into module %s, but this constant was already declared as another identifier.\n",
			  current->node_data->name,
			  module->name,
			  name_of_module_currently_being_parsed);
    }
    else if ( constant == NULL ) {
      /* It has not been declared previously as something other than a constant 
         and it is not a constant,
         so we know it hasn't been declared */

      insert_identifier(current->node_data->name);
      insert_constant(current->node_data->name, current->node_data->num_arguments, current->node_data->arguments);
      /* We also need to update its kind and domain */
      init_string_list(&current_string_list_first,
		       &current_string_list_last);
      insert_string_into_list(&current_string_list_first,
			      &current_string_list_last,
			      current->node_data->name);
      update_constants(current_string_list_first, current->node_data->kind, current->node_data->domain);
      
    }
    else {
      /* The constant has already been declared so there's no need to add it,
	 but we need to check that it was declared in the same way */
      
      if ( (constant->node_data->kind != current->node_data->kind)
	   || (strcmp( constant->node_data->domain, current->node_data->domain) != 0)
	   || (constant->node_data->num_arguments != current->node_data->num_arguments)
	   || (!string_list_same(constant->node_data->arguments, current->node_data->arguments)) )
	{
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Trying to merge constant %s (from module %s) into module %s, but this constant was already declared in a different way.\n",
			      current->node_data->name,
			      module->name,
			      name_of_module_currently_being_parsed); 
	}
    }
    current = current->next;
  }
}


/* for debugging */
void print_current_constants(FILE * stream) {
  struct constant_list_node * current_constant;
  int counter;
  struct string_list_node * current_arg;

  counter = 1;
  fprintf(stream, "Constants:\n");
  current_constant = constant_list_first;
  while ( current_constant != NULL ) {

    if (current_constant->node_data != NULL) {
      fprintf(stream, "%d: ", counter);

      /* print name first */
      if( current_constant->node_data->name != NULL ) {
	fprintf(stream, "%s", current_constant->node_data->name);
      }
      else {
	fprintf(stream, "NULL-constant-name");
      }
      /* print arguments if there are any.
	 This assumes that all arguments are in a linked
	 list of character strings */
      if (current_constant->node_data->num_arguments > 0) {
	
	fprintf(stream, "(");
	
	/* start with the first argument in the list */
	current_arg = current_constant->node_data->arguments;
	if( current_arg != NULL ) {
	  if (current_arg->node_data != NULL) {
	    fprintf(stream, "%s", current_arg->node_data);
	  }
	  else {
	    fprintf(stream, "NULL-argument-name");
	  }
	  current_arg = current_arg->next;
	  while (current_arg != NULL) {
	    fprintf(stream, ",");
	    if (current_arg->node_data != NULL) {
	      fprintf(stream, "%s", current_arg->node_data);
	    }
	    else {
	      fprintf(stream, "NULL-argument-name");
	    }
	    current_arg = current_arg->next;
	  }
	}
	fprintf(stream, ")");
      }
      
      /* print the kind of constant it is */
      switch ( current_constant->node_data->kind ) {
      case SIMPLE_FL_CONST:
	fprintf(stream, " : simpleFluent");
	break;
      case SD_FL_CONST:
	fprintf(stream, " : sdFluent");
	break;
      case ACTION_CONST:
	fprintf(stream, " : action");
	break;
      case RIGID_CONST:
	fprintf(stream, " : rigid constant with domain ");
	break;
      }
      
      if ( current_constant->node_data->kind != RIGID_CONST ) {
	/* print the domain of the fluent or action constant only if it's not Boolean */
	if ( current_constant->node_data->domain != NULL ) {
	  if ( strcmp(current_constant->node_data->domain, "Boolean") != 0 ) {
	    fprintf(stream, "(%s)", current_constant->node_data->domain);
	  }
	}
	else {
	  fprintf(stream, "(NULL-domain-name)");
	}
      }
      else { /* print the domain of the rigid constant */
	if ( current_constant->node_data->domain != NULL ) {
	  fprintf(stream, "%s", current_constant->node_data->domain);
	}
	else {
	  fprintf(stream, "NULL-domain-name");
	}
      }
      
      fprintf(stream, "\n");

    counter++;
    }
    current_constant = current_constant->next;
  }
}

/* for debugging.  
   To print out a specific list of constants (e.g. of a specific module). */
void print_constant_list(FILE * stream, struct constant_list_node * list) {
  struct constant_list_node * current_constant;
  int counter;
  struct string_list_node * current_arg;

  counter = 1;
  fprintf(stream, "Constants:\n");
  current_constant = list;
  while ( current_constant != NULL ) {

    /* print name first */
    fprintf(stream, "%d: %s", counter, current_constant->node_data->name);

    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current_constant->node_data->num_arguments > 0) {

      fprintf(stream, "(");

      /* start with the first argument in the list */
      current_arg = current_constant->node_data->arguments;
      fprintf(stream, "%s", current_arg->node_data);
      current_arg = current_arg->next;

      while (current_arg != NULL) {
	fprintf(stream, ",%s", current_arg->node_data);
	current_arg = current_arg->next;
      }

      fprintf(stream, ")");
    }

    /* print the kind of constant it is */
    switch ( current_constant->node_data->kind ) {
    case SIMPLE_FL_CONST:
      fprintf(stream, " : simpleFluent");
      break;
    case SD_FL_CONST:
      fprintf(stream, " : sdFluent");
      break;
    case ACTION_CONST:
      fprintf(stream, " : action");
      break;
    case RIGID_CONST:
      fprintf(stream, " : rigid constant with domain ");
      break;
    }

    if ( current_constant->node_data->kind != RIGID_CONST ) {
      /* print the domain of the fluent or action constant only if it's not Boolean */
      if ( strcmp(current_constant->node_data->domain, "Boolean") != 0 ) {
	fprintf(stream, "(%s)", current_constant->node_data->domain);
      }
    }
    else { /* print the domain of the rigid constant */
      fprintf(stream, "%s", current_constant->node_data->domain);
    }

    fprintf(stream, "\n");

    counter++;
    current_constant = current_constant->next;
  }
}

/* for printing out a constant declaration section to be used as ccalc input */
void print_constant_list_for_ccalc(FILE * stream, struct constant_list_node * list) {
  struct constant_list_node * current;
  struct string_list_node * current_arg;

  current = list;
  if ( current != NULL ) {
    fprintf(stream, ":- constants\n");
    fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current->node_data->num_arguments > 0) {
      fprintf(stream, "(");
      /* start with the first argument in the list */
      current_arg = current->node_data->arguments;
      if ( strcmp(current_arg->node_data, "action") == 0 ) {
	fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
	fprintf(stream, "(boolean)");
      }
      else if ( strcmp(current_arg->node_data, "explicitAction") == 0 ) {
	/* !!!: CCalc doesn't have explicit actions so we need to print
	   "action" instead of "explicitaction" */
	fprintf(stream, "%s", prepare_string_for_ccalc("action"));
	fprintf(stream, "(boolean)");
      }
      else if ( is_an_integer_range(current_arg->node_data) ) {
	fprintf(stream, 
		"%s", 
		convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
      }
      else {
	fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
      }
      current_arg = current_arg->next;
      while (current_arg != NULL) {
	if ( strcmp(current_arg->node_data, "action") == 0 ) {
	  fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	  fprintf(stream, "(boolean)");
	}
	else if ( strcmp(current_arg->node_data, "explicitAction") == 0 ) {
	  /* !!!: CCalc doesn't have explicit actions so we need to print
	     "action" instead of "explicitaction" */
	  fprintf(stream, ", %s", prepare_string_for_ccalc("action"));
	  fprintf(stream, "(boolean)");
	}
	else if ( is_an_integer_range(current_arg->node_data) ) {
	  fprintf(stream, 
		  ", %s", 
		  convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	}
	else {
	  fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	}
	current_arg = current_arg->next;
      }
      fprintf(stream, ")");
    }
    /* print the kind of constant it is */
    switch ( current->node_data->kind ) {
    case SIMPLE_FL_CONST:
      fprintf(stream, " :: simpleFluent");
      break;
    case SD_FL_CONST:
      fprintf(stream, " :: sdFluent");
      break;
    case ACTION_CONST:
      fprintf(stream, " :: action");
      break;
    case RIGID_CONST:
      fprintf(stream, " :: "); /* We will print the domain shortly. */
      break;
    }

    if ( current->node_data->kind != RIGID_CONST ) {
      /* print the domain of the fluent or action constant only if it's not Boolean */
      if ( strcmp(current->node_data->domain, "Boolean") != 0 ) {
	if ( is_an_integer_range(current->node_data->domain) ) {
	  fprintf(stream, 
		  "(%s)", 
		  convert_integer_range_to_ccalc_acceptable_form(current->node_data->domain));
	}
	else {
	  fprintf(stream, "(%s)", prepare_string_for_ccalc(current->node_data->domain));
	}
      }
    }
    else { /* print the domain of the rigid constant */
      if ( is_an_integer_range(current->node_data->domain) ) {
	fprintf(stream, 
		"%s", 
		convert_integer_range_to_ccalc_acceptable_form(current->node_data->domain));
      }
      else {
	fprintf(stream, "%s", prepare_string_for_ccalc(current->node_data->domain));
      }
    }

    current = current->next;

    while ( current != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream, "    %s", prepare_string_for_ccalc(current->node_data->name) );
      /* print arguments if there are any.
	 This assumes that all arguments are in a linked
	 list of character strings */
      if (current->node_data->num_arguments > 0) {
	fprintf(stream, "(");
	/* start with the first argument in the list */
	current_arg = current->node_data->arguments;
	if ( strcmp(current_arg->node_data, "action") == 0 ) {
	  fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
	  fprintf(stream, "(boolean)");
	}
	else if ( strcmp(current_arg->node_data, "explicitAction") == 0 ) {
	  /* !!!: CCalc doesn't have explicit actions so we need to print
	     "action" instead of "explicitaction" */
	  fprintf(stream, "%s", prepare_string_for_ccalc("action"));
	  fprintf(stream, "(boolean)");
	}
	else if ( is_an_integer_range(current_arg->node_data) ) {
	  fprintf(stream, 
		  "%s", 
		  convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	}
	else {
	  fprintf(stream, "%s", prepare_string_for_ccalc(current_arg->node_data));
	}
	current_arg = current_arg->next;
	while (current_arg != NULL) {
	  if ( strcmp(current_arg->node_data, "action") == 0 ) {
	    fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	    fprintf(stream, "(boolean)");
	  }
	  else if ( strcmp(current_arg->node_data, "explicitAction") == 0 ) {
	    /* !!!: CCalc doesn't have explicit actions so we need to print
	       "action" instead of "explicitaction" */
	    fprintf(stream, ", %s", prepare_string_for_ccalc("action"));
	    fprintf(stream, "(boolean)");
	  }
	  else if ( is_an_integer_range(current_arg->node_data) ) {
	    fprintf(stream, 
		    ", %s", 
		    convert_integer_range_to_ccalc_acceptable_form(current_arg->node_data));
	  }
	  else {
	    fprintf(stream, ", %s", prepare_string_for_ccalc(current_arg->node_data));
	  }
	  current_arg = current_arg->next;
	}
	fprintf(stream, ")");
      }
      /* print the kind of constant it is */
      switch ( current->node_data->kind ) {
      case SIMPLE_FL_CONST:
	fprintf(stream, " :: simpleFluent");
	break;
      case SD_FL_CONST:
	fprintf(stream, " :: sdFluent");
	break;
      case ACTION_CONST:
	fprintf(stream, " :: action");
	break;
      case RIGID_CONST:
	fprintf(stream, " :: "); /* We will print the domain shortly. */
	break;
      }
      
      if ( current->node_data->kind != RIGID_CONST ) {
	/* print the domain of the fluent or action constant only if it's not Boolean */
	if ( strcmp(current->node_data->domain, "Boolean") != 0 ) {
	  if ( is_an_integer_range(current->node_data->domain) ) {
	    fprintf(stream, 
		    "(%s)", 
		    convert_integer_range_to_ccalc_acceptable_form(current->node_data->domain));
	  }
	  else {
	    fprintf(stream, "(%s)", prepare_string_for_ccalc(current->node_data->domain));
	  }
	}
      }
      else { /* print the domain of the rigid constant */
	if ( is_an_integer_range(current->node_data->domain) ) {
	  fprintf(stream, 
		  "%s", 
		  convert_integer_range_to_ccalc_acceptable_form(current->node_data->domain));
	}
	else {
	  fprintf(stream, "(%s)", prepare_string_for_ccalc(current->node_data->domain));
	}
      }
      
      current = current->next;
    }
    fprintf(stream, ".\n");
  }
}


/****** Managing lists of variables **************/


/* initialize the list of variables */
void init_variables() {
  variable_list_first = variable_list_last = NULL;
}


/* Looks up a variable and returns a pointer to its list entry.
   Returns NULL if not found. */
struct variable_list_node * lookup_variable(char * lookup_name) {
  struct variable_list_node * current;

  current = variable_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Looks up a variable in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct variable_list_node * lookup_variable_in_module(char * lookup_name, struct mad_module * module) {
  struct variable_list_node * current;

  current = module->module_variables;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* Goes through the variable list and looks for a variable of a
   particular sort.  
   If it finds one, it returns the name,
   otherwise it returns NULL. */
char * find_variable_of_sort(char * sort_to_lookup) {
  struct variable_list_node * current;

  current = variable_list_first;
  if ( sort_to_lookup != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->sort, sort_to_lookup) == 0 ) {
	return current->node_data->name;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* Goes through the variable list of a given module and looks for a variable of a
   particular sort.  
   If it finds one, it returns the name,
   otherwise it returns NULL. */
char * find_variable_of_sort_in_module(char * sort_to_lookup, struct mad_module * module) {
  struct variable_list_node * current;

  current = module->module_variables;
  if ( sort_to_lookup != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->sort, sort_to_lookup) == 0 ) {
	return current->node_data->name;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}

/* This inserts a new variable into the list of variables */
void insert_variable (char * name) {
  struct variable_list_node * new_node;

  new_node = (struct variable_list_node *) malloc( sizeof(struct variable_list_node) );
  new_node->node_data = make_new_variable(name);
  new_node->next = NULL;

  /* variable_list_last is the pointer to the end of the list */
  if ( variable_list_last == NULL ) { /* the list is empty */
    variable_list_first = new_node;
    variable_list_last = new_node;
  }
  else {
    variable_list_last->next = new_node;
    variable_list_last = new_node;
  }

  /* update the list of identifiers to indicate this
     identifier is a variable */
  update_id_type(name, VARIABLE_ID);

}

/* This inserts a new variable into the list of variables of a given module. 
   This is called when new variables are created during the processing of 
   import statements, so we know the sort of the variable. 
   
   Note that this is a little different from insert_variable, because it
   does not call update_id_type at the end.  This is because we don't keep a list
   of identifiers for each module, only one for the module currently being read in.
   So, variables added to a module will not have a corresponding identifier at this
   point.  Later on, when this module is merged with the one which is importing it,
   these variables will be added to the "global" identifiers and their id type
   will be updated.

   IMPORTANT: Now we also call this function after parsing, when making the
   action description definite, where we unify arguments by making new variables.
   Since there is no merging after that point we need to remember to update the
   id type too, after calling this function.
*/
void insert_variable_into_module(struct mad_module * module, char * name, char * sort) {
  struct variable_list_node * new_node;

  struct variable_list_node * current;

  new_node = (struct variable_list_node *) malloc( sizeof(struct variable_list_node) );
  new_node->node_data = make_new_variable(name);
  new_node->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(sort) + 1 ) );
  strcpy( new_node->node_data->sort, sort );
  new_node->next = NULL;

  current = module->module_variables;
  if (current == NULL ) {
    module->module_variables = new_node;
  }
  else {
    /* Go to the last element */
    while ( current->next != NULL ) {
      current = current->next;
    }
    current->next = new_node;
  }

}


/* Called when inserting a new variable.  Makes a new variable */
struct mad_variable * make_new_variable (char * name) {
  struct mad_variable * new_variable ;

  new_variable = (struct mad_variable *) malloc( sizeof(struct mad_variable) );
  new_variable->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_variable->name, name );

  return new_variable;
}


/* Add information about sort of a group of variables.
   This is separate from making a new variable because we get this
   information later during the declaration. */
void * update_variables (struct string_list_node * current_entry, char * sort) {

  struct variable_list_node * this_variable;

  while(current_entry != NULL) { 

    this_variable = lookup_variable(current_entry->node_data);

    if (this_variable != NULL) {
      
      this_variable->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(sort) + 1 ) );
      strcpy( this_variable->node_data->sort, sort );
    }
    else {
      print_error_message(stderr, "Internal error: trying to update a non-existent variable %s!\n",
	      current_entry->node_data);
    }

    current_entry = current_entry->next;
  }
}


/* Makes a copy of a given variable list */
struct variable_list_node * copy_variable_list(struct variable_list_node * variable_list) {
  struct variable_list_node * copied_variable_list;
  
  if ( variable_list != NULL ) {
    /* First copy current node's name and sort */
    copied_variable_list = (struct variable_list_node *) malloc( sizeof(struct variable_list_node) );
    copied_variable_list->node_data = make_new_variable(variable_list->node_data->name);
    copied_variable_list->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(variable_list->node_data->sort) + 1 ) );
    strcpy( copied_variable_list->node_data->sort, variable_list->node_data->sort );
    
    /* Then copy the rest */
    copied_variable_list->next = copy_variable_list(variable_list->next);
  }
  else {
    copied_variable_list = NULL;
  }

  return copied_variable_list;
}

/* Given a list of strings corresponding to variables, variable_list, 
   returns a new list which is a "parallel copy" of the first, and 
   whose members are obtained by creating new variables
   of the sort corresponding to each variable from variable_list. */
struct string_list_node * make_new_variable_list_and_add_to_module(struct mad_module * module, 
								   struct string_list_node * variable_list) {

  struct string_list_node * new_variable_list;
  struct string_list_node * new_variable_list_end;

  struct string_list_node * current; /* Used to go down list. */

  struct variable_list_node * variable_as_declared; /* To point to the variable declaration 
						       of the variable from variable_list.
						       Used to determine the right sort. */

  char * new_variable_name; /* Holds the name of the new variable, as each new variable is created. */


  init_string_list(&new_variable_list, &new_variable_list_end);
  
  current = variable_list;
  
  while ( current != NULL ) {
    variable_as_declared = lookup_variable(current->node_data);
    if ( variable_as_declared != NULL ) {
      new_variable_name = make_new_variable_name(variable_as_declared->node_data->sort);

      insert_string_into_list(&new_variable_list, 
			      &new_variable_list_end,
			      new_variable_name);
      /* Add the new variable to the variable declarations of the module. */
      insert_variable_into_module(module, new_variable_name, variable_as_declared->node_data->sort);
    }
    else {
      print_error_message(stderr, "Internal error: Free variable in renaming clause doesn't appear to be declared.\n");
    }
    current = current->next;
  }
  
  return new_variable_list;
}


/* Similar to make_new_variable_list_and_add_to_module but differs in
   that the variables given are not variables of the module being currently
   parsed.  They are variables of the given module. */
struct string_list_node * make_new_variable_list_of_given_module_and_add_to_module(struct mad_module * module,
										   struct string_list_node * variable_list) {

  struct string_list_node * new_variable_list;
  struct string_list_node * new_variable_list_end;

  struct string_list_node * current; /* Used to go down list. */

  struct variable_list_node * variable_as_declared; /* To point to the variable declaration 
						       of the variable from variable_list.
						       Used to determine the right sort. */

  char * new_variable_name; /* Holds the name of the new variable, as each new variable is created. */


  init_string_list(&new_variable_list, &new_variable_list_end);
  
  current = variable_list;
  
  while ( current != NULL ) {
    variable_as_declared = lookup_variable_in_module(current->node_data, module);
    if ( variable_as_declared != NULL ) {
      new_variable_name = make_new_variable_name(variable_as_declared->node_data->sort);

      insert_string_into_list(&new_variable_list, 
			      &new_variable_list_end,
			      new_variable_name);
      /* Add the new variable to the variable declarations of the module. */
      insert_variable_into_module(module, new_variable_name, variable_as_declared->node_data->sort);
    }
    else {
      print_error_message(stderr, "Internal error: Free variable in renaming clause doesn't appear to be declared.\n");
    }
    current = current->next;
  }
  
  return new_variable_list;
}


/* Called to merge variables of an imported module with the current variables */
void merge_variables_from_module(struct mad_module * module) {

  struct variable_list_node * current;
  struct id_list_node * id;

  struct string_list_node * current_string_list_first;
  struct string_list_node * current_string_list_last;

  current = module->module_variables;
  while ( current != NULL ) {
    /* First check if the identifier has been used already */
    id = lookup_identifier(current->node_data->name);
    if ( id != NULL && id->node_data->id_type != VARIABLE_ID ) {
      /* It has been declared previously as something other than a variable */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Trying to merge variable %s (from module %s) into module %s, but this variable was already declared as another identifier.\n",
			  current->node_data->name,
			  module->name,
			  name_of_module_currently_being_parsed);
    }
    else if ( lookup_variable(current->node_data->name) == NULL ) {
      /* It has not been declared previously as something other than a variable 
         and it is not a variable,
         so we know it hasn't been declared */
      insert_identifier(current->node_data->name);
      insert_variable(current->node_data->name);
      /* We also need to update its sort */
      init_string_list(&current_string_list_first,
		       &current_string_list_last);
      insert_string_into_list(&current_string_list_first,
			      &current_string_list_last,
			      current->node_data->name);
      update_variables(current_string_list_first, current->node_data->sort);
    }
    else {
      /* The variable has already been declared so there's no need to add it,
	 but we need to check that it was declared to as a variable of the
	 same sort. */
      if ( !strcmp( (lookup_variable(current->node_data->name))->node_data->sort,
		    current->node_data->sort), 0) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "Trying to merge variable %s (from module %s) into module %s, but this variable was already declared as having a different sort.\n",
			    current->node_data->name,
			    module->name,
			    name_of_module_currently_being_parsed);
      }
    }
    current = current->next;
  }
}


/* for debugging */
void print_current_variables(FILE * stream) {
  struct variable_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Variables:\n");
  current = variable_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->name != NULL ) {
	fprintf(stream, "%s", current->node_data->name);
      }
      else {
	fprintf(stream, "NULL-variable-name");
      }
      fprintf(stream, ", of sort ");
      if ( current->node_data->sort != NULL ) {
	fprintf(stream, "%s", current->node_data->sort);
      }
      else {
	fprintf(stream, "NULL-sort-name");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of variables (e.g. of a specific module). */
void print_variable_list(FILE * stream, struct variable_list_node * list) {
  struct variable_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Variables:\n");
  current = list;
  while ( current != NULL ) {
    fprintf(stream, "%d: %s, of sort %s\n", counter, current->node_data->name, current->node_data->sort);
    counter++;
    current = current->next;
  }
}

/* for printing out a variable declaration section to be used as ccalc input. */
void print_variable_list_for_ccalc(FILE * stream, struct variable_list_node * variable_list) {
  struct variable_list_node * current;

  current = variable_list;
  if ( current != NULL ) {
    fprintf(stream, ":- variables\n");
    fprintf(stream, "    %s :: %s",
	    prepare_string_for_ccalc(current->node_data->name),
	    prepare_string_for_ccalc(current->node_data->sort) );
    current = current->next; 
    while ( current != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream, "    %s :: %s",
	      prepare_string_for_ccalc(current->node_data->name),
	      prepare_string_for_ccalc(current->node_data->sort) );
      current = current->next;
    }
    fprintf(stream, ".\n");
  }
}

/* for printing out a variable declaration section to be used as ccalc input.
   This version prints all variables except explicitAction variables.
*/
void print_non_explicit_action_variable_list_for_ccalc(FILE * stream, 
						       struct variable_list_node * variable_list) {
  struct variable_list_node * current;

  int header_printed;

  header_printed = 0;
  current = variable_list;
  while ( current != NULL ) {
    if ( strcmp(current->node_data->sort, "explicitAction") != 0 ) {
      if ( header_printed == 0 ) {
	fprintf(stream, ":- variables\n");
	fprintf(stream, 
		"    %s :: %s",
		prepare_string_for_ccalc(current->node_data->name),
		prepare_string_for_ccalc(current->node_data->sort) );
	header_printed = 1;
      }
      else {
	fprintf(stream, ";\n");
	fprintf(stream, 
		"    %s :: %s",
		prepare_string_for_ccalc(current->node_data->name),
		prepare_string_for_ccalc(current->node_data->sort) );
      }
    }
    else {
      if ( header_printed == 0 ) {
	fprintf(stream, ":- variables\n");
	fprintf(stream, 
		"    %s :: action",
		prepare_string_for_ccalc(current->node_data->name));
	header_printed = 1;
      }
      else {
	fprintf(stream, ";\n");
	fprintf(stream, 
		"    %s :: action",
		prepare_string_for_ccalc(current->node_data->name));
      }
    }
    current = current->next;
  }
  if ( header_printed != 0 ) {
    fprintf(stream, ".\n");
  }
}


/****** Managing lists of renaming clauses **************/

/* initialize the list of renaming clauses */
void init_renaming_clauses() {
  renaming_list_first = renaming_list_last = NULL;
}

/* initialize a list of renaming_cases */
void init_renaming_case_list(struct renaming_case_list_node ** list_start,
			     struct renaming_case_list_node ** list_end) {
  (*list_start) = (*list_end) = NULL;
}


/* initialize the list of renaming clauses for a specific import. */
void init_import_renaming_clauses() {
  import_renaming_list_first = import_renaming_list_last = NULL;
}

/* Looks up a sort or constant name in the list of renamings 
   currently being processed and returns a pointer to its list entry.
   Returns NULL if not found. 
   (This is used to prevent multiple renamings of the same sort/constant
   during the same import.)
*/
struct renaming_list_node * lookup_renaming(char * lookup_name) {
  struct renaming_list_node * current;
  
  current = import_renaming_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->atom->LHS_term, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* Called when inserting a new renaming clause.  Makes a new renaming case */
struct mad_renaming_case * make_new_renaming_case (struct mad_formula * condition, 
						   struct mad_formula * formula) {

  struct mad_renaming_case * new_renaming_case ;

  new_renaming_case = (struct mad_renaming_case *) malloc( sizeof(struct mad_renaming_case) );
  new_renaming_case->condition = condition;
  new_renaming_case->formula = formula;

  return new_renaming_case;
}

/* This inserts a new renaming case into a list of renaming cases 
   for a specific import. */
void insert_renaming_case_into_list(struct renaming_case_list_node ** list_start,
				    struct renaming_case_list_node ** list_end,
				    struct mad_renaming_case * new_case) {
  
  struct renaming_case_list_node * new_node;

  new_node = (struct renaming_case_list_node *) malloc( sizeof(struct renaming_case_list_node) );
  new_node->node_data = new_case;
  new_node->next = NULL;

  /* renaming_list_last is the pointer to the end of the list */
  if ( (*list_end) == NULL ) { /* the list is empty */
    (*list_start) = new_node;
    (*list_end) = new_node;
  }
  else {
    (*list_end)->next = new_node;
    (*list_end) = new_node;
  }
  
}


/* Called when inserting a new renaming clause.  Makes a new renaming clause */
struct mad_renaming_clause * make_new_renaming_clause (int kind, 
						       struct mad_formula * atom, 
						       struct mad_formula * formula,
						       struct string_list_node * free_variables,
						       int num_cases,
						       struct renaming_case_list_node * cases,
						       struct mad_formula * generalized_atom, 
						       struct renaming_case_list_node * generalized_atom_cases ) {

  struct mad_renaming_clause * new_renaming_clause ;

  new_renaming_clause = (struct mad_renaming_clause *) malloc( sizeof(struct mad_renaming_clause) );
  new_renaming_clause->kind;
  new_renaming_clause->atom = atom;
  new_renaming_clause->formula = formula;
  new_renaming_clause->list_of_free_variables = free_variables;
  new_renaming_clause->num_cases = num_cases;
  new_renaming_clause->cases = cases;
  new_renaming_clause->generalized_atom = generalized_atom;
  new_renaming_clause->generalized_atom_cases = generalized_atom_cases;

  return new_renaming_clause;
}

/* This inserts a new renaming clause into the list of renaming clauses */
void insert_renaming_clause (int kind, 
			     struct mad_formula * atom, 
			     struct mad_formula * formula,
			     struct string_list_node * free_variables,
			     int num_cases,
			     struct renaming_case_list_node * cases,
			     struct mad_formula * generalized_atom, 
			     struct renaming_case_list_node * generalized_atom_cases) {

  struct renaming_list_node * new_node;

  new_node = (struct renaming_list_node *) malloc( sizeof(struct renaming_list_node) );
  new_node->node_data = make_new_renaming_clause(kind, atom, formula, free_variables, num_cases, cases, generalized_atom, generalized_atom_cases);
  new_node->next = NULL;

  /* renaming_list_last is the pointer to the end of the list */
  if ( renaming_list_last == NULL ) { /* the list is empty */
    renaming_list_first = new_node;
    renaming_list_last = new_node;
  }
  else {
    renaming_list_last->next = new_node;
    renaming_list_last = new_node;
  }

}

/* This inserts a new renaming clause into the list of renaming clauses 
   for a specific import. */
void insert_import_renaming_clause (int kind, 
				    struct mad_formula * atom, 
				    struct mad_formula * formula,
				    struct string_list_node * free_variables,
				    int num_cases,
				    struct renaming_case_list_node * cases,
				    struct mad_formula * generalized_atom,
				    struct renaming_case_list_node * generalized_atom_cases) {

  struct renaming_list_node * new_node;

  new_node = (struct renaming_list_node *) malloc( sizeof(struct renaming_list_node) );
  new_node->node_data = make_new_renaming_clause(kind, atom, formula, free_variables, num_cases, cases, generalized_atom, generalized_atom_cases);
  new_node->next = NULL;

  /* renaming_list_last is the pointer to the end of the list */
  if ( import_renaming_list_last == NULL ) { /* the list is empty */
    import_renaming_list_first = new_node;
    import_renaming_list_last = new_node;
  }
  else {
    import_renaming_list_last->next = new_node;
    import_renaming_list_last = new_node;
  }

}

/* Makes a copy of a given renaming case list */
struct renaming_case_list_node * copy_renaming_case_list(struct renaming_case_list_node * cases) {
  struct renaming_case_list_node * copied_cases;
  
  if (cases != NULL ) {
    /* First copy current node's information */
    copied_cases = (struct renaming_case_list_node *) malloc( sizeof(struct renaming_case_list_node) );
    copied_cases->node_data = make_new_renaming_case(copy_formula(cases->node_data->condition),
						     copy_formula(cases->node_data->formula));
    /* Then copy the rest */
    copied_cases->next = copy_renaming_case_list(cases->next);
  }
  else {
    copied_cases = NULL;
  }
  return copied_cases;
}


/* Makes a copy of a given renaming list */
struct renaming_list_node * copy_renaming_list(struct renaming_list_node * renaming_list) {
  struct renaming_list_node * copied_renaming_list;
  
  if ( renaming_list != NULL ) {
    /* First copy current node's information */
    copied_renaming_list = (struct renaming_list_node *) malloc( sizeof(struct renaming_list_node) );
    copied_renaming_list->node_data = make_new_renaming_clause(renaming_list->node_data->kind,
							       copy_formula(renaming_list->node_data->atom),
							       copy_formula(renaming_list->node_data->formula),
							       copy_string_list(renaming_list->node_data->list_of_free_variables),
							       renaming_list->node_data->num_cases,
							       copy_renaming_case_list(renaming_list->node_data->cases),
							       copy_formula(renaming_list->node_data->generalized_atom),
							       copy_renaming_case_list(renaming_list->node_data->generalized_atom_cases));
    
    /* Then copy the rest */
    copied_renaming_list->next = copy_renaming_list(renaming_list->next);
  }
  else {
    copied_renaming_list = NULL;
  }
  
  return copied_renaming_list;
}

/* Called to merge renaming clauses of an imported module with the current renamings. */
void merge_renamings_from_module(struct mad_module * module) {

  struct renaming_list_node * current;

  struct renaming_list_node * place_holder_for_renaming_list_first;

  struct mad_formula * atom;
  struct mad_formula * formula;
  struct renaming_case_list_node * cases;
  struct mad_formula * generalized_atom;
  struct renaming_case_list_node * generalized_atom_cases;

  /* The renaming list for a module is kept in reverse order so
     we need to add everything to the beginning of the list. 
     At this point, there is just the renaming list for the
     module being parsed.  In front of that we need to add renamings
     from the current import followed by the renaming list for the module 
     being imported. */

  /* First we make a copy of the pointer to the current renaming list. */

  place_holder_for_renaming_list_first = renaming_list_first;  

  /* Now that we have a copy, we re-initialize it and populate it with
     renamings from the current import followed by the renaming list
     for the module being imported. */
  init_renaming_clauses();

  /* Copy each renaming in the current import. */
  current = import_renaming_list_first;
  while ( current != NULL ) {
    atom = copy_formula(current->node_data->atom);
    formula = copy_formula(current->node_data->formula);
    cases = copy_renaming_case_list(current->node_data->cases);
    generalized_atom = copy_formula(current->node_data->generalized_atom);
    generalized_atom_cases = copy_renaming_case_list(current->node_data->generalized_atom_cases);
    insert_renaming_clause( current->node_data->kind, atom, formula, NULL, current->node_data->num_cases, cases, generalized_atom, generalized_atom_cases);
    current = current->next;
  }

  /* Copy renamings from the module being imported. */
  current = module->module_renamings;
  while ( current != NULL ) {
    atom = copy_formula(current->node_data->atom);
    formula = copy_formula(current->node_data->formula);
    cases = copy_renaming_case_list(current->node_data->cases);
    generalized_atom = copy_formula(current->node_data->generalized_atom);
    generalized_atom_cases = copy_renaming_case_list(current->node_data->generalized_atom_cases);
    insert_renaming_clause( current->node_data->kind, atom, formula, NULL, current->node_data->num_cases, cases, generalized_atom, generalized_atom_cases);
    current = current->next;
  }
  
  /* Now finally put back the renamings which were already part of the 
     currently parsed module. */
  current = place_holder_for_renaming_list_first;
  while ( current != NULL ) {
    atom = copy_formula(current->node_data->atom);
    formula = copy_formula(current->node_data->formula);
    cases = copy_renaming_case_list(current->node_data->cases);
    generalized_atom = copy_formula(current->node_data->generalized_atom);
    generalized_atom_cases = copy_renaming_case_list(current->node_data->generalized_atom_cases);
    insert_renaming_clause( current->node_data->kind, atom, formula, NULL, current->node_data->num_cases, cases, generalized_atom, generalized_atom_cases);
    current = current->next;
  }

}


/* for debugging */
void print_current_renamings(FILE * stream) {
  struct renaming_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Renamings:\n");
  current = renaming_list_first;
  while ( current != NULL ) {

    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->atom != NULL ) {
	print_formula(stream, current->node_data->atom);
      }
      else {
	fprintf(stream, "NULL-LHS");
      }
      fprintf(stream, "  is  ");
      if ( current->node_data->formula != NULL ) {
	print_formula(stream, current->node_data->formula);
      }
      else {
	fprintf(stream, "NULL-RHS");
      }
      fprintf(stream, "\n");
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of renamings (e.g. of a specific module). */
void print_renaming_list(FILE * stream, struct renaming_list_node * list) {
  struct renaming_list_node * current;
  struct renaming_case_list_node * current_case;
  int counter;

  counter = 1;
  fprintf(stream, "Renamings:\n");
  current = list;
  while ( current != NULL ) {

    fprintf(stream, "%d: ", counter);
    if ( current->node_data->atom != NULL ) {
      print_formula(stream, current->node_data->atom);
    }
    else {
      fprintf(stream, "NULL formula in renaming!");
    }
    fprintf(stream, "\n");
    fprintf(stream, "    is     ");
    if ( current->node_data->formula != NULL ) {
      print_formula(stream, current->node_data->formula);
    }
    else {
      fprintf(stream, "NULL formula in renaming!");
    }
    fprintf(stream, "\n");
    /* Now print cases */
    if (current->node_data->num_cases == 1) {
      fprintf(stream, "  This renaming just has the single case above.\n");
    }
    else {
      fprintf(stream, "  Cases:\n");
      current_case = current->node_data->cases;
      while ( current_case != NULL ) {
	fprintf(stream, "    ");
	print_formula(stream, current_case->node_data->condition);
	fprintf(stream, " : ");
	print_formula(stream, current_case->node_data->formula);
	fprintf(stream, "\n");
	current_case = current_case->next;
      }
    }
    /* Print the generalized atom and cases, if they exist */
    if ( current->node_data->generalized_atom != NULL ) {
      fprintf(stream, "There are also generalized cases:\n");
      print_formula(stream, current->node_data->generalized_atom);
      fprintf(stream, "\n");
      fprintf(stream, "    is     ");
      fprintf(stream, "  Cases:\n");
      current_case = current->node_data->generalized_atom_cases;
      while ( current_case != NULL ) {
	fprintf(stream, "    ");
	print_formula(stream, current_case->node_data->condition);
	fprintf(stream, " : ");
	print_formula(stream, current_case->node_data->formula);
	fprintf(stream, "\n");
	current_case = current_case->next;
      }
    }
    counter++;
    current = current->next;
  }
}

/****** Managing lists of formulas **************/


/* initialize the list of formulas */
void init_formula_list(struct formula_list_node ** first, struct formula_list_node ** last) {
  (*first) = (*last) = NULL;
}

/* This inserts a new formula into a list of formulas */
void insert_formula_into_list(struct formula_list_node ** first, 
			      struct formula_list_node ** last, 
			      struct mad_formula * new_formula) {
  struct formula_list_node * new_node;

  new_node = (struct formula_list_node *) malloc( sizeof(struct formula_list_node) );
  new_node->node_data = new_formula;
  new_node->next = NULL;

  /* last is the pointer to the end of the list */
  if ( (*last) == NULL ) { /* the list is empty */
    (*first) = new_node;
    (*last) = new_node;
  }
  else {
    (*last)->next = new_node;
    (*last) = new_node;
  }
}



/****** Managing lists of axioms **************/


/* initialize the list of axioms */
void init_axioms() {
  axiom_list_first = axiom_list_last = NULL;
}

/* initialize a given list of axioms */
void init_axiom_list(struct axiom_list_node ** first_axiom, struct axiom_list_node ** last_axiom) {
  (*first_axiom) = (*last_axiom) = NULL;
}

/* This inserts a new axiom into the list of axioms */
void insert_axiom (struct mad_axiom * new_axiom) {

  struct axiom_list_node * new_node;

  new_node = (struct axiom_list_node *) malloc( sizeof(struct axiom_list_node) );
  new_node->node_data = new_axiom;
  new_node->next = NULL;

  /* axiom_list_last is the pointer to the end of the list */
  if ( axiom_list_last == NULL ) { /* the list is empty */
    axiom_list_first = new_node;
    axiom_list_last = new_node;
  }
  else {
    axiom_list_last->next = new_node;
    axiom_list_last = new_node;
  }

}

/* This inserts a new axiom into a given list of axioms */
void insert_axiom_into_list (struct mad_axiom * new_axiom, 
			     struct axiom_list_node ** first_axiom,
			     struct axiom_list_node ** last_axiom) {

  struct axiom_list_node * new_node;

  new_node = (struct axiom_list_node *) malloc( sizeof(struct axiom_list_node) );
  new_node->node_data = new_axiom;
  new_node->next = NULL;

  /* last_axiom is the pointer to the end of the list */
  if ( (*last_axiom) == NULL ) { /* the list is empty */
    (*first_axiom) = new_node;
    (*last_axiom) = new_node;
  }
  else {
    (*last_axiom)->next = new_node;
    (*last_axiom) = new_node;
  }
}

/* Called when inserting a new axiom.  Makes a new axiom */
struct mad_axiom * make_new_axiom (struct mad_formula * F_formula,
				   struct mad_formula * G_formula,
				   struct mad_formula * H_formula) {

  struct mad_axiom * new_axiom ;

  new_axiom = (struct mad_axiom *) malloc( sizeof(struct mad_axiom) );
  new_axiom->F = F_formula;
  new_axiom->G = G_formula;
  new_axiom->H = H_formula;

  return new_axiom;
}


/* Called when registering arguments as part of a constant appearing in a formula */
struct mad_constant_argument * make_new_argument (int argument_kind, char * argument_name) {
  struct mad_constant_argument * new_argument ;
  
  new_argument = (struct mad_constant_argument *) malloc( sizeof(struct mad_constant_argument) );
  
  new_argument->argument_kind = argument_kind;

  new_argument->name = (char *) malloc( sizeof(char) * ( strlen(argument_name) + 1 ) );
  strcpy( new_argument->name, argument_name );

  /* At this time we assume it will have no value attached. */
  new_argument->has_a_value = 0;

  new_argument->arguments = NULL;
  new_argument->next_argument = NULL;
  
  return new_argument;

}


/* Called when a formula is encountered as part of an axiom.  Makes a new formula */
struct mad_formula * make_new_formula (int kind) {
  struct mad_formula * new_formula ;
  
  new_formula = (struct mad_formula *) malloc( sizeof(struct mad_formula) );
  
  new_formula->formula_kind = kind;

  new_formula->connective = -1;
  new_formula->quantifier = -1;
  new_formula->inequality_kind = -1;
  new_formula->quantified_variable = NULL;
  new_formula->LHS_term = NULL;
  new_formula->LHS_arguments = NULL;
  new_formula->RHS_term = NULL;
  new_formula->RHS_arguments = NULL;
  new_formula->left_formula = NULL;
  new_formula->right_formula = NULL;

  new_formula->LHS_value = NULL;
  new_formula->RHS_value = NULL;
  
  return new_formula;
}

/* Checks if two argument structures are exactly the same. */
int are_argument_structures_same(struct mad_constant_argument * arguments1,
				 struct mad_constant_argument * arguments2) {

  struct mad_constant_argument * current_arg1;
  struct mad_constant_argument * current_arg2;

  if ( arguments1 != NULL && arguments2 != NULL ) {
    if ( strcmp(arguments1->name, arguments2->name) != 0 ) {
      return 0;
    }
    else if ( arguments1->argument_kind != arguments2->argument_kind ) {
      return 0;
    }
    else if ( arguments1->has_a_value != arguments2->has_a_value 
	      || (arguments1->has_a_value == 1
		  && strcmp(arguments1->value, arguments2->value) != 0) ) {
      return 0;
    }
    else if ( !are_argument_structures_same(arguments1->arguments, 
					    arguments2->arguments) ) {
      return 0;
    }
    else if ( !are_argument_structures_same(arguments1->next_argument, 
					    arguments2->next_argument) ) {
      return 0;
    }
  }
  // If they were not both non-Null, then both must be Null
  else if ( !(arguments1 == NULL && arguments2 == NULL) ) {
    return 0;
  }
  
  return 1;
}

/* Makes a copy of a given argument structure. */
struct mad_constant_argument * copy_arguments(struct mad_constant_argument * arguments) {

  struct mad_constant_argument * copied_argument;

  if ( arguments != NULL ) {
    /* First we copy the argument name. */
    copied_argument = make_new_argument(arguments->argument_kind, arguments->name);
    
    /* Now that the argument name has been copied, if it has a value, copy the value */
    if (arguments->has_a_value == 1) {
      copied_argument->value = (char *) malloc( sizeof(char) * ( strlen(arguments->value) + 1 ) );
      strcpy(copied_argument->value, arguments->value);
    }
    
    /* After copying the name and the value, we need to copy any possible arguments to this argument
       and then copy the next argument. */
    
    if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      copied_argument->arguments = copy_arguments(arguments->arguments);
    }
    if ( arguments->next_argument != NULL ) {
      copied_argument->next_argument = copy_arguments(arguments->next_argument);
    }
  }
  else {// if the original argument structure is NULL
    copied_argument = NULL;
  }

  return copied_argument;
}

/* Makes a copy of a single argument, 
   including any sub_arguments (if it is an ARGUMENTED_ARGUMENT), 
   but without copying the next_argument field. */
struct mad_constant_argument * copy_single_argument(struct mad_constant_argument * arguments) {

  struct mad_constant_argument * copied_argument;

  if ( arguments != NULL ) {
    /* First we copy the argument name. */
    copied_argument = make_new_argument(arguments->argument_kind, arguments->name);
    
    /* Now that the argument name has been copied, if it has a value, copy the value */
    if (arguments->has_a_value == 1) {
      copied_argument->value = (char *) malloc( sizeof(char) * ( strlen(arguments->value) + 1 ) );
      strcpy(copied_argument->value, arguments->value);
    }
    
    /* After copying the name and the value, we need to copy any possible arguments to this argument. */
    if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      copied_argument->arguments = copy_arguments(arguments->arguments);
    }

    /* We don't copy the next_argument, leaving it NULL instead. */
    copied_argument->next_argument = NULL;
  }
  else {// if the original argument structure is NULL
    copied_argument = NULL;
  }

  return copied_argument;
}


/* Takes an argument structure and renames variables from a given module,
   by prepending an import prefix to all variable occurrences in those arguments. */
void rename_variables_of_module_in_arguments(struct mad_module * module, 
					     struct mad_constant_argument * arguments, 
					     int import_index) {

  if ( arguments != NULL ) {
    /* First we rename the argument name, if it is a variable. */
    if ( lookup_variable_in_module(arguments->name, module) != NULL ) {
      arguments->name = prepend_import_prefix_to_string(arguments->name, import_index);
    }
    
    /* Now that the argument name has been copied, if it has a value, and that value is a variable, rename it. */
    if ( (arguments->has_a_value == 1) && (lookup_variable_in_module(arguments->value, module) != NULL) ) {
      arguments->value = prepend_import_prefix_to_string(arguments->value, import_index);
    }
    
    /* After renaming the name and the value, we need to rename any possible arguments to this argument
       and then rename the next argument. */
    
    if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      rename_variables_of_module_in_arguments(module, arguments->arguments, import_index);
    }
    if ( arguments->next_argument != NULL ) {
      rename_variables_of_module_in_arguments(module, arguments->next_argument, import_index);
    }
  }
  
}

/* Renames occurrences of a given constant in an argument structure, adding
   extra variables if necessary. */
int rename_constant_in_arguments(char * constant_name,
				  struct mad_constant_argument * arguments,
				  int import_index, 
				  struct string_list_node * new_variable_list) {

  int constant_encountered = 0;

  /* If we rename a constant occurrence, we may need to add extra variables.
     This is used to go to the end of the arguments. */
  struct mad_constant_argument * current_argument_argument;

  /* This is used to create extra arguments corresponding to extra variables. */
  struct mad_constant_argument * new_argument_argument;

  /* This is used to go down the new_variable_list if we need to add extra arguments
     to this constant occurrence. */
  struct string_list_node * current_new_variable;
  
  if ( arguments != NULL ) {
    /* First we rename the argument name, if it matches the constant name. */
    if ( strcmp(arguments->name, constant_name) == 0 ) {

      constant_encountered = 1;

      /* First prepend the import prefix to the constant name. */
      arguments->name = prepend_import_prefix_to_string(arguments->name, import_index);
      
      /* Now add extra arguments to the constant occurrence, if any. */
      if ( new_variable_list != NULL ) {
	
	/* To do this we need to go to the end of the arguments, if there are any. */
	current_argument_argument = arguments->arguments;
	
	/* If there are any arguments, go to the last one */
	if ( arguments->arguments != NULL ) {
	  /* Go down the chain of arguments to reach the last argument */
	  current_argument_argument = current_argument_argument->next_argument;
	  while ( current_argument_argument->next_argument != NULL ) {
	    current_argument_argument = current_argument_argument->next_argument;
	  }
	  /* Now add the first of the extra variables to the end of the arguments. */
	  current_argument_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	  /* Now update new_argument_argument to point to the last argument added so far. */
	  new_argument_argument = current_argument_argument->next_argument;
	}
	else { /* The argument has no arguments so we create a new argument
		  corresponding to the first of the extra variables. */
	  arguments->arguments = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	  /* Now update new_argument_argument to point to the last argument added so far. */
	  new_argument_argument = arguments->arguments;
	}
	
	/* Move on to the next extra variable to be added */
	current_new_variable = new_variable_list->next;
	
	/* At this point, we have already added the first extra variable as an
	   argument to this constant occurrence, new_argument_argument points to the
	   last argument for this constant occurrence, and current_new_variable points
	   to the remaining new variables to be added. Now we can just go add them
	   one by one. */
	while ( current_new_variable != NULL ) {
	  new_argument_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_new_variable->node_data);
	  new_argument_argument = new_argument_argument->next_argument;
	  current_new_variable = current_new_variable->next;
	}
      }
    }
    
    /* If the name of the argument matched the constant to be renamed, we have renamed 
       the constant occurrence and added any extra arguments to the end.
       
       However, we are not done yet.  The argument may have arguments itself which are constants
       themselves, so we need to rename constant occurrences in the arguments. 
       
       After renaming the name, we need to rename any possible arguments to this argument
       and then rename the next argument. */
    
    if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      // update constant_encountered ONLY IF WE SEE IT 
      // (i.e. don't mark it unseen here because we might have encountered it above)
      if ( rename_constant_in_arguments(constant_name, 
					arguments->arguments, 
					import_index, 
					new_variable_list) == 1 ) {
	constant_encountered = 1;
      }
    }
    if ( arguments->next_argument != NULL ) {
      // update constant_encountered ONLY IF WE SEE IT 
      // (i.e. don't mark it unseen here because we might have encountered it above)
      if ( rename_constant_in_arguments(constant_name,
					arguments->next_argument,
					import_index, 
					new_variable_list) == 1 ) {
	constant_encountered = 1;
      }
    }
  }
  
  return constant_encountered;
  
}


/* Given an argument structure and a list of variables, returns a list of any
   free variables in the argument structure, other than those in the given list.
   Assumes the variables have all been declared.  It won't recognize 
   undeclared variables. 
   Also, we assume that the formula in which the argument structure appears
   has already been checked for being correctly formed. 
   (Hence We don't check if identifiers are valid or not.  In fact, sometimes
   they might not appear to be, for example if we are looking for free variables
   in a constant appearing on the left hand side of an "is" statement.  In such
   a case the identifier for the constant hasn't been added to the list yet.
   However, the variables have, which is all that matters.) */
struct string_list_node * find_free_variables_in_arguments(struct mad_constant_argument * arguments, struct string_list_node * given_variables) {

  /* This will be the list of free variables in the arguments */
  struct string_list_node * free_variables_start;
  struct string_list_node * free_variables_end;

  /* As we check the argument, its value, its subarguments and the next arguments,
     we will wish to find the free variables which are different from those we already
     found, and also different from the given_variables.  For this, we temporarily modify
     The given variables, adding the quantified variable, but we will remove it
     later, so we keep a marker to the end of the list */
  struct string_list_node * end_of_given_variables;
  
  /* To iterate through the list */
  struct string_list_node * current;

  /* needed to look up an identifier and check if it's a variable */
  struct id_list_node * id;

  /* Initialize the list of free variables */
  init_string_list(&free_variables_start, &free_variables_end);

  if ( arguments != NULL ) {
    /* First we check if the argument name is a free variable. */

      id = lookup_identifier(arguments->name);
      if ( id == NULL ) {
	/* It didn't find the identifier.  This shouldn't happen because we
	   assume the formula in which the argument appears has already been checked for correctness. */
	/* However, we only care about the variables here so we don't do anything. */
	/* print_error_message(stderr, "Internal error: Formula contains an atom with unregistered identifier as argument.\n"); */
      }
      else if ( id->node_data->id_type == VARIABLE_ID ) {
	/* The argument name is a variable, so we check if it's among the given_variables. */
	current = given_variables;
	while (current != NULL ) {
	  if (strcmp(arguments->name, current->node_data) == 0) {
	    // We found the variable in the given_variable list so we quit looking
	    break;
	  }
	  current = current->next;
	}
	if ( current == NULL ) { // We didn't find the identifier among the given_variables
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  arguments->name);
	}
	else { // the argument name is an identifier but not a variable identifier
	  /* No need to do anything in this case */
	}
      }

    /* Now that the argument name has been checked, if it has a value, check if the value is a free variable */
    if (arguments->has_a_value == 1) {

      id = lookup_identifier(arguments->value);
      if ( id == NULL ) {
	/* It didn't find the identifier.  This shouldn't happen because we
	   assume the formula in which the argument appears has already been checked for correctness. */
	/* However, we only care about the variables here so we don't do anything. */
	/* print_error_message(stderr, "Internal error: Formula contains an atom with unregistered identifier as its value.\n"); */
      }
      else if ( id->node_data->id_type == VARIABLE_ID ) {
	/* The value is a variable, so we check if it's among the given_variables. */
	current = given_variables;
	while (current != NULL ) {
	  if (strcmp(arguments->value, current->node_data) == 0) {
	    // We found the variable in the given_variable list so we quit looking
	    break;
	  }
	  current = current->next;
	}
	if ( current == NULL ) { // We didn't find the identifier among the given_variables
	  
	  /* But it may be among the free_variables coming from the argument name
	     So we search those too.  (Actually, there may be only one variable there so we
	     don't need to loop, but I keep it like this in case things get switched around later.) */
	  current = free_variables_start;
	  while (current != NULL ) {
	    if (strcmp(arguments->value, current->node_data) == 0) {
	      // We found the variable in the free_variables list so we quit looking
	      break;
	    }
	    current = current->next;
	  }
	  if ( current == NULL ) { // We didn't find the identifier among the free_variables either
	    insert_string_into_list(&free_variables_start, 
				    &free_variables_end, 
				    arguments->value);
	  }
	}
      }
      else { // the value is an identifier but not a variable identifier
	/* No need to do anything in this case */
      }
    }
    
    /* After checking the name and the value, we need to check any possible arguments to this argument
       and then check the next argument. */
    
    if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      
      /* !!!: Actually, if the argument name was a variable, then we probably won't allow
	 any arguments in it, for a formula to be correctly formed.  However, I'll process the argumens
	 without checking that, just in case.  It won't hurt. */
      
      /* While looking for the free variables in the arguments, we don't want
	 to find the one corresponding to the argument name or value so we temporarily add
	 any free variables to the list of given_variables. */
      
      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;
	
	/* Link the end of given_variables to the start of free_variables (which came
	   from the argument name and value).  Note that this may be empty, if the argument name
	   and the value weren't a free variable. */
	current->next = free_variables_start;
	
	/* Add the free variables found in the arguments to the list of free variables */
	current = find_free_variables_in_arguments(arguments->arguments, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	
	/* Now unlink the end of the original given_variables from the free_variables */
	end_of_given_variables->next = NULL;
      }
      else { // if the given_variables list was NULL
	/* In this case we simply give the list of free_variables (which may include 
	   up to two entries coming from the argument name and value) as the 
	   given_variables. There is no need to add anything to the end of the list */
	
	/* Add the free variables found in the arguments to the list of free variables */
	current = find_free_variables_in_arguments(arguments->arguments, free_variables_start);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	/* Since we didn't modify the given_variables list, there is no need to fix it. */
      }
    }
    
    if ( arguments->next_argument != NULL ) {

      /* While looking for the free variables in the next arguments, we don't want
	 to find any corresponding to those found so far.  Therefore we temporarily add
	 any free variables to the list of given_variables. */
      
      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;
	
	/* Link the end of given_variables to the start of free_variables (which came
	   from the argument name, value and the arguments to this argument).  Note that 
	   this may be empty, if no free variables were found so far. */
	current->next = free_variables_start;
	
	/* Add the free variables found in the next arguments to the list of free variables */
	current = find_free_variables_in_arguments(arguments->next_argument, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	
	/* Now unlink the end of the original given_variables from the free_variables */
	end_of_given_variables->next = NULL;
      }
      else { // if the given_variables list was NULL
	/* In this case we simply give the list of free_variables found so far as the 
	   given_variables. There is no need to add anything to the end of the list */
	
	/* Add the free variables found in the next arguments to the list of free variables */
	current = find_free_variables_in_arguments(arguments->next_argument, free_variables_start);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	/* Since we didn't modify the given_variables list, there is no need to fix it. */
      }
    }
  }
  else {// if the original argument structure is NULL
    /* No need to do anything in this case.  We will return the list which was initialized to NULL */
  }
  
  return free_variables_start;
}

/* Makes a copy of a given formula. */
struct mad_formula * copy_formula(struct mad_formula * formula) {

  struct mad_formula * copied_formula; 

  if ( formula != NULL ) {
    /* First make a copy of the formula with the same kind */
    copied_formula = make_new_formula( formula->formula_kind );
    
    /* Copy the connective.  If it's -1, it doesn't matter that it gets copied. */
    copied_formula->connective = formula->connective;
    
    /* Copy the quantifier.  If it's -1, it doesn't matter that it gets copied. */
    copied_formula->quantifier = formula->quantifier;

    /* Copy the quantifier.  If it's -1, it doesn't matter that it gets copied. */
    copied_formula->inequality_kind = formula->inequality_kind;
    
    /* If it's a quantified formula, copy the quantified variable */
    if ( formula->formula_kind == QUANTIFIER ) {
      copied_formula->quantified_variable = (char *) malloc( sizeof(char) * ( strlen(formula->quantified_variable) + 1 ) );
      strcpy(copied_formula->quantified_variable, formula->quantified_variable);
    }
    
    /* If there is a left_formula, copy it */
    if ( formula->left_formula != NULL ) {
      copied_formula->left_formula = copy_formula(formula->left_formula);
    }
    
    /* If it's an atomic_formula, copy the LHS_term, LHS_arguments, RHS_term, RHS_arguments. */
    if ( formula->formula_kind == ATOMIC_FORMULA
	 || formula->formula_kind == INEQUALITY) {
      if (formula->LHS_term != NULL) {
	copied_formula->LHS_term = (char *) malloc( sizeof(char) * ( strlen(formula->LHS_term) + 1 ) );
	strcpy(copied_formula->LHS_term, formula->LHS_term);
	copied_formula->LHS_arguments = copy_arguments(formula->LHS_arguments);

	if ( formula->LHS_value != NULL ) {
	  copied_formula->LHS_value = (char *) malloc( sizeof(char) * ( strlen(formula->LHS_value) + 1 ) );
	  strcpy(copied_formula->LHS_value, formula->LHS_value);
	}
      }
      if (formula->RHS_term != NULL) {
	copied_formula->RHS_term = (char *) malloc( sizeof(char) * ( strlen(formula->RHS_term) + 1 ) );
	strcpy(copied_formula->RHS_term, formula->RHS_term);
	copied_formula->RHS_arguments = copy_arguments(formula->RHS_arguments);

	if ( formula->RHS_value != NULL ) {
	  copied_formula->RHS_value = (char *) malloc( sizeof(char) * ( strlen(formula->RHS_value) + 1 ) );
	  strcpy(copied_formula->RHS_value, formula->RHS_value);
	}
      }
    }
    
    /* If there is a right_formula, copy it */
    if ( formula->right_formula != NULL ) {
      copied_formula->right_formula = copy_formula(formula->right_formula);
    }
  }
  else { // if the original formula is NULL
    copied_formula = NULL;
  }
  return copied_formula;
}

/* Takes a formula and renames variables from a given module,
   by prepending an import prefix to all variable occurrences in that formula. */
void rename_variables_of_module_in_formula(struct mad_module * module, struct mad_formula * formula, int import_index) {

  if (formula != NULL) {
    
    /* If it's a quantified formula, rename the quantified variable */
    if ( formula->formula_kind == QUANTIFIER ) {
      /* We shouldn't need to check if this identifier is a variable, since it must be. 
         However, we check, just in case it doesn't turn out to be variable of that module,
         and report an error in that case. */
      if ( lookup_variable_in_module(formula->quantified_variable, module) != NULL ) {
	formula->quantified_variable = prepend_import_prefix_to_string(formula->quantified_variable, import_index);
      }
      else {
	print_error_message(stderr, "Internal error: Quantified variable %s to be renamed is not a declared variable of the module %s.\n",
	       formula->quantified_variable,
	       module->name);
      }
    }
    
    /* If there is a left_formula, rename variables in it */
    if ( formula->left_formula != NULL ) {
      rename_variables_of_module_in_formula(module, formula->left_formula, import_index);
    }
    
    /* This is where we do the essential renaming */
    if ( formula->formula_kind == ATOMIC_FORMULA 
	 || formula->formula_kind == INEQUALITY) {
      /* If it's an atom, check all identifiers, including the LHS_term, LHS_arguments, RHS_term, RHS_arguments,
	 renaming any variables encountered. */
      
      if (formula->LHS_term != NULL) {
	if ( lookup_variable_in_module(formula->LHS_term, module) != NULL ) {
	  formula->LHS_term = prepend_import_prefix_to_string(formula->LHS_term, import_index);
	}
	rename_variables_of_module_in_arguments(module, formula->LHS_arguments, import_index);
      }
      if (formula->RHS_term != NULL) {
	if ( lookup_variable_in_module(formula->RHS_term, module) != NULL ) {
	  formula->RHS_term = prepend_import_prefix_to_string(formula->RHS_term, import_index);
	}
	rename_variables_of_module_in_arguments(module, formula->RHS_arguments, import_index);
      }
    }
    
    /* If there is a right_formula, rename variables in it. */
    if ( formula->right_formula != NULL ) {
      rename_variables_of_module_in_formula(module, formula->right_formula, import_index);
    }
  }
  
}

/* Renames occurrences of a given constant in a formula, adding
   extra variables if necessary. 
   Returns 1 if it encountered an occurrence of the constant, 0 otherwise.
*/
int rename_constant_in_formula(char * constant_name,
			       struct mad_formula * formula,
			       int import_index, 
			       struct string_list_node * new_variable_list) {

  int constant_encountered = 0;

  /* If we rename a constant occurrence, we may need to add extra variables.
     This is used to go to the end of the arguments. */
  struct mad_constant_argument * current_constant_argument;
  
  /* This is used to create extra arguments corresponding to extra variables. */
  struct mad_constant_argument * new_constant_argument;

  /* This is used to go down the new_variable_list if we need to add extra arguments
     to this constant occurrence. */
  struct string_list_node * current_new_variable;

  if (formula != NULL) {
    
    /* If there is a left_formula, rename occurrences of the constant in it. */
    if ( formula->left_formula != NULL ) {
      constant_encountered = rename_constant_in_formula(constant_name, 
							formula->left_formula, 
							import_index, 
							new_variable_list);
    }
    
    /* This is where we do the essential renaming */
    if ( formula->formula_kind == ATOMIC_FORMULA 
	 || formula->formula_kind == INEQUALITY) {
      /* If it's an atomic_formula, check the LHS_term, LHS_arguments, RHS_term, RHS_arguments,
	 renaming any occurrences of the constant encountered. */

      if (formula->LHS_term != NULL ) {
	if ( strcmp(formula->LHS_term, constant_name) == 0 ) {

	  constant_encountered = 1;

	  /* First prepend the import prefix to the constant name. */
	  formula->LHS_term = prepend_import_prefix_to_string(formula->LHS_term, import_index);
	  
	  /* Now add extra arguments to the constant occurrence, if any. */
	  if ( new_variable_list != NULL ) {
	    
	    /* To do this we need to go to the end of the arguments, if there are any. */
	    current_constant_argument = formula->LHS_arguments;
	    
	    /* If there are any arguments, go to the last one */
	    if ( current_constant_argument != NULL ) {
	      /* Go down the chain of arguments to reach the last argument */
	      // current_constant_argument = current_constant_argument->next_argument;
	      while ( current_constant_argument->next_argument != NULL ) {
		current_constant_argument = current_constant_argument->next_argument;
	      }
	      /* Now add the first of the extra variables to the end of the arguments. */
	      current_constant_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	      /* Now update new_constant_argument to point to the last argument added so far. */
	      new_constant_argument = current_constant_argument->next_argument;
	    }
	    else { /* The LHS has no arguments so we create a new argument
		      corresponding to the first of the extra variables. */
	      formula->LHS_arguments = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	      /* Now update new_constant_argument to point to the last argument added so far. */
	      new_constant_argument = formula->LHS_arguments;
	    }
	    
	    /* Move on to the next extra variable to be added */
	    current_new_variable = new_variable_list->next;
	    
	    /* At this point, we have already added the first extra variable as an
	       argument to this constant occurrence, new_constant_argument points to the
	       last argument for this constant occurrence, and current_new_variable points
	       to the remaining new variables to be added. Now we can just go add them
	       one by one. */
	    while ( current_new_variable != NULL ) {
	      new_constant_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_new_variable->node_data);
	      new_constant_argument = new_constant_argument->next_argument;
	      current_new_variable = current_new_variable->next;
	    }
	  }
	}
	
	/* If the LHS_term of the atom matched the constant to be renamed, we have renamed 
	   the constant occurrence and added any extra arguments to the end.
	   
	   However, we are not done yet.  The LHS of the atomic formula may have arguments which are constants
	   themselves, so we need to rename constant occurrences in the arguments. */
	
	// update constant_encountered ONLY IF WE SEE IT 
	// (i.e. don't mark it unseen here because we might have encountered it above)
	if ( rename_constant_in_arguments(constant_name, 
					  formula->LHS_arguments, 
					  import_index, 
					  new_variable_list) == 1 ) {
	  constant_encountered = 1;
	}
      }
      
      /* Now do the renaming for the RHS term and arguments, if there are any. */
      if (formula->RHS_term != NULL ) {
	if ( strcmp(formula->RHS_term, constant_name) == 0 ) {

	  constant_encountered = 1;

	  /* First prepend the import prefix to the constant name. */
	  formula->RHS_term = prepend_import_prefix_to_string(formula->RHS_term, import_index);
	  
	  /* Now add extra arguments to the constant occurrence, if any. */
	  if ( new_variable_list != NULL ) {
	    
	    /* To do this we need to go to the end of the arguments, if there are any. */
	    current_constant_argument = formula->RHS_arguments;
	    
	    /* If there are any arguments, go to the last one */
	    if ( current_constant_argument != NULL ) {
	      /* Go down the chain of arguments to reach the last argument */
	      // current_constant_argument = current_constant_argument->next_argument;
	      while ( current_constant_argument->next_argument != NULL ) {
		current_constant_argument = current_constant_argument->next_argument;
	      }
	      /* Now add the first of the extra variables to the end of the arguments. */
	      current_constant_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	      /* Now update new_constant_argument to point to the last argument added so far. */
	      new_constant_argument = current_constant_argument->next_argument;
	    }
	    else { /* The RHS has no arguments so we create a new argument
		      corresponding to the first of the extra variables. */
	      formula->RHS_arguments = make_new_argument(PLAIN_ARGUMENT, new_variable_list->node_data);
	      /* Now update new_constant_argument to point to the last argument added so far. */
	      new_constant_argument = formula->RHS_arguments;
	    }
	    
	    /* Move on to the next extra variable to be added */
	    current_new_variable = new_variable_list->next;
	    
	    /* At this point, we have already added the first extra variable as an
	       argument to this constant occurrence, new_constant_argument points to the
	       last argument for this constant occurrence, and current_new_variable points
	       to the remaining new variables to be added. Now we can just go add them
	       one by one. */
	    while ( current_new_variable != NULL ) {
	      new_constant_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_new_variable->node_data);
	      new_constant_argument = new_constant_argument->next_argument;
	      current_new_variable = current_new_variable->next;
	    }
	  }
	}
	
	/* If the RHS_term of the atom matched the constant to be renamed, we have renamed 
	   the constant occurrence and added any extra arguments to the end.
	   
	   However, we are not done yet.  The RHS of the atomic formula may have arguments which are constants
	   themselves, so we need to rename constant occurrences in the arguments. */

	// update constant_encountered ONLY IF WE SEE IT 
	// (i.e. don't mark it unseen here because we might have encountered it above)
	if ( rename_constant_in_arguments(constant_name, 
					  formula->RHS_arguments, 
					  import_index, 
					  new_variable_list) == 1 ) {
	  constant_encountered = 1;
	}
      }

    }

    /* If there is a right_formula, rename occurrences of the constant in it. */
    if ( formula->right_formula != NULL ) {
      // update constant_encountered ONLY IF WE SEE IT 
      // (i.e. don't mark it unseen here because we might have encountered it above)
      if ( rename_constant_in_formula(constant_name, 
				      formula->right_formula, 
				      import_index, 
				      new_variable_list) == 1 ) {
	constant_encountered = 1;
      }
    }
  }
  
  return constant_encountered;
  
}


/* Given a formula and a list of variables, returns a list of any
   free variables in the formula, other than those in the given list.
   Assumes the variables have all been declared.  It won't recognize 
   undeclared variables. 
   Also, we assume that the formula has already been checked for being 
   correctly formed. 
   (Hence We don't check if identifiers are valid or not.  In fact, sometimes
   they might not appear to be, for example if we are looking for free variables
   in a constant appearing on the left hand side of an "is" statement.  In such
   a case the identifier for the constant hasn't been added to the list yet.
   However, the variables have, which is all that matters.) */
struct string_list_node * find_free_variables_in_formula(struct mad_formula * formula, struct string_list_node * given_variables) {

  /* This will be the list of free variables in the formula */
  struct string_list_node * free_variables_start;
  struct string_list_node * free_variables_end;

  /* If we encounter a quantified formula, we will wish to find the free
     variables in the subformula, which are different from the quantified variable
     and also different from the given_variables.  For this, we temporarily modify
     The given variables, adding the quantified variable, but we will remove it
     later, so we keep a marker to the end of the list */
  struct string_list_node * end_of_given_variables;
  
  /* To iterate through the list */
  struct string_list_node * current;

  /* needed to look up an identifier and check if it's a variable */
  struct id_list_node * id;


  /* Initialize the list of free variables */
  init_string_list(&free_variables_start, &free_variables_end);

  if ( formula != NULL ) {
    
    /* If it's a quantified formula, the quantified variable is not free */
    if ( formula->formula_kind == QUANTIFIER ) {

      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;

	/* Add the quantified variable to the list of given variables so that this variable
	   is not treated as a free variable when searching the subformula. */
	insert_string_into_list(&given_variables, 
				&current, 
				formula->quantified_variable);

	/* Add the free variables found in the subformula to the list of free variables */
        current = find_free_variables_in_formula(formula->right_formula, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	
	/* Now remove the quantified formula from the list of given variables */
	end_of_given_variables->next = NULL;
      }
      else { // given_variables is NULL
	
	// in this case current is NULL

	/* Add the quantified variable to the list of given variables so that this variable
	   is not treated as a free variable when searching the subformula. */
	insert_string_into_list(&given_variables, 
				&current, 
				formula->quantified_variable);

	/* Add the free variables found in the subformula to the list of free variables */
        current = find_free_variables_in_formula(formula->right_formula, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	
	/* Now remove the quantified variable from the list of given variables,
	   making given_variables point to NULL again. */
	given_variables = NULL;
      }
    }
    else if (formula->formula_kind == BINARY_CONNECTIVE) {
      /* Add the free variables found in the left subformula to the list of free variables */
      current = find_free_variables_in_formula(formula->left_formula, given_variables);
      while ( current != NULL ) {
	insert_string_into_list(&free_variables_start, 
				&free_variables_end, 
				current->node_data);
	current = current->next;
      }

      /* While looking for the free variables in the right subformula, we don't want
	 to find the ones already found in the left subformula, so we temporarily add
	 these to the list of given_variables. */

      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;

	/* Link the end of given_variables to the start of free_variables (which came
	   from the left subformula). */
	current->next = free_variables_start;
      
	/* Add the free variables found in the right subformula to the list of free variables */
	current = find_free_variables_in_formula(formula->right_formula, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}

	/* Now unlink the end of the original given_variables from the free_variables */
	end_of_given_variables->next = NULL;
      }
      else {  // given_variables is NULL
	
	/* In this case we simply give the list of free_variables as the given_variables.
	   There is no need to add anything to the end of the list */

	/* Add the free variables found in the subformula to the list of free variables */
        current = find_free_variables_in_formula(formula->right_formula, free_variables_start);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	
	/* Since we didn't modify the given_variables list, there is no need to fix it. */
      }
    }
    else if (formula->formula_kind == UNARY_CONNECTIVE) {
      /* No need to modify the given_variables list in this case */

      /* Add the free variables found in the subformula to the list of free variables */
      current = find_free_variables_in_formula(formula->right_formula, given_variables);
      while ( current != NULL ) {
	insert_string_into_list(&free_variables_start, 
				&free_variables_end, 
				current->node_data);
	current = current->next;
      }
    }
    else if (formula->formula_kind == ZERO_PLACE_CONNECTIVE) {
      // No free variables in this case, so do nothing, eventually returning the NULL list
    }
    else if ( formula->formula_kind == ATOMIC_FORMULA 
	      || formula->formula_kind == INEQUALITY) {
      /* This is the actual place we encounter free variables. */

      /* We need to check the LHS_term first, in case it is a variable */
      
      id = lookup_identifier(formula->LHS_term);
      if ( id == NULL ) {
	/* It didn't find the identifier.  This shouldn't happen because we
	   assume the formula has already been checked for correctness. */
	/* However, we only care about the variables here so we don't do anything. */
	/* print_error_message(stderr, "Internal error: Formula contains an atom with unregistered identifier.\n"); */
      }
      else if ( id->node_data->id_type == VARIABLE_ID ) {
	/* The LHS_term is a variable, so we check if it's among the given_variables. */
	current = given_variables;
	while (current != NULL ) {
	  if (strcmp(formula->LHS_term, current->node_data) == 0) {
	    // We found the variable in the given_variable list so we quit looking
	    break;
	  }
	  current = current->next;
	}
	if ( current == NULL ) { // We didn't find the identifier among the given_variables
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  formula->LHS_term);
	}
	else { // the LHS_term is an identifier but not a variable identifier
	  /* No need to do anything in this case */
	}
      }
      
      /* Now we need to check for free variables in the LHS_arguments */

      /* !!!: Actually, if the LHS_term was a variable, then we probably won't allow
	 any arguments, for a formula to be correctly formed.  However, I'll process the arguments
	 without checking that, just in case.  It won't hurt. */

      /* While looking for the free variables in the LHS_arguments, we don't want
	 to find the one corresponding to the "LHS_term" so we temporarily add
	 any free variables to the list of given_variables. */

      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;
	
	/* Link the end of given_variables to the start of free_variables (which came
	   from the LHS_term).  Note that this may be empty, if the LHS_term
	   wasn't a free variable. */
	current->next = free_variables_start;
	
	/* Add the free variables found in the arguments to the list of free variables */
	current = find_free_variables_in_arguments(formula->LHS_arguments, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}

	/* Now unlink the end of the original given_variables from the free_variables */
	end_of_given_variables->next = NULL;
      }
      else { // if the given_variables list was NULL
	/* In this case we simply give the list of free_variables (which may include a single
	   entry coming from LHS_term) as the given_variables. There is no need to add 
	   anything to the end of the list */

	/* Add the free variables found in the subformula to the list of free variables */
	current = find_free_variables_in_arguments(formula->LHS_arguments, free_variables_start);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	/* Since we didn't modify the given_variables list, there is no need to fix it. */
      }
    

      /* Now we check if the RHS_term is a free variable. */

      id = lookup_identifier(formula->RHS_term);
      if ( id == NULL ) {
	/* It didn't find the identifier.  This shouldn't happen because we
	   assume the formula has already been checked for correctness. */
	/* However, we only care about the variables here so we don't do anything. */
	/* print_error_message(stderr, "Internal error: Formula contains an atom with unregistered identifier.\n"); */
      }
      else if ( id->node_data->id_type == VARIABLE_ID ) {
	/* The RHS_term is a variable, so we check if it's among the given_variables. */
	current = given_variables;
	while (current != NULL ) {
	  if (strcmp(formula->RHS_term, current->node_data) == 0) {
	    // We found the variable in the given_variable list so we quit looking
	    break;
	  }
	  current = current->next;
	}
	if ( current == NULL ) { // We didn't find the identifier among the given_variables

	  /* But it may be among the free_variables coming from LHS_term or the LHS_arguments
	     So we search those too. */
	  current = free_variables_start;
	  while (current != NULL ) {
	    if (strcmp(formula->RHS_term, current->node_data) == 0) {
	      // We found the variable in the free_variables list so we quit looking
	      break;
	    }
	    current = current->next;
	  }
	  if ( current == NULL ) { // We didn't find the identifier among the free_variables either
	    insert_string_into_list(&free_variables_start, 
				    &free_variables_end, 
				    formula->RHS_term);
	  }
	}
	else { // the RHS_term is an identifier but not a variable identifier
	  /* No need to do anything in this case */
	}
      }
      
      /* Now we need to check for free variables in the RHS_arguments */

      /* !!!: Actually, if the RHS_term was a variable, then we probably won't allow
	 any arguments, for a formula to be correctly formed.  However, I'll process the arguments
	 without checking that, just in case.  It won't hurt. */

      /* While looking for the free variables in the RHS_arguments, we don't want
	 to find any free variables already found in the LHS_term, LHS_arguments, or RHS_term,
         so we temporarily add any free variables found so far to the list of given_variables. */

      current = given_variables;
      // first check that the given_variables list is not empty
      if ( current != NULL ) {
	while ( current->next != NULL ) {
	  current = current->next;
	}
	/* Now current will point to the last element of the list */
	end_of_given_variables = current;
	
	/* Link the end of given_variables to the start of free_variables (which came
	   from the RHS_term).  Note that this may be empty, if the RHS_term
	   wasn't a free variable. */
	current->next = free_variables_start;
	
	/* Add the free variables found in the arguments to the list of free variables */
	current = find_free_variables_in_arguments(formula->RHS_arguments, given_variables);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}

	/* Now unlink the end of the original given_variables from the free_variables */
	end_of_given_variables->next = NULL;
      }
      else { // if the given_variables list was NULL
	/* In this case we simply give the list of free_variables (which may include a single
	   entry coming from RHS_term) as the given_variables. There is no need to add 
	   anything to the end of the list */

	/* Add the free variables found in the subformula to the list of free variables */
	current = find_free_variables_in_arguments(formula->RHS_arguments, free_variables_start);
	while ( current != NULL ) {
	  insert_string_into_list(&free_variables_start, 
				  &free_variables_end, 
				  current->node_data);
	  current = current->next;
	}
	/* Since we didn't modify the given_variables list, there is no need to fix it. */
      }

    }
    else {
      print_error_message(stderr, "Internal error: Trying to find free variables in an invalid formula.\n");
    }
  }
  
  return free_variables_start;
}

/* Takes an argument structure, and checks if it has
   been formed in a valid way, using identifiers in a manner consistent
   with their declarations. */
int is_a_valid_argument(struct mad_constant_argument * argument) {

  /* To check validity of each argument if the term is a constant */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* needed to look up identifiers and check their declarations */
  struct id_list_node * id;
  struct id_list_node * value_id;

  /* For working with identifiers appearing in the term */
  struct object_list_node * current_object;
  struct constant_list_node * argument_constant;
  struct object_list_node * argument_object;
  struct constant_list_node * constant_for_argument_of_argument;
  struct variable_list_node * current_variable;
  
  if (argument != NULL ) {
    /* If the argument has arguments, we should check that they are correctly formed too. */
    if (argument->argument_kind == ARGUMENTED_ARGUMENT) {
      // This argument must be a constant or object, since it is argumented.
      // Actually, now arithmetic operators are represented this way too,
      // so it could also be an arithmetic operator.
      argument_constant = lookup_constant(argument->name);
      argument_object = lookup_object(argument->name);
      if (argument_constant == NULL 
	  && argument_object == NULL
	  && !is_an_arithmetic_operator(argument->name) ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "Identifier %s has arguments but it's not a constant/object/operator.\n", 
			    argument->name);
	/* The argument is invalid since it was given arguments,
	   even though it is not a constant or object. */
	return 0;
      }
      else if ( !is_an_arithmetic_operator(argument->name) ) { 
	// we have a constant/object with arguments
	/* We need to check that all of its arguments match its declaration and
	   that they are valid arguments too. */
	
	/* Check arguments */
	if ( argument_constant != NULL ) {
	  current_declaration_argument = argument_constant->node_data->arguments;
	}
	else {
	  current_declaration_argument = argument_object->node_data->arguments;
	}
	current_argument = argument->arguments;
	/* Run through as many arguments as were declared. */
	while (current_declaration_argument != NULL) {
	  /* First make sure that there is an argument to check. :) */
	  if ( current_argument == NULL ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Not enough arguments for constant/object %s.\n", 
				argument->name);
	    return 0;
	  }
	  
	  /* Check the argument and see if it is a constant, object or variable. */
	  id = lookup_identifier(current_argument->name);
	  if ( id == NULL 
	       && current_argument->name != NULL 
	       && !is_an_integer(current_argument->name) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Undeclared identifier %s in argument.\n", 
				       current_argument->name);
	    /* The given argument is not valid since this argument isn't. */
	    return 0;
	  }
	  else if ( current_argument->name == NULL ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "NULL identifier in term.\n");
	    /* The given argument is not valid since this argument isn't. */
	    return 0;
	  }
	  else if ( id != NULL && id->node_data->id_type == CONSTANT_ID ) {
	    constant_for_argument_of_argument = lookup_constant(current_argument->name);
	    /* If the declaration requires an action, we check that 
	       a) the argument constant is an action constant
	       b) there is no argument value and the argument constant is a Boolean action
	       c) there is an arg. value which matches the domain of the argument constant */
	    if ( strcmp(current_declaration_argument->node_data, "action") == 0 ) {
	      if ( constant_for_argument_of_argument->node_data->kind != ACTION_CONST ) {
		print_parser_error_message(stderr,
				    input_file_name, 
				    error_lineno,
				    "Action constant required as argument to %s.\n", 
				    argument->name);
		return 0;
	      }
	      if ( current_argument->has_a_value == 0 ) {
		if ( strcmp(constant_for_argument_of_argument->node_data->domain, "Boolean") != 0 ) {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      error_lineno,
				      "Non-Boolean action constant in argument has no value.\n");
		  return 0;
		}
	      }
	      else { /* The argument has a value */
		/* If the value is one of the built-in objects "true" or "false",
		   then the sort of the argument constant must be Boolean or a supersort
		   of Boolean. */
		if ( (strcmp(current_argument->value, "true") == 0 )
		     || (strcmp(current_argument->value, "false") == 0) ) {
		  if ( (strcmp(constant_for_argument_of_argument->node_data->domain, "Boolean") != 0)
		       && !(is_a_known_constant_domain_sort(constant_for_argument_of_argument->node_data->domain)
			    && is_a_subsort_of("Boolean", constant_for_argument_of_argument->node_data->domain) ) ) {
		    print_parser_error_message(stderr,
					       input_file_name, 
					       error_lineno,
					       "Value of constant %s not of the right sort.\n", 
					       current_argument->name);
		    return 0;
		  }
		}
		/* Check the value and see if it is an object or a variable. */
		value_id = lookup_identifier(current_argument->value);
		if ( value_id == NULL ) {
		  if ( current_argument->value != NULL ) {
		    print_parser_error_message(stderr,
					input_file_name, 
					error_lineno,
					"Undeclared identifier %s in argument.\n", 
					current_argument->value);
		  }
		  else {
		    print_parser_error_message(stderr,
					input_file_name, 
					error_lineno,
					"NULL identifier in argument.\n");
		  }
		  /* The given argument is not valid since this argument isn't. */
		  return 0;
		}
		else if ( value_id->node_data->id_type == OBJECT_ID ) {
		  current_object = lookup_object(current_argument->value);
		  if ( (strcmp(current_object->node_data->sort,
			       constant_for_argument_of_argument->node_data->domain) != 0)
		       && !(is_a_known_object_sort(current_object->node_data->sort)
			    && is_a_known_constant_domain_sort(constant_for_argument_of_argument->node_data->domain)
			    && is_a_subsort_of(current_object->node_data->sort, 
					       constant_for_argument_of_argument->node_data->domain) ) ) {
		    print_parser_error_message(stderr,
					       input_file_name, 
					       error_lineno,
					       "Value of constant %s not of correct sort.\n", 
					       constant_for_argument_of_argument->node_data->name);
		    return 0;
		  }
		}
		else if ( value_id->node_data->id_type == VARIABLE_ID ) {
		  current_variable = lookup_variable(current_argument->value);
		  if ( (strcmp(current_variable->node_data->sort, 
			       constant_for_argument_of_argument->node_data->domain) != 0)
		       && !(is_a_known_variable_sort(current_variable->node_data->sort)
			    && is_a_known_constant_domain_sort(constant_for_argument_of_argument->node_data->domain)
			    && is_a_subsort_of(current_variable->node_data->sort, 
					       constant_for_argument_of_argument->node_data->domain) ) ) {
		    print_parser_error_message(stderr,
					       input_file_name, 
					       error_lineno,
					       "Value of constant %s not of correct sort.\n", 
					       constant_for_argument_of_argument->node_data->name);
		    return 0;
		  }
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      error_lineno,
					     "Value of constant %s must be a variable, an object, \"true\" or \"false\".\n", 
					     constant_for_argument_of_argument->node_data->name);
		  return 0;
		}
	      }
	    }
	    else { // the declaration requires a non-action constant (i.e. a non-constant)
	      /* When declaring arguments for a constant, we allow sorts or "action"
		 so no non-action constants should normally appear as arguments of a term.
		 However, we allow constants without values,
		 as shorthand for the value of the constant at that time. 
		 (This will be expanded later on.) */
	      
	      /* We should check that there is no argument value
		 and that the domain of the constant matches the declaration */
	      if ( current_argument->has_a_value == 1 ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s cannot have a value when it appears as an argument which was declared not to require an action.\n", 
					   argument->name);
		return 0;
	      }
	      if ( (strcmp(constant_for_argument_of_argument->node_data->domain, 
			   current_declaration_argument->node_data) != 0)
		   && !(is_a_known_constant_domain_sort(constant_for_argument_of_argument->node_data->domain)
			&& is_a_known_constant_argument_sort(current_declaration_argument->node_data)
			&& is_a_subsort_of(constant_for_argument_of_argument->node_data->domain,
					   current_declaration_argument->node_data))
		   && !(is_an_integer_range(constant_for_argument_of_argument->node_data->domain)
			&& is_an_integer_range(current_declaration_argument->node_data)
			&& (lower_integer_of_range(current_declaration_argument->node_data) 
			    <= lower_integer_of_range(constant_for_argument_of_argument->node_data->domain) )
			&& (upper_integer_of_range(current_declaration_argument->node_data) 
			    >= upper_integer_of_range(constant_for_argument_of_argument->node_data->domain) ) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Domain of %s doesn't match argument declaration of %s.\n", 
					   current_argument->name, 
					   argument->name);
		/* The given argument isn't valid since an arg doesn't match the declaration. */
		return 0;
	      }
	    }
	  }
	  else if ( id != NULL && id->node_data->id_type == OBJECT_ID ) {
	    if ( current_argument->has_a_value == 1 ) {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Object argument %s has a value.\n",
				  current_argument->name);
	      return 0;
	    }
	    if ( strcmp(current_declaration_argument->node_data, "action") == 0 ) {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Action argument required but object seen instead.\n");
	      return 0;
	    }
	    current_object = lookup_object(current_argument->name);
	    if ( (strcmp(current_object->node_data->sort, 
			 current_declaration_argument->node_data) != 0) 
		 && !(is_a_known_object_sort(current_object->node_data->sort)
		      && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		      && is_a_subsort_of(current_object->node_data->sort,
					 current_declaration_argument->node_data) ) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Argument %s of constant/object %s not of correct sort.\n", 
					 current_argument->name,
					 argument->name);
	      return 0;
	    }
	  }
	  else if ( id != NULL && id->node_data->id_type == VARIABLE_ID ) {
	    if ( current_argument->has_a_value == 1 ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Variable argument %s has a value.\n",
					 current_argument->name);
	      return 0;
	    }
	    current_variable = lookup_variable(current_argument->name);
	    /* If an action is required by the declaration, we can have an action variable
	       or an explicitAction variable. */
	    if ( (strcmp(current_declaration_argument->node_data, "action") == 0)
		 && (strcmp(current_variable->node_data->sort, "action") != 0)
		 && (strcmp(current_variable->node_data->sort, "explicitAction") != 0) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Action argument required but %s variable seen instead.\n",
					 current_variable->node_data->sort);
	      return 0;
	    }
	    // If the declaration does not require an action argument, then it
	    // the argument variable's sort has to match the declaration
	    current_variable = lookup_variable(current_argument->name);
	    if ( (strcmp(current_declaration_argument->node_data, "action") != 0)
		 && (strcmp(current_variable->node_data->sort, 
			    current_declaration_argument->node_data) != 0)
		 && !(is_a_known_variable_sort(current_variable->node_data->sort)
		      && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		      && is_a_subsort_of(current_variable->node_data->sort,
					 current_declaration_argument->node_data) ) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Argument %s of constant/object %s not of correct sort.\n",
					 current_argument->name,
					 argument->name);
	      return 0;
	    }
	  }
	  else if ( is_an_integer(current_argument->name) ) {
	    if ( !(is_an_integer_range(current_declaration_argument->node_data)
		   && (lower_integer_of_range(current_declaration_argument->node_data) 
		       <= atoi(current_argument->name))
		   && (upper_integer_of_range(current_declaration_argument->node_data) 
		       >= atoi(current_argument->name)) ) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Argument %s of constant %s not within declared range for that argument.\n", 
					 current_argument->name, 
					 argument->name);
	      /* The term isn't valid since an arg doesn't match the declaration. */
	      return 0;
	    }
	  }
	  else {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Argument to constant/object %s must be a constant, variable, object, or integer.\n",
				  argument->name);
	    return 0;
	  }
	  
	  /* We finished checking that the argument matches the declaration of 
	     the given constant/object (which was given as the "argument" of this function).
	     Now we check that it has also been formed validly. */
	  
	  if ( !is_a_valid_argument(current_argument) ) {
	    /* The given argument is not valid since one of its arguments is incorrectly formed. */
	      return 0;
	    } 

	  /* Move on, to check the next arguments */
	  current_declaration_argument = current_declaration_argument->next;
	  current_argument = current_argument->next_argument;
	}
	/* After checking matches for all declared arguments, make sure there
	   are no more arguments left in the given argument */
	if ( current_argument != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Too many arguments for constant/object %s.\n", 
			      argument->name);
	  /* The given argument is not valid because a constant/object 
	     has more arguments than were declared */
	  return 0;
	}
	
      }
      else {
	// We have an arithmetic operator
	// check that all arguments are integers
	current_argument = argument->arguments;
	while (current_argument != NULL) {
	  /* Check the argument and see if it is a constant, object or variable. */
	  id = lookup_identifier(current_argument->name);
	  if ( id == NULL ) {
	    if (current_argument->name != NULL 
		&& !is_an_integer(current_argument->name) 
		&& !is_an_arithmetic_operator(current_argument->name) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Undeclared identifier %s in argument.\n", 
					 current_argument->name);
	      /* The given argument is not valid since this argument isn't. */
	      return 0;
	    }
	    else if ( current_argument->name == NULL ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "NULL identifier in term.\n");
	      /* The given argument is not valid since this argument isn't. */
	      return 0;
	    }
	  }
	  else if ( id != NULL && id->node_data->id_type == CONSTANT_ID ) {
	    constant_for_argument_of_argument = lookup_constant(current_argument->name);
	    /* When declaring arguments for a constant, we allow sorts or "action"
	       so no non-action constants should normally appear as arguments of a term.
	       However, we allow constants without values,
	       as shorthand for the value of the constant at that time. 
	       (This will be expanded later on.) */
	    
	    /* We should check that there is no argument value
	       and that the domain of the constant matches the declaration */
	    if ( current_argument->has_a_value == 1 ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Constant %s cannot have a value when it appears as an argument which was declared not to require an action.\n", 
					 argument->name);
	      return 0;
	    }
	    if ( !is_an_integer_range(constant_for_argument_of_argument->node_data->domain) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Domain of %s must be an integer to be used with %s.\n", 
					 current_argument->name, 
					 argument->name);
	      /* The given argument isn't valid since an arg doesn't match the declaration. */
	      return 0;
	    }
	  }
	  else if ( id != NULL && id->node_data->id_type == VARIABLE_ID ) {
	    if ( current_argument->has_a_value == 1 ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Variable argument %s has a value.\n",
					 current_argument->name);
	      return 0;
	    }
	    current_variable = lookup_variable(current_argument->name);
	    if ( !is_an_integer_range(current_variable->node_data->sort) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Argument %s of operator %s is not an integer.\n",
					 current_argument->name,
					 argument->name);
	      return 0;
	    }
	  }
	  else {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Argument to arithmetic operator %s must be a constant, variable, or integer.\n",
					 argument->name);
	      return 0;
	  }
	  
	  /* We finished checking that the argument matches the declaration of 
	     the given operator.
	     Now we check that it has also been formed validly. */
	  
	  if ( !is_a_valid_argument(current_argument) ) {
	    /* The given argument is not valid since one of its arguments is incorrectly formed. */
	    return 0;
	  } 
	  
	  /* Move on, to check the next arguments */
	  current_argument = current_argument->next_argument;
	}
      }
    }
    else { // the argument is plain (no arguments)
      /* In this case we just check that 
	 a) if it is a constant, its value must match the domain
	                         or its domain must be boolean if there is no value
	 b) if it is an object or variable, it shouldn't have a value
	 c) if it is true or false, it shouldn't have a value
      */
      /* First check if the term is just "true" or "false" */
      if ( (strcmp(argument->name, "true") == 0)
	   || (strcmp(argument->name, "false") == 0) ) {
	if ( argument->has_a_value == 1 ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "true/false argument %s has a value.\n",
				     argument->name);
	  return 0;
	}
	if ( argument->arguments != NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Argument \"%s\" has arguments.", 
				     argument->name);
	  return 0;
	}  
      }
      else {
	id = lookup_identifier(argument->name);
	if ( id == NULL && argument->name != NULL && !is_an_integer(argument->name) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Undeclared identifier %s in argument.\n", 
				     argument->name);
	  /* The given argument is not valid since this argument isn't. */
	  return 0;
	}	
	else if ( argument->name == NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "NULL identifier in argument.\n");
	  /* The given argument is not valid since this argument isn't. */
	  return 0;
	}
	else if ( id != NULL && id->node_data->id_type == CONSTANT_ID ) {
	  argument_constant = lookup_constant(argument->name);
	  /* If the argument is a constant, we check that 
	     a) there is no argument value and the argument constant is a Boolean constant
	     b) or the constant is an action constant
	     c)    and there is an arg. value which matches the domain of the argument constant */
	  if ( argument->has_a_value == 0 ) {
	    // If the argument does not have a value, we don't have anything to check here.
	    // So the 4-line code below is commented out!
	    // This is the 
	    // if ( strcmp(argument_constant->node_data->domain, "Boolean") != 0 ) {
	    //  print_parser_error_message(stderr,
	    //		 	    input_file_name, 
	    //			    error_lineno,
	    //                      "Non-Boolean constant in argument has no value.\n");
	    //  return 0;
	    // }
	  }
	  else { /* The argument has a value */
	    if ( argument_constant->node_data->kind != ACTION_CONST ) {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Non-action constant %s cannot have a value when it occurs as an argument.\n", 
				  argument->name);
	      return 0;
	    }
	    /* If the value is one of the built-in objects "true" or "false",
	       then the sort of the argument constant must be Boolean or a supersort
	       of Boolean. */
	    if ( (strcmp(argument->value, "true") == 0 )
		 || (strcmp(argument->value, "false") == 0) ) {
	      if ( (strcmp(argument_constant->node_data->domain, "Boolean") != 0)
		   && !(is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			&& is_a_subsort_of("Boolean", argument_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Value of constant %s not of the right sort.\n", 
					   argument->name);
		return 0;
	      }
	    }
	    /* Check the value and see if it is an object or a variable. */
	    value_id = lookup_identifier(argument->value);
	    if ( value_id == NULL ) {
	      if ( argument->value != NULL ) {
		print_parser_error_message(stderr,
				    input_file_name, 
				    error_lineno,
				    "Undeclared identifier %s in argument.\n", 
				    argument->value);
	      }
	      else {
		print_parser_error_message(stderr,
				    input_file_name, 
				    error_lineno,
				    "NULL identifier in argument.\n");
	      }
	      /* The given argument is not valid since this argument isn't. */
	      return 0;
	    }
	    else if ( value_id->node_data->id_type == OBJECT_ID ) {
	      current_object = lookup_object(argument->value);
	      if ( (strcmp(current_object->node_data->sort, 
			   argument_constant->node_data->domain) != 0)
		   && !(is_a_known_object_sort(current_object->node_data->sort)
			&& is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			&& is_a_subsort_of(current_object->node_data->sort, 
					   argument_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Value of constant %s not of correct sort.\n", 
					   argument_constant->node_data->name);
		return 0;
	      }
	    }
	    else if ( value_id->node_data->id_type == VARIABLE_ID ) {
	      current_variable = lookup_variable(argument->value);
	      if ( (strcmp(current_variable->node_data->sort, 
			   argument_constant->node_data->domain) != 0)
		   && !(is_a_known_variable_sort(current_variable->node_data->sort)
			&& is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			&& is_a_subsort_of(current_variable->node_data->sort, 
					   argument_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Value of constant %s not of correct sort.\n", 
					   argument_constant->node_data->name);
		return 0;
	      }
	    }
	    else {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Value of constant %s must be a variable, an object, \"true\" or \"false\".\n", 
				  argument_constant->node_data->name);
	      return 0;
	    }
	  }
	}
	else if ( id != NULL && id->node_data->id_type == OBJECT_ID ) {
	  if ( argument->has_a_value == 1 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Object argument %s has a value.\n",
				argument->name);
	    return 0;
	  }
	  /* Nothing to check about an object, 
	     but we should make sure the argument has no arguments. */
	  if ( argument->arguments != NULL ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Object argument has arguments.");
	    return 0;
	  }
	}
	else if ( id != NULL && id->node_data->id_type == VARIABLE_ID ) {
	  if ( argument->has_a_value == 1 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Variable argument %s has a value.\n",
				argument->name);
	    return 0;
	  }
	  /* Nothing to check about a variable, 
	     but we should make sure the argument has no arguments. */
	  if ( argument->arguments != NULL ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Variable argument has arguments.");
	    return 0;
	  }
	}
	else if ( is_an_integer(argument->name) ) {
	  if ( argument->has_a_value == 1 ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Integer argument %s has a value.\n",
				       argument->name);
	    return 0;
	  }
	  /* Nothing to check about an integer, 
	     but we should make sure the argument has no arguments. */
	  if ( argument->arguments != NULL ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Integer argument has arguments.");
	    return 0;
	  }
	}
	else { // the term is not "true", "false", a constant, object or variable 
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Only \"true\", \"false\", constants, objects, variables, integers, or arithmetic expressions may appear as arguments.\n");
	  return 0;
	}
      }
    }
    
    /* The argument was not NULL and we checked all possibilities
       for an invalid form, not encountering at any. */
    /* The argument is valid */
    return 1;
  }
  
  print_parser_error_message(stderr,
			     input_file_name, 
			     error_lineno,
			     "NULL argument.\n");
  /* A NULL argument is invalid */
  return 0;
}

/* Takes a term and an argument structure, and checks if the term is
   a numerical expression

   Assumes that is_a_valid_term has already been called on the term,
   so the structure is not checked.
*/
int is_a_valid_numerical_expression(char * term, 
				    struct mad_constant_argument * term_arguments) {

  struct constant_list_node * term_constant;
  struct variable_list_node * term_variable;

  //The only things that can have variable values are
  // i)  constant domains
  // ii) variables
  // iii) integers
  //
  // (All integers are built-in so explicity declared 
  // objects cannot be numerical, though their arguments can.)
  
  term_constant = lookup_constant(term);
  term_variable = lookup_variable(term);

  //if ( term_constant != NULL 
  //     && is_an_integer_range(term_constant->node_data->domain) ) {
  //  return 1;
  //}
  //else 
  if ( term_variable != NULL 
       && is_an_integer_range(term_variable->node_data->sort ) ) {
    return 1;
  }
  else if ( term_constant != NULL 
	    && is_an_integer_range(term_constant->node_data->domain) ) {
    return 1;
  }
  else if ( is_an_arithmetic_operator(term) ) {
    return 1;
  }
  else if ( is_an_integer(term) ) {
    return 1;
  }

  // None of the valid cases were encountered
  print_parser_error_message(stderr,
			     input_file_name, 
			     error_lineno,
			     "Term \"%s\" is not a numerical expression.", 
			     term);
  return 0;
  
}

/* Takes a term and an argument structure, and checks if the term has
   been formed in a valid way, using identifiers in a manner consistent
   with their declarations. 

   The grammar in yaccer.y accepts "sort-name formulas" under terms but
   whether a formula is such a sort-name formula is checked before
   is_a_valid_term is called, so there's no need to check it here.
*/
int is_a_valid_term(char * term, struct mad_constant_argument * term_arguments) {

  /* To check validity of each argument if the term is a constant */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* needed to look up identifiers and check their declarations */
  struct id_list_node * id;
  struct id_list_node * argument_id;
  struct id_list_node * value_id;

  /* For working with identifiers appearing in the term */
  struct object_list_node * current_object;
  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct constant_list_node * argument_constant;
  struct variable_list_node * current_variable;

  /* First check if the term is just "true" or "false" */
  if ( (strcmp(term, "true") == 0)
       || (strcmp(term, "false") == 0) ) {
    if ( term_arguments != NULL ) {
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Term \"%s\" has arguments.", 
			  term);
      return 0;
    }  
  }
  else {
    /* Check the term and see if it is a constant, object or variable. */
    id = lookup_identifier(term);
    if ( id == NULL && term!=NULL 
	 && !is_an_integer(term) 
	 && !is_an_arithmetic_operator(term) ) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "Undeclared identifier %s used in term.\n", 
				 term);
      // The term is not valid.
      return 0;
    }
    else if ( term == NULL ) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "NULL identifier in term.\n");
      /* The term is not valid. */
      return 0;
    }
    else if ( id != NULL && id->node_data->id_type == CONSTANT_ID ) {
      /* Since the term is a constant, we need to check
	 that all of its arguments (if any) match the declaration */
      
      term_constant = lookup_constant(term);
      /* Check arguments */
      current_declaration_argument = term_constant->node_data->arguments;
      current_argument = term_arguments;
      /* Run through as many arguments as were declared. */
      while (current_declaration_argument != NULL) {
	/* First make sure that there is an argument to check. :) */
	if ( current_argument == NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Not enough arguments for constant %s.\n", 
			      term);
	  return 0;
	}
	
	/* Check the argument and see if it is a constant, object or variable. */
	argument_id = lookup_identifier(current_argument->name);
	if ( argument_id == NULL  
	     && current_argument->name!=NULL 
	     && !is_an_integer(current_argument->name)
	     && !is_an_arithmetic_operator(current_argument->name) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Undeclared identifier %s in term.\n", 
				     current_argument->name);
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( current_argument->name == NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "NULL identifier in term.\n");
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == CONSTANT_ID ) {
	  argument_constant = lookup_constant(current_argument->name);
	  /* If the declaration requires an action, we check that 
	     a) the argument constant is an action constant
	     b) there is no argument value and the argument constant is a Boolean action
	     c) there is an arg. value which matches the domain of the argument constant */
	  if ( strcmp(current_declaration_argument->node_data, "action") == 0 ) {
	    if ( argument_constant->node_data->kind != ACTION_CONST ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Action constant required as argument to %s.\n", 
					 term);
	      return 0;
	    }
	    if ( current_argument->has_a_value == 0 ) {
	      if ( strcmp(argument_constant->node_data->domain, "Boolean") != 0 ) {
		print_parser_error_message(stderr,
				    input_file_name, 
				    error_lineno,
				    "Non-Boolean action constant in argument has no value.\n");
		return 0;
	      }
	    }
	    else { /* The argument has a value */
	      /* If the value is one of the built-in objects "true" or "false",
		 then the sort of the argument constant must be Boolean or a supersort
		 of Boolean. */
	      if ( (strcmp(current_argument->value, "true") == 0 )
		   || (strcmp(current_argument->value, "false") == 0) ) {
		if ( (strcmp(argument_constant->node_data->domain, "Boolean") != 0)
		     && !(is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			  && is_a_subsort_of("Boolean", argument_constant->node_data->domain) ) ) {
		  print_parser_error_message(stderr,
					     input_file_name, 
					     error_lineno,
					     "Value of constant %s not of the right sort.\n", 
					     current_argument->name);
		  return 0;
		}
	      }
	      /* Check the value and see if it is an object or a variable. */
	      value_id = lookup_identifier(current_argument->value);
	      if ( value_id == NULL ) {
		if ( current_argument->value != NULL ) {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      error_lineno,
				      "Undeclared identifier %s in term.\n", 
				      current_argument->value);
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      error_lineno,
				      "NULL identifier in term.\n");
		}
		/* The term is not valid since the argument isn't. */
		return 0;
	      }
	      else if ( value_id->node_data->id_type == OBJECT_ID ) {
		current_object = lookup_object(current_argument->value);
		if ( (strcmp(current_object->node_data->sort, 
			     argument_constant->node_data->domain) != 0)
		     && !(is_a_known_object_sort(current_object->node_data->sort)
			  && is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			  && is_a_subsort_of(current_object->node_data->sort, 
					     argument_constant->node_data->domain) ) ) {
		  print_parser_error_message(stderr,
					     input_file_name, 
					     error_lineno,
					     "Value of constant %s not of correct sort.\n", 
					     argument_constant->node_data->name);
		  return 0;
		}
	      }
	      else if ( value_id->node_data->id_type == VARIABLE_ID ) {
		current_variable = lookup_variable(current_argument->value);
		if ( (strcmp(current_variable->node_data->sort, 
			     argument_constant->node_data->domain) != 0)
		     && !(is_a_known_variable_sort(current_variable->node_data->sort)
			  && is_a_known_constant_domain_sort(argument_constant->node_data->domain)
			  && is_a_subsort_of(current_variable->node_data->sort, 
					     argument_constant->node_data->domain) ) ) {
		  print_parser_error_message(stderr,
					     input_file_name, 
					     error_lineno,
					     "Value of constant %s not of correct sort.\n", argument_constant->node_data->name);
		  return 0;
		}
	      }
	      else {
		print_parser_error_message(stderr,
				    input_file_name, 
				    error_lineno,
				    "Value of constant %s must be a variable, an object, \"true\" or \"false\".\n",
				    argument_constant->node_data->name);
		return 0;
	      }
	    }
	  }
	  else { // the declaration requires a non-action-constant (i.e. a non-constant)
	    /* When declaring arguments for a constant, we allow sorts or "action"
	       so no non-action constants should normally appear as arguments of a term.
	       However, we allow constants without values,
	       as shorthand for the value of the constant at that time. 
	       (This will be expanded later on.) */
	    
	    /* We should check that there is no argument value
	       and that the domain of the constant matches the declaration */
	    if ( current_argument->has_a_value == 1 ) {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Argument %s of %s cannot appear with a value.\n", 
				  current_argument->name, term);
	      return 0;
	    }
	    if ( (strcmp(argument_constant->node_data->domain, 
			 current_declaration_argument->node_data) != 0)
		 && !(is_a_known_constant_domain_sort(argument_constant->node_data->domain)
		      && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		      && is_a_subsort_of(argument_constant->node_data->domain,
					 current_declaration_argument->node_data) ) 
		 && !(is_an_integer_range(argument_constant->node_data->domain)
		      && is_an_integer_range(current_declaration_argument->node_data)
		      && (lower_integer_of_range(current_declaration_argument->node_data) 
			  <= lower_integer_of_range(argument_constant->node_data->domain) )
		      && (upper_integer_of_range(current_declaration_argument->node_data) 
			  >= upper_integer_of_range(argument_constant->node_data->domain) ) ) ) {
	      print_parser_error_message(stderr,
					 input_file_name, 
					 error_lineno,
					 "Domain of %s doesn't match argument declaration of %s.\n", 
					 current_argument->name, 
					 term);
	      /* The term isn't valid since an arg doesn't match the declaration. */
	      return 0;
	    }
	  }
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == OBJECT_ID ) {
	  if ( strcmp(current_declaration_argument->node_data, "action") == 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Action argument required but object seen instead.\n");
	    return 0;
	  }
	  current_object = lookup_object(current_argument->name);
	  if ( (strcmp(current_object->node_data->sort, 
		      current_declaration_argument->node_data) != 0)
	       && !(is_a_known_object_sort(current_object->node_data->sort)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(current_object->node_data->sort,
				       current_declaration_argument->node_data) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of constant %s not of correct sort.\n", 
				       current_argument->name,
				       term_constant->node_data->name);
	    return 0;
	  }
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == VARIABLE_ID ) {
	  current_variable = lookup_variable(current_argument->name);
	  /* If an action is required by the declaration, we can have an action variable
	     or an explicitAction variable. */
	  if ( (strcmp(current_declaration_argument->node_data, "action") == 0)
	       && (strcmp(current_variable->node_data->sort, "action") != 0)
	       && (strcmp(current_variable->node_data->sort, "explicitAction") != 0) ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Action argument required but %s variable seen instead.\n",
				current_variable->node_data->sort);
	    return 0;
	  }
	  // If the declaration does not require an action argument, then it
	  // the argument variable's sort has to match the declaration
	  if ( (strcmp(current_declaration_argument->node_data, "action") != 0)
	       && (strcmp(current_variable->node_data->sort, 
			  current_declaration_argument->node_data) != 0)
	       && !(is_a_known_variable_sort(current_variable->node_data->sort)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(current_variable->node_data->sort,
				       current_declaration_argument->node_data) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of constant %s not of correct sort.\n",
				       current_argument->name,
				       term_constant->node_data->name);
	    return 0;
	  }
	}
	else if ( is_an_integer(current_argument->name) ) {
	  if ( !(is_an_integer_range(current_declaration_argument->node_data)
		 && (lower_integer_of_range(current_declaration_argument->node_data) 
		     <= atoi(current_argument->name))
		 && (upper_integer_of_range(current_declaration_argument->node_data) 
		     >= atoi(current_argument->name)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of constant %s not within declared range for that argument.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else if ( is_an_arithmetic_operator(current_argument->name) ) {
	  if ( !is_an_integer_range(current_declaration_argument->node_data) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Arithmetic argument given to constant %s, which expects non-integer argument.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Argument to constant %s must be a constant, variable or object, or integer.\n", 
			      term_constant->node_data->name);
	  return 0;
	}

	/* We should also check that the argument is correctly formed. */
	if ( !is_a_valid_argument(current_argument) ) {
	  /* The term is not valid since one of its arguments is incorrectly formed. */
	  return 0;
	}
	
	/* Move on, to check the next arguments */
	current_declaration_argument = current_declaration_argument->next;
	current_argument = current_argument->next_argument;
      }
      /* After checking matches for all declared arguments, make sure there
	 are no more arguments left in the term */
      if ( current_argument != NULL ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "Too many arguments for constant %s.\n", 
			    term);
	/* The term is not valid because a constant has more arguments than were
	   declared */
	return 0;
      }
      
    }
    else if ( id != NULL && id->node_data->id_type == OBJECT_ID ) {
      /* Since the term is an object, we need to check
	 that all of its arguments (if any) match the declaration */
      term_object = lookup_object(term);
      /* Check arguments */
      current_declaration_argument = term_object->node_data->arguments;
      current_argument = term_arguments;
      /* Run through as many arguments as were declared. */
      while (current_declaration_argument != NULL) {
	/* First make sure that there is an argument to check. :) */
	if ( current_argument == NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Not enough arguments for object %s.\n", 
				     term);
	  return 0;
	}
	/* Check the argument and see if it is a constant, object or variable. */
	argument_id = lookup_identifier(current_argument->name);
	if ( argument_id == NULL  
	     && current_argument->name!=NULL 
	     && !is_an_integer(current_argument->name)
	     && !is_an_arithmetic_operator(current_argument->name) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Undeclared identifier %s in term.\n", 
				     current_argument->name);
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( current_argument->name == NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "NULL identifier in term.\n");
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == CONSTANT_ID ) {
	  argument_constant = lookup_constant(current_argument->name);
	  /* When declaring arguments for an object, we allow only sorts
	     so no constants should normally appear as arguments of a term.
	     However, we allow constants without values,
	     as shorthand for the value of the constant at that time. 
	     (This will be expanded later on.) */
	  
	  /* We should check that there is no argument value
	     and that the domain of the constant matches the declaration */
	  if ( current_argument->has_a_value == 1 ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of %s cannot appear with a value.\n", 
				       current_argument->name, term);
	    return 0;
	  }
	  if ( (strcmp(argument_constant->node_data->domain, 
		       current_declaration_argument->node_data) != 0)
	       && !(is_a_known_constant_domain_sort(argument_constant->node_data->domain)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(argument_constant->node_data->domain,
				       current_declaration_argument->node_data) ) 
	       && !(is_an_integer_range(argument_constant->node_data->domain)
		    && is_an_integer_range(current_declaration_argument->node_data)
		    && (lower_integer_of_range(current_declaration_argument->node_data) 
			<= lower_integer_of_range(argument_constant->node_data->domain) )
		    && (upper_integer_of_range(current_declaration_argument->node_data) 
			>= upper_integer_of_range(argument_constant->node_data->domain) ) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Domain of %s doesn't match argument declaration of %s.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == OBJECT_ID ) {
	  current_object = lookup_object(current_argument->name);
	  if ( (strcmp(current_object->node_data->sort, 
		       current_declaration_argument->node_data) != 0)
	       && !(is_a_known_object_sort(current_object->node_data->sort)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(current_object->node_data->sort,
				       current_declaration_argument->node_data) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of object %s not of correct sort.\n", 
				       current_argument->name,
				       term_object->node_data->name);
	    return 0;
	  }
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == VARIABLE_ID ) {
	  current_variable = lookup_variable(current_argument->name);
	  if ( (strcmp(current_variable->node_data->sort, 
		       current_declaration_argument->node_data) != 0)
	       && !(is_a_known_variable_sort(current_variable->node_data->sort)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(current_variable->node_data->sort,
				       current_declaration_argument->node_data) )
	       && !(is_an_integer_range(current_variable->node_data->sort)
		    && is_an_integer_range(current_declaration_argument->node_data)
		    && (lower_integer_of_range(current_declaration_argument->node_data) 
			<= lower_integer_of_range(current_variable->node_data->sort) )
		    && (upper_integer_of_range(current_declaration_argument->node_data) 
			>= upper_integer_of_range(current_variable->node_data->sort)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of object %s not of correct sort.\n",
				       current_argument->name,
				       term_object->node_data->name);
	    return 0;
	  }
	}
	else if ( is_an_integer(current_argument->name) ) {
	  if ( !(is_an_integer_range(current_declaration_argument->node_data)
		 && (lower_integer_of_range(current_declaration_argument->node_data) 
		     <= atoi(current_argument->name))
		 && (upper_integer_of_range(current_declaration_argument->node_data) 
		     >= atoi(current_argument->name)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of object %s not within declared range for that argument.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else if ( is_an_arithmetic_operator(current_argument->name) ) {
	  if ( !is_an_integer_range(current_declaration_argument->node_data) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Arithmetic argument given to object %s, which expects non-integer argument.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Argument to object %s must be a constant, variable or object, or integer.\n", 
				     term_constant->node_data->name);
	  return 0;
	}
	
	/* We should also check that the argument is correctly formed. */
	if ( !is_a_valid_argument(current_argument) ) {
	  /* The term is not valid since one of its arguments is incorrectly formed. */
	  return 0;
	}
	
	/* Move on, to check the next arguments */
	current_declaration_argument = current_declaration_argument->next;
	current_argument = current_argument->next_argument;
      }
      /* After checking matches for all declared arguments, make sure there
	 are no more arguments left in the term */
      if ( current_argument != NULL ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Too many arguments for object %s.\n", 
				   term);
	/* The term is not valid because an object has more arguments than were
	   declared */
	return 0;
      }
    }
    else if ( id != NULL && id->node_data->id_type == VARIABLE_ID ) {
      /* Nothing to check about a variable, 
	 but we should make sure the term has no arguments. */
      if ( term_arguments != NULL ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "Variable term has arguments.");
	return 0;
      }
    }
    else if ( is_an_integer(term) ) {
      /* Nothing to check about an integer, 
	 but we should make sure the term has no arguments. */
      if ( term_arguments != NULL ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Integer term has arguments.");
	return 0;
      }
    }
    else if ( is_an_arithmetic_operator(term) ) {
      // We should make sure that all of the arguments are integers,
      // variables for integers, constants with an integer range domain,
      // or arithmetic operators
      current_argument = term_arguments;
      /* Run through as many arguments as were declared. */
      while (current_argument != NULL) {
	/* Check the argument and see if it is a constant, object or variable. */
	argument_id = lookup_identifier(current_argument->name);
	if ( argument_id == NULL  
	     && current_argument->name!=NULL 
	     && !is_an_integer(current_argument->name)
	     && !is_an_arithmetic_operator(current_argument->name) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Undeclared identifier %s in term.\n", 
				     current_argument->name);
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( current_argument->name == NULL ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "NULL identifier in term.\n");
	  /* The term is not valid since the argument isn't. */
	  return 0;
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == CONSTANT_ID ) {
	  argument_constant = lookup_constant(current_argument->name);
	  /* When declaring arguments for an object, we allow only sorts
	     so no constants should normally appear as arguments of a term.
	     However, we allow constants without values,
	     as shorthand for the value of the constant at that time. 
	     (This will be expanded later on.) */
	  
	  /* We should check that there is no argument value
	     and that the domain of the constant matches the declaration */
	  if ( current_argument->has_a_value == 1 ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s of %s cannot appear with a value.\n", 
				       current_argument->name, term);
	    return 0;
	  }
	  if ( !is_an_integer_range(argument_constant->node_data->domain) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Domain of %s must be an integer to be used with %s.\n", 
				       current_argument->name, 
				       term);
	    /* The term isn't valid since an arg doesn't match the declaration. */
	    return 0;
	  }
	}
	else if ( argument_id != NULL && argument_id->node_data->id_type == VARIABLE_ID ) {
	  current_variable = lookup_variable(current_argument->name);
	  if ( !is_an_integer_range(current_variable->node_data->sort) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Sort of variable %s must be an integer to be used with %s.\n",
				       current_argument->name,
				       term_object->node_data->name);
	    return 0;
	  }
	}
	
	/* We should also check that the argument is correctly formed. */
	if ( !is_a_valid_argument(current_argument) ) {
	  /* The term is not valid since one of its arguments is incorrectly formed. */
	  return 0;
	}
	
	/* Move on, to check the next arguments */
	current_argument = current_argument->next_argument;
      }
    }
    else { // the term is not "true", "false", a constant, object or variable 
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "Only \"true\", \"false\", integers, constants, objects or variables may appear as terms.\n");
      return 0;
    }
  }
  
  /* The term is valid */
  return 1;
}

/* Takes a formula and checks if it has been formed in valid way, 
   using identifiers in a manner consistent with their declarations. */
int is_a_valid_formula(struct mad_formula * formula) {

  /* needed to look up identifiers and check their declarations */
  struct id_list_node * id;
  struct id_list_node * id2;

  /* For working with identifiers appearing in formulas */
  struct constant_list_node * LHS_constant;
  struct constant_list_node * RHS_constant;
  struct variable_list_node * LHS_variable;
  struct variable_list_node * RHS_variable;
  struct object_list_node * LHS_object;
  struct object_list_node * RHS_object;

  /* If the formula is an atomic formula containing a constant c and a non-constant v/o,
     we allow both c=v/o and v/o=c, but in the latter case we will swap the LHS term
     and the RHS term so that the formula becomes c=v/o. */
  char * temporary_term_holder;
  struct mad_constant_argument * temporary_argument_holder;

  if ( formula != NULL ) {
    
    if ( formula->formula_kind == QUANTIFIER ) {
      /* If it's a quantified formula, check that the quantified variable 
	 has actually been declared as a variable */
      id = lookup_identifier(formula->quantified_variable);
      if ( id == NULL ) {
	if ( is_an_integer(formula->quantified_variable) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Integer %s used where a quantified variable is needed.\n", 
				     formula->quantified_variable);
	}
	else if ( formula->quantified_variable != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Undeclared identifier %s used as quantified variable.\n", 
			      formula->quantified_variable);
	}
	else {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Quantified variable is NULL.\n");
	}
	/* The formula is not valid since the quantified variable isn't. */
	return 0;
      }
      else {
	if ( id->node_data->id_type != VARIABLE_ID ) {
	  if ( formula->quantified_variable != NULL ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"Non-variable identifier %s used as quantified variable.\n", 
				formula->quantified_variable);
	  }
	  else {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"NULL identifier used as quantified variable.\n");
	  }
	  /* The formula is not valid because the quantified variable is not a variable. */
	  return 0;
	}
      }
      if (!is_a_valid_formula(formula->right_formula) ) {
	/* The formula is not valid because the quantified subformula is not valid. */
	return 0;
      }
    }
    else if (formula->formula_kind == BINARY_CONNECTIVE) {
      if ( !is_a_valid_formula(formula->left_formula)
	   || !is_a_valid_formula(formula->right_formula) ) {
	/* The formula is not valid because one of its subformulas is not valid. */
	return 0;
      }
    }
    else if (formula->formula_kind == UNARY_CONNECTIVE) {
      if ( !is_a_valid_formula(formula->right_formula) ) {
	/* The formula is not valid because its subformulas is not valid. */
	return 0;
      }
    }
    else if (formula->formula_kind == ZERO_PLACE_CONNECTIVE) {
      // Nothing to check in this case, so do nothing
    }
    else if ( formula->formula_kind == ATOMIC_FORMULA ) {
      /* This is the main place where we check for usage of identifiers. */
      
      /* Check if this atomic formula is a "sort name formula",
	 i.e. if it is of the form SortName(variable) */
      if ( is_a_known_sort(formula->LHS_term) ) {
	if ( formula->RHS_term != NULL 
	     || formula->RHS_arguments != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "\"Sort formula\" for %s cannot have any \"=RHS\".\n", 
			      formula->LHS_term);
	  return 0;
	}
	if ( formula->LHS_arguments == NULL 
	     || formula->LHS_arguments->next_argument != NULL) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Sort name %s must have a single argument.\n", 
			      formula->LHS_term);
	  return 0;
	}
	if ( formula->LHS_arguments->argument_kind != PLAIN_ARGUMENT 
	     || formula->LHS_arguments->arguments != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Sort name %s cannot have a nested argument.\n", 
			      formula->LHS_term);
	  return 0;
	}
	if ( formula->LHS_arguments->has_a_value == 1
	     || formula->LHS_arguments->value != NULL) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Sort name %s cannot have an argument with a value.\n", 
			      formula->LHS_term);
	  return 0;
	}
	if ( lookup_variable(formula->LHS_arguments->name) == NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Sort name %s has a non-variable argument.\n", 
			      formula->LHS_term);
	  return 0;
	}
      
	/* The atomic formula is a valid "sort name formula". */
	return 1;
      }

      /* Now that we know it is not a "sort name formula", we can check
	 validity for regular formulas. */
      
      if ( !is_a_valid_term(formula->LHS_term, formula->LHS_arguments) ) {
	/* The formula is not valid because its LHS term is not valid. */
	return 0;
      }
      
      /* The RHS term may be NULL, either if the LHS term is a Boolean constant,
	 or if the LHS term is an action variable.
	 Otherwise the RHS must be a valid term. */
      if (formula->RHS_term == NULL ) {
	LHS_constant = lookup_constant(formula->LHS_term);
	LHS_variable = lookup_variable(formula->LHS_term);
	if ( !(LHS_constant != NULL
	       && strcmp(LHS_constant->node_data->domain, "Boolean") == 0)
	     && !(LHS_variable != NULL
		  && (strcmp(LHS_variable->node_data->sort, "action") == 0
		      || strcmp(LHS_variable->node_data->sort, "explicitAction") == 0) ) ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Term %s does not have an associated value, even though it is not an action variable or a Boolean constant. \n",
			      formula->LHS_term);
	  return 0;
	}
      }
      else {
	
	if (!is_a_valid_term(formula->RHS_term, formula->RHS_arguments) ) {
	  /* The formula is not valid because its RHS_term is not valid. */
	  return 0;
	}
      }


      /* If both sides are non-NULL, then we also check that, if one side
	 is an action variable, the other side is an action variable or
	 a Boolean action constant. */
      if ( formula->LHS_term != NULL 
	   && formula->RHS_term != NULL ) {
	LHS_variable = lookup_variable(formula->LHS_term);
	RHS_variable = lookup_variable(formula->RHS_term);
	LHS_constant = lookup_constant(formula->LHS_term);
	RHS_constant = lookup_constant(formula->RHS_term);

	if ( LHS_variable != NULL
	     && (strcmp(LHS_variable->node_data->sort, "action") == 0
		 || strcmp(LHS_variable->node_data->sort, "explicitAction") == 0) ) {
	  /* check that the RHS is an action variable or Boolean constant */
	  if ( RHS_variable != NULL
	       && (strcmp(RHS_variable->node_data->sort, "action") == 0
		   || strcmp(RHS_variable->node_data->sort, "explicitAction") == 0) ) {
	    /* Both sides are action variables, so there's no problem.
	       Do nothing in this case. */
	    return 1;
	  }
	  else if ( RHS_constant != NULL
		    && RHS_constant->node_data->kind == ACTION_CONST
		    && strcmp(RHS_constant->node_data->domain, "Boolean") == 0 ) {
	    /* The LHS is an action variable and the RHS is a Boolean action constant. */
	    // Swap the LHS_term with the RHS_term, that we get c=v instead of v=c.
	    temporary_term_holder = formula->RHS_term;
	    temporary_argument_holder = formula->RHS_arguments;
	    formula->RHS_term = formula->LHS_term;
	    formula->RHS_arguments = formula->LHS_arguments;
	    formula->LHS_term = temporary_term_holder;
	    formula->LHS_arguments = temporary_argument_holder;
	    return 1;
	  }
	  else {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"An atomic formula with an action variable as one term must have either an action variable or a Boolean action constant as its other term.\n");
	    return 0;
	  }
	}
	if ( RHS_variable != NULL
	     && (strcmp(RHS_variable->node_data->sort, "action") == 0
		 || strcmp(RHS_variable->node_data->sort, "explicitAction") == 0) ) {
	  /* check that the LHS is an action variable or Boolean constant */
	  if ( LHS_variable != NULL
	       && (strcmp(LHS_variable->node_data->sort, "action") == 0
		   || strcmp(LHS_variable->node_data->sort, "explicitAction") == 0)) {
	    /* Both sides are action variables, so there's no problem.
	       Do nothing in this case. */
	    return 1;
	  }
	  else if ( LHS_constant != NULL
		    && LHS_constant->node_data->kind == ACTION_CONST
		    && strcmp(LHS_constant->node_data->domain, "Boolean") == 0 ) {
	    /* The RHS is an action variable and the LHS is a Boolean action constant. */
	    // Swap the LHS_term with the RHS_term, that we get c=v instead of v=c.
	    //
	    // Wait a minute!  It's already c=v! So let's comment out the code below
	    // temporary_term_holder = formula->RHS_term;
	    // temporary_argument_holder = formula->RHS_arguments;
	    // formula->RHS_term = formula->LHS_term;
	    // formula->RHS_arguments = formula->LHS_arguments;
	    // formula->LHS_term = temporary_term_holder;
	    // formula->LHS_arguments = temporary_argument_holder;
	    return 1;
	  }
	  else {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,
				"An atomic formula with an action variable as one term must have either an action variable or a Boolean action constant as its other term.\n");
	    return 0;
	  }
	}
      }

      
      /* Both terms are validly formed so we check if they match each other.
         These are the valid cases:
	 a) true/false = Boolean constant / object / variable
         b) constant = object/variable (domain matches sort)
	 c) Boolean constant = true/false
	 d) constant = constant (of the same domain)
	 e) object/variable = constant (domain matches sort)
	 f) object/variable = object/variable/true/false
      */ 

      id = lookup_identifier(formula->LHS_term);
      id2 = lookup_identifier(formula->RHS_term);
      if ( strcmp(formula->LHS_term, "true") == 0
           || strcmp(formula->LHS_term, "false") == 0 ) {
	// The only possibility for the atomic formula to be invalid is
	// when the RHS_term is a constant whose sort is not Boolean or a supersort of Boolean.
	if ( (id2 != NULL) 
	     && (id2->node_data->id_type == CONSTANT_ID) ) {
	  RHS_constant = lookup_constant(formula->RHS_term);
	  if ( (strcmp(RHS_constant->node_data->domain, "Boolean") != 0)
	       && !(is_a_known_constant_domain_sort(RHS_constant->node_data->domain)
		    && is_a_subsort_of("Boolean", RHS_constant->node_data->domain) ) ){
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Constant %s has value \"%s\", not of the right sort.\n", 
				       formula->RHS_term,
				       formula->LHS_term);
	    return 0;
	  }
	  else { /* The LHS_term is true/false and the RHS term is a constant */
	    // Swap LHS_term with RHS_term, that we get c=true/false instead of true/false=c.
	    temporary_term_holder = formula->RHS_term;
	    temporary_argument_holder = formula->RHS_arguments;
	    formula->RHS_term = formula->LHS_term;
	    formula->RHS_arguments = formula->LHS_arguments;
	    formula->LHS_term = temporary_term_holder;
	    formula->LHS_arguments = temporary_argument_holder;
	  }
	}
      }
      else { // The LHS_term is not "true" or "false"
	if (id != NULL && id->node_data->id_type == CONSTANT_ID ) {
	  // The LHS term is a constant
	  LHS_constant = lookup_constant(formula->LHS_term);
	  /* If the RHS term is also a constant, and they have the same domain,
	     then we allow this expression constant1 = constant 2 
	     as an abbreviation for  (exists v in common_domain LHS_constant=v & RHS_constant=v)

             But if the RHS term is not a constant, we check that it matches the sort 
             of the domain of the constant in the LHS_term. */
	  RHS_constant = lookup_constant(formula->RHS_term);
	  if ( RHS_constant == NULL ) {
	    RHS_variable = lookup_variable(formula->RHS_term);
	    RHS_object = lookup_object(formula->RHS_term);
	    if ( RHS_variable != NULL ) {
	      // If it's a variable for an integer range, then check if the
	      // domain of the constant is also an integer range,
	      // and that the variable's range lies within the domain
	      if ( is_an_integer_range(RHS_variable->node_data->sort)
		   && !(is_an_integer_range(LHS_constant->node_data->domain)
			&& (lower_integer_of_range(LHS_constant->node_data->domain) 
			    <= lower_integer_of_range(RHS_variable->node_data->sort))
			&& (upper_integer_of_range(LHS_constant->node_data->domain) 
			    >= upper_integer_of_range(RHS_variable->node_data->sort)) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has variable value %s, not within range.\n",
					   formula->LHS_term,
					   formula->RHS_term);
		return 0;
	      }
	      else if ( !is_an_integer_range(RHS_variable->node_data->sort)
			&& (strcmp(RHS_variable->node_data->sort, 
				   LHS_constant->node_data->domain) != 0)
			&& !(is_a_known_variable_sort(RHS_variable->node_data->sort)
			     && is_a_known_constant_domain_sort(LHS_constant->node_data->domain)
			     && is_a_subsort_of(RHS_variable->node_data->sort, 
						LHS_constant->node_data->domain)) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has value %s not of the right sort.\n", 
					   formula->LHS_term,
					   formula->RHS_term);
		return 0;
	      }
	    }
	    else if ( RHS_object != NULL ) {
	      if ( (strcmp(RHS_object->node_data->sort, 
			   LHS_constant->node_data->domain) != 0)
		   && !(is_a_known_object_sort(RHS_object->node_data->sort)
			&& is_a_known_constant_domain_sort(LHS_constant->node_data->domain)
			&& is_a_subsort_of(RHS_object->node_data->sort, 
					   LHS_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has value %s not of the right sort.\n", 
					   formula->LHS_term,
					   formula->RHS_term);
		return 0;
	      }
	    }
	    // It might be an integer object, which is not declared
	    else if ( formula->RHS_term != NULL 
		      && is_an_integer(formula->RHS_term) ) {
	      if ( !(is_an_integer_range(LHS_constant->node_data->domain) 
		     && (lower_integer_of_range(LHS_constant->node_data->domain) 
			 <= atoi(formula->RHS_term))
		     && (upper_integer_of_range(LHS_constant->node_data->domain) 
			 >= atoi(formula->RHS_term)) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has integer value %s, not within range.\n", 
					   formula->LHS_term,
					   formula->RHS_term);
		return 0;
	      }
	    }
	    else if ( formula->RHS_term != NULL 
		      && is_an_arithmetic_operator(formula->RHS_term) ) {
	      // No need to check anything here, because this arithmetic term 
	      // has already been checked by is_a_valid_term
	    }
	    else { /* RHS_term is not a constant, object or variable, or integer, or arithmetic
		      so it must be "true" or "false"
		      We check if these are in the domain of the LHS constant */
	      if ( (strcmp(LHS_constant->node_data->domain, "Boolean") != 0)
		   && !(is_a_known_constant_domain_sort(RHS_constant->node_data->domain)
			&& is_a_subsort_of("Boolean", RHS_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has value %s not of the right sort.\n", 
					   formula->LHS_term,
					   formula->RHS_term);
		return 0;
	      }
	    }
	  }
	  else { /* Both the LHS term and the RHS term are constants.
		    We check that they have the same domain. */
	    if ( strcmp(RHS_constant->node_data->domain, 
			LHS_constant->node_data->domain) != 0 ) {
	      print_parser_error_message(stderr,
				  input_file_name, 
				  error_lineno,
				  "Constant %s and constant %s don't have the same domain.\n", 
				  formula->LHS_term,
				  formula->RHS_term);
	      return 0;
	    }
	  }
	}
	else { /* The LHS term is not a constant, nor "true"/"false",
		  so it must be an object/variable, integer, or arithmetic operator. */
	  RHS_constant = lookup_constant(formula->RHS_term);
	  if ( RHS_constant == NULL ) {
	    /* Both the RHS term and the LHS term are not constants. 
	       We allow all such cases, so there's nothing to check. */
	  }
	  else {
	    /* The RHS term is a constant and the LHS term is an object/variable,
	       so we check that the sort of the latter matches the domain of the former. */
	    LHS_variable = lookup_variable(formula->LHS_term);
	    LHS_object = lookup_object(formula->LHS_term);
	    if ( LHS_variable != NULL ) {
	      // If it's a variable for an integer range, then check if the
	      // domain of the constant is also an integer range,
	      // and that the variable's range lies within the domain
	      if ( is_an_integer_range(LHS_variable->node_data->sort)
		   && !(is_an_integer_range(RHS_constant->node_data->domain)
			&& (lower_integer_of_range(RHS_constant->node_data->domain) 
			    <= lower_integer_of_range(LHS_variable->node_data->sort))
			&& (upper_integer_of_range(RHS_constant->node_data->domain) 
			    >= upper_integer_of_range(LHS_variable->node_data->sort)) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has variable value %s, not within range.\n",
					   formula->RHS_term,
					   formula->LHS_term);
		return 0;
	      }
	      else if ( !is_an_integer_range(LHS_variable->node_data->sort)
			&& (strcmp(LHS_variable->node_data->sort, 
				   RHS_constant->node_data->domain) != 0)
			&& !(is_a_known_variable_sort(LHS_variable->node_data->sort)
			     && is_a_known_constant_domain_sort(RHS_constant->node_data->domain)
			     && is_a_subsort_of(LHS_variable->node_data->sort, 
						RHS_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has value %s not of the right sort.\n", 
					   formula->RHS_term,
					   formula->LHS_term);
		return 0;
	      }
	    }
	    else if ( LHS_object != NULL ) {
	      if ( (strcmp(LHS_object->node_data->sort, 
			   RHS_constant->node_data->domain) != 0)
		   && !(is_a_known_object_sort(LHS_object->node_data->sort)
			&& is_a_known_constant_domain_sort(RHS_constant->node_data->domain)
			&& is_a_subsort_of(LHS_object->node_data->sort, 
					   RHS_constant->node_data->domain) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has value %s not of the right sort.\n", 
					   formula->RHS_term,
					   formula->LHS_term);
		return 0;
	      }
	    }
	    // It might be an integer object, which is not declared
	    else if ( formula->LHS_term != NULL 
		      && is_an_integer(formula->LHS_term) ) {
	      if ( !(is_an_integer_range(RHS_constant->node_data->domain) 
		     && (lower_integer_of_range(RHS_constant->node_data->domain) 
			 <= atoi(formula->LHS_term))
		     && (upper_integer_of_range(RHS_constant->node_data->domain) 
			 >= atoi(formula->LHS_term)) ) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Constant %s has integer value %s, not within range.\n", 
					   formula->RHS_term,
					   formula->LHS_term);
		return 0;
	      }
	    }
	    else if ( formula->LHS_term != NULL 
		      && is_an_arithmetic_operator(formula->LHS_term) ) {
	      // No need to check anything here, because this arithmetic term 
	      // has already been checked by is_a_valid_term
	    }

	    /* The LHS_term is an object/variable and the RHS term is a constant */
	    // Swap LHS_term with RHS_term, that we get c=o/v instead of o/v=c.
	    temporary_term_holder = formula->RHS_term;
	    temporary_argument_holder = formula->RHS_arguments;
	    formula->RHS_term = formula->LHS_term;
	    formula->RHS_arguments = formula->LHS_arguments;
	    formula->LHS_term = temporary_term_holder;
	    formula->LHS_arguments = temporary_argument_holder;
	  }
	}
      }
	  
    }
    else if ( formula->formula_kind == INEQUALITY ) {
      /* This is the main place where we check for usage of identifiers. */
      
      /* Check if this atomic formula is a "sort name formula",
	 i.e. if it is of the form SortName(variable) */
      if ( is_a_known_sort(formula->LHS_term) ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "A \"sort formula\" for %s cannot occur as one side of an inequality.\n", 
				   formula->LHS_term);
	return 0;
      }
      
      /* Now that we know it is not a "sort name formula", we can check
	 validity for regular formulas. */
      
      if ( !is_a_valid_term(formula->LHS_term, formula->LHS_arguments) ) {
	/* The formula is not valid because its LHS term is not valid. */
	return 0;
      }
      
      if (!is_a_valid_term(formula->RHS_term, formula->RHS_arguments) ) {
	/* The formula is not valid because its RHS_term is not valid. */
	return 0;
      }
      
      /* Both terms are validly formed so we check if they are also both
	 numerical expressions.
      */ 
      if ( !is_a_valid_numerical_expression(formula->LHS_term, 
					    formula->LHS_arguments) ) {
	/* The formula is not valid because its LHS term is not numerical. */
	return 0;
      }

      if ( !is_a_valid_numerical_expression(formula->RHS_term, 
					    formula->RHS_arguments) ) {
	/* The formula is not valid because its RHS term is not numerical. */
	return 0;
      }
      
    }

    /* The formula is valid */
    return 1;
  }
  
  /* A NULL formula is invalid */
  print_parser_error_message(stderr,
		      input_file_name, 
		      error_lineno,
		      "NULL formula.\n");
  return 0;
}

/* Takes an atomic formula and a module, and checks if
   the atomic formula is a valid LHS of a constant
   renaming statement, where the constant being renamed
   comes from the given module. 

   Assumes the formula contains an LHS term which is a constant.
*/
int is_a_valid_constant_renaming_LHS_from_module(struct mad_formula * formula, 
						 struct mad_module * module) {

  /* To check validity of each argument if the LHS term has arguments */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* For working with identifiers appearing in the term */
  struct object_list_node * current_object;
  struct constant_list_node * term_constant;
  struct constant_list_node * argument_constant;
  struct variable_list_node * current_variable;

  /* To keep a list of the variables appearing as arguments to the constant */
  struct string_list_node * list_of_variables_seen_first;
  struct string_list_node * list_of_variables_seen_last;

  init_string_list(&list_of_variables_seen_first,
		   &list_of_variables_seen_last);
  
  /* The constant on the LHS must be followed by exactly
     the number of arguments as was declared.  
     Each argument and value must be 
     - a variable of the correct sort,
     - or, if the constant is Boolean
       + a variable of a subsort of the correct sort, or
       + an object of the correct sort (or a subsort of it).
     Furthermore, each variable must only appear once.
  */
  term_constant = lookup_constant_in_module(formula->LHS_term, module);

  /* Check arguments */
  current_declaration_argument = term_constant->node_data->arguments;
  current_argument = formula->LHS_arguments;
  /* Run through as many arguments as were declared. */
  while (current_declaration_argument != NULL) {
    /* First make sure that there is an argument to check. :) */
    if ( current_argument == NULL ) {
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "Not enough arguments for constant %s.\n", 
			  formula->LHS_term);
      return 0;
    }

    if ( strcmp(term_constant->node_data->domain, "Boolean") == 0 ) {
      /* Check if the argument is a variable. */
      current_variable = lookup_variable(current_argument->name);
      if ( current_variable != NULL ) {
	/* The sort of the variable should match the declaration. */
	if ( !is_a_known_variable_sort(current_variable->node_data->sort) 
	     && !is_an_integer_range(current_variable->node_data->sort) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Argument %s on the LHS of \"is\" has undeclared sort (%s).\n",
				     current_argument->name, 
				     current_variable->node_data->sort);
	  return 0;
	}
	if ( !is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module) 
	     && !is_an_integer_range(current_declaration_argument->node_data) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Constant %s on the LHS of \"is\" was declared with unknown sort argument (%s).\n",
				     term_constant->node_data->name, 
				     current_declaration_argument->node_data);
	  return 0;
	}
	if ( strcmp(current_variable->node_data->sort, 
		    current_declaration_argument->node_data) != 0
	     && !(is_a_known_variable_sort(current_variable->node_data->sort)
		  && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		  && is_a_subsort_of(current_variable->node_data->sort,
				     current_declaration_argument->node_data) )
	     && !(is_a_known_variable_sort_in_module(current_variable->node_data->sort, module)
		  && is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module)
		  && is_a_subsort_of_in_module(current_variable->node_data->sort,
					       current_declaration_argument->node_data,
					       module) ) 
	     && !(is_an_integer_range(current_variable->node_data->sort)
		  && is_an_integer_range(current_declaration_argument->node_data)
		  && (lower_integer_of_range(current_declaration_argument->node_data) 
		      <= lower_integer_of_range(current_variable->node_data->sort))
		  && (upper_integer_of_range(current_declaration_argument->node_data) 
		      >= upper_integer_of_range(current_variable->node_data->sort)) ) ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Sort of argument %s on the LHS of \"is\" doesn't match declaration of constant.\n",
				     current_argument->name);
	  return 0;
	}
	/* Check that the variable hasn't appeared in an argument before this one. */
	if ( lookup_string_in_list(current_argument->name, list_of_variables_seen_first) != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Variable arguments on the LHS if \"is\" must be distinct.\n");
	  return 0;
	}
	else {
	  insert_string_into_list(&list_of_variables_seen_first,
				  &list_of_variables_seen_first,
				  current_argument->name);
	}
      }
      else {
	current_object = lookup_object(current_argument->name);
	if ( current_object != NULL ) {
	  /* The sort of the object should match the declaration. */
	  if ( !is_a_known_object_sort(current_object->node_data->sort) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Argument %s on the LHS of \"is\" has undeclared sort (%s).\n",
				       current_argument->name, 
				       current_object->node_data->sort);
	    return 0;
	  }
	  if ( !is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Constant %s on the LHS of \"is\" was declared with unknown sort argument (%s).\n",
				       term_constant->node_data->name, 
				       current_declaration_argument->node_data);
	  return 0;
	  }
	  if ( strcmp(current_object->node_data->sort, 
		      current_declaration_argument->node_data) != 0 
	       && !(is_a_known_object_sort(current_object->node_data->sort)
		    && is_a_known_constant_argument_sort(current_declaration_argument->node_data)
		    && is_a_subsort_of(current_object->node_data->sort,
				       current_declaration_argument->node_data) )
	       && !(is_a_known_object_sort_in_module(current_object->node_data->sort, module)
		    && is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module)
		    && is_a_subsort_of_in_module(current_object->node_data->sort,
						 current_declaration_argument->node_data,
						 module) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Sort of argument %s on the LHS of \"is\" doesn't match declaration of constant.\n",
				       current_argument->name);
	    return 0;
	  }
	  // Since objects can have arguments too now, we need to check
	  // if the object has any variable arguments
	  if ( current_argument->arguments != NULL ) {
	    if ( !is_fully_instantiated_object_in_module_import(current_argument->name, 
								current_argument->arguments,
								module) );
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Object %s occurring on the LHS of \"is\" not fully instantiated.\n",
				       current_argument->name);
	    return 0;
	  }
	}
	else if ( is_an_integer(current_argument->name) ) {
	  if ( !is_a_known_constant_argument_sort_in_module(current_declaration_argument->node_data, module) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Constant %s on the LHS of \"is\" was declared with unknown sort argument (%s).\n",
				       term_constant->node_data->name, 
				       current_declaration_argument->node_data);
	    return 0;
	  }
	  if ( !(is_an_integer_range(current_declaration_argument->node_data)
		 && (lower_integer_of_range(current_declaration_argument->node_data) 
		     <= atoi(current_argument->name))
		 && (upper_integer_of_range(current_declaration_argument->node_data) 
		     >= atoi(current_argument->name)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Integer argument %s on the LHS of \"is\" isn't within range of declaration of constant.\n",
				       current_argument->name);
	    return 0;
	  }
	}
	else { /* The argument is neither an object nor a variable. */
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Argument %s on the LHS of \"is\" should be a variable or an object.\n",
			      current_argument->name);
	  return 0;
	}
      }
    }
    else { /* The constant is not Boolean */
      /* Check that the argument is a variable. */
      current_variable = lookup_variable(current_argument->name);
      if ( current_variable == NULL ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Argument %s on the LHS of \"is\" should be a variable.\n",
				   current_argument->name);
	return 0;
      }
      
      /* The sort of the variable should match the declaration. */
      if ( strcmp(current_variable->node_data->sort, 
		  current_declaration_argument->node_data) != 0 
	   && !(is_an_integer_range(current_variable->node_data->sort)
		&& is_an_integer_range(current_declaration_argument->node_data)
		&& (lower_integer_of_range(current_variable->node_data->sort)
		    == lower_integer_of_range(current_declaration_argument->node_data))
		&& (upper_integer_of_range(current_variable->node_data->sort)
		    == upper_integer_of_range(current_declaration_argument->node_data)) ) ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Sort of argument %s on the LHS of \"is\" doesn't match declaration of constant.\n",
				   current_argument->name);
	return 0;
      }
      
      
      /* Check that the variable hasn't appeared in an argument before this one. */
      if ( lookup_string_in_list(current_argument->name, list_of_variables_seen_first) != NULL ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Arguments on the LHS if \"is\" must be distinct.\n");
	return 0;
      }
      else {
	insert_string_into_list(&list_of_variables_seen_first,
				&list_of_variables_seen_first,
				current_argument->name);
      }
    }
    
    
    /* Move on, to check the next argument */
    current_declaration_argument = current_declaration_argument->next;
    current_argument = current_argument->next_argument;
  }
  
  /* After checking matches for all declared arguments, make sure there
     are no more arguments left in the term */
  if ( current_argument != NULL ) {
    print_parser_error_message(stderr,
			       input_file_name, 
			       error_lineno,
			       "Too many arguments for constant %s.\n", 
			       formula->LHS_term);
    /* The term is not valid because a constant has more arguments than were
       declared */
    return 0;
  }
  
  /* We don't allow constants on the RHS to have a value, and this is
     checked earlier (in yaccer.y, right after realizing that we have
     a constant renaming "is" in our hands).
     Before this modification, we used to allow values and the code
     commented out below checked that for correctness.

     // We also need to check that
     // a) there is no value on the RHS and the constant is Boolean
     // b) or the constant is not Boolean and the RHS is a variable of
     //   the same sort as the domain declared.
  
     if ( formula->RHS_term == NULL ) {
       if ( strcmp(term_constant->node_data->domain, "Boolean") != 0 ) {
         print_parser_error_message(stderr,
                             input_file_name, 
		  	     error_lineno,
			     "Non-Boolean constant %s before \"is\" doesn't have a value.\n",
			     formula->LHS_term);
	 return 0;
       }
     }
     else {
     current_variable = lookup_variable(formula->RHS_term);
     if ( current_variable == NULL ) {
       print_parser_error_message(stderr,
                           input_file_name, 
                	   error_lineno,
			   "Value of constant %s before \"is\" isn't a variable.\n",
			   formula->LHS_term);
       return 0;
     }
     else if ( strcmp(term_constant->node_data->domain, 
                      current_variable->node_data->sort) != 0 ) {
       print_parser_error_message(stderr,
                           input_file_name, 
   	                   error_lineno,
			   "Sort of variable %s before \"is\" doesn't match domain of %s.\n",
			   formula->RHS_term,
			   formula->LHS_term);
       return 0;
     }
   }
  */
  
  /* We checked all possibilities for problems and none of them was the
     case so the formula is valid as a constant renaming. */
  return 1;
  
}


/* Takes an atomic formula and a domain name, and checks if
   the atomic formula is a valid RHS of a constant
   renaming statement, where the constant being renamed
   has the given domain. 
*/
int is_a_valid_constant_renaming_RHS(struct mad_formula * formula, 
				     char * domain) {

  /* For working with identifiers appearing in the term */
  struct constant_list_node * LHS_constant;
  struct constant_list_node * RHS_constant;
  struct object_list_node * LHS_object;

  /* Used when checking whether the RHS contains more than one constant */
  struct constant_list_node * single_constant;

  /* If the formula is an atomic formula containing a constant c and 
     a non-constant true/false, we allow both c=true/false and true/false=c, but in
     the latter case we will swap the LHS term and the RHS term so that
     the formula becomes c=true/false. */
  char * temporary_term_holder;
  struct mad_constant_argument * temporary_argument_holder;

  
  /* The constant on the RHS must be followed by exactly
     the number of arguments as was declared.  Finally,
     there must be a value if and only if the constant being renamed
     is a Boolean constant.  (A value of true is like no value.  A value
     of false is as if no value but with the negated constant.)
     Each argument and value must be a variable of the correct sort.
  */
  
  /*
    We need to check that the formula is either
    - an atom (of a Boolean sort) or false, in case the given domain is Boolean
    - a constant of the right sort, if the given domain isn't Boolean
    - an object of the right sort, if the given domain isn't Boolean
  */
  if ( strcmp(domain, "Boolean") == 0 ) {
    if ( formula->formula_kind == ATOMIC_FORMULA ) {
      LHS_constant = lookup_constant(formula->LHS_term);
      /* This should be of the form of a Boolean atom. */
      if ( formula->RHS_term == NULL ) {
	/* An atomic formula with just a single LHS_term should be a constant. */
	if ( LHS_constant == NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,
			      "Only a constant, \"true\", or \"false\" is allowed on the RHS when renaming a Boolean constant.\n");
	  return 0;
	}
	else {
	  if ( strcmp(LHS_constant->node_data->domain, "Boolean") != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Non-Boolean constant seen on the RHS when renaming a Boolean constant.\n");
	    return 0;
	  }
	  /* Check that the constant is formed correctly */
	  if ( !is_a_valid_term(formula->LHS_term, formula->LHS_arguments) ) {
	    return 0;
	  }
	  else if ( !term_contains_exactly_one_constant(formula->LHS_term, 
							formula->LHS_arguments,
				                        &single_constant) ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The RHS of \"is\" contains more than one constant.\n");
	    return 0;
	  }
	}
      }
      else {
	/* Both LHS_term and and RHS_term exist.  
	   So the formula must be c=true, c=false, true=c or false=c,
	   where c is a Boolean constant. */
	if ( LHS_constant != NULL ) {
	  /* check for c=true, or c=false */
	  if ( strcmp(LHS_constant->node_data->domain, "Boolean") != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Non-Boolean constant seen on the RHS when renaming a Boolean constant.\n");
	    return 0;
	  }
	  RHS_constant = lookup_constant(formula->RHS_term);
	  if ( RHS_constant != NULL ) { 
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"More than one constant on the RHS of \"is\".\n");
	    return 0;
	  }
	  if ( strcmp(formula->RHS_term, "true") != 0
	       &&  strcmp(formula->RHS_term, "false") != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Value of Boolean atom on the RHS of \"is\" must be \"true\" or \"false\".\n");
	    return 0;
	  }
	  /* Check that constant c is formed correctly */
	  if ( !is_a_valid_term(formula->LHS_term, formula->LHS_arguments) ) {
	    return 0;
	  }
	  else if ( !term_contains_exactly_one_constant(formula->LHS_term, 
							formula->LHS_arguments,
				                        &single_constant) ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The RHS of \"is\" contains more than one constant.\n");
	    return 0;
	  }
	}
	else {
	  /* check for true=c or false=c */
	  RHS_constant = lookup_constant(formula->RHS_term);
	  if ( RHS_constant == NULL ) { 
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Two objects/variables are not allowed on the RHS when renaming a Boolean constant.\n");
	    return 0;
	  }
	  if ( strcmp(RHS_constant->node_data->domain, "Boolean") != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Non-Boolean constant seen on the RHS when renaming a Boolean constant.\n");
	    return 0;
	  }
	  if ( strcmp(formula->LHS_term, "true") != 0
	       &&  strcmp(formula->LHS_term, "false") != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"Value of Boolean atom on the RHS of \"is\" must be \"true\" or \"false\".\n");
	    return 0;
	  }
	  /* Check that constant c is formed correctly */
	  if ( !is_a_valid_term(formula->RHS_term, formula->RHS_arguments) ) {
	    return 0;
	  }
	  else if ( !term_contains_exactly_one_constant(formula->RHS_term, 
							formula->RHS_arguments,
				                        &single_constant) ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The RHS of \"is\" contains more than one constant.\n");
	    return 0;
	  }
	  /* The LHS_term is a true/false and the RHS term is a constant */
	  // Swap LHS_term with RHS_term, that we get c=true/false instead of false/true=c.
	  temporary_term_holder = formula->RHS_term;
	  temporary_argument_holder = formula->RHS_arguments;
	  formula->RHS_term = formula->LHS_term;
	  formula->RHS_arguments = formula->LHS_arguments;
	  formula->LHS_term = temporary_term_holder;
	  formula->LHS_arguments = temporary_argument_holder;
	}
      }
    }
    else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
      /* We allow the negation of Boolean constants. (To stand for c=false.) */
      if ( formula->connective != NEG ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,	    
			    "The only unary connective allowed after \"is\" is negation.\n");
	return 0;
      }
      if ( formula->right_formula->formula_kind != ATOMIC_FORMULA ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,	    
			    "Only Boolean constants may be negated on the RHS when renaming a Boolean constant.\n");
	return 0;
      }
      else {
	if ( formula->right_formula->RHS_term != NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,	    
			      "Only a single Boolean constant (with no value) may appear negated on the RHS of \"is\" when renaming a Boolean constant.\n");
	return 0;
	}
	LHS_constant = lookup_constant(formula->right_formula->LHS_term);
	if ( LHS_constant == NULL ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,	    
			      "Only Boolean constants may be negated on the RHS when renaming a Boolean constant.\n");
	  return 0;
	}
	if ( strcmp(LHS_constant->node_data->domain, "Boolean") != 0 ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,	    
			      "Non-Boolean constant seen on the RHS when renaming a Boolean constant.\n");
	  return 0;
	}
	/* Check that constant c is formed correctly */
	if ( !is_a_valid_term(formula->right_formula->LHS_term, formula->right_formula->LHS_arguments) ) {
	  return 0;
	}
	else if ( !term_contains_exactly_one_constant(formula->right_formula->LHS_term, 
						      formula->right_formula->LHS_arguments,
						      &single_constant) ) {
	  print_parser_error_message(stderr,
			      input_file_name, 
			      error_lineno,	    
			      "The RHS of \"is\" contains more than one constant.\n");
	  return 0;
	}
      }
    }
    else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
      if ( formula->connective != TRUE && formula->connective != FALSE) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,	    
			    "Renaming a Boolean constant must have an atom, \"true\", or \"false\" on the RHS.\n");
	return 0;
      }
    }
    else {
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,	    
			  "Renaming a Boolean constant must have an atom, \"true\",  or \"false\" on the RHS.\n");
      return 0;
    }
  }
  else { /* The constant being renamed is multi-valued */
    if ( formula->formula_kind == ATOMIC_FORMULA ) {
      if ( formula->RHS_term != NULL ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,	    
			    "Renaming a multi-valued constant must have a single constant or an object on the RHS.\n");
	return 0;
      }
      else {
	/* There is a single term.
	   Check if it's a constant or object of the right sort.
	   (i.e. is it of the given domain?) */
	LHS_constant = lookup_constant(formula->LHS_term);
	LHS_object = lookup_object(formula->LHS_term);
	if ( LHS_constant != NULL ) {
	  if ( strcmp(domain,
		      LHS_constant->node_data->domain) != 0 ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The constant on the RHS must have the same domain as the renamed constant.\n");
	    return 0;
	  }
	  /* We also need to check that the constant term is formed correctly */
	  if ( !is_a_valid_term(formula->LHS_term, formula->LHS_arguments) ) {
	    return 0;
	  }
	  else if ( !term_contains_exactly_one_constant(formula->LHS_term, 
							formula->LHS_arguments,
				                        &single_constant) ) {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The RHS of \"is\" contains more than one constant.\n");
	    return 0;
	  }
	}
	else if ( LHS_object != NULL ) {
	  if ( strcmp(domain, 
		      LHS_object->node_data->sort) != 0 
	       && !(is_a_known_object_sort(LHS_object->node_data->sort)
		    && is_a_known_constant_domain_sort(domain)
		    && is_a_subsort_of(LHS_object->node_data->sort, domain) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,	    
				       "The object on the RHS must belong to the domain of the renamed constant.\n");
	    return 0;
	  }
	  else {
	    // We need to check that it is fully instantiated
	    // (Since objects can have arguments too now, we need to check
	    //  if the object has any variable arguments)
	    if ( formula->LHS_arguments != NULL ) {
	      if ( !is_fully_instantiated_object(formula->LHS_term, 
						 formula->LHS_arguments) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "Object %s occurring on the RHS of \"is\" not fully instantiated.\n",
					   formula->LHS_term);
		return 0;
	      }
	    }
	  }
	}
	else if ( is_an_integer(formula->LHS_term) ) {
	  if ( !(is_an_integer_range(domain)
		 && (lower_integer_of_range(domain) 
		     <= atoi(formula->LHS_term))
		 && (upper_integer_of_range(domain) 
		     >= atoi(formula->LHS_term)) ) ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "Integer %s on the RHS of \"is\" isn't within domain of renamed constant.\n",
				       formula->LHS_term);
	    return 0;
	  }
	}
	else {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,	    
				     "Renaming a multi-valued constant must have a constant, object, or integer on the RHS.\n");
	  return 0;
	}
      }
    }
    else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
      /* Even though we expect only a single constant or object,
	 it is possible that we see "true" or "false".  These are
	 zero place connectives but also Boolean objects. */
      if ( formula->connective != TRUE && formula->connective != FALSE) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,	    
			    "Renaming a multi-valued constant must have a constant or an object on the RHS.\n");
	return 0;
      }      
      else {
	if ( !(is_a_known_constant_domain_sort(domain)
	       && is_a_subsort_of("Boolean", domain) ) ) {
	  if ( formula->connective == TRUE ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,	    
				       "The object \"true\" on the RHS does not belong to the domain of the renamed constant.\n");
	  }
	  else {
	    print_parser_error_message(stderr,
				input_file_name, 
				error_lineno,	    
				"The object \"false\" on the RHS does not belong to the domain of the renamed constant.\n");
	  }
	  return 0;
	}
      }

    }
    else {
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,	    
			  "Renaming a multi-valued constant must have a constant or an object on the RHS.\n");
      return 0;
    }
  }
  
  /* We checked all possibilities for problems and none of them was the
     case so the formula is valid as a constant renaming. */
  return 1;
  
}

/* When the LHS constant in a renaming is Boolean, we need to check if
   the constant has variable or object arguments that need to be "generalized".
   This function computes the "generalized atom" and the corresponding renaming cases. */
void obtain_generalization_of_Boolean_renamed_constant(struct mad_formula * renaming_atom,
						       struct mad_module * module,
						       struct mad_formula ** generalized_atom,
						       struct renaming_case_list_node ** cases) {
  
  /* To check validity of each argument if the LHS term has arguments */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* For working with identifiers appearing in the term */
  struct object_list_node * current_object;
  struct constant_list_node * term_constant;
  struct constant_list_node * argument_constant;
  struct variable_list_node * current_variable;

  struct string_list_node * new_variable_string_list_first;
  struct string_list_node * new_variable_string_list_last;

  struct renaming_case_list_node * generalized_atom_renaming_case_list_first;
  struct renaming_case_list_node * generalized_atom_renaming_case_list_last;;

  char * new_variable;

  struct mad_formula * new_case_condition;
  struct mad_formula * new_case_formula;

  int generalization_necessary;

  /* We start by assuming that no generalization is necessary */
  generalization_necessary = 0;

  (*generalized_atom) = copy_formula(renaming_atom);
    
  init_renaming_case_list(&generalized_atom_renaming_case_list_first,
			  &generalized_atom_renaming_case_list_last);

  term_constant = lookup_constant_in_module((*generalized_atom)->LHS_term, module);
  
  /* Check arguments */
  current_declaration_argument = term_constant->node_data->arguments;
  current_argument = (*generalized_atom)->LHS_arguments;
  /* Run through as many arguments as were declared. */
  while (current_declaration_argument != NULL) {
    /* Check if the argument is a variable. */
    current_variable = lookup_variable(current_argument->name);
    if ( current_variable != NULL ) {
      /* See if the sort of the variable matches the declaration. */
      if ( strcmp(current_variable->node_data->sort, 
		  current_declaration_argument->node_data) != 0 ) {

	generalization_necessary = 1;

	/* Generalize the current variable to have the sort of the declaration,
	   and add a case that expresses the constant is false when the
	   generalized argument is not of the same sort as the current_variable */

	new_variable = make_new_variable_name(current_declaration_argument->node_data);
	/* Add this new variable to the list of identifiers and to the list of variables */
	insert_identifier(new_variable);
	insert_variable(new_variable);
	init_string_list(&new_variable_string_list_first,
			 &new_variable_string_list_last);
	insert_string_into_list(&new_variable_string_list_first,
				&new_variable_string_list_last,
				new_variable);
	update_variables(new_variable_string_list_first, current_declaration_argument->node_data);

	/* The condition for the new case will be "-Sort_of_current_argument(new_variable)". */
	new_case_condition = make_new_formula(UNARY_CONNECTIVE);
	new_case_condition->connective = NEG;
	new_case_condition->right_formula = make_new_formula(ATOMIC_FORMULA);
	new_case_condition->right_formula->LHS_term = (char *) malloc( sizeof(char) * strlen(current_variable->node_data->sort) + 1);
	strcpy(new_case_condition->right_formula->LHS_term, current_variable->node_data->sort);
	new_case_condition->right_formula->LHS_arguments = make_new_argument(PLAIN_ARGUMENT,
									     new_variable);

	new_case_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	new_case_formula->connective = FALSE;

	insert_renaming_case_into_list(&generalized_atom_renaming_case_list_first,
				       &generalized_atom_renaming_case_list_last,
				       make_new_renaming_case(new_case_condition, 
							      new_case_formula));

	/* Now we generalize the current argument */
	current_argument->name = (char *) malloc( sizeof(char) * strlen(new_variable) + 1);
	strcpy(current_argument->name, new_variable);
      }
    }
    else {
      current_object = lookup_object(current_argument->name);
      if ( current_object != NULL ) {

	generalization_necessary = 1;
	
	/* Generalize the current argument( which is an object) to be 
	   a variable having the sort of the declaration,
	   and add a case that expresses the constant is false when the
	   generalized argument is not the same as the current argument */
	
	new_variable = make_new_variable_name(current_declaration_argument->node_data);
	/* Add this new variable to the list of identifiers and to the list of variables */
	insert_identifier(new_variable);
	insert_variable(new_variable);
	init_string_list(&new_variable_string_list_first,
			 &new_variable_string_list_last);
	insert_string_into_list(&new_variable_string_list_first,
				&new_variable_string_list_last,
				new_variable);
	update_variables(new_variable_string_list_first, current_declaration_argument->node_data);
	
	/* The condition for the new case will be "-(current_argument=new_variable)". */
	new_case_condition = make_new_formula(UNARY_CONNECTIVE);
	new_case_condition->connective = NEG;
	new_case_condition->right_formula = make_new_formula(ATOMIC_FORMULA);
	new_case_condition->right_formula->LHS_term = (char *) malloc( sizeof(char) * strlen(current_argument->name) + 1);
	strcpy(new_case_condition->right_formula->LHS_term, current_argument->name);
	new_case_condition->right_formula->LHS_arguments = copy_arguments(current_argument->arguments);
	new_case_condition->right_formula->RHS_term = (char *) malloc( sizeof(char) * strlen(new_variable) + 1);
	strcpy(new_case_condition->right_formula->RHS_term, new_variable);
	
	new_case_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	new_case_formula->connective = FALSE;
	
	insert_renaming_case_into_list(&generalized_atom_renaming_case_list_first,
				       &generalized_atom_renaming_case_list_last,
				       make_new_renaming_case(new_case_condition, 
							      new_case_formula));
	
	/* Now we generalize the current argument */
	current_argument->name = (char *) malloc( sizeof(char) * strlen(new_variable) + 1);
	strcpy(current_argument->name, new_variable);
      }
      else if ( is_an_integer(current_argument->name) ){
	generalization_necessary = 1;
	
	/* Generalize the current argument( which is an integer) to be 
	   a variable having the sort of the declaration,
	   and add a case that expresses the constant is false when the
	   generalized argument is not the same as the current argument */
	
	new_variable = make_new_variable_name(current_declaration_argument->node_data);
	/* Add this new variable to the list of identifiers and to the list of variables */
	insert_identifier(new_variable);
	insert_variable(new_variable);
	init_string_list(&new_variable_string_list_first,
			 &new_variable_string_list_last);
	insert_string_into_list(&new_variable_string_list_first,
				&new_variable_string_list_last,
				new_variable);
	update_variables(new_variable_string_list_first, current_declaration_argument->node_data);
	
	/* The condition for the new case will be "-(current_argument=new_variable)". */
	new_case_condition = make_new_formula(UNARY_CONNECTIVE);
	new_case_condition->connective = NEG;
	new_case_condition->right_formula = make_new_formula(ATOMIC_FORMULA);
	new_case_condition->right_formula->LHS_term = (char *) malloc( sizeof(char) * strlen(current_argument->name) + 1);
	strcpy(new_case_condition->right_formula->LHS_term, current_argument->name);
	new_case_condition->right_formula->LHS_arguments = copy_arguments(current_argument->arguments);
	new_case_condition->right_formula->RHS_term = (char *) malloc( sizeof(char) * strlen(new_variable) + 1);
	strcpy(new_case_condition->right_formula->RHS_term, new_variable);
	
	new_case_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	new_case_formula->connective = FALSE;
	
	insert_renaming_case_into_list(&generalized_atom_renaming_case_list_first,
				       &generalized_atom_renaming_case_list_last,
				       make_new_renaming_case(new_case_condition, 
							      new_case_formula));
	
	/* Now we generalize the current argument */
	current_argument->name = (char *) malloc( sizeof(char) * strlen(new_variable) + 1);
	strcpy(current_argument->name, new_variable);
      }
    }
    
    
    /* Move on, to check the next argument */
    current_declaration_argument = current_declaration_argument->next;
    current_argument = current_argument->next_argument;
  }

  /* Now we have checked whether generalization was necessary and 
     if it was, made the generalized atom and cases. 
     Since we started by copying the renaming atom to generalized atom,
     if no generalization was done, we make generalized_atom NULL again.)
  */
  if ( !generalization_necessary ) {
    (*generalized_atom) = NULL;
  }

  (*cases) = generalized_atom_renaming_case_list_first;
    
}


/* Takes a term and an argument structure, and a pointer to a constant_list_node.
   Checks if the term has exactly one constant in it and returns 0 or 1.
   If it returns 1, it also sets the given pointer to point to the constant
   which is the only constant in the term. 
   If the single constant is an action variable, the pointer is set to NULL.
*/
int term_contains_exactly_one_constant(char * term, 
				       struct mad_constant_argument * term_arguments,
				       struct constant_list_node ** single_constant_ptr) {

  /* To check each argument if the term is a constant */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* For working with identifiers appearing in the term */
  struct constant_list_node * term_constant;
  struct constant_list_node * argument_constant;
  struct variable_list_node * term_variable;
  struct variable_list_node * argument_variable;
  struct object_list_node * term_object;

  // To flag whether we've seen a constant yet, used while
  // going through arguments of an object
  int one_constant_seen;

  term_constant = lookup_constant(term);
  if ( term_constant != NULL ) {
    /* We've already found one constant.  Now check the arguments. */
    current_declaration_argument = term_constant->node_data->arguments;
    current_argument = term_arguments;
    /* Run through as many arguments as were declared. */
    while (current_declaration_argument != NULL && current_argument != NULL) {
      /* We only need to examine cases where the declaration doesn't require
	 an action.  (Because if an action is required, the action name 
	 appearing there will not be a constant but an "object"). */
      if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	argument_constant = lookup_constant(current_argument->name);
	if ( argument_constant != NULL ) {
	  /* More than one constant in the term */
	  return 0;
	}
	else {  // the argument is not a constant
	  /* This argument is not a constant.  We need to check if
	     it is a variable for an action or explicitAction, because in
	     these cases the ground term would contain more than one constant. */
	  argument_variable = lookup_variable(current_argument->name);
	  if (argument_variable != NULL
	      && is_a_known_variable_sort(argument_variable->node_data->sort)
	      && ( (strcmp(argument_variable->node_data->sort, "action") == 0)
		   || (strcmp(argument_variable->node_data->sort, "explicitAction") == 0)
		   || is_a_subsort_of("action", argument_variable->node_data->sort)
		   || is_a_subsort_of("explicitAction", argument_variable->node_data->sort) ) ) {
	    (*single_constant_ptr) = NULL;
	    return 0;
	  }
	}
      }
      /* Besides checking if this argument is a constant,
	 if it's a constant (which may just be an action-name "object"),
	 or an object, we need to check if it contains a constant 
	 in its arguments. */
      
      if ( find_formula_kind_of_argument_arguments(current_argument) != CONSTANTLESS_FORMULA ) {
	return 0;
      }
      /* Move on, to check the next arguments */
      current_declaration_argument = current_declaration_argument->next;
      current_argument = current_argument->next_argument;
    }
    /* No constants were found in the arguments */
    (*single_constant_ptr) = term_constant;
    return 1;
  }
  else {
    /* The term is not a constant.  We need to check if
       it is a variable for an action or explicitAction, because in
       these cases the ground term would contain exactly one constant. */
    term_variable = lookup_variable(term);
    if ( term_variable != NULL
	 && ((strcmp(term_variable->node_data->sort, "action") == 0)
	     || strcmp(term_variable->node_data->sort, "explicitAction") == 0) ) {
      (*single_constant_ptr) = NULL;
      return 1;
    }
    // It may also be an object which contains "shorthand" constants nested
    // in it as arguments, so we need to check those
    // 
    // Arithmetic operators can contain constants in their arguments.
    // so we should consider their arguments too
    term_object = lookup_object(term);
    if ( term_object != NULL || is_an_arithmetic_operator) {
      /* Haven't seen any constants yet, but need to check the arguments. */
      one_constant_seen = 0;
      current_argument = term_arguments;
      while ( current_argument != NULL ) {
	argument_constant = lookup_constant(current_argument->name);
	if ( argument_constant != NULL ) {
	  if ( one_constant_seen ) {
	    // More than one constant in the term
	    return 0;
	  }
	  else {
	    // Record seeing this constant
	    (*single_constant_ptr) = argument_constant;
	    one_constant_seen = 1;
	  }
	}
	else {  // the argument is not a constant
	  /* This argument is not a constant.  We need to check if
	     it is a variable for an action or explicitAction, because in
	     these cases the ground term would contain more than one constant. */
	  argument_variable = lookup_variable(current_argument->name);
	  if (argument_variable != NULL
	      && is_a_known_variable_sort(argument_variable->node_data->sort)
	      && ( (strcmp(argument_variable->node_data->sort, "action") == 0)
		   || (strcmp(argument_variable->node_data->sort, "explicitAction") == 0)
		   || is_a_subsort_of("action", argument_variable->node_data->sort)
		   || is_a_subsort_of("explicitAction", argument_variable->node_data->sort) ) ) {
	    if ( one_constant_seen ) {
	      // More than one constant in the term
	      (*single_constant_ptr) = NULL;
	      return 0;
	    }
	    else {
	      // Record seeing this constant
	      (*single_constant_ptr) = NULL;
	      one_constant_seen = 1;
	    }
	  }

	  /* Besides checking if this argument is a constant,
	     if it's a constant (which may just be an action-name "object"),
	     or an object, we need to check if it contains a constant 
	     in its arguments. */

	  // If we haven't seen any constants yet, we need to make see if the current
	  // argument has exactly one constant
	  if ( !one_constant_seen) {
	      if ( term_contains_exactly_one_constant(current_argument->name, current_argument->arguments, single_constant_ptr) ) {
		// Record seeing this constant
		one_constant_seen = 1;
	      }
	  }
	  // but if we've already seen a constant, we need to make sure there are no more
	  else {
	    if ( find_formula_kind_of_argument_arguments(current_argument) != CONSTANTLESS_FORMULA ) {						      
	      return 0;
	    }
	  }
	}
	
	/* Move on, to check the next arguments */
	current_argument = current_argument->next_argument;
      }
      
      // If we saw a constant while going through all arguments,
      // and we were able to reach the end (without seeing another), report this
      if ( one_constant_seen ) {
	return 1;
      }
    }
    
    /* No constants in the term. */
    return 0;
  }
}


/* Takes a formula, which comes from the head of an axiom,
   and checks if this formula would allow the axiom to be
   a definite causal law. 
   i.e., the formula is either
         a) the negation of an atomic formula with exactly one constant 
	    (and that constant is Boolean)
         b) a single atomic formula with at most one constant
         c) the zero-place connective false

   Assumes the formula has been correctly formed.
*/ 
int is_a_definite_head_formula(struct mad_formula * formula) {

  struct constant_list_node * single_constant;

  /* Initially assume there is no single_constant in the formula */
  single_constant = NULL;

  if ( formula != NULL ) {
    if ( formula->formula_kind == QUANTIFIER ) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,	  
				 "A quantifier cannot appear in the head of an axiom.\n");
      return 0;
    }
    else if (formula->formula_kind == BINARY_CONNECTIVE) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "A binary connective cannot appear in the head of an axiom.\n");
      return 0;
    }
    else if (formula->formula_kind == UNARY_CONNECTIVE) {
      if ( formula->connective != NEG ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "No unary connective other than negation can appear in the head of an axiom.\n");
	return 0;
      }
      else {
	/* Check if the formula in the scope of negation is an atomic formula
	   with exactly one constant (and that constant is Boolean). */
	if ( formula->right_formula != NULL ) {
	  if (formula->right_formula->formula_kind != ATOMIC_FORMULA) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "A negation in the head of an axiom must be followed by an atomic formula.\n");
	    return 0;
	  }
	  /* Before we check that the atomic formula has exactly one constant,
	     we should check that it is not a "sort name predicate" */
	  if ( is_a_known_sort(formula->right_formula->LHS_term) ) { 
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "A negation in the head of an axiom cannot be followed by a \"sort name predicate\" (%s).\n",
				       formula->right_formula->LHS_term);
	    return 0;
	  }
	  else {
	    if ( term_contains_exactly_one_constant(formula->right_formula->LHS_term, 
						    formula->right_formula->LHS_arguments,
						    &single_constant) ) {
	      if ( (single_constant == NULL) // the constant was a variable for actions
		   || (strcmp(single_constant->node_data->domain, "Boolean") != 0) ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "A negation in the head of an axiom cannot precede non-Boolean constants.\n");
		return 0;
	      }
	      /* Since we already found a single Boolean constant in the LHS term,
		 we need to make sure that the RHS term is constantless */
	      if ( formula->right_formula->RHS_term != NULL ) {
		if ( find_formula_kind_of_term(formula->right_formula->RHS_term, 
					       formula->right_formula->RHS_arguments)
		     != CONSTANTLESS_FORMULA ) {
		  print_parser_error_message(stderr,
					     input_file_name, 
					     error_lineno,
					     "At most one constant may appear in the head of an axiom.\n");
		  return 0;
		}
	      }
	    }
	    else {  /* The LHS term contains zero or more constants */
	      if ( find_formula_kind_of_term(formula->right_formula->LHS_term, 
					     formula->right_formula->LHS_arguments)
		   != CONSTANTLESS_FORMULA ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "At most one constant may appear in the head of an axiom.\n");
		return 0;
	      }
	      else { /* The LHS term has no constants, so we check the RHS term. */
		if ( formula->right_formula->RHS_term != NULL
		     && formula->right_formula->RHS_arguments != NULL
		     && term_contains_exactly_one_constant(formula->right_formula->RHS_term, 
							   formula->right_formula->RHS_arguments,
							   &single_constant) ) {
		  if ( (single_constant == NULL) // the constant was a variable for actions
		       || (strcmp(single_constant->node_data->domain, "Boolean") != 0) ) {
		    print_parser_error_message(stderr,
					       input_file_name, 
					       error_lineno,
					       "A negation in the head of an axiom cannot precede non-Boolean constants.\n");
		    return 0;
		  }
		}
		else { /* The RHS term contains zero or more constants */
		  print_parser_error_message(stderr,
					     input_file_name, 
					     error_lineno,
					     "Exactly one constant must appear after a negation in the head of an axiom.\n");
		  return 0;
		}
	      }
	    }
	  }
	}
	else {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "NULL formula follows negation.\n");
	  return 0;
	}
      }
    }
    else if (formula->formula_kind == ZERO_PLACE_CONNECTIVE) {
      if ( formula->connective == TRUE ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Zero-place connective \"true\" cannot appear as the head of an axiom.\n");
	return 0;
      }
    }
    else if ( formula->formula_kind == ATOMIC_FORMULA ) {
      /* Before we check that the atomic formula has at most one constant,
	 we should check that it is not a "sort name predicate" */
      if ( is_a_known_sort(formula->LHS_term) ) {
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "The head of an axiom cannot contain a \"sort name predicate\" (%s).\n",
				   formula->LHS_term);
	return 0;
      }
      /* Check that the formula has at most one constant. */
      if ( term_contains_exactly_one_constant(formula->LHS_term, 
					      formula->LHS_arguments,
					      &single_constant) ) {
	/* Since we already found a single constant in the LHS term,
	   we need to make sure that the RHS term is constantless */
	if ( formula->RHS_term != NULL ) {
	  if ( find_formula_kind_of_term(formula->RHS_term, 
					 formula->RHS_arguments)
	       != CONSTANTLESS_FORMULA ) {
	    print_parser_error_message(stderr,
				       input_file_name, 
				       error_lineno,
				       "At most one constant may appear in the head of an axiom.\n");
	    return 0;
	  }
	}
      }
      else {  /* The LHS term contains zero or more constants */
	if ( find_formula_kind_of_term(formula->LHS_term, 
				       formula->LHS_arguments)
	     != CONSTANTLESS_FORMULA ) {
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "At most one constant may appear in the head of an axiom.\n");
	  return 0;
	}
	else { /* The LHS term has no constants, so we check the RHS term. */
	  if ( formula->RHS_term != NULL ) {
	    if ( !term_contains_exactly_one_constant(formula->RHS_term, 
						     formula->RHS_arguments,
						     &single_constant) ) {
	      /* The RHS term contains zero or more constants */
	      if ( find_formula_kind_of_term(formula->RHS_term, 
					     formula->RHS_arguments)
		   != CONSTANTLESS_FORMULA ) {
		print_parser_error_message(stderr,
					   input_file_name, 
					   error_lineno,
					   "At most one constant may appear in the head of an axiom.\n");
		return 0;
	      }
	    }
	  }
	}
      }
    }
    else if (formula->formula_kind == INEQUALITY) {
      print_parser_error_message(stderr,
				 input_file_name, 
				 error_lineno,
				 "An inequality cannot appear in the head of an axiom.\n");
      return 0;
    }
    
    /* We checked all possibilities that may lead to a nondefinite head
       and none of them were problematic so the formula is a good definite head. */
    return 1;
  }
  /* A NULL formula is not a good definite head */
  print_parser_error_message(stderr,
		      input_file_name, 
		      error_lineno,
		      "NULL formula.\n");
  return 0;
}

/* Takes two formula kinds and returns the kind of 
   a new formula would have if it were composed of two
   subformulas with the given formula kinds. */
enum formula_kind_type combination_of_two_formula_kinds(enum formula_kind_type kind1,
							enum formula_kind_type kind2) {
  if ( kind1 == CONSTANTLESS_FORMULA ) {
    return kind2;
  }
  else if ( kind1 == SIMPLE_FLUENT_FORMULA ) {
    switch ( kind2 ) {
    case CONSTANTLESS_FORMULA:
      return SIMPLE_FLUENT_FORMULA;
      break;
    case SIMPLE_FLUENT_FORMULA:
      return SIMPLE_FLUENT_FORMULA;
      break;
    case RIGID_FORMULA:
      return FLUENT_FORMULA;
      break;
    case FLUENT_FORMULA:
      return FLUENT_FORMULA;
      break;
    case ACTION_FORMULA:
      return FORMULA;
      break;
    case FORMULA:
      return FORMULA;
      break;
    }
  }
  else if ( kind1 == RIGID_FORMULA ) {
    switch ( kind2 ) {
    case CONSTANTLESS_FORMULA:
      return RIGID_FORMULA;
      break;
    case SIMPLE_FLUENT_FORMULA:
      return FLUENT_FORMULA;
      break;
    case RIGID_FORMULA:
      return RIGID_FORMULA;
      break;
    case FLUENT_FORMULA:
      return FLUENT_FORMULA;
      break;
    case ACTION_FORMULA:
      return FORMULA;
      break;
    case FORMULA:
      return FORMULA;
      break;
    }
  }
  else if ( kind1 == FLUENT_FORMULA ) {
    switch ( kind2 ) {
    case CONSTANTLESS_FORMULA:
      return FLUENT_FORMULA;
      break;
    case SIMPLE_FLUENT_FORMULA:
      return FLUENT_FORMULA;
      break;
    case RIGID_FORMULA:
      return FLUENT_FORMULA;
      break;
    case FLUENT_FORMULA:
      return FLUENT_FORMULA;
      break;
    case ACTION_FORMULA:
      return FORMULA;
      break;
    case FORMULA:
      return FORMULA;
      break;
    }
  }
  else if ( kind1 == ACTION_FORMULA ) {
    switch ( kind2 ) {
    case CONSTANTLESS_FORMULA:
      return ACTION_FORMULA;
      break;
    case SIMPLE_FLUENT_FORMULA:
      return FORMULA;
      break;
    case RIGID_FORMULA:
      return FORMULA;
      break;
    case FLUENT_FORMULA:
      return FORMULA;
      break;
    case ACTION_FORMULA:
      return ACTION_FORMULA;
      break;
    case FORMULA:
      return FORMULA;
      break;
    }
  }
  else if ( kind1 == FORMULA ) {
    return FORMULA;
  }
}

/* Takes an argument structure and returns a formula kind 
   (i.e., action formula, fluent formula, etc.) for it,
   based on any constants appearing as its arguments.
   This doesn't check for constants in the argument name itself,
   because that will already have been checked when this function
   is called!

   Assumes the argument has been formed validly. */
enum formula_kind_type find_formula_kind_of_argument_arguments(struct mad_constant_argument * argument) {
  
  enum formula_kind_type kind;
  
  struct constant_list_node * given_argument_constant;
  struct object_list_node * given_argument_object;
  struct constant_list_node * argument_of_argument_constant;

  struct variable_list_node * argument_of_argument_variable;
  
  /* If the argument is a constant/object, we'll check it to see if it
     contains constants, even though the declared argument isn't an action.
     (This is shorthand for a formula where that constant appears
     as a conjunct, so it affects the kind of the formula). */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;
  
  /* To hold the formula kind coming from an argument if the term
     is a constant. */
  enum formula_kind_type argument_formula_kind;


  /* By default, there are no constants. */
  kind = CONSTANTLESS_FORMULA;
  
  /* We only need to check things if the given argument is a constant or object,
     or if it is an arithmetic operator. */
  given_argument_constant = lookup_constant(argument->name);
  given_argument_object = lookup_object(argument->name);
  if (given_argument_constant != NULL) {
    /* We check the arguments, since there may be a constant
       appearing in an argument. */
    current_declaration_argument = given_argument_constant->node_data->arguments;
    current_argument = argument->arguments;
    /* Run through as many arguments as were declared. */
    while (current_declaration_argument != NULL && current_argument != NULL) {
      /* We only need to examine cases where the declaration doesn't require
	 an action.  (Because if an action is required, the action name 
	 appearing there will not be a constant but an "object"). */
      if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	argument_of_argument_constant = lookup_constant(current_argument->name);
	if ( argument_of_argument_constant != NULL ) {
	  /* This constant "c" which appears as an argument is actually
	     used as shorthand for a formula in which the "c"
	     is replaced with a variable "v" and then the atomic formula
	     "c=v" is added as a conjunct to the end of the whole formula. */
	  /* Find out what kind of constant it is */
	  switch (argument_of_argument_constant->node_data->kind) {
	  case SIMPLE_FL_CONST: 
	    argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	    break;
	  case SD_FL_CONST:
	    argument_formula_kind = FLUENT_FORMULA;
	    break;
	  case ACTION_CONST:
	    argument_formula_kind = ACTION_FORMULA;
	    break;
	  case RIGID_CONST:
	    argument_formula_kind = RIGID_FORMULA;
	    break;
	  }
	  kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	}
	else {  // the argument of the given argument is not a constant
	  /* If the argument of the given argument is a variable, 
	     it may be a variable of sort "action" or "explicitAction",
	     and then it will be an action formula. */
	  argument_of_argument_variable = lookup_variable(current_argument->name);
	  if (argument_of_argument_variable != NULL
	      && is_a_known_variable_sort(argument_of_argument_variable->node_data->sort)
	      && ( (strcmp(argument_of_argument_variable->node_data->sort, "action") == 0)
		   || (strcmp(argument_of_argument_variable->node_data->sort, "explicitAction") == 0)
		   || is_a_subsort_of("action", argument_of_argument_variable->node_data->sort)
		   || is_a_subsort_of("explicitAction", argument_of_argument_variable->node_data->sort) ) ) {
	    kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	  }
	}
	
	/* If we already determined that the formula kind is FORMULA,
	   there is no need to look further. */
	if ( kind == FORMULA ) {
	  return kind;
	}
      }

      /* Besides checking if this argument is a constant,
	 if it's a constant (which may just be an action-name "object"),
	 or if it's an object,
	 we need to check if it contains a constant in its arguments. */
      
      argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
      kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
      
      /* If we already determined that the formula kind is FORMULA,
	 there is no need to look further. */
      if ( kind == FORMULA ) {
	return kind;
      }
      
      /* Move on, to check the next arguments */
      current_declaration_argument = current_declaration_argument->next;
      current_argument = current_argument->next_argument;
    }
  }
  else if (given_argument_object != NULL) {
    /* We check the arguments, since there may be a constant
       appearing in an argument. */
    current_declaration_argument = given_argument_object->node_data->arguments;
    current_argument = argument->arguments;
    /* Run through as many arguments as were declared. */
    while (current_declaration_argument != NULL && current_argument != NULL) {
      argument_of_argument_constant = lookup_constant(current_argument->name);
      if ( argument_of_argument_constant != NULL ) {
	/* This constant "c" which appears as an argument is actually
	   used as shorthand for a formula in which the "c"
	   is replaced with a variable "v" and then the atomic formula
	   "c=v" is added as a conjunct to the end of the whole formula. */
	/* Find out what kind of constant it is */
	switch (argument_of_argument_constant->node_data->kind) {
	case SIMPLE_FL_CONST: 
	  argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	  break;
	case SD_FL_CONST:
	  argument_formula_kind = FLUENT_FORMULA;
	  break;
	case ACTION_CONST:
	  argument_formula_kind = ACTION_FORMULA;
	  break;
	case RIGID_CONST:
	  argument_formula_kind = RIGID_FORMULA;
	  break;
	}
	kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
      }
      else {  // the argument of the given argument is not a constant
	/* If the argument of the given argument is a variable, 
	   it may be a variable of sort "action" or "explicitAction",
	   and then it will be an action formula. */
	argument_of_argument_variable = lookup_variable(current_argument->name);
	if (argument_of_argument_variable != NULL
	    && is_a_known_variable_sort(argument_of_argument_variable->node_data->sort)
	    && ( (strcmp(argument_of_argument_variable->node_data->sort, "action") == 0)
		 || (strcmp(argument_of_argument_variable->node_data->sort, "explicitAction") == 0)
		 || is_a_subsort_of("action", argument_of_argument_variable->node_data->sort)
		 || is_a_subsort_of("explicitAction", argument_of_argument_variable->node_data->sort) ) ) {
	  kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	}
      }
      
      /* If we already determined that the formula kind is FORMULA,
	 there is no need to look further. */
      if ( kind == FORMULA ) {
	return kind;
      }
      
      /* Besides checking if this argument is a constant,
	 if it's a constant (which may just be an action-name "object"),
	 or if it's an object,
	 we need to check if it contains a constant in its arguments. */
      
      argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
      kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
      
      /* If we already determined that the formula kind is FORMULA,
	 there is no need to look further. */
      if ( kind == FORMULA ) {
	return kind;
      }
      
      /* Move on, to check the next arguments */
      current_declaration_argument = current_declaration_argument->next;
      current_argument = current_argument->next_argument;
    }
  }
  else if ( is_an_arithmetic_operator(argument->name) ) {
    /* We check the arguments, since there may be a constant
       appearing in an argument. */
    current_argument = argument->arguments;
    while ( current_argument != NULL ) {
      argument_of_argument_constant = lookup_constant(current_argument->name);
      if ( argument_of_argument_constant != NULL ) {
	/* This constant "c" which appears as an argument is actually
	   used as shorthand for a formula in which the "c"
	   is replaced with a variable "v" and then the atomic formula
	   "c=v" is added as a conjunct to the end of the whole formula. */
	/* Find out what kind of constant it is */
	switch (argument_of_argument_constant->node_data->kind) {
	case SIMPLE_FL_CONST: 
	  argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	  break;
	case SD_FL_CONST:
	  argument_formula_kind = FLUENT_FORMULA;
	  break;
	case ACTION_CONST:
	  argument_formula_kind = ACTION_FORMULA;
	  break;
	case RIGID_CONST:
	  argument_formula_kind = RIGID_FORMULA;
	  break;
	}
	kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
      }
      else {  // the argument of the given argument is not a constant
	/* If the argument of the given argument is a variable, 
	   it may be a variable of sort "action" or "explicitAction",
	   and then it will be an action formula. */
	argument_of_argument_variable = lookup_variable(current_argument->name);
	if (argument_of_argument_variable != NULL
	    && is_a_known_variable_sort(argument_of_argument_variable->node_data->sort)
	    && ( (strcmp(argument_of_argument_variable->node_data->sort, "action") == 0)
		 || (strcmp(argument_of_argument_variable->node_data->sort, "explicitAction") == 0)
		 || is_a_subsort_of("action", argument_of_argument_variable->node_data->sort)
		 || is_a_subsort_of("explicitAction", argument_of_argument_variable->node_data->sort) ) ) {
	  kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	}
      }
      
      /* If we already determined that the formula kind is FORMULA,
	 there is no need to look further. */
      if ( kind == FORMULA ) {
	return kind;
      }
      
      /* Besides checking if this argument is a constant,
	 if it's a constant (which may just be an action-name "object"),
	 or if it's an object,
	 we need to check if it contains a constant in its arguments. */
      
      argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
      kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
      
      /* If we already determined that the formula kind is FORMULA,
	 there is no need to look further. */
      if ( kind == FORMULA ) {
	return kind;
      }
      
      /* Move on, to check the next arguments */
      current_argument = current_argument->next_argument;
    }
  }

  return kind;
}

/* Takes a term and an argument structure and returns a formula kind 
   (i.e., action formula, fluent formula, etc.) for it,
   by treating it as if it were a formula composed of a single term.
   (Technically for a term to be a formula there should be something
   it is equal to, but if we just want to determine what how a formula
   kind is affected based on the constants in the term, it doesn't matter
   what the term is equal to.
   
   Assumes the term has been formed validly. */
enum formula_kind_type find_formula_kind_of_term(char * term,
						 struct mad_constant_argument * term_arguments) {
  
  enum formula_kind_type kind;
  
  struct constant_list_node * term_constant;
  struct constant_list_node * argument_constant;

  struct variable_list_node * term_variable;
  struct variable_list_node * argument_variable;
  
  struct object_list_node * term_object;

  /* If the term has any arguments, we'll check them to see if they
     contain constants, even when an argument isn't an action.
     (This is shorthand for a formula where that constant appears
     as a conjunct, so it affects the kind of the formula). */
  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  /* To hold the formula kind coming from an argument if the term
     is a constant/object. */
  enum formula_kind_type argument_formula_kind;

  
  /* By default, a term is assumed to have no constants. */
  kind = CONSTANTLESS_FORMULA;
  
  if (term != NULL) {
    
    /* Check if the term is a constant */
    term_constant = lookup_constant(term);
    if (term_constant != NULL) {
      /* Find out what kind of constant it is */
      switch (term_constant->node_data->kind) {
      case SIMPLE_FL_CONST: 
	kind = SIMPLE_FLUENT_FORMULA;
	break;
      case SD_FL_CONST:
	kind = FLUENT_FORMULA;
	break;
      case ACTION_CONST:
	kind = ACTION_FORMULA;
	break;
      case RIGID_CONST:
	kind = RIGID_FORMULA;
	break;
      }
      
      /* We also need to check the arguments, since there may be a constant
	 appearing in an argument. */
      current_declaration_argument = term_constant->node_data->arguments;
      current_argument = term_arguments;
      /* Run through as many arguments as were declared. */
      while (current_declaration_argument != NULL && current_argument != NULL) {
	/* We only need to examine cases where the declaration doesn't require
	   an action.  (Because if an action is required, the action name 
	   appearing there will not be a constant but an "object"). */
	if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	  argument_constant = lookup_constant(current_argument->name);
	  if ( argument_constant != NULL ) {
	    /* This constant "c" which appears as an argument is actually
	       used as shorthand for a formula in which the "c"
	       is replaced with a variable "v" and then the atomic formula
	       "c=v" is added as a conjunct to the end of the whole formula. */
	    /* Find out what kind of constant it is */
	    switch (argument_constant->node_data->kind) {
	    case SIMPLE_FL_CONST: 
	      argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	      break;
	    case SD_FL_CONST:
	      argument_formula_kind = FLUENT_FORMULA;
	      break;
	    case ACTION_CONST:
	      argument_formula_kind = ACTION_FORMULA;
	      break;
	    case RIGID_CONST:
	      argument_formula_kind = RIGID_FORMULA;
	      break;
	    }
	    kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	  }
	  else {  // the argument is not a constant
	    /* If the argument is a variable, 
	       it may be a variable of sort "action" or "explicitAction",
	       and then it will be an action formula. */
	    argument_variable = lookup_variable(current_argument->name);
	    if (argument_variable != NULL
		&& is_a_known_variable_sort(argument_variable->node_data->sort)
		&& ( (strcmp(argument_variable->node_data->sort, "action") == 0)
		     || (strcmp(argument_variable->node_data->sort, "explicitAction") == 0)
		     || is_a_subsort_of("action", argument_variable->node_data->sort)
		     || is_a_subsort_of("explicitAction", argument_variable->node_data->sort) ) ) {
	      kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	    }
	  }
	}
	
	/* If we already determined that the formula kind is FORMULA,
	   there is no need to look further. */
	if ( kind == FORMULA ) {
	  return kind;
	}
	
	/* Besides checking if this argument is a constant,
	   if it's a constant (which may just be an action-name "object", 
           or if it's an object,
	   we need to check if it contains a constant in its arguments. */
	
	argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
	kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	
	/* If we already determined that the formula kind is FORMULA,
	   there is no need to look further. */
	if ( kind == FORMULA ) {
	  return kind;
	}
	
	/* Move on, to check the next arguments */
	current_declaration_argument = current_declaration_argument->next;
	current_argument = current_argument->next_argument;
      }
    }
    else {  // the term is not a constant
      /* If the term is a variable, 
	 it may be a variable of sort "action" or "explicitAction",
	 and then it will be an action formula. */
      term_variable = lookup_variable(term);
      if (term_variable != NULL
	  && is_a_known_variable_sort(term_variable->node_data->sort)
	  && ( (strcmp(term_variable->node_data->sort, "action") == 0)
	       || (strcmp(term_variable->node_data->sort, "explicitAction") == 0)
	       || is_a_subsort_of("action", term_variable->node_data->sort)
	       || is_a_subsort_of("explicitAction", term_variable->node_data->sort) ) ) {
	kind = ACTION_FORMULA;
      }
      // The term may be an object, which may contain constants 
      // appearing in it as arguments
      term_object = lookup_object(term);
      if (term_object != NULL) {
	current_declaration_argument = term_object->node_data->arguments;
	current_argument = term_arguments;
	/* Run through as many arguments as were declared. */
	while (current_declaration_argument != NULL && current_argument != NULL) {
	  argument_constant = lookup_constant(current_argument->name);
	  if ( argument_constant != NULL ) {
	    /* This constant "c" which appears as an argument is actually
	       used as shorthand for a formula in which the "c"
	       is replaced with a variable "v" and then the atomic formula
	       "c=v" is added as a conjunct to the end of the whole formula. */
	    /* Find out what kind of constant it is */
	    switch (argument_constant->node_data->kind) {
	    case SIMPLE_FL_CONST: 
	      argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	      break;
	    case SD_FL_CONST:
	      argument_formula_kind = FLUENT_FORMULA;
	      break;
	    case ACTION_CONST:
	      argument_formula_kind = ACTION_FORMULA;
	      break;
	    case RIGID_CONST:
	      argument_formula_kind = RIGID_FORMULA;
	      break;
	    }
	    kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	  }
	  else {  // the argument is not a constant
	    /* If the argument is a variable, 
	       it may be a variable of sort "action" or "explicitAction",
	       and then it will be an action formula. */
	    argument_variable = lookup_variable(current_argument->name);
	    if (argument_variable != NULL
		&& is_a_known_variable_sort(argument_variable->node_data->sort)
		&& ( (strcmp(argument_variable->node_data->sort, "action") == 0)
		     || (strcmp(argument_variable->node_data->sort, "explicitAction") == 0)
		     || is_a_subsort_of("action", argument_variable->node_data->sort)
		     || is_a_subsort_of("explicitAction", argument_variable->node_data->sort) ) ) {
	      kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	    }
	  }
	  
	  /* If we already determined that the formula kind is FORMULA,
	     there is no need to look further. */
	  if ( kind == FORMULA ) {
	    return kind;
	  }
	  
	  /* Besides checking if this argument is a constant,
	     if it's a constant (which may just be an action-name "object", 
	     or if it's an object,
	     we need to check if it contains a constant in its arguments. */
	  
	  argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
	  kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	  
	  /* If we already determined that the formula kind is FORMULA,
	     there is no need to look further. */
	  if ( kind == FORMULA ) {
	    return kind;
	  }
	  
	  /* Move on, to check the next arguments */
	  current_declaration_argument = current_declaration_argument->next;
	  current_argument = current_argument->next_argument;
	}
      }
      
      // The term may also be an arithmetic operator, which may contain constants 
      // appearing in it as arguments
      if ( is_an_arithmetic_operator(term) ) {
	current_argument = term_arguments;
	while ( current_argument != NULL) {
	  argument_constant = lookup_constant(current_argument->name);
	  if ( argument_constant != NULL ) {
	    /* This constant "c" which appears as an argument is actually
	       used as shorthand for a formula in which the "c"
	       is replaced with a variable "v" and then the atomic formula
	       "c=v" is added as a conjunct to the end of the whole formula. */
	    /* Find out what kind of constant it is */
	    switch (argument_constant->node_data->kind) {
	    case SIMPLE_FL_CONST: 
	      argument_formula_kind = SIMPLE_FLUENT_FORMULA;
	      break;
	    case SD_FL_CONST:
	      argument_formula_kind = FLUENT_FORMULA;
	      break;
	    case ACTION_CONST:
	      argument_formula_kind = ACTION_FORMULA;
	      break;
	    case RIGID_CONST:
	      argument_formula_kind = RIGID_FORMULA;
	      break;
	    }
	    kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	  }
	  else {  // the argument is not a constant
	    /* If the argument is a variable, 
	       it may be a variable of sort "action" or "explicitAction",
	       and then it will be an action formula. */
	    argument_variable = lookup_variable(current_argument->name);
	    if (argument_variable != NULL
		&& is_a_known_variable_sort(argument_variable->node_data->sort)
		&& ( (strcmp(argument_variable->node_data->sort, "action") == 0)
		     || (strcmp(argument_variable->node_data->sort, "explicitAction") == 0)
		     || is_a_subsort_of("action", argument_variable->node_data->sort)
		     || is_a_subsort_of("explicitAction", argument_variable->node_data->sort) ) ) {
	      kind = combination_of_two_formula_kinds(kind, ACTION_FORMULA);
	    }
	  }
	  
	  /* If we already determined that the formula kind is FORMULA,
	     there is no need to look further. */
	  if ( kind == FORMULA ) {
	    return kind;
	  }
	  
	  /* Besides checking if this argument is a constant,
	     if it's a constant (which may just be an action-name "object", 
	     or if it's an object,
	     we need to check if it contains a constant in its arguments. */
	  
	  argument_formula_kind = find_formula_kind_of_argument_arguments(current_argument);
	  kind = combination_of_two_formula_kinds(kind, argument_formula_kind);
	  
	  /* If we already determined that the formula kind is FORMULA,
	     there is no need to look further. */
	  if ( kind == FORMULA ) {
	    return kind;
	  }
	  
	  /* Move on, to check the next arguments */
	  current_argument = current_argument->next_argument;
	}
      }
    }
  }
  
  return kind;
}


/* Takes a formula and returns its kind (i.e., action formula, fluent formula, etc.) 
   Assumes formula has been formed validly. */
enum formula_kind_type find_formula_kind(struct mad_formula * formula) {
  
  enum formula_kind_type kind;
  enum formula_kind_type left_formula_kind;
  enum formula_kind_type right_formula_kind;
  enum formula_kind_type LHS_term_formula_kind;
  enum formula_kind_type RHS_term_formula_kind;
  
  /* By default, a formula is assumed to have no constants. */
  kind = CONSTANTLESS_FORMULA;
 
  if (formula != NULL) {
    
    if ( formula->formula_kind == QUANTIFIER ) {
      /* The kind is the same as that of the formula in the scope of the quantifier */
      right_formula_kind = find_formula_kind(formula->right_formula);
      kind = right_formula_kind;
    }
    else if (formula->formula_kind == BINARY_CONNECTIVE) {
      left_formula_kind = find_formula_kind(formula->left_formula);
      /* If the kind of the left formula has already been determined to be FORMULA,
	 there is no need to look at the right formula. */
      if (left_formula_kind == FORMULA) {
	return FORMULA;
      }
      right_formula_kind = find_formula_kind(formula->right_formula);

      return combination_of_two_formula_kinds(left_formula_kind,
					      right_formula_kind);
    }
    else if (formula->formula_kind == UNARY_CONNECTIVE) {
      /* The kind is the same as that of the formula in the scope of the quantifier */
      right_formula_kind = find_formula_kind(formula->right_formula);
      kind = right_formula_kind;
    }
    else if (formula->formula_kind == ZERO_PLACE_CONNECTIVE) {
      /* No constants here. */
      kind = CONSTANTLESS_FORMULA;
    }
    else if ( formula->formula_kind == ATOMIC_FORMULA 
	      || formula->formula_kind == INEQUALITY ) {
      /* This is the main place where we check for constant occurrence. */
      
      LHS_term_formula_kind = find_formula_kind_of_term(formula->LHS_term,
							formula->LHS_arguments);
      
      /* If the kind of the LHS_term has already been determined to be FORMULA,
	 there is no need to look at the RHS_term. */
      if (LHS_term_formula_kind == FORMULA) {
	return FORMULA;
      }
      /* The RHS term may be NULL if the LHS term was a Boolean constant. */
      if (formula->RHS_term != NULL ) {
	RHS_term_formula_kind = find_formula_kind_of_term(formula->RHS_term,
							  formula->RHS_arguments);
	return combination_of_two_formula_kinds(LHS_term_formula_kind,
						RHS_term_formula_kind);
      }
      return LHS_term_formula_kind;
    }
  }
  return kind;
}

/* Takes an axiom (with three parts F, G, H) 
   and checks if this would constitute a valid axiom.
   These conditions are checked:
   1. F must be definite
      (In general, F cannot have both fluents and actions, 
      but this condition is satisfied if it's definite.)
   2. if H is NULL, 
        if F is a fluent formula, G must be a fluent formula
        if F is an action formula, G can be anything
   3. if H is not NULL,
        F and G must be fluent formulas
        F must not contain any sdFluents or rigid constants.
   4. if F is has a rigid constant,
        the axiom must not contain any non-rigid constants.

   Assumes the formulas in the axiom have been formed validly.
*/
int is_a_valid_axiom(struct mad_axiom * axiom) {
  
  enum formula_kind_type F_formula_kind;
  enum formula_kind_type G_formula_kind;
  enum formula_kind_type H_formula_kind;

  /* Start by assuming the axiom is valid. */
  int valid = 1;

  F_formula_kind = find_formula_kind(temp_axiom->F);
  G_formula_kind = find_formula_kind(temp_axiom->G);
  H_formula_kind = find_formula_kind(temp_axiom->H);

  if ( temp_axiom->H == NULL ) {
    if ( F_formula_kind != ACTION_FORMULA
	 && F_formula_kind != FORMULA ) {
      /* F is a fluent formula, so this must be a static law. */
      if ( G_formula_kind == ACTION_FORMULA
	   || G_formula_kind == FORMULA ) {
	/* G is not a fluent formula */
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "In a static law, the body must be a fluent formula.\n");
	valid = 0;
      }
    }
  }
  else { /* H is not NULL, so this must be a fluent dynamic law. */
    if ( F_formula_kind != CONSTANTLESS_FORMULA
	 && F_formula_kind != SIMPLE_FLUENT_FORMULA ) {
      /* F contains a rigid, sdFLuent, or action constant. */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "In a fluent dynamic law, the head must be a fluent formula without any rigid or sdFluent constants.\n");
      valid = 0;
    }
    if ( G_formula_kind == ACTION_FORMULA
	 || G_formula_kind == FORMULA ) {
      /* G is not a fluent formula */
      print_parser_error_message(stderr,
			  input_file_name, 
			  error_lineno,
			  "In a fluent dynamic law, the formula after \"if\" must be a fluent formula.\n");
      valid = 0;
    }
  }

  /* Check that the head is definite. */
  if ( !is_a_definite_head_formula(temp_axiom->F) ) {
    valid = 0;
    print_parser_error_message(stderr,
			input_file_name, 
			error_lineno,
			"Since the head is not definite, we won't check if it contains rigid constants.\n");
  }
  else {
    /* Since the head is known to be definite, it contains at most one constant.
       Therefore, if the head contains a rigid constant, it must be a RIGID_FORMULA. */
    if ( F_formula_kind == RIGID_FORMULA ) {
      /* Check if all the other constants are rigid */
      if ( (G_formula_kind != CONSTANTLESS_FORMULA
	    && G_formula_kind != RIGID_FORMULA )
	   || (H_formula_kind != CONSTANTLESS_FORMULA
	       && H_formula_kind != RIGID_FORMULA ) ) {
	print_parser_error_message(stderr,
			    input_file_name, 
			    error_lineno,
			    "If the head of an axiom contains a rigid constant, all constants in the axiom must be rigid.\n");
	valid = 0;
      }
    }
  }

  return valid;
}


/* Called by expand_formula, to expand arguments. */
struct mad_constant_argument * expand_argument(struct mad_constant_argument * argument,
					       struct formula_list_node ** formula_list_first_ptr,
					       struct formula_list_node ** formula_list_last_ptr) {

  struct mad_formula * new_formula; 
  
  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct constant_list_node * argument_constant;

  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  struct string_list_node * string_list_first;
  struct string_list_node * string_list_last;

  char * new_variable;

  if (argument != NULL ) {
    term_constant = lookup_constant(argument->name);
    term_object = lookup_object(argument->name);
    // if this argument isn't a constant, an object, or an arithmetic operator,
    // there is nothing to expand
    if ( term_constant != NULL ) {
      current_argument = argument->arguments;
      current_declaration_argument = term_constant->node_data->arguments;
      while ( current_argument != NULL ) {
	/* Check if it's a constant where a non-constant is expected */
	if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	  argument_constant = lookup_constant(current_argument->name);
	  if ( argument_constant != NULL ) {
	    /* Make a new variable to replace it */
	    /* We make a new one that is different from the other identifiers. */
	    new_variable = make_new_variable_name(argument_constant->node_data->domain);
	    /* Add this new variable to the list of identifiers and to the list of variables */
	    insert_identifier(new_variable);
	    insert_variable(new_variable);
	    init_string_list(&string_list_first,
			     &string_list_last);
	    insert_string_into_list(&string_list_first,
				    &string_list_last,
				    new_variable);
	    update_variables(string_list_first, argument_constant->node_data->domain);
	    
	    /* Prepare a new atomic formula to be conjoined with the current atomic formula
	       when we replace this constant with a variable */
	    new_formula = make_new_formula(ATOMIC_FORMULA);
	    new_formula->LHS_term = current_argument->name;
	    new_formula->LHS_arguments = current_argument->arguments;
	    new_formula->RHS_term = new_variable;
	    
	    /* Insert this new formula into the list of new formulas to be conjoined */
	    insert_formula_into_list(formula_list_first_ptr,
				     formula_list_last_ptr,
				     new_formula);
	      
	    /* Now replace this argument with the new variable */
	    current_argument->name = new_variable;
	    current_argument->argument_kind = PLAIN_ARGUMENT;
	    current_argument->arguments = NULL;
	  }
	  else {
	    // The argument declaration requires a non-action,
	    // and the argument is not a constant.
	    // In this case, we may still have an object argument
	    // which contains a constant, so we need to expand this
	    // argument
	    current_argument = expand_argument(current_argument, formula_list_first_ptr, formula_list_last_ptr);
	  }
	}
	else { /* If the current_argument spot requires an action constant, this argument won't 
		  be "shorthand" but the arguments to it may be shorthand. */
	  current_argument = expand_argument(current_argument, formula_list_first_ptr, formula_list_last_ptr);
	}
	
	/* Move on to the next argument */
	current_argument = current_argument->next_argument;
	current_declaration_argument = current_declaration_argument->next;
      }
    }
    else if (term_object != NULL) {
      current_argument = argument->arguments;
      current_declaration_argument = term_object->node_data->arguments;
      while ( current_argument != NULL ) {
	/* Normally we don't expect any constants as arguments to objects,
	   so if the argument is a constant, it should be expanded. */
	argument_constant = lookup_constant(current_argument->name);
	if ( argument_constant != NULL ) {
	  /* Make a new variable to replace it */
	  /* We make a new one that is different from the other identifiers. */
	  new_variable = make_new_variable_name(argument_constant->node_data->domain);
	  /* Add this new variable to the list of identifiers and to the list of variables */
	  insert_identifier(new_variable);
	  insert_variable(new_variable);
	  init_string_list(&string_list_first,
			   &string_list_last);
	  insert_string_into_list(&string_list_first,
				  &string_list_last,
				  new_variable);
	  update_variables(string_list_first, argument_constant->node_data->domain);
	  
	  /* Prepare a new atomic formula to be conjoined with the current atomic formula
	     when we replace this constant with a variable */
	  new_formula = make_new_formula(ATOMIC_FORMULA);
	  new_formula->LHS_term = current_argument->name;
	  new_formula->LHS_arguments = current_argument->arguments;
	  new_formula->RHS_term = new_variable;
	  
	  /* Insert this new formula into the list of new formulas to be conjoined */
	  insert_formula_into_list(formula_list_first_ptr,
				   formula_list_last_ptr,
				   new_formula);
	  
	  /* Now replace this argument with the new variable */
	  current_argument->name = new_variable;
	  current_argument->argument_kind = PLAIN_ARGUMENT;
	  current_argument->arguments = NULL;
	}
	else { /* If the argument is not a constant, then it is not "shorthand".
		  But the arguments to it may be shorthand. */
	  current_argument = expand_argument(current_argument, formula_list_first_ptr, formula_list_last_ptr);
	}
	
	/* Move on to the next argument */
	current_argument = current_argument->next_argument;
	current_declaration_argument = current_declaration_argument->next;
      }
    }
    else if (is_an_arithmetic_operator(argument->name) ) {
      current_argument = argument->arguments;
      while ( current_argument != NULL ) {
	/* Normally we don't expect any constants as arguments to arithmetic operators,
	   so if the argument is a constant, it should be expanded. */
	argument_constant = lookup_constant(current_argument->name);
	if ( argument_constant != NULL ) {
	  /* Make a new variable to replace it */
	  /* We make a new one that is different from the other identifiers. */
	  new_variable = make_new_variable_name(argument_constant->node_data->domain);
	  /* Add this new variable to the list of identifiers and to the list of variables */
	  insert_identifier(new_variable);
	  insert_variable(new_variable);
	  init_string_list(&string_list_first,
			   &string_list_last);
	  insert_string_into_list(&string_list_first,
				  &string_list_last,
				  new_variable);
	  update_variables(string_list_first, argument_constant->node_data->domain);
	  
	  /* Prepare a new atomic formula to be conjoined with the current atomic formula
	     when we replace this constant with a variable */
	  new_formula = make_new_formula(ATOMIC_FORMULA);
	  new_formula->LHS_term = current_argument->name;
	  new_formula->LHS_arguments = current_argument->arguments;
	  new_formula->RHS_term = new_variable;
	  
	  /* Insert this new formula into the list of new formulas to be conjoined */
	  insert_formula_into_list(formula_list_first_ptr,
				   formula_list_last_ptr,
				   new_formula);
	  
	  /* Now replace this argument with the new variable */
	  current_argument->name = new_variable;
	  current_argument->argument_kind = PLAIN_ARGUMENT;
	  current_argument->arguments = NULL;
	}
	else { /* If the argument is not a constant, then it is not "shorthand".
		  But the arguments to it may be shorthand. */
	  current_argument = expand_argument(current_argument, formula_list_first_ptr, formula_list_last_ptr);
	}
	
	/* Move on to the next argument */
	current_argument = current_argument->next_argument;
      }
    }

  }
  return argument;
}

/* Given a formula, it checks if there are any constants which
   appear as arguments, even though no constant is required.
   (This is shorthand, to stand for "the argument has the value 
   of the constant").  Such arguments are replaced with a variable
   and conjuncts are added specifying that the constant has the
   same value as the variable. 

   Assumes the formula has been formed validly. 
*/
struct mad_formula * expand_formula(struct mad_formula * formula) {

  struct mad_formula * new_formula; 
  
  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct constant_list_node * argument_constant;

  struct mad_constant_argument * current_argument;
  struct string_list_node * current_declaration_argument;

  struct string_list_node * string_list_first;
  struct string_list_node * string_list_last;

  struct formula_list_node * formula_list_first;
  struct formula_list_node * formula_list_last;
  struct formula_list_node * current_atomic_formula_to_be_expanded; 
  struct formula_list_node * current_formula_to_be_conjoined;

  char * new_variable;

  if ( formula != NULL ) {
    
    /* If there is a left_formula, expand it */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = expand_formula(formula->left_formula);
    }
    
    /* If it's an atomic_formula, expand the LHS_term, with LHS_arguments and RHS_term with RHS_arguments. */
    if ( formula->formula_kind == ATOMIC_FORMULA ) {

      init_formula_list(&formula_list_first,
			&formula_list_last);

      /* First expand the LHS term */
      if (formula->LHS_arguments != NULL ) {
	term_constant = lookup_constant(formula->LHS_term);
	term_object = lookup_object(formula->LHS_term);
	if (term_constant != NULL) {
	  current_argument = formula->LHS_arguments;
	  current_declaration_argument = term_constant->node_data->arguments;
	  while ( current_argument != NULL ) {
	    /* Check if it's a constant where a non-constant is expected */
	    if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	      argument_constant = lookup_constant(current_argument->name);
	      if ( argument_constant != NULL ) {
		/* Make a new variable to replace it */
		/* We make a new one that is different from the other identifiers. */
		new_variable = make_new_variable_name(argument_constant->node_data->domain);
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(new_variable);
		insert_variable(new_variable);
		init_string_list(&string_list_first,
				 &string_list_last);
		insert_string_into_list(&string_list_first,
					&string_list_last,
					new_variable);
		update_variables(string_list_first, argument_constant->node_data->domain);
		
		/* Prepare a new atomic formula to be conjoined with the current atomic formula
		   when we replace this constant with a variable */
		new_formula = make_new_formula(ATOMIC_FORMULA);
		new_formula->LHS_term = current_argument->name;
		new_formula->LHS_arguments = current_argument->arguments;
		new_formula->RHS_term = new_variable;
		
		/* Insert this new formula into the list of new formulas to be conjoined */
		insert_formula_into_list(&formula_list_first,
					 &formula_list_last,
					 new_formula);
		
		/* Now replace this argument with the new variable */
	      current_argument->name = new_variable;
	      current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	      }
	      else {
		// The argument declaration requires a non-action,
		// and the argument is not a constant.
		// In this case, we may still have an object argument
		// which contains a constant, so we need to expand this
		// argument
		current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	      }
	    }
	    else { /* If the term requires an action constant, this argument won't 
		      be "shorthand" but the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	    current_declaration_argument = current_declaration_argument->next;
	  }
	}
	else if (term_object != NULL) {
	  current_argument = formula->LHS_arguments;
	  current_declaration_argument = term_object->node_data->arguments;
	  while ( current_argument != NULL ) {
	    /* Normally we don't expect any constants as arguments to objects,
	       so if the argument is a constant, it should be expanded. */
	    argument_constant = lookup_constant(current_argument->name);
	    if ( argument_constant != NULL ) {
	      /* Make a new variable to replace it */
	      /* We make a new one that is different from the other identifiers. */
	      new_variable = make_new_variable_name(argument_constant->node_data->domain);
	      /* Add this new variable to the list of identifiers and to the list of variables */
	      insert_identifier(new_variable);
	      insert_variable(new_variable);
	      init_string_list(&string_list_first,
			       &string_list_last);
	      insert_string_into_list(&string_list_first,
				      &string_list_last,
				      new_variable);
	      update_variables(string_list_first, argument_constant->node_data->domain);
	      
	      /* Prepare a new atomic formula to be conjoined with the current atomic formula
		 when we replace this constant with a variable */
	      new_formula = make_new_formula(ATOMIC_FORMULA);
	      new_formula->LHS_term = current_argument->name;
	      new_formula->LHS_arguments = current_argument->arguments;
	      new_formula->RHS_term = new_variable;
	      
	      /* Insert this new formula into the list of new formulas to be conjoined */
	      insert_formula_into_list(&formula_list_first,
				       &formula_list_last,
				       new_formula);
	      
	      /* Now replace this argument with the new variable */
	      current_argument->name = new_variable;
	      current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	    }
	    else { /* If the argument is not a constant, then it is not "shorthand".
		      But the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	    current_declaration_argument = current_declaration_argument->next;
	  }
	}
	else if ( is_an_arithmetic_operator( formula->LHS_term ) ) {
	  current_argument = formula->LHS_arguments;
	  while ( current_argument != NULL ) {
	    /* Normally we don't expect any constants as arguments to arithmetic operators,
	       so if the argument is a constant, it should be expanded. */
	    argument_constant = lookup_constant(current_argument->name);
	    if ( argument_constant != NULL ) {
	      /* Make a new variable to replace it */
	      /* We make a new one that is different from the other identifiers. */
	      new_variable = make_new_variable_name(argument_constant->node_data->domain);
	      /* Add this new variable to the list of identifiers and to the list of variables */
	      insert_identifier(new_variable);
	      insert_variable(new_variable);
	      init_string_list(&string_list_first,
			       &string_list_last);
	      insert_string_into_list(&string_list_first,
				      &string_list_last,
				      new_variable);
	      update_variables(string_list_first, argument_constant->node_data->domain);
	      
	      /* Prepare a new atomic formula to be conjoined with the current atomic formula
		 when we replace this constant with a variable */
	      new_formula = make_new_formula(ATOMIC_FORMULA);
	      new_formula->LHS_term = current_argument->name;
	      new_formula->LHS_arguments = current_argument->arguments;
	      new_formula->RHS_term = new_variable;
	      
	      /* Insert this new formula into the list of new formulas to be conjoined */
	      insert_formula_into_list(&formula_list_first,
				       &formula_list_last,
				       new_formula);
	      
	      /* Now replace this argument with the new variable */
	      current_argument->name = new_variable;
	      current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	    }
	    else { /* If the argument is not a constant, then it is not "shorthand".
		      But the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	  }
	}
      }
      

      /* Now expand the RHS term */

      if (formula->RHS_arguments != NULL ) {
	term_constant = lookup_constant(formula->RHS_term);
	term_object = lookup_object(formula->RHS_term);
	if ( term_constant != NULL ) {
	  current_argument = formula->RHS_arguments;
	  current_declaration_argument = term_constant->node_data->arguments;
	  while ( current_argument != NULL ) {
	    /* Check if it's a constant where a non-constant is expected */
	    if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	      argument_constant = lookup_constant(current_argument->name);
	      if ( argument_constant != NULL ) {
		/* Make a new variable to replace it */
		/* We make a new one that is different from the other identifiers. */
		new_variable = make_new_variable_name(argument_constant->node_data->domain);
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(new_variable);
		insert_variable(new_variable);
		init_string_list(&string_list_first,
				 &string_list_last);
		insert_string_into_list(&string_list_first,
					&string_list_last,
					new_variable);
		update_variables(string_list_first, argument_constant->node_data->domain);
		
		/* Prepare a new atomic formula to be conjoined with the current atomic formula
		   when we replace this constant with a variable */
		new_formula = make_new_formula(ATOMIC_FORMULA);
		new_formula->LHS_term = current_argument->name;
		new_formula->LHS_arguments = current_argument->arguments;
		new_formula->RHS_term = new_variable;
		
		/* Insert this new formula into the list of new formulas to be conjoined */
		insert_formula_into_list(&formula_list_first,
					 &formula_list_last,
					 new_formula);
		
		/* Now replace this argument with the new variable */
		current_argument->name = new_variable;
		current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	      }
	      else {
		// The argument declaration requires a non-action,
		// and the argument is not a constant.
		// In this case, we may still have an object argument
		// which contains a constant, so we need to expand this
		// argument
		current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	      }
	    }
	    else { /* If the current_argument spot requires an action constant, 
		      this argument won't be "shorthand" but the arguments
		      to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	    current_declaration_argument = current_declaration_argument->next;
	  }
	}
	else if (term_object != NULL) {
	  current_argument = formula->RHS_arguments;
	  current_declaration_argument = term_object->node_data->arguments;
	  while ( current_argument != NULL ) {
	    /* Normally we don't expect any constants as arguments to objects,
	       so if the argument is a constant, it should be expanded. */
	    argument_constant = lookup_constant(current_argument->name);
	    if ( argument_constant != NULL ) {
	      /* Make a new variable to replace it */
	      /* We make a new one that is different from the other identifiers. */
	      new_variable = make_new_variable_name(argument_constant->node_data->domain);
	      /* Add this new variable to the list of identifiers and to the list of variables */
	      insert_identifier(new_variable);
	      insert_variable(new_variable);
	      init_string_list(&string_list_first,
			       &string_list_last);
	      insert_string_into_list(&string_list_first,
				      &string_list_last,
				      new_variable);
	      update_variables(string_list_first, argument_constant->node_data->domain);
	      
	      /* Prepare a new atomic formula to be conjoined with the current atomic formula
		 when we replace this constant with a variable */
	      new_formula = make_new_formula(ATOMIC_FORMULA);
	      new_formula->LHS_term = current_argument->name;
	      new_formula->LHS_arguments = current_argument->arguments;
	      new_formula->RHS_term = new_variable;
	      
	      /* Insert this new formula into the list of new formulas to be conjoined */
	      insert_formula_into_list(&formula_list_first,
				       &formula_list_last,
				       new_formula);
	      
	      /* Now replace this argument with the new variable */
	      current_argument->name = new_variable;
	      current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	    }
	    else { /* If the argument is not a constant, then it is not "shorthand".
		      But the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	    current_declaration_argument = current_declaration_argument->next;
	  }
	}
	else if ( is_an_arithmetic_operator( formula->RHS_term ) ) {
	  current_argument = formula->RHS_arguments;
	  while ( current_argument != NULL ) {
	    /* Normally we don't expect any constants as arguments to arithmetic operators,
	       so if the argument is a constant, it should be expanded. */
	    argument_constant = lookup_constant(current_argument->name);
	    if ( argument_constant != NULL ) {
	      /* Make a new variable to replace it */
	      /* We make a new one that is different from the other identifiers. */
	      new_variable = make_new_variable_name(argument_constant->node_data->domain);
	      /* Add this new variable to the list of identifiers and to the list of variables */
	      insert_identifier(new_variable);
	      insert_variable(new_variable);
	      init_string_list(&string_list_first,
			       &string_list_last);
	      insert_string_into_list(&string_list_first,
				      &string_list_last,
				      new_variable);
	      update_variables(string_list_first, argument_constant->node_data->domain);
	      
	      /* Prepare a new atomic formula to be conjoined with the current atomic formula
		 when we replace this constant with a variable */
	      new_formula = make_new_formula(ATOMIC_FORMULA);
	      new_formula->LHS_term = current_argument->name;
	      new_formula->LHS_arguments = current_argument->arguments;
	      new_formula->RHS_term = new_variable;
	      
	      /* Insert this new formula into the list of new formulas to be conjoined */
	      insert_formula_into_list(&formula_list_first,
				       &formula_list_last,
				       new_formula);
	      
	      /* Now replace this argument with the new variable */
	      current_argument->name = new_variable;
	      current_argument->argument_kind = PLAIN_ARGUMENT;
	      current_argument->arguments = NULL;
	    }
	    else { /* If the argument is not a constant, then it is not "shorthand".
		      But the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	    
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	  }
	}
      }
      
      /* At this point, the current atomic formula has no more unexpanded
	 constants as arguments, and we have a list of new atomic formulas
	 to be conjoined with the atomic formula.
	 Before conjoining them, we need to expand them further too. */
      current_atomic_formula_to_be_expanded = formula_list_first;
      while ( current_atomic_formula_to_be_expanded != NULL ) {
	/* Expand the LHS term for this formula.  
	   (The RHS term is a variable so there is nothing to be expanded there.) */
	if ( current_atomic_formula_to_be_expanded->node_data->LHS_arguments != NULL ) {
	  term_constant = lookup_constant(current_atomic_formula_to_be_expanded->node_data->LHS_term);
	  current_argument = current_atomic_formula_to_be_expanded->node_data->LHS_arguments;
	  current_declaration_argument = term_constant->node_data->arguments;
	  while ( current_argument != NULL ) {
	    /* Check if it's a constant where a non-constant is expected */
	    if ( strcmp(current_declaration_argument->node_data, "action") != 0 ) {
	      argument_constant = lookup_constant(current_argument->name);
	      if ( argument_constant != NULL ) {
		/* Make a new variable to replace it */
		/* We make a new one that is different from the other identifiers. */
		new_variable = make_new_variable_name(argument_constant->node_data->domain);
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(new_variable);
		insert_variable(new_variable);
		init_string_list(&string_list_first,
				 &string_list_last);
		insert_string_into_list(&string_list_first,
					&string_list_last,
					new_variable);
		update_variables(string_list_first, argument_constant->node_data->domain);
		
		/* Prepare a new atomic formula to be conjoined with the current atomic formula
		   when we replace this constant with a variable */
		new_formula = make_new_formula(ATOMIC_FORMULA);
		new_formula->LHS_term = current_argument->name;
		new_formula->LHS_arguments = current_argument->arguments;
		new_formula->RHS_term = new_variable;
		
		/* Insert this new formula into the list of new formulas to be conjoined */
		insert_formula_into_list(&formula_list_first,
					 &formula_list_last,
					 new_formula);
		
		/* Now replace this argument with the new variable */
		current_argument->name = new_variable;
		current_argument->argument_kind = PLAIN_ARGUMENT;
		current_argument->arguments = NULL;
	      }
	      else { 
		// The argument declaration requires a non-action,
		// and the argument is not a constant.
		// In this case, we may still have an object argument
		// which contains a constant, so we need to expand this
		// argument
		current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	      }
	    }
	    else { /* If the current_argument spot requires an action constant, this argument won't 
		      be "shorthand" but the arguments to it may be shorthand. */
	      current_argument = expand_argument(current_argument, &formula_list_first, &formula_list_last);
	    }
	  
	    /* Move on to the next argument */
	    current_argument = current_argument->next_argument;
	    current_declaration_argument = current_declaration_argument->next;
	  }
	}
	
	/* Move on to the next formula in the list of formulas to be expanded */
	current_atomic_formula_to_be_expanded = current_atomic_formula_to_be_expanded->next;
      }

      /* Now we have expanded this atomic formula and its arguments, as much as possible.
	 We will now change it into a conjunction of all of the new atomic formulas 
	 created as a result of expanding. */
    
      /* Go down the list, conjoining new formulas and updating the given formula,
	 as you add conjuncts. */
      current_formula_to_be_conjoined = formula_list_first;
      
      while ( current_formula_to_be_conjoined != NULL ) {
	new_formula = make_new_formula(BINARY_CONNECTIVE);
	new_formula->connective = AND;
	new_formula->left_formula = formula;
	new_formula->right_formula = current_formula_to_be_conjoined->node_data;
	
	formula = new_formula;
	
	current_formula_to_be_conjoined = current_formula_to_be_conjoined->next;
      }
    }
    
    /* If there is a right_formula, expand it */
    if ( formula->right_formula != NULL ) {
      formula->right_formula = expand_formula(formula->right_formula);
    }
  }

  return formula;
}

struct mad_axiom * copy_axiom(struct mad_axiom * axiom) {
  if ( axiom != NULL ) {
    return make_new_axiom(copy_formula(axiom->F),
			  copy_formula(axiom->G),
			  copy_formula(axiom->H));
  }
  else {
    return NULL;
  }
    
}    

/* Makes a copy of a given axiom list */
struct axiom_list_node * copy_axiom_list(struct axiom_list_node * axiom_list) {
  struct axiom_list_node * copied_axiom_list;
  
  if ( axiom_list != NULL ) {
    /* First copy current node's formulas */
    copied_axiom_list = (struct axiom_list_node *) malloc( sizeof(struct axiom_list_node) );
    copied_axiom_list->node_data = copy_axiom(axiom_list->node_data);

    /* Then copy the rest */
    copied_axiom_list->next = copy_axiom_list(axiom_list->next);
  }
  else {
    copied_axiom_list = NULL;
  }
  
  return copied_axiom_list;
}

/* Called to merge axioms of an imported module with the current axioms */
void merge_axioms_from_module(struct mad_module * module) {

  struct axiom_list_node * current;

  struct mad_formula * F;
  struct mad_formula * G;
  struct mad_formula * H;

  current = module->module_axioms;

  /* We need to go down the list, copying each axiom. */
  
  while ( current != NULL ) {

    F = copy_formula(current->node_data->F);
    G = copy_formula(current->node_data->G);
    /* Not all axioms have an H formula so we need to check if it exists before
       trying to copy it. */
    if ( current->node_data->H != NULL) { 
      H = copy_formula(current->node_data->H);
    }
    else {
      H = NULL;
    }

    insert_axiom( make_new_axiom(F,G,H) );

    current = current->next;
  }
}

/* for debugging */
void print_arithmetic_operator_arguments(FILE * stream, struct mad_constant_argument * arguments) {
  if (arguments != NULL ) {
    if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
      fprintf(stream, "%s", arguments->name);
    }
    else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      if ( is_an_arithmetic_operator(arguments->name) ) {
	if ( strcmp(arguments->name, "+") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments(stream, arguments->arguments);
	  fprintf(stream, "+");
	  print_arithmetic_operator_arguments(stream, arguments->arguments->next_argument);
	  fprintf(stream, ")");
	}
	else if ( strcmp(arguments->name, "*") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments(stream, arguments->arguments);
	  fprintf(stream, "*");
	  print_arithmetic_operator_arguments(stream, arguments->arguments->next_argument);
	  fprintf(stream, ")");
	}
      }
      else {
	/* Only print one argument, not the whole chain */
	fprintf(stream, "%s", arguments->name);
	fprintf(stream, "(");
	print_constant_arguments(stream, arguments->arguments);
	fprintf(stream, ")");
      }
    }
    else {
      print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      fprintf(stream, "=%s", arguments->value);
    }
  }
}

/* for printing out arithmetic operator arguments in an axiom to be used as input for ccalc. */
void print_arithmetic_operator_arguments_for_ccalc(FILE * stream, 
						   struct mad_constant_argument * arguments) {
  if (arguments != NULL ) {
    if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
      fprintf(stream, "%s",  prepare_string_for_ccalc(arguments->name));
    }
    else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      if ( is_an_arithmetic_operator(arguments->name) ) {
	if ( strcmp(arguments->name, "+") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->arguments);
	  fprintf(stream, "+");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->arguments->next_argument);
	  fprintf(stream, ")");
	}
	else if ( strcmp(arguments->name, "*") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->arguments);
	  fprintf(stream, "*");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->arguments->next_argument);
	  fprintf(stream, ")");
	}
      }
      else {
	/* Only print one argument, not the whole chain */
	fprintf(stream, "%s", prepare_string_for_ccalc(arguments->name));
	fprintf(stream, "(");
	print_constant_arguments_for_ccalc(stream, arguments->arguments);
	fprintf(stream, ")");
      }
    }
    else {
      print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      fprintf(stream, "=%s",  prepare_string_for_ccalc(arguments->value));
    }
  }
}


/* for debugging */
void print_constant_arguments(FILE * stream, struct mad_constant_argument * arguments) {

  if (arguments != NULL ) {
    if ( is_an_arithmetic_operator(arguments->name) ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_arguments(stream, arguments);
    }
    else {
      if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
	fprintf(stream, "%s", arguments->name);
      }
      else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
	/* We need to print the name first, followed by the arguments of this argument! */
	fprintf(stream, "%s", arguments->name);
	fprintf(stream, "(");
	print_constant_arguments(stream, arguments->arguments);
	fprintf(stream, ")");
      }
      else {
	print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
      }
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      fprintf(stream, "=%s", arguments->value);
    }

    /* Move on to the next argument after printing this one */
    arguments = arguments->next_argument;

    while (arguments != NULL ) {
      if ( is_an_arithmetic_operator(arguments->name) ) {
	// Arithmetic operators may have their own syntax.
	// E.g., we print "a + b", not "+(a,b)"
	print_arithmetic_operator_arguments(stream, arguments);
      }
      else {
	if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
	  fprintf(stream, ",%s", arguments->name);
	}
	else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
	  /* We need to print the name first, followed by the arguments of this argument! */
	  fprintf(stream,",%s", arguments->name);
	  fprintf(stream, "(");
	  print_constant_arguments(stream, arguments->arguments);
	  fprintf(stream, ")");
	}
	else {
	  print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
	}
      }
      /* print the value if this argument is a constant with a value */
      if ( arguments->has_a_value == 1 ) {
	fprintf(stream, "=%s", arguments->value);
      }

      /* Move on to the next argument after printing this one */
      arguments = arguments->next_argument;
    }
  }
}

/* for printing out constant arguments in an axiom to be used as input for ccalc. */
void print_constant_arguments_for_ccalc(FILE * stream, 
					struct mad_constant_argument * arguments) {

  if (arguments != NULL ) {
    if ( is_an_arithmetic_operator(arguments->name) ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_arguments_for_ccalc(stream, arguments);
    }
    else {
      if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
	fprintf(stream, "%s", prepare_string_for_ccalc(arguments->name));
      }
      else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
	/* We need to print the name first, followed by the arguments of this argument! */
	fprintf(stream, "%s", prepare_string_for_ccalc(arguments->name));
	fprintf(stream, "(");
	print_constant_arguments_for_ccalc(stream, arguments->arguments);
	fprintf(stream, ")");
      }
      else {
	print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
      }
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      if ( arguments->value != NULL ) {
	if ( (strcmp(arguments->value, "true") != 0)
	     && (strcmp(arguments->value, "false") != 0) ) {
	  fprintf(stream, "=%s", prepare_string_for_ccalc(arguments->value));
	}
      }
    }
     
    /* Move on to the next argument after printing this one */
    arguments = arguments->next_argument;
    
    while (arguments != NULL ) {
      if ( is_an_arithmetic_operator(arguments->name) ) {
	// Arithmetic operators may have their own syntax.
	// E.g., we print "a + b", not "+(a,b)"
	print_arithmetic_operator_arguments_for_ccalc(stream, arguments);
      }
      else {
	if ( arguments->argument_kind == PLAIN_ARGUMENT ) {
	  fprintf(stream, ",%s", prepare_string_for_ccalc(arguments->name));
	}
	else if ( arguments->argument_kind == ARGUMENTED_ARGUMENT ) {
	  /* We need to print the name first, followed by the arguments of this argument! */
	  fprintf(stream, ",%s", prepare_string_for_ccalc(arguments->name));
	  fprintf(stream, "(");
	  print_constant_arguments_for_ccalc(stream, arguments->arguments);
	  fprintf(stream, ")");
	}
	else {
	  print_error_message(stderr, "\nInternal error: Trying to print an invalid kind of argument?\n");
	}
      }
      /* print the value if this argument is a constant with a value */
      if ( arguments->has_a_value == 1 ) {
	if ( arguments->value != NULL ) {
	  if ( (strcmp(arguments->value, "true") != 0)
	       && (strcmp(arguments->value, "false") != 0) ) {
	    fprintf(stream, "=%s", prepare_string_for_ccalc(arguments->value));
	  }
	}
      }

      /* Move on to the next argument after printing this one */
      arguments = arguments->next_argument;
    }
  }
}

// for debugging
void print_arithmetic_operator_term(FILE * stream, 
				    char * term,
				    struct mad_constant_argument * arguments) {
  if (term != NULL ) {
    if ( arguments == NULL ) {
      fprintf(stream, "%s", term);
    }
    else {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      if ( is_an_arithmetic_operator(term) ) {
	if ( strcmp(term, "+") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments(stream, arguments);
	  fprintf(stream, "+");
	  print_arithmetic_operator_arguments(stream, arguments->next_argument);
	  fprintf(stream, ")");
	}
	else if ( strcmp(term, "*") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments(stream, arguments);
	  fprintf(stream, "*");
	  print_arithmetic_operator_arguments(stream, arguments->next_argument);
	  fprintf(stream, ")");
	}
      }
      else {
	fprintf(stream, "%s", term);
	/* we also need to print the arguments, if there are any. */
	if (arguments != NULL ) {
	  fprintf(stream, "(");
	  print_constant_arguments(stream, arguments);
	  fprintf(stream, ")");
	}
      }
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      fprintf(stream, "=%s", arguments->value);
    }
  }
}

// for printing out a term (and arguments) in an axiom to be used as input for ccalc.
void print_arithmetic_operator_term_for_ccalc(FILE * stream, 
					      char * term,
					      struct mad_constant_argument * arguments) {
  if (term != NULL ) {
    if ( arguments == NULL ) {
      fprintf(stream, "%s", prepare_string_for_ccalc(term));
    }
    else {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      if ( is_an_arithmetic_operator(term) ) {
	if ( strcmp(term, "+") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments);
	  fprintf(stream, "+");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->next_argument);
	  fprintf(stream, ")");
	}
	else if ( strcmp(term, "*") == 0 ) {
	  fprintf(stream, "(");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments);
	  fprintf(stream, "*");
	  print_arithmetic_operator_arguments_for_ccalc(stream, arguments->next_argument);
	  fprintf(stream, ")");
	}
      }
      else {
	fprintf(stream, "%s", prepare_string_for_ccalc(term));
	/* we also need to print the arguments, if there are any. */
	if (arguments != NULL ) {
	  fprintf(stream, "(");
	  print_constant_arguments_for_ccalc(stream, arguments);
	  fprintf(stream, ")");
	}
      }
    }
    /* print the value if this argument is a constant with a value */
    if ( arguments->has_a_value == 1 ) {
      fprintf(stream, "=%s", prepare_string_for_ccalc(arguments->value));
    }
  }
}

/* for debugging */
void print_formula(FILE * stream, struct mad_formula * formula) {
  
  if ( formula->formula_kind == BINARY_CONNECTIVE ) {
    fprintf(stream, "(");
    if ( formula->left_formula != NULL ) {
      print_formula(stream, formula->left_formula);
    }
    switch ( formula->connective ) {
    case AND:
      fprintf(stream, " & ");
      break;
    case OR:
      fprintf(stream, " | ");
      break;
    case EQUIV:
      fprintf(stream, " <-> ");
      break;
    case IMPLIES:
      fprintf(stream, " -> ");
      break;
    }
    if ( formula->right_formula != NULL ) {
      print_formula(stream, formula->right_formula);
    }
    fprintf(stream, ")");
  }
  else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
    switch ( formula->connective ) {
    case NEG:
      fprintf(stream, "-");
      break;
    }
    fprintf(stream, "(");
    if ( formula->right_formula != NULL ) {
      print_formula(stream, formula->right_formula);
    }
    fprintf(stream, ")");
  }
  else if ( formula->formula_kind == ATOMIC_FORMULA ) {
    if ( is_an_arithmetic_operator(formula->LHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term(stream, 
				     formula->LHS_term, 
				     formula->LHS_arguments);
    }
    else {
      fprintf(stream, "%s", formula->LHS_term);
      /* we also need to print the arguments, if there are any. */
      if (formula->LHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments(stream, formula->LHS_arguments);
	fprintf(stream, ")");
      }
    }
    /* If the value is given explicitly, print it */
    if ( formula->RHS_term != NULL ) {
      fprintf(stream, "=");
      if ( is_an_arithmetic_operator(formula->RHS_term) ) {
	// Arithmetic operators may have their own syntax.
	// E.g., we print "a + b", not "+(a,b)"
	print_arithmetic_operator_term(stream, 
				       formula->RHS_term, 
				       formula->RHS_arguments);
      }
      else {
	fprintf(stream, "%s", formula->RHS_term);
	if (formula->RHS_arguments != NULL ) {
	  fprintf(stream, "(");
	  print_constant_arguments(stream, formula->RHS_arguments);
	  fprintf(stream, ")");
	}
      }
    }
  }
  else if ( formula->formula_kind == INEQUALITY ) {
    if ( is_an_arithmetic_operator(formula->LHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term(stream, 
				     formula->LHS_term, 
				     formula->LHS_arguments);
    }
    else {
      fprintf(stream, "%s", formula->LHS_term);
      /* we also need to print the arguments, if there are any. */
      if (formula->LHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments(stream, formula->LHS_arguments);
	fprintf(stream, ")");
      }
    }
    if ( formula->inequality_kind == LESS_THAN ) {
      fprintf(stream, "<");
    }
    if ( is_an_arithmetic_operator(formula->RHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.g., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term(stream, 
				     formula->RHS_term, 
				     formula->RHS_arguments);
    }
    else {
      fprintf(stream, "%s", formula->RHS_term);
      if (formula->RHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments(stream, formula->RHS_arguments);
	fprintf(stream, ")");
      }
    }
  }
  else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
    if (formula->connective == TRUE) {
	fprintf(stream, "true");
    }
    if (formula->connective == FALSE) {
	fprintf(stream, "false");
    }
  }
  else if ( formula->formula_kind == QUANTIFIER ) {
    fprintf(stream, "(");
    if (formula->quantifier == EXISTS) {
      fprintf(stream, "exists ");
    }
    else if (formula->quantifier == FORALL) {
      fprintf(stream, "forall ");
    }
    else {
      print_error_message(stderr, "Formula with quantifier doesn't have a valid quantifier!");
    }
    if ( formula->quantified_variable != NULL ) {
      fprintf(stream, "%s ", formula->quantified_variable);
    }
    else {
      print_error_message(stderr, "Formula with quantifier has no quantified variable!");
    }
    if ( formula->right_formula != NULL ) {
      print_formula(stream, formula->right_formula);
    }
    fprintf(stream, ")");
  }
  else {
    print_error_message(stderr, "Formula doesn't have a valid type!");
  }
}

/* for printing out a formula in an axiom to be used as input for ccalc. */
void print_formula_for_ccalc(FILE * stream, struct mad_formula * formula) {

  if ( formula->formula_kind == BINARY_CONNECTIVE ) {
    fprintf(stream, "(");
    if ( formula->left_formula != NULL ) {
      print_formula_for_ccalc(stream, formula->left_formula);
    }
    switch ( formula->connective ) {
    case AND:
      fprintf(stream, " & ");
      break;
    case OR:
      fprintf(stream, " ++ ");
      break;
    case EQUIV:
      fprintf(stream, " <-> ");
      break;
    case IMPLIES:
      fprintf(stream, " ->> ");
      break;
    }
    if ( formula->right_formula != NULL ) {
      print_formula_for_ccalc(stream, formula->right_formula);
    }
    fprintf(stream, ")");
  }
  else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
    switch ( formula->connective ) {
    case NEG:
      fprintf(stream, "-");
      break;
    }
    fprintf(stream, "(");
    if ( formula->right_formula != NULL ) {
      print_formula_for_ccalc(stream, formula->right_formula);
    }
    fprintf(stream, ")");
  }
  else if ( formula->formula_kind == ATOMIC_FORMULA ) {
    /* If the formula is a sort_predicate formula sort_name(v),
       we will just print it as "[\/var(sort_name) | var(sort_name)=v ]" */
    if ( is_a_known_sort(formula->LHS_term) ) {
      /* !!!: The way we print CCalc code corresponding to such sort_predicate formulas
	 doesn't work if the sort_predicate is action. */
      if ( strcmp(formula->LHS_term, "action") == 0 ) {
	fprintf(stream, 
		"[\\/ var(%s, -1) | var(%s, -1)=%s ]",
		prepare_string_for_ccalc(formula->LHS_term),
		prepare_string_for_ccalc(formula->LHS_term),
		prepare_string_for_ccalc(formula->LHS_arguments->name));
	fprintf(stderr, "***************************************************************\n");
	fprintf(stderr, "We don't know how to write \"action\" sort-name formulas in CCalc!\n");
	fprintf(stderr, "(So the output file just written is incorrect.)\n");
	fprintf(stderr, "***************************************************************\n");
      }
      else {
	if ( is_an_integer_range(formula->LHS_term) ) {
	  fprintf(stream, 
		  "[\\/ var(%s, -1) | var(%s, -1)=%s ]",
		  convert_integer_range_to_ccalc_acceptable_form(formula->LHS_term),
		  convert_integer_range_to_ccalc_acceptable_form(formula->LHS_term),
		  prepare_string_for_ccalc(formula->LHS_arguments->name));
	}
	else {
	  fprintf(stream, 
		  "[\\/ var(%s, -1) | var(%s, -1)=%s ]",
		  prepare_string_for_ccalc(formula->LHS_term),
		  prepare_string_for_ccalc(formula->LHS_term),
		  prepare_string_for_ccalc(formula->LHS_arguments->name));
	}
      }
    }
    else if ( is_an_arithmetic_operator(formula->LHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.eg., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term_for_ccalc(stream, 
					       formula->LHS_term, 
					       formula->LHS_arguments);
    }
    else {
      fprintf(stream, "%s", prepare_string_for_ccalc(formula->LHS_term));
      /* we also need to print the arguments, if there are any. */
      if (formula->LHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments_for_ccalc(stream, formula->LHS_arguments);
	fprintf(stream, ")");
      }
    }
    /* If the RHS term is NULL then it is shorthand for value "true"
       for a Boolean constant.  In this case the value isn't printed. */
    if ( formula->RHS_term != NULL ) {
      /* We only print the RHS_term if it is not "true"
	 If it's "true", then we assume the LHS_term is
	 a Boolean constant so we don't print the value. */
      if ( strcmp(formula->RHS_term, "true") != 0 ) {
	fprintf(stream, "=");
	if ( is_an_arithmetic_operator(formula->RHS_term) ) {
	  // Arithmetic operators may have their own syntax.
	  // E.eg., we print "a + b", not "+(a,b)"
	  print_arithmetic_operator_term_for_ccalc(stream, 
						   formula->RHS_term, 
						   formula->RHS_arguments);
	}
	else {
	  fprintf(stream, "%s", prepare_string_for_ccalc(formula->RHS_term));
	  if (formula->RHS_arguments != NULL ) {
	    fprintf(stream, "(");
	    print_constant_arguments_for_ccalc(stream, formula->RHS_arguments);
	    fprintf(stream, ")");
	  }
	}
      }
    }
  }
  else if ( formula->formula_kind == INEQUALITY ) {
    if ( is_an_arithmetic_operator(formula->LHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.eg., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term_for_ccalc(stream, 
					       formula->LHS_term, 
					       formula->LHS_arguments);
    }
    else {
      fprintf(stream, "%s", prepare_string_for_ccalc(formula->LHS_term));
      /* we also need to print the arguments, if there are any. */
      if (formula->LHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments_for_ccalc(stream, formula->LHS_arguments);
	fprintf(stream, ")");
      }
    }
    if ( formula->inequality_kind == LESS_THAN ) {
      fprintf(stream, "<");
    }
    if ( is_an_arithmetic_operator(formula->RHS_term) ) {
      // Arithmetic operators may have their own syntax.
      // E.eg., we print "a + b", not "+(a,b)"
      print_arithmetic_operator_term_for_ccalc(stream, 
					       formula->RHS_term, 
					       formula->RHS_arguments);
    }
    else {
      fprintf(stream, "%s", prepare_string_for_ccalc(formula->RHS_term));
      if (formula->RHS_arguments != NULL ) {
	fprintf(stream, "(");
	print_constant_arguments_for_ccalc(stream, formula->RHS_arguments);
	fprintf(stream, ")");
      }
    }
  }
  else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
    if (formula->connective == TRUE) {
      fprintf(stream, "true");
    }
    if (formula->connective == FALSE) {
      fprintf(stream, "false");
    }
  }
  else if ( formula->formula_kind == QUANTIFIER ) {
    if (formula->quantifier == EXISTS) {
      fprintf(stream, "[\\/ ");
    }
    else if (formula->quantifier == FORALL) {
      fprintf(stream, "[/\\ ");
    }
    else {
      print_error_message(stderr, "Formula with quantifier doesn't have a valid quantifier!");
    }
    if ( formula->quantified_variable != NULL ) {
      fprintf(stream, "%s | ", prepare_string_for_ccalc(formula->quantified_variable));
    }
    else {
      print_error_message(stderr, "Formula with quantifier has no quantified variable!");
    }
    print_formula_for_ccalc(stream, formula->right_formula);
    fprintf(stream, "]");
  }
  else {
    print_error_message(stderr, "Formula doesn't have a valid type!");
  }
}

/* for debugging */
void print_current_axioms(FILE * stream) {
  struct axiom_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Axioms:\n");
  current = axiom_list_first;
  while ( current != NULL ) {

    if ( current->node_data != NULL ) {
      fprintf(stream, "%d: ", counter);
      if ( current->node_data->F != NULL ) {
	print_formula(stream, current->node_data->F);
      }
      else {
	fprintf(stream, "NULL-formula");
      }
      fprintf(stream, "\n");
      fprintf(stream, "    if     ");
      if ( current->node_data->G != NULL ) {
	print_formula(stream, current->node_data->G);
      }
      else {
	fprintf(stream, "NULL-formula");
      }
      fprintf(stream, "\n");
      /* print last part if it exists */
      if ( current->node_data->H != NULL ) {
	fprintf(stream, "    after  ");
	print_formula(stream, current->node_data->H);
	fprintf(stream, "\n");
      }
      counter++;
    }
    current = current->next;
  }
}

/* for debugging.  
   To print out a specific list of axioms (e.g. of a specific module). */
void print_axiom_list(FILE * stream, struct axiom_list_node * list) {
  struct axiom_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Axioms:\n");
  current = list;
  while ( current != NULL ) {
    
    fprintf(stream, "%d: ", counter);
    if (current->node_data == NULL) {
      fprintf(stream, "Axiom list node has no data!\n");
    }
    else {
      if ( current->node_data->F != NULL ) {
	print_formula(stream, current->node_data->F);
      }
      else {
	print_error_message(stderr, "NULL formula in axiom!");
      }
      fprintf(stream, "\n");
      fprintf(stream, "    if     ");
      if ( current->node_data->G != NULL ) {
	print_formula(stream, current->node_data->G);
      }
      else {
	print_error_message(stderr, "NULL formula in axiom!");
      }
      fprintf(stream, "\n");
      /* print last part if it exists */
      if ( current->node_data->H != NULL ) {
	fprintf(stream, "    after  ");
	print_formula(stream, current->node_data->H);
	fprintf(stream, "\n");
      }
    }
    counter++;
    current = current->next;
  }
}

/* for printing out an axiom declaration section to be used as ccalc input */
void print_axiom_list_for_ccalc(FILE * stream, struct axiom_list_node * list) {
  struct axiom_list_node * current;

  current = list;
  if ( current != NULL ) {
    //fprintf(stream, "% :- axioms\n\n");
    while ( current != NULL ) {
      if ( current->node_data->F != NULL ) {
	/* !!!!!!!!!!!! Just a hack to see what happens when we don't print
	   axioms with equivalence heads
	if ( current->node_data->F->formula_kind == BINARY_CONNECTIVE ) {
	  current = current->next;
	  continue;
	}
	*/
	fprintf(stream, "caused ");
	print_formula_for_ccalc(stream, current->node_data->F);
      }
      else {
	print_error_message(stderr, "NULL formula in axiom!");
      }
      fprintf(stream, "\n");
      fprintf(stream, "    if     ");
      if ( current->node_data->G != NULL ) {
	print_formula_for_ccalc(stream, current->node_data->G);
      }
      else {
	print_error_message(stderr, "NULL formula in axiom!");
      }
      /* print last part if it exists */
      if ( current->node_data->H != NULL ) {
	fprintf(stream, "\n");
	fprintf(stream, "    after  ");
	print_formula_for_ccalc(stream, current->node_data->H);
      }
      fprintf(stream, ".\n");
      current = current->next;
    }
  }
}
  

/****** Managing lists of modules **************/

/* initialize the list of modules */
void init_modules() {
  module_list_first = module_list_last = NULL;
}

/* Looks up a module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct module_list_node * lookup_module(char * lookup_name) {
  struct module_list_node * current;
  int counter;

  current = module_list_first;
  if ( lookup_name != NULL ) {
    while ( current != NULL ) {
      if( strcmp(current->node_data->name, lookup_name) == 0 ) {
	return current;
      }
      current = current->next;
    }
  }
  /* It was not found in the list */
  return NULL;
}


/* Called when inserting a new module.  Makes a new module */
struct mad_module * make_new_module (char * name) {

  struct mad_module * new_module ;

  new_module = (struct mad_module *) malloc( sizeof(struct mad_module) );
  new_module->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_module->name, name );
  
  /* Initially the import index of a module is 0.
     It will increase by importing modules */
  new_module->import_index = 0;
  return new_module;
}

/* This inserts a new module into the list of modules */
void insert_module (char * name) {

  struct module_list_node * new_node;

  new_node = (struct module_list_node *) malloc( sizeof(struct module_list_node) );
  new_node->node_data = make_new_module(name);
  new_node->next = NULL;

  /* module_list_last is the pointer to the end of the list */
  if ( module_list_last == NULL ) { /* the list is empty */
    module_list_first = new_node;
    module_list_last = new_node;
  }
  else {
  module_list_last->next = new_node;
  module_list_last = new_node;
  }

}

/* Called at the end of a module description, to update the
   information (sorts, objects, etc.) about the module.  We use
   this to assign the parsed information to a specific module. */
void update_module (char * name) {

  struct module_list_node * this_module;

  this_module = lookup_module(name);

  if ( this_module != NULL ) {
    this_module->node_data->module_sorts = sort_list_first;
    this_module->node_data->module_inclusions = inclusion_list_first;
    this_module->node_data->module_sort_dependencies = sort_dependency_list_first;
    this_module->node_data->module_objects = object_list_first;;
    this_module->node_data->module_constants = constant_list_first;
    this_module->node_data->module_variables = variable_list_first;
    this_module->node_data->module_axioms = axiom_list_first;
    this_module->node_data->module_renamings = renaming_list_first;

    this_module->node_data->import_index = global_import_index;
  }
  else {
    print_error_message(stderr, "Internal error: trying to update nonexistent module?\n");
  }

};  

/* This inserts names of all modules into the list of identifiers */
void insert_module_names_into_identifiers() {

  struct module_list_node * current;

  current = module_list_first;
  while ( current != NULL ) {
    if ( lookup_identifier(current->node_data->name) == NULL ) {
      insert_identifier(current->node_data->name);
      update_id_type(current->node_data->name, MODULE_ID);
    }
    else {
      print_error_message(stderr, "Internal error: Module name already an identifier even though\n");
      print_error_message(stderr, "                the list should have been cleared?\n");
    }

    current = current->next;
  }


}

/* Makes a copy of a module to be used during importing.
   This is necessary because sort and constant renaming
   statements modify an imported module but we need to
   keep the original copy to be able to import it again 
   later. */
struct mad_module * copy_module(struct mad_module * module) {

  struct mad_module * copied_module;

  copied_module = make_new_module(module->name);
  /* A new module has import index 0 but our copied module should have the 
     same import index as the original. */

  copied_module->import_index = module->import_index;
  copied_module->module_sorts = copy_sort_list(module->module_sorts);
  copied_module->module_inclusions = copy_inclusion_list(module->module_inclusions);
  copied_module->module_sort_dependencies = copy_sort_dependency_list(module->module_sort_dependencies);
  copied_module->module_objects = copy_object_list(module->module_objects);
  copied_module->module_constants = copy_constant_list(module->module_constants);
  copied_module->module_variables = copy_variable_list(module->module_variables);
  copied_module->module_axioms = copy_axiom_list(module->module_axioms);
  copied_module->module_renamings = copy_renaming_list(module->module_renamings);

  return copied_module;
}



/* Takes a formula and renames sorts from a given module,
   by replacing them with new sorts, according to sort renaming 
   clauses currently being processed. */
void rename_sorts_of_module_in_formula(struct mad_module * module, struct mad_formula * formula) {

  struct renaming_list_node * current_renaming;

  if (formula != NULL) {
    
    /* If there is a left_formula, rename sorts in it */
    if ( formula->left_formula != NULL ) {
      rename_sorts_of_module_in_formula(module, formula->left_formula);
    }
    
    /* This is where we do the essential renaming */
    if ( formula->formula_kind == ATOMIC_FORMULA ) {
      /* If it's an atom, the only possibility for this to include a sort name
	 is for the sort name to appear as the LHS_term
	 (with no LHS_arguments, no RHS_term, and no RHS_arguments). */
      
      if (formula->LHS_term != NULL) {
	if ( is_a_known_sort_in_module(formula->LHS_term, module) ) {
	  /* Go down the list of renaming clauses one by one */
	  current_renaming = import_renaming_list_first;
	  while (current_renaming != NULL) {
	    if ( strcmp(formula->LHS_term, current_renaming->node_data->atom->LHS_term) == 0 ) {
	      formula->LHS_term = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	      strcpy( formula->LHS_term, current_renaming->node_data->formula->LHS_term );
	      break; /* Move on to the next sort. */
	    }
	    /* Move on to the next renaming clause */
	    current_renaming = current_renaming->next;
	  }
	}
      }
    }
    
    /* If there is a right_formula, rename variables in it. */
    if ( formula->right_formula != NULL ) {
      rename_sorts_of_module_in_formula(module, formula->right_formula);
    }
  }
  
}


/* Renames sorts in a given module, based on renaming clauses */
void rename_sorts_in_module(struct mad_module * module) {

  struct renaming_list_node * current_renaming;

  struct sort_list_node * current_sort;
  struct inclusion_list_node * current_inclusion;
  struct sort_dependency_list_node * current_sort_dependency;
  struct object_list_node * current_object;
  struct constant_list_node * current_constant;
  struct string_list_node * current_argument; /* To rename arguments of a constant */
  struct variable_list_node * current_variable;
  struct axiom_list_node * current_axiom;
  struct renaming_case_list_node * current_renaming_case;
    
  /* Only rename sorts if there are sort renaming statements. */
  if ( import_renaming_list_first != NULL ) {

    /* Now that we allow "sort name predicates" in formulas (i.e. Agent(u))
       we may have occurrences of sort names in axioms or renaming statements.
       We rename those before renaming anything in declarations, because otherwise we
       won't be able to tell which identifiers are sorts. */
    current_axiom = module->module_axioms;
    while ( current_axiom != NULL ) {
      rename_sorts_of_module_in_formula(module, current_axiom->node_data->F);
      rename_sorts_of_module_in_formula(module, current_axiom->node_data->G);
      /* Not all axioms have an H formula so we need to check if it exists before
	 trying to rename variables in it. */
      if ( current_axiom->node_data->H != NULL) { 
	rename_sorts_of_module_in_formula(module, current_axiom->node_data->H);
      }
      current_axiom = current_axiom->next;
    }

    current_renaming = module->module_renamings;
    while ( current_renaming != NULL ) {
      rename_sorts_of_module_in_formula(module, current_renaming->node_data->atom);
      rename_sorts_of_module_in_formula(module, current_renaming->node_data->formula);

      /* Also rename the occurrences in the cases of the renaming. */
      current_renaming_case = current_renaming->node_data->cases;
      while ( current_renaming_case != NULL ) {
	rename_sorts_of_module_in_formula(module, current_renaming_case->node_data->condition);
	rename_sorts_of_module_in_formula(module, current_renaming_case->node_data->formula);
	current_renaming_case = current_renaming_case->next;
      }

      /* Also do this for the generalized_renaming */
      rename_sorts_of_module_in_formula(module, current_renaming->node_data->generalized_atom);
      current_renaming_case = current_renaming->node_data->generalized_atom_cases;
      while ( current_renaming_case != NULL ) {
	rename_sorts_of_module_in_formula(module, current_renaming_case->node_data->condition);
	rename_sorts_of_module_in_formula(module, current_renaming_case->node_data->formula);
	current_renaming_case = current_renaming_case->next;
      }

      current_renaming = current_renaming->next;
    }


    /* First rename the sort declaration itself */
    current_sort = module->module_sorts;
    while ( current_sort != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_sort->node_data->name, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_sort->node_data->name = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_sort->node_data->name, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next sort. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_sort = current_sort->next;
    }
    
    /* We need to do some renaming for inclusions too.  Since there are 
       two sorts (a supersort and a subsort) in every inclusion, and we want to
       concurrently do each renaming, we go down the list of inclusions twice,
       once for supersorts and once for subsorts. */
    current_inclusion = module->module_inclusions;
    while ( current_inclusion != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_inclusion->node_data->supersort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_inclusion->node_data->supersort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_inclusion->node_data->supersort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next supersort. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_inclusion = current_inclusion->next;
    }
    /* Now do the same for subsorts in the inclusion statements. */
    current_inclusion = module->module_inclusions;
    while ( current_inclusion != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_inclusion->node_data->subsort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_inclusion->node_data->subsort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_inclusion->node_data->subsort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next subsort. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_inclusion = current_inclusion->next;
    }
    /* After renaming all sorts in the inclusions, we need to check if this
       process introduced any cycles to the inclusion graph. */
    current_sort = module->module_sorts;
    while ( current_sort != NULL ) {
      if ( is_a_known_sort_in_module(current_sort->node_data->name, module) ) {
	if ( is_a_subsort_of_in_module(current_sort->node_data->name, 
				       current_sort->node_data->name,
				       module) ) {
	  renaming_sorts_leads_to_error = 1;
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Renaming sorts in module %s leads to a cycle in the inclusion graph.\n",
				     module->name);
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "   (\"%s << %s\") is one cycle but there may be more.\n",
				     current_sort->node_data->name,
				     current_sort->node_data->name);
	  break;
	}
      }
      else {
	renaming_sorts_leads_to_error = 1;
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Renaming sorts in module %s leads to having undeclared sort %s.\n",
				   module->name,
				   current_sort->node_data->name);
	break;
      }
      current_sort = current_sort->next;
    }
    if ( renaming_sorts_leads_to_error  == 1 ) {
      return;
    }


    /* We need to do some renaming for sort dependencies too.  Since there are 
       two sorts (a supersort and a subsort) in every sort dependency, and we want to
       concurrently do each renaming, we go down the list of sort dependencies twice,
       once for supersorts and once for subsorts. */
    current_sort_dependency = module->module_sort_dependencies;
    while ( current_sort_dependency != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_sort_dependency->node_data->supersort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_sort_dependency->node_data->supersort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_sort_dependency->node_data->supersort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next supersort. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_sort_dependency = current_sort_dependency->next;
    }
    /* Now do the same for subsorts in the sort dependency statements. */
    current_sort_dependency = module->module_sort_dependencies;
    while ( current_sort_dependency != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_sort_dependency->node_data->subsort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_sort_dependency->node_data->subsort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_sort_dependency->node_data->subsort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next subsort. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_sort_dependency = current_sort_dependency->next;
    }
    /* After renaming all sorts in the sort dependencies, we need to check if this
       process introduced any cycles to the sort dependency graph. */
    current_sort = module->module_sorts;
    while ( current_sort != NULL ) {
      if ( is_a_known_sort_in_module(current_sort->node_data->name, module) ) {
	if ( is_a_subsort_of_in_module(current_sort->node_data->name, 
				       current_sort->node_data->name,
				       module) ) {
	  renaming_sorts_leads_to_error = 1;
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "Renaming sorts in module %s leads to a cycle in the sort_dependency graph.\n",
				     module->name);
	  print_parser_error_message(stderr,
				     input_file_name, 
				     error_lineno,
				     "   (%s depens on %s) is one cycle but there may be more.\n",
				     current_sort->node_data->name,
				     current_sort->node_data->name);
	  break;
	}
      }
      else {
	renaming_sorts_leads_to_error = 1;
	print_parser_error_message(stderr,
				   input_file_name, 
				   error_lineno,
				   "Renaming sorts in module %s leads to having undeclared sort %s.\n",
				   module->name,
				   current_sort->node_data->name);
	break;
      }
      current_sort = current_sort->next;
    }
    if ( renaming_sorts_leads_to_error  == 1 ) {
      return;
    }

    
    // Then rename the sorts of object declarations,
    // and the sort names that occur as arguments in declarations
    current_object = module->module_objects;
    while ( current_object != NULL ) {
      /* rename arguments */
      current_argument = current_object->node_data->arguments;
      while (current_argument != NULL ) {
	/* Go down the list of renaming clauses one by one */
	current_renaming = import_renaming_list_first;
	while (current_renaming != NULL) {
	  if ( strcmp(current_argument->node_data, current_renaming->node_data->atom->LHS_term) == 0 ) {
	    current_argument->node_data = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	    strcpy( current_argument->node_data, current_renaming->node_data->formula->LHS_term );
	    break; /* Move on to the next argument of this object. */
	  }
	  /* Move on to the next renaming clause */
	  current_renaming = current_renaming->next;
	}
	current_argument = current_argument->next;
      }
      
      // rename the object sort
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_object->node_data->sort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_object->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_object->node_data->sort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next object. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_object = current_object->next;
    }
    
    /* For constant declarations, rename sorts of arguments and domains */
    current_constant = module->module_constants;
    while ( current_constant != NULL ) {
      /* rename arguments */
      current_argument = current_constant->node_data->arguments;
      while (current_argument != NULL ) {
	/* Go down the list of renaming clauses one by one */
	current_renaming = import_renaming_list_first;
	while (current_renaming != NULL) {
	  if ( strcmp(current_argument->node_data, current_renaming->node_data->atom->LHS_term) == 0 ) {
	    current_argument->node_data = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	    strcpy( current_argument->node_data, current_renaming->node_data->formula->LHS_term );
	    break; /* Move on to the next argument of this constant. */
	  }
	  /* Move on to the next renaming clause */
	  current_renaming = current_renaming->next;
	}
	current_argument = current_argument->next;
      }
      
      /* rename the domain sort */
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_constant->node_data->domain, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_constant->node_data->domain = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_constant->node_data->domain, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next constant. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_constant = current_constant->next;
    }
    
    /* Finally rename the sorts of variables */
    current_variable = module->module_variables;
    while ( current_variable != NULL ) {
      /* Go down the list of renaming clauses one by one */
      current_renaming = import_renaming_list_first;
      while (current_renaming != NULL) {
	if ( strcmp(current_variable->node_data->sort, current_renaming->node_data->atom->LHS_term) == 0 ) {
	  current_variable->node_data->sort = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->formula->LHS_term) + 1 ) );
	  strcpy( current_variable->node_data->sort, current_renaming->node_data->formula->LHS_term );
	  break; /* Move on to the next variable. */
	}
	/* Move on to the next renaming clause */
	current_renaming = current_renaming->next;
      }
      current_variable = current_variable->next;
    }    
    
  }

}

/* Renames variables in a given module by prepending an
   import prefix to all variables */
void rename_variables_in_module(struct mad_module * module, int import_index) {

  struct variable_list_node * current_variable;

  struct axiom_list_node * current_axiom;

  struct renaming_list_node * current_renaming;
  struct renaming_case_list_node * current_renaming_case;

  /* First we need to rename the variables in the axioms.
     (If we rename the variables in the declarations first, then we have no way left
     of recognizing which identifiers in axioms are variables, since lookups won't match
     the renamed version). */

  current_axiom = module->module_axioms;
  while ( current_axiom != NULL ) {
    rename_variables_of_module_in_formula(module, current_axiom->node_data->F, import_index);
    rename_variables_of_module_in_formula(module, current_axiom->node_data->G, import_index);
    /* Not all axioms have an H formula so we need to check if it exists before
       trying to rename variables in it. */
    if ( current_axiom->node_data->H != NULL) { 
      rename_variables_of_module_in_formula(module, current_axiom->node_data->H, import_index);
    }
    
    current_axiom = current_axiom->next;
  }

  /* It isn't really necessary to rename variables in the renaming
     statements for the module, because these variables are never
     used after the initial processing of the import which introduced
     the renaming to the module.  But we still rename these variables
     in case we use them later. */
  current_renaming = module->module_renamings;
  while ( current_renaming != NULL ) {
    rename_variables_of_module_in_formula(module, current_renaming->node_data->atom, import_index);
    rename_variables_of_module_in_formula(module, current_renaming->node_data->formula, import_index);

    /* We don't need to rename variables in the list of free variables
       for a renaming because it is empty at this point.  After a renaming 
       has been processed both the LHS and the RHS will have the same variables
       and the free variable list of the renaming will be reset to NULL. */

    /* Also rename the occurrences in the cases of the renaming. */
    current_renaming_case = current_renaming->node_data->cases;
    while ( current_renaming_case != NULL ) {
      rename_variables_of_module_in_formula(module, current_renaming_case->node_data->condition, import_index);
      rename_variables_of_module_in_formula(module, current_renaming_case->node_data->formula, import_index);
      current_renaming_case = current_renaming_case->next;
    }

    /* Also do this for the generalized atom and its cases */
    rename_variables_of_module_in_formula(module, current_renaming->node_data->generalized_atom, import_index);
    current_renaming_case = current_renaming->node_data->generalized_atom_cases;
    while ( current_renaming_case != NULL ) {
      rename_variables_of_module_in_formula(module, current_renaming_case->node_data->condition, import_index);
      rename_variables_of_module_in_formula(module, current_renaming_case->node_data->formula, import_index);
      current_renaming_case = current_renaming_case->next;
    }
    
    current_renaming = current_renaming->next;
  }

  /* Now rename variable occurrences in the declarations */
  current_variable = module->module_variables;
  while ( current_variable != NULL ) {
    current_variable->node_data->name = prepend_import_prefix_to_string(current_variable->node_data->name, import_index);
    current_variable = current_variable->next;
  }
  
}

/* Renames constants in a given module based on renaming clauses seen.
   It adds an import prefix to constant names and adds extra arguments as necessary. */
void rename_constants_in_module(struct mad_module * module, int import_index) {

  /* To loop through renaming lists */
  struct renaming_list_node * current_renaming;
  struct renaming_list_node * current_renaming2;

  /* To add equivalences for each case of a renaming */
  struct renaming_case_list_node * current_renaming_case;

  struct constant_list_node * current_constant;

  // This and the next are to rename arguments of a constant
  struct string_list_node * first_argument; 

  struct string_list_node * current_argument; 

  // Used to go down list of extra free variables on the RHS of the renaming clause.
  struct string_list_node * current_free_variable;

  // This is the declared variable found by looking
  // up a free variable corresponding to a renaming clause.
  struct variable_list_node * free_variable_as_declared;

  struct axiom_list_node * current_axiom;

  // When adding extra arguments to constant occurrences
  // in axioms, we need to use new variables instead of
  // those used in the renaming statement.
  struct string_list_node * new_variable_list;

  // This is used when renaming constant occurrences in previous renamings, 
  // to keep track of whether we have already added the list of new variables
  // to that renaming (we may encounter the constant multiple times if there are
  // cases, and we want to add variables only once)
  int new_variables_have_been_added;

  // This is used to make a copy of the constant name currently being renamed
  char * constant_name_copy; 

  /* The list of the equivalence axioms (corresponding to the renaming statements)
     which will be added to the beginning of the axioms of the module. */
  struct axiom_list_node * new_equivalence_axioms_first; 
  struct axiom_list_node * new_equivalence_axioms_last; 
  struct mad_formula * new_equivalence_formula; 
  struct mad_formula * true_formula; // To make a new formula containing 0-place connective "top"
  struct mad_axiom * new_equivalence_axiom;
  
  /* Go down the list of renaming clauses one by one */
  current_renaming = import_renaming_list_first;
  if (current_renaming != NULL) {
    /* There is at least one constant that is renamed in this import */

    init_axiom_list(&new_equivalence_axioms_first, &new_equivalence_axioms_last);

    while (current_renaming != NULL) {

      constant_name_copy = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->atom->LHS_term) + 1 ) );
      strcpy( constant_name_copy, current_renaming->node_data->atom->LHS_term );
      
      /* First rename occurrences in the constant declarations. */
      current_constant = module->module_constants;
      while ( current_constant != NULL ) {
	/* Check if the constant name is among those to be renamed */
	if ( strcmp(current_constant->node_data->name, constant_name_copy) == 0 ) {
	  current_constant->node_data->name = prepend_import_prefix_to_string(current_constant->node_data->name, import_index);
	  
	  /* Add extra arguments to the constant declaration, if any. */
	  if ( current_renaming->node_data->list_of_free_variables != NULL ) {
	    
	    /* If there are any arguments, go to the last one */
	    if ( current_constant->node_data->arguments != NULL ) {
	      /* Set first_argument to point to the first argument of the constant */
	      first_argument = current_constant->node_data->arguments;
	      /* Go down the list of arguments to reach the last argument */
	      current_argument = current_constant->node_data->arguments;
	      while ( current_argument->next != NULL ) {
		current_argument = current_argument->next;
	      }
	    }
	    else { /* The constant has no arguments so we create a new list 
		      of arguments corresponding to the extra variables. */
	      init_string_list(&first_argument, &current_argument);
	    }
	    /* At this point, first_argument points to the beginning of the argument list and 
	       current_argument points to the last argument in the list of arguments.
	       Now we can add the sorts for the extra free variables in the renaming clause. */
	    current_free_variable = current_renaming->node_data->list_of_free_variables;
	    while ( current_free_variable != NULL ) {
	      /* Find the sort of the variable */
	      free_variable_as_declared = lookup_variable(current_free_variable->node_data);
	      if ( free_variable_as_declared != NULL ) {
		insert_string_into_list(&first_argument, &current_argument, free_variable_as_declared->node_data->sort);
		/* Also increment the number of arguments in the declaration. */
		current_constant->node_data->num_arguments = current_constant->node_data->num_arguments + 1;
	      }
	      else {
		print_error_message(stderr, "Internal error: Free variable in renaming clause doesn't appear to be declared.\n");
	      }
	      current_free_variable = current_free_variable->next;
	    }
	  }
	}
	
	/* Move to the next constant declaration */
	current_constant = current_constant->next;
      }
      
      /* Now we're finished renaming occurrences of the constant in declaration.
	 We move onto renaming occurrences in axioms. */
      
      /* When we add extra variables as arguments to constant occurrences in formulas,
	 we need to create a new set of variables because i) we don't want to accidentally
	 use variables which already occur in the axiom, and ii) in case the axiom contains
	 different renamed constants.  (In such a case, introducing the same variables for
	 different renamed constants may change the meaning of the axiom.) */
      new_variable_list = make_new_variable_list_and_add_to_module(module,
								   current_renaming->node_data->list_of_free_variables);
            
      current_axiom = module->module_axioms;
      while ( current_axiom != NULL ) {
	rename_constant_in_formula(constant_name_copy, 
				   current_axiom->node_data->F, 
				   import_index, 
				   new_variable_list);
	rename_constant_in_formula(constant_name_copy,
				   current_axiom->node_data->G,
				   import_index, 
				   new_variable_list);
	/* Not all axioms have an H formula so we need to check if it exists before
	   trying to rename constants in it. */
	if ( current_axiom->node_data->H != NULL) { 
	  rename_constant_in_formula(constant_name_copy,
				     current_axiom->node_data->H, 
				     import_index, 
				     new_variable_list);
	}
	
	current_axiom = current_axiom->next;
      }
      
      /* We need to rename occurrences of the constant in the renaming clauses which have
	 been added to the module in earlier imports. */
      
      current_renaming2 = module->module_renamings;
      while (current_renaming2 != NULL) {
	
	// We might encounter the constant more than once, in the main formula and
	// also in formulas for cases (if there are any)
	// However, we want to add the new variables 
	// to the list of free variables only once 
	new_variables_have_been_added = 0;
	/* Update the RHS of the already existing renaming */
	if ( rename_constant_in_formula(constant_name_copy,
					current_renaming2->node_data->formula, 
					import_index, 
					new_variable_list) == 1 
	     && new_variables_have_been_added == 0) {
	  // If the constant was encountered and renamed in the formula, then
	  // any new variables added while renaming the constant will be extra
	  // variables occurring on the RHS of this old renaming statement.
	  append_string_list(&(current_renaming2->node_data->list_of_free_variables), 
			     copy_string_list(new_variable_list));
	  new_variables_have_been_added = 1;
	}
	/* Also rename the occurrences in the cases of the earlier renaming. */
	current_renaming_case = current_renaming2->node_data->cases;
	while ( current_renaming_case != NULL ) {
	  if ( rename_constant_in_formula(constant_name_copy,
					  current_renaming_case->node_data->formula, 
					  import_index, 
					  new_variable_list) == 1
	       && new_variables_have_been_added == 0) {
	    // If the constant was encountered and renamed in the formula, then
	    // any new variables added while renaming the constant will be extra
	    // variables occurring on the RHS of this old renaming statement.
	    append_string_list(&(current_renaming2->node_data->list_of_free_variables), 
			       copy_string_list(new_variable_list));
	    new_variables_have_been_added = 1;
	  }
	  current_renaming_case = current_renaming_case->next;
	}

	/* Note that we don't need to do anything for the generalized atom and 
	   generalized cases, because the generalized atom will be handled at the
	   same time as the atom (see a few lines below)
	   and the generalized cases have formulas which are all "false". */

	current_renaming2 = current_renaming2->next;
      }

      /* Finally, rename the constant in this renaming clause.  The extra variables
         added will be those from the renaming. */
      rename_constant_in_formula(constant_name_copy,
				 current_renaming->node_data->atom, 
				 import_index, 
				 current_renaming->node_data->list_of_free_variables);
      rename_constant_in_formula(constant_name_copy,
				 current_renaming->node_data->generalized_atom, 
				 import_index, 
				 current_renaming->node_data->list_of_free_variables);

      /* As a result of processing this renaming statement, we have made the set
	 of free variables on the LHS and the RHS the same.  So there are no more
	 extra free variables right now. */
      current_renaming->node_data->list_of_free_variables = NULL;

      /* After renaming the constant occurrence in the renaming clause, we add the
	 equivalence axioms corresponding to it. */
      if ( current_renaming->node_data->num_cases == 1) {
	new_equivalence_formula = make_new_formula(BINARY_CONNECTIVE);
	new_equivalence_formula->connective = EQUIV;
	new_equivalence_formula->left_formula = copy_formula(current_renaming->node_data->atom);
	new_equivalence_formula->right_formula = copy_formula(current_renaming->node_data->formula);
	
	true_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	true_formula->connective = TRUE;
	
	new_equivalence_axiom = make_new_axiom(new_equivalence_formula, true_formula, NULL);

	insert_axiom_into_list(new_equivalence_axiom,
			       &new_equivalence_axioms_first,
			       &new_equivalence_axioms_last);
      }
      else {
	current_renaming_case = current_renaming->node_data->cases;
	while ( current_renaming_case != NULL ) {
	  new_equivalence_formula = make_new_formula(BINARY_CONNECTIVE);
	  new_equivalence_formula->connective = EQUIV;
	  new_equivalence_formula->left_formula = copy_formula(current_renaming->node_data->atom);
	  new_equivalence_formula->right_formula = copy_formula(current_renaming_case->node_data->formula);
	
	  true_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	  true_formula->connective = TRUE;
	  
	  new_equivalence_axiom = make_new_axiom(new_equivalence_formula, 
						 current_renaming_case->node_data->condition,
						 NULL);
	  
	  insert_axiom_into_list(new_equivalence_axiom,
				 &new_equivalence_axioms_first,
				 &new_equivalence_axioms_last);

	  current_renaming_case = current_renaming_case->next;
	}
      }
      /* If there is a generalized atom, we also need to add equivalences for
	 those cases. */
      if ( current_renaming->node_data->generalized_atom != NULL ) {
	current_renaming_case = current_renaming->node_data->generalized_atom_cases;
	while ( current_renaming_case != NULL ) {
	  new_equivalence_formula = make_new_formula(BINARY_CONNECTIVE);
	  new_equivalence_formula->connective = EQUIV;
	  new_equivalence_formula->left_formula = copy_formula(current_renaming->node_data->generalized_atom);
	  new_equivalence_formula->right_formula = copy_formula(current_renaming_case->node_data->formula);
	  
	  true_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	  true_formula->connective = TRUE;
	  
	  new_equivalence_axiom = make_new_axiom(new_equivalence_formula, 
						 current_renaming_case->node_data->condition,
						 NULL);
	  
	  insert_axiom_into_list(new_equivalence_axiom,
				 &new_equivalence_axioms_first,
				 &new_equivalence_axioms_last);

	  current_renaming_case = current_renaming_case->next;
	}
      }
      
      
      /* Move on to the next renaming clause */
      current_renaming = current_renaming->next;
    }
    
    /* Okay.  Now all the renaming clauses from this import have been processed but
       now we also should process all the renaming clauses from previous imports.
       We add import prefixes to all and we also might need to add extra variables 
       (if they were defined in terms of constants renamed this time around.)

       This is basically just like the renaming clauses from this import, with the
       difference that the extra free variables in these statements are variables
       of "module" (the module being imported), instead of the module which is
       currently being read in by the parser.  Therefore, when we create a new
       list of variables corresponding to the extra variables, we need to search
       for them among "module"s variables. */

    /* Go down the list of the renaming clauses of the module one by one */
    current_renaming = module->module_renamings;

    while (current_renaming != NULL) {

      constant_name_copy = (char *) malloc( sizeof(char) * ( strlen(current_renaming->node_data->atom->LHS_term) + 1 ) );
      strcpy( constant_name_copy, current_renaming->node_data->atom->LHS_term );
      
      /* First rename occurrences in the constant declarations. */
      current_constant = module->module_constants;
      while ( current_constant != NULL ) {
	/* Check if the constant name is among those to be renamed */
	if ( strcmp(current_constant->node_data->name, constant_name_copy) == 0 ) {
	  current_constant->node_data->name = prepend_import_prefix_to_string(current_constant->node_data->name, import_index);
	  
	  /* Add extra arguments to the constant declaration, if any. */
	  if ( current_renaming->node_data->list_of_free_variables != NULL ) {
	    
	    /* If there are any arguments, go to the last one */
	    if ( current_constant->node_data->arguments != NULL ) {
	      /* Set first_argument to point to the first argument of the constant */
	      first_argument = current_constant->node_data->arguments;
	      /* Go down the list of arguments to reach the last argument */
	      current_argument = current_constant->node_data->arguments;
	      while ( current_argument->next != NULL ) {
		current_argument = current_argument->next;
	      }
	    }
	    else { /* The constant has no arguments so we create a new list 
		      of arguments corresponding to the extra variables. */
	      init_string_list(&first_argument, &current_argument);
	    }
	    /* At this point, first_argument points to the beginning of the argument list and 
	       current_argument points to the last argument in the list of arguments.
	       Now we can add the sorts for the extra free variables in the renaming clause. */
	    current_free_variable = current_renaming->node_data->list_of_free_variables;
	    while ( current_free_variable != NULL ) {
	      /* Find the sort of the variable.  Here we lookup the variable in the
		 given module because when processing renamings already part of the module,
	         the extra free variables are those we have created anew and added to the
		 module.  (I.e. They are not variables which are declared in the module
		 currently being parsed.) */
	      free_variable_as_declared = lookup_variable_in_module(current_free_variable->node_data, module);
	      if ( free_variable_as_declared != NULL ) {
		insert_string_into_list(&first_argument, &current_argument, free_variable_as_declared->node_data->sort);
		/* Also increment the number of arguments in the declaration. */
		current_constant->node_data->num_arguments = current_constant->node_data->num_arguments + 1;
	      }
	      else {
		print_error_message(stderr, "Internal error: Free variable in renaming clause doesn't appear to be declared.\n");
	      }
	      current_free_variable = current_free_variable->next;
	    }
	  }
	}
	
	/* Move to the next constant declaration */
	current_constant = current_constant->next;
      }
      
      /* Now we're finished renaming occurrences of the constant in declaration.
	 We move onto renaming occurrences in axioms. */
      
      /* When we add extra variables as arguments to constant occurrences in formulas,
	 we need to create a new set of variables because i) we don't want to accidentally
	 use variables which already occur in the axiom, and ii) in case the axiom contains
	 different renamed constants.  (In such a case, introducing the same variables for
	 different renamed constants may change the meaning of the axiom.) */
      new_variable_list = make_new_variable_list_of_given_module_and_add_to_module(module,
										   current_renaming->node_data->list_of_free_variables);
      
      current_axiom = module->module_axioms;
      while ( current_axiom != NULL ) {
	/* We need to check if the axiom in which this constant appears is an equivalence
	   which was added when the constant was renamed.  If so, we use the
	   list_of_free_variables (not the new_variable_list) in order to match the
	   variables used on the RHS of the equivalence. */ 
	if (current_axiom->node_data->F != NULL 
	    && current_axiom->node_data->F->formula_kind == BINARY_CONNECTIVE
	    && current_axiom->node_data->F->connective == EQUIV
	    && current_axiom->node_data->F->left_formula != NULL
	    && current_axiom->node_data->F->left_formula->formula_kind == ATOMIC_FORMULA
	    && strcmp(current_axiom->node_data->F->left_formula->LHS_term, constant_name_copy) == 0) {
	  /* We just need to rename the F part of the axiom since the G part is "top" and the H part is empty. */
	  rename_constant_in_formula(constant_name_copy, 
				     current_axiom->node_data->F, 
				     import_index, 
				     current_renaming->node_data->list_of_free_variables);
	}
	else {
	  rename_constant_in_formula(constant_name_copy, 
				     current_axiom->node_data->F, 
				     import_index, 
				     new_variable_list);
	  rename_constant_in_formula(constant_name_copy,
				     current_axiom->node_data->G,
				     import_index, 
				     new_variable_list);
	  /* Not all axioms have an H formula so we need to check if it exists before
	     trying to rename constants in it. */
	  if ( current_axiom->node_data->H != NULL) { 
	    rename_constant_in_formula(constant_name_copy,
				       current_axiom->node_data->H, 
				       import_index, 
				       new_variable_list);
	  }
	}
	current_axiom = current_axiom->next;
      }
      
      /* We need to rename occurrences of the constant in the other renaming clauses which have
	 been added to the module in earlier imports. */
      
      current_renaming2 = current_renaming->next;
      while (current_renaming2 != NULL) {
	// We might encounter the constant more than once, in the main formula and
	// also in formulas for cases (if there are any)
	// However, we want to add the new variables 
	// to the list of free variables only once 
	new_variables_have_been_added = 0;

	/* Update the RHS of the already existing renaming */
	if ( rename_constant_in_formula(constant_name_copy,
					current_renaming2->node_data->formula, 
					import_index, 
					new_variable_list) == 1
	     && new_variables_have_been_added == 0 ) {
	  // If the constant was encountered and renamed in the formula, then
	  // any new variables added while renaming the constant will be extra
	  // variables occurring on the RHS of this old renaming statement.
	  append_string_list(&(current_renaming2->node_data->list_of_free_variables), 
			     copy_string_list(new_variable_list));
	  new_variables_have_been_added = 1;
	}
	/* Also rename the occurrences in the cases of the earlier renaming. */
	current_renaming_case = current_renaming2->node_data->cases;
	while ( current_renaming_case != NULL ) {
	  if ( rename_constant_in_formula(constant_name_copy,
					  current_renaming_case->node_data->formula, 
					  import_index, 
					  new_variable_list) == 1
	       && new_variables_have_been_added == 0 ) {
	    // If the constant was encountered and renamed in the formula, then
	    // any new variables added while renaming the constant will be extra
	    // variables occurring on the RHS of this old renaming statement.
	    append_string_list(&(current_renaming2->node_data->list_of_free_variables), 
			       copy_string_list(new_variable_list));
	    new_variables_have_been_added = 1;
	  }
	  current_renaming_case = current_renaming_case->next;
	}
	
	/* Note that we don't need to do anything for the generalized atom and 
	   generalized cases, because the generalized atom will be handled at the
	   same time as the atom (see a few lines below)
	   and the generalized cases have formulas which are all "false". */
	
	current_renaming2 = current_renaming2->next;
      }
      
      /* Finally, rename the constant in this renaming clause.  The extra variables
         added will be those from the renaming. */
      rename_constant_in_formula(constant_name_copy,
				 current_renaming->node_data->atom, 
				 import_index, 
				 current_renaming->node_data->list_of_free_variables);
      rename_constant_in_formula(constant_name_copy,
				 current_renaming->node_data->generalized_atom, 
				 import_index, 
				 current_renaming->node_data->list_of_free_variables);
      /* As a result of processing this renaming statement, we have made the set
	 of free variables on the LHS and the RHS the same.  So there are no more
	 extra free variables right now. */
      current_renaming->node_data->list_of_free_variables = NULL;
      
      /* Move on to the next renaming clause */
      current_renaming = current_renaming->next;
    }

    /* Now we simply add all the new equivalence axioms (corresponding to the renaming
       statements) to the beginning of the module_axioms. */
    new_equivalence_axioms_last->next = module->module_axioms;
    module->module_axioms = new_equivalence_axioms_first;

  }
  
}

/* For debugging */
void print_module(FILE * stream, struct mad_module * module) {

  if ( module != NULL ) {
    if ( module->name!= NULL ) {
      fprintf(stream, "***************************\n");
      fprintf(stream, "Module %s\n", module->name);
      fprintf(stream, "***************************\n");
    }
    else {
      fprintf(stream, "***************************\n");
      fprintf(stream,"Module with NULL name!\n");
      fprintf(stream, "***************************\n");
    }
      
    if ( module->module_sorts != NULL )
      print_sort_list(stream, module->module_sorts);
    if ( module->module_inclusions != NULL )
      print_inclusion_list(stream, module->module_inclusions);
    if ( module->module_objects != NULL )
      print_object_list(stream, module->module_objects);
    if ( module->module_constants != NULL )
      print_constant_list(stream, module->module_constants);
    if ( module->module_variables != NULL )
      print_variable_list(stream, module->module_variables);
    if ( module->module_axioms != NULL )
      print_axiom_list(stream, module->module_axioms);
    if ( module->module_renamings != NULL )
      print_renaming_list(stream, module->module_renamings);
      
  }
}

/* For debugging.
   It prints out the internal lists of the module which is 
   being parsed.  This may be incomplete due to an error 
   during parsing.
*/
void print_current_internal_lists(FILE * stream) {
  if ( sort_list_first != NULL )
    print_current_sorts(stream);  
  if ( inclusion_list_first != NULL )
    print_current_inclusions(stream);  
  if ( object_list_first != NULL )
    print_current_objects(stream);
  if ( constant_list_first != NULL )
    print_current_constants(stream);  
  if ( variable_list_first != NULL )
    print_current_variables(stream);
  if ( axiom_list_first != NULL )
    print_current_axioms(stream);
  if ( renaming_list_first != NULL )
    print_current_renamings(stream);
}

/* for debugging */
void print_current_modules(FILE * stream) {

  struct module_list_node * current;
  int counter;

  counter = 1;
  fprintf(stream, "Modules:\n");
  current = module_list_first;
  while ( current != NULL ) {
    if ( current->node_data != NULL ) {
      fprintf(stream, "\n", counter);
      fprintf(stream, "Module %d:\n", counter);
      
      print_module(stream, current->node_data);
      
      counter++;
    }
    current = current->next;
  }
  
}


/****** Managing lists of ground_actions **************/

/* initialize the lists of ground actions */
void init_ground_actions() {
  ground_action_list_first = ground_action_list_last = NULL;
}

void init_ground_explicit_actions() {
  ground_explicit_action_list_first = ground_explicit_action_list_last = NULL;
}

/* Called when inserting a new ground action.  Makes a new ground action */
struct mad_ground_action * make_new_ground_action (char * name,
						   struct string_list_node * arguments,
						   int num_arguments,
						   char * value) {
  struct mad_ground_action * new_ground_action ;

  new_ground_action = (struct mad_ground_action *) malloc( sizeof(struct mad_ground_action) );
  new_ground_action->name = (char *) malloc( sizeof(char) * ( strlen(name) + 1 ) );
  strcpy( new_ground_action->name, name );
  
  new_ground_action->arguments = arguments;
  new_ground_action->num_arguments = num_arguments;
  //new_ground_action->value = value;
  new_ground_action->value = NULL;

  return new_ground_action;
}

/* This inserts a new ground action into a list of ground actions. */
void insert_ground_action (char * name,
			   struct string_list_node * arguments,
			   int num_arguments,
			   char * value) {

  struct ground_action_list_node * new_node;

  new_node = (struct ground_action_list_node *) malloc( sizeof(struct ground_action_list_node) );
  new_node->node_data = make_new_ground_action(name, arguments, num_arguments, value);
  new_node->next = NULL;
  
  /* ground_action_list_last is the pointer to the end of the list */
  if ( ground_action_list_last == NULL ) { /* the list is empty */
    ground_action_list_first = new_node;
    ground_action_list_last = new_node;
  }
  else {
    ground_action_list_last->next = new_node;
    ground_action_list_last = new_node;
  }
}

void insert_ground_explicit_action (char * name, 
				    struct string_list_node * arguments,
				    int num_arguments,
				    char * value) {

  struct ground_action_list_node * new_node;

  new_node = (struct ground_action_list_node *) malloc( sizeof(struct ground_action_list_node) );
  new_node->node_data = make_new_ground_action(name, arguments, num_arguments, value);
  new_node->next = NULL;
  
  /* ground_explicit_action_list_last is the pointer to the end of the list */
  if ( ground_explicit_action_list_last == NULL ) { /* the list is empty */
    ground_explicit_action_list_first = new_node;
    ground_explicit_action_list_last = new_node;
  }
  else {
    ground_explicit_action_list_last->next = new_node;
    ground_explicit_action_list_last = new_node;
  }
}

/* for debugging */
void print_ground_actions(FILE * stream) {
  struct ground_action_list_node * current_ground_action;
  int counter;
  struct string_list_node * current_arg;

  fprintf(stream, "There are %d ground actions.\n", num_ground_actions);
  counter = 1;
  fprintf(stream, "Ground Actions:\n");
  current_ground_action = ground_action_list_first;
  while ( current_ground_action != NULL ) {

    /* print name first */
    fprintf(stream, "%d: %s", counter, current_ground_action->node_data->name);

    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current_ground_action->node_data->num_arguments > 0) {
      
      fprintf(stream, "(");
      
      /* start with the first argument in the list */
      current_arg = current_ground_action->node_data->arguments;
      fprintf(stream, "%s", current_arg->node_data);
      current_arg = current_arg->next;
      
      while (current_arg != NULL) {
	fprintf(stream, ",%s", current_arg->node_data);
	current_arg = current_arg->next;
      }
      
      fprintf(stream, ")");
    }
    
    /* print the value */
    //fprintf(stream, "=%s", current_ground_action->node_data->value);

    fprintf(stream, "\n");

    counter++;
    current_ground_action = current_ground_action->next;
  }
}

void print_ground_explicit_actions(FILE * stream) {
  struct ground_action_list_node * current_ground_action;
  int counter;
  struct string_list_node * current_arg;
  
  fprintf(stream, "There are %d ground actions.\n", num_ground_explicit_actions);
  counter = 1;
  fprintf(stream, "Ground Explicit Actions:\n");
  current_ground_action = ground_explicit_action_list_first;
  while ( current_ground_action != NULL ) {

    /* print name first */
    fprintf(stream, "%d: %s", counter, current_ground_action->node_data->name);

    /* print arguments if there are any.
       This assumes that all arguments are in a linked
       list of character strings */
    if (current_ground_action->node_data->num_arguments > 0) {
      
      fprintf(stream, "(");
      
      /* start with the first argument in the list */
      current_arg = current_ground_action->node_data->arguments;
      fprintf(stream, "%s", current_arg->node_data);
      current_arg = current_arg->next;
      
      while (current_arg != NULL) {
	fprintf(stream, ",%s", current_arg->node_data);
	current_arg = current_arg->next;
      }
      
      fprintf(stream, ")");
    }
    
    /* print the value */
    //fprintf(stream, "=%s", current_ground_action->node_data->value);

    fprintf(stream,"\n");

    counter++;
    current_ground_action = current_ground_action->next;
  }
}



/****** Function declarations for trees **************/

/* These aren't used yet.............. */

/* initialize the list of identifiers */
void init_tree() {
  root = NULL;
}


/****** Function declarations for after parsing ******/

/* Checks if the identifier conversion for ccalc will
   lead to multiple identifiers mapping to the same string
   If so, a warning is given */
void check_module_for_ccalc_identifier_clashes(struct mad_module * module) {

  struct string_list_node * module_ids_first;
  struct string_list_node * module_ids_last;

  struct string_list_node * module_ids_ccalc_first;
  struct string_list_node * module_ids_ccalc_last;

  struct string_list_node * current_id;
  struct string_list_node * current_id2;
  struct string_list_node * current_ccalc_id;
  struct string_list_node * current_ccalc_id2;

  struct sort_list_node * current_sort;
  struct object_list_node * current_object;
  struct constant_list_node * current_constant;
  struct variable_list_node * current_variable;

  int clash_seen_yet;

  // make a list of the identifiers in the module in ccalc form

  init_string_list(&module_ids_first, &module_ids_last);
  
  current_sort = module->module_sorts;
  while (current_sort != NULL ) {
    // add this to the list of identifiers
    if ( !is_an_integer_range(current_sort->node_data->name) ) {
      insert_string_into_list(&module_ids_first, 
			      &module_ids_last,
			      current_sort->node_data->name);
    }
    current_sort = current_sort->next;
  }

  current_object = module->module_objects;
  while (current_object != NULL ) {
    // add this to the list of identifiers
    insert_string_into_list(&module_ids_first, 
			    &module_ids_last,
			    current_object->node_data->name);
    current_object = current_object->next;
  }

  current_constant = module->module_constants;
  while (current_constant != NULL ) {
    // add this to the list of identifiers
    insert_string_into_list(&module_ids_first, 
			    &module_ids_last,
			    current_constant->node_data->name);
    current_constant = current_constant->next;
  }

  current_variable = module->module_variables;
  while (current_variable != NULL ) {
    // add this to the list of identifiers
    insert_string_into_list(&module_ids_first, 
			    &module_ids_last,
			    current_variable->node_data->name);
    current_variable = current_variable->next;
  }


  // Make a parallel list with ccalc-converted versions of ids
  // to avoid doing the ccalc-conversion every time we compare
  init_string_list(&module_ids_ccalc_first, &module_ids_ccalc_last);
  current_id = module_ids_first;
  while ( current_id != NULL ) {
    insert_string_into_list(&module_ids_ccalc_first, 
			    &module_ids_ccalc_last,
			    prepare_string_for_ccalc(current_id->node_data));
    current_id = current_id->next;
  }

  // Now go down list of modules again and see if there are any clashes
  current_id = module_ids_first;
  current_ccalc_id = module_ids_ccalc_first;
  while ( current_id != NULL ) {
    clash_seen_yet = 0;
    current_id2 = current_id->next;
    current_ccalc_id2 = current_ccalc_id->next;
    while ( current_id2 != NULL ) {
      if (strcmp(current_ccalc_id->node_data, current_ccalc_id2->node_data) == 0) {
	if ( !clash_seen_yet ) {
	  fprintf(stderr, 
		  "WARNING: Identifiers %s, %s", 
		  current_id->node_data,
		  current_id2->node_data);
	  clash_seen_yet = 1;
	}
	else {
	  fprintf(stderr, ", %s", current_id2->node_data);
	}
      }
      current_id2 = current_id2->next;
      current_ccalc_id2 = current_ccalc_id2->next;
    }
    if (clash_seen_yet) {
      // finish printing output
      fprintf(stderr, 
	      " turn into %s when converted to CCalc format\n",
	      current_ccalc_id->node_data);
    }
    current_id = current_id->next;
    current_ccalc_id = current_ccalc_id->next;
  }

}

/* Prints a ccalc command to show only non-renamed fluents
   in a module
   (i.e. those that are not of the form In.xxx)
 */
void print_ccalc_show_command_for_explicit_constants(FILE * stream, struct mad_module * module) {

  /* We want to print something like:
%:- show
%    location(i30dotx)=i30dotp;
%    support(i33doti17doti7doty)=i33doti17doti7dots;
%    supported(i33doti17doti7doty,i33doti17doti7dots)=i30doti29doti11doti0dotboolean_var0;
%    top(i30dotx)=i30dotp;
%    elevation(i33dotx)=i33dotp;
%    toplevel(i33dotx)=i33dotp;
%    at(i33dotf,i33dotp)=i30doti29doti11doti0dotboolean_var0.
*/

  struct constant_list_node * current_constant;
  struct string_list_node * current_arg;

  // When we have a constant with arguments, we need to write the show command
  // using variables.  In order to make sure we have a declared variable of the 
  // right sort, we will make new variables.
  
  // This list will hold the new variables created for each constant
  struct string_list_node * new_variable_list_first;
  struct string_list_node * new_variable_list_last;

  // This list will hold the sorts of the new variables created for each constant
  // It is needed in order to have a proper variable declaration.
  struct string_list_node * new_variable_sort_list_first;
  struct string_list_node * new_variable_sort_list_last;

  // To loop through the lists of new variables and their sorts
  struct string_list_node * current_variable;
  struct string_list_node * current_variable_sort;

  // To keep track of whether we have printed any constants in a show command yet
  int first_constant_printed;

  // initialize list of variables and corresponding sorts
  init_string_list(&new_variable_list_first, &new_variable_list_last);
  init_string_list(&new_variable_sort_list_first, &new_variable_sort_list_last);

  current_constant = module->module_constants;
  while ( current_constant != NULL ) {
    if ( !string_begins_with_an_import_prefix(current_constant->node_data->name) 
	 && strcmp(current_constant->node_data->name, "Actor")!=0
	 && strcmp(current_constant->node_data->name, "Theme")!=0 ) {
	 //	 && current_constant->node_data->kind != ACTION_CONST) {

      // See what the arguments are
      // This assumes that all arguments are in a linked
      // list of character strings      
      if (current_constant->node_data->num_arguments > 0) {
	
	// start with the first argument in the list
	current_arg = current_constant->node_data->arguments;

	while (current_arg != NULL) {
	  if (strcmp(current_arg->node_data, "explicitAction") == 0 ) {
	    // !!!: CCalc doesn't have explicit actions so we need to use
	    //   "action" instead of "explicitaction"
	    insert_string_into_list(&new_variable_list_first, 
				    &new_variable_list_last,
				    make_new_variable_name("action"));
	    insert_string_into_list(&new_variable_sort_list_first, 
				    &new_variable_sort_list_last,
				    "action");
	  }  
	  else {
	    insert_string_into_list(&new_variable_list_first, 
				    &new_variable_list_last,
				    make_new_variable_name(current_arg->node_data));
	    insert_string_into_list(&new_variable_sort_list_first, 
				    &new_variable_sort_list_last,
				    current_arg->node_data);
	  }
	  
	  current_arg = current_arg->next;
	}
      }
      // We also need a variable for the domain of the constant
      insert_string_into_list(&new_variable_list_first, 
			      &new_variable_list_last,
			      make_new_variable_name(current_constant->node_data->domain));
      insert_string_into_list(&new_variable_sort_list_first, 
			      &new_variable_sort_list_last,
			      current_constant->node_data->domain);

    }
    current_constant = current_constant->next;
  }

  // Now we have a list of all the new variables needed for specifying constants 
  // in the show command

  // Print declarations of variables needed for show command
  current_variable = new_variable_list_first;
  current_variable_sort = new_variable_sort_list_first;
  if ( current_variable != NULL ) {
    fprintf(stream, ":- variables\n");
    fprintf(stream, "    %s :: %s",
	    prepare_string_for_ccalc(current_variable->node_data),
	    prepare_string_for_ccalc(current_variable_sort->node_data) );
    current_variable = current_variable->next; 
    current_variable_sort = current_variable_sort->next; 
    while ( current_variable != NULL ) {
      fprintf(stream, ";\n");
      fprintf(stream, "    %s :: %s",
	      prepare_string_for_ccalc(current_variable->node_data),
	      prepare_string_for_ccalc(current_variable_sort->node_data) );
      current_variable = current_variable->next; 
      current_variable_sort = current_variable_sort->next; 
    }
    fprintf(stream, ".\n");
  }

  // Now that we have the necessary variables declared,
  // we can print the show command for all constants

  fprintf(stream, "\n");

  current_constant = module->module_constants;
  if ( current_constant != NULL ) {
    fprintf(stream, ":- show");
  
    first_constant_printed = 0;
    // We need to reset the pointer to the list of variables which will be used
    // in the show command
    current_variable = new_variable_list_first;
    
    while ( current_constant != NULL ) {
      if ( !string_begins_with_an_import_prefix(current_constant->node_data->name)
	   && strcmp(current_constant->node_data->name, "Actor")!=0
	   && strcmp(current_constant->node_data->name, "Theme")!=0 ) {
	//	 && current_constant->node_data->kind != ACTION_CONST) {
	
	// We will print a "; after each constant, unless it is the last
	if ( first_constant_printed != 1 ) {
	  fprintf(stream, "\n  %s", prepare_string_for_ccalc(current_constant->node_data->name));
	  first_constant_printed = 1;
	}
	else {
	  fprintf(stream, ";\n  %s", prepare_string_for_ccalc(current_constant->node_data->name));
	}
	
	if (current_constant->node_data->num_arguments > 0) {
	  fprintf(stream, "(");
	  
	  current_arg = current_constant->node_data->arguments;
	  if ( current_arg != NULL ) {
	    fprintf(stream, "%s", prepare_string_for_ccalc(current_variable->node_data));
	    current_arg = current_arg->next;
	    // Also need to advance the variables to be printed
	    current_variable = current_variable->next; 
	  }
	  while (current_arg != NULL) {
	    fprintf(stream, ", %s", prepare_string_for_ccalc(current_variable->node_data));
	    current_arg = current_arg->next;
	    // Also need to advance the variables to be printed
	    current_variable = current_variable->next; 
	  }
	  
	  fprintf(stream, ")");
	}
	// Print variable for the domain of the constant.
	fprintf(stream, "=%s", prepare_string_for_ccalc(current_variable->node_data));
	// Also need to advance the variables to be printed
	current_variable = current_variable->next; 
      }
      current_constant = current_constant->next;
    }
    fprintf(stream, ".\n");
  }
}


/* Prints Ccalc input corresponding to a module */
void print_module_as_ccalc_input(FILE * stream, struct mad_module * module) {

  // We first check if the identifier conversion for ccalc will
  // lead to multiple identifiers mapping to the same string
  // If so, a warning will be given
  check_module_for_ccalc_identifier_clashes(module);

  //fprintf(stream, "CCalc input corresponding to module %s:\n", module->name);
  // print_sort_and_inclusion_list_for_ccalc(stream,
  //                                         module->module_sorts,
  //	       			             module->module_inclusions);
  print_sort_list_for_ccalc(stream, module->module_sorts);
  fprintf(stream, "\n");
  print_inclusion_list_for_ccalc(stream, module->module_inclusions);
  fprintf(stream, "\n");
  print_integer_range_object_list_for_ccalc(stream, module->module_sorts);
  fprintf(stream, "\n");
  print_object_list_for_ccalc(stream, module->module_objects);
  fprintf(stream, "\n");
  print_constant_list_for_ccalc(stream, module->module_constants);
  fprintf(stream, "\n");
  //print_variable_list_for_ccalc(stream, module->module_variables);
  print_non_explicit_action_variable_list_for_ccalc(stream, 
						    module->module_variables);
  fprintf(stream, "\n");
  print_axiom_list_for_ccalc(stream, module->module_axioms);
  fprintf(stream, "\n");
  print_ccalc_show_command_for_explicit_constants(stream, module);
}


/* Takes a renaming list and returns a list which is the reverse of the first,
   The actual renaming data is not copied, it is simply re-used. */
struct renaming_list_node * reverse_renaming_list(struct renaming_list_node * list) {
  
  struct renaming_list_node * current;
  struct renaming_list_node * new_node;
  struct renaming_list_node * last_added_node; /* The node we added last, to the reversed list */

  new_node = NULL;
  current = list;
  if (current != NULL) {
    new_node = (struct renaming_list_node *) malloc( sizeof(struct renaming_list_node) );
    new_node->node_data = current->node_data;
    new_node->next = NULL;
    last_added_node = new_node;
    current = current->next;
  }
  while ( current != NULL ) {
    new_node = (struct renaming_list_node *) malloc( sizeof(struct renaming_list_node) );
    new_node->node_data = current->node_data;
    new_node->next = last_added_node;
    last_added_node = new_node;
    current = current->next;
  }
  return new_node;
}


/* Given a string_list of variables, it creates a parallel
   list by making new variables of the same sorts, and
   returns a substitution to be used for replacing the old 
   with the new.

   This is called when we want to standardize apart before
   unification, to rename some variables.
*/
struct mad_substitution * get_substitution_to_replace_variables_by_new_variables(struct string_list_node * variables_to_be_replaced) {
  
  struct mad_substitution * subst;
  
  struct string_list_node * current_var_name;

  struct variable_list_node * current_variable;

  char * new_argument_variable;

  struct mad_constant_argument * old_args;
  struct mad_constant_argument * new_args;

  struct mad_constant_argument * current_old_args;
  struct mad_constant_argument * current_new_args;


  subst = (struct mad_substitution *) malloc( sizeof(struct mad_substitution) );
  subst->old_arguments = NULL;
  subst->new_arguments = NULL;

  current_old_args = NULL;
  current_new_args = NULL;

  current_var_name = variables_to_be_replaced;
  while ( current_var_name != NULL ) {
    current_variable = lookup_variable(current_var_name->node_data);
    if ( current_variable == NULL ) {
      print_error_message(stderr, "Internal error: Variable to be substituted is not declared.\n");
      break;
    }
    else {
      new_argument_variable = make_new_variable_name(current_variable->node_data->sort);
      /* Add this new variable to the list of identifiers and to the list of variables */
      insert_identifier(new_argument_variable);
      /* update the list of identifiers to indicate this
	 identifier is a variable */
      update_id_type(new_argument_variable, VARIABLE_ID);
      insert_variable_into_module(working_module, 
				  new_argument_variable, 
				  current_variable->node_data->sort);
      if ( current_old_args == NULL ) {
	current_old_args = make_new_argument(PLAIN_ARGUMENT, current_var_name->node_data);
	current_new_args = make_new_argument(PLAIN_ARGUMENT, new_argument_variable);
	subst->old_arguments = current_old_args;
	subst->new_arguments = current_new_args;
      }
      else {
	current_old_args->next_argument = make_new_argument(PLAIN_ARGUMENT, current_var_name->node_data);
	current_new_args->next_argument = make_new_argument(PLAIN_ARGUMENT, new_argument_variable);
	current_old_args = current_old_args->next_argument;
	current_new_args = current_new_args->next_argument;
      }
    }
    current_var_name = current_var_name->next;
  }

  return subst;
}


/* Takes two lists of arguments, which are the same length, and
   sees if it can unify them.  If it can, it will return 1 and
   update the pointer to a substitution containing two argument 
   structures which allow unification with maximal overlap.
*/
int unify_argument_lists(struct mad_constant_argument * arguments1, 
			 struct mad_constant_argument * arguments2,
			 struct mad_substitution ** subst_ptr) {

  struct object_list_node * arguments1_object;
  struct object_list_node * arguments2_object;
  struct variable_list_node * arguments1_variable;
  struct variable_list_node * arguments2_variable;

  struct mad_constant_argument * old_argument_to_be_added;
  struct mad_constant_argument * new_argument_to_be_added;

  char * common_subsort_name;

  struct sort_list_node * current_sort;

  char * new_argument_variable;

  // To deal with integer ranges
  int lo1, hi1, lo2, hi2;

  /* We compare each argument from the two argument structures.  */
  arguments1 = arguments1;
  arguments2 = arguments2;  
  if ( arguments1 != NULL && arguments2 != NULL ) {

    arguments1_object = lookup_object(arguments1->name);
    arguments2_object = lookup_object(arguments2->name);
    arguments1_variable = lookup_variable(arguments1->name);
    arguments2_variable = lookup_variable(arguments2->name);

    if ( arguments1_object != NULL
	 && arguments2_object != NULL ) {
      // The arguments have to have the same name in order to unify
      if ( strcmp(arguments1->name, arguments2->name) != 0 ) {
	return 0;
      }
      else {
	if ( unify_argument_lists(arguments1->arguments,
				  arguments2->arguments,
				  subst_ptr) ) {
	  return unify_argument_lists(arguments1->next_argument,
				      arguments2->next_argument,
				      subst_ptr);
	}
	else {
	  return 0;
	}
      }

    }
    else if ( arguments1_variable != NULL
	      && arguments2_object != NULL ) {
      /* The first argument is a variable and the second an object.  They match only if
	 the object belongs to the sort of the variable. */
      if ( strcmp(arguments2_object->node_data->sort,
		  arguments1_variable->node_data->sort) == 0 
	   || (is_a_known_object_sort(arguments2_object->node_data->sort)
	       && is_a_known_variable_sort(arguments1_variable->node_data->sort)
	       && is_a_subsort_of(arguments2_object->node_data->sort,
				  arguments1_variable->node_data->sort) ) ) {
	// Add var/obj to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments2);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments1);
	  new_argument_to_be_added = copy_single_argument(arguments2);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else if ( arguments2_variable != NULL
	      && arguments1_object != NULL ) {
      /* This is symmetrical to the previous case.  

	 The second argument is a variable and the first an object.  They match only if
	 the object belongs to the sort of the variable. */
      if ( strcmp(arguments1_object->node_data->sort,
		  arguments2_variable->node_data->sort) == 0 
	   || (is_a_known_object_sort(arguments1_object->node_data->sort)
	       && is_a_known_variable_sort(arguments2_variable->node_data->sort)
	       && is_a_subsort_of(arguments1_object->node_data->sort,
				  arguments2_variable->node_data->sort) ) ) {
	// Add var/obj to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments2);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments1);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments2);
	  new_argument_to_be_added = copy_single_argument(arguments1);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else if ( arguments1_variable != NULL
	      && arguments2_variable != NULL ) {
      /* Both arguments are variables.  They can be unified only if they 
	 are of the same sort, one's sort is a subsort of the other's,
	 or if they have a common subsort. */
      if ( strcmp(arguments1_variable->node_data->sort,
		  arguments2_variable->node_data->sort) == 0 ) {
	// if they have the same sort, just substitute the first with the second
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments2);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments1);
	  new_argument_to_be_added = copy_single_argument(arguments2);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else if ( is_a_known_variable_sort(arguments1_variable->node_data->sort)
		&& is_a_known_variable_sort(arguments2_variable->node_data->sort)
		&& is_a_subsort_of(arguments1_variable->node_data->sort,
				   arguments2_variable->node_data->sort) ) {
	// if the first's sort is a subsort of the second's, 
	// then substitute the second with the first
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments2);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments1);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments2);
	  new_argument_to_be_added = copy_single_argument(arguments1);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else if ( is_a_known_variable_sort(arguments1_variable->node_data->sort)
		&& is_a_known_variable_sort(arguments2_variable->node_data->sort)
		&& is_a_subsort_of(arguments2_variable->node_data->sort,
				   arguments1_variable->node_data->sort) ) {
	// if the second's sort is a subsort of the first's, 
	// then substitute the first with the second
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments2);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments1);
	  new_argument_to_be_added = copy_single_argument(arguments2);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	// Even if one variable is not a subsort of the other, if they have a 
	// common subsort, then they can be unified.
	// We need to find the maximal common subsort
	common_subsort_name = NULL;
	// Try "Boolean" first"
	if ( is_a_known_variable_sort(arguments1_variable->node_data->sort)
	     && is_a_known_variable_sort(arguments2_variable->node_data->sort)
	     && is_a_subsort_of("Boolean",
				arguments1_variable->node_data->sort) 
	     && is_a_subsort_of("Boolean",
				arguments2_variable->node_data->sort) ) {
	  // Ok, we found common subsort.  Update it, if it's more general than
	  // what we found so far.
	  if ( common_subsort_name == NULL
	       || is_a_subsort_of(common_subsort_name, "Boolean") ) {
	    common_subsort_name = "Boolean";
	    }
	}
	current_sort = sort_list_first;
	while ( current_sort != NULL ) {
	  if ( is_a_known_variable_sort(current_sort->node_data->name)
	       && is_a_known_variable_sort(arguments1_variable->node_data->sort)
	       && is_a_known_variable_sort(arguments2_variable->node_data->sort)
	       && is_a_subsort_of(current_sort->node_data->name,
				  arguments1_variable->node_data->sort) 
	       && is_a_subsort_of(current_sort->node_data->name,
				  arguments2_variable->node_data->sort) ) {
	    // Ok, we found common subsort.  Update it, if it's more general than
	    // what we found so far.
	    if ( common_subsort_name == NULL
		 || is_a_subsort_of(common_subsort_name,
				    current_sort->node_data->name) ) {
	      common_subsort_name = current_sort->node_data->name;
	    }
	  }
	  current_sort = current_sort->next;
	}
	// The two variables belong to different sorts and we have looked for
	// a common subsort among the list of declared sorts, 
	// but if they are both integer ranges, then their intersection is
	// their common subsort
	if ( is_an_integer_range(arguments1_variable->node_data->sort)
	     && is_an_integer_range(arguments2_variable->node_data->sort) ) {
	  lo1 = lower_integer_of_range(arguments1_variable->node_data->sort);
	  hi1 = upper_integer_of_range(arguments1_variable->node_data->sort);
	  lo2 = lower_integer_of_range(arguments2_variable->node_data->sort);
	  hi2 = upper_integer_of_range(arguments2_variable->node_data->sort);
	  // First determine which range begins "earlier", 
	  // i.e., has a smaller lower integer
	  if ( lo1 <= lo2 ) {
	    // first range begins earlier,
	    if ( hi1 >= lo2 ) {
	      // They overlap
	      if ( hi1 >= hi2 ) {
		// first range includes second range completely
		common_subsort_name = arguments2_variable->node_data->sort;
	      }
	      else {
		// the overlap is from the lo2 to hi1
		common_subsort_name
		  = (char *) malloc( sizeof(char) * (strlen(integer_to_string(lo2)) 
						     + strlen(integer_to_string(hi1)) 
						     + 2 // for ".."
						     + 1 ));
		strcpy(common_subsort_name, integer_to_string(lo2));
		strcat(common_subsort_name, "..");
		strcat(common_subsort_name, integer_to_string(hi1));
	      }
	    }
	  }
	  else { // second range begins earlier
	    if ( hi2 >= lo1 ) {
	      // They overlap
	      if ( hi2 >= hi1 ) {
		// second range includes first range completely
		common_subsort_name = arguments1_variable->node_data->sort;
	      }
	      else {
		// the overlap is from the lo1 to hi2
		common_subsort_name
		  = (char *) malloc( sizeof(char) * (strlen(integer_to_string(lo1)) 
						     + strlen(integer_to_string(hi2)) 
						     + 2 // for ".."
						     + 1 ));
		strcpy(common_subsort_name, integer_to_string(lo1));
		strcat(common_subsort_name, "..");
		strcat(common_subsort_name, integer_to_string(hi2));
	      }
	    }
	  }
	}
	if ( common_subsort_name != NULL ) {
	  // We need to make a new variable of this common subsort
	  // and insert substitutions to replace both the first and the second
	  // argument with this new variable
	  new_argument_variable = make_new_variable_name(common_subsort_name);
	  /* Add this new variable to the list of identifiers and to the list of variables */
	  insert_identifier(new_argument_variable);
	  /* update the list of identifiers to indicate this
	     identifier is a variable */
	  update_id_type(new_argument_variable, VARIABLE_ID);
	  insert_variable_into_module(working_module, 
				      new_argument_variable, 
				      common_subsort_name);
	  
	  if ( (*subst_ptr)->old_arguments == NULL ) {
	    (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	    (*subst_ptr)->new_arguments = make_new_argument(PLAIN_ARGUMENT,
							    new_argument_variable);
	    (*subst_ptr)->old_arguments->next_argument = copy_single_argument(arguments2);
	    (*subst_ptr)->new_arguments->next_argument = make_new_argument(PLAIN_ARGUMENT,
									   new_argument_variable);
	  }
	  else {
	    // We insert these two new pairs at the beginning, not at the end
	    old_argument_to_be_added = copy_single_argument(arguments1);
	    old_argument_to_be_added->next_argument = copy_single_argument(arguments2);
	    new_argument_to_be_added = make_new_argument(PLAIN_ARGUMENT,
							 new_argument_variable);
	    new_argument_to_be_added->next_argument = make_new_argument(PLAIN_ARGUMENT,
									new_argument_variable);
	    old_argument_to_be_added->next_argument->next_argument = (*subst_ptr)->old_arguments;
	    new_argument_to_be_added->next_argument->next_argument = (*subst_ptr)->new_arguments;
	    (*subst_ptr)->old_arguments = old_argument_to_be_added;
	    (*subst_ptr)->new_arguments = new_argument_to_be_added;
	  }
	  return unify_argument_lists(arguments1->next_argument,
				      arguments2->next_argument,
				      subst_ptr);
	}
	else {
	  return 0;
	}
      }
    }
    else if ( (strcmp(arguments1->name, "true") == 0
	       || strcmp(arguments1->name, "false") == 0) 
	      && (strcmp(arguments2->name, "true") == 0
		  || strcmp(arguments2->name, "false") == 0) ) {
      /* The arguments are both true/false so they unify if and only if they are the same. */
      if ( strcmp(arguments1->name,
		  arguments2->name) == 0 ) {
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else if ( (strcmp(arguments1->name, "true") == 0
	       || strcmp(arguments1->name, "false") == 0) 
	      && arguments2_variable != NULL ) {
      /* The first argument is true/false and the second a variable.  
	 They unify only if the sort of the variable is Boolean or
	 a supersort of Boolean. */
      if ( strcmp("Boolean",
		  arguments2_variable->node_data->sort) == 0 
	   || (is_a_known_variable_sort(arguments2_variable->node_data->sort)
	       && is_a_subsort_of("Boolean", arguments2_variable->node_data->sort) ) ) {
	// Add var/"true" or var/"false" to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments2);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments1);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments2);
	  new_argument_to_be_added = copy_single_argument(arguments1);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else { /* The two arguments didn't match, so the argument lists are not unifiable */
	return 0;
      }
    }
    else if ( (strcmp(arguments2->name, "true") == 0
	       || strcmp(arguments2->name, "false") == 0) 
	      && arguments1_variable != NULL ) {
      /* The second argument is true/false and the first a variable.  
	 They unify only if the sort of the variable is Boolean or
	 a supersort of Boolean. */
      if ( strcmp("Boolean",
		  arguments1_variable->node_data->sort) == 0 
	   || (is_a_known_variable_sort(arguments1_variable->node_data->sort)
	       && is_a_subsort_of("Boolean", arguments1_variable->node_data->sort) ) ) {
	// Add var/"true" or var/"false" to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments2);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments1);
	  new_argument_to_be_added = copy_single_argument(arguments2);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else { /* The two arguments didn't match, so the argument lists are not unifiable */
	return 0;
      }
    }
    else if ( is_an_integer(arguments1->name)
	      && is_an_integer(arguments2->name) ) {
      // Integers unify only if they have the same value
      if ( atoi(arguments1->name) == atoi(arguments2->name) ) {
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else if ( is_an_integer(arguments1->name)
	      && arguments2_variable != NULL ) {
      /* The first argument is an integer and the second a variable.  
	 They unify only if the sort of the variable is an integer range which
	 contains the first argument. */
      if ( is_an_integer_range(arguments2_variable->node_data->sort)
	   && (lower_integer_of_range(arguments2_variable->node_data->sort)
	       <= atoi(arguments1->name))
	   && (upper_integer_of_range(arguments2_variable->node_data->sort)
	       >= atoi(arguments1->name)) ) {
	// Add var/integer to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments2);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments1);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments2);
	  new_argument_to_be_added = copy_single_argument(arguments1);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else if ( is_an_integer(arguments2->name)
	      && arguments1_variable != NULL ) {
      /* The second argument is an integer and the first a variable.  
	 They unify only if the sort of the variable is an integer range which
	 contains the second argument. */
      if ( is_an_integer_range(arguments1_variable->node_data->sort)
	   && (lower_integer_of_range(arguments1_variable->node_data->sort)
	       <= atoi(arguments2->name))
	   && (upper_integer_of_range(arguments1_variable->node_data->sort)
	       >= atoi(arguments2->name)) ) {
	// Add var/integer to substitution
	if ( (*subst_ptr)->old_arguments == NULL ) {
	  (*subst_ptr)->old_arguments = copy_single_argument(arguments1);
	  (*subst_ptr)->new_arguments = copy_single_argument(arguments2);
	}
	else {
	  // We insert this new pair at the beginning, not at the end
	  old_argument_to_be_added = copy_single_argument(arguments1);
	  new_argument_to_be_added = copy_single_argument(arguments2);
	  old_argument_to_be_added->next_argument = (*subst_ptr)->old_arguments;
	  new_argument_to_be_added->next_argument = (*subst_ptr)->new_arguments;
	  (*subst_ptr)->old_arguments = old_argument_to_be_added;
	  (*subst_ptr)->new_arguments = new_argument_to_be_added;
	}
	return unify_argument_lists(arguments1->next_argument,
				    arguments2->next_argument,
				    subst_ptr);
      }
      else {
	return 0;
      }
    }
    else { /* At least one of the arguments is neither an object nor a variable nor true/false
	      nor an integer.
	      Assuming we got these arguments from validly formed formulas,
	      then one must be an action constant. */
      
      /* !!!: For now we will assume having action constants breaks unification. */
      return 0;

    }

  }
  else {
    // If they're not both non-Null, then they must both be NULL
    if ( arguments1 == NULL && arguments2 == NULL ) {
      return 1;
    }
    else {
      return 0;
    }
  }
  
}

/* Called when making an action description definite.
   It takes a list of arguments to be replaced and a list of arguments to replace the former,
   and an argument structure in which this replacement should be done.

   It is intended for situations in which the argument structure has arguments of its own.
   (i.e. it is an ARGUMENTED_ARGUMENT.)

   The replacement is done in parallel.
*/
struct mad_constant_argument * replace_arguments_with_arguments_in_argument(struct mad_constant_argument * arguments1, 
									    struct mad_constant_argument * arguments2,
									    struct mad_constant_argument * argument) {
  
  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct variable_list_node * argument_variable;
  
  struct mad_constant_argument * current_argument_argument;

  struct mad_constant_argument * current_argument1;
  struct mad_constant_argument * current_argument2;

  if ( argument != NULL ) {
          
    term_constant = lookup_constant(argument->name);
    term_object = lookup_object(argument->name);
    // if this argument isn't a constant or object, or an arithmetic operator,
    // there are no arguments nested further
    if ( term_constant != NULL 
	 || term_object != NULL 
	 || is_an_arithmetic_operator(argument->name) ) {
      
      current_argument_argument = argument->arguments;
      while ( current_argument_argument != NULL ) {
	argument_variable = lookup_variable(current_argument_argument->name);
	if ( argument_variable != NULL ) {
	  
	  current_argument1 = arguments1;
	  current_argument2 = arguments2;
	  while ( current_argument1 != NULL ) {
	    
	    if ( strcmp(current_argument_argument->name, current_argument1->name) == 0 ) {
	      if ( current_argument2->argument_kind == PLAIN_ARGUMENT ) {
		current_argument_argument->name = current_argument2->name;
	      }
	      else { /* current_argument2 is an action constant, so we need to replace whole current_argument_argument */
		current_argument_argument->name = current_argument2->name;
		current_argument_argument->arguments = current_argument2->arguments;
		/* This next field is normally only meant for when we have ground actions,
		   but we should set it here, just in case */
		if ( current_argument2->has_a_value == 1 ) {
		  current_argument_argument->value = current_argument2->value;
		}
	      }
	      
	      break;
	    }

	    current_argument1 = current_argument1->next_argument;
	    current_argument2 = current_argument2->next_argument;
	  }
	  
	}
	else {
	  /* The argument isn't the variable we are looking for,
	     but it might be a constant with variables in its argument */
	  current_argument_argument = replace_arguments_with_arguments_in_argument(arguments1,
										   arguments2,
										   current_argument_argument);
	}

	current_argument_argument = current_argument_argument->next_argument;
      }
    }
  }
  return argument;
}

/* Called when making an action description definite.
   It takes a list of arguments to be replaced and a list of arguments to replace the former,
   and a formula in which this replacement should be done.

   The replacement is done in parallel.
*/
struct mad_formula * replace_arguments_with_arguments_in_formula(struct mad_constant_argument * arguments1,
								 struct mad_constant_argument * arguments2,
								 struct mad_formula * formula) {
  
  struct variable_list_node * term_variable;
  struct variable_list_node * argument_variable;

  struct mad_constant_argument * current_formula_argument;

  struct mad_constant_argument * current_argument1;
  struct mad_constant_argument * current_argument2;

  struct mad_constant_argument * arguments_without_quantified_variable1;
  struct mad_constant_argument * arguments_without_quantified_variable2;
  
  struct mad_constant_argument * current_argument_without_quantified_variable1;
  struct mad_constant_argument * current_argument_without_quantified_variable2;


  if ( formula != NULL ) {

   /* If there is a left_formula, replace the argument in it. */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = replace_arguments_with_arguments_in_formula(arguments1,
									  arguments2,
									  formula->left_formula);
    }
    
    /* If it's an atomic formula, replace any instance of arguments1 with arguments2. */
    if ( formula->formula_kind == ATOMIC_FORMULA 
	 || formula->formula_kind == INEQUALITY ) {
      
      /* arg2 is an argument of the RHS of the renaming.  It can either be
	 an object or a variable.  If it is an object, it must match arg1, otherwise we
	 wouldn't be able to call replacement on the two constants.  
         So arg2 must be a variable for us to need to replace it. */

      term_variable = lookup_variable(formula->LHS_term);
      if ( term_variable != NULL ) {
	
	current_argument1 = arguments1;
	current_argument2 = arguments2;
	while ( current_argument1 != NULL ) {
	  
	  if ( strcmp(formula->LHS_term, current_argument1->name) == 0 ) {
	    if ( current_argument2->argument_kind == PLAIN_ARGUMENT ) {
	      formula->LHS_term = current_argument2->name;
	    }
	    else { /* current_argument2 is an action constant, so we need to replace whole LHS */
	      formula->LHS_term = current_argument2->name;
	      formula->LHS_arguments = current_argument2->arguments;
	      /* This next field is normally only meant for when we have ground actions,
		 but we should set it here, just in case */
	      if ( current_argument2->has_a_value == 1 ) {
		formula->LHS_value = current_argument2->value;
	      }
	    }

	    break;
	  }
	  
	  current_argument1 = current_argument1->next_argument;
	  current_argument2 = current_argument2->next_argument;
	}
      }
      else { /* The LHS term isn't the variable we are looking for 
		but it may be a constant with variables in the arguments */
	
	if (formula->LHS_arguments != NULL ) {
	  current_formula_argument = formula->LHS_arguments;
	  while ( current_formula_argument != NULL ) {
	    argument_variable = lookup_variable(current_formula_argument->name);
	    if ( argument_variable != NULL ) {
	      
	      current_argument1 = arguments1;
	      current_argument2 = arguments2;
	      while ( current_argument1 != NULL ) {
		
		if ( strcmp(current_formula_argument->name, current_argument1->name) == 0 ) {
		  if ( current_argument2->argument_kind == PLAIN_ARGUMENT ) {
		    current_formula_argument->name = current_argument2->name;
		  }
		  else { /* current_argument2 is an action constant, so we need to replace whole current_formula_argument */
		    current_formula_argument->name = current_argument2->name;
		    current_formula_argument->arguments = current_argument2->arguments;
		    /* This next field is normally only meant for when we have ground actions,
		       but we should set it here, just in case */
		    if ( current_argument2->has_a_value == 1 ) {
		      current_formula_argument->value = current_argument2->value;
		    }
		  }

		  break;
		}
		
		current_argument1 = current_argument1->next_argument;
		current_argument2 = current_argument2->next_argument;
	      }
	    }
	    else {
	      /* The argument isn't a variable, but it might be a constant with variables in its argument */
	      current_formula_argument = replace_arguments_with_arguments_in_argument(arguments1,
										      arguments2,
										      current_formula_argument);
	    }
	    current_formula_argument = current_formula_argument->next_argument;
	  }
	}
      }
      
      /* Now do the same for the RHS term */
      if ( formula->RHS_term != NULL ) {
	
	term_variable = lookup_variable(formula->RHS_term);
	if ( term_variable != NULL ) {
	  
	  current_argument1 = arguments1;
	  current_argument2 = arguments2;
	  while ( current_argument1 != NULL ) {
	    
	    if ( strcmp(formula->RHS_term, current_argument1->name) == 0 ) {
	      if ( current_argument2->argument_kind == PLAIN_ARGUMENT ) {
		formula->RHS_term = current_argument2->name;
	      }
	      else { /* current_argument2 is an action constant, so we need to replace whole RHS */
		formula->RHS_term = current_argument2->name;
		formula->RHS_arguments = current_argument2->arguments;
		/* This next field is normally only meant for when we have ground actions,
		   but we should set it here, just in case */
		if ( current_argument2->has_a_value == 1 ) {
		  formula->RHS_value = current_argument2->value;
		}
	      }
	      
	      break;
	    }
	    
	    current_argument1 = current_argument1->next_argument;
	    current_argument2 = current_argument2->next_argument;
	  }
	}
	else { /* The RHS term isn't the variable we are looking for 
		  but it may be a constant with variables in the arguments */
	  
	  if (formula->RHS_arguments != NULL ) {
	    current_formula_argument = formula->RHS_arguments;
	    while ( current_formula_argument != NULL ) {
	      argument_variable = lookup_variable(current_formula_argument->name);
	      if ( argument_variable != NULL ) {
		
		current_argument1 = arguments1;
		current_argument2 = arguments2;
		while ( current_argument1 != NULL ) {
		  
		  if ( strcmp(current_formula_argument->name, current_argument1->name) == 0 ) {
		    if ( current_argument2->argument_kind == PLAIN_ARGUMENT ) {
		      current_formula_argument->name = current_argument2->name;
		    }
		    else { /* current_argument2 is an action constant, so we need to replace whole current_formula_argument */
		      current_formula_argument->name = current_argument2->name;
		      current_formula_argument->arguments = current_argument2->arguments;
		      /* This next field is normally only meant for when we have ground actions,
			 but we should set it here, just in case */
		      if ( current_argument2->has_a_value == 1 ) {
			current_formula_argument->value = current_argument2->value;
		      }
		    }

		    break;
		  }
		  
		  current_argument1 = current_argument1->next_argument;
		  current_argument2 = current_argument2->next_argument;
		}
	      }
	      else {
		/* The argument isn't a variable, but it might be a constant with variables in its argument */
		current_formula_argument = replace_arguments_with_arguments_in_argument(arguments1,
											arguments2,
											current_formula_argument);
	      }
	      current_formula_argument = current_formula_argument->next_argument;
	    }
	  }
	}
      }
    }
    /* If it's a quantified formula, then replace instances of the arguments 
       only for those arguments different from the quantified variable.  (Because otherwise
       the variable won't be free.)
       e.g. the second conjunct in (F(x) & exists x G(x) doesn't have x free. */
    if ( formula->formula_kind == QUANTIFIER ) {
      /* If any of the arguments we are replacing is the same with the quantified
	 variable, then we need to skip over that. Therefore we now make two
	 new copies of the arguments to be replaced (arguments1) and those which
	 replace them (arguments2). */

      /* Initially the new lists are null */
      arguments_without_quantified_variable1 = arguments_without_quantified_variable2 = NULL;
      /* Since we haven't copied anything over yet, our current position in the new lists is also null. */
      current_argument_without_quantified_variable1 = NULL;
      current_argument_without_quantified_variable2 = NULL;

      /* We compare each argument from the two argument structures.  */
      current_argument1 = arguments1;
      current_argument2 = arguments2;  
      while ( current_argument1 != NULL ) {
	/* Check that the argument is different from the quantified variable.
	   If it is different, from the quantified variable, then carry it into the new lists. */
	if ( strcmp(formula->quantified_variable, current_argument1->name) != 0 ) {
	  if ( current_argument_without_quantified_variable1 == NULL ) {
	    current_argument_without_quantified_variable1 = copy_single_argument(current_argument1);
	    current_argument_without_quantified_variable2 = copy_single_argument(current_argument2);
	  }
	  else {
	    current_argument_without_quantified_variable1->next_argument = copy_single_argument(current_argument1);
	    current_argument_without_quantified_variable2->next_argument = copy_single_argument(current_argument2);
	    current_argument_without_quantified_variable1 = current_argument_without_quantified_variable1->next_argument;
	    current_argument_without_quantified_variable2 = current_argument_without_quantified_variable2->next_argument;
	  }
	  /* If this is the first argument to be carried over into the new lists, set the new lists now */
	  if ( arguments_without_quantified_variable1 == NULL ) {
	    arguments_without_quantified_variable1 = current_argument_without_quantified_variable1;
	    arguments_without_quantified_variable2 = current_argument_without_quantified_variable2;
	  }
	}

	current_argument1 = current_argument1->next_argument;
	current_argument2 = current_argument2->next_argument;
      }

      /* Now that we have our new lists ready, go ahead and replace arguments in the quantified right_formula */
      formula->right_formula = replace_arguments_with_arguments_in_formula(arguments_without_quantified_variable1, 
									   arguments_without_quantified_variable2,
									   formula->right_formula);	
    }
    /* If there is a right_formula, that is not attached to a quantifier,
       replace the variable in it. */
    else if ( formula->right_formula != NULL ) {
      formula->right_formula = replace_arguments_with_arguments_in_formula(arguments1, 
									   arguments2,
									   formula->right_formula);
    }
  }
  return formula;
}


/* Called when an action description is being made definite.
   It takes 

   - an axiom whose head is the LHS of an "is" statement,
   - a formula to replace that head (either the RHS formula of "is" or a case formula of the "is")
   - the condition upon which this replacement is valid (the conditon from the corresponding case of "is")

   and makes the replacement, returning the new axiom.
*/
struct mad_axiom * replace_formula_in_head_of_axiom_by_formula_upon_condition(struct mad_axiom * axiom,
									      struct mad_formula * formula,
									      struct mad_formula * condition) {
  struct mad_axiom * new_axiom;

  struct mad_formula * new_axiom_head;

  struct mad_formula * new_axiom_G_with_condition;

  struct mad_formula * true_formula; /* This'll be a formula with the 0-place connective TRUE */


  struct constant_list_node * axiom_head_constant;


  /* If the condition is not NULL, it must appear at the end of the new axiom.
     Since a condition does not contain any constants, it doesn't matter whether
     the causal law corresponding to the axiom contains the condition with
     timestamp i: or i+1:, as long as it appears in the body.  So we will put
     the condition in formula G of the axiom. */
  if ( condition != NULL ) {
    new_axiom_G_with_condition = make_new_formula(BINARY_CONNECTIVE);
    new_axiom_G_with_condition->connective = AND;
    new_axiom_G_with_condition->left_formula = axiom->G;
    new_axiom_G_with_condition->right_formula = condition;
    axiom->G = new_axiom_G_with_condition;
  }

  true_formula = make_new_formula(ATOMIC_FORMULA);
  true_formula->LHS_term = "true";
  
  
  if ( axiom->F->formula_kind != UNARY_CONNECTIVE ) {
    axiom_head_constant = lookup_constant(axiom->F->LHS_term);
    if ( strcmp(axiom_head_constant->node_data->domain, "Boolean") == 0 ) {
      /* The head we are replacing is a Boolean atom so 
	 there are 4 possibilities for axiom->F:
	 c, c=true, c=false, c=var */
      if ( axiom->F->RHS_term == NULL 
	   || strcmp(axiom->F->RHS_term, "true") == 0 ) {
	/* There are 6 possibilities for formula:
	   c, c=true, c=false, -c, true, false */
	if ( formula->formula_kind == ATOMIC_FORMULA
	     && (formula->RHS_term == NULL 
		 || strcmp(formula->RHS_term, "true") == 0 
		 || strcmp(formula->RHS_term, "false") == 0 ) ) {
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	  /* formula is -c, we'll make it -(c=false) and use c=false */
	  formula->right_formula->RHS_term = "false";
	  new_axiom = make_new_axiom(formula->right_formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	  /* In this case, if the head we are replacing is a fluent constant,
	     we can replace it okay, but if it is an action constant, we need
	     to take extra care, since this is an action dynamic law and
	     such laws must have an action in the head.
	     
	     Since an action dynamic law "A if F" is shorthand for 
	     i:A  causal_arrow  i:F  (i<m)
	     when we replace A by true/false, we wish to get
	     i:true/false  causal arrow  i:F  (i<m).
	     
	     But "true/false if F" will give
	     i:true/false  causal_arrow  i:F  (i<m+1),
	     
	     so we use "true/false if true after F" instead, which gives
	     i+1:true/false  causal_arrow i+1:true & i:F  (i<m)
	     and this is equivalent to what we want.
	  */
	  axiom_head_constant = lookup_constant(axiom->F->LHS_term);
	  if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	    /* In this case there is no H part, since it must be an action dynamic law */
	    new_axiom = make_new_axiom(formula, 
				       true_formula,
				       axiom->G);
	  }
	  else {
	    new_axiom = make_new_axiom(formula, 
				       axiom->G,
				       axiom->H);
	  }
	}
      }
      else if ( strcmp(axiom->F->RHS_term, "false") == 0 ) {
	/* There are 6 possibilities for formula:
	   c, c=true, c=false, -c, true, false */
	if ( formula->formula_kind == ATOMIC_FORMULA
	     && (formula->RHS_term == NULL 
		 || strcmp(formula->RHS_term, "true") == 0) ) {
	  formula->RHS_term = "false";
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == ATOMIC_FORMULA
		  && strcmp(formula->RHS_term, "false") == 0 ) {
	  formula->RHS_term = "true";
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	  /* formula is -c, we'll make it -(c=true) and use c=true */
	  formula->right_formula->RHS_term = "true";
	  new_axiom = make_new_axiom(formula->right_formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	  /* Since the head we are replacing is c=false, we must use the complement of 
	     the formula to replace it */
	  if ( formula->connective == TRUE ) {
	    formula->connective = FALSE;
	  }
	  else {
	    formula->connective = TRUE;
	  }
	  
	  /* In this case, if the head we are replacing is a fluent constant,
	     we can replace it okay, but if it is an action constant, we need
	     to take extra care, since this is an action dynamic law and
	     such laws must have an action in the head.
	     
	     Since an action dynamic law "A if F" is shorthand for 
	     i:A  causal_arrow  i:F  (i<m)
	     when we replace A by true/false, we wish to get
	     i:true/false  causal arrow  i:F  (i<m).
	     
	     But "true/false if F" will give
	     i:true/false  causal_arrow  i:F  (i<m+1),
	     
	     so we use "true/false if true after F" instead, which gives
	     i+1:true/false  causal_arrow  i+1:true & i:F  (i<m)
	     and this is equivalent to what we want.
	  */
	  axiom_head_constant = lookup_constant(axiom->F->LHS_term);
	  if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	    /* In this case there is no H part, since it must be an action dynamic law */
	    new_axiom = make_new_axiom(formula, 
				       true_formula,
				       axiom->G);
	  }
	  else {
	    new_axiom = make_new_axiom(formula, 
				       axiom->G,
				       axiom->H);
	  }
	}
      }
      else { /* axiom->F->RHS_term must be a Boolean variable */
	/* There are 6 possibilities for formula:
	   c, c=true, c=false, -c, true, false */
	if ( formula->formula_kind == ATOMIC_FORMULA
	     && (formula->RHS_term == NULL 
		 || strcmp(formula->RHS_term, "true") == 0 ) ) {
	  formula->RHS_term = axiom->F->RHS_term;
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == ATOMIC_FORMULA
		  && strcmp(formula->RHS_term, "false") == 0 ) {
	  new_axiom_head = make_new_formula(UNARY_CONNECTIVE);
	  new_axiom_head->connective = NEG;
	  new_axiom_head->right_formula = formula;
	  new_axiom_head->right_formula->RHS_term = axiom->F->RHS_term;
	  new_axiom = make_new_axiom(new_axiom_head, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	  formula->right_formula->RHS_term = axiom->F->RHS_term;
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
	else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	  
	  /*if ( formula->connective == TRUE ) {
	    new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	    new_axiom_head->LHS_term = "true";
	    new_axiom_head->RHS_term = axiom->F->RHS_term;
	  }
	  else {
	    new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	    new_axiom_head->LHS_term = "false";
	    new_axiom_head->RHS_term = axiom->F->RHS_term;
	  }
	  */
	  /*if ( formula->connective == TRUE ) {
	    new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	    new_axiom_head->LHS_term = axiom->F->RHS_term;
	  }
	  else {
	    new_axiom_head = make_new_formula(UNARY_CONNECTIVE);
	    new_axiom_head->connective = NEG;
	    new_axiom_head->right_formula = make_new_formula(ATOMIC_FORMULA);
	    new_axiom_head->right_formula->LHS_term = axiom->F->RHS_term;
	  }
	  */
	  
	  /* In this case, if the head we are replacing is a fluent constant,
	     we can replace it okay, but if it is an action constant, we need
	     to take extra care, since this is an action dynamic law and
	     such laws must have an action in the head.
	     
	     Since an action dynamic law "A if F" is shorthand for 
	     i:A  causal_arrow  i:F  (i<m)
	     when we replace A by true/false, we wish to get
	     i:true/false  causal arrow  i:F  (i<m).
	     
	     But "true/false if F" will give
	     i:true/false  causal_arrow  i:F  (i<m+1),
	     
	     so we use "true/false if true after F" instead, which gives
	     i+1:true/false  causal_arrow  i+1:true & i:F  (i<m)
	     and this is equivalent to what we want.
	  */
	  axiom_head_constant = lookup_constant(axiom->F->LHS_term);
	  if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	    /* In this case there is no H part, since it must be an action dynamic law */
	    if ( formula->connective == TRUE ) {
	      new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      new_axiom_head->connective = FALSE;
	      new_axiom = make_new_axiom(new_axiom_head, 
					 true_formula,
					 replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->RHS_term),
										     make_new_argument(PLAIN_ARGUMENT, "false"),
										     axiom->G));
	    }
	    else {
	      new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      new_axiom_head->connective = FALSE;
	      new_axiom = make_new_axiom(new_axiom_head, 
					 true_formula,
					 replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->RHS_term),
										     make_new_argument(PLAIN_ARGUMENT, "true"),
										     axiom->G));
	    }
	  }
	  else {
	    if ( formula->connective == TRUE ) {
	      new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      new_axiom_head->connective = FALSE;
	      new_axiom = make_new_axiom(new_axiom_head, 
					 axiom->G,
					 replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->RHS_term),
										     make_new_argument(PLAIN_ARGUMENT, "false"),
										     axiom->H));
	    }
	    else {
	      new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      new_axiom_head->connective = FALSE;
	      new_axiom = make_new_axiom(new_axiom_head, 
					 axiom->G,
					 replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->RHS_term),
										     make_new_argument(PLAIN_ARGUMENT, "true"),
										     axiom->H));
	    }
	  }
	}
      }
    }
    else { /* The head we are replacing is a non-Boolean atom. */

      /* There are 4 possibilities for axiom->F:
	   c1=object, c1=true, c1=false, c1=var

	 There are 4 possibilities for formula:
	   c2=var, object=var, true=var, false=var,

	 but in each of these, the variable in axiom->F and the
	 variable in formula have been unified, so we can just
	 replace the whole formula.
	 
	 The only thing we need to watch out for is that, 
	 if c1 is an action constant, it is an action dynamic law
	 which must have an action in the head.  If formula
	 does not have a constant c2, when we replace the LHS_term,
	 the new head will be a constantless formula.
	 
	 Since an action dynamic law "A if F" is shorthand for 
	   i:A  causal_arrow  i:F  (i<m)
	 when we replace A by a constantless formula G, we wish to get
	   i:G  causal arrow  i:F  (i<m).
	 
	 But "G if F" will give
	   i:G  causal_arrow  i:F  (i<m+1),
	     
	 so we use "G if true after F" instead, which gives
	   i+1:G  causal_arrow i+1:true & i:F  (i<m)
	 and this is equivalent to what we want.
      */
      axiom_head_constant = lookup_constant(axiom->F->LHS_term);
      if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	/* In this case there is no H part, since it must be an action dynamic law */
	new_axiom = make_new_axiom(formula, 
				   true_formula,
				   axiom->G);
      }
      else {
	new_axiom = make_new_axiom(formula, 
				   axiom->G,
				   axiom->H);
      }
    }
  }
  else { /* The head we are replacing is the negation of a Boolean atom so 
	    there are 4 possibilities for axiom->F:
	    -(c), -(c=true), -(c=false), -(c=var) */
    if ( axiom->F->right_formula->RHS_term == NULL 
	 || strcmp(axiom->F->right_formula->RHS_term, "true") == 0 ) {
      /* There are 6 possibilities for formula:
	 c, c=true, c=false, -c, true, false */
      if ( formula->formula_kind == ATOMIC_FORMULA
	   && (formula->RHS_term == NULL 
	       || strcmp(formula->RHS_term, "true") == 0 
	       || strcmp(formula->RHS_term, "false") == 0 ) ) {
	new_axiom_head = make_new_formula(UNARY_CONNECTIVE);
	new_axiom_head->connective = NEG;
	new_axiom_head->right_formula = formula;
	new_axiom = make_new_axiom(new_axiom_head, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	/* formula is -c, and when we replace it in the negated head, we get --c,
	   so we'll just make the new head c */
	new_axiom = make_new_axiom(formula->right_formula, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	/* Since the head we are replacing is -c, we must use the complement of 
	   the formula to replace it */
	if ( formula->connective == TRUE ) {
	  formula->connective = FALSE;
	}
	else {
	  formula->connective = TRUE;
	}
	
	/* In this case, if the head we are replacing is a fluent constant,
	   we can replace it okay, but if it is an action constant, we need
	   to take extra care, since this is an action dynamic law and
	   such laws must have an action in the head.
	   
	   Since an action dynamic law "A if F" is shorthand for 
	   i:A  causal_arrow  i:F  (i<m)
	   when we replace A by true/false, we wish to get
	   i:true/false  causal arrow  i:F  (i<m).
	   
	   But "true/false if F" will give
	   i:true/false  causal_arrow  i:F  (i<m+1),
	   
	   so we use "true/false if true after F" instead, which gives
	   i+1:true/false  causal_arrow i+1:true & i:F  (i<m)
	   and this is equivalent to what we want.
	*/
	axiom_head_constant = lookup_constant(axiom->F->right_formula->LHS_term);
	if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	  /* In this case there is no H part, since it must be an action dynamic law */
	  new_axiom = make_new_axiom(formula,
				     true_formula,
				     axiom->G);
	}
	else {
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
      }
    }
    else if ( strcmp(axiom->F->right_formula->RHS_term, "false") == 0 ) {
      /* There are 6 possibilities for formula:
	 c, c=true, c=false, -c, true, false */
      if ( formula->formula_kind == ATOMIC_FORMULA
	   && (formula->RHS_term == NULL 
	       || strcmp(formula->RHS_term, "true") == 0
	       || strcmp(formula->RHS_term, "false") == 0) ) {
	new_axiom = make_new_axiom(formula, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	/* renaming is c1 <-> -c2, and when we replace it in the negated head -(c1=false)
	   we get -c2 so we'll just make the new head -c2 */
	new_axiom = make_new_axiom(formula,
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	/* In this case, if the head we are replacing is a fluent constant,
	   we can replace it okay, but if it is an action constant, we need
	   to take extra care, since this is an action dynamic law and
	   such laws must have an action in the head.
	   
	   Since an action dynamic law "A if F" is shorthand for 
	   i:A  causal_arrow  i:F  (i<m)
	   when we replace A by true/false, we wish to get
	   i:true/false  causal arrow  i:F  (i<m).
	   
	   But "true/false if F" will give
	   i:true/false  causal_arrow  i:F  (i<m+1),
	   
	   so we use "true/false if true after F" instead, which gives
	   i+1:true/false  causal_arrow  i+1:true & i:F  (i<m)
	   and this is equivalent to what we want.
	*/
	axiom_head_constant = lookup_constant(axiom->F->right_formula->LHS_term);
	if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	  new_axiom = make_new_axiom(formula, 
				     true_formula,
				     axiom->G);
	}
	else {
	  new_axiom = make_new_axiom(formula, 
				     axiom->G,
				     axiom->H);
	}
      }
    }
    else { /* axiom->F->right_formula->RHS_term must be a Boolean variable */
      /* There are 6 possibilities for formula:
	 c, c=true, c=false, -c, true, false */
      if ( formula->formula_kind == ATOMIC_FORMULA
	   && (formula->RHS_term == NULL 
	       || strcmp(formula->RHS_term, "true") == 0 ) ) {
	new_axiom_head = copy_formula(axiom->F);
	new_axiom_head->right_formula->RHS_term = formula->RHS_term;
	new_axiom = make_new_axiom(new_axiom_head, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == ATOMIC_FORMULA
		&& strcmp(formula->RHS_term, "false") == 0 ) {
	new_axiom_head = copy_formula(formula);
	new_axiom_head->RHS_term = axiom->F->right_formula->RHS_term;
	new_axiom = make_new_axiom(new_axiom_head, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == UNARY_CONNECTIVE ) {
	new_axiom_head = copy_formula(formula->right_formula);
	new_axiom_head->RHS_term = axiom->F->right_formula->RHS_term;
	new_axiom = make_new_axiom(new_axiom_head, 
				   axiom->G,
				   axiom->H);
      }
      else if ( formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
	
	/* if ( formula->connective == TRUE ) {
	  new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	  new_axiom_head->LHS_term = "false";
	  new_axiom_head->RHS_term = axiom->F->right_formula->RHS_term;
	}
	else {
	  new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	  new_axiom_head->LHS_term = "true";
	  new_axiom_head->RHS_term = axiom->F->right_formula->RHS_term;
	}
	*/
	/*if ( formula->connective == TRUE ) {
	  make_new_formula(UNARY_CONNECTIVE);
	  new_axiom_head->connective = NEG;
	  new_axiom_head->right_formula = make_new_formula(ATOMIC_FORMULA);
	  new_axiom_head->right_formula->LHS_term = axiom->F->right_formula->RHS_term;
	}
	else {
	  new_axiom_head = make_new_formula(ATOMIC_FORMULA);
	  new_axiom_head->LHS_term =axiom->F->right_formula->RHS_term;
	}
	*/
	
	/* In this case, if the head we are replacing is a fluent constant,
	   we can replace it okay, but if it is an action constant, we need
	   to take extra care, since this is an action dynamic law and
	   such laws must have an action in the head.
	   
	   Since an action dynamic law "A if F" is shorthand for 
	   i:A  causal_arrow  i:F  (i<m)
	   when we replace A by true/false, we wish to get
	   i:true/false  causal arrow  i:F  (i<m).
	   
	   But "true/false if F" will give
	   i:true/false  causal_arrow  i:F  (i<m+1),
	   
	   so we use "true/false if true after F" instead, which gives
	   i+1:true/false  causal_arrow  i+1:true & i:F  (i<m)
	   and this is equivalent to what we want.
	*/
	axiom_head_constant = lookup_constant(axiom->F->right_formula->LHS_term);
	if ( axiom_head_constant->node_data->kind == ACTION_CONST ) {
	  if ( formula->connective == TRUE ) {
	    new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    new_axiom_head->connective = FALSE;
	    new_axiom = make_new_axiom(new_axiom_head, 
				       true_formula,
				       replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->right_formula->RHS_term),
										   make_new_argument(PLAIN_ARGUMENT, "true"),
										   axiom->G));
	  }
	  else {
	    new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    new_axiom_head->connective = FALSE;
	    new_axiom = make_new_axiom(new_axiom_head, 
				       true_formula,
				       replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->right_formula->RHS_term),
										   make_new_argument(PLAIN_ARGUMENT, "false"),
										   axiom->G));
	  }
	}
	else {
	  if ( formula->connective == TRUE ) {
	    new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    new_axiom_head->connective = FALSE;
	    new_axiom = make_new_axiom(new_axiom_head, 
				       axiom->G,
				       replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->right_formula->RHS_term),
										   make_new_argument(PLAIN_ARGUMENT, "true"),
										   axiom->H));
	  }
	  else {
	    new_axiom_head = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    new_axiom_head->connective = FALSE;
	    new_axiom = make_new_axiom(new_axiom_head, 
				       axiom->G,
				       replace_arguments_with_arguments_in_formula(make_new_argument(PLAIN_ARGUMENT, axiom->F->right_formula->RHS_term),
										   make_new_argument(PLAIN_ARGUMENT, "false"),
										     axiom->H));
	  }
	}
      }
    }
  }
  
  return new_axiom;
}



/* Turn the nondefinite action description into a definite description:
   - Reverse the renaming list so that the oldest renaming appears first.
   - Go down renaming list, and
     - Use the renaming equivalence to replace occurrences
       of the renamed constant in the heads of non-equivalence axioms
     - Replace the equivalence axiom for the renamed constant by a
       definite law 
       (using Proposition 3 from "Actions as Special Cases" (Erdogan & Lifschitz, 2006)
*/
void make_action_description_definite() {

  struct renaming_list_node * current_renaming;
  struct axiom_list_node * current_axiom;
  struct mad_formula * current_axiom_head;

  struct axiom_list_node * previous_axiom; /* To point to the axiom which precedes the
					      current axiom being processed.  It is set
					      at the end of processing each axiom. */


  struct constant_list_node * current_renaming_constant;
  struct constant_list_node * current_axiom_head_constant;

  struct mad_constant_argument * current_renaming_arguments;
  struct mad_constant_argument * current_axiom_head_arguments;
  struct mad_constant_argument * copied_axiom_head_arguments;

  struct mad_constant_argument * unified_arguments;

  struct string_list_node * free_variable_list1_first;
  struct string_list_node * free_variable_list2_first;
  struct string_list_node * common_free_variable_list;
  
  struct mad_substitution * subst;

  /* This is used to loop through arguments, because we need to add
     an extra argument to the argument list sometimes.  
     (We add the value of a non-Boolean constant). */
  struct mad_constant_argument * current_argument;

  struct mad_formula * copied_atom;
  struct mad_formula * copied_formula;
  struct renaming_case_list_node * copied_cases;
  struct mad_formula * copied_generalized_atom;
  struct renaming_case_list_node * copied_generalized_atom_cases;

  struct mad_formula * copied_axiom_F;
  struct mad_formula * copied_axiom_G;
  struct mad_formula * copied_axiom_H;


  struct renaming_case_list_node * current_renaming_case;

  /* These hold a list of new axioms to replace the current one */
  struct axiom_list_node * new_axioms_first;
  struct axiom_list_node * new_axioms_last;

  /* For cycling through the list of new axioms */
  struct axiom_list_node * current_new_axiom;

  /* When we replace an equivalence by a definite law, we 
     move the right_formula of the equivalence to the body,
     so that it is conjoined with the old G part of the axiom. 
     Also, if the constant being renamed is Boolean,
     we definitize a copy with the sides of the equivalence negated. */
  struct mad_formula * new_axiom_F;
  struct mad_formula * new_axiom_G;

  int constants_match;

  /* First reverse the list of renamings */
  reversed_renaming_list_first = reverse_renaming_list(renaming_list_first);

  current_renaming = reversed_renaming_list_first;
  while ( current_renaming != NULL ) {

    current_renaming_constant = lookup_constant(current_renaming->node_data->atom->LHS_term);

    /* We'll first replace occurrences of this constant in all 
       non-equivalence axioms (these are all definite). */
    current_axiom = axiom_list_first;
    while ( current_axiom != NULL ) {

      /* initialize list of axioms to replace current_axiom */
      init_axiom_list(&new_axioms_first, &new_axioms_last);

      current_axiom_head = current_axiom->node_data->F;
      /* The axioms with equivalences in the head are those which have
	 a binary connective in the head.  (Other axioms are all definite,
	 with the head formula being an atomic formula, a negated formula, or a zero-place connective.
	 And there are no heads with binary connectives other than equivalence.) 

	 Since we want to replace axioms where this constant occurs in the head,
	 we only consider the heads which are atomic formulas or negations.
      */
      if ( current_axiom_head->formula_kind != BINARY_CONNECTIVE 
	   && current_axiom_head->formula_kind != ZERO_PLACE_CONNECTIVE ) {

	constants_match = 0;
	
	if ( strcmp(current_renaming_constant->node_data->domain, "Boolean") == 0 ) {
	  /* First check if the constant occurrence in the head matches that in
	     the renaming */
	  if ( current_axiom_head->formula_kind != UNARY_CONNECTIVE ) {
	    if ( strcmp(current_renaming->node_data->atom->LHS_term, current_axiom_head->LHS_term) == 0 ) {
	      constants_match = 1;
	      current_axiom_head_arguments =copy_arguments( current_axiom_head->LHS_arguments);
	      current_renaming_arguments = copy_arguments(current_renaming->node_data->atom->LHS_arguments);
	    }
	  }
	  else { /* the current axiom's head is a negation */
	    if ( strcmp(current_renaming->node_data->atom->LHS_term, current_axiom_head->right_formula->LHS_term) == 0 ) {
	      constants_match = 1;
	      current_axiom_head_arguments = copy_arguments(current_axiom_head->right_formula->LHS_arguments);
	      current_renaming_arguments = copy_arguments(current_renaming->node_data->atom->LHS_arguments);
	    }
	  }
	}
	else { /* the constant being renamed is not Boolean */
	  // In this case, we know that any matching head
	  // must be an atomic_formula, not a negation
	  if ( current_axiom_head->formula_kind != UNARY_CONNECTIVE ) {
	    if ( strcmp(current_renaming->node_data->atom->LHS_term, current_axiom_head->LHS_term) == 0 ) {
	      constants_match = 1;
	      current_axiom_head_arguments = copy_arguments(current_axiom_head->LHS_arguments);
	      current_renaming_arguments = copy_arguments(current_renaming->node_data->atom->LHS_arguments);
	      /* We also need to add the value of the constants in the axiom head and in the renaming 
		 to the end of the arguments, since these need to be unified. */
	      if ( current_axiom_head_arguments == NULL ) {
		/* There are no arguments, so just add the value */
		current_axiom_head_arguments = make_new_argument(PLAIN_ARGUMENT, current_axiom_head->RHS_term);
	      }
	      else {
		current_argument = current_axiom_head_arguments;
		while ( current_argument->next_argument != NULL ) {
		  current_argument = current_argument->next_argument;
		}
		current_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_axiom_head->RHS_term);
	      }
	      if ( current_renaming_arguments == NULL ) {
		/* There are no arguments, so just add the value */
		current_renaming_arguments = make_new_argument(PLAIN_ARGUMENT, 
							       current_renaming->node_data->formula->RHS_term);
	      }
	      else {
		current_argument = current_renaming_arguments;
		while ( current_argument->next_argument != NULL ) {
		  current_argument = current_argument->next_argument;
		}
		current_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, 
								    current_renaming->node_data->formula->RHS_term);
	      }
	    }
	  }
	}

	if ( constants_match ) {
	  /* Okay, we found a constant to be replaced. 
	     Now we need to be careful about the arguments matching,
	     how to replace negated things, "true"/"false" etc. */
	  
	  /* If there is a generalized atom, then the atom in the renaming
	     does not cover all possible ground instances of the constant to
	     be renamed.  First check if the ground instances of this constant 
	     and the ground instances of the atom from the renaming overlap,
	     and determine the maximal arguments for this overlap. */


	  // We will modify the current axiom's formulas and head arguments
	  // by making substitutions, when we unify the head with the renaming
	  // Since we need clean (i.e. unmodified) versions later, we make copies
	  copied_axiom_F = copy_formula(current_axiom->node_data->F);
	  copied_axiom_G = copy_formula(current_axiom->node_data->G);
	  copied_axiom_H = copy_formula(current_axiom->node_data->H);
	  copied_axiom_head_arguments = copy_arguments(current_axiom_head_arguments);


	  // Before attempting to unify the two argument structures, we need
	  // to see if there are free variables common to both of them and
	  // "standardize apart" by renaming these
	  free_variable_list1_first = find_free_variables_in_arguments(current_axiom_head_arguments, 
								       NULL);
	  free_variable_list2_first = find_free_variables_in_arguments(current_renaming_arguments,
								       NULL);
	  common_free_variable_list = set_intersection_of_string_lists(free_variable_list1_first, 
								       free_variable_list2_first);
	  if ( common_free_variable_list != NULL ) {
	    subst = get_substitution_to_replace_variables_by_new_variables(common_free_variable_list);
	    copied_axiom_F = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_F);
	    copied_axiom_G = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_G);
	    copied_axiom_H = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_G);
	    
	    copied_axiom_head_arguments = replace_arguments_with_arguments_in_argument(subst->old_arguments,
										       subst->new_arguments,
										       copied_axiom_head_arguments);
	  }
	  
	  // Make a new, empty, substitution
	  subst = (struct mad_substitution *) malloc( sizeof(struct mad_substitution) );
	  subst->old_arguments = NULL;
	  subst->new_arguments = NULL;

	  if ( unify_argument_lists(copied_axiom_head_arguments,
				    current_renaming_arguments,
				    &subst) == 1 ) {
	    /* The arguments were compatible, 
	       so we will replace this constant with the atom and its cases */
	    
	    // Make a copy of the formulas in the renamings and the
	    // formulas in the axiom
	    copied_atom = copy_formula(current_renaming->node_data->atom);
	    copied_formula = copy_formula(current_renaming->node_data->formula);
	    copied_cases = copy_renaming_case_list(current_renaming->node_data->cases);
	    
	    // Now replace the arguments in these copies, using the unified arguments
	    copied_atom = replace_arguments_with_arguments_in_formula(subst->old_arguments,
								      subst->new_arguments,
								      copied_atom);
	    copied_formula = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_formula);
	    
	    current_renaming_case = copied_cases;
	    while ( current_renaming_case != NULL ) {
	      
	      current_renaming_case->node_data->condition = replace_arguments_with_arguments_in_formula(subst->old_arguments,
													subst->new_arguments,
													current_renaming_case->node_data->condition);
	      
	      current_renaming_case->node_data->formula = replace_arguments_with_arguments_in_formula(subst->old_arguments,
												      subst->new_arguments,
												      current_renaming_case->node_data->formula);
	      
	      current_renaming_case = current_renaming_case->next;
	    }
	    copied_axiom_F = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_F);
	    copied_axiom_G = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_G);
	    copied_axiom_H = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									 subst->new_arguments,
									 copied_axiom_H);
	    
	    
	    /* make a list of new axioms which will replace this one */
	    if ( current_renaming->node_data->num_cases == 1 ) {
	      insert_axiom_into_list(replace_formula_in_head_of_axiom_by_formula_upon_condition(copy_axiom(make_new_axiom(copied_axiom_F,
															  copied_axiom_G,
															  copied_axiom_H)),
												copied_formula,
												NULL),
				     &new_axioms_first,
				     &new_axioms_last);
	    }
	    else { /* There is more than 1 case, so we use the cases. */
	      current_renaming_case = copied_cases;
	      while ( current_renaming_case != NULL ) {
		insert_axiom_into_list(replace_formula_in_head_of_axiom_by_formula_upon_condition(copy_axiom(make_new_axiom(copied_axiom_F,
															    copied_axiom_G,
															    copied_axiom_H)),
												  current_renaming_case->node_data->formula,
												  current_renaming_case->node_data->condition),
				       &new_axioms_first,
				       &new_axioms_last);
		
		current_renaming_case = current_renaming_case->next;
	      }
	    }
	  }

	  /* Now we need to do the same for the generalized atom and its cases */
	  if ( current_renaming->node_data->generalized_atom != NULL ) {

	    current_renaming_arguments = copy_arguments(current_renaming->node_data->generalized_atom->LHS_arguments);
	    

	    // We will modify the current axiom's formulas and head arguments
	    // by making substitutions, when we unify the head with the renaming
	    // Since we need clean (i.e. unmodified) versions later, we make copies
	    copied_axiom_F = copy_formula(current_axiom->node_data->F);
	    copied_axiom_G = copy_formula(current_axiom->node_data->G);
	    copied_axiom_H = copy_formula(current_axiom->node_data->H);
	    copied_axiom_head_arguments = copy_arguments(current_axiom_head_arguments);
	    

	    // Before attempting to unify the two argument structures, we need
	    // to see if there are free variables common to both of them and
	    // "standardize apart" by renaming these
	    free_variable_list1_first = find_free_variables_in_arguments(current_axiom_head_arguments, 
									 NULL);
	    free_variable_list2_first = find_free_variables_in_arguments(current_renaming_arguments,
									 NULL);
	    common_free_variable_list = set_intersection_of_string_lists(free_variable_list1_first, 
									 free_variable_list2_first);
	    if ( common_free_variable_list != NULL ) {
	      subst = get_substitution_to_replace_variables_by_new_variables(common_free_variable_list);
	      copied_axiom_F = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_F);
	      copied_axiom_G = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_G);
	      copied_axiom_H = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_G);
	      
	      copied_axiom_head_arguments = replace_arguments_with_arguments_in_argument(subst->old_arguments,
											 subst->new_arguments,
											 copied_axiom_head_arguments);
	    }
	    
	    // Make a new, empty, substitution
	    subst = (struct mad_substitution *) malloc( sizeof(struct mad_substitution) );
	    subst->old_arguments = NULL;
	    subst->new_arguments = NULL;
	    
	    /* We first unify the arguments. */
	    if ( unify_argument_lists(copied_axiom_head_arguments,
				      current_renaming_arguments,
				      &subst) == 0 ) {
	      print_error_message(stderr, "There is a problem! The generalized atom in the renaming cannot be unified with the constant occurrence.\n");

	      printf("Attempting to unify: ");
	      print_constant_arguments(stdout, copied_axiom_head_arguments);
	      printf("  with  ");
	      print_constant_arguments(stdout, current_renaming_arguments);
	      printf(" \n");

	    }
	    else {
	      // Make a copy of the formulas in the renamings and the formulas in the axiom
	      copied_generalized_atom = copy_formula(current_renaming->node_data->generalized_atom);
	      copied_generalized_atom_cases = copy_renaming_case_list(current_renaming->node_data->generalized_atom_cases);
	      
	      
	      // Now replace the arguments in these copies, using the unified arguments		
	      copied_generalized_atom = replace_arguments_with_arguments_in_formula(subst->old_arguments,
										    subst->new_arguments,
										    copied_generalized_atom);
	      
	      current_renaming_case = copied_generalized_atom_cases;
	      while ( current_renaming_case != NULL ) {
		
		current_renaming_case->node_data->condition = replace_arguments_with_arguments_in_formula(subst->old_arguments,
													  subst->new_arguments,
													  current_renaming_case->node_data->condition);
		
		current_renaming_case->node_data->formula = replace_arguments_with_arguments_in_formula(subst->old_arguments,
													subst->new_arguments,
													current_renaming_case->node_data->formula);
		
		current_renaming_case = current_renaming_case->next;
	      }
	      
	      copied_axiom_F = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_F);
	      copied_axiom_G = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_G);
	      copied_axiom_H = replace_arguments_with_arguments_in_formula(subst->old_arguments,
									   subst->new_arguments,
									   copied_axiom_H);
	      
	      
	      /* make a list of new axioms which will replace this one */
	      current_renaming_case = copied_generalized_atom_cases;
	      while ( current_renaming_case != NULL ) {
		insert_axiom_into_list(replace_formula_in_head_of_axiom_by_formula_upon_condition(copy_axiom(make_new_axiom(copied_axiom_F,
															    copied_axiom_G,
															    copied_axiom_H)),
												  current_renaming_case->node_data->formula,
												  current_renaming_case->node_data->condition),
				       &new_axioms_first,
				       &new_axioms_last);
		
		current_renaming_case = current_renaming_case->next;
	      }
	    }
	  }
	}
      }
      
      if ( new_axioms_first != NULL ) {
	
	/* Print the axiom getting replaced, which axioms are replacing it, and which renaming was used. 
	 
	fprintf(stdout, "Replaced axiom\n");
	fprintf(stdout, "   ");
	print_formula(stdout, current_axiom->node_data->F);
	fprintf(stdout, "\n");
	fprintf(stdout, "      if ");
	print_formula(stdout, current_axiom->node_data->G);
	fprintf(stdout, "\n");
	if ( current_axiom->node_data->H != NULL ) {
	  fprintf(stdout, "      after ");
	  print_formula(stdout, current_axiom->node_data->H);
	  fprintf(stdout, "\n");
	}
	fprintf(stdout, "with\n");
	current_new_axiom = new_axioms_first;
	while (current_new_axiom != NULL) {
	  print_formula(stdout, current_new_axiom->node_data->F);
	  fprintf(stdout, "\n");
	  fprintf(stdout, "      if ");
	  print_formula(stdout, current_new_axiom->node_data->G);
	  fprintf(stdout, "\n");
	  if ( current_new_axiom->node_data->H != NULL ) {
	    fprintf(stdout,"      after ");
	    print_formula(stdout, current_new_axiom->node_data->H);
	    fprintf(stdout, "\n");
	  }
	  current_new_axiom = current_new_axiom->next;
	}
	fprintf(stdout, "using renaming\n");
	if ( current_renaming->node_data->atom != NULL ) {
	  print_formula(stdout, current_renaming->node_data->atom);
	}
	else {
	  print_error_message(stderr, "NULL formula in renaming!");
	}
	fprintf(stdout, "\n");
	fprintf(stdout, "    is     ");
	if ( current_renaming->node_data->formula != NULL ) {
	  print_formula(stdout, current_renaming->node_data->formula);
	}
	else {
	  print_error_message(stderr, "NULL formula in renaming!");
	}
	fprintf(stdout, "\n");
	// Now print cases
	if (current_renaming->node_data->num_cases == 1) {
	  fprintf(stdout, "  This renaming just has the single case above.\n");
	}
	else {
	  fprintf(stdout, "  Cases:\n");
	  current_renaming_case = current_renaming->node_data->cases;
	  while ( current_renaming_case != NULL ) {
	    fprintf(stdout, "    ");
	    print_formula(stdout, current_renaming_case->node_data->condition);
	    fprintf(stdout, " : ");
	    print_formula(stdout, current_renaming_case->node_data->formula);
	    fprintf(stdout, "\n");
	    current_renaming_case = current_renaming_case->next;
	  }
	}
	// Print the generalized atom and cases, if they exist
	if ( current_renaming->node_data->generalized_atom != NULL ) {
	  fprintf(stdout, "There are also generalized cases:\n");
	  print_formula(stdout, current_renaming->node_data->generalized_atom);
	  fprintf(stdout, "\n");
	  fprintf(stdout, "    is     ");
	  fprintf(stdout, "  Cases:\n");
	  current_renaming_case = current_renaming->node_data->generalized_atom_cases;
	  while ( current_renaming_case != NULL ) {
	    fprintf(stdout, "    ");
	    print_formula(stdout, current_renaming_case->node_data->condition);
	    fprintf(stdout, " : ");
	    print_formula(stdout, current_renaming_case->node_data->formula);
	    fprintf(stdout, "\n");
	    current_renaming_case = current_renaming_case->next;
	  }
	}
	fprintf(stdout, "-------\n");
	*/
	
	new_axioms_last->next = current_axiom->next;
	current_axiom->node_data = new_axioms_first->node_data;
	current_axiom->next = new_axioms_first->next;
      }

      current_axiom = current_axiom->next;
    }

    /* At this point the only axioms with the current constant 
       in the head are the equivalence axioms.  Replace those
       with definite laws, according to 
       Proposition 3 from "Actions as Special Cases" (Erdogan & Lifschitz, 2006) */


    /* We'll find all equivalence axioms which have this constant in the left_formula of the head
       and replace those axioms with definite laws. */
    current_axiom = axiom_list_first;
    while ( current_axiom != NULL ) {

      current_axiom_head = current_axiom->node_data->F;
      /* The axioms with equivalences in the head are those which have
	 a binary connective in the head.  (And there are no heads with 
	 binary connectives other than equivalence.) */

      if ( current_axiom_head->formula_kind == BINARY_CONNECTIVE ) {
	if ( strcmp(current_renaming->node_data->atom->LHS_term, current_axiom_head->left_formula->LHS_term) == 0 ) {
	  /* Okay, we found an equivalence corresponding to this renaming. */

	  /* initialize list of axioms to replace current_axiom */
	  init_axiom_list(&new_axioms_first, &new_axioms_last);

	  /* If the constant in the renaming is multi-valued, we have equivalences for
	     each value.  We can simply turn each one into a definite law.
	     On the other hand, if the constant is Boolean, we only have a single 
	     equivalence where the c=true is renamed.  So we need to turn such equivalences
	     into two definite laws, one for the equivalence and one for both sides of
	     the equivalence negated. */
	  if ( strcmp(current_renaming_constant->node_data->domain, "Boolean") == 0 ) {
	    new_axiom_G = make_new_formula(BINARY_CONNECTIVE);
	    new_axiom_G->connective = AND;
	    new_axiom_G->left_formula = current_axiom_head->right_formula;
	    new_axiom_G->right_formula = current_axiom->node_data->G;

	    insert_axiom_into_list(make_new_axiom(current_axiom_head->left_formula,
						  new_axiom_G,
						  NULL),
				   &new_axioms_first,
				   &new_axioms_last);
	    /* Now make a version with the LHS and RHS of the equivalence negated. */
	    new_axiom_F = make_new_formula(UNARY_CONNECTIVE);
	    new_axiom_F->connective = NEG;
	    new_axiom_F->right_formula = current_axiom_head->left_formula;
	    new_axiom_G = make_new_formula(BINARY_CONNECTIVE);
	    new_axiom_G->connective = AND;
	    new_axiom_G->left_formula = make_new_formula(UNARY_CONNECTIVE);
	    new_axiom_G->left_formula->connective = NEG;
	    new_axiom_G->left_formula->right_formula = current_axiom_head->right_formula;
	    new_axiom_G->right_formula = current_axiom->node_data->G;
	    insert_axiom_into_list(make_new_axiom(new_axiom_F,
						  new_axiom_G,
						  NULL),
				   &new_axioms_first,
				   &new_axioms_last);
	  }
	  else {
	    /* Move the right_formula of the head (i.e. the LHS of the equivalence)
	       to the body and conjoin it with the old body (which is the condition from the renaming case). */
	    
	    new_axiom_G = make_new_formula(BINARY_CONNECTIVE);
	    new_axiom_G->connective = AND;
	    new_axiom_G->left_formula = current_axiom_head->right_formula;
	    new_axiom_G->right_formula = current_axiom->node_data->G;
	    
	    insert_axiom_into_list(make_new_axiom(current_axiom_head->left_formula,
						  new_axiom_G,
						  NULL),
				   &new_axioms_first,
				   &new_axioms_last);
	  }
	  
	  if ( new_axioms_first != NULL ) {
	    /* Print the equivalence axiom getting replaced, and the definite law replacing it
	    
	    fprintf(stdout, "Replaced equivalence axiom\n");
	    fprintf(stdout, "   ");
	    print_formula(stdout, current_axiom->node_data->F);
	    fprintf(stdout, "\n");
	    fprintf(stdout, "      if ");
	    print_formula(stdout, current_axiom->node_data->G);
	    fprintf(stdout, "\n");
	    if ( current_axiom->node_data->H != NULL ) {
	    fprintf(stdout, "      after ");
	    print_formula(stdout, current_axiom->node_data->H);
	    fprintf(stdout, "\n");
	    }
	    fprintf(stdout, "with\n");
	    current_new_axiom = new_axioms_first;
	    while (current_new_axiom != NULL) {
	      print_formula(stdout, current_new_axiom->node_data->F);
	      fprintf(stdout, "\n");
	      fprintf(stdout, "      if ");
	      print_formula(stdout, current_new_axiom->node_data->G);
	      fprintf(stdout, "\n");
	      if ( current_new_axiom->node_data->H != NULL ) {
		fprintf(stdout, "      after ");
		print_formula(stdout, current_new_axiom->node_data->H);
		fprintf(stdout, n"\n");
	      }
	      current_new_axiom = current_new_axiom->next;
	    }
	    fprintf(stdout, "-------\n");
	    */

	    new_axioms_last->next = current_axiom->next;
	    current_axiom->node_data = new_axioms_first->node_data;
	    current_axiom->next = new_axioms_first->next;
	  }
	}
      }
      
      current_axiom = current_axiom->next;
    }

    /* Move on to the next renaming */

    current_renaming = current_renaming->next;
  }
  
}

/* Called after reading all input.
   Computes the objects for a given sort.

   This is called by the function
     build_object_lists_for_sort
*/
void build_object_list_for_single_sort(char * sort_name) {

  struct sort_list_node * sort;
  struct sort_dependency_list_node * current_dependency_node;

  struct string_list_node * first_object_of_sort;
  struct string_list_node * last_object_of_sort;

  struct object_list_node * current_object;

  struct sort_list_node * current_sort;

  struct string_list_node * current_string;

  int index;


  sort = lookup_sort_in_module(sort_name, working_module);

  if ( !sort->node_data->objects_built ) {
    // make sure that all of the sorts to which this sort
    // has dependencies have their objects built
    current_dependency_node = working_module->module_sort_dependencies;
    while ( current_dependency_node != NULL ) {
      if ( strcmp(current_dependency_node->node_data->supersort, sort_name) == 0 ) {
	build_object_list_for_single_sort(current_dependency_node->node_data->subsort);
      }
      current_dependency_node = current_dependency_node->next;
    }
    
    // Now all the sorts which this sort depends upon are ready
    // So we can build the objects for this sort
    init_string_list(&first_object_of_sort,
		     &last_object_of_sort);
    sort->node_data->num_objects = 0;

    current_object = working_module->module_objects;
    while (current_object != NULL) {
      /* Check if it is an object of this sort or an object of a subsort of the current sort. */
      if ( strcmp(current_object->node_data->sort, sort->node_data->name) == 0
	   || (is_a_known_object_sort(current_object->node_data->sort)
	       && is_a_known_sort(sort->node_data->name)
	       && is_a_subsort_of(current_object->node_data->sort, 
				  sort->node_data->name) ) ) {

	// If the object has no arguments, then we can simply add it to the objects
	// of the sort
	if ( current_object->node_data->num_arguments == 0 ) {
	  insert_string_into_list(&first_object_of_sort,
				  &last_object_of_sort,
				  current_object->node_data->name);
	  sort->node_data->num_objects++;
	}
	else {
	// If this object has arguments, then we need to ground them

	  /* Given an object with arguments, we will generate instances of
	     it as follows:
	     Make an array of pointers to the mad_sorts of each argument.
	     Also make two arrays corresponding to the arguments.
	     One of these arrays will hold the number of objects of the sort of argument
	     The other will hold indexes to objects for each sort that are used to generate
	     the current ground instance. */
	  int num_arguments;
	  struct mad_sort ** argument_sorts;
	  int * num_objects;
	  int * indexes;

	  int length;

	  char * ground_object_name;

	  int i;
	  
	  /* An index into the arrays of indexes/num_objects.  It's to keep track
	     of the argument whose object we are currently varying to generate
	     new ground object instances. */
	  int current_index;
	  
	  /* To flag whether an object schema is valid or not. */
	  int void_object_schema; 

	  // It is an action constant "schema" since it has arguments
	  // 
	  // We need to be careful: it will be void if any one of the arguments
	  // is a non-existent sort, or a sort which has no objects
	  void_object_schema = 0;
	  	  
	  int done; /* To signal when we are done generating ground instances for a
		       specific object. */
	  
	  num_arguments = current_object->node_data->num_arguments;
	  
	  // Create arrays corresponding to the arguments of the object.
	  argument_sorts = (struct mad_sort **) malloc( sizeof(struct mad_sort *) * num_arguments);
	  num_objects = (int *) malloc( sizeof(int) * num_arguments);
	  indexes = (int *) malloc( sizeof(int) * num_arguments);
	  
	  /* Set all indexes to zero. */
	  for (i = 0; i < num_arguments; i++) {
	    indexes[i] = 0;
	  }
	  
	  /* Set the sort and number of objects for each argument and value. */
	  current_string = current_object->node_data->arguments;
	  i = 0;
	  while ( current_string != NULL ) {
	    current_sort = lookup_sort_in_module(current_string->node_data, working_module);
	    /* Normally the sort should be found, but right now when the parser 
	       encounters an object with an argument sort which isn't declared,
	       it only generates an error message and then records the constant as is.
	       So we check, in order to avoid segfaults. */
	    if ( current_sort == NULL ) {
	      void_object_schema = 1;
	    }
	    else {
	      argument_sorts[i] = current_sort->node_data;
	      if ( current_sort->node_data->num_objects == 0) {
		/* If there are no objects to ground a sort argument of the action, 
		   then this constant schema is void. */
		void_object_schema = 1;
	      }
	      else {
		num_objects[i] = current_sort->node_data->num_objects;
	      }
	    }
	    i++;
	    current_string = current_string->next;
	  }
	  

	  if (void_object_schema == 1) {
	    /* There's nothing to ground, because the object schema is void.
	       Just skip this object. */
	    print_error_message(stderr, 
				"Object schema %s has argument sorts without any objects.  Discarding without grounding...\n", 
				current_object->node_data->name);
	    current_object = current_object->next;
	    continue;
	  }

	  /* Start from the last argument, and make your way backwards through the arguments.
	     (Where backwards means right to left.) */
	  current_index = num_arguments - 1;

	  done = 0;
	  while ( !done ) {
	    
	    /* Generate ground instance here */
	    
	    // first calculate the length of the string to contain the ground object

	    // at least 2 for ( and ), and 1 for each comma between arguments
	    length = strlen(current_object->node_data->name) + 2 + num_arguments - 1;
	    // now add the lengths of the arguments
	    for (i = 0; i < num_arguments; i++) {
	      length = length + strlen( (argument_sorts[i])->objects[indexes[i]] );
	    }
	    ground_object_name = (char *) malloc( sizeof(char) * ( length + 1 ) );
	    strcpy(ground_object_name, current_object->node_data->name);
	    strcat(ground_object_name, "(");
	    strcat(ground_object_name, (argument_sorts[0])->objects[indexes[0]] );
	    for (i = 1; i < num_arguments; i++) {
	      strcat(ground_object_name, ",");
	      strcat(ground_object_name, (argument_sorts[i])->objects[indexes[i]] );
	    }
	    strcat(ground_object_name, ")");
	    
	    insert_string_into_list(&first_object_of_sort,
				    &last_object_of_sort,
				    ground_object_name);
	    sort->node_data->num_objects++;
	    
	    
	    /* Now, for the next ground instance,
	       go to the next set of argument indexes */
	    if ( indexes[current_index] < (num_objects[current_index]-1) ) {
	      // If there are more objects for the current argument, 
	      // just move to the next object.
	      indexes[current_index]++;
	    }
	    else {
	      /* Go left until you see an index which is less than the number of objects. */
	      while ( (current_index >= 0)
		      && (indexes[current_index] == (num_objects[current_index]-1)) ) {
		current_index--;
	      }
	      /* Check if you've gone off the left-most edge */
	      if ( current_index < 0 ) {
		/* We are done, no more grounding left to do. */
		done = 1;
	      }
	      else {
		/* increment current_index */
		indexes[current_index]++;
		/* Then go back all the way to the right, setting indexes to zero on your way. */
		while ( current_index < (num_arguments-1) ) {
		  current_index++;
		  indexes[current_index] = 0;
		}
	      }
	    }
	  }
	}
      }
      
      current_object = current_object->next;
    }
    
    if (sort->node_data->num_objects > 0) {
      /* Copy object name pointers */
      index = 0;
      sort->node_data->objects = (char **) malloc( sizeof(char *) * sort->node_data->num_objects );
      current_string = first_object_of_sort;
      while ( current_string != NULL ) {
	sort->node_data->objects[index++] = current_string->node_data;
	current_string = current_string->next;
      }
    }
    else {
      // Now that we have all library sorts declared by the ontology,
      // there will almost always be sorts which are not used (and have no objects)
      // so we comment out the error message
      //
      // print_error_message(stderr, "Sort %s was declared but it has no objects.\n", current_sort->node_data->name);
    }
    
    // Mark that this sort's objects have been built
    sort->node_data->objects_built = 1;
  }
  
}

/* Called after reading all input.
   Computes the objects for each sort and updates the sort list
   for the last module, with this info. */
void build_object_lists_for_sorts() {

  struct sort_list_node * new_Boolean_sort_node;
  struct sort_list_node * new_integer_range_sort_node;
  struct sort_list_node * last_added_new_sort_node;
  
  struct sort_list_node * current_sort;

  struct object_list_node * current_object;
  struct constant_list_node * current_constant;
  struct variable_list_node * current_variable;

  struct string_list_node * current_object_arg;
  struct string_list_node * current_constant_arg;
  
  int lower, upper;
  
  int i;

  // We first add all built-in sorts to the beginning of the lists
  // of sorts, with their objects.  These include all integer ranges
  // occurring in the action description and also Boolean

  // We'll make a list with these new sorts and when we're done
  // we will add this new list to the beginning of the list of sorts

  // Make a new Boolean sort, with objects "true" and "false"
  new_Boolean_sort_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
  new_Boolean_sort_node->node_data = make_new_sort("Boolean");
  new_Boolean_sort_node->node_data->num_objects = 2;
  new_Boolean_sort_node->node_data->objects = (char **) malloc( sizeof(char *) * 2 );
  new_Boolean_sort_node->node_data->objects[0] = "true";
  new_Boolean_sort_node->node_data->objects[1] = "false";
  new_Boolean_sort_node->node_data->objects_built = 1;
  last_added_new_sort_node = new_Boolean_sort_node;
  
  // Add integer ranges.  
  // These come from object arguments, constant arguments, 
  // constant domains, and variables

  current_object =  working_module->module_objects;
  while ( current_object != NULL ) {
    current_object_arg = current_object->node_data->arguments;
    while ( current_object_arg != NULL ) {
      if ( is_an_integer_range(current_object_arg->node_data) ) {
	lower = lower_integer_of_range(current_object_arg->node_data);
	upper = upper_integer_of_range(current_object_arg->node_data);
	new_integer_range_sort_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
	new_integer_range_sort_node->node_data = make_new_sort(current_object_arg->node_data);
	new_integer_range_sort_node->node_data->num_objects = upper - lower + 1;
	new_integer_range_sort_node->node_data->objects 
	  = (char **) malloc( sizeof(char *) * new_integer_range_sort_node->node_data->num_objects );
	for (i = 0; i <= new_integer_range_sort_node->node_data->num_objects; i++ ) {
	  new_integer_range_sort_node->node_data->objects[i] = integer_to_string(i + lower);
	}
	new_integer_range_sort_node->node_data->objects_built = 1;
	// Add this node to the end of the list of new sorts
	last_added_new_sort_node->next = new_integer_range_sort_node;
	last_added_new_sort_node = last_added_new_sort_node->next;
      }
      current_object_arg = current_object_arg->next;
    }
    current_object = current_object->next;
  }

  current_constant =  working_module->module_constants;
  while ( current_constant != NULL ) {
    current_constant_arg = current_constant->node_data->arguments;
    while ( current_constant_arg != NULL ) {
      if ( is_an_integer_range(current_constant_arg->node_data) ) {
	lower = lower_integer_of_range(current_constant_arg->node_data);
	upper = upper_integer_of_range(current_constant_arg->node_data);
	new_integer_range_sort_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
	new_integer_range_sort_node->node_data = make_new_sort(current_constant_arg->node_data);
	new_integer_range_sort_node->node_data->num_objects = upper - lower + 1;
	new_integer_range_sort_node->node_data->objects 
	  = (char **) malloc( sizeof(char *) * new_integer_range_sort_node->node_data->num_objects );
	for (i = 0; i <= new_integer_range_sort_node->node_data->num_objects; i++ ) {
	  new_integer_range_sort_node->node_data->objects[i] = integer_to_string(i + lower);
	}
	new_integer_range_sort_node->node_data->objects_built = 1;
	// Add this node to the end of the list of new sorts
	last_added_new_sort_node->next = new_integer_range_sort_node;
	last_added_new_sort_node = last_added_new_sort_node->next;
      }
      current_constant_arg = current_constant_arg->next;
    }
    // Also need to see if the domain is an integer range
    if ( is_an_integer_range(current_constant->node_data->domain) ) {
      lower = lower_integer_of_range(current_constant->node_data->domain);
      upper = upper_integer_of_range(current_constant->node_data->domain);
      new_integer_range_sort_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
      new_integer_range_sort_node->node_data = make_new_sort(current_constant->node_data->domain);
      new_integer_range_sort_node->node_data->num_objects = upper - lower + 1;
      new_integer_range_sort_node->node_data->objects 
	= (char **) malloc( sizeof(char *) * new_integer_range_sort_node->node_data->num_objects );
      for (i = 0; i <= new_integer_range_sort_node->node_data->num_objects; i++ ) {
	new_integer_range_sort_node->node_data->objects[i] = integer_to_string(i + lower);
      }
      new_integer_range_sort_node->node_data->objects_built = 1;
      // Add this node to the end of the list of new sorts
      last_added_new_sort_node->next = new_integer_range_sort_node;
      last_added_new_sort_node = last_added_new_sort_node->next;
    }
    current_constant = current_constant->next;
  }

  current_variable =  working_module->module_variables;
  while ( current_variable != NULL ) {
    if ( is_an_integer_range(current_variable->node_data->sort) ) {
      lower = lower_integer_of_range(current_variable->node_data->sort);
      upper = upper_integer_of_range(current_variable->node_data->sort);
      new_integer_range_sort_node = (struct sort_list_node *) malloc( sizeof(struct sort_list_node) );
      new_integer_range_sort_node->node_data = make_new_sort(current_variable->node_data->sort);
      new_integer_range_sort_node->node_data->num_objects = upper - lower + 1;
      new_integer_range_sort_node->node_data->objects 
	= (char **) malloc( sizeof(char *) * new_integer_range_sort_node->node_data->num_objects );
      for (i = 0; i <= new_integer_range_sort_node->node_data->num_objects; i++ ) {
	new_integer_range_sort_node->node_data->objects[i] = integer_to_string(i + lower);
      }
      new_integer_range_sort_node->node_data->objects_built = 1;
      // Add this node to the end of the list of new sorts
      last_added_new_sort_node->next = new_integer_range_sort_node;
      last_added_new_sort_node = last_added_new_sort_node->next;
    }
    current_variable = current_variable->next;
  }


  // Okay, now that we have created a list of the new sorts,
  // put it at the beginning of the sort list
  last_added_new_sort_node->next = working_module->module_sorts;
  working_module->module_sorts = new_Boolean_sort_node;

  // Now build objects for the rest of the sorts
  current_sort = last_added_new_sort_node->next;
  while (current_sort != NULL) {
    
    build_object_list_for_single_sort(current_sort->node_data->name);
    
    current_sort = current_sort->next;
  }
}

/* for debugging */
void print_objects_of_all_sorts(FILE * stream) {
  struct sort_list_node * current;
  int i;

  fprintf(stream, "Sorts:\n");
  current = working_module->module_sorts;
  while ( current != NULL ) {
    if ( current->node_data->num_objects > 0 ) {
      fprintf(stream, "Sort %s has objects:\n", current->node_data->name);
      fprintf(stream, "  ");
      for (i = 0; i < current->node_data->num_objects; i++) {
	fprintf(stream, "%s ", (current->node_data->objects)[i]);
      }
      fprintf(stream, "\n");
    }
    else {
      fprintf(stream, "Sort %s has no objects.\n", current->node_data->name);
    }
    current = current->next;
  }
}

/* Called after reading all input and building the lists of objects for each sort.
   Computes the ground actions (including ground explicit actions). 

   Assumes all sorts have at least one object.
*/
void build_ground_action_lists() {

  struct sort_list_node * current_sort;

  struct object_list_node * current_object;
  struct constant_list_node * current_constant;

  struct string_list_node * argument_list_first;
  struct string_list_node * argument_list_last;

  struct ground_action_list_node * current_ground_action;
  
  struct string_list_node * current_string;

  int i;

  /* Given a constant (possibly with arguments), we will generate instances of
     it as follows:
     Make an array of pointers to the mad_sorts of each argument and the value.
     Also make two arrays corresponding to the arguments and the value.
     One of these arrays will hold the number of objects of the sort of argument (or value)
     The other will hold indexes to objects for each sort that are used to generate
     the current ground instance. */
  int num_arguments;
  struct mad_sort ** argument_and_value_sorts;
  int * num_objects;
  int * indexes;

  /* An index into the arrays of indexes/num_objects.  It's to keep track
     of the argument/value whose object we are currently varying to generate
     new ground action instances. */
  int current_index;

  int void_action_constant_schema; /* To flag whether an action constant schema is valid or not. */

  int done; /* To signal when we are done generating ground instances for a
	       specific action. */

  int explicit; /* To signal whether the action we are grounding now is an
 		   explicit action. */

  init_ground_actions();
  num_ground_actions = 0;
  init_ground_explicit_actions();
  num_ground_explicit_actions = 0;

  /* To ground actions we need to go down the list of constants
     and find the action constants. */
  current_constant = working_module->module_constants;
  while (current_constant != NULL) {
    if ( current_constant->node_data->kind == ACTION_CONST ) {

      /* Check if it's an explicit action. */
      if ( string_begins_with_an_import_prefix(current_constant->node_data->name) ) {
	explicit = 0;
      }
      else {
	explicit = 1;
      }

      num_arguments = current_constant->node_data->num_arguments;

      if (num_arguments == 0) {
	// There's nothing to ground.  Just insert the same action into the ground action list.
	//fprintf(stdout, "Nothing to ground.  Inserting action as is...\n");

	insert_ground_action(current_constant->node_data->name, 
			     NULL, // no argument list
			     num_arguments,
			     NULL); // no value is inserted since we only allow Boolean actions now
	num_ground_actions++;
	
	if ( explicit ) {
	  insert_ground_explicit_action(current_constant->node_data->name, 
					NULL, // no argument list
					num_arguments,
					NULL); // no value is inserted since we only allow Boolean actions now
	  num_ground_explicit_actions++;
	}

	current_constant = current_constant->next;
	continue;
      }
      

      // It is an action constant "schema" since it has arguments
      // 
      // We need to be careful: it will be void if any one of the arguments
      // is a non-existent sort, or a sort which has no objects
      void_action_constant_schema = 0;

      /* Create arrays corresponding to the arguments and value of the constant.
	 (num_arguments + 1 because of the value.) */
      argument_and_value_sorts = (struct mad_sort **) malloc( sizeof(struct mad_sort *) * num_arguments);
      num_objects = (int *) malloc( sizeof(int) * num_arguments);
      indexes = (int *) malloc( sizeof(int) * num_arguments);

      /* Set all indexes to zero. */
      for (i = 0; i < num_arguments; i++) {
	indexes[i] = 0;
      }

      /* Set the sort and number of objects for each argument and value. */
      current_string = current_constant->node_data->arguments;
      i = 0;
      while ( current_string != NULL ) {
	current_sort = lookup_sort_in_module(current_string->node_data, working_module);
	/* Normally the sort should be found, but right now when the parser 
	   encounters a constant with an argument sort which isn't declared,
	   it only generates an error message and then records the constant as is.
	   So we check, in order to avoid segfaults. */
	if ( current_sort == NULL ) {
	  void_action_constant_schema = 1;
	}
	else {
	  argument_and_value_sorts[i] = current_sort->node_data;
	  if ( current_sort->node_data->num_objects == 0) {
	    /* If there are no objects to ground a sort argument of the action, 
	       then this constant schema is void. */
	    void_action_constant_schema = 1;
	  }
	  else {
	    num_objects[i] = current_sort->node_data->num_objects;
	  }
	}
	i++;
	current_string = current_string->next;
      }
	

      if (void_action_constant_schema == 1) {
	/* There's nothing to ground, because the constant schema is void.
	   Just skip this constant. */
	print_error_message(stderr, 
		"Action constant schema %s has argument sorts without any objects.  Discarding without grounding...\n", 
		current_constant->node_data->name);
	current_constant = current_constant->next;
	continue;
      }

      // This might need to be updated to take into account the fact that
      // the sort might be undeclared -- Boolean
      //current_sort = lookup_sort(current_constant->node_data->domain, working_module);
      //argument_and_value_sorts[i] = current_sort->node_data;
      //num_objects[i] = current_sort->node_data->num_objects;
      
      /* Start from the value, and make your way backwards through the arguments.
         (Where backwards means right to left.) */
      current_index = num_arguments - 1;
      //if ( num_objects[current_index] == 0 ) {
      //  fprintf(stdout, "The domain of action %s is empty!\n", current_constant->node_data->name);
      //  fprintf(stdout, "No ground actions will be generated for it.\n", current_constant->node_data->name);
      //  /* Move to the next action. */
      //  break;
      //}
      
      done = 0;
      while ( !done ) {
	
	/* Generate ground instance here */
	
	init_string_list(&argument_list_first,
			 &argument_list_last);
	for (i = 0; i < num_arguments; i++) {
	  insert_string_into_list(&argument_list_first,
				  &argument_list_last,
				  (argument_and_value_sorts[i])->objects[indexes[i]]);
	}
	
	insert_ground_action(current_constant->node_data->name, 
			     argument_list_first, 
			     num_arguments,
			     //(argument_and_value_sorts[i])->objects[indexes[i]]);
			     NULL);
	num_ground_actions++;

	if ( explicit ) {
	  insert_ground_explicit_action(current_constant->node_data->name, 
					argument_list_first, 
					num_arguments,
					//(argument_and_value_sorts[i])->objects[indexes[i]]);
					NULL);
	  num_ground_explicit_actions++;
	}

	/* Now, for the next ground instance,
	   go to the next set of argument/value indexes */
	if ( indexes[current_index] < (num_objects[current_index]-1) ) {
	  /* If there are more objects for the current argument/value, just move to the next object. */
	  indexes[current_index]++;
	}
	else {
	  /* Go left until you see an index which is less than the number of objects. */
	  while ( (current_index >= 0)
		  && (indexes[current_index] == (num_objects[current_index]-1)) ) {
	    current_index--;
	  }
	  /* Check if you've gone off the left-most edge */
	  if ( current_index < 0 ) {
	    /* We are done, no more grounding left to do. */
	    done = 1;
	  }
	  else {
	    /* increment current_index */
	    indexes[current_index]++;
	    /* Then go back all the way to the right, setting indexes to zero on your way. */
	    while ( current_index < (num_arguments-1) ) {
	      current_index++;
	      indexes[current_index] = 0;
	    }
	  }
	}
      }

    }
    current_constant = current_constant->next;
  }

  /* Now the lists have been built.  
     Copy the lists into arrays, for access during grounding of axioms. */
  ground_actions = (struct mad_ground_action **) malloc( sizeof(struct mad_ground_action *)
							 * num_ground_actions );
  current_ground_action = ground_action_list_first;
  i = 0;
  while ( current_ground_action != NULL ) {
    ground_actions[i] = current_ground_action->node_data;
    i++;
    current_ground_action = current_ground_action->next;
  }

  ground_explicit_actions = (struct mad_ground_action **) malloc( sizeof(struct mad_ground_action *)
								  * num_ground_explicit_actions );
  current_ground_action = ground_explicit_action_list_first;
  i = 0;
  while ( current_ground_action != NULL ) {
    ground_explicit_actions[i] = current_ground_action->node_data;
    i++;
    current_ground_action = current_ground_action->next;
  }
}

/* Assumes formula has been formed validly and that the variable and
   object are valid.
 */
struct mad_constant_argument * replace_object_variable_with_object_in_argument(char * variable, 
								     char * object,
								     struct mad_constant_argument * argument) {
  
  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct variable_list_node * argument_variable;
  
  struct mad_constant_argument * current_argument;

  if ( argument != NULL ) {
          
    term_constant = lookup_constant(argument->name);
    term_object = lookup_object(argument->name);
    // if this argument isn't a constant/object or an arithmetic operator, 
    // there are no arguments nested further */
    if ( term_constant != NULL 
	 || term_object != NULL	 
	 || is_an_arithmetic_operator(argument->name) ) {
      
      current_argument = argument->arguments;
      while ( current_argument != NULL ) {
	argument_variable = lookup_variable(current_argument->name);
	if ( argument_variable != NULL 
	     && strcmp(current_argument->name, variable) == 0 ) {
	  current_argument->name = object;
	}
	else {
	  /* The argument isn't the variable we are looking for,
	     but it might be a constant with variables in its argument */
	  current_argument = replace_object_variable_with_object_in_argument(variable,
									     object,
									     current_argument);
	}
	
	current_argument = current_argument->next_argument;
      }
    }
  }
  return argument;
}

/* Assumes formula has been formed validly and that the variable and
   object are valid.
 */
struct mad_formula * replace_object_variable_with_object_in_formula(char * variable, 
								    char * object,
								    struct mad_formula * formula) {

  struct variable_list_node * term_variable;
  struct variable_list_node * argument_variable;

  struct mad_constant_argument * current_argument;

  if ( formula != NULL ) {

   /* If there is a left_formula, replace the variable in it. */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = replace_object_variable_with_object_in_formula(variable, 
									     object,
									     formula->left_formula);
    }
    /* If it's an atomic formula, replace instances of the variable with the object. */
    if ( formula->formula_kind == ATOMIC_FORMULA 
	 || formula->formula_kind == INEQUALITY ) {
      
      term_variable = lookup_variable(formula->LHS_term);
      if ( term_variable != NULL 
	   && strcmp(formula->LHS_term, variable) == 0 ) {
	formula->LHS_term = object;
      }
      else { /* The LHS term isn't the variable we are looking for 
		but it may be a constant with variables in the arguments */
	if (formula->LHS_arguments != NULL ) {
	  current_argument = formula->LHS_arguments;
	  while ( current_argument != NULL ) {
	    argument_variable = lookup_variable(current_argument->name);
	    if ( argument_variable != NULL 
		 && strcmp(current_argument->name, variable) == 0 ) {
	      current_argument->name = object;
	    }
	    else {
	      /* The argument isn't a variable, but it might be a constant with variables in its argument */
	      current_argument = replace_object_variable_with_object_in_argument(variable,
										 object,
										 current_argument);
	    }
	    
	    current_argument = current_argument->next_argument;
	  }
	}
      }

      /* Now do the same for the RHS term */
      if ( formula->RHS_term != NULL ) {
	term_variable = lookup_variable(formula->RHS_term);
	if ( term_variable != NULL 
	     && strcmp(formula->RHS_term, variable) == 0 ) {
	  formula->RHS_term = object;
	}
	else { /* The RHS term isn't the variable we are looking for 
		  but it may be a constant with variables in the arguments */
	  if (formula->RHS_arguments != NULL ) {
	    current_argument = formula->RHS_arguments;
	    while ( current_argument != NULL ) {
	      argument_variable = lookup_variable(current_argument->name);
	      if ( argument_variable != NULL 
		   && strcmp(current_argument->name, variable) == 0 ) {
		current_argument->name = object;
	      }
	      else {
		/* The argument isn't a variable, but it might be a constant with variables in its argument */
		current_argument = replace_object_variable_with_object_in_argument(variable,
										   object,
										   current_argument);
	      }
	      
	      current_argument = current_argument->next_argument;
	    }
	  }
	}
      }
    }
    /* If it's a quantified formula, then replace instances of the variable 
       only if the quantified variable is different.  (Because otherwise
       the variable won't be free.)
       e.g. the second conjunct in (F(x) & exists x G(x) doesn't have x free. */
    if ( formula->formula_kind == QUANTIFIER ) {
      if ( strcmp(formula->quantified_variable, variable) != 0 ) {
	formula->right_formula = replace_object_variable_with_object_in_formula(variable, 
										object,
										formula->right_formula);	
      }
    }
    /* If there is a right_formula, that is not attached to a quantifier,
       replace the variable in it. */
    else if ( formula->right_formula != NULL ) {
      formula->right_formula = replace_object_variable_with_object_in_formula(variable, 
									      object,
									      formula->right_formula);
    }
  }
  return formula;
}

/* Assumes formula has been formed validly. */
struct mad_formula * ground_quantified_objects_in_formula(struct mad_formula * formula) {

  struct variable_list_node * variable_node;

  struct sort_list_node * variable_sort;

  struct mad_formula * formula_to_be_reproduced;
  struct mad_formula * new_formula;

  int connective;
  char * variable_to_be_replaced;

  int i;

  if (formula != NULL) {

    /* If there is a left_formula, ground the quantified objects in it */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = ground_quantified_objects_in_formula(formula->left_formula);
    }
    
    /* If it's a quantified formula, ground the quantified objects in it */
    if ( formula->formula_kind == QUANTIFIER ) {
      variable_node = lookup_variable(formula->quantified_variable);
      variable_sort = lookup_sort_in_module(variable_node->node_data->sort, working_module);
      if ( variable_sort != NULL 
	   && strcmp(variable_sort->node_data->name, "action") != 0
	   && strcmp(variable_sort->node_data->name, "explicitAction") != 0 ) {
	/*
	fprintf(stdout, "Found variable %s to ground (with %d objs) in a quantifier\n", 
	        formula->quantified_variable,
	        variable_sort->node_data->num_objects);
	*/
	if (variable_sort->node_data->num_objects == 0) {
	  /* There are no objects so the formula is replaced by true or false. */
	  if ( formula->quantifier == FORALL ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = TRUE;
	  }
	  if ( formula->quantifier == EXISTS ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = FALSE;
	  }
	}
	else { 
	  if (formula->quantifier == FORALL) {
	    connective = AND;
	  }
	  if (formula->quantifier == EXISTS) {
	    connective = OR;
	  }
	  
	  formula_to_be_reproduced = formula->right_formula;
	  variable_to_be_replaced = formula->quantified_variable;
	  formula = replace_object_variable_with_object_in_formula(variable_to_be_replaced, 
								   (variable_sort->node_data->objects)[0],
								   copy_formula(formula_to_be_reproduced));
	  for (i= 1; i < variable_sort->node_data->num_objects; i++) {
	    new_formula = make_new_formula(BINARY_CONNECTIVE);
	    new_formula->connective = connective;
	    new_formula->left_formula = formula;
	    new_formula->right_formula = replace_object_variable_with_object_in_formula(variable_to_be_replaced, 
											(variable_sort->node_data->objects)[i],
											copy_formula(formula_to_be_reproduced));
	    
	    formula = new_formula;
	  }
	
	}
	/* After grounding the quantified variable at the head of this formula,
	   we get a new formula which may be composed of subformulas which have
	   quantifiers, so these need to be ground too. */
	ground_quantified_objects_in_formula(formula);
      }
    }
    else if ( formula->right_formula != NULL ) {
      /* If there is a right_formula, that is not attached to a quantifier  */
      formula->right_formula = ground_quantified_objects_in_formula(formula->right_formula);
    }
  }
  return formula;
}

void ground_quantified_objects_in_axioms() {

  struct axiom_list_node * current_axiom;

  current_axiom = working_module->module_axioms;
  while (current_axiom != NULL) {
    current_axiom->node_data->F = ground_quantified_objects_in_formula(current_axiom->node_data->F);
    current_axiom->node_data->G = ground_quantified_objects_in_formula(current_axiom->node_data->G);
    current_axiom->node_data->H = ground_quantified_objects_in_formula(current_axiom->node_data->H);

    current_axiom = current_axiom->next;
  }
  
}



/* 
   Actually, this will probably never be used, because an action variable cannot be nested
   within an argument. 

   Assumes formula has been formed validly and that the variable and
   action are valid.
 */
struct mad_constant_argument * replace_action_variable_with_ground_action_in_argument(char * variable, 
									   struct mad_ground_action * ground_action,
									   struct mad_constant_argument * argument) {
  

  struct constant_list_node * term_constant;
  struct object_list_node * term_object;
  struct variable_list_node * argument_variable;
  
  struct mad_constant_argument * current_argument;
  struct mad_constant_argument * current_argument_argument;

  struct string_list_node * current_ground_action_argument;

  if ( argument != NULL ) {

    /*
    fprintf(stdout, "Replacing variable %s with action ", variable);
    fprintf(stdout, "%s", ground_action->name);
    if ( ground_action->arguments != NULL ) {
      fprintf(stdout, "(");
      current_ground_action_argument = ground_action->arguments;
      fprintf(stdout, "%s", current_ground_action_argument->node_data);
      current_ground_action_argument = current_ground_action_argument->next;
      while( current_ground_action_argument != NULL ) {
	fprintf(stdout, ", %s", current_ground_action_argument->node_data);
	current_ground_action_argument = current_ground_action_argument->next;
      }
      fprintf(stdout, ")");
    }
    fprintf(stdout, "=%s ", ground_action->value);
    fprintf(stdout, "in formula:\n");
    print_formula(stdout, formula);
    fprintf(stdout, "\n");
    */
          
    term_constant = lookup_constant(argument->name);
    term_object = lookup_object(argument->name);
    /* if this argument isn't a constant, there are no arguments nested further */
    if ( term_constant != NULL
	 || term_object != NULL 
	 || is_an_arithmetic_operator(argument->name) ) {
      
      current_argument = argument->arguments;
      while ( current_argument != NULL ) {
	argument_variable = lookup_variable(current_argument->name);
	if ( argument_variable != NULL 
	     && strcmp(current_argument->name, variable) == 0 ) {
	  /* The argument argument is the variable we are replacing */
	  current_ground_action_argument = ground_action->arguments;
	  /* If there is at least one argument, make an argument structure. */
	  if (  current_ground_action_argument != NULL ) {
	    current_argument->name = ground_action->name;
	    current_argument->argument_kind = ARGUMENTED_ARGUMENT;
	    current_argument_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	    current_argument->arguments = current_argument_argument;
	    current_ground_action_argument = current_ground_action_argument->next;
	    /* As long as there are more arguments, add them to the argument structure. */
	    while ( current_ground_action_argument != NULL ) {
	      current_argument_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	      current_argument_argument = current_argument_argument->next_argument;
	      current_ground_action_argument = current_ground_action_argument->next;
	    }
	  }
	  else { /* The ground action has no arguments */
	    current_argument->name = ground_action->name;
	  }
	  ///* Record the value of the ground action */
	  //current_argument->has_a_value = 1;
	  //current_argument->value = ground_action->value;
	}
	else {
	  /* The argument isn't the variable we are looking for,
	     but it might be a constant with variables in its argument */
	  current_argument = replace_action_variable_with_ground_action_in_argument(variable,
									     ground_action,
									     current_argument);
	}

	current_argument = current_argument->next_argument;
      }
    }
  }
  return argument;
}



/* 
   Assumes formula has been formed validly and that the variable and
   action are valid.
 */
struct mad_formula * replace_action_variable_with_ground_action_in_formula(char * variable, 
								    struct mad_ground_action * ground_action,
								    struct mad_formula * formula) {

  struct constant_list_node * term_constant;

  struct variable_list_node * term_variable;
  struct variable_list_node * argument_variable;

  struct mad_constant_argument * current_argument;
  struct mad_constant_argument * current_argument_argument;

  struct mad_constant_argument * current_LHS_argument;
  struct mad_constant_argument * current_RHS_argument;

  struct string_list_node * current_ground_action_argument;

  struct mad_formula * new_formula;

  int mismatch;

  if ( formula != NULL ) {

    /*
    fprintf(stdout, "Replacing variable %s with action ", variable);
    fprintf(stdout, "%s", ground_action->name);
    if ( ground_action->arguments != NULL ) {
      fprintf(stdout, "(");
      current_ground_action_argument = ground_action->arguments;
      fprintf(stdout, "%s", current_ground_action_argument->node_data);
      current_ground_action_argument = current_ground_action_argument->next;
      while( current_ground_action_argument != NULL ) {
	fprintf(stdout, ", %s", current_ground_action_argument->node_data);
	current_ground_action_argument = current_ground_action_argument->next;
      }
      fprintf(stdout, ")");
    }
    fprintf(stdout, "=%s ", ground_action->value);
    fprintf(stdout, "in formula:\n");
    print_formula(stdout, formula);
    fprintf(stdout, "\n");
    */

   /* If there is a left_formula, replace the variable in it. */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = replace_action_variable_with_ground_action_in_formula(variable, 
									     ground_action,
									     formula->left_formula);
    }
    
    /* If it's an atomic formula, replace instances of the variable with the action. */
    if ( formula->formula_kind == ATOMIC_FORMULA 
	 || formula->formula_kind == INEQUALITY ) {

      term_variable = lookup_variable(formula->LHS_term);
      if ( term_variable != NULL 
	   && strcmp(formula->LHS_term, variable) == 0 ) {
	/* The LHS is the action variable we are replacing. */
	
	/* If the atomic formula consists of a single action variable term,
	   we replace it by the ground action where the value becomes the RHS term. */
	if ( formula->RHS_term == NULL ) {
	  formula->LHS_term = ground_action->name;
	  current_ground_action_argument = ground_action->arguments;
	  /* If there is at least one argument, make an argument structure. */
	  if (  current_ground_action_argument != NULL ) {
	    current_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	    formula->LHS_arguments = current_argument;
	    current_ground_action_argument = current_ground_action_argument->next;
	    /* As long as there are more arguments, add them to the argument structure. */
	    while ( current_ground_action_argument != NULL ) {
	      current_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	      current_argument = current_argument->next_argument;
	      current_ground_action_argument = current_ground_action_argument->next;
	    }
	  }
	  /* Record the value.  Even if the action is Boolean and the value is "true", we record
	     it.  This allows us to check whether a term has a value, and thus determine whether it
	     is the result of grounding an action. */ 
	  //if ( strcmp(ground_action->value, "true") != 0
 	  //     || strcmp((lookup_constant(ground_action->name))->node_data->domain, "Boolean") != 0 ) {
	  //formula->RHS_term = ground_action->value;
	  //}
	}
	else { /* The LHS is the action variable we are grounding, but the RHS isn't NULL.
		  So the RHS must be either another action variable or an action. */

	  /* If the RHS is a ground action, we can compare it with the ground action
	     and replace the whole formula by true or false. */
	  new_formula = make_new_formula(ATOMIC_FORMULA);
	  new_formula->LHS_term = formula->RHS_term;
	  new_formula->LHS_arguments = formula->RHS_arguments;
	  if ( find_free_variables_in_formula(new_formula, NULL) == NULL ) {
	    /* The RHS_term is ground */
	    term_constant = lookup_constant(formula->RHS_term);
	    if ( term_constant != NULL ) {
	      /* RHS is an action constant, so we compare it with the ground action we
		 are using to replace the LHS variable. */
	      mismatch = 0;
	      if ( strcmp(ground_action->name, formula->RHS_term) != 0 ) {
		mismatch = 1;
	      }
	      /* If they have arguments, see if those match. */
	      current_ground_action_argument = ground_action->arguments;
	      current_RHS_argument = formula->RHS_arguments;
	      while ( current_ground_action_argument != NULL
		      && current_RHS_argument != NULL ) {
		if ( strcmp( current_ground_action_argument->node_data,
			     current_RHS_argument->name ) != 0 ) {
		  mismatch = 1;
		  break;
		}
		current_ground_action_argument = current_ground_action_argument->next;
		current_RHS_argument = current_RHS_argument->next_argument;
	      }
	      if ( current_ground_action_argument != NULL 
		   || current_RHS_argument != NULL ) {
		/* The number of arguments doesn't match */
		mismatch = 1;
	      }
	      ///* Also see if the values match. Be careful here, because the RHS
	      // may have no value, which matches "true" */
	      //if ( strcmp(ground_action->value, formula->RHS_value) != 0 
	      //   && !(strcmp(ground_action->value, "true") == 0 
	      //	&& formula->RHS_value == NULL) ) {
	      //mismatch = 1;
	      //}
	      if ( mismatch ) {
		formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		formula->connective = FALSE;
	      }
	      else {
		formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		formula->connective = TRUE;
	      }
	      return formula;
	    }
	  }
	  
	  /* The RHS is not ground yet so we just record the ground action on the LHS. */
	  formula->LHS_term = ground_action->name;
	  current_ground_action_argument = ground_action->arguments;
	  /* If there is at least one argument, make an argument structure. */
	  if (  current_ground_action_argument != NULL ) {
	    current_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	    formula->LHS_arguments = current_argument;
	    current_ground_action_argument = current_ground_action_argument->next;
	    /* As long as there are more arguments, add them to the argument structure. */
	    while ( current_ground_action_argument != NULL ) {
	      current_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	      current_argument = current_argument->next_argument;
	      current_ground_action_argument = current_ground_action_argument->next;
	    }
	  }
	  /* Record the value.  Even if the action is Boolean and the value is "true", we record
	     it.  This allows us to check whether a term has a value, and thus determine whether it
	     is the result of grounding an action. */ 
	  //formula->LHS_value = ground_action->value;
	}
      }
      else { /* The LHS term isn't the action variable we are looking for
		but it may be a constant with variables in the arguments */
	if (formula->LHS_arguments != NULL ) {
	  current_argument = formula->LHS_arguments;
	  while ( current_argument != NULL ) {
	    argument_variable = lookup_variable(current_argument->name);
	    if ( argument_variable != NULL 
		 && strcmp(current_argument->name, variable) == 0 ) {
	      /* The current_argument of the term is the variable to be replaced,
		 so we put the ground action there instead.  If the ground action constant 
		 has arguments, the argument will be an ARGUMENTED_ARGUMENT, 
		 otherwise it will remain a PLAIN_ARGUMENT (which it must have been, as a variable). */
	      current_ground_action_argument = ground_action->arguments;
	      /* If there is at least one argument, make an argument structure. */
	      if (  current_ground_action_argument != NULL ) {
		current_argument->name = ground_action->name;
		current_argument->argument_kind = ARGUMENTED_ARGUMENT;
		current_argument_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
		current_argument->arguments = current_argument_argument;
		current_ground_action_argument = current_ground_action_argument->next;
		/* As long as there are more arguments, add them to the argument structure. */
		while ( current_ground_action_argument != NULL ) {
		  current_argument_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
		  current_argument_argument = current_argument_argument->next_argument;
		  current_ground_action_argument = current_ground_action_argument->next;
		}
	      }
	      else { /* The ground action has no arguments */
		current_argument->name = ground_action->name;
	      }
	      /* Record the value of the ground action */
	      //current_argument->has_a_value = 1;
	      //current_argument->value = ground_action->value;
	    }
	    else {
	      /* The argument isn't the variable we are looking for, but it might be a constant 
		 with variables in its argument */
	      current_argument = replace_action_variable_with_ground_action_in_argument(variable,
										 ground_action,
										 current_argument);
	    }

	    current_argument = current_argument->next_argument;
	  }
	}
      }

      /* Now do a similar thing for the RHS term. It won't be exactly the same because the RHS term
	 cannot appear alone, but the LHS term can. */
      if (formula->RHS_term != NULL) {
	term_variable = lookup_variable(formula->RHS_term);
	if ( term_variable != NULL 
	     && strcmp(formula->RHS_term, variable) == 0 ) {
	  /* The RHS is the action variable we are replacing. */
	  
	  /* The RHS is the action variable we are grounding, and the LHS isn't NULL.
	     So the LHS must be either another action variable or an action. */
	  
	  /* If the LHS is a ground action, we can compare it with the ground action
	     and replace the whole formula by true or false. */
	  new_formula = make_new_formula(ATOMIC_FORMULA);
	  new_formula->LHS_term = formula->LHS_term;
	  new_formula->LHS_arguments = formula->LHS_arguments;
	  if ( find_free_variables_in_formula(new_formula, NULL) == NULL ) {
	    /* The LHS_term is ground */
	    term_constant = lookup_constant(formula->LHS_term);
	    if ( term_constant != NULL ) {
	      /* LHS is an action constant, so we compare it with the ground action we
		 are using to replace the RHS variable. */
	      mismatch = 0;
	      if ( strcmp(ground_action->name, formula->LHS_term) != 0 ) {
		mismatch = 1;
	      }
	      /* If they have arguments, see if those match. */
	      current_ground_action_argument = ground_action->arguments;
	      current_LHS_argument = formula->LHS_arguments;
	      while ( current_ground_action_argument != NULL
		      && current_LHS_argument != NULL ) {
		if ( strcmp( current_ground_action_argument->node_data,
			     current_LHS_argument->name ) != 0 ) {
		  mismatch = 1;
		  break;
		}
		current_ground_action_argument = current_ground_action_argument->next;
		current_LHS_argument = current_LHS_argument->next_argument;
	      }
	      if ( current_ground_action_argument != NULL 
		   || current_LHS_argument != NULL ) {
		/* The number of arguments doesn't match */
		mismatch = 1;
	      }
	      /* Also see if the values match. Be careful here, because the LHS
		 may have no value, which matches "true" */
	      //if ( strcmp(ground_action->value, formula->LHS_value) != 0 
	      //   && !(strcmp(ground_action->value, "true") == 0 
	      //	&& formula->LHS_value == NULL) ) {
	      //mismatch = 1;
	      //}
	      if ( mismatch ) {
		formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		formula->connective = FALSE;
	      }
	      else {
		formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		formula->connective = TRUE;
	      }
	      return formula;
	    }
	  }
	  
	  /* The LHS is not ground yet so we just record the ground action on the RHS. */
	  formula->RHS_term = ground_action->name;
	  current_ground_action_argument = ground_action->arguments;
	  /* If there is at least one argument, make an argument structure. */
	  if (  current_ground_action_argument != NULL ) {
	    current_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	    formula->RHS_arguments = current_argument;
	    current_ground_action_argument = current_ground_action_argument->next;
	    /* As long as there are more arguments, add them to the argument structure. */
	    while ( current_ground_action_argument != NULL ) {
	      current_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
	      current_argument = current_argument->next_argument;
	      current_ground_action_argument = current_ground_action_argument->next;
	    }
	  }
	  /* Record the value.  Even if the action is Boolean and the value is "true", we record
	     it.  This allows us to check whether a term has a value, and thus determine whether it
	     is the result of grounding an action. */ 
	  //formula->RHS_value = ground_action->value;
	}
	else { /* The RHS term isn't the action variable we are looking for
		  but it may be a constant with variables in the arguments */
	  if (formula->RHS_arguments != NULL ) {
	    current_argument = formula->RHS_arguments;
	    while ( current_argument != NULL ) {
	      argument_variable = lookup_variable(current_argument->name);
	      if ( argument_variable != NULL 
		   && strcmp(current_argument->name, variable) == 0 ) {
		/* The current_argument of the term is the variable to be replaced,
		   so we put the ground action there instead.  If the ground action constant 
		   has arguments, the argument will be an ARGUMENTED_ARGUMENT, 
		   otherwise it will remain a PLAIN_ARGUMENT (which it must have been, as a variable). */
		current_ground_action_argument = ground_action->arguments;
		/* If there is at least one argument, make an argument structure. */
		if (  current_ground_action_argument != NULL ) {
		  current_argument->name = ground_action->name;
		  current_argument->argument_kind = ARGUMENTED_ARGUMENT;
		  current_argument_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
		  current_argument->arguments = current_argument_argument;
		  current_ground_action_argument = current_ground_action_argument->next;
		  /* As long as there are more arguments, add them to the argument structure. */
		  while ( current_ground_action_argument != NULL ) {
		    current_argument_argument->next_argument = make_new_argument(PLAIN_ARGUMENT, current_ground_action_argument->node_data);
		    current_argument_argument = current_argument_argument->next_argument;
		    current_ground_action_argument = current_ground_action_argument->next;
		  }
		}
		else { /* The ground action has no arguments */
		  current_argument->name = ground_action->name;
		}
		/* Record the value of the ground action */
		//current_argument->has_a_value = 1;
		//current_argument->value = ground_action->value;
	      }
	      else {
		/* The argument isn't the variable we are looking for, but it might be a constant 
		   with variables in its argument */
		current_argument = replace_action_variable_with_ground_action_in_argument(variable,
											  ground_action,
											  current_argument);
	      }
	      
	      current_argument = current_argument->next_argument;
	    }
	  }
	}
      }
    }
    /* If it's a quantified formula, then replace instances of the variable 
       only if the quantified variable is different.  (Because otherwise
       the variable won't be free.)
       e.g. the second conjunct in (F(x) & exists x G(x) doesn't have x free. */
    if ( formula->formula_kind == QUANTIFIER ) {
      if ( strcmp(formula->quantified_variable, variable) != 0 ) {
	formula->right_formula = replace_action_variable_with_ground_action_in_formula(variable, 
										       ground_action,
										       formula->right_formula);	
      }
    }
    /* If there is a right_formula, that is not attached to a quantifier,
       replace the variable in it. */
    else if ( formula->right_formula != NULL ) {
      //    /* If there is a right_formula, replace the variable in it. */
      //    if ( formula->right_formula != NULL ) {
      formula->right_formula = replace_action_variable_with_ground_action_in_formula(variable, 
									      ground_action,
									      formula->right_formula);
    }
  }
  return formula;
}



/* 
   Assumes that this is called after all quantified objects have been ground.
   (i.e. any quantified variable left must be an action or explicitAction variable.)

    Assumes formula has been formed validly.
 */
struct mad_formula * ground_quantified_actions_in_formula(struct mad_formula * formula) {

  struct variable_list_node * variable_node;

  struct mad_formula * formula_to_be_reproduced;
  struct mad_formula * new_formula;

  struct ground_action_list_node * current_ground_action;

  int connective;
  char * variable_to_be_replaced;

  if (formula != NULL) {

    /* If there is a left_formula, ground the quantified actions in it */
    if ( formula->left_formula != NULL ) {
      formula->left_formula = ground_quantified_actions_in_formula(formula->left_formula);
    }
    
    /* If it's a quantified formula, ground the quantified actions in it */
    if ( formula->formula_kind == QUANTIFIER ) {
      variable_node = lookup_variable(formula->quantified_variable);
      if ( strcmp(variable_node->node_data->sort, "action") == 0 ) {
	//fprintf(stdout, "Grounding a quantified action.\n");
	if (num_ground_actions == 0) {
	  /* There are no actions so the formula is replaced by true or false. */
	  if ( formula->quantifier == FORALL ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = TRUE;
	  }
	  if ( formula->quantifier == EXISTS ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = FALSE;
	  }
	}
	else { 
	  if (formula->quantifier == FORALL) {
	    connective = AND;
	  }
	  if (formula->quantifier == EXISTS) {
	    connective = OR;
	  }
	  
	  formula_to_be_reproduced = formula->right_formula;
	  variable_to_be_replaced = formula->quantified_variable;
	  
	  current_ground_action = ground_action_list_first;
	  formula = replace_action_variable_with_ground_action_in_formula(variable_to_be_replaced, 
								   current_ground_action->node_data,
								   copy_formula(formula_to_be_reproduced));
	  current_ground_action = current_ground_action->next;
	  while ( current_ground_action != NULL ) {
	    new_formula = make_new_formula(BINARY_CONNECTIVE);
	    new_formula->connective = connective;
	    new_formula->left_formula = formula;
	    new_formula->right_formula = replace_action_variable_with_ground_action_in_formula(variable_to_be_replaced, 
											current_ground_action->node_data,
											copy_formula(formula_to_be_reproduced));
	    
	    formula = new_formula;

	    current_ground_action = current_ground_action->next;
	  }
	  
	}

	/* After grounding the quantified variable at the head of this formula,
	   we get a new formula which may be composed of subformulas which have
	   quantifiers, so these need to be ground too. */
	ground_quantified_actions_in_formula(formula);

      }
      if ( strcmp(variable_node->node_data->sort, "explicitAction") == 0 ) {
	if (num_ground_explicit_actions == 0) {
	  /* There are no actions so the formula is replaced by true or false. */
	  if ( formula->quantifier == FORALL ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = TRUE;
	  }
	  if ( formula->quantifier == EXISTS ) {
	    formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    formula->connective = FALSE;
	  }
	}
	else { 
	  if (formula->quantifier == FORALL) {
	    connective = AND;
	  }
	  if (formula->quantifier == EXISTS) {
	    connective = OR;
	  }
	  
	  formula_to_be_reproduced = formula->right_formula;
	  variable_to_be_replaced = formula->quantified_variable;
	  
	  current_ground_action = ground_explicit_action_list_first;
	  formula = replace_action_variable_with_ground_action_in_formula(variable_to_be_replaced, 
								   current_ground_action->node_data,
								   copy_formula(formula_to_be_reproduced));
	  current_ground_action = current_ground_action->next;
	  while ( current_ground_action != NULL ) {
	    new_formula = make_new_formula(BINARY_CONNECTIVE);
	    new_formula->connective = connective;
	    new_formula->left_formula = formula;
	    new_formula->right_formula = replace_action_variable_with_ground_action_in_formula(variable_to_be_replaced, 
											current_ground_action->node_data,
											copy_formula(formula_to_be_reproduced));
	    
	    formula = new_formula;
	    
	    current_ground_action = current_ground_action->next;
	  }
	  
	}

	/* After grounding the quantified variable at the head of this formula,
	   we get a new formula which may be composed of subformulas which have
	   quantifiers, so these need to be ground too. */
	ground_quantified_actions_in_formula(formula);

      }
    }
    else if ( formula->right_formula != NULL ) {
      /* If there is a right_formula, that is not attached to a quantifier  */
      formula->right_formula = ground_quantified_actions_in_formula(formula->right_formula);
    }
  }
  return formula;
}


void ground_quantified_actions_in_axioms() {

  struct axiom_list_node * current_axiom;

  current_axiom = working_module->module_axioms;
  while (current_axiom != NULL) {
    current_axiom->node_data->F = ground_quantified_actions_in_formula(current_axiom->node_data->F);
    current_axiom->node_data->G = ground_quantified_actions_in_formula(current_axiom->node_data->G);
    current_axiom->node_data->H = ground_quantified_actions_in_formula(current_axiom->node_data->H);

    current_axiom = current_axiom->next;
  }
  
}


void build_ground_axiom_list() {

  struct axiom_list_node * current_axiom;

  struct sort_list_node * current_sort;

  struct object_list_node * current_object;
  struct constant_list_node * current_constant;

  struct string_list_node * argument_list_first;
  struct string_list_node * argument_list_last;

  struct string_list_node * free_variable_list_first;
  struct string_list_node * free_variables_in_F;
  struct string_list_node * free_variables_not_in_F_but_in_G;
  struct string_list_node * free_variables_not_in_F_or_G_but_in_H;
  
  struct string_list_node * current_string;

  struct mad_axiom * new_axiom;

  int i;

  int axiom_schema_number; /* The number of the axiom we are grounding right now. */

  int void_axiom_schema; /* To flag whether an axiom schema is valid or not. */

  /* Given an axiom without any bound variables (i.e., with the quantifiers already ground)
     we will generate instances of it as follows:
     Make a list of the free variables in the axiom
     Make an array of pointers to the mad_sorts of each free variable.
     Also make two arrays corresponding to the free variables.
     One of these arrays will hold the number of objects of the sort of variable.
     The other will hold indexes to objects for each sort that are used to generate
     the current ground instance. */
  int num_free_vars;
  struct mad_sort ** free_variable_sorts;
  char ** free_variable_names;
  int * num_objects;
  int * indexes;
  
  /* An index into the arrays of indexes/num_objects.  It's to keep track
     of the free variable whose object we are currently varying to generate
     new ground action instances. */
  int current_index;

  int done; /* To signal when we are done generating ground instances for a
	       specific axiom. */

  /* To keep a count of how many axioms were grounded. */
  int num_axioms_grounded;
  int num_total_axioms_grounded;

  init_axiom_list(&ground_axiom_list_first,
		  &ground_axiom_list_last);

  num_total_axioms_grounded = 0;
  
  axiom_schema_number = 1;
  current_axiom = working_module->module_axioms;
  while (current_axiom != NULL) {
    
    /* 
    fprintf(stdout, "Grounding axiom %d:...\n", axiom_schema_number++);
    print_formula(stdout, current_axiom->node_data->F);
    fprintf(stdout, "\n");
    fprintf(stdout, "    if     ");
    print_formula(stdout, current_axiom->node_data->G);
    fprintf(stdout, "\n");
    // print last part if it exists 
    if ( current_axiom->node_data->H != NULL ) {
      fprintf(stdout,"    after  ");
      print_formula(stdout, current_axiom->node_data->H);
      fprintf(stdout, "\n");
    }
    */

    void_axiom_schema = 0; /* Begin with the assumption that the axiom is valid. */

    num_axioms_grounded = 0;

    /* Find the free variables in the axiom */
    free_variable_list_first = NULL;
    free_variables_in_F = find_free_variables_in_formula(current_axiom->node_data->F, NULL);
    free_variables_not_in_F_but_in_G = find_free_variables_in_formula(current_axiom->node_data->G, free_variables_in_F);
    free_variable_list_first = free_variables_in_F;
    append_string_list(&free_variable_list_first, free_variables_not_in_F_but_in_G);
    free_variables_not_in_F_or_G_but_in_H = find_free_variables_in_formula(current_axiom->node_data->H, free_variable_list_first);
    append_string_list(&free_variable_list_first, free_variables_not_in_F_or_G_but_in_H);

    num_free_vars = 0;
    current_string = free_variable_list_first;
    while ( current_string != NULL ) {
      num_free_vars++;
      current_string = current_string->next;
    }

    if (num_free_vars == 0) {
      /* There's nothing to ground.  Just insert the same axiom into the ground axiom list. */
      //fprintf(stdout, "Nothing to ground.  Inserting axiom as is...\n");
      insert_axiom_into_list(copy_axiom(current_axiom->node_data),
			     &ground_axiom_list_first,
			     &ground_axiom_list_last);
      //insert_axiom_into_list(current_axiom->node_data,
      //&ground_axiom_list_first,
      //			     &ground_axiom_list_last);
      num_total_axioms_grounded++;
      current_axiom = current_axiom->next;
      continue;
    }

    /* Create arrays corresponding to the free variables. */
    free_variable_names = (char **) malloc( sizeof(char *) * num_free_vars);
    free_variable_sorts = (struct mad_sort **) malloc( sizeof(struct mad_sort *) * num_free_vars);
    num_objects = (int *) malloc( sizeof(int) * num_free_vars);
    indexes = (int *) malloc( sizeof(int) * num_free_vars);
    
    /* Set all indexes to zero. */
    for (i = 0; i < num_free_vars; i++) {
      indexes[i] = 0;
    }

    /* Set the name, sort and number of objects for each variable. */
    current_string = free_variable_list_first;
    i = 0;
    while ( current_string != NULL ) {
      free_variable_names[i] = current_string->node_data;
      current_sort = lookup_sort_in_module((lookup_variable(current_string->node_data))->node_data->sort,
					   working_module);
      if ( current_sort != NULL ) {
	if ( current_sort->node_data->num_objects == 0 ) {
	  /* If a sort has no objects then any axiom with variables of that sort is void. */
	  void_axiom_schema = 1;
	}
	else {
	  free_variable_sorts[i] = current_sort->node_data;
	  num_objects[i] = current_sort->node_data->num_objects;
	}
      }
      else { /* The sort of the variable is not in the sort list. */
	/* The variable must be an action or explicitAction */
	if ( strcmp((lookup_variable(current_string->node_data))->node_data->sort, "action") == 0 ) {
	  if ( num_ground_actions == 0) {
	    /* If there are no ground actions then any axiom with an action variable is void. */
	    void_axiom_schema = 1;
	  }
	  else {
	    /* Even though, "action" is not a sort, for uniformity, we
	       store it as a sort name.  But we don't access anything
	       other than its name. */
	    free_variable_sorts[i] = make_new_sort("action");
	    num_objects[i] = num_ground_actions;
	    //i--;
	  }
	}
	else if ( strcmp((lookup_variable(current_string->node_data))->node_data->sort, "explicitAction") == 0 ) {
	  if ( num_ground_explicit_actions == 0 ) {
	    /* If there are no ground explicit actions then any axiom with an explicit action variable is void. */
	    void_axiom_schema = 1;
	  }
	  else {
	    /* Even though, "explicitAction" is not a sort, for uniformity, we
	       store it as a sort name.  But we don't access anything
	       other than its name. */
	    free_variable_sorts[i] = make_new_sort("explicitAction");
	    num_objects[i] = num_ground_explicit_actions;
	    //i--;
	  }
	}
	else {
	  print_error_message(stderr, 
			      "Variable %s has no valid sort, so not grounding axiom with it.\n", current_string->node_data);
	  fflush(stdout);
	  void_axiom_schema = 1;
	}
      }
      i++;
      current_string = current_string->next;
    }

    if (void_axiom_schema == 1) {
      /* There's nothing to ground, because the axiom is void.
	 Just skip this axiom. */
      //print_error_message(stderr, "Void axiom schema.  Discarding without grounding...\n");
      current_axiom = current_axiom->next;
      continue;
    }

    //num_free_vars = i;
    
    /* Start from the last variable, and make your way backwards through the variables.
       (Where backwards means right to left.) */
    current_index = num_free_vars - 1 ;
      
    done = 0;
    while ( !done ) {
      
      /* Generate ground instance here */

      new_axiom = make_new_axiom(copy_formula(current_axiom->node_data->F),
				 copy_formula(current_axiom->node_data->G),
				 copy_formula(current_axiom->node_data->H));

      //fprintf(stdout, "Axiom:\n");
      //print_formula(stdout, current_axiom->node_data->F);
      //fprintf(stdout, "  if ");
      //print_formula(stdout, current_axiom->node_data->G);
      //fprintf(stdout, "\n");
      //if (current_axiom->node_data->H != NULL ) {
      //fprintf(stdout, " after ");
      //print_formula(stdout, current_axiom->node_data->H);
      //fprintf(stdout, "\n");
      //}

      /* !!!!!!!!!!!!! This can be done more efficiently, by replacing all variables in parallel, 
	 instead of one by one. */

      for (i = 0; i < num_free_vars; i++) {
	if ( strcmp(free_variable_sorts[i]->name, "action") == 0 ) {
	  /*
	  if ( ground_actions[indexes[i]] != NULL ) {
	    fprintf(stdout, "Grounding it with action %s\n", ground_actions[indexes[i]]->name);
	    fflush(stdout);
	  }
	  else {
	    fprintf(stdout, "Grounding with NULL action!\n");
	    fflush(stdout);
	  }
	  */
	  new_axiom->F = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_actions[indexes[i]],
									       new_axiom->F);
	  new_axiom->G = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_actions[indexes[i]],
									       new_axiom->G);
	  new_axiom->H = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_actions[indexes[i]],
									       new_axiom->H);
	}
	else if ( strcmp(free_variable_sorts[i]->name, "explicitAction") == 0 ) {

	  //if ( ground_explicit_actions[indexes[i]] != NULL ) {
	  //fprintf(stdout, "Grounding it with explicit action %s\n", ground_explicit_actions[indexes[i]]->name);
	  //fflush(stdout);
	  //}
	  //else {
	  //fprintf(stdout, "Grounding with NULL explicit action!\n");
	  //fflush(stdout);
	  //}

	  new_axiom->F = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_explicit_actions[indexes[i]],
									       new_axiom->F);
	  new_axiom->G = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_explicit_actions[indexes[i]],
									       new_axiom->G);
	  new_axiom->H = replace_action_variable_with_ground_action_in_formula(free_variable_names[i],
									       ground_explicit_actions[indexes[i]],
									       new_axiom->H);
	}
	else { /* The variable to be ground is a non-action. */
	  new_axiom->F = replace_object_variable_with_object_in_formula(free_variable_names[i],
									       (free_variable_sorts[i])->objects[indexes[i]],
									       new_axiom->F);
	  new_axiom->G = replace_object_variable_with_object_in_formula(free_variable_names[i],
									       (free_variable_sorts[i])->objects[indexes[i]],
									       new_axiom->G);
	  new_axiom->H = replace_object_variable_with_object_in_formula(free_variable_names[i],
									(free_variable_sorts[i])->objects[indexes[i]],
									new_axiom->H);
	}
      }

      insert_axiom_into_list(new_axiom,
			     &ground_axiom_list_first,
			     &ground_axiom_list_last);
      num_axioms_grounded++;

      /*
      fprintf(stdout, "indexes: ");
      for (i = 0; i < num_free_vars; i++) {
	fprintf(stdout, "%d ", indexes[i]);
      }
      fprintf(stdout, "\n");
      fprintf(stdout, "num objects: ");
      for (i = 0; i < num_free_vars; i++) {
	fprintf(stdout, "%d ", num_objects[i]);
      }
      fprintf(stdout, "\n");
      
      fprintf(stdout, "Axiom: \n");
      print_formula(stdout, new_axiom->F);
      fprintf(stdout, "\n");
      fprintf(stdout, "    if     ");
      print_formula(stdout, new_axiom->G);
      fprintf(stdout, "\n");
      // print last part if it exists
      if ( new_axiom->H != NULL ) {
	fprintf(stdout, "    after  ");
	print_formula(stdout, new_axiom->H);
	fprintf("\n");
      }
      
      fflush(stdout);
      */

      /* Now, for the next ground instance,
	 go to the next set of variable indexes */
      if ( indexes[current_index] < (num_objects[current_index]-1) ) {
	/* If there are more objects for the current free variable, just move to the next object. */
	indexes[current_index]++;
      }
      else {
	/* Go left until you see an index which is less than the number of objects. */
	while ( (current_index >= 0)
		&& (indexes[current_index] == (num_objects[current_index]-1)) ) {
	  current_index--;
	}
	/* Check if you've gone off the left-most edge */
	if ( current_index < 0 ) {
	  /* We are done, no more grounding left to do. */
	  done = 1;
	}
	else {
	  /* increment current_index */
	  indexes[current_index]++;
	  /* Then go back all the way to the right, setting indexes to zero on your way. */
	  while ( current_index < (num_free_vars-1) ) {
	    current_index++;
	    indexes[current_index] = 0;
	  }
	}
      }
    }
    num_total_axioms_grounded = num_total_axioms_grounded + num_axioms_grounded;
    //fprintf(stdout, "%d axioms were grounded.  (Bringing the total number to %d.)\n",
    //num_axioms_grounded,
    //num_total_axioms_grounded);
        
    current_axiom = current_axiom->next;
  }
}
