/* MAD: Software for the Modular Action Description Language
   Copyright (C) 2008  Selim T. Erdogan

   This file is part of MAD.

   MAD is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   MAD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with MAD.  If not, see <http://www.gnu.org/licenses/>.


   Contact info: selim@cs.utexas.edu
*/


/* File: data-types.h
   This file contains data structures and function declarations used in MAD. 
*/


#ifndef DATA_TYPES_H
#define DATA_TYPES_H 1

/* types of identifiers */
#define UNINITIALIZED_ID 0
#define SORT_ID 1
#define VARIABLE_ID 2
#define CONSTANT_ID 3
#define OBJECT_ID 4
#define MODULE_ID 5

/* kinds of constants */
#define SIMPLE_FL_CONST 0
#define SD_FL_CONST 1
#define ACTION_CONST 2
#define RIGID_CONST 3

/* kinds of formula nodes */
#define BINARY_CONNECTIVE 0
#define UNARY_CONNECTIVE 1
#define ATOMIC_FORMULA 2
#define ZERO_PLACE_CONNECTIVE 3
#define QUANTIFIER 4
#define INEQUALITY 5

/* kinds of arguments of consants appearing in formulas */
#define PLAIN_ARGUMENT 0 /* The argument does not have any arguments itself */
#define ARGUMENTED_ARGUMENT 1  /* The argument has arguments which take arguments */

/* kinds of renaming clauses */
#define SORT_RENAMING_CLAUSE 0
#define CONSTANT_RENAMING_CLAUSE 1

/* kinds of formulas */
enum formula_kind_type {
  CONSTANTLESS_FORMULA,  /* No constants. */
  SIMPLE_FLUENT_FORMULA, /* At least one simple fluent constant,
			    no sdFluents, no actions. */
  RIGID_FORMULA,          /* At least one rigid constant, 
			    no fluents, no actions. */
  FLUENT_FORMULA,        /* At least one sdFluent or rigid, no actions. */
  ACTION_FORMULA,        /* At least one action, no fluents. */
  FORMULA,               /* At least one action 
			    and at least (one fluent or one rigid). */
};

/*********************************************
 * This section has variables and functions
 * to keep track of input and output files 
 *********************************************/

/* Upper limit on number of input files given */
#define MAX_INPUT_FILES 256

char * input_file_names[MAX_INPUT_FILES];
int num_input_files;

FILE * input_file_stream;
char * input_file_name;

FILE * old_input_file_stream;

FILE * output_file_stream;
char * output_file_name;

/* To keep a stack of input file data, as we recursively encounter
   files in MAD include statements a. */
#define INPUT_FILE_STACK_SIZE 200

char * input_file_name_stack[INPUT_FILE_STACK_SIZE];
int input_line_number_stack[INPUT_FILE_STACK_SIZE];
int input_file_name_stack_pointer;
int input_line_number_stack_pointer;

/* To push/pop input file data, as we recursively encounter
   files in MAD include statements a. */
void push_input_file_name(char * name);
void push_line_no(int line_number);
char * pop_input_file_name();
int pop_line_no();

/* The line number when we reach the end of an "include"d file
   is recorded and later used to calculate the line number in
   the next file. */
int oldyylineno;

/* Since only the code in the main parser (yaccer.y) has access to
   the line numbers of tokens, we need a way for other functions
   to know the line number when reporting errors.  
   This variable should be set before calling one of these
   auxiliary functions which has the potential to report an error. */
int error_lineno;

/* In some cases, errors arise when we do something general with an
   import block (like merging modules).  When these errors occur, we
   will be in the middle or the end of the import block, but we would
   like to report the line number of the "import" keyword.  Whenever 
   we encounter this keyword, we store the location in a variable. */
int import_lineno;

/* To store an error message which will be generated and printed out 
   when necessary. */
char * error_message;

/* To flag if an error was encountered during parsing */
int parse_error_encountered;

/* To keep track of whether we have seen a module, a sort declaration section,
   or an inclusion declaration section last.
   Has a value of SORTS, INCLUSIONS, or MODULE.
*/
int last_component_seen;

/* To keep track of whether a problematic constant was seen during
   a constant declaration */
int bad_constant_seen_in_declaration;

/* To keep track of whether we have seen a constant renaming clause in
   a particular import section */
int constant_renaming_clause_encountered;

/* To keep track of whether we have seen the keyword IF when parsing
   a "first_formula_follower" in the grammar. */
int keyword_if_seen_in_first_formula_follower;

/* To keep track of whether any formulas about to be combined
   into an axiom were invalid. */
int invalid_formula_seen_while_parsing_axiom;

/* To keep track of whether the LHS or RHS of an "is" statement
   is invalidly formed. */
int invalid_is_statement_part_seen_while_parsing_import;

/* To indicate whether a serious problem (such as a cyclical
   inclusion or an undeclared sort) was introduced when
   trying to do a sort renaming */
int renaming_sorts_leads_to_error;

/* To indicate whether a serious problem (such as a cyclical
   dependency) was introduced when trying to do update sort dependencies
   according to inclusion information. */
int updating_sort_dependencies_leads_to_an_error;

/* To indicate whether a serious problem (such as a cyclical
   dependency) was introduced when trying to do merge a module. */
int merging_sort_dependencies_leads_to_error;

/* To keep track of the number of arguments when processing
   a declaration */
int temp_num_arguments;

/* To keep track of the domain of a constant when processing
   a declaration */
char * temp_domain;

/* To keep track of the kind of a constant when processing a
   declaration */
int temp_kind;

/* To keep track of how many renaming cases there are for
   an "is" statement. */
int temp_num_cases;

/* To keep track of the kind of a formula when processing an
   axiom. Note that this DOES NOT necessarily represent the
   kind of "temp_formula" (introduced below). */
enum formula_kind_type temp_formula_kind;

/* A temporary variable created for recording axioms about
   exogenousness.  
   It is also used to create new variable names when importing
   causes adding In (where n is the import index). */
char * temp_variable;

/* Holds the name of the module currently being parsed. */
char * name_of_module_currently_being_parsed;

/* To keep track of the import index used for importing modules.
   Has the value of the highest import index used so far. */
int global_import_index;

/* Used to append an integer to the end of new variables we create,
   in order to ensure uniqueness. */
int new_variable_index;

/* In order to standardize error messages and make changes to
   all error message printing easily, we use this wrapper around
   fprintf(stderr, ...) 

   This is called when errors are encountered during the parsing stage,
   and reports the line number. 

   The implementation is based on print_to_end_of_string() below.
*/
void print_parser_error_message(FILE * error_stream, char * file_name, int line_number, const char *fmt, ...);

/* In order to standardize error messages and make changes to
   all error message printing easily, we use this wrapper around
   fprintf(stderr, ...) 

   This is called for errors that are not related to a specific file 
   and line number.

   The implementation is based on print_to_end_of_string() below.
*/
void print_error_message(FILE * error_stream, const char *fmt, ...);

/* Converts a nonnegative integer to a string of characters.
   If the number is negative it returns a NULL string. */
char * integer_to_string(int integer);

/* Given a string s and an integer n, returns a new string
   obtained by prepending "In." to s. */
char * prepend_import_prefix_to_string(char * s, int import_index);

/* Given a string s, it checks to see if the string begins with an import
   prefix of the form "I<integer>". */ 
int string_begins_with_an_import_prefix(const char * s);

/* Given a string s, returns
   a new string obtained by prepending "I0." and appending 
   "_var<new_variable_index>" (without < and >) to s. */
char * make_new_variable_name(char *s);

/* Takes a string and prepares it to be used as input for CCalc.
   If there are any import prefixes "I<m>." it turns them into
   "i<m>dot" and for the part after any import prefixes, it
   converts all letters to lower case.

   (It assumes the string is in the correct form for MAD, so it just
   replaces "." by "dot" and makes all letters lower case.)

   If the string does not begin with "I", then it just turns it into
   lower case, but doesn't replace "."s  (This is necessary for
   integer ranges which may be of the for "lo..hi" and the dots
   mustn't be replaced.)

   !!!: This doesn't check for any name clashes after conversion
   so, for example, if identifiers agent and Agent exist in
   the MAD input, they will both turn into agent. */
char * prepare_string_for_ccalc(char * s);

/* Takes a character string "message" and prints to the end of 
   that string.  If the string is NULL, it creates it. 
   The resulting string is returned.
   
   This function is based on the example from the man page of sprintf. 
*/
char * print_to_end_of_string(char * message, const char *fmt, ...);


/****** A List of character strings ****/

/* This is used to deal with lists of character strings
   which may be needed elsewhere, as a list of the arguments
   of a constant, etc. */

/* A node for a linked list of identifiers */
struct string_list_node {
  char * node_data;
  struct string_list_node * next;
};

/* Temporary string lists to use during parsing */
struct string_list_node * temp_string_list_first;
struct string_list_node * temp_string_list_last;

struct string_list_node * temp_string_list2_first;
struct string_list_node * temp_string_list2_last;

struct string_list_node * temp_string_list3_first;
struct string_list_node * temp_string_list3_last;

/*******************************************/
/*******************************************/
/****** Data types for identifiers *********/
/*******************************************/
/*******************************************/

/* The data for a numeric symbol */
struct mad_numeric_symbol {
  char * name;
  char * integer;
};


/* The data for an identifier */
struct id_data {
  char * id_name; /* name of the identifier */
  int id_type; /* sort, variable, constant, object, etc. */
};


/* The data for a sort */
struct mad_sort {
  char * name; /* name of the sort */

  /* After reading in all input, we make an array with all objects of this sort.
     The following are not used Until the end of reading all input. */
  char ** objects;
  int num_objects;
  int objects_built; // 0 initially, 1 after the object list for this sort has been built
};

/* The data for an inclusion statement */
struct mad_inclusion {
  char * supersort; /* name of supersort */
  char * subsort; /* name of subsort */
};

/* The data for a sort dependency.  Each sort will depend
   on sorts which appear as argument in object declarations
   belonging to the first sort.

   This is very similar to an inclusion, but needs to be
   tracked separately.  If there is a cycle in inclusions
   that just means the sorts are the same (i.e. they have
   the same objects) but if there is a cycle in the dependencies
   then grounding would lead to an infinite number of objects.
*/
struct mad_sort_dependency {
  char * supersort; /* name of depending sort */
  char * subsort; /* name of sort which is depended upon */
};

/* The data for an object */
struct mad_object {
  char * name; /* name of the object */
  char * sort; /* the (name of the) sort of the object */
  int num_arguments; /* the number of arguments the object has */
  struct string_list_node * arguments; /* A list of the (names of the) sorts of each argument. */
};


/* The data for a constant */
struct mad_constant {
  char * name; /* name of the constant */
  int kind; /* simple fluent, statically det. fl., action, rigid */
  char * domain; /* Boolean, an already declared sort or a special
                    sort of an already declared sort */
  int num_arguments; /* the number of arguments the constant has */
  struct string_list_node * arguments; /* A list of the (names of the) sorts of each argument. */
};

/* The data for a variable */
struct mad_variable {
  char * name; /* name of the variable */
  char * sort; /* the (name of the) sort of the variable */
};

/* The data for an argument of a constant in formulas.
   This is necessary in case the argument of a constant is a
   constant itself. */
struct mad_constant_argument {
  int argument_kind; /* It can be a constant with or without arguments */

  char * name; /* This is the argument, or the name of the 
			   constant which is the argument */

  int has_a_value; /* If the argument is a constant then it may have a value.
		      Then this will be 1.  Otherwise it is 0. */
  char * value;

  struct mad_constant_argument * arguments; /* If it is a constant with arguments,
					       this points to the first one */
  struct mad_constant_argument * next_argument; /* The next argument at the same
						   level */
};

/* An argument pointer to temporarily keep track of arguments while parsing. */
struct mad_constant_argument * temp_argument;


/* The data for a formula */
struct mad_formula {

  /* a formula may be in the following forms: 
     1. left_formula binary_connective right_formula 
     2. unary_connective right_formula 
     3. term = term   ("this is what an atomic_formula is")
     4. zero_place_connective 
     5. quantifier quantified_variable right_formula */
  int formula_kind; /* Binary connective, unary connective, atomic_formula, zero_place_connective, quantifier */

  int connective; /* kind of connective it is a connective */
  int quantifier; /* kind of quantifier if it is a quantifier */
  int inequality_kind; /* kind of inequality sign if the formula is an inequality */

  char * quantified_variable; /* name of the variable if it is a quantified formula */

  char * LHS_term; /* the term on the LHS of the '=' */
  struct mad_constant_argument * LHS_arguments; /* list of the arguments of the LHS_term */
  char * RHS_term; /* the term on the RHS of the '=' */
  struct mad_constant_argument * RHS_arguments; /* list of the arguments of the RHS_term */

  struct mad_formula * left_formula;  
  struct mad_formula * right_formula;  

  /* The following two fields are an extension.  They are only used when
     grounding actions.  If an atomic formula is of the form a=a1 where
     a and a1 are action variables, then, at the time of grounding the 
     first action variable we need to record the value of the action
     in order to compare it to the other, when the second one is ground.

     Since they are not used until grounding, I think the only function
     that needs to change is copy_formula. */
  char * LHS_value;
  char * RHS_value;
};

/* To keep track of a formula when processing an axiom */
struct mad_formula * temp_formula;

/* Some more formulas for temporary use */
struct mad_formula * temp_formula2;
struct mad_formula * temp_formula3;
struct mad_formula * temp_formula4;

/* To keep track of the LHS of "is" when there are cases in a
   constant renaming */
struct mad_formula * constant_atom_being_renamed;

/* To build the generalized version of a LHS of "is" when the
   atom is Boolean but the arguments are not all variables of
   the declared sort. */
struct mad_formula * temp_generalized_atom;

/* The data for a case in a renaming clause */
struct mad_renaming_case {

  struct mad_formula * condition;  /* The condition under which the case applies. */
  struct mad_formula * formula;
};

/* A node for a linked list of renaming cases */
struct renaming_case_list_node {
  struct mad_renaming_case * node_data;
  struct renaming_case_list_node * next;
};

struct renaming_case_list_node * temp_renaming_case_list_first;
struct renaming_case_list_node * temp_renaming_case_list_last;
struct renaming_case_list_node * temp_renaming_case_list2;

struct renaming_case_list_node * temp_generalized_atom_cases;

/* The data for a renaming clause */
struct mad_renaming_clause {
  /* A mad renaming clause is either a sort renaming clause or a
     constant renaming clause.  Either way, 
     - the LHS of the "is" statement can be parsed as an atom 
       (an identifier with optional arguments and an optional value)
     - the RHS of the "is" statement can be parsed as a formula
     (a sort is just an identifier, which can be an atom/formula) */

  int kind; /* SORT_RENAMING_CLAUSE or CONSTANT_RENAMING_CLAUSE */

  struct mad_formula * atom;
  struct mad_formula * formula;

  /* In a constant_renaming clause, if the RHS has more free variables 
     than the LHS, we record the list of extra free variables
     in order to refer to it easily later. 
     (Instead of computing it over and over every time.) */
  struct string_list_node * list_of_free_variables; 

  int num_cases;
  struct renaming_case_list_node * cases;

  /* If a Boolean constant is being renamed, then the arguments to
     the constant may not be as general as the declared arguments,
     so we also need to have a version of the constant with
     the arguments generalized. */
  struct mad_formula * generalized_atom;
  struct renaming_case_list_node * generalized_atom_cases;
};

/* The data for an axiom */
struct mad_axiom {
  /* axioms have three parts, as in C+:
     caused F if G after H */
  struct mad_formula * F;
  struct mad_formula * G;
  struct mad_formula * H;
};

/*To keep track of an axiom when processing an axiom */
struct mad_axiom * temp_axiom;

/* The data for a module */
struct mad_module {
  char * name; /* name of the module */ 
  struct sort_list_node * module_sorts;
  struct inclusion_list_node * module_inclusions;
  struct sort_dependency_list_node * module_sort_dependencies;
  struct object_list_node * module_objects;
  struct constant_list_node * module_constants;
  struct variable_list_node * module_variables;;
  struct axiom_list_node * module_axioms;

  int import_index; /* This is the largest integer n, 
		       such that In is included in the module */

  /* We keep a list of the renaming clauses processed so
     far in this module.  The list is in reverse order.
     I.e. a new renaming is added to the front of the list. */
  struct renaming_list_node * module_renamings;

};  

/* To keep track of a module when importing. */
struct mad_module * temp_module;

/* When parsing is completed successfully, this points to
   the last module parsed. */
struct mad_module * last_module;
/* And this points to a copy of the last module, which we
   will use for the rest of the processing. */
struct mad_module * working_module;

/* The data for a ground action */
struct mad_ground_action {
  char * name; /* name of the action constant */ 
  struct string_list_node * arguments;
  int num_arguments;
  char * value;

};  

/* Arrays and sizes which will be populated after grounding
   actions.  These arrays will then be used during the grounding of axioms. */
struct mad_ground_action ** ground_actions;
int num_ground_actions;
struct mad_ground_action ** ground_explicit_actions;
int num_ground_explicit_actions;


/* We need to have a way to represent substitution lists
   (showing which variables need to be replaced by what)
   when doing unification.

   (We use two lists of argument structures, instead
    of one list of variables and another of arguments.
    This is purely for convenience, because when we
    added this data type we already had functions to
    replace arguments by arguments.)
*/
struct mad_substitution {
  struct mad_constant_argument * old_arguments;
  struct mad_constant_argument * new_arguments;
};


/*******************************************/
/*******************************************/
/****** Data types for lists ***************/
/*******************************************/
/*******************************************/

/****** List of numeric symbols ****/

/* A node for a linked list of inclusions */
struct numeric_symbol_list_node {
  struct mad_numeric_symbol * node_data;
  struct numeric_symbol_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct numeric_symbol_list_node * numeric_symbol_list_first;
struct numeric_symbol_list_node * numeric_symbol_list_last;

/* Numeric symbol list node for temporary use (during parsing) */
struct numeric_symbol_list_node * temp_numeric_symbol_list_node;


/****** List of identifiers ****/

/* A node for a linked list of identifiers */
struct id_list_node {
  struct id_data * node_data;
  struct id_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct id_list_node * id_list_first;
struct id_list_node * id_list_last;


/****** List of sorts ****/

/* A node for a linked list of sorts */
struct sort_list_node {
  struct mad_sort * node_data;
  struct sort_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct sort_list_node * sort_list_first;
struct sort_list_node * sort_list_last;

/* Pointers to the first and last nodes on the global ontology list of sorts */
struct sort_list_node * global_sort_list_first;
struct sort_list_node * global_sort_list_last;

/****** List of inclusions ****/

/* A node for a linked list of inclusions */
struct inclusion_list_node {
  struct mad_inclusion * node_data;
  struct inclusion_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct inclusion_list_node * inclusion_list_first;
struct inclusion_list_node * inclusion_list_last;

/* Pointers to the first and last nodes on the global ontology list of inclusions */
struct inclusion_list_node * global_inclusion_list_first;
struct inclusion_list_node * global_inclusion_list_last;

/****** List of sort dependencies ****/

/* A node for a linked list of sort dependencies */
struct sort_dependency_list_node {
  struct mad_sort_dependency * node_data;
  struct sort_dependency_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct sort_dependency_list_node * sort_dependency_list_first;
struct sort_dependency_list_node * sort_dependency_list_last;

/* Pointers to the first and last nodes on the global ontology list of sort dependencies */
struct sort_dependency_list_node * global_sort_dependency_list_first;
struct sort_dependency_list_node * global_sort_dependency_list_last;


/****** List of objects ****/

/* A node for a linked list of objects */
struct object_list_node {
  struct mad_object * node_data;
  struct object_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct object_list_node * object_list_first;
struct object_list_node * object_list_last;

/* object list nodes for temporary use */
struct object_list_node * temp_object_list_node;
struct object_list_node * temp_object_list_node2;
struct object_list_node * temp_object_list_node3;

/****** List of constants ****/

/* A node for a linked list of constants */
struct constant_list_node {
  struct mad_constant * node_data;
  struct constant_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct constant_list_node * constant_list_first;
struct constant_list_node * constant_list_last;

/* constant list nodes for temporary use */
struct constant_list_node * temp_constant_list_node;
struct constant_list_node * temp_constant_list_node2;
struct constant_list_node * temp_constant_list_node3;

/****** List of variables ****/

/* A node for a linked list of variables */
struct variable_list_node {
  struct mad_variable * node_data;
  struct variable_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct variable_list_node * variable_list_first;
struct variable_list_node * variable_list_last;

/****** List of renaming clauses ****/

/* A node for a linked list of renaming clauses */
struct renaming_list_node {
  struct mad_renaming_clause * node_data;
  struct renaming_list_node * next;
};

/* Pointers to the first and last nodes on the list 
   of renamings built up during processing of a module. 
   (This list contains renamings coming from any imports
   so far.) */
struct renaming_list_node * renaming_list_first;
struct renaming_list_node * renaming_list_last;

/* Pointers to the first and last nodes on the list
   of renamings seen for a specific import statement. 
   (This is reset every time an import section is processed.) */
struct renaming_list_node * import_renaming_list_first;
struct renaming_list_node * import_renaming_list_last;

/* After reading in all the input, before we turn our
   nondefinite action description into a definite one,
   we reverse the renaming list, for easier processing.
   This will hold a reversed version of renaming_list_first. */
struct renaming_list_node * reversed_renaming_list_first;

/****** List of formulas ****/

/* A node for a linked list of axioms */
struct formula_list_node {
  struct mad_formula * node_data;
  struct formula_list_node * next;
};


/****** List of axioms ****/

/* A node for a linked list of axioms */
struct axiom_list_node {
  struct mad_axiom * node_data;
  struct axiom_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct axiom_list_node * axiom_list_first;
struct axiom_list_node * axiom_list_last;

/* Pointers to the first and last nodes on the list of ground axioms */
struct axiom_list_node * ground_axiom_list_first;
struct axiom_list_node * ground_axiom_list_last;


/****** List of modules ****/

/* A node for a linked list of modules */
struct module_list_node {
  struct mad_module * node_data;
  struct module_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct module_list_node * module_list_first;
struct module_list_node * module_list_last;

/****** Lists of ground actions ******/

/* A node for a linked list of ground actions */
struct ground_action_list_node {
  struct mad_ground_action * node_data;
  struct ground_action_list_node * next;
};

/* Pointers to the first and last nodes on the list */
struct ground_action_list_node * ground_action_list_first;
struct ground_action_list_node * ground_action_list_last;

/* Pointers to the first and last nodes on the list for explicit actions */
struct ground_action_list_node * ground_explicit_action_list_first;
struct ground_action_list_node * ground_explicit_action_list_last;


/*******************************************/
/*******************************************/
/****** Data types for trees ***************/
/*******************************************/
/*******************************************/

/* These aren't used yet.............. */

/* A node for a tree of tokens */
struct tree_node {
  int node_type; /* what type of node? data, keyword, ... */
  char * node_data; 
  int num_children; /* the number of children of the node */
  struct tree_node * children; /* pointer to array of children */
};

/* Pointer to the root of the tree */
struct tree_node * root;


/*******************************************/
/*******************************************/
/*******************************************/
/****** Function declarations **************/
/*******************************************/
/*******************************************/


/****** Managing lists of numeric symbols **************/

/* initialize the list of numeric_symbols */
void init_numeric_symbols();

/* Looks up a numeric_symbol and returns a pointer to its list entry.
   Returns NULL if not found. */
struct numeric_symbol_list_node * lookup_numeric_symbol(char * lookup_name);

/* Called when inserting a new numeric symbol.  Makes a new numeric symbol. */
struct mad_numeric_symbol * make_new_numeric_symbol (char * name, char * integer);

/* This inserts a new numeric symbol into the list of numeric symbols */
void insert_numeric_symbol (char * name, char * integer);

/* for debugging */
void print_current_numeric_symbols(FILE * stream);


/****** Managing lists of character strings **************/


/* initialize a list of strings */
void init_string_list(struct string_list_node ** first, struct string_list_node ** last);

/* This inserts a new string into a list of strings */
void insert_string_into_list(struct string_list_node ** first, 
			     struct string_list_node ** last, 
			     char * new_string);

struct string_list_node * lookup_string_in_list(char * lookup_string, struct string_list_node * list);

/* This checks whether two lists of strings are the same */
int string_list_same(struct string_list_node * first, 
		     struct string_list_node * second);

/* Makes a copy of a given string list */
struct string_list_node * copy_string_list(struct string_list_node * string_list);

/* Finds the set difference between two given lists of strings and 
   returns it as a list of strings.
   (One use of this is to compare the lists of free variables in two
    formulas.)

   Assumes that the given string lists have no repetitions.
 */
struct string_list_node * set_difference_of_string_lists(struct string_list_node * first, 
							 struct string_list_node * second);

/* This adds a string list onto the end of another */
void append_string_list(struct string_list_node ** first, 
			struct string_list_node * second);

/* for debugging */
void print_string_list(FILE * stream, struct string_list_node * string_list);

/****** Managing lists of identifiers **************/


/* initialize the list of identifiers */
void init_identifiers();

/* initialize the list of identifiers to contain the list of global sorts */
void init_identifiers_with_global_sorts();

/* Looks up an identifier and returns a pointer to its list entry.
   Returns NULL if not found. */
struct id_list_node * lookup_identifier(char * lookup_name);


/* This inserts a new identifier into the list of identifiers */
void insert_identifier (char * name);


/* Called when inserting a new identifier.  Makes a new identifier */
struct id_data * make_id_data (char * name);

/* For indicating a type of an identifier.
   The id is found during lexical analysis but we only
   learn it's type during parsing. */
void update_id_type(char * name, int type);

/* for debugging */
void print_id_list(FILE * stream);

/****** Managing lists of sorts **************/


/* initialize the list of sorts */
void init_sorts();

/* initialize the global list of sorts */
void init_global_sorts();

/* Looks up a sort and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_list_node * lookup_sort(char * lookup_name);

/* Looks up a sort in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_list_node * lookup_sort_in_module(char * lookup_name, struct mad_module * module);

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action". */
int is_a_known_sort(char * name);

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action". */
int is_a_known_sort_in_module(char * name, struct mad_module * module);

/* Checks if a given string is a declared sort or the keyword "Boolean". */
int is_a_known_object_sort(char * name);

/* Checks if a given string is a declared sort in a particular module,
   or the keyword "Boolean". */
int is_a_known_object_sort_in_module(char * name, struct mad_module * module);

/* Checks if a given string is a declared sort or the keyword "Boolean". */
int is_a_known_constant_domain_sort(char * name);

/* Checks if a given string is a declared sort in a particular module,
   or the keyword "Boolean". */
int is_a_known_constant_domain_sort_in_module(char * name, struct mad_module * module);

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action". */
int is_a_known_constant_argument_sort(char * name);

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action". */
int is_a_known_constant_argument_sort_in_module(char * name, struct mad_module * module);

/* Checks if a given string is a declared sort or one of the
   keywords "Boolean", "action", "explicitAction". */
int is_a_known_variable_sort(char * name);

/* Checks if a given string is a declared sort in a particular module,
   or one of the keywords "Boolean", "action", "explicitAction". */
int is_a_known_variable_sort_in_module(char * name, struct mad_module * module);

/* Called when inserting a new sort.  Makes a new sort */
struct mad_sort * make_new_sort (char * name);

/* This inserts a new sort into the list of sorts */
void insert_sort (char * name);

/* Makes a copy of a given sort list */
struct sort_list_node * copy_sort_list(struct sort_list_node * sort_list);

/* Makes the list of sorts a copy of the global list of sorts */ 
void copy_global_sorts();

/* Called to merge sorts of an imported module with the current sorts */
void merge_sorts_from_module(struct mad_module * module);

/* for debugging */
void print_current_sorts(FILE * stream);

/* for debugging.  
   To print out a specific list of sorts (e.g. of a specific module). */
void print_sort_list(FILE * stream, struct sort_list_node * list);

/* for printing out a sort declaration section to be used as ccalc input */
void print_sort_list_for_ccalc(FILE * stream, struct sort_list_node * list);


/****** Managing integer ranges and arithmetic **************/

// Takes two strings "m", "n" containing integer values and returns
// an integer range string "m..n"
char * make_integer_range_string(char * begin, char * end);

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and makes sure that "lo <= hi"
//
// We assume that it's syntactically valid, having been already
// processed and marked as an integer range in the lexical analyzer
int is_a_valid_integer_range(char * range_string);

// Takes a string and check if it is an integer range 
// in the form "lo..hi" (where lo and hi may be negative too) 
//
// Doesn't check wheter lo<=hi because we assume that has already
// been checked
int is_an_integer_range(char * range_string);

// Takes a string and check if it is an integer
int is_an_integer(char * s);

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and returns the value of lo
//
// We assume that the string given is syntactically valid, 
// having been already processed and marked as an integer 
// range in the lexical analyzer
int lower_integer_of_range(char * range_string);

// Takes an integer range in the form "lo..hi" (where lo and hi may 
// be negative too) and returns the value of hi
//
// We assume that the string given is syntactically valid, 
// having been already processed and marked as an integer 
// range in the lexical analyzer
int upper_integer_of_range(char * range_string);

// Takes an integer range of the form "lo..hi" and turns it
// returns the string "ccalc_int_range_lotohi"
char * convert_integer_range_to_ccalc_acceptable_form(char * range);

/*  If there are built-in integer ranges used in the action description,
    then the integer objects in these need to be declared explicitly.

    This function prints out an object declaration section to with these integers.

 */
void print_integer_range_object_list_for_ccalc(FILE * stream, struct sort_list_node * list);


// Takes a string and check if it is an arithmetic operator
int is_an_arithmetic_operator(char * s);


/****** Managing lists of inclusions **************/


/* initialize the list of inclusions */
void init_inclusions();

/* initialize the global list of inclusions */
void init_global_inclusions();

/* Looks up an inclusion and returns a pointer to its list entry.
   Returns NULL if not found. */
struct inclusion_list_node * lookup_inclusion(char * lookup_supersort, char * lookup_subsort);

/* Looks up an inclusion in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct inclusion_list_node * lookup_inclusion_in_module(char * lookup_supersort, 
							char * lookup_subsort, 
							struct mad_module * module);

/* Called when inserting a new inclusion.  Makes a new inclusion */
struct mad_inclusion * make_new_inclusion (char * supersort, char * subsort);

/* This inserts a new inclusion into the list of inclusions */
void insert_inclusion (char * supersort, char * subsort);

/* Makes a copy of a given inclusion list */
struct inclusion_list_node * copy_inclusion_list(struct inclusion_list_node * inclusion_list);

/* Makes the list of inclusions a copy of the global list of inclusions */ 
void copy_global_inclusions();

/* Checks if a given sort is a subsort of another given sort.  
   If the two sorts are the same, then it won't consider them 
   to be subsorts of each other, unless there's a cycle in the
   inclusion graph. (So this can be used as a cycle detector.)

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
 */
int is_a_subsort_of(char * first_sort, char * second_sort);

/* Checks if a given sort is a subsort of another given sort, in a given module.
   If the two sorts are the same, then it won't consider them 
   to be subsorts of each other, unless there's a cycle in the
   inclusion graph. (So this can be used as a cycle detector.) 

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
*/
int is_a_subsort_of_in_module(char * first_sort, 
			      char * second_sort, 
			      struct mad_module * module);

/* Called to merge inclusions of an imported module with the current inclusions */
void merge_inclusions_from_module(struct mad_module * module);

/* for debugging */
void print_current_inclusions(FILE * stream);

/* for debugging.  
   To print out a specific list of inclusions (e.g. of a specific module). */
void print_inclusion_list(FILE * stream, struct inclusion_list_node * list);

/* for printing an inclusion declaration section to be used as ccalc input. */
void print_inclusion_list_for_ccalc(FILE * stream,
				    struct inclusion_list_node * inclusion_list);

/* for printing out a sort declaration section to be used as ccalc input. */
void print_sort_and_inclusion_list_for_ccalc(FILE * stream,
					     struct sort_list_node * sort_list,
					     struct inclusion_list_node * inclusion_list);



/****** Managing lists of sort dependencies **************/


/* initialize the list of sort dependencies */
void init_sort_dependencies();

/* initialize the global list of sort dependencies */
void init_global_sort_dependencies();

/* Looks up a sort dependency and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_dependency_list_node * lookup_sort_dependency(char * lookup_supersort, 
							  char * lookup_subsort);

/* Looks up a sort_dependency in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct sort_dependency_list_node * lookup_sort_dependency_in_module(char * lookup_supersort, 
								    char * lookup_subsort, 
								    struct mad_module * module);

/* Called when inserting a new sort_dependency.  Makes a new sort_dependency */
struct mad_sort_dependency * make_new_sort_dependency (char * supersort, char * subsort);

/* This inserts a new sort_dependency into the list of sort_dependencies */
void insert_sort_dependency (char * supersort, char * subsort);

/* Makes a copy of a given sort_dependency list */
struct sort_dependency_list_node * copy_sort_dependency_list(struct sort_dependency_list_node * sort_dependency_list);

/* Makes the list of sort dependencies a copy of the global list of sort dependencies */ 
void copy_global_sort_dependencys();

/* Checks if a given sort is depended upon by another given sort.  
   If the two sorts are the same, then it won't consider them 
   to depend upon each other, unless there's a cycle in the
   sort dependency graph. (So this can be used as a cycle detector.)

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
 */
int is_a_sort_depended_upon_by_another_sort(char * first_sort, char * second_sort);

/* Checks if a given sort is depended upon by another given sort, 
   in a given module.
   If the two sorts are the same, then it won't consider them 
   to depend upon each other, unless there's a cycle in the
   sort_dependency graph. (So this can be used as a cycle detector.) 

   This first makes sure that the given sorts are known.
   (It uses is_a_known_variable_sort because that's the most general
    sort checking function.)
*/
int is_a_sort_depended_upon_by_another_sort_in_module(char * first_sort, 
						      char * second_sort, 
						      struct mad_module * module);

// Given a list of argument sorts for an object in a string list,
// and the sort of the object,
// this function records sort dependencies between each argument 
// and the object sort
void update_sort_dependencies_according_to_object_declaration(struct string_list_node * args,
							      char * object_sort_name);

// When we have an inclusion, this means that the supersort of
// the inclusion depends on the dependencies of the subsort
//
// This function updates looks at all inclusions and updates
// the sort dependencies accordingly
void update_sort_dependencies_according_to_inclusions();

/* Called to merge sort dependencies of an imported module with the current sort dependencies */
void merge_sort_dependencies_from_module(struct mad_module * module);

/* for debugging */
void print_current_sort_dependencies(FILE * stream);

/* for debugging.  
   To print out a specific list of sort dependencies (e.g. of a specific module). */
void print_sort_dependency_list(FILE * stream, struct sort_dependency_list_node * list);


/****** Managing lists of objects **************/

/* initialize the list of objects */
void init_objects();

/* initialize a list of objects */
void init_object_list(struct object_list_node ** first, struct object_list_node ** last);

/* Looks up an object and returns a pointer to its list entry.
   Returns NULL if not found. */
struct object_list_node * lookup_object(char * lookup_name);

/* This inserts a new object into the list of objects */
void insert_object (char * name, int num_arguments, struct string_list_node * arguments);

/* This inserts a new object into a given list of objects */
void insert_object_into_list (struct object_list_node ** first, 
			      struct object_list_node ** last, 
			      char * name, 
			      int num_arguments, 
			      struct string_list_node * arguments);

/* Called when inserting a new object.  Makes a new object */
struct mad_object * make_new_object (char * name, 
				     int num_arguments,
				     struct string_list_node * arguments);

/* Add information about sort of a group of objects.
   This is separate from making a new object because we get this
   information later during the declaration. */
void * update_objects (struct string_list_node * current_entry, char * sort);

/* Checks whether an object is fully instantiated, i.e., has no variables. */
int is_fully_instantiated_object(char * name, 
				 struct mad_constant_argument * arguments);

/* Checks whether an object is fully instantiated, i.e., has no variables.
   when it comes from a given module that is being imported at the moment. */
int is_fully_instantiated_object_in_module_import(char * name, 
						  struct mad_constant_argument * arguments,
						  struct mad_module * module);

/* Makes a copy of a given object list */
struct object_list_node * copy_object_list(struct object_list_node * object_list);

/* Called to merge objects of an imported module with the current objects */
void merge_objects_from_module(struct mad_module * module);

/* for debugging */
void print_current_objects(FILE * stream);

/* for debugging.  
   To print out a specific list of objects (e.g. of a specific module). */
void print_object_list(FILE * stream, struct object_list_node * list);

/* for printing out an object declaration section to be used as ccalc input. */
void print_object_list_for_ccalc(FILE * stream, struct object_list_node * object_list);

/****** Managing lists of constants **************/


/* initialize the list of constants */
void init_constants();

/* initialize a list of constants */
void init_constant_list(struct constant_list_node ** first, struct constant_list_node ** last);

/* Looks up a constant and returns a pointer to its list entry.
   Returns NULL if not found. */
struct constant_list_node * lookup_constant(char * lookup_name);

/* Looks up a constant in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct constant_list_node * lookup_constant_in_module(char * lookup_name, struct mad_module * module);

/* This inserts a new constant into the list of constants */
void insert_constant (char * name, int num_arguments, struct string_list_node * arguments);

/* This inserts a new constant into a given list of constants */
void insert_constant_into_list (struct constant_list_node ** first, 
				struct constant_list_node ** last, 
				char * name, 
				int num_arguments, 
				struct string_list_node * arguments);

/* Called when inserting a new constant.  Makes a new constant */
struct mad_constant * make_new_constant (char * name, int num_arguments,
					 struct string_list_node * arguments);


/* Add information about kind and domain of a group of constants
   This is separate from making a new constant because we get this
   information later during the declaration. */
void update_constants(struct string_list_node * current_entry, int kind, char * domain);

/* Makes a copy of a given constant list */
struct constant_list_node * copy_constant_list(struct constant_list_node * constant_list);

/* Called to merge constants of an imported module with the current constants */
void merge_constants_from_module(struct mad_module * module);

/* for debugging */
void print_current_constants(FILE * stream);

/* for debugging.  
   To print out a specific list of constants (e.g. of a specific module). */
void print_constant_list(FILE * stream, struct constant_list_node * list);

/* for printing out a constant declaration section to be used as ccalc input */
void print_constant_list_for_ccalc(FILE * stream, struct constant_list_node * list);

/****** Managing lists of variables **************/


/* initialize the list of variables */
void init_variables();


/* Looks up a variable and returns a pointer to its list entry.
   Returns NULL if not found. */
struct variable_list_node * lookup_variable(char * lookup_name);

/* Looks up a variable in a particular module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct variable_list_node * lookup_variable_in_module(char * lookup_name, struct mad_module * module);

/* Goes through the variable list and looks for a variable of a
   particular sort.  
   If it finds one, it returns the name,
   otherwise it returns NULL. */
char * find_variable_of_sort(char * sort_to_lookup);

/* Goes through the variable list of a given module and looks for a variable of a
   particular sort.  
   If it finds one, it returns the name,
   otherwise it returns NULL. */
char * find_variable_of_sort_in_module(char * sort_to_lookup, struct mad_module * module);

/* This inserts a new variable into the list of variables */
void insert_variable (char * name);

/* This inserts a new variable into the list of variables of a given module. 
   This is called when new variables are created during the processing of 
   import statements, so we know the sort of the variable. 
   
   Note that this is a little different from insert_variable, because it
   does not call update_id_type at the end.  This is because we don't keep a list
   of identifiers for each module, only one for the module currently being read in.
   So, variables added to a module will not have a corresponding identifier at this
   point.  Later on, when this module is merged with the one which is importing it,
   these variables will be added to the "global" identifiers and their id type
   will be updated.

   IMPORTANT: Now we also call this function after parsing, when making the
   action description definite, where we unify arguments by making new variables.
   Since there is no merging after that point we need to remember to update the
   id type too, after calling this function.
*/
void insert_variable_into_module(struct mad_module * module, char * name, char * sort);

/* Called when inserting a new variable.  Makes a new variable */
struct mad_variable * make_new_variable (char * name);

/* Add information about sort of a group of variables.
   This is separate from making a new variable because we get this
   information later during the declaration. */
void * update_variables (struct string_list_node * current_entry, char * sort);

/* Makes a copy of a given variable list */
struct variable_list_node * copy_variable_list(struct variable_list_node * variable_list);

/* Given a list of strings corresponding to variables, variable_list, 
   returns a new list which is a "parallel copy" of the first, and 
   whose members are obtained by creating new variables
   of the sort corresponding to each variable from variable_list. */
struct string_list_node * make_new_variable_list_and_add_to_module(struct mad_module * module, 
								   struct string_list_node * variable_list);

/* Similar to make_new_variable_list_and_add_to_module but differs in
   that the variables given are not variables of the module being currently
   parsed.  They are variables of the given module. */
struct string_list_node * make_new_variable_list_of_given_module_and_add_to_module(struct mad_module * module,
										   struct string_list_node * variable_list);

/* Called to merge variables of an imported module with the current variables */
void merge_variables_from_module(struct mad_module * module);

/* for debugging */
void print_current_variables(FILE * stream);

/* for debugging.  
   To print out a specific list of variables (e.g. of a specific module). */
void print_variable_list(FILE * stream, struct variable_list_node * list);

/* for printing out a variable declaration section to be used as ccalc input. */
void print_variable_list_for_ccalc(FILE * stream, struct variable_list_node * variable_list);

/* for printing out a variable declaration section to be used as ccalc input.
   This version prints all variables except explicitAction variables.
*/
void print_non_explicit_action_variable_list_for_ccalc(FILE * stream, 
						       struct variable_list_node * variable_list);

/****** Managing lists of renaming clauses **************/

/* initialize the list of renaming clauses */
void init_renaming_clauses();

/* initialize a list of renaming_cases */
void init_renaming_case_list(struct renaming_case_list_node ** list_start,
			     struct renaming_case_list_node ** list_end);

/* initialize the list of renaming clauses for a specific import. */
void init_import_renaming_clauses();

/* Looks up a sort or constant name in the list of renamings 
   currently being processed and returns a pointer to its list entry.
   Returns NULL if not found. 
   (This is used to prevent multiple renamings of the same sort/constant
   during the same import.)
*/
struct renaming_list_node * lookup_renaming(char * lookup_name);

/* Called when inserting a new renaming clause.  Makes a new renaming case */
struct mad_renaming_case * make_new_renaming_case (struct mad_formula * condition, 
						   struct mad_formula * formula);

/* This inserts a new renaming case into a list of renaming cases 
   for a specific import. */
void insert_renaming_case_into_list(struct renaming_case_list_node ** list_start,
				    struct renaming_case_list_node ** list_end,
				    struct mad_renaming_case * new_case);

/* Called when inserting a new renaming clause.  Makes a new renaming clause */
struct mad_renaming_clause * make_new_renaming_clause (int kind, 
						       struct mad_formula * atom, 
						       struct mad_formula * formula,
						       struct string_list_node * free_variables,
						       int num_cases,
						       struct renaming_case_list_node * cases,
						       struct mad_formula * generalized_atom,
						       struct renaming_case_list_node * generalized_atom_cases);

/* This inserts a new renaming clause into the list of renaming clauses */
void insert_renaming_clause (int kind,
			     struct mad_formula * atom,
			     struct mad_formula * formula,
			     struct string_list_node * free_variables,
			     int num_cases,
			     struct renaming_case_list_node * cases,
			     struct mad_formula * generalized_atom,
			     struct renaming_case_list_node * generalized_atom_cases);

/* This inserts a new renaming clause into the list of renaming clauses 
   for a specific import. */
void insert_import_renaming_clause (int kind, 
				    struct mad_formula * atom, 
				    struct mad_formula * formula,
				    struct string_list_node * free_variables,
				    int num_cases,
				    struct renaming_case_list_node * cases,
				    struct mad_formula * generalized_atom,
				    struct renaming_case_list_node * generalized_atom_cases);

/* Makes a copy of a given renaming case list */
struct renaming_case_list_node * copy_renaming_case_list(struct renaming_case_list_node * cases);

/* Makes a copy of a given renaming list */
struct renaming_list_node * copy_renaming_list(struct renaming_list_node * renaming_list);

/* Called to merge renaming clauses of an imported module with the current renamings. */
void merge_renamings_from_module(struct mad_module * module);

/* for debugging */
void print_current_renamings(FILE * stream);

/* for debugging.  
   To print out a specific list of renamings (e.g. of a specific module). */
void print_renaming_list(FILE * stream, struct renaming_list_node * list);

/****** Managing lists of formulas **************/


/* initialize the list of formulas */
void init_formula_list(struct formula_list_node ** first, struct formula_list_node ** last);

/* This inserts a new formula into a list of formulas */
void insert_formula_into_list(struct formula_list_node ** first, 
			      struct formula_list_node ** last, 
			      struct mad_formula * new_formula);


/****** Managing lists of axioms **************/

/* initialize the list of axioms */
void init_axioms();

/* initialize a given list of axioms */
void init_axiom_list(struct axiom_list_node ** first_axiom, struct axiom_list_node ** last_axiom);

/* This inserts a new axiom into the list of axioms */
void insert_axiom (struct mad_axiom * new_axiom);

/* This inserts a new axiom into a given list of axioms */
void insert_axiom_into_list (struct mad_axiom * new_axiom, 
			     struct axiom_list_node ** first_axiom,
			     struct axiom_list_node ** last_axiom);

/* Called when inserting a new axiom.  Makes a new axiom */
struct mad_axiom * make_new_axiom (struct mad_formula * F_formula,
				   struct mad_formula * G_formula,
				   struct mad_formula * H_formula);

/* Called when registering arguments as part of a constant appearing in a formula */
struct mad_constant_argument * make_new_argument (int argument_kind, char * argument_name);

/* Called when a formula is encountered as part of an axiom.  Makes a new formula */
struct mad_formula * make_new_formula (int kind);

/* Checks if two argument structures are exactly the same. */
int are_argument_structures_same(struct mad_constant_argument * arguments1,
				 struct mad_constant_argument * arguments2);

/* Makes a copy of a given argument structure. */
struct mad_constant_argument * copy_arguments(struct mad_constant_argument * arguments);

/* Makes a copy of a single argument, 
   including any sub_arguments (if it is an ARGUMENTED_ARGUMENT), 
   but without copying the next_argument field. */
struct mad_constant_argument * copy_single_argument(struct mad_constant_argument * arguments);

/* Takes an argument structure and renames variables from a given module,
   by prepending an import prefix to all variable occurrences in those arguments. */
void rename_variables_of_module_in_arguments(struct mad_module * module, 
					     struct mad_constant_argument * arguments, 
					     int import_index);

/* Renames occurrences of a given constant in an argument structure, adding
   extra variables if necessary. */
int rename_constant_in_arguments(char * constant_name,
				  struct mad_constant_argument * arguments,
				  int import_index, 
				  struct string_list_node * new_variable_list);

/* Given an argument structure and a list of variables, returns a list of any
   free variables in the argument structure, other than those in the given list.
   Assumes the variables have all been declared.  It won't recognize 
   undeclared variables. 
   Also, we assume that the formula in which the argument structure appears
   has already been checked for being correctly formed. 
   (Hence We don't check if identifiers are valid or not.  In fact, sometimes
   they might not appear to be, for example if we are looking for free variables
   in a constant appearing on the left hand side of an "is" statement.  In such
   a case the identifier for the constant hasn't been added to the list yet.
   However, the variables have, which is all that matters.) */
struct string_list_node * find_free_variables_in_arguments(struct mad_constant_argument * arguments, struct string_list_node * given_variables);

/* Makes a copy of a given formula. */
struct mad_formula * copy_formula(struct mad_formula * formula);

/* Takes a formula and renames variables from a given module,
   by prepending an import prefix to all variable occurrences in that formula. */
void rename_variables_of_module_in_formula(struct mad_module * module, struct mad_formula * formula, int import_index);

/* Renames occurrences of a given constant in a formula, adding
   extra variables if necessary. 
   Returns 1 if it encountered an occurrence of the constant, 0 otherwise.
*/
int rename_constant_in_formula(char * constant_name,
			       struct mad_formula * formula,
			       int import_index, 
			       struct string_list_node * new_variable_list);

/* Given a formula and a list of variables, returns a list of any
   free variables in the formula, other than those in the given list.
   Assumes the variables have all been declared.  It won't recognize 
   undeclared variables. 
   Also, we assume that the formula has already been checked for being 
   correctly formed. 
   (Hence We don't check if identifiers are valid or not.  In fact, sometimes
   they might not appear to be, for example if we are looking for free variables
   in a constant appearing on the left hand side of an "is" statement.  In such
   a case the identifier for the constant hasn't been added to the list yet.
   However, the variables have, which is all that matters.) */
struct string_list_node * find_free_variables_in_formula(struct mad_formula * formula, struct string_list_node * given_variables);

/* Takes an argument structure, and checks if it has
   been formed in a valid way, using identifiers in a manner consistent
   with their declarations. */
int is_a_valid_argument(struct mad_constant_argument * argument);


/* Takes a term and an argument structure, and checks if the term is
   a numerical expression

   Assumes that is_a_valid_term has already been called on the term,
   so the structure is not checked.
*/
int is_a_valid_numerical_expression(char * term, 
				    struct mad_constant_argument * term_arguments);

/* Takes a term and an argument structure, and checks if the term has
   been formed in a valid way, using identifiers in a manner consistent
   with their declarations. 

   The grammar in yaccer.y accepts "sort-name formulas" under terms but
   whether a formula is such a sort-name formula is checked before
   is_a_valid_term is called, so there's no need to check it here.
*/
int is_a_valid_term(char * term, struct mad_constant_argument * term_arguments);

/* Takes a formula and checks if it has been formed in valid way, 
   using identifiers in a manner consistent with their declarations. */
int is_a_valid_formula(struct mad_formula * formula); 

/* Takes an atomic formula and a module, and checks if
   the atomic formula is a valid LHS of a constant
   renaming statement, where the constant being renamed
   comes from the given module. 

   Assumes the formula contains an LHS term which is a constant.
*/
int is_a_valid_constant_renaming_LHS_from_module(struct mad_formula * formula, 
						 struct mad_module * module);

/* Takes an atomic formula and a domain name, and checks if
   the atomic formula is a valid RHS of a constant
   renaming statement, where the constant being renamed
   has the given domain. 
*/
int is_a_valid_constant_renaming_RHS(struct mad_formula * formula, 
				     char * domain);

/* When the LHS constant in a renaming is Boolean, we need to check if
   the constant has variable or object arguments that need to be "generalized".
   This function computes the "generalized atom" and the corresponding renaming cases. */
void obtain_generalization_of_Boolean_renamed_constant(struct mad_formula * renaming_atom,
						       struct mad_module * module,
						       struct mad_formula ** generalized_atom,
						       struct renaming_case_list_node ** cases);

/* Takes a term and an argument structure, and a pointer to a constant_list_node.
   Checks if the term has exactly one constant in it and returns 0 or 1.
   If it returns 1, it also sets the given pointer to point to the constant
   which is the only constant in the term. 
   If the single constant is an action variable, the pointer is set to NULL.
*/
int term_contains_exactly_one_constant(char * term, 
				       struct mad_constant_argument * term_arguments,
				       struct constant_list_node ** single_constant_ptr);

/* Takes a formula, which comes from the head of an axiom,
   and checks if this formula would allow the axiom to be
   a definite causal law.  
   i.e., the formula is either
         a) the negation of an atomic formula with exactly one constant 
            (and that constant is Boolean)
         b) a single atomic formula with at most one constant
	 c) the zero-place connective false

   Assumes the formula has been correctly formed.
*/
int is_a_definite_head_formula(struct mad_formula * formula);

/* Takes two formula kinds and returns the kind of 
   a new formula would have if it were composed of two
   subformulas with the given formula kinds. */
enum formula_kind_type combination_of_two_formula_kinds(enum formula_kind_type kind1,
							enum formula_kind_type kind2);

/* Takes an argument structure and returns a formula kind 
   (i.e., action formula, fluent formula, etc.) for it,
   based on any constants appearing as its arguments.
   This doesn't check for constants in the argument name itself,
   because that will already have been checked when this function
   is called!

   Assumes the argument has been formed validly. */
enum formula_kind_type find_formula_kind_of_argument_arguments(struct mad_constant_argument * argument);

/* Takes a term and an argument structure and returns a formula kind 
   (i.e., action formula, fluent formula, etc.) for it,
   by treating it as if it were a formula composed of a single term.
   (Technically for a term to be a formula there should be something
   it is equal to, but if we just want to determine what how a formula
   kind is affected based on the constants in the term, it doesn't matter
   what the term is equal to.
   
   Assumes the term has been formed validly. */
enum formula_kind_type find_formula_kind_of_term(char * term,
						 struct mad_constant_argument * term_arguments);

/* Takes a formula and returns its kind (i.e., action formula, fluent formula, etc.) 
   Assumes formula has been formed validly. */
enum formula_kind_type find_formula_kind(struct mad_formula * formula);

/* Takes an axiom (with three parts F, G, H) 
   and checks if this would constitute a valid axiom.
   These conditions are checked:
   1. F must be definite
      (In general, F cannot have both fluents and actions, 
      but this condition is satisfied if it's definite.)
   2. if H is NULL, 
        if F is a fluent formula, G must be a fluent formula
        if F is an action formula, G can be anything
   3. if H is not NULL,
        F and G must be fluent formulas
        F must not contain any sdFluents or rigid constants.
   4. if F is has a rigid constant,
        the axiom must not contain any non-rigid constants.

   Assumes the formulas in the axiom have been formed validly.
*/
int is_a_valid_axiom(struct mad_axiom * axiom);

/* Called by expand_formula, to expand arguments. */
struct mad_constant_argument * expand_argument(struct mad_constant_argument * argument,
					       struct formula_list_node ** formula_list_first_ptr,
					       struct formula_list_node ** formula_list_last_ptr);

/* Given a formula, it checks if there are any constants which
   appear as arguments, even though no constant is required.
   (This is shorthand, to stand for "the argument has the value 
   of the constant").  Such arguments are replaced with a variable
   and conjuncts are added specifying that the constant has the
   same value as the variable. 

   Assumes the formula has been formed validly. 
*/
struct mad_formula * expand_formula(struct mad_formula * formula);

struct mad_axiom * copy_axiom(struct mad_axiom * axiom);

/* Makes a copy of a given axiom list */
struct axiom_list_node * copy_axiom_list(struct axiom_list_node * axiom_list);

/* Called to merge axioms of an imported module with the current axioms */
void merge_axioms_from_module(struct mad_module * module);

/* for debugging */
void print_arithmetic_operator_arguments(FILE * stream, struct mad_constant_argument * arguments);

/* for printing out arithmetic operator arguments in an axiom to be used as input for ccalc. */
void print_arithmetic_operator_arguments_for_ccalc(FILE * stream, 
						   struct mad_constant_argument * arguments);

/* for debugging */
void print_constant_arguments(FILE * stream, struct mad_constant_argument * arguments);

/* for printing out constant arguments in an axiom to be used as input for ccalc. */
void print_constant_arguments_for_ccalc(FILE * stream, 
					struct mad_constant_argument * arguments);

// for debugging
void print_arithmetic_operator_term(FILE * stream, 
				    char * term,
				    struct mad_constant_argument * arguments);

// for printing out a term (and arguments) in an axiom to be used as input for ccalc.
void print_arithmetic_operator_term_for_ccalc(FILE * stream, 
					      char * term,
					      struct mad_constant_argument * arguments);

/* for debugging */
void print_formula(FILE * stream, struct mad_formula * formula);

/* for printing out a formula in an axiom to be used as input for ccalc. */
void print_formula_for_ccalc(FILE * stream, struct mad_formula * formula);

/* for debugging */
void print_current_axioms(FILE * stream);

/* for debugging.  
   To print out a specific list of axioms (e.g. of a specific module). */
void print_axiom_list(FILE * stream, struct axiom_list_node * list);

/* for printing out an axiom declaration section to be used as ccalc input */
void print_axiom_list_for_ccalc(FILE * stream, struct axiom_list_node * list);

/****** Managing lists of modules **************/

/* initialize the list of modules */
void init_modules();

/* Looks up a module and returns a pointer to its list entry.
   Returns NULL if not found. */
struct module_list_node * lookup_module(char * lookup_name);

/* Called when inserting a new module.  Makes a new module */
struct mad_module * make_new_module (char * name);

/* This inserts a new module into the list of modules */
void insert_module (char * name);

/* Called at the end of a module description, to update the
   information (sorts, objects, etc.) about the module.  We use
   this to assign the parsed information to a specific module. */
void update_module (char * name);

/* This inserts names of all modules into the list of identifiers */
void insert_module_names_into_identifiers();

/* Makes a copy of a module to be used during importing.
   This is necessary because sort and constant renaming
   statements modify an imported module but we need to
   keep the original copy to be able to import it again 
   later. */
struct mad_module * copy_module(struct mad_module * module);

/* Takes a formula and renames sorts from a given module,
   by replacing them with new sorts, according to sort renaming 
   clauses currently being processed. */
void rename_sorts_of_module_in_formula(struct mad_module * module, struct mad_formula * formula);

/* Renames sorts in a given module, based on renaming clauses */
void rename_sorts_in_module(struct mad_module * module);

/* Renames variables in a given module by prepending an
   import prefix to all variables */
void rename_variables_in_module(struct mad_module * module, int import_index);

/* Renames constants in a given module based on renaming clauses seen.
   It adds an import prefix to constant names and adds extra arguments as necessary. */
void rename_constants_in_module(struct mad_module * module, int import_index);

/* For debugging */
void print_module(FILE * stream, struct mad_module * module);

/* For debugging.
   It prints out the internal lists of the module which is 
   being parsed.  This may be incomplete due to an error 
   during parsing.
*/
void print_current_internal_lists(FILE * stream);

/* for debugging */
void print_current_modules(FILE * stream);


/****** Managing lists of ground_actions **************/

/* initialize the lists of ground actions */
void init_ground_actions();

void init_ground_explicit_actions();

/* Called when inserting a new ground action.  Makes a new ground action */
struct mad_ground_action * make_new_ground_action (char * name,
						   struct string_list_node * arguments,
						   int num_arguments,
						   char * value);

/* This inserts a new ground action into a list of ground actions. */
void insert_ground_action (char * name,
			   struct string_list_node * arguments,
			   int num_arguments,
			   char * value);

void insert_ground_explicit_action (char * name,
				    struct string_list_node * arguments,
				    int num_arguments,
				    char * value);

/* for debugging */
void print_ground_actions(FILE * stream);

void print_ground_explicit_actions(FILE * stream);


/****** Function declarations for trees **************/

/* These aren't used yet.............. */

/* initialize the list of identifiers */
void init_tree();

/****** Function declarations for after parsing ******/

/* Checks if the identifier conversion for ccalc will
   lead to multiple identifiers mapping to the same string
   If so, a warning is given */
void check_module_for_ccalc_identifier_clashes(struct mad_module * module);

/* Prints a ccalc command to show only non-renamed fluents
   in a module
   (i.e. those that are not of the form In.xxx)
 */
void print_ccalc_show_command_for_explicit_constants(FILE * stream, struct mad_module * module);

/* Prints Ccalc input corresponding to a module */
void print_module_as_ccalc_input(FILE * stream, struct mad_module * module);

/* Takes a renaming list and returns a list which is the reverse of the first,
   The actual renaming data is not copied, it is simply re-used. */
struct renaming_list_node * reverse_renaming_list(struct renaming_list_node * list);

/* Given a string_list of variables, it creates a parallel
   list by making new variables of the same sorts, and
   returns a substitution to be used for replacing the old 
   with the new.

   This is called when we want to standardize apart before
   unification, to rename some variables.
*/
struct mad_substitution * get_substitution_to_replace_variables_by_new_variables(struct string_list_node * variables_to_be_replaced);

/* Takes two lists of arguments, which are the same length, and
   sees if it can unify them.  If it can, it will return 1 and
   update the pointer to a substitution containing two argument 
   structures which allow unification with maximal overlap.
*/
int unify_argument_lists(struct mad_constant_argument * arguments1, 
			 struct mad_constant_argument * arguments2,
			 struct mad_substitution ** subst_ptr);

/* Called when making an action description definite.
   It takes a list of arguments to be replaced and a list of arguments to replace the former,
   and an argument structure in which this replacement should be done.

   It is intended for situations in which the argument structure has arguments of its own.
   (i.e. it is an ARGUMENTED_ARGUMENT.)

   The replacement is done in parallel.
*/
struct mad_constant_argument * replace_arguments_with_arguments_in_argument(struct mad_constant_argument * arguments1, 
									    struct mad_constant_argument * arguments2,
									    struct mad_constant_argument * argument);

/* Called when making an action description definite.
   It takes a list of arguments to be replaced and a list of arguments to replace the former,
   and a formula in which this replacement should be done.

   The replacement is done in parallel.
*/
struct mad_formula * replace_arguments_with_arguments_in_formula(struct mad_constant_argument * arguments1,
								 struct mad_constant_argument * arguments2,
								 struct mad_formula * formula);

/* Called when an action description is being made definite.
   It takes 

   - an axiom whose head is the LHS of an "is" statement,
   - a formula to replace that head (either the RHS formula of "is" or a case formula of the "is")
   - the condition upon which this replacement is valid (the conditon from the corresponding case of "is")

   and makes the replacement, returning the new axiom.
*/
struct mad_axiom * replace_formula_in_head_of_axiom_by_formula_upon_condition(struct mad_axiom * axiom,
									      struct mad_formula * formula,
									      struct mad_formula * condition);

/* Turn the nondefinite action description into a definite description:
   - Reverse the renaming list so that the oldest renaming appears first.
   - Go down renaming list, and
     - Use the renaming equivalence to replace occurrences
       of the renamed constant in the heads of non-equivalence axioms
     - Replace the equivalence axiom for the renamed constant by a
       definite law 
       (using Proposition 3 from "Actions as Special Cases" (Erdogan & Lifschitz, 2006)
*/
void make_action_description_definite();


/* Called after reading all input.
   Computes the objects for each sort and updates the sort list
   for the last module, with this info. */
void build_object_lists_for_sorts();

/* for debugging */
void print_objects_of_all_sorts(FILE * stream);

/* Called after reading all input and building the lists of objects for each sort.
   Computes the ground actions (including ground explicit actions). 

   Assumes all sorts have at least one object.
*/
void build_ground_action_lists();

/* Assumes formula has been formed validly and that the variable and
   object are valid.
 */
struct mad_constant_argument * replace_object_variable_with_object_in_argument(char * variable, 
									       char * object,
									       struct mad_constant_argument * argument);

/* Assumes formula has been formed validly and that the variable and
   object are valid.
 */
struct mad_formula * replace_object_variable_with_object_in_formula(char * variable, 
								    char * object,
								    struct mad_formula * formula);

/* Assumes formula has been formed validly. */
struct mad_formula * ground_quantified_objects_in_formula(struct mad_formula * formula);

void ground_quantified_objects_in_axioms();

/* 
   Actually, this will probably never be used, because an action variable cannot be nested
   within an argument. 

   Assumes formula has been formed validly and that the variable and
   action are valid.
 */
struct mad_constant_argument * replace_action_variable_with_ground_action_in_argument(char * variable, 
										      struct mad_ground_action * ground_action,
										      struct mad_constant_argument * argument);

/* 
   Assumes formula has been formed validly and that the variable and
   action are valid.
 */
struct mad_formula * replace_action_variable_with_ground_action_in_formula(char * variable, 
									   struct mad_ground_action * ground_action,
									   struct mad_formula * formula);

/* 
   Assumes that this is called after all quantified objects have been ground.
   (i.e. any quantified variable left must be an action or explicitAction variable.)

    Assumes formula has been formed validly.
 */
struct mad_formula * ground_quantified_actions_in_formula(struct mad_formula * formula);

void ground_quantified_actions_in_axioms();

void build_ground_axiom_list();


#endif /* DATA_TYPES_H */
