/* MAD: Software for the Modular Action Description Language
   Copyright (C) 2008, 2009  Selim T. Erdogan

   This file is part of MAD.

   MAD is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   MAD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with MAD.  If not, see <http://www.gnu.org/licenses/>.


   Contact info: selim@cs.utexas.edu
*/


/* File: flexer.l
   This file is used in generating a simple parsing program for MAD files.
   It contains token descriptions and instructions to be used as input to
   Flex, a lexical analyzer generator.
*/


%{
/* need this for the call to atof() below */
#include <math.h>

/* Not necessary any more: #include "yaccer.tab.h" */
%}

DIGIT      [0-9]
IDENT      [a-zA-Z][a-zA-Z0-9_]*
INT_RANGE  "-"?{DIGIT}+"..""-"?{DIGIT}+

%x comment
%x str

%option header-file="lex.yy.h"


%option yylineno

/* %option reentrant */
%option bison-bridge
%option bison-locations

%%
 /* These will be used later, when we are reading in a quoted string */
 char * string_buf; /* To hold the string */
 char *string_buf_ptr; /* To act as a pointer to a location in the string,
                          as we process it.  */


 /*
 {DIGIT}+    {
            printf( "An integer: %s (%d)\n", yytext,
                   atoi( yytext ) );
            }

 {DIGIT}+"."{DIGIT}*        {
            printf( "A float: %s (%g)\n", yytext,
                    atof( yytext ) );
            }
 */
include|module|import|is|case|numeric_symbol|sorts|inclusions|objects|actions|fluents|variables|action|explicitAction|simple|staticallyDetermined|rigid|axioms|if|after|constraint|default|exogenous|causes|nonexecutable|always|may|cause|inertial|exists|forall|true|false|Boolean        {
            /* decide what to return */ 
            if ( strcmp(yytext, "include") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return INCLUDE;
            }
            else if ( strcmp(yytext, "module") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return MODULE;
            }
            else if ( strcmp(yytext, "import") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return IMPORT;
	    }
            else if ( strcmp(yytext, "is") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return IS;
	    }
            else if ( strcmp(yytext, "case") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return CASE;
	    }
            else if ( strcmp(yytext, "numeric_symbol") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return NUMERIC_SYMBOL;
	    }
            else if ( strcmp(yytext, "sorts") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return SORTS;
	    }
            else if ( strcmp(yytext, "inclusions") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return INCLUSIONS;
	    }
            else if ( strcmp(yytext, "objects") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return OBJECTS;
	    }
            else if ( strcmp(yytext, "actions") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return ACTIONS;
	    }
            else if ( strcmp(yytext, "fluents") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return FLUENTS;
	    }
            else if ( strcmp(yytext, "variables") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return VARIABLES;
	    }
            else if ( strcmp(yytext, "simple") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return SIMPLE;
            }
            else if ( strcmp(yytext, "staticallyDetermined") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return SD_FLUENT;
            }
            else if ( strcmp(yytext, "rigid") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return RIGID;
            }
            else if ( strcmp(yytext, "action") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return ACTION;
            }
            else if ( strcmp(yytext, "explicitAction") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return EXPLICIT_ACTION;
            }
            else if ( strcmp(yytext, "axioms") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return AXIOMS;
	    }
            else if ( strcmp(yytext, "if") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return IF;
	    }
            else if ( strcmp(yytext, "after") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return AFTER;
	    }
            else if ( strcmp(yytext, "constraint") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return CONSTRAINT;
	    }
            else if ( strcmp(yytext, "default") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return DEFAULT;
	    }
            else if ( strcmp(yytext, "exogenous") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return EXOGENOUS;
	    }
            else if ( strcmp(yytext, "causes") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return CAUSES;
	    }
            else if ( strcmp(yytext, "nonexecutable") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return NONEXECUTABLE;
	    }
            else if ( strcmp(yytext, "always") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return ALWAYS;
	    }
            else if ( strcmp(yytext, "rigid") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return RIGID;
	    }
            else if ( strcmp(yytext, "may") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return MAY;
	    }
            else if ( strcmp(yytext, "cause") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return CAUSE;
	    }
            else if ( strcmp(yytext, "inertial") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return INERTIAL;
	    }
            else if ( strcmp(yytext, "exists") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return EXISTS;
	    }
            else if ( strcmp(yytext, "forall") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return FORALL;
	    }
            else if ( strcmp(yytext, "true") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return TRUE;
	    }
            else if ( strcmp(yytext, "false") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return FALSE;
	    }
            else if ( strcmp(yytext, "Boolean") == 0 ) {
               yylval->name = strdup(yytext);
               yylloc->first_line = yylineno;
               return BOOLEAN;
	    }
            else {
            printf( "A keyword: %s\n", yytext );
            }
            }

  /* I used to have {IDENT}(\.{IDENT})* here for identifiers with periods in the names. */
{IDENT}*    { 
               /* I used to update the list of identifiers here, but now this is
                  done when an ID token is seen in by the parser.  This way the
                  list of identifiers is updated only in declaration sections, and not
                  in import or axiom sections. 

                  if ( lookup_identifier(yytext) == NULL ) {
                  insert_identifier(yytext);
	          }
               */
               /* printf("%s, %d: returning identifier token for: %s\n", input_file_name, yylineno, yytext ); */

               // First check if this identifier has been declared to be a numeric symbol
               // and, if so, then return the corresponding integer
               temp_numeric_symbol_list_node = lookup_numeric_symbol(yytext);
               if ( temp_numeric_symbol_list_node != NULL ) {
                yylval->name = strdup(temp_numeric_symbol_list_node->node_data->integer);
                yylloc->first_line = yylineno;
                return INTEGER; 
               }
               else {
                 if ( string_begins_with_an_import_prefix(yytext) ) {
	  	    printf("%s, %d: Identifier %s begins with \"I<integer>\".  This isn't allowed.\n", 
                           input_file_name,
                           yylineno, 
                           yytext);
  	            /* printf("%d: Identifiers can't begin with \"I<integer>\".\n", yylineno); */
                 }
                 yylval->name = strdup(yytext);
                 yylloc->first_line = yylineno;
	         return ID;
               }
            }
"-"?{DIGIT}+ {
                yylval->name = strdup(yytext);
                yylloc->first_line = yylineno;
                return INTEGER; 
             }
".."        {  /* printf(" returning \"..\"\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return TWO_DOTS;
            }

"&"         {  /* printf(" returning an AND\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return AND;
            }
"|"        {  /* printf(" returning an OR\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return OR;
            }
"-"         {  /* printf(" returning a NEG\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return NEG;
            }
"<->"       {  /* printf(" returning an EQUIV\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return EQUIV;
            }

"->"       {  /* printf(" returning an IMPLIES\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return IMPLIES;
            }
"!="        {  /* printf(" returning a NEQ\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return NEQ;
            }
"<<"        {  /* printf(" returning an inclusion operator\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return INCLUSION_OPERATOR;
            }
"<"         {  /* printf(" returning a LESS_THAN\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return LESS_THAN;
            }
"+"         {  /* printf(" returning a PLUS\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return PLUS;
            }
"*"         {  /* printf(" returning a TIMES\n"); */
               yylval->name = malloc(1 * sizeof(char) );
               yylval->name[0] = yytext[0];
               yylloc->first_line = yylineno;
               return TIMES;
            }
%            /* printf( "This was a comment:\n--> %s \n", yytext); */ { BEGIN(comment); }

<comment>.     /* eat up everything within comments */
<comment>\n    { /* printf( "\n") */; BEGIN(INITIAL); }
<comment><<EOF>> { /* printf( "\n") */; BEGIN(INITIAL); } 

\"      { /* We are reading in a a string */ 
          BEGIN(str); 
          yylloc->first_line = yylineno;
        }
 /* The section about <str> was influenced by the flex manual. 
    This version is simplified since we don't care about any special characters in a string. 
 */
<str>\"\n { /* saw closing quote and newline - all done */
            BEGIN(INITIAL);
            *string_buf_ptr = '\0';
            yylval->name = strdup(string_buf);
            /* yylloc is set when we see the first quote.  No need to set it here. */
            free(string_buf);
            return STRING;
          }
<str>\n {
          *string_buf_ptr = '\0';
          /* Since we just saw a newline, the position should be reported as 1 less than yylineno. */
          printf("%s, %d: Newline before end of quoted string.\n", input_file_name, yylineno - 1 );
          exit(EXIT_FAILURE);
         }
<str>[^\n\"]+ {
                char *yptr = yytext;
                string_buf = (char * ) malloc( sizeof(char) * strlen(yytext));
                string_buf_ptr = string_buf;

                while ( *yptr )
                  *string_buf_ptr++ = *yptr++;
              }

[ \t\n]+         /* eat up whitespace except newlines */

.           {
             /* printf( "returning single character token: %s\n", yytext ); */
             yylval->name = malloc(1 * sizeof(char) );
             yylval->name[0] = yytext[0];
             yylloc->first_line = yylineno;
             return yytext[0];
            }
<<EOF>> { /* At the end, reset line number,  in preparation for more files to parse. */;
          /* Check if this is the end of an "include"d file */
          if ( input_file_name_stack_pointer != 0) {
            //fprintf(stdout, "Parsed file %s successfully.\n", input_file_name);
	    fclose(yyin);
	    yypop_buffer_state();
            oldyylineno = yylineno;
            return END_OF_INCLUDE;
          }
          else {
            yylval->name = malloc(1 * sizeof(char) );
            yylval->name[0] = yytext[0];
            yylloc->first_line = yylineno;
            /* This is where we reset it */
            yylineno = 1;
            return yytext[0];
          }
        } 

%%
