/* MAD: Software for the Modular Action Description Language
   Copyright (C) 2008  Selim T. Erdogan

   This file is part of MAD.

   MAD is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   MAD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with MAD.  If not, see <http://www.gnu.org/licenses/>.


   Contact info: selim@cs.utexas.edu
*/


/* File: mparse.c
   This file contains the main part of the parsing program for MAD files.
   It directs the parsing and the subsequent output generation.
*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "data-types.h"
#include "yaccer.tab.h"
#include "lex.yy.h"

int main(int argc, char **argv )
{

  /* To indicate whether we're generating a CCalc file directly
     or asking the user to choose an action. */
  int interactive_mode = 0;  

  char answer[10];

  /* To iterate through the list of modules parsed. */
  struct module_list_node * current;

  int i;


  /* Initialize variables to keep track of input and output files. */
  num_input_files = 0;

  input_file_stream = NULL;
  
  old_input_file_stream = NULL;
  
  output_file_stream = NULL;
  output_file_name = NULL;

  input_file_name_stack_pointer = 0;
  input_line_number_stack_pointer = 0;


  if ( argc <= 1 ) {
    print_error_message(stderr, "Please specify an input file.\n");
    exit(EXIT_FAILURE);
  }
  else { /* This code is unreachable but if we ever decide to 
            take input straight from the standard input, we can activate it. */
    yyin = stdin;
  }

  ++argv, --argc;  /* skip over program name */
  while ( argc > 0 ) {

    /* Check if we are being given the name for an output file */
    if ( strcmp(argv[0], "-o") == 0 ) {

      /* Make sure there are no two such output file names */
      if ( output_file_name != NULL ) {
	print_error_message(stderr, "Only one output file can be specified!  Don't use -o more than once.\n");
	exit(EXIT_FAILURE);
      }

      ++argv, --argc;  /* skip over option argument -o */

      /* The next argument should be name of the output file */
      if ( argc > 0 ) {
	output_file_name = argv[0];
	++argv, --argc;  /* skip over output file name */
      }
      else {
	fprintf(stdout, "No output file name provided after option -o.  Using \"mparse.output\".\n");
	output_file_name = "mparse.output";
      }
    }
    else if ( strcmp(argv[0], "-i") == 0 ) {
      /* We are running in interactive mode instead of
         just creating a CCalc file directly. */
      interactive_mode = 1;

      ++argv, --argc;  /* move to next current argument */
    }
    else {

      if ( num_input_files >= MAX_INPUT_FILES ) {
        print_error_message(stderr, 
			    "Cannot have more than %d input files.  Sorry...\n",
			    MAX_INPUT_FILES);
        exit(EXIT_FAILURE);
      }
      /* Current argument is not an option or output file name so it must be an input file */
      input_file_names[num_input_files] = argv[0];
      num_input_files++;

      ++argv, --argc;  /* move to next current argument */
    }
  }


  /* Initialize all lists */

  init_numeric_symbols();

  init_identifiers();

  init_sorts();
  init_objects();
  init_constants();
  init_variables();
  init_axioms();
  init_modules();
  
  global_import_index = 0;

  /* We can pretend that the last component seen was a module,
     not a sort or inclusion declaration section,
     because this has no effect. */
  last_component_seen = MODULE;

  /* Make sure an input file was specified */
  if ( num_input_files == 0 ) {
    print_error_message(stderr, "Please specify an input file.\n");
    exit(EXIT_FAILURE);
  }

  // Flag in case extra-syntactic errors happen while parsing
  parse_error_encountered = 0;

  i = 0;
  while ( i < num_input_files ) {
    
    input_file_stream = fopen( input_file_names[i], "r" );
    if ( input_file_stream == NULL) {
      print_error_message(stderr, "Couldn't open file %s.  Does it exist?\n", input_file_names[i]);
      exit(EXIT_FAILURE);
    }
    else { /* File opened successfully.  Parse it. */
      
      yyin = input_file_stream;
      input_file_name = input_file_names[i];

      if ( yyparse() != 0 ) {
	/* We are done with this file */
	fclose(yyin);

	print_error_message(stderr, "An error was encountered while parsing file %s.\n", input_file_name);
	fprintf(stdout, "Would you like to see the modules parsed so far? (y/n)  ");
	if (fgets(answer, 10, stdin) == NULL) {
	  fprintf(stderr, "An error occurred when reading your answer. Quitting.\n");
	  exit(EXIT_FAILURE);
	}
	while ( (strcmp(answer, "y\n") != 0) && (strcmp(answer, "n\n") != 0) ) {
	  fprintf(stdout, "Please just type y or n, followed by enter.\n");
	  fprintf(stdout, "Would you like to see the modules parsed so far? (y/n)  ");
	  if (fgets(answer, 10, stdin) == NULL) {
	    fprintf(stderr, "An error occurred when reading your answer.  Quitting.\n");
	    exit(EXIT_FAILURE);
	  }
	}
	if (strcmp(answer, "y\n") == 0) {
	  output_file_stream = stdout;
	  fprintf(output_file_stream, "Modules parsed so far:\n");
	  print_current_modules(output_file_stream);
	  fprintf(output_file_stream, "Internal lists built at the time of error:\n");
	  print_current_internal_lists(output_file_stream);
	}
	exit(EXIT_FAILURE);

      } // end of case when there was a parse error
      else { 
	/* File parsed without errors.  Nothing else to do. */
	if ( parse_error_encountered == 0 ) {
	  fprintf(stdout, "Parsed file %s successfully.\n", input_file_name);
	}
	fclose(yyin);
      }
    }

    /* Move to next input file */
    i++;
  }
  
  if ( parse_error_encountered == 1 ) {
    print_error_message(stderr, "Errors were encountered during parsing.\n");
    exit(EXIT_FAILURE);
  }

  if ( module_list_first == NULL ) {
    print_error_message(stderr, "No modules in action description!\n");
    // exit(EXIT_FAILURE);
  }
  else {
    /* Set last module, since that is the one we are mainly interested in. */
    current = module_list_first;
    while ( current != NULL && current->next != NULL) {
      current = current->next;
    }
    last_module = current->node_data;

    // Before building any object lists or doing grounding,
    // which may change data structures, we make a copy of 
    // the last module, and also copy have our global lists
    // point to this module.
    working_module = copy_module(last_module);
    sort_list_first = working_module->module_sorts;
    inclusion_list_first = working_module->module_inclusions;
    sort_dependency_list_first = working_module->module_sort_dependencies;
    object_list_first = working_module->module_objects;
    constant_list_first = working_module->module_constants;
    variable_list_first = working_module->module_variables;
    axiom_list_first = working_module->module_axioms;
    renaming_list_first = working_module->module_renamings;


    make_action_description_definite();
    fprintf(stdout, "Made nondefinite action description definite.\n");

    build_object_lists_for_sorts();
    // The sort list changed because we added Boolean to the
    //   beginning. Update global list pointer.
    sort_list_first = working_module->module_sorts;
    //print_objects_of_all_sorts(stdout);    
    
    build_ground_action_lists();
    //print_ground_actions(stdout);
    //print_ground_explicit_actions(stdout);
    
    ground_quantified_objects_in_axioms();
    ground_quantified_actions_in_axioms();
    
    build_ground_axiom_list();
    working_module->module_axioms = ground_axiom_list_first;
    //fflush(stdout);


    if ( interactive_mode == 1) {  
      fprintf(stdout, "What would you like to do?\n");
      fprintf(stdout, "1. Print modules parsed.\n");
      fprintf(stdout, "2. Print last module.\n");
      fprintf(stdout, "3. Print last module as CCalc input.\n");
      fprintf(stdout, "4. Quit without printing anything.\n");
      fprintf(stdout, "Type the number of your choice, followed by enter.\n");
      if (fgets(answer, 10, stdin) == NULL) {
	print_error_message(stderr, "An error occurred when reading input. Quitting.\n");
	exit(EXIT_FAILURE);
      }
    }
    else { /* Not interactive, so just print CCalc input. */
      strcpy(answer, "3\n");
    }
    
    while ( (strcmp(answer, "1\n") != 0) 
	    && (strcmp(answer, "2\n") != 0)
	    && (strcmp(answer, "3\n") != 0)
	    && (strcmp(answer, "4\n") != 0) ) {
      fprintf(stdout, "Please just type the number of your choice, followed by enter.\n");
      if (fgets(answer, 10, stdin) == NULL) {
	print_error_message(stderr, "An error occurred when reading input.  Quitting.\n");
	exit(EXIT_FAILURE);
      }
    }
    
    if (strcmp(answer, "4\n") != 0) {
      if ( output_file_name == NULL ) {
	output_file_name = "mparse.output";
      }
      output_file_stream = fopen(output_file_name, "w");
      if ( output_file_stream == NULL ) {
	print_error_message(stderr, "Couldn't open file %s for output.\n", output_file_name);
	exit(EXIT_FAILURE);
      }
      
      if (strcmp(answer, "1\n") == 0) {
	print_current_modules(output_file_stream);
      }
      else if (strcmp(answer, "2\n") == 0) {
	print_module(output_file_stream, working_module);
      }
      else if (strcmp(answer, "3\n") == 0) {
	print_module_as_ccalc_input(output_file_stream, working_module);
      }
      
      fprintf(stdout,"Output printed to file \"%s\".\n", output_file_name);
      
      fclose(output_file_stream);    
    }
  }
  
  exit(EXIT_SUCCESS);
  
}
