/* MAD: Software for the Modular Action Description Language
   Copyright (C) 2008, 2009  Selim T. Erdogan

   This file is part of MAD.

   MAD is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   MAD is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with MAD.  If not, see <http://www.gnu.org/licenses/>.


   Contact info: selim@cs.utexas.edu
*/


/* File: yaccer.y
   This file is used in generating a simple parsing program for MAD files.
   It contains a grammar and semantic actions to be used as input to
   GNU Bison, a parser generator.
*/


%{
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "data-types.h"

// #include "data-manipulation.c"

/* I copied the following from yaccer.tab.h so that YYLTYPE in the
   definition of yyerror below will be recognized properly. */
#if ! defined (YYLTYPE) && ! defined (YYLTYPE_IS_DECLARED)
typedef struct YYLTYPE
{

  int first_line;
  int first_column;
  int last_line;
  int last_column;

} YYLTYPE;

# define YYLTYPE_IS_DECLARED 1
# define YYLTYPE_IS_TRIVIAL 1
#endif

/* For a GLR parser with the "locations" and "pure-parser" options on.  */
void yyerror (YYLTYPE *locp, char const *msg);

/* For a non-GLR parser. */
/* 
void yyerror (char *s)
{
  fprintf (stderr, "%d: %s\n", yyloc, s);
}
*/

%}
%union {
  char * name;   /* For returning character strings (e.g. identifiers).  */
  struct mad_inclusion * inclusion_ptr; /* For returning pointers to inclusions. */
  struct mad_formula * formula_ptr;   /* For returning pointers to formulas.  */
  struct mad_constant_argument * argument_ptr;   /* For returning pointers to arguments
						    of constants appearing in formulas.  */
  struct mad_renaming_case * renaming_case_ptr;  /* For returning pointers to renaming cases */
}

%{
#include "lex.yy.c"
%}

// To be able to debug the parser if it does something weird.
%debug

//for more verbose error-reporting upon syntax errors 
%error-verbose

//%locations 
%pure-parser

// This is to branch on different possibilities (might be needed since import 
// statements may appear anywhere)
%glr-parser

/* declarations */

%token <name> ID "identifier"
%token <name> INTEGER "integer"
%token INCLUDE "include"
%token END_OF_INCLUDE "end_of_include"
%token <name> STRING "string"
%token MODULE "module"
%token IMPORT "import"
%token IS "is"
%token CASE "case"
%token NUMERIC_SYMBOL "numeric symbol"
%token SORTS "sorts"
%token INCLUSIONS "inclusions"
%token OBJECTS "objects"
%token ACTIONS "actions"
%token FLUENTS "fluents"
%token VARIABLES "variables"
%token SIMPLE "simple"
%token SD_FLUENT "sdFluent"
%token RIGID "rigid"
%token <name> ACTION "action"
%token <name> EXPLICIT_ACTION "explicitAction"
%token IF "if"
%token AFTER "after"
%token AXIOMS "axioms"
%token CONSTRAINT "constraint"
%token DEFAULT "default"
%token EXOGENOUS "exogenous"
%token CAUSES "causes"
%token NONEXECUTABLE "nonexecutable"
%token ALWAYS "always"
%token MAY "may"
%token CAUSE "cause"
%token INERTIAL "inertial"
%token EXISTS "exists"
%token FORALL "forall"
%token <name> TRUE "true"
%token <name> FALSE "false"
%token <name> BOOLEAN "boolean"
%type <renaming_case_ptr> single_case_statement
%type <renaming_case_ptr> possible_default_case_statement
%type <renaming_case_ptr> default_case_statement
%type <name> integer_range
%type <name> simple_sort_id object_group_sort_id variable_group_sort_id
%type <inclusion_ptr> inclusion_expression
%type <name> object_id
%type <name> constant_id
%type <name> variable_id
%type <formula_ptr> term
%type <argument_ptr> arithmetic_expression
%type <argument_ptr> constant_arguments
%type <argument_ptr> constant_argument_breakup
%type <argument_ptr> constant_argument_atom
%type <argument_ptr> constant_argument_constant
/* We used to allow allow non-Boolean actions which could have a value
   when they appeared as an argument to a constant.  Now this is prohibited
   so we don't need it anymore.
%type <name> value
*/
%type <formula_ptr> formula atomic_formula inequality
%type <formula_ptr> possible_if_follower

// used to make an integer range
%token TWO_DOTS /* defined in flexer.l as .. */

// operators which may occur in formulas
// precedences increase downwards
%nonassoc QUANTIFIER_PRECEDENCE /* In order to give rules with quantifiers lower
				   precedence than rules for other types of formulas. 
				   This way, "exists a  a=b & c=d" will be parsed as
				     (exists a ((a=b) & (c=d)))
				   and not as
				     ((exists a (a=b)) & (c=d)) */
%left EQUIV   /* defined in flexer.l as <-> */
%left IMPLIES /* defined in flexer.l as ->  */
%left OR      /* defined in flexer.l as |   */
%left AND     /* defined in flexer.l as &   */
%nonassoc NEG /* defined in flexer.l as -   */
%nonassoc NEQ /* defined in flexer.l as !=  */
// the inclusion operator which may occur in inclusion declarations
%left INCLUSION_OPERATOR /* defined in flexer.l as << */

// inequality operators
%left LESS_THAN /* defined in flexer.l as <   */

// arithmetic operators
%left PLUS  /* defined in flexer.l as +   */
%left TIMES /* defined in flexer.l as *   */


%%
action_description : module action_description_breakup
                   | symbol_declaration action_description_breakup
                   | sort_declaration action_description_breakup
                   | sort_inclusion_declaration action_description_breakup
/*		   | /* empty */
                   | included_action_description action_description
                   ;
included_action_description : INCLUDE STRING 
                              {
				/* Push current input stream and line number onto stack, for returning later.
				   Set new input stream to file to be included, and reset line number.
				*/
				input_file_stream = fopen( $2, "r" );
				if ( input_file_stream == NULL) {
				  print_error_message(stderr, "Couldn't open file %s.  Does it exist?\n", $2);
				  exit(EXIT_FAILURE);
				}
				else {
				  push_input_file_name(input_file_name);
				  input_file_name = $2;
				  push_line_no(yylineno);
				  yylineno = 1;
				  yyin = input_file_stream;

				  yypush_buffer_state(yy_create_buffer( yyin, YY_BUF_SIZE ));
				}
			      }
                              action_description END_OF_INCLUDE
                              {
				/* After reading an action description from an include statement,
				   continue this one, where we left off, 
				   by resetting the input buffer and line number. */

				/* No need to pop the buffer state here.
                                   When the lexical analyzer encounters an end-of-file,
				   it will pop the buffer state. */

				//yylineno = pop_line_no();
				yylineno = pop_line_no() + yylineno - oldyylineno;
				input_file_name = pop_input_file_name();
			      }
                            ;
action_description_breakup : module action_description_breakup
                           | symbol_declaration action_description_breakup
                           | sort_declaration action_description_breakup
                           | sort_inclusion_declaration action_description_breakup
                           | /* empty */
                           ;
module : MODULE ID ';'
           {
	     // If we have seen a sort or inclusion declaration section 
	     // right before this one, then we must make sure those declarations
	     // are made global
	     if ( last_component_seen == SORTS ) {
	       init_global_sorts();
	       global_sort_list_first = copy_sort_list(sort_list_first);
	     }
	     if ( last_component_seen == INCLUSIONS ) {
	       init_global_inclusions();
	       global_inclusion_list_first = copy_inclusion_list(inclusion_list_first);
	     }

	     last_component_seen = MODULE;

	     /* When you see a module beginning, get ready to read in a new module */

	     // We can't simply initialize identifiers anymore, because the 
	     // global "ontology" of sorts (which now appear outside of modules)
	     // needs to be added to the identifiers
	     // init_identifiers();
	     //
	     // Instead, we copy those sorts from the global "ontology" lists
	     init_identifiers_with_global_sorts();

	     /* We need to add the module names as identifiers */
	     insert_module_names_into_identifiers();
	     
	     // No need to initialize sorts and inclusions anymore,
	     // since those declarations now appear outside of modules
	     // init_sorts();
	     // init_inclusions();
	     //
	     // Instead, we copy those from the global "ontology" lists
	     copy_global_sorts();
	     copy_global_inclusions();

	     init_sort_dependencies();

	     init_objects();
	     init_constants();
	     init_variables();
	     init_axioms();
	     init_renaming_clauses();
	     
	     /* Initialize index to append to newly created variables. */
	     new_variable_index = 0;
	     
	     if ( string_begins_with_an_import_prefix($2) ) {
	       // !!! Do something
	     }
	     /* Register identifier */
	     if ( lookup_identifier($2) == NULL ) {
	       insert_identifier($2);
	     }

	     // The name of the module being parsed is needed during the rest of the parsing
	     name_of_module_currently_being_parsed = (char *) malloc( sizeof(char) * strlen($2) + 1);
	     strcpy(name_of_module_currently_being_parsed, $2);
	   }
         module_body
           {
	     /* make a new module with this name */
	     if ( lookup_identifier($2)->node_data->id_type == UNINITIALIZED_ID ) {
	       if ( lookup_module($2) == NULL ) {
		 update_id_type($2, MODULE_ID);
		 insert_module($2);
	       }
	       else {
		 print_parser_error_message(stderr, 
					    input_file_name, 
					    @1.first_line, 
					    "Attempt to declare an already-declared module: %s\n", 
					    $2);
	       }
	     }
	     else {
	       print_parser_error_message(stderr, 
					  input_file_name, 
					  @2.first_line, 
					  "Identifier %s was already declared.\n",
					  $2);
	     }

	     /* When done reading in a module, move the current
                information (sorts, objects, etc.) over to the
                that specific module */
	     update_module($2);
	   }
       ;
module_body : possible_import_declarations { }
              possible_object_declaration { }
              possible_action_declaration { }
              possible_fluent_declaration { }
              possible_variable_declaration { }
              possible_axiom_part { }
              ;
possible_import_declarations : import_declaration possible_import_declarations
                            | /* empty */
                            ;
possible_object_declaration : object_declaration possible_import_declarations
                            | /* empty */
                            ;
possible_action_declaration : action_declaration possible_import_declarations
                            | /* empty */
                            ;
possible_fluent_declaration : fluent_declaration possible_import_declarations
                            | /* empty */
                            ;
possible_variable_declaration : variable_declaration possible_import_declarations
                              | /* empty */
                              ;
possible_axiom_part : axiom_part possible_import_declarations
                    | /* empty */
                    ;

import_declaration : IMPORT ID ';'
                     {
		       /* Record the line number of the "import" keyword.  This will serve as the
			  error line number when errors are encountered due to general importing
			  processes like renaming sorts or merging modules. */
		       import_lineno = @1.first_line;
		       
		       @$ = @2;
		       if ( string_begins_with_an_import_prefix($2) ) {
			 // !!! Do something
		       }
		       /* Register identifier */
		       if ( lookup_identifier($2) == NULL ) {
			 print_parser_error_message(stderr, 
					     input_file_name, 
					     @2.first_line, 
					     "Identifier %s not declared.\n",
					     $2);
			 insert_identifier($2);
		       }
		       /* check that module exists */
		       if ( lookup_module($2) == NULL ) {
			 print_parser_error_message(stderr, 
						    input_file_name, 
						    @2.first_line, 
						    "Trying to import nonexistent module %s.\n",
						    $2);
			 print_parser_error_message(stderr, 
						    input_file_name, 
						    @2.first_line, 
						    "Aborting parsing...\n");
			 YYABORT;
		       }
		       else {
			 /* set the module for processing possible renaming statements */
			 temp_module = copy_module(lookup_module($2)->node_data);
 
			 /* make the import index one more than global_import_index */
			 global_import_index =  global_import_index + 1;
			 
			 rename_variables_in_module(temp_module, global_import_index);
			 
			 /* at this point we haven't encountered any constant renaming clauses
			    for this import */
			 constant_renaming_clause_encountered = 0;

			 /* So far no renaming clauses have been seen */
			 init_import_renaming_clauses();

			 invalid_is_statement_part_seen_while_parsing_import = 0;
			 temp_generalized_atom = NULL;
			 temp_generalized_atom_cases = NULL;
		       }
		     }
                     possible_renaming_clauses
                     {
		       if ( lookup_module($2) != NULL ) {
			 
			 /* At this point we are done with parsing an import section.
			    We need to check if we have seen any constant renaming clauses.
			    If we did, all sort renaming clauses would have been processed
			    then.  (Since we need to have the renamed sorts of the module
			    to check for correctness of a constant renaming clause.)
			    If no constant renaming clauses have been seen, we should rename
			    the sorts now. */
			 if ( constant_renaming_clause_encountered == 0 ) {
			   
			   /* Rename sorts */
			   renaming_sorts_leads_to_error = 0;
			   error_lineno = import_lineno;
			   rename_sorts_in_module(temp_module);
			   // If renaming sorts screwed up the sorts, inclusions, or sort
			   // dependencies (by introducing a cyclical inclusion, an undeclared
			   // sort, or a cyclical sort dependency)
			   // that can lead to serious problems later,
			   // so we abort parsing
			   if ( renaming_sorts_leads_to_error == 1 ) {
			     print_parser_error_message(stderr, 
							input_file_name, 
							error_lineno,
							"Aborting parsing...\n");
			     YYABORT;
			   }

			   /* After processing sort renamings, we clear the renaming list. */
			   init_import_renaming_clauses();
			 }
			 
			 /* Now what remains is to rename the constants,
			    insert the equivalence axioms and merge the modules. */
			 rename_constants_in_module(temp_module, global_import_index);

			 /* Prepare for error messages during merging */
			 error_lineno = import_lineno;
			 merge_sorts_from_module(temp_module);
			 merge_inclusions_from_module(temp_module);

			 updating_sort_dependencies_leads_to_an_error = 0;
			 update_sort_dependencies_according_to_inclusions();
			 // If updating sort dependencies acccording to inclusions
			 // screwed up (by introducing a cyclical dependency)
			 // that can lead to serious problems later,
			 // so we abort parsing
			 if ( updating_sort_dependencies_leads_to_an_error == 1 ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      error_lineno,
						      "Aborting parsing...\n");
			   YYABORT;
			 }
			 /* Merge sort dependencies */
			 merging_sort_dependencies_leads_to_error = 0;
			 merge_sort_dependencies_from_module(temp_module);
			 // If merging sort dependencies screwed up
			 // (by introducing a cyclical dependency)
			 // that can lead to serious problems later,
			 // so we abort parsing
			 if ( merging_sort_dependencies_leads_to_error == 1 ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      error_lineno,
						      "Aborting parsing...\n");
			   YYABORT;
			 }
			 merge_objects_from_module(temp_module);
			 merge_constants_from_module(temp_module);
			 merge_variables_from_module(temp_module);
			 merge_axioms_from_module(temp_module);
			 merge_renamings_from_module(temp_module);

		       }
		     }
                   ;

/* A sort or a constant both have the form an atomic_formula.  
   The former without any arguments and no specific value,
   the latter with possible arguments or a value (so in the 
   form of an atom, rather than a term in full generality) */
possible_renaming_clauses : atomic_formula IS formula ';' 
                          {
			    /* We need to check whether the "atom" is actually a sort name or a constant name. */
			    if ( is_a_known_sort_in_module($1->LHS_term, temp_module) 
				 || is_an_integer_range($1->LHS_term) ) {
			      /* It's a sort name, so we check that it is a single term,
				 and that the RHS of the "is" is a declared sort from the module
				 currently being parsed. */
			      if ( strcmp($1->LHS_term, "Boolean") == 0 ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @1.first_line, 
							   "A sort name on the LHS of \"is\" cannot be \"Boolean\".\n");
			      }
			      else if ( strcmp($1->LHS_term, "action") == 0 ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @1.first_line, 
							   "A sort name on the LHS of \"is\" cannot be \"action\".\n");
			      }
			      else if ( is_an_integer_range($1->LHS_term) ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @1.first_line, 
							   "A sort name on the LHS of \"is\" cannot be an integer range.\n");
			      }
			      else if ($1->LHS_arguments != NULL
				  || $1->RHS_term != NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @1.first_line, 
							   "A sort name on the LHS of \"is\" cannot be followed by arguments or a value.\n");
			      }
			      if ( strcmp($3->LHS_term, "Boolean") == 0 ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @3.first_line, 
							   "A sort name on the RHS of \"is\" cannot be \"Boolean\".\n");
			      }
			      else if ( strcmp($3->LHS_term, "action") == 0 ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @3.first_line, 
							   "A sort name on the RHS of \"is\" cannot be \"action\".\n");
			      }
			      else if ( is_an_integer_range($3->LHS_term) ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @3.first_line, 
							   "A sort name on the RHS of \"is\" cannot be an integer range.\n");
			      }
			      else if ( is_an_integer($3->LHS_term) ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
							   input_file_name, 
							   @3.first_line, 
							   "An integer cannot appear on the RHS of a sort renaming statement.\n");
			      }
			      else if ( lookup_identifier($3->LHS_term) == NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				if ( $3->LHS_term != NULL ) {
				  print_parser_error_message(stderr, 
							     input_file_name, 
						      @3.first_line, 
						      "Undeclared identifier %s on the RHS of \"is\".\n",
						      $3->LHS_term);
				}
				else {
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "NULL identifier on the RHS of \"is\".\n");
				}
			      }
			      else if ( !is_a_known_sort($3->LHS_term) ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "Identifier %s on the RHS of \"is\" was not declared as a sort.\n",
						      $3->LHS_term);
			      }
			      if ($3->LHS_arguments != NULL
				  || $3->RHS_term != NULL) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @3.first_line, 
						    "A sort name on the RHS of \"is\" cannot be followed by arguments or a value.\n");
			      }
			      
			      /* The same sort cannot be renamed twice in the same import */
			      if ( lookup_renaming($1->LHS_term) != NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "Sort %s cannot be renamed twice in the same import.\n",
						    $1->LHS_term);
			      }


			      /* All sort renaming clauses must appear before any constant
				 renaming clauses so we make sure that this is the case */
			      if ( constant_renaming_clause_encountered == 1 ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "All sort renaming clauses for an import must occur before constant renaming clauses.\n");
			      }
			      			      
			      if ( !invalid_is_statement_part_seen_while_parsing_import ) {
				/* Add this to the list of import renaming clauses */
				insert_import_renaming_clause(SORT_RENAMING_CLAUSE, $1, $3, NULL, 0, NULL, NULL, NULL);
			      }
			      else {
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "Ill-formed renaming clause.  Ignoring it...\n");
			      }
			    }
			    else if ( lookup_constant_in_module($1->LHS_term, temp_module) != NULL ) {

			      /* All sort renaming clauses must have occurred before the first
				 constant_renaming clause so we process those sort renamings
				 before doing anything about constants.  (This is necessary,
				 since we must check whether a renamed constant is being 
				 used consistently with its declaration, and the declaration 
				 may have changed by renaming sorts.) We do this only before
				 the first constant renaming clause seen.*/
			      if ( constant_renaming_clause_encountered == 0 ) {
				
				/* To indicate that no more sort renamings can occur after this */
				constant_renaming_clause_encountered = 1;
				
				/* Rename sorts */
				renaming_sorts_leads_to_error = 0;
				error_lineno = import_lineno;
				rename_sorts_in_module(temp_module);
				// If renaming sorts screwed up the sorts, inclusions, or
				// sort dependencies (by introducing a cyclical inclusion,
				// an undeclared sort, or a cyclical sort dependency)
				// that can lead to serious problems later,
				// so we abort parsing
				if ( renaming_sorts_leads_to_error == 1 ) {
				  print_parser_error_message(stderr, 
							     input_file_name, 
							     error_lineno,
							     "Aborting parsing...\n");
				  YYABORT;
				}
				
				/* After processing sort renamings, we only need to record constant renamings. */
				init_import_renaming_clauses();
			      }

			      
			      /* We used to allow the constant being renamed to have a value,
				 but that isn't allowed anymore. */
			      if ( $1->RHS_term != NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "A constant name on the LHS of \"is\" cannot be followed by a value.\n");
			      }
			      else {
				error_lineno = @1.first_line;
				if ( !is_a_valid_constant_renaming_LHS_from_module($1, temp_module) ) {
				  /* First check if the LHS and RHS are validly formed 
				     (i.e. see if the identifiers are used in the way declared). */
				  
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "The constant on the LHS of \"is\" is ill-formed.\n");
				}
				/* This used to check if the RHS of "is" is a valid formula, but we
				   don't allow full formulas anymore.  Only atoms, constants or
				   objects are allowed now, depending on the LHS. 
				   
				   error_lineno = @3.first_line;
				   if ( !is_a_valid_formula($3) ) {
  				     invalid_is_statement_part_seen_while_parsing_import = 1;
				     print_parser_error_message(stderr,
				                         input_file_name, 
							 @3.first_line, 
							 "The formula on the RHS of \"is\" is ill-formed.\n");
				 }
				*/
				else {
				  temp_constant_list_node = lookup_constant_in_module($1->LHS_term, temp_module);
				  
				  error_lineno = @3.first_line;
				  if ( !is_a_valid_constant_renaming_RHS($3,
									 temp_constant_list_node->node_data->domain) ) {
				    invalid_is_statement_part_seen_while_parsing_import = 1;
				    print_parser_error_message(stderr, 
							input_file_name, 
							@3.first_line, 
							"The RHS of \"is\" is ill-formed.\n");
				  }
				  else if ( lookup_renaming($1->LHS_term) != NULL ) {
				    /* The same constant cannot be renamed twice in the same import */
				    invalid_is_statement_part_seen_while_parsing_import = 1;
				    print_parser_error_message(stderr, 
							input_file_name, 
							@1.first_line, 
							"Constant %s cannot be renamed twice in the same import.\n",
							$1->LHS_term);
				  }
				  else{
				    
				    /* The code below was used when we allowed a full formula, not
				       just an atom, constant or object on the RHS of a constant
				       renaming statement.  But it continues to be applicable
				       in exactly the same way, so there's no need to change it.
				       
				       If the constant being renamed is
				       - is an action constant, the renaming formula cannot have fluents
				       - is a statically determined fluent, the renaming formula cannot have actions
				       - is a rigid constant, the only constants it can have are rigid constants
				       - is a simple fluent constant, the renaming formula cannot have action constants or statically determined constants (nor rigid constants).
				    */
				    
				    temp_formula_kind = find_formula_kind($3);
				    if ( temp_constant_list_node->node_data->kind == ACTION_CONST ) {
				      if ( temp_formula_kind != CONSTANTLESS_FORMULA
					   && temp_formula_kind != ACTION_FORMULA ) {
					invalid_is_statement_part_seen_while_parsing_import = 1;
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @3.first_line, 
							    "Renaming an action constant using fluent constants.\n");
				      }
				    }
				    else if ( temp_constant_list_node->node_data->kind == SD_FL_CONST ) {
				      if ( temp_formula_kind == FORMULA
					   || temp_formula_kind == ACTION_FORMULA ) {
					invalid_is_statement_part_seen_while_parsing_import = 1;
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @3.first_line, 
							    "Renaming an sdFluent constant using action constants.\n");
				      }
				    }
				    else if ( temp_constant_list_node->node_data->kind == RIGID_CONST ) {
				      if ( temp_formula_kind != CONSTANTLESS_FORMULA
					   && temp_formula_kind != RIGID_FORMULA ) {
					invalid_is_statement_part_seen_while_parsing_import = 1;
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @3.first_line, 
							    "Renaming a rigid constant using non-rigid constants.\n");
				      }
				    }
				    else if ( temp_constant_list_node->node_data->kind == SIMPLE_FL_CONST ) {
				      if ( temp_formula_kind != CONSTANTLESS_FORMULA
					   && temp_formula_kind != SIMPLE_FLUENT_FORMULA ) {
					invalid_is_statement_part_seen_while_parsing_import = 1;
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @3.first_line, 
							    "Renaming a simple fluent constant using non-simple fluent constants or action constants.\n");
				      }
				    }
				  }
			      
				  /* !!!: We should check the semantic condition that the 
				     the renaming clause is an explicit definition. */
				  
				  
				  /* Before finding free variables, or checking that the RHS 
				     is a single atom/constant/object, the RHS formula must be expanded. */
				  /* But there is no need for this if we don't allow a formula with
				     more than one constant.
				     
				     If we uncomment this, there needs to be a call
				     to is_a_valid_formula($3) because otherwise expansion
				     may segfault.

				     temp_formula = expand_formula($3);
				  */
				  temp_formula = copy_formula($3);
				  
				  /* Find the free variables in the LHS and RHS of the is statement */
				  temp_string_list_first = find_free_variables_in_formula($1, NULL);
				  temp_string_list2_first = find_free_variables_in_formula(temp_formula, NULL);
				  
				  /* We allow the LHS to have free variables which are not in
				     the RHS.  So this code is no longer necessary.

				     if ( set_difference_of_string_lists(temp_string_list_first,
				     temp_string_list2_first) 
				     != NULL ) {
				     print_parser_error_message(stderr, 
							 input_file_name, 
							 @1.first_line, 
							 "Left-hand side of is statement has free variables not in right-hand side:\n");
				     fprintf(stderr, "    ");
				     print_string_list(stderr, set_difference_of_string_lists(temp_string_list_first,
				     temp_string_list2_first) );
				     fprintf(stderr, "\n");
				     }
				  */
				  
				  /* Now check to see if the RHS has any free variables not in the
				     LHS.  If so, we need to add additional arguments to the 
				     explicitly defined constant while importing it. 
				     We record any such extra free variables in the renaming clause. */
				  
				  /* If the LHS is a non-Boolean constant c, the RHS is
				     a constant d with the same domain, or an object o.
				     We make a new variable var of the sort of the domain
				     and make the LHS c=var.  Then
				     - if the RHS of the case is an object, we change it to
				       o=var
				     - if it's a constant, we make the RHS d=var.

				  */

				  if ( strcmp(temp_constant_list_node->node_data->domain,
					      "Boolean") != 0 ) {
				    temp_variable = make_new_variable_name(temp_constant_list_node->node_data->domain);
				    /* Add this new variable to the list of identifiers and to the list of variables */
				    insert_identifier(temp_variable);
				    insert_variable(temp_variable);
				    init_string_list(&temp_string_list3_first,
						     &temp_string_list3_last);
				    insert_string_into_list(&temp_string_list3_first,
							    &temp_string_list3_last,
							    temp_variable);
				    update_variables(temp_string_list3_first, temp_constant_list_node->node_data->domain);
				    
				    $1->RHS_term = temp_variable;
				    
				    /* The formula on the RHS of the renaming will be an atomic formula,
				       and the new variable will be in its RHS_term.  If the original formula
				       was just true/false, we change it from a 0-place connective to an atomic
				       formula. */
				    if ( temp_formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
				      if ( temp_formula->connective == TRUE ) {
					temp_formula = make_new_formula(ATOMIC_FORMULA);
					temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("true") + 1);
					strcpy(temp_formula->LHS_term, "true");
				      }
				      else {
					temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("false") + 1);
					strcpy(temp_formula->LHS_term, "false");
				      }
				    }
				    temp_formula->RHS_term = temp_variable;
				  }
				  else { /* The LHS constant is Boolean.  So we check if
					    the constant has variable or object arguments
					    that need to be "generalized" */
				    obtain_generalization_of_Boolean_renamed_constant($1,
										      temp_module,
										      &temp_generalized_atom,
										      &temp_generalized_atom_cases);
				  }
				}
			      }  
			      if ( !invalid_is_statement_part_seen_while_parsing_import ) {
				/* Add this to the list of import renaming clauses */
				insert_import_renaming_clause(CONSTANT_RENAMING_CLAUSE,
							      $1, 
							      temp_formula, 
							      set_difference_of_string_lists(temp_string_list2_first,
											     temp_string_list_first) ,
							      1, /* Just a single case */
							      NULL,
							      temp_generalized_atom,
							      temp_generalized_atom_cases);
			      }
			      else {
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "Ill-formed renaming clause.  Ignoring it...\n");
			      }
			    }
			    else {
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @1.first_line, 
						  "%s in \"is\" statement is not a sort or constant of module %s.\n",
						  $1->LHS_term, 
						  temp_module->name);
			      //fprintf(stderr, "    (Maybe I can't find it because it was renamed.\n");
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @1.first_line, 
						  "Ill-formed renaming clause.  Ignoring it...\n");
			    }
			    
			    /* Get ready for any more "is" statements that might follow. */
			    invalid_is_statement_part_seen_while_parsing_import = 0;
			    temp_generalized_atom = NULL;
			    temp_generalized_atom_cases = NULL;
			  }
                          possible_renaming_clauses
			| atomic_formula IS CASE 
                          {
			    /* We need to check that the "atom" is actually constant name. */
			    if ( is_a_known_sort_in_module($1->LHS_term, temp_module) 
				 || is_an_integer_range($1->LHS_term) ) {
			      invalid_is_statement_part_seen_while_parsing_import = 1;
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @1.first_line, 
						  "A sort renaming cannot have cases.\n");
			    }
			    else if ( lookup_constant_in_module($1->LHS_term, temp_module) != NULL ) {

			      /* All sort renaming clauses must have occurred before the first
				 constant_renaming clause so we process those sort renamings
				 before doing anything about constants.  (This is necessary,
				 since we must check whether a renamed constant is being 
				 used consistently with its declaration, and the declaration 
				 may have changed by renaming sorts.) We do this only before
				 the first constant renaming clause seen.*/
			      if ( constant_renaming_clause_encountered == 0 ) {
				
				/* To indicate that no more sort renamings can occur after this */
				constant_renaming_clause_encountered = 1;
				
				/* Rename sorts */
				error_lineno = import_lineno;
				rename_sorts_in_module(temp_module);
				
				/* After processing sort renamings, we only need to record constant renamings. */
				init_import_renaming_clauses();
			      }
			      
			      
			      /* We used to allow the constant being renamed to have a value,
				 but that isn't allowed anymore. */
			      if ( $1->RHS_term != NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @1.first_line, 
						    "A constant name on the LHS of \"is\" cannot be followed by a value.\n");
			      }
			      else {
				error_lineno = @1.first_line;
				if ( !is_a_valid_constant_renaming_LHS_from_module($1, temp_module) ) {
				  /* First check if the LHS and RHS are validly formed 
				     (i.e. see if the identifiers are used in the way declared). */
				  
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "The constant on the LHS of \"is\" is ill-formed.\n");
				}
				else if ( lookup_renaming($1->LHS_term) != NULL ) {
				  /* The same sort cannot be renamed twice in the same import */
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "Constant %s cannot be renamed twice in the same import.\n",
						      $1->LHS_term);
				}
			      }
				
			      temp_constant_list_node = lookup_constant_in_module($1->LHS_term, temp_module);			      
			      constant_atom_being_renamed = copy_formula($1);
			      
			      /* If the LHS is a non-Boolean constant c, then, in
				 every case, the RHS is 
				 a constant d with the same domain, or an object o.
				 We make a new variable var of the sort of the domain
				 and make the LHS c=var.  Then
				 - if the RHS of the case is an object, we change it to
				 o=var
				 - if it's a constant, we make the RHS d=var.
				 
				 We make this new variable only once, below, and use
				 it in all cases.
			      */
			      
			      /* Finish building this case. */
			      if ( strcmp(temp_constant_list_node->node_data->domain,
					  "Boolean") != 0 ) {
				temp_variable = make_new_variable_name(temp_constant_list_node->node_data->domain);
				/* Add this new variable to the list of identifiers and to the list of variables */
				insert_identifier(temp_variable);
				insert_variable(temp_variable);
				init_string_list(&temp_string_list3_first,
						 &temp_string_list3_last);
				insert_string_into_list(&temp_string_list3_first,
							&temp_string_list3_last,
							temp_variable);
				update_variables(temp_string_list3_first, temp_constant_list_node->node_data->domain);
				
				$1->RHS_term = temp_variable;
			      }
			      else { /* The LHS constant is Boolean.  So we check if
					the constant has variable or object arguments
					that need to be "generalized" */
				obtain_generalization_of_Boolean_renamed_constant($1,
										  temp_module,
										  &temp_generalized_atom,
										  &temp_generalized_atom_cases);
			      }
			      
			      /* Now make the list of renamings */
			      init_renaming_case_list(&temp_renaming_case_list_first,
						      &temp_renaming_case_list_last);
			      
			    }
			    else {
			      invalid_is_statement_part_seen_while_parsing_import = 1;
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @1.first_line, 
						  "%s in \"is\" statement is not a sort or constant of module %s.\n",
						  $1->LHS_term,
						  temp_module->name);
			    }
			  }
                          formula ':' formula ';' 
                          {
			    if ( constant_atom_being_renamed != NULL 
				 && !invalid_is_statement_part_seen_while_parsing_import) {
			      
			      temp_constant_list_node = lookup_constant_in_module(constant_atom_being_renamed->LHS_term, temp_module);

			      /* Check that the condition formula has been formed validly */
			      error_lineno = @5.first_line;
			      if ( !is_a_valid_formula($5) ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @5.first_line, 
						    "Ill-formed condition for a constant renaming case.\n");
			      }
			      else { 
				/* Check that the condition for the case has no constants,
				   and that it doesn't include free variables which are not in
				   the constant atom_being_renamed */
				temp_formula_kind = find_formula_kind($5);
				if ( temp_formula_kind != CONSTANTLESS_FORMULA ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @5.first_line, 
						      "The condition for a constant renaming case cannot include constants.\n");
				}
				else {
				  temp_string_list_first = find_free_variables_in_formula(constant_atom_being_renamed, NULL);
				  temp_string_list2_first = find_free_variables_in_formula($5, NULL);
				  if ( set_difference_of_string_lists(temp_string_list2_first,
								      temp_string_list_first) != NULL ) {
				    invalid_is_statement_part_seen_while_parsing_import = 1;
				    print_parser_error_message(stderr, 
							input_file_name, 
							@5.first_line, 
							"The condition for a constant renaming case cannot include free variables that don't appear on the LHS of \"is\".\n");
				  }
				  else {
				    error_lineno = @7.first_line;
				    if ( !is_a_valid_constant_renaming_RHS($7,
									   temp_constant_list_node->node_data->domain) ) {
				      invalid_is_statement_part_seen_while_parsing_import = 1;
				      print_parser_error_message(stderr, 
							  input_file_name, 
							  @7.first_line, 
							  "The RHS of \"is\" is ill-formed.\n");
				    }
				    else{				  
				    
				      /* The code below was used when we allowed a full formula, not
					 just an atom, constant or object on the RHS of a constant
					 renaming statement.  But it continues to be applicable
					 in exactly the same way, so there's no need to change it.
					 
					 If the constant being renamed is
					 - is an action constant, the renaming formula cannot have fluents
					 - is a statically determined fluent, the renaming formula cannot have actions
					 - is a rigid constant, the only constants it can have are rigid constants
					 - is a simple fluent constant, the renaming formula cannot have action constants or statically determined constants (nor rigid constants).
				      */
				      
				      temp_formula_kind = find_formula_kind($7);
				      if ( temp_constant_list_node->node_data->kind == ACTION_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != ACTION_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @7.first_line, 
							      "Renaming an action constant using fluent constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == SD_FL_CONST ) {
					if ( temp_formula_kind == FORMULA
					     || temp_formula_kind == ACTION_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @7.first_line, 
							      "Renaming an sdFluent constant using action constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == RIGID_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != RIGID_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @7.first_line, 
							      "Renaming a rigid constant using non-rigid constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == SIMPLE_FL_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != SIMPLE_FLUENT_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @7.first_line, 
							      "Renaming a simple fluent constant using non-simple fluent constants or action constants.\n");
					}
				      }
				    }
				    /* At this point, we are done with checking that 
				       this case is validly formed. */
				    
				    temp_formula = copy_formula($7);
				    
				    /* Finish building this case. */
				    if ( strcmp(temp_constant_list_node->node_data->domain,
						"Boolean") != 0 ) {
				      /* The formula on the RHS of the renaming will be an atomic formula,
					 and the new variable will be in its RHS_term.  If the original formula
					 was just true/false, we change it from a 0-place connective to an atomic
					 formula. */
				      if ( temp_formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
					if ( temp_formula->connective == TRUE ) {
					  temp_formula = make_new_formula(ATOMIC_FORMULA);
					  temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("true") + 1);
					  strcpy(temp_formula->LHS_term, "true");
					}
					else {
					  temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("false") + 1);
					  strcpy(temp_formula->LHS_term, "false");
					}
				      }
				      temp_formula->RHS_term = temp_variable;
				    }
				    
				    insert_renaming_case_into_list(&temp_renaming_case_list_first,
								   &temp_renaming_case_list_last,
								   make_new_renaming_case($5, $7));
				  }
				}
			      }
			    }
			  }
                          case_statement_breakup possible_default_case_statement
                          {
			    if ( constant_atom_being_renamed != NULL 
				 && !invalid_is_statement_part_seen_while_parsing_import) {
			      
			      /* If the constant being renamed is non-Boolean, we must
				 have an explicit default case. */
			      temp_constant_list_node = lookup_constant_in_module(constant_atom_being_renamed->LHS_term, temp_module);
			      if ( strcmp(temp_constant_list_node->node_data->domain,
					  "Boolean") != 0
				   && $11 == NULL ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				/* To report the error, we use the line number of the last ';' before cases */
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @8.first_line, 
						    "A non-Boolean constant renaming with cases must include a valid default case.\n");
			      }
			      else {
				if ( $11 == NULL ) { 
				/* The constant being renamed is Boolean but with an implicit 
				   "default : false" case. */
				temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
				temp_formula->connective = TRUE;
				temp_formula2 = make_new_formula(ZERO_PLACE_CONNECTIVE);
				temp_formula2->connective = FALSE;
				insert_renaming_case_into_list(&temp_renaming_case_list_first,
							       &temp_renaming_case_list_last,
							       make_new_renaming_case(temp_formula,
										      temp_formula2));
				}
				else {
				  insert_renaming_case_into_list(&temp_renaming_case_list_first,
								 &temp_renaming_case_list_last,
								 $11);
				}
				
				/* At this point, we have a bunch of validly formed cases
				   (This one and the rest, including the default case).
				   Now we should build the formula resulting from 
				   combining the cases (and the default). */
				
				temp_num_cases = 0;
				/* temp_formula2 will hold the conjunction of the negations of 
				   conditions seen so far. */
				temp_renaming_case_list2 = temp_renaming_case_list_first;
				if ( temp_renaming_case_list2 != NULL ) {
				  temp_num_cases++;
				  
				  /* Our renaming formula so far will be Condition1 -> Formula1 */
				  temp_formula = make_new_formula(BINARY_CONNECTIVE);
				  temp_formula-> connective = IMPLIES;
				  temp_formula->left_formula = temp_renaming_case_list2->node_data->condition;
				  temp_formula->right_formula = temp_renaming_case_list2->node_data->formula;
				  
				  /* Now negate Condition1 to prepare for the next case. */
				  temp_formula2 = make_new_formula(UNARY_CONNECTIVE);
				  temp_formula2-> connective = NEG;
				  temp_formula2->right_formula = temp_renaming_case_list2->node_data->condition;
				  
				  temp_renaming_case_list2 =  temp_renaming_case_list2->next;
				}
				while ( temp_renaming_case_list2 != NULL ) {
				  temp_num_cases++;

				  /* If this is not the default case (which is the last case), then 
				     Our renaming formula so far will be 
				     temp_formula & ( (temp_formula2 & Condition_of_this_case) -> Formula_of_this_case) */
				  if ( temp_renaming_case_list2->next != NULL ) {
				    temp_formula3 = make_new_formula(BINARY_CONNECTIVE);
				    temp_formula3->connective = AND;
				    temp_formula3->left_formula = temp_formula;
				    temp_formula3->right_formula = make_new_formula(BINARY_CONNECTIVE);
				    temp_formula3->right_formula->connective = IMPLIES;
				    temp_formula3->right_formula->left_formula = make_new_formula(BINARY_CONNECTIVE);
				    temp_formula3->right_formula->left_formula->connective = AND;
				    temp_formula3->right_formula->left_formula->left_formula = temp_formula2;
				    temp_formula3->right_formula->left_formula->right_formula = temp_renaming_case_list2->node_data->condition;
				    temp_formula3->right_formula->right_formula = temp_renaming_case_list2->node_data->formula;
				    
				    temp_formula = temp_formula3;
				  }
				  else {
				    /* If this is the default case, then
				       Our renaming formula so far will be 
				       temp_formula & ( temp_formula2 -> Formula_of_this_case) */
				    temp_formula3 = make_new_formula(BINARY_CONNECTIVE);
				    temp_formula3->connective = AND;
				    temp_formula3->left_formula = temp_formula;
				    temp_formula3->right_formula = make_new_formula(BINARY_CONNECTIVE);
				    temp_formula3->right_formula->connective = IMPLIES;
				    temp_formula3->right_formula->left_formula = temp_formula2;
				    temp_formula3->right_formula->right_formula = temp_renaming_case_list2->node_data->formula;				    
				    temp_formula = temp_formula3;
				  }
				  
				  /* Now negate Condition_of_this_case and conjoin it with temp_formula2 */
				  temp_formula3 = make_new_formula(UNARY_CONNECTIVE);
				  temp_formula3-> connective = NEG;
				  temp_formula3->right_formula = temp_renaming_case_list2->node_data->condition;
				  
				  temp_formula4 = make_new_formula(BINARY_CONNECTIVE);
				  temp_formula4-> connective = AND;
				  temp_formula4->left_formula = temp_formula2;
				  temp_formula4->right_formula = temp_formula3;
				  
				  temp_formula2 = temp_formula4;
				  
				  /* We also need to update the condition of this processed case
				     to reflect when it is applicable. */
				  temp_renaming_case_list2->node_data->condition = temp_formula->right_formula->left_formula;
				  
				  
				  temp_renaming_case_list2 =  temp_renaming_case_list2->next;
				  
				}
				
				
				/* Before finding free variables, or checking that the RHS
				   is a single atom/constant/object, the RHS formula must be expanded. */
				/* But there is no need for this if we don't allow a formula with
				   more than one constant on the RHSs.
				   
				   If we uncomment this, there needs to be a call
				   to is_a_valid_formula($3) because otherwise expansion
				   may segfault.

				   temp_formula = expand_formula(temp_formula);
				*/
				
				/* Find the free variables in the LHS and RHS of the is statement */
				temp_string_list_first = find_free_variables_in_formula($1, NULL);
				temp_string_list2_first = find_free_variables_in_formula(temp_formula, NULL);
				
				/* We allow the LHS to have free variables which are not in the RHS.
				   if ( set_difference_of_string_lists(temp_string_list_first,
				   temp_string_list2_first) 
				   != NULL ) {
				   print_parser_error_message(stderr, 
				                       input_file_name, 
						       @1.first_line, 
						       "Left-hand side of is statement has free variables not in right-hand side:\n");
				   fprintf(stderr, "    ");
				   print_string_list(stderr, set_difference_of_string_lists(temp_string_list_first,
				   temp_string_list2_first) );
				   fprintf(stderr, "\n");
				   }
				*/
				
				
				/* Now check to see if the RHS has any free variables not in the
				   LHS.  If so, we need to add additional arguments to the 
				   explicitly defined constant while importing it. 
				   We record any such extra free variables in the renaming clause. */
				
				if ( !invalid_is_statement_part_seen_while_parsing_import ) {
				  /* Add this to the list of import renaming clauses */
				  insert_import_renaming_clause(CONSTANT_RENAMING_CLAUSE,
								$1, 
								temp_formula, 
								set_difference_of_string_lists(temp_string_list2_first,
											       temp_string_list_first) ,
								temp_num_cases,
								temp_renaming_case_list_first,
								temp_generalized_atom,
								temp_generalized_atom_cases);
				}
				else {
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "Ill-formed renaming clause.  Ignoring it...\n");
				}
			      }
			    }
			    else {
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @1.first_line, 
						  "Ill-formed renaming clause.  Ignoring it...\n");
			    }
			    
			    /* Get ready for any more "is" statements that might follow. */
			    invalid_is_statement_part_seen_while_parsing_import = 0;
			    temp_generalized_atom = NULL;
			    temp_generalized_atom_cases = NULL;
			  }
                          possible_renaming_clauses
                        | /* empty */
                        ;

case_statement_breakup : single_case_statement ';' 
                         {
			   insert_renaming_case_into_list(&temp_renaming_case_list_first,
							  &temp_renaming_case_list_last,
							  $1);
			 }
                         case_statement_breakup
                       | /* empty */
                       ;
single_case_statement : CASE formula ':' formula
                        {
			  if ( constant_atom_being_renamed != NULL 
			       && !invalid_is_statement_part_seen_while_parsing_import) {
			    
			    temp_constant_list_node = lookup_constant_in_module(constant_atom_being_renamed->LHS_term, temp_module);

			    /* Check that the condition formula has been formed validly */
			    error_lineno = @2.first_line;
			    if ( !is_a_valid_formula($2) ) {
			      invalid_is_statement_part_seen_while_parsing_import = 1;
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @2.first_line, 
						  "Ill-formed condition for a constant renaming case.\n");
			    }
			    else { 
			      /* Check that the condition for the case has no constants,
				 and that it doesn't include free variables which are not in
				 the constant atom_being_renamed */
			      temp_formula_kind = find_formula_kind($2);
			      if ( temp_formula_kind != CONSTANTLESS_FORMULA ) {
				invalid_is_statement_part_seen_while_parsing_import = 1;
				print_parser_error_message(stderr, 
						    input_file_name, 
						    @2.first_line, 
						    "The condition for a constant renaming case cannot include constants.\n");
			      }
			      else {
				temp_string_list_first = find_free_variables_in_formula(constant_atom_being_renamed, NULL);
				temp_string_list2_first = find_free_variables_in_formula($2, NULL);
				if ( set_difference_of_string_lists(temp_string_list2_first,
								    temp_string_list_first) != NULL ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @2.first_line, 
						      "The condition for a constant renaming case cannot include free variables that don't appear on the LHS of \"is\".\n");
				}
				else {
				  error_lineno = @4.first_line;
				  if ( !is_a_valid_constant_renaming_RHS($4,
									 temp_constant_list_node->node_data->domain) ) {
				    invalid_is_statement_part_seen_while_parsing_import = 1;
				    print_parser_error_message(stderr, 
							input_file_name, 
							@4.first_line, 
							"The RHS of \"is\" is ill-formed.\n");
				    }
				    else{				  
				  
				      /* The code below was used when we allowed a full formula, not
					 just an atom, constant or object on the RHS of a constant
					 renaming statement.  But it continues to be applicable
					 in exactly the same way, so there's no need to change it.
					 
					 If the constant being renamed is
					 - is an action constant, the renaming formula cannot have fluents
					 - is a statically determined fluent, the renaming formula cannot have actions
					 - is a rigid constant, the only constants it can have are rigid constants
					 - is a simple fluent constant, the renaming formula cannot have action constants or statically determined constants (nor rigid constants).
				      */
				      
				      temp_formula_kind = find_formula_kind($4);
				      if ( temp_constant_list_node->node_data->kind == ACTION_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != ACTION_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @4.first_line, 
							      "Renaming an action constant using fluent constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == SD_FL_CONST ) {
					if ( temp_formula_kind == FORMULA
					     || temp_formula_kind == ACTION_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @4.first_line, 
							      "Renaming an sdFluent constant using action constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == RIGID_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != RIGID_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @4.first_line, 
							      "Renaming a rigid constant using non-rigid constants.\n");
					}
				      }
				      else if ( temp_constant_list_node->node_data->kind == SIMPLE_FL_CONST ) {
					if ( temp_formula_kind != CONSTANTLESS_FORMULA
					     && temp_formula_kind != SIMPLE_FLUENT_FORMULA ) {
					  invalid_is_statement_part_seen_while_parsing_import = 1;
					  print_parser_error_message(stderr, 
							      input_file_name, 
							      @4.first_line, 
							      "Renaming a simple fluent constant using non-simple fluent constants or action constants.\n");
					}
				      }
				    }
				  
				  /* At this point, we are done with checking that 
				     this case is validly formed. */
				  
				  temp_formula = copy_formula($4);
				  
				  /* Finish building this case. */
				  if ( strcmp(temp_constant_list_node->node_data->domain,
					      "Boolean") != 0 ) {
				    /* The formula on the RHS of the renaming will be an atomic formula,
				       and the new variable will be in its RHS_term.  If the original formula
				       was just true/false, we change it from a 0-place connective to an atomic
				       formula. */
				    if ( temp_formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
				      if ( temp_formula->connective == TRUE ) {
					temp_formula = make_new_formula(ATOMIC_FORMULA);
					temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("true") + 1);
					strcpy(temp_formula->LHS_term, "true");
				      }
				      else {
					temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("false") + 1);
					strcpy(temp_formula->LHS_term, "false");
				      }
				    }
				    temp_formula->RHS_term = temp_variable;
				  }
				  
				  $$ = make_new_renaming_case($2, temp_formula);
				  
				}
			      }
			    }
			  }
			  else {
			    $$ = NULL;
			  }
			}
                      ; 
possible_default_case_statement : default_case_statement { $$ = $1; }
                                | /* empty */ { $$ = NULL; }
                                ;
default_case_statement : DEFAULT ':' formula ';'
                        {
			  if ( constant_atom_being_renamed != NULL 
			       && !invalid_is_statement_part_seen_while_parsing_import) {
			    
			    temp_constant_list_node = lookup_constant_in_module(constant_atom_being_renamed->LHS_term, temp_module);
			    
			    error_lineno = @3.first_line;
			    if ( !is_a_valid_constant_renaming_RHS($3,
								   temp_constant_list_node->node_data->domain) ) {
			      invalid_is_statement_part_seen_while_parsing_import = 1;
			      print_parser_error_message(stderr, 
						  input_file_name, 
						  @3.first_line, 
						  "The RHS of \"is\" is ill-formed.\n");
			    }
			    else{				  
			      
			      /* The code below was used when we allowed a full formula, not
				 just an atom, constant or object on the RHS of a constant
				 renaming statement.  But it continues to be applicable
				 in exactly the same way, so there's no need to change it.
				 
				 If the constant being renamed is
				 - is an action constant, the renaming formula cannot have fluents
				 - is a statically determined fluent, the renaming formula cannot have actions
				 - is a rigid constant, the only constants it can have are rigid constants
				 - is a simple fluent constant, the renaming formula cannot have action constants or statically determined constants (nor rigid constants).
			      */
			      
			      temp_formula_kind = find_formula_kind($3);
			      if ( temp_constant_list_node->node_data->kind == ACTION_CONST ) {
				if ( temp_formula_kind != CONSTANTLESS_FORMULA
				     && temp_formula_kind != ACTION_FORMULA ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "Renaming an action constant using fluent constants.\n");
				}
			      }
			      else if ( temp_constant_list_node->node_data->kind == SD_FL_CONST ) {
				if ( temp_formula_kind == FORMULA
				     || temp_formula_kind == ACTION_FORMULA ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "Renaming an sdFluent constant using action constants.\n");
				}
			      }
			      else if ( temp_constant_list_node->node_data->kind == RIGID_CONST ) {
				if ( temp_formula_kind != CONSTANTLESS_FORMULA
				     && temp_formula_kind != RIGID_FORMULA ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "Renaming a rigid constant using non-rigid constants.\n");
				}
			      }
			      else if ( temp_constant_list_node->node_data->kind == SIMPLE_FL_CONST ) {
				if ( temp_formula_kind != CONSTANTLESS_FORMULA
				     && temp_formula_kind != SIMPLE_FLUENT_FORMULA ) {
				  invalid_is_statement_part_seen_while_parsing_import = 1;
				  print_parser_error_message(stderr, 
						      input_file_name, 
						      @3.first_line, 
						      "Renaming a simple fluent constant using non-simple fluent constants or action constants.\n");
				}
			      }

				    
			      /* At this point, we are done with checking that 
				 this case is validly formed. */
			      
			      temp_formula = copy_formula($3);
			      
			      /* Finish building this case. */
			      if ( strcmp(temp_constant_list_node->node_data->domain,
					  "Boolean") != 0 ) {
				/* The formula on the RHS of the renaming will be an atomic formula,
				   and the new variable will be in its RHS_term.  If the original formula
				   was just true/false, we change it from a 0-place connective to an atomic
				   formula. */
				if ( temp_formula->formula_kind == ZERO_PLACE_CONNECTIVE ) {
				  if ( temp_formula->connective == TRUE ) {
				    temp_formula = make_new_formula(ATOMIC_FORMULA);
				    temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("true") + 1);
				    strcpy(temp_formula->LHS_term, "true");
				  }
				  else {
				    temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen("false") + 1);
				    strcpy(temp_formula->LHS_term, "false");
				  }
				}
				temp_formula->RHS_term = temp_variable;
			      }
			      
			      /* The default case has condition "true" */
			      temp_formula2 = make_new_formula(ZERO_PLACE_CONNECTIVE);
			      temp_formula2->connective = TRUE;
			      
			      $$ = make_new_renaming_case(temp_formula2, temp_formula);
			      
			    }
			  }
			  else {
			    $$ = NULL;
			  }
			}
                      ;

symbol_declaration : NUMERIC_SYMBOL ID '=' INTEGER
                     {
		       // The identifier should not have been seen before
		       if ( lookup_identifier($2) == NULL ) {
			 // Normally we should also look up if the identifier has been 
			 // declared as a numeric symbol before, 
			 // but we don't because if that had been the case, then the
			 // lexical analyzer would have given us an integer instead of the id!
			 insert_numeric_symbol($2, $4);
		       }
		       else {
			 print_parser_error_message(stderr, 
						    input_file_name, 
						    @2.first_line, 
						    "Attempt to turn an already-declared identifier into a numeric symbol: %s\n",
						    $2);
		       }
		     }

integer_range : INTEGER TWO_DOTS INTEGER
                {
		  $$ = make_integer_range_string($1, $3);
		}

sort_declaration : SORTS 
                   {
		     // If we have seen a sort or inclusion declaration section 
		     // right before this one, then we must make sure those declarations
		     // are made global
		     if ( last_component_seen == SORTS ) {
		       init_global_sorts();
		       global_sort_list_first = copy_sort_list(sort_list_first);
		     }
		     if ( last_component_seen == INCLUSIONS ) {
		       init_global_inclusions();
		       global_inclusion_list_first = copy_inclusion_list(inclusion_list_first);
		     }
		     
		     // We need to set the sorts and inclusions to be used in this section
		     // to those which
		     // have already been declared, by copying from the
		     // global "ontology" lists
		     copy_global_sorts();
		     copy_global_inclusions();
		     
		     last_component_seen = SORTS;
		   }
                   simple_sort_id ';' 
                   {
		     if ( strcmp($3, "Boolean") == 0) {
		       print_parser_error_message(stderr, 
						  input_file_name, 
						  @3.first_line, 
						  "Warning: The sort Boolean is declared internally. (No need to declare it explicitly.)\n");
		     }
		     else {
		       /* Register identifier */
		       if ( lookup_identifier($3) == NULL ) {
			 insert_identifier($3);
		       }
		       if ( lookup_sort($3) == NULL ) {
			 insert_sort($3);
			 /***********************************************
                 		    We'll probably also need to declare special sorts for this sort
			 **********************************************/
		       }
		       else {
			 print_parser_error_message(stderr, 
						    input_file_name, 
						    @3.first_line, 
						    "Attempt to declare an already-declared sort: %s\n",
						    $3);
		       }
		     }
		   }
                   sort_declaration_breakup 
                   { 
                     /* fprintf("Done with a sorts declaration\n"); 
			print_current_sorts(stdout); 
			fprintf(stdout, "\n"); */
		   }
                 ;
sort_declaration_breakup : simple_sort_id ';' 
                           {
			     if ( strcmp($1, "Boolean") == 0) {
			       print_parser_error_message(stderr, 
						   input_file_name, 
						   @1.first_line, 
						   "Warning: The sort Boolean is declared internally. (No need to declare it explicitly.)\n");
			     }
			     else {
			       /* Register identifier */
			       if ( lookup_identifier($1) == NULL ) {
				 insert_identifier($1);
			       }
			       if ( lookup_sort($1) == NULL ) {
				 insert_sort($1);
				 /***********************************************
                 		    We'll probably also need to declare special sorts for this sort
				 **********************************************/
			       }
			       else {
				 print_parser_error_message(stderr, 
						     input_file_name, 
						     @1.first_line, 
						     "Attempt to declare an already-declared sort: %s\n",
						     $1);
			       }
			     }
			   }
                           sort_declaration_breakup
                         | /* empty */
                         ;
simple_sort_id : ID 
                 { 
		   if ( string_begins_with_an_import_prefix($1) ) {
		     // !!! Do something
		   }
		   $$ = $1; 
		 }
               | BOOLEAN { $$ = $1; }
               ;


sort_inclusion_declaration : INCLUSIONS 
                             {
		               // If we have seen a sort or inclusion declaration section 
			       // right before this one, then we must make sure those declarations
			       // are made global
			       if ( last_component_seen == SORTS ) {
				 init_global_sorts();
				 global_sort_list_first = copy_sort_list(sort_list_first);
			       }
			       if ( last_component_seen == INCLUSIONS ) {
				 init_global_inclusions();
				 global_inclusion_list_first = copy_inclusion_list(inclusion_list_first);
			       }
		     
			       // We need to set the sorts and inclusions to be used in this section
			       // to those which
			       // have already been declared, by copying from the
			       // global "ontology" lists
			       copy_global_sorts();
			       copy_global_inclusions();
			       
			       last_component_seen = INCLUSIONS;
			     }
                             inclusion_declaration_breakup
                             { 
			       /* fprintf(stdout, "Done with an inclusion declaration\n"); 
				  print_inclusion_list(stdout); 
				  fprintf(stdout, "\n"); */
			     }
                           ;
inclusion_declaration_breakup : inclusion_expression ';' inclusion_declaration_breakup { }
                              | /* empty */
                              ;
/* For every inclusion_expression, we return an inclusion -- a pair of sorts */
inclusion_expression : simple_sort_id INCLUSION_OPERATOR inclusion_expression
                       {
			 if ( !is_a_known_object_sort($1) ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @1.first_line, 
					       "Reference to undeclared sort %s.\n", 
					       $1);
			 }
			 else if ( !is_a_known_object_sort($3->subsort) ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @3.first_line, 
					       "Reference to undeclared sort %s.\n",
					       $3->subsort);
			 }
			 else if ( strcmp($3->subsort, "Boolean") == 0) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @3.first_line, 
					       "Boolean cannot be declared as a supersort.\n");
			 }
			 /* Since $3 is an inclusion, we look at its subsort to get the
			    sort which is declared to be a supersort of $1. */
			 else if ( strcmp($1, $3->subsort) == 0 ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @1.first_line, 
					       "%s is declared as a subsort of itself.\n",
					       $1);
			 }
			 else if ( lookup_inclusion($3->subsort, $1) != NULL ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @1.first_line, 
					       "Warning: %s << %s is an already-declared inclusion.\n",
					       $1,
					       $3->subsort);
			 }
			 /* Check that adding this inclusion won't lead 
			    to a cycle in the inclusion graph. */
			 // We already checked above that both arguments to is_a_subsort_of
			 // are declared sorts or Boolean.
			 else if ( is_a_subsort_of($3->subsort, $1) ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "Warning: %s is already a subsort of %s.\n",
						      $3->subsort,
						      $1);
			 }
			 else {
			   insert_inclusion($3->subsort, $1);
			 }
			 $$ = make_new_inclusion($1, $3->supersort);
		       }
                     | simple_sort_id INCLUSION_OPERATOR simple_sort_id
                       {
			 if ( !is_a_known_object_sort($1) ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @1.first_line, 
					       "Reference to undeclared sort %s.\n",
					       $1);
			 }
			 else if ( !is_a_known_object_sort($3) ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @3.first_line, 
					       "Reference to undeclared sort %s.\n",
					       $3);
			 }
			 else if ( strcmp($3, "Boolean") == 0) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @3.first_line, 
					       "Boolean cannot be declared as a supersort.\n");
			 }
			 /* Check that both sorts are not the same */
			 else if ( strcmp($1, $3) == 0 ) {
			   print_parser_error_message(stderr, 
					       input_file_name, 
					       @1.first_line, 
					       "%s is declared as a subsort of itself.\n",
					       $1);
			 }
			 else if ( lookup_inclusion($3, $1) != NULL ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "Warning: %s << %s is an already-declared inclusion.\n",
						      $1,
						      $3);
			 }
			 /* Check that adding this inclusion won't lead 
			    to a cycle in the inclusion graph. */
			 // We already checked above that both arguments to is_a_subsort_of
			 // are declared sorts or Boolean.
			 else if ( is_a_subsort_of($3, $1) ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line, 
						      "Warning: %s is already a subsort of %s.\n",
						      $3,
						      $1);
			 }
			 else {
			   insert_inclusion($3, $1);
			 }
			 $$ = make_new_inclusion($3, $1);
		       }
                     ;

object_declaration : OBJECTS 
                     { 
		       /* prepare to declare a group of objects
			  of the same sort */
		       init_string_list(&temp_string_list2_first,
					&temp_string_list2_last);
		       init_object_list(&temp_object_list_node,
					&temp_object_list_node2);
		     }
                     object_specification ';'
                     {
		       /* prepare for another group of objects with
			  the same sort */
		       init_string_list(&temp_string_list2_first,
					&temp_string_list2_last);
		       init_object_list(&temp_object_list_node,
					&temp_object_list_node2);
                     }
                     object_specification_breakup
                     { 
		       /* fprintf(stdout, "Done with an objects declaration\n");
			  print_current_objects(stdout); 
			  fprintf(stdout, "\n"); */
		     }
                   ;
object_specification_breakup : object_specification ';'
                               {
				 /* prepare for another group of objects with
				    the same sort */
				 init_string_list(&temp_string_list2_first,
						  &temp_string_list2_last);
				 init_object_list(&temp_object_list_node,
						  &temp_object_list_node2);
				 
			       }
                               object_specification_breakup { }
                             | /* empty */
                             ;
object_specification : object_id_group ':' object_group_sort_id
                       {
			 /* Now we have a list of objects with arguments
			    and we have a sort for all those objects.
			    We will insert the valid objects into
			    the list of objects of this module and then
			    update their sort accordingly.
			 */
			 
			 /* We start with the first object in the list */
			 
			 temp_object_list_node3 = temp_object_list_node;
			 while ( temp_object_list_node3 != NULL ) {
			   if ( lookup_object(temp_object_list_node3->node_data->name) == NULL ) {			       
			     /* insert the object with its name and its list of arguments */
			     insert_object(temp_object_list_node3->node_data->name,
					   temp_object_list_node3->node_data->num_arguments,
					   temp_object_list_node3->node_data->arguments);
			     
			     /* Add object to the group of objects declared together, 
				which will have the same sort */
			     insert_string_into_list(&temp_string_list2_first,
						     &temp_string_list2_last,
						     temp_object_list_node3->node_data->name);

			     // Now add the dependencies this object brings
			     updating_sort_dependencies_leads_to_an_error = 0;
			     update_sort_dependencies_according_to_object_declaration(temp_object_list_node3->node_data->arguments,
										      $3);
			     // If updating sort dependencies according to object declarations
			     // screwed up (by introducing a cyclical dependency)
			     // that can lead to serious problems later,
			     // so we abort parsing
			     if ( updating_sort_dependencies_leads_to_an_error == 1 ) {
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  error_lineno,
							  "Aborting parsing...\n");
			       YYABORT;
			     }
			   }
			   else {
			     print_parser_error_message(stderr, 
							input_file_name, 
							@1.first_line, 
							"Attempt to declare an already-declared object: %s\n",
							temp_object_list_node3->node_data->name);
			   }
			   
			   temp_object_list_node3 = temp_object_list_node3->next;
			 }
			 
			 update_objects(temp_string_list2_first, $3);

			 error_lineno = @1.first_line;
			 updating_sort_dependencies_leads_to_an_error = 0;
			 update_sort_dependencies_according_to_inclusions();
			 // If updating sort dependencies acccording to inclusions
			 // screwed up (by introducing a cyclical dependency)
			 // that can lead to serious problems later,
			 // so we abort parsing
			 if ( updating_sort_dependencies_leads_to_an_error == 1 ) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      error_lineno,
						      "Aborting parsing...\n");
			   YYABORT;
			 }
		       }
                     ;
object_id_group : object_id object_id_breakup { }
                  ;
object_id_breakup : ',' object_id object_id_breakup { }
                  | /* empty */
                  ;
object_id : ID '(' 
              { 
		if ( string_begins_with_an_import_prefix($1) ) {
		  // !!! Do something
		}
		/* Register identifier */
		if ( lookup_identifier($1) == NULL ) {
		  insert_identifier($1);
		}

		/* since we know it is an object with arguments
		   we can start counting its arguments */
		temp_num_arguments = 0;
		/* Make a temporary list of strings to hold the arguments */
		init_string_list(&temp_string_list_first,
				 &temp_string_list_last);
	      }
              object_id_declaration_arguments ')' 
              { 
		/* first we check if this was already declared */
		if ( lookup_identifier($1)->node_data->id_type == UNINITIALIZED_ID ) {
		  /* insert the object (with its name and its list of arguments) into
		     a temporary list of objects -- all objects which are declared
		     as part of the same object specification.  We will update their
		     sort at the end of the specification. 
		  */
		  insert_object_into_list(&temp_object_list_node,
					    &temp_object_list_node2,
					    $1,
					    temp_num_arguments, 
					    temp_string_list_first);
		  $$ = $1;
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      @1.first_line, 
				      "Identifier %s was already declared.\n",
				      $1);
		}
	      }
            | ID 
              {
		if ( string_begins_with_an_import_prefix($1) ) {
		  // !!! Do something
		}
		/* Register identifier */
		if ( lookup_identifier($1) == NULL ) {
		  insert_identifier($1);
		}

		/* first we check if this was already declared */
		if ( lookup_identifier($1)->node_data->id_type == UNINITIALIZED_ID ) {
		    insert_object_into_list(&temp_object_list_node,
					      &temp_object_list_node2,
					      $1,
					      0,
					      NULL);
		    $$ = $1;
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      @1.first_line, 
				      "Identifier %s was already declared.\n",
				      $1);
		}
	      }
            ;
object_id_declaration_arguments : object_id_declaration_argument object_declaration_argument_breakup 
object_declaration_argument_breakup : ',' object_id_declaration_argument object_declaration_argument_breakup
                                      | /* empty */
                                      ;
object_id_declaration_argument : integer_range
                                 {
				   error_lineno = @1.first_line;
				   // Check that the range is valid
				   if ( !is_a_valid_integer_range($1) ) {
				     print_parser_error_message(stderr, 
								input_file_name, 
								@1.first_line,
								"Invalid integer range \"%s\".\n",
								$1);
				   }
				   /* increment number of arguments */
				   temp_num_arguments = temp_num_arguments + 1;
				   /* And add first argument to the list of strings */
				   insert_string_into_list(&temp_string_list_first,
							   &temp_string_list_last,
							   $1);
				 }
                               | simple_sort_id 
                                 {
				   if ( !is_a_known_constant_argument_sort($1) ) { 
				     print_parser_error_message(stderr, 
								input_file_name, 
								@1.first_line,
								"Reference to undeclared sort %s.\n",
								$1);
				   }
				   /* increment number of arguments */
				   temp_num_arguments = temp_num_arguments + 1;
				   /* And add first argument to the list of strings */
				   insert_string_into_list(&temp_string_list_first,
							   &temp_string_list_last,
							   $1);
				 }
                               ;

object_group_sort_id : simple_sort_id 
                       { 
		         if ( strcmp($1, "Boolean") == 0) {
			   print_parser_error_message(stderr, 
						      input_file_name, 
						      @1.first_line,
						      "Only \"true\" or \"false\" can be objects of sort Boolean.\n");
			   
			   //$$ = $1;
			 }
			 else {
			   if ( lookup_sort($1) == NULL ) {
			     print_parser_error_message(stderr, 
						 input_file_name, 
						 @1.first_line,
						 "Reference to undeclared sort %s.\n",
						 $1);
			   }
			   else {
			     $$ = $1;
			   }
			 }
		       }
                     ;


action_declaration : ACTIONS
                       { 
			 /* prepare to declare a group of actions
			    with the same kind and domain */
			 init_string_list(&temp_string_list2_first,
					  &temp_string_list2_last);
			 init_constant_list(&temp_constant_list_node,
					    &temp_constant_list_node2);

		       }
                       action_specification
                       { 
			 /* fprintf(stdout, "Done with an action declaration\n");
			    print_current_constants(stdout); 
			    fprintf(stdout, "\n"); */
		       }
                     ;
fluent_declaration : FLUENTS
                       { 
			 /* prepare to declare a group of constants
			    with the same kind and domain */
			 init_string_list(&temp_string_list2_first,
					  &temp_string_list2_last);
			 init_constant_list(&temp_constant_list_node,
					    &temp_constant_list_node2);
		       }
                       fluent_specification ';' 
                       {
			 /* prepare to declare a group of fluents
			    with the same kind and domain */
			 init_string_list(&temp_string_list2_first,
					  &temp_string_list2_last);
			 init_constant_list(&temp_constant_list_node,
					    &temp_constant_list_node2);
		       }
                       fluent_specification_breakup
                       { 
			 /* fprintf(stdout, "Done with a fluent declaration\n");
			    print_current_constants(stdout); 
			    fprintf(stdout, "\n"); */
		       }
                     ;
fluent_specification_breakup : fluent_specification ';' 
                                 {
				   /* prepare for another group of fluents with
				      the same kind and domain */
				   init_string_list(&temp_string_list2_first,
						    &temp_string_list2_last);
				   init_constant_list(&temp_constant_list_node,
						      &temp_constant_list_node2);
				 }
                                 fluent_specification_breakup { }
                               | /* empty */
                               ;
action_specification : action_id_group
                         {
			   // Action specifications specify action constants
			   // whose domain is Boolean.  
			   // (We don't allow multi-valued actions.)
			   temp_kind = ACTION_CONST;
			   temp_domain = "Boolean";

			   /* Now we have a list of actions with arguments .  
			      We will insert the valid actions into the list
			      of constants of this module and then update their
			      kind and domain (as action and Boolean, 
			      respectively) accordingly.
 
			      One special case that must be checked is as
			      follows:
			      if the kind is "action", then none of the
			      constants can have argument of type action.

			      (Otherwise, imagine two actions a(action) and b.  
			      When we ground, would get b, a(b), a(a(b)), etc., 
			      never ending.)
			   */
			   
			   /* We start with the first constant in the list */

			   temp_constant_list_node3 = temp_constant_list_node;
			   while ( temp_constant_list_node3 != NULL ) {
			     if ( lookup_constant(temp_constant_list_node3->node_data->name) == NULL ) {			       
			       bad_constant_seen_in_declaration = 0;
			       /* Go through arguments for actions and 
				  check if they have action arguments */
			       temp_string_list_first = temp_constant_list_node3->node_data->arguments;
			       while ( temp_string_list_first != NULL ) {
				 if ( strcmp(temp_string_list_first->node_data, "action") == 0 ) {
				   print_parser_error_message(stderr, 
							      input_file_name, 
							      @1.first_line, 
							      "Attempt to declare action constant %s with action argument.\n",
							      temp_constant_list_node3->node_data->name);
				   bad_constant_seen_in_declaration = 1;
				   break;
				 }
				 temp_string_list_first = temp_string_list_first->next;
			       }
			       
			       if ( bad_constant_seen_in_declaration == 0 ) {
				 /* insert the action with its name and 
				    its list of arguments */
				 insert_constant(temp_constant_list_node3->node_data->name,
						 temp_constant_list_node3->node_data->num_arguments,
						 temp_constant_list_node3->node_data->arguments);
				 
				 /* Add action to the group of actions declared 
				    together. */
				 insert_string_into_list(&temp_string_list2_first,
							 &temp_string_list2_last,
							 temp_constant_list_node3->node_data->name);
			       }
			       else {
				 print_parser_error_message(stderr, 
							    input_file_name, 
							    @1.first_line, 
							    "Ignoring declaration of action constant %s.\n",
							    temp_constant_list_node3->node_data->name);
			       }
			     }
			     else {
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  @1.first_line, 
							  "Attempt to declare an already-declared constant: %s\n",
							  temp_constant_list_node3->node_data->name);
			     }
			     
			     temp_constant_list_node3 = temp_constant_list_node3->next;
			   }
			   
			   update_constants(temp_string_list2_first, temp_kind, temp_domain);
			 }
                       ;
fluent_specification : constant_id_group ':' fluent_kind_and_domain
                         {
			   /* Now we have a list of fluents with arguments
			      and we have a kind and domain for all those
			      fluents.  We will insert the valid fluents into
			      the list of constants of this module and then
			      update their kind and domain accordingly.
			   */
			   
			   /* We start with the first constant in the list */

			   temp_constant_list_node3 = temp_constant_list_node;
			   while ( temp_constant_list_node3 != NULL ) {
			     if ( lookup_constant(temp_constant_list_node3->node_data->name) == NULL ) {			       
			       /* insert the constant with its name and its list of arguments */
			       insert_constant(temp_constant_list_node3->node_data->name,
					       temp_constant_list_node3->node_data->num_arguments,
					       temp_constant_list_node3->node_data->arguments);
			       
			       /* Add constant to the group of constants declared together, 
				  which will have the same kind and domain */
			       insert_string_into_list(&temp_string_list2_first,
						       &temp_string_list2_last,
						       temp_constant_list_node3->node_data->name);
			     }
			     else {
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  @1.first_line, 
							  "Attempt to declare an already-declared constant: %s\n",
							  temp_constant_list_node3->node_data->name);
			     }
			     
			     temp_constant_list_node3 = temp_constant_list_node3->next;
			   }
			   
			   update_constants(temp_string_list2_first, temp_kind, temp_domain);
			 }
                       ;
action_id_group : constant_id ';' action_id_breakup { }
                ;
action_id_breakup : constant_id ';' action_id_breakup { }
                  | /* empty */                  
                  ;
constant_id_group : constant_id constant_id_breakup { }
                  ;
constant_id_breakup : ',' constant_id constant_id_breakup { }
                    | /* empty */                  
                    ;
constant_id : ID '(' 
              { 
		if ( string_begins_with_an_import_prefix($1) ) {
		  // !!! Do something
		}
		/* Register identifier */
		if ( lookup_identifier($1) == NULL ) {
		  insert_identifier($1);
		}

		/* since we know it is a constant with arguments
		   we can start counting its arguments */
		temp_num_arguments = 0;
		/* Make a temporary list of strings to hold the arguments */
		init_string_list(&temp_string_list_first,
				 &temp_string_list_last);
	      }
              constant_id_declaration_arguments ')' 
              { 
		/* first we check if this was already declared */
		if ( lookup_identifier($1)->node_data->id_type == UNINITIALIZED_ID ) {

		  /*
		  if ( lookup_constant($1) == NULL ) {
		    // Add constant to the group of constants declared together, 
		    //   which will have the same kind and domain
		    insert_string_into_list(&temp_string_list2_first,
					    &temp_string_list2_last,
					    $1);
		  */
		  
		  /* insert the constant (with its name and its list of arguments) into
		     a temporary list of constants -- all constants which are declared
		     as part of the same constant specification.  We will update their
		     kind and domain at the end of the specification. 
		  */
		  insert_constant_into_list(&temp_constant_list_node,
					    &temp_constant_list_node2,
					    $1,
					    temp_num_arguments, 
					    temp_string_list_first);
		  $$ = $1;
		    
		  //}
		  //else {
		  //print_parser_error_message(stderr, 
		  //	      input_file_name, 
		  //	      @1.first_line, 
		  //	      "Attempt to declare an already-declared constant: %s\n",
		  //	      $1);
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      @1.first_line, 
				      "Identifier %s was already declared.\n",
				      $1);
		}
	      }
            | ID 
              {
		if ( string_begins_with_an_import_prefix($1) ) {
		  // !!! Do something
		}
		/* Register identifier */
		if ( lookup_identifier($1) == NULL ) {
		  insert_identifier($1);
		}

		/* first we check if this was already declared */
		if ( lookup_identifier($1)->node_data->id_type == UNINITIALIZED_ID ) {
		  /* if ( lookup_constant($1) == NULL ) {
		     // Add constant to the group of constants 
		     //  declared together, which will have the same
		       // sort
		    insert_string_into_list(&temp_string_list2_first,
					    &temp_string_list2_last,
					    $1);
		  */
		    insert_constant_into_list(&temp_constant_list_node,
					      &temp_constant_list_node2,
					      $1,
					      0,
					      NULL);
		    $$ = $1;
		    //}
		    //else {
		    //print_parser_error_message(stderr,
		    //		input_file_name, 
		    //		@1.first_line, 
		    //		"Attempt to declare an already-declared constant: %s\n",
		    //		$1);
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      @1.first_line, 
				      "Identifier %s was already declared.\n",
				      $1);
		}
	      }
            ;
constant_id_declaration_arguments : constant_id_declaration_argument constant_declaration_argument_breakup 
constant_declaration_argument_breakup : ',' constant_id_declaration_argument constant_declaration_argument_breakup
                                      | /* empty */
                                      ;
constant_id_declaration_argument : ACTION
                                    {
				      /* increment number of arguments */
				      temp_num_arguments = temp_num_arguments + 1;
				      /* And add first argument to the list of strings */
				      insert_string_into_list(&temp_string_list_first,
							      &temp_string_list_last,
							      $1);

				      /* We shouldn't allow explicitAction as the sort of constant argument.
					 Imagine we did.  Then a formula will be correctly formed if the 
					 argument for such a constant is an action.  However, if we later
					 import this module into another, and rename the action, the formula
					 will become incorrectly formed, since the argument will no longer
					 be an explicit action.
				      */
				    }
                                  | simple_sort_id 
                                    {
				      if ( !is_a_known_constant_argument_sort($1) ) { 
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @1.first_line,
							    "Reference to undeclared sort %s.\n",
							    $1);
				      }
				      /* increment number of arguments */
				      temp_num_arguments = temp_num_arguments + 1;
				      /* And add first argument to the list of strings */
				      insert_string_into_list(&temp_string_list_first,
							      &temp_string_list_last,
							      $1);
				    }
                                  | integer_range 
                                    {
				      if ( !is_a_valid_integer_range($1) ) { 
					print_parser_error_message(stderr, 
							    input_file_name, 
							    @1.first_line,
							    "Reference to invalid integer_range.\n",
							    $1);
				      }
				      /* increment number of arguments */
				      temp_num_arguments = temp_num_arguments + 1;
				      /* And add first argument to the list of strings */
				      insert_string_into_list(&temp_string_list_first,
							      &temp_string_list_last,
							      $1);
				    }
                                  ;

fluent_kind_and_domain : SIMPLE '(' simple_sort_id ')'
                           {
			     temp_kind = SIMPLE_FL_CONST;
			     
			     if ( !is_a_known_constant_domain_sort($3) ) {
			       print_parser_error_message(stderr,
						   input_file_name, 
						   @3.first_line, 
						   "Reference to undeclared sort %s.\n",
						   $3);
			     }
			     else if ( strcmp($3, "Boolean") == 0) {
			       print_parser_error_message(stderr,
						   input_file_name, 
						   @3.first_line, 
						   "Warning: Explicit declaration of Boolean value for constant is unnecessary.\n");
			     }
			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         | SD_FLUENT '(' simple_sort_id ')'
                           {
			     temp_kind = SD_FL_CONST;
			     
			     if ( !is_a_known_constant_domain_sort($3) ) {
			       print_parser_error_message(stderr,
						   input_file_name, 
						   @3.first_line, 
						   "Reference to undeclared sort %s.\n",
						   $3);
			     }
			     else if ( strcmp($3, "Boolean") == 0) {
			       print_parser_error_message(stderr,
						   input_file_name, 
						   @3.first_line, 
						   "Warning: Explicit declaration of Boolean value for constant is unnecessary.\n");
			     }
			     
			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         | RIGID '(' simple_sort_id ')'
                           {
			     temp_kind = RIGID_CONST;
			     
			     if ( !is_a_known_constant_domain_sort($3) ) {
			       print_parser_error_message(stderr,
							  input_file_name, 
							  @3.first_line, 
							  "Reference to undeclared sort %s.\n",
							  $3);
			     }
			     else if ( strcmp($3, "Boolean") == 0) {
			       print_parser_error_message(stderr,
							  input_file_name, 
							  @3.first_line, 
							  "Warning: Explicit declaration of Boolean value for constant is unnecessary.\n");
			     }
			     
			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         | SIMPLE
                           {
			     temp_kind = SIMPLE_FL_CONST;
			     temp_domain = "Boolean";
			   }
                         | SD_FLUENT
                           {
			     temp_kind = SD_FL_CONST;
			     temp_domain = "Boolean";
			   }
                         | RIGID
                           {
			     temp_kind = RIGID_CONST;
			     temp_domain = "Boolean";
			   }
                         | SIMPLE '(' integer_range ')'
                           {
			     temp_kind = SIMPLE_FL_CONST;
			     
			     if ( !is_a_valid_integer_range($3) ) { 
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  @3.first_line,
							  "Reference to invalid integer_range.\n",
							  $3);
			     }

			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         | SD_FLUENT '(' integer_range ')'
                           {
			     temp_kind = SD_FL_CONST;
			     
			     if ( !is_a_valid_integer_range($3) ) { 
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  @3.first_line,
							  "Reference to invalid integer_range.\n",
							  $3);
			     }
			     
			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         | RIGID '(' integer_range ')'
                           {
			     temp_kind = RIGID_CONST;

			     if ( !is_a_valid_integer_range($3) ) { 
			       print_parser_error_message(stderr, 
							  input_file_name, 
							  @3.first_line,
							  "Reference to invalid integer_range.\n",
							  $3);
			     }

			     temp_domain = (char *) malloc( sizeof(char) * strlen($3) + 1);
			     strcpy( temp_domain, $3);
			   }
                         ;


variable_declaration : VARIABLES 
                       { 
			 /* prepare to declare a group of variables
			    of the same sort */
			 init_string_list(&temp_string_list2_first,
					  &temp_string_list2_last);
		       }
                       variable_specification_breakup
                       { 
			 /* fprintf(stdout, "Done with a variables declaration\n");
			    print_current_variables(stdout); 
			    fprintf(stdout, "\n"); */
		       }
                     ;
variable_specification_breakup : variable_specification ';' 
                                 {
				   /* prepare for another group of variables with
				      the same sort */
				   init_string_list(&temp_string_list2_first,
						    &temp_string_list2_last);
				 }
                               variable_specification_breakup { }
                               | /* empty */
                               ;
variable_specification : variable_id_group ':' variable_group_sort_id 
                         {
			   update_variables(temp_string_list2_first, $3);
			 }
                       ;
variable_id_group : variable_id variable_id_breakup { }
                  ;
variable_id_breakup : ',' variable_id variable_id_breakup { }
                    | /* empty */
                    ;
variable_id : ID 
              {
		if ( string_begins_with_an_import_prefix($1) ) {
		  // !!! Do something
		}
		/* Register identifier */
		if ( lookup_identifier($1) == NULL ) {
		  insert_identifier($1);
		}

		/* first we check if this was already declared */
		if ( lookup_identifier($1)->node_data->id_type == UNINITIALIZED_ID ) {
		  if ( lookup_variable($1) == NULL ) {
		    /* Add variable to the group of variables 
		       declared together, which will have the same
		       sort */
		    insert_string_into_list(&temp_string_list2_first,
					    &temp_string_list2_last,
					    $1);
		    insert_variable($1);
		    $$ = $1;
		  }
		  else {
		    print_parser_error_message(stderr,
					input_file_name, 
					@1.first_line, 
					"Attempt to declare an already-declared variable: %s\n",
					$1);
		  }
		}
		else {
		  print_parser_error_message(stderr,
				      input_file_name, 
				      @1.first_line, 
				      "Identifier %s was already declared.\n",
				      $1);
		}
	      }
            ;

variable_group_sort_id : ACTION
                         {
			   $$ = $1;
                         }
                       | EXPLICIT_ACTION
                         { 
			   $$ = $1;                         
			 }
                       | simple_sort_id 
                         { 
			   if ( !is_a_known_variable_sort($1) ) {
			     print_parser_error_message(stderr,
							input_file_name, 
							@1.first_line, 
							"Reference to undeclared sort %s.\n",
							$1);
			   }
			   else {
			     $$ = $1;
			   }
			 }
		       | integer_range
			 {
			   if ( !is_a_valid_integer_range($1) ) { 
			     print_parser_error_message(stderr, 
							input_file_name, 
							@1.first_line,
							"Reference to invalid integer_range.\n",
							$1);
			   }
			   else {
			     $$ = $1;
			   }
			 }
                       ;


axiom_part : AXIOMS { }
             { 
	       temp_axiom = NULL;
	       keyword_if_seen_in_first_formula_follower = 0;
	       invalid_formula_seen_while_parsing_axiom = 0;
             }
             axiom_part_breakup
             { 
               /* fprintf(stdout, "Done with an axioms section\n");
		  print_current_axioms(stdout); 
		  fprintf(stdout, "\n"); */
	     }
           ;
axiom_part_breakup : axiom ';' 
                     {
		       if (temp_axiom != NULL) {
			 error_lineno = @1.first_line;
			 if ( invalid_formula_seen_while_parsing_axiom 
			      || !is_a_valid_axiom(temp_axiom) ) {
			   print_parser_error_message(stderr,
					       input_file_name, 
					       @1.first_line, 
					       "Ill-formed axiom.  Ignoring it...\n");
			 }
			 else {
			   insert_axiom(temp_axiom);
			 }
		       }
		       else {
			 print_parser_error_message(stderr,
						    input_file_name, 
						    @1.first_line, 
						    "Ill-formed axiom.  Ignoring it...\n");
		       }
		       temp_axiom = NULL;
		       keyword_if_seen_in_first_formula_follower = 0;
		       invalid_formula_seen_while_parsing_axiom = 0;
		     }
                     axiom_part_breakup
                   | /* empty */
                   ;
axiom : formula first_formula_follower
        {
	  /* We should check that the formulas are validly formed. */
	  error_lineno = @1.first_line;
	  if ( !is_a_valid_formula($1) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@1.first_line, 
				"The head formula is ill-formed.\n");
	  }
	  error_lineno = @2.first_line;
	  if ( keyword_if_seen_in_first_formula_follower 
	       && temp_axiom != NULL
	       && temp_axiom->G != NULL
	       && !is_a_valid_formula(temp_axiom->G) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"if\" is ill-formed.\n");
	  }
	  error_lineno = @2.first_line;
	  if ( temp_axiom != NULL
	       && temp_axiom->H != NULL
	       && !is_a_valid_formula(temp_axiom->H) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"after\" is ill-formed.\n");
	  }
	  
	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Starting with formula $1
	       this axiom can be one of 
	       $1
	       $1 if F
	       $1 after G
	       $1 if F after G          */
	    
	    /* first check if there was a follower */
	    if ( temp_axiom == NULL ) {
	      /* If there was no follower, then the G part
		 of the axiom must have been true.
		 So we first create a new true formula. */
	      temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      temp_formula->connective = TRUE;
	      temp_axiom = make_new_axiom(expand_formula($1), temp_formula, NULL);
	    }
	    else {
	      if (temp_axiom->G != NULL) {
		temp_axiom->G = expand_formula(temp_axiom->G);
	      }
	      else {
		temp_axiom->G = make_new_formula(ZERO_PLACE_CONNECTIVE);
		temp_axiom->G->connective = TRUE;
	      }
	      if (temp_axiom->H != NULL) {
		temp_axiom->H = expand_formula(temp_axiom->H);
	      }
	      temp_axiom->F = expand_formula($1);
	    }
	  }
        }
      | formula CAUSES formula possible_if_follower
        {
	  /* We should check that the first formula is validly formed and that it is
	     an action formula */
	  error_lineno = @1.first_line;
	  if ( !is_a_valid_formula($1) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@1.first_line, 
				"The formula before \"causes\" is ill-formed.\n");
	  }
	  if ( find_formula_kind($1) != ACTION_FORMULA ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@1.first_line, 
				"The formula before \"causes\" should be an action formula.\n");
	  }
	  /* We also check that the other formulas are validly formed. */
	  error_lineno = @3.first_line;
	  if ( !is_a_valid_formula($3) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@3.first_line, 
				"The formula after \"causes\" is ill-formed.\n");
	  }
	  error_lineno = @4.first_line;
	  if ( $4 != NULL && !is_a_valid_formula($4) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@4.first_line, 
				"The formula after \"if\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom 
               $1 CAUSES $3 if $4
	       is an abbreviation for a formula of the form
	       F if G after H.
	       If $3 is an action formula then it is an abbreviation for
	       $3 if $1 & $4.
	       If it's a fluent formula, it is an abbreviation for
               $3 if true after $1 & $4.
	       (In both cases "& $4" is dropped if there is no possible_if_follower.)
	    */
	    
	    temp_formula_kind = find_formula_kind($3);
	    if ( temp_formula_kind == ACTION_FORMULA ) {
	      temp_axiom = make_new_axiom(expand_formula($3), NULL, NULL);
	      
	      /* First check if there was anything following the if */
	      if ( $4 != NULL ) {
		/* We need to construct the axiom's G part by conjoining two formulas */
		temp_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula-> connective = AND;
		temp_formula->left_formula = $1;
		temp_formula->right_formula = $4;
		temp_axiom->G = expand_formula(temp_formula);
	      }
	      else {
		temp_axiom->G = expand_formula($1);
	      }
	    }
	    else if ( temp_formula_kind != FORMULA ) {
	      temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	      temp_formula-> connective = TRUE;
	      temp_axiom = make_new_axiom(expand_formula($3), temp_formula, NULL);
	      
	      /* First check if there was anything following the if */
	      if ( $4 != NULL ) {
		/* We need to construct the axiom's H part by conjoining two formulas */
		temp_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula-> connective = AND;
		temp_formula->left_formula = $1;
		temp_formula->right_formula = $4;
		temp_axiom->H = expand_formula(temp_formula);
	      }
	      else {
		temp_axiom->H = expand_formula($1);
	      }
	    }
	    else { /* It is not an action formula or a fluent formula */
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
					 input_file_name, 
					 @3.first_line, 
					 "The formula after \"causes\" should be an action formula or a fluent formula.\n");
	      temp_axiom = NULL;
	    }
	  }
	}
      | DEFAULT formula possible_if_follower possible_after_formula
        {
	  /* We should check that the formulas are validly formed. */
	  error_lineno = @2.first_line;
	  if ( !is_a_valid_formula($2) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"default\" is ill-formed.\n");
	  }
	  error_lineno = @3.first_line;
	  if ( $3 != NULL && !is_a_valid_formula($3) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@3.first_line, 
				"The formula after \"if\" is ill-formed.\n");
	  }
	  error_lineno = @4.first_line;
	  if ( temp_axiom != NULL
	       && temp_axiom->H != NULL 
	       && !is_a_valid_formula(temp_axiom->H) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@4.first_line, 
				"The formula after \"after\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom 
	       default $2 if $3 after $4
	       is an abbreviation for 
	       $2 if $2 & $3
	       or
 	       $2 if $2 & $3 after $4
	       depending on whether $4 exists. */
	  
	    /* first check if there was an after part */
	    if ( temp_axiom == NULL ) {
	      /* If there was no follower, then the axiom has no H part. */
	      temp_axiom = make_new_axiom(expand_formula($2), NULL, NULL);
	    }
	    else {
	      if ( temp_axiom->H != NULL ) {
		temp_axiom->H = expand_formula(temp_axiom->H);
	      }
	      temp_axiom->F = expand_formula($2);
	    }
	    /* Now check if there was an if part */
	    if ( $3 != NULL ) {
	      /* We need to construct the axiom's G part by conjoining two formulas */
	      temp_formula = make_new_formula(BINARY_CONNECTIVE);
	      temp_formula-> connective = AND;
	      temp_formula->left_formula = $2;
	      temp_formula->right_formula = $3;
	      temp_axiom->G = expand_formula(temp_formula);
	    }
	    else {
	      temp_axiom->G = expand_formula($2);
	    }
	  }
	}
      | EXOGENOUS term possible_if_follower
        {
	  /* Such an axiom 
	       exogenous $2 if $3
             is an abbreviation (depending on whether $3 exists) for 
	       $2=v if $2=v
             or
 	       $2=v if $2=v & $3
             for all values v in the domain of $2. */

	  /* We should check that the term is a constant. */
	  if ( $2 != NULL ) {
	    temp_constant_list_node = lookup_constant($2->LHS_term);
	    if ( temp_constant_list_node == NULL ) {
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
				  input_file_name, 
				  @2.first_line, 
				  "%s in \"exogenous\" axiom was not declared as a constant.\n",
				  $2->LHS_term);
	    }
	    else {
	      /* We should check that the term and, if it exists, the formula after if, are validly formed. */
	      error_lineno = @2.first_line;
	      if (! is_a_valid_term($2->LHS_term, $2->LHS_arguments) ) {
		invalid_formula_seen_while_parsing_axiom = 1;
		print_parser_error_message(stderr,
				    input_file_name, 
				    @2.first_line, 
				    "The term after \"exogenous\" is ill-formed.\n" );
	      }
	      error_lineno = @3.first_line;
	      if ( $3 != NULL && !is_a_valid_formula($3) ) {
		invalid_formula_seen_while_parsing_axiom = 1;
		print_parser_error_message(stderr,
				    input_file_name, 
				    @3.first_line, 
				    "The formula after \"if\" is ill-formed.\n");
	      }
	      
	      if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
		temp_formula = $2;
		
		/* Since there is no value given and we need to assert that all
		   values are possible, we make a variable for the
		   sort of the values, and use that variable for the value. */
		
		/* We make a new one that is different from the other identifiers. */
		
		temp_variable = make_new_variable_name(temp_constant_list_node->node_data->domain);
		
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(temp_variable);
		insert_variable(temp_variable);
		init_string_list(&temp_string_list2_first,
				 &temp_string_list2_last);
		insert_string_into_list(&temp_string_list2_first,
					&temp_string_list2_last,
					temp_variable);
		update_variables(temp_string_list2_first, temp_constant_list_node->node_data->domain);
		
		temp_formula->RHS_term = temp_variable;
		temp_formula->LHS_arguments = $2->LHS_arguments;
		
		/* Such an axiom is an abbreviation for
		   F if F */                              
		temp_axiom = make_new_axiom(expand_formula(temp_formula), NULL, NULL);
		
		/* Check if there was an if part */
		if ( $3 != NULL ) {
		  /* We need to construct the axiom's G part by conjoining two formulas */
		  temp_formula = make_new_formula(BINARY_CONNECTIVE);
		  temp_formula-> connective = AND;
		  temp_formula->left_formula = $2;
		  temp_formula->left_formula->RHS_term = temp_variable;
		  temp_formula->right_formula = $3;
		  temp_axiom->G = expand_formula(temp_formula);
		}
		else {
		  temp_axiom->G = expand_formula($2);
		}
	      }
	    }
	  }
	  else { /* The term is NULL */
	    /* We should never encounter this situation but just in case we do... */
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				       input_file_name, 
				       @2.first_line, 
				       "NULL term following \"exogenous\".\n");
	  }
	}
      | INERTIAL term possible_if_follower
        {
	  /* Such an axiom 
	       inertial $2 if $3
             is an abbreviation (depending on whether $3 exists) for 
	       $2=v if $2=v after $2=v
             or
 	       $2=v if $2=v after $2=v & $3
             for all values v in the domain of $2. */

	  /* We should check that the term is a fluent constant. */
	  if ( $2 != NULL ) {
	    temp_constant_list_node = lookup_constant($2->LHS_term);
	    if ( temp_constant_list_node == NULL ) {
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
				  input_file_name, 
				  @2.first_line, 
				  "%s in \"inertial\" axiom was not declared as a constant.\n",
				  $2->LHS_term);
	    }
	    else if ( temp_constant_list_node->node_data != NULL 
		      && temp_constant_list_node->node_data->kind == ACTION_CONST ) {
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
				  input_file_name, 
				  @2.first_line, 
				  "%s in \"inertial\" was not declared as a fluent constant.\n",
				  $2->LHS_term);
	    }
	    else {
	      /* We should check that the term and, if it exists, the formula after if, are validly formed. */
	      error_lineno = @2.first_line;
	      if (! is_a_valid_term($2->LHS_term, $2->LHS_arguments) ) {
		invalid_formula_seen_while_parsing_axiom = 1;
		print_parser_error_message(stderr,
				    input_file_name, 
				    @2.first_line, 
				    "The term after \"inertial\" is ill-formed.\n");
	      }
	      error_lineno = @3.first_line;
	      if ( $3 != NULL && !is_a_valid_formula($3) ) {
		invalid_formula_seen_while_parsing_axiom = 1;
		print_parser_error_message(stderr,
				    input_file_name, 
				    @3.first_line, 
				    "The formula after \"if\" is ill-formed.\n");
	      }

	      if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
		temp_formula = $2;
		
		/* Since there is no value given and we need to assert that all
		   values are possible, we make a variable for the
		   sort of the values, and use that variable for the value. */
		
		/* We make a new variable that is different from the other identifiers. */
		
		temp_variable = make_new_variable_name(temp_constant_list_node->node_data->domain);
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(temp_variable);
		insert_variable(temp_variable);
		init_string_list(&temp_string_list2_first,
				 &temp_string_list2_last);
		insert_string_into_list(&temp_string_list2_first,
					&temp_string_list2_last,
					temp_variable);
		update_variables(temp_string_list2_first, temp_constant_list_node->node_data->domain);
		
		temp_formula->RHS_term = temp_variable;
		temp_formula->LHS_arguments = $2->LHS_arguments;
		
		/* Such an axiom is an abbreviation for
		   F if F after F */                              
		temp_axiom = make_new_axiom(expand_formula(temp_formula), expand_formula(temp_formula), NULL);
		
		/* Check if there was an if part */
		if ( $3 != NULL ) {
		  /* We need to construct the axiom's H part by conjoining two formulas */
		  temp_formula = make_new_formula(BINARY_CONNECTIVE);
		  temp_formula-> connective = AND;
		  temp_formula->left_formula = $2;
		  temp_formula->left_formula->RHS_term = temp_variable;
		  temp_formula->right_formula = $3;
		  temp_axiom->H = expand_formula(temp_formula);
		}
		else {
		  temp_axiom->H = expand_formula($2);
		}
	      }
	    }
	  }
	  else { /* The term is NULL */
	    /* We should never encounter this situation but just in case we do... */
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				       input_file_name, 
				       @2.first_line, 
				       "NULL term following \"inertial\".\n");
	  }
	}
      | CONSTRAINT formula possible_after_formula
        {
	  /* We should check that the formulas are validly formed. */
	  error_lineno = @2.first_line;
	  if ( !is_a_valid_formula($2) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"constraint\" is ill-formed.\n");
	  }
	  error_lineno = @3.first_line;
	  if ( temp_axiom != NULL
	       && temp_axiom->H != NULL 
	       && !is_a_valid_formula(temp_axiom->H) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@3.first_line, 
				"The formula after \"after\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom 
	       constraint $2 after $3
	       is an abbreviation for 
	       false if -$2
	       or
 	       false if -$2 after $3
	       depending on whether $3 exists. */
	    
	    /* We create a new formula which is the negation of
	       the formula $2. */
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
	    temp_formula->connective = NEG;
	    temp_formula->right_formula = $2;
	    
	    /* first check if there was an after part */
	    if ( temp_axiom == NULL ) {
	      /* If there was no follower, then the axiom has no H part. */
	      temp_axiom = make_new_axiom(NULL, expand_formula(temp_formula), NULL);
	    }
	    else {
	      if ( temp_axiom->H != NULL ) {
		temp_axiom->H = expand_formula(temp_axiom->H);
	      }
	      temp_axiom->G = expand_formula(temp_formula);
	    }
	    /* We create a new false formula corresponding to
	       the "bottom" in the first part of the axiom. */
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula->connective = FALSE;
	    temp_axiom->F = temp_formula;
	  }
	}
      | NONEXECUTABLE formula possible_if_follower
        {
	  /* We should check that formula $2 is validly formed and that it is
	     an action formula */
	  error_lineno = @2.first_line;
	  if ( !is_a_valid_formula($2) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"nonexecutable\" is ill-formed.\n");
	  }
	  if ( find_formula_kind($2) != ACTION_FORMULA ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"nonexecutable\" should be an action formula.\n");
	  }
	  /* We also check that formula $3, if it exists, is validly formed. */
	  error_lineno = @3.first_line;
	  if ( $3 != NULL && !is_a_valid_formula($3) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@3.first_line, 
				"The formula after \"if\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom is an abbreviation for
	       false if true after  $2
	       if there was no possible_if_follower, or
	       false if true after  $2 & $3 
	       if there was a valid possible_if_follower*/

	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula-> connective = FALSE;
	    temp_axiom = make_new_axiom(temp_formula, NULL, NULL);
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula-> connective = TRUE;
	    temp_axiom->G = temp_formula;
	    
	    /* Check if there was an if part */
	    if ( $3 != NULL ) {
	      /* We need to construct the axiom's H part by conjoining two formulas */
	      temp_formula = make_new_formula(BINARY_CONNECTIVE);
	      temp_formula-> connective = AND;
	      temp_formula->left_formula = $2;
	      temp_formula->right_formula = $3;
	      temp_axiom->H = expand_formula(temp_formula);
	    }
	    else {
	      temp_axiom->H = expand_formula($2);
	    }
	  }
        }
      | ALWAYS formula
        {
	  /* We should check that the formula is validly formed. */
	  error_lineno = @2.first_line;
	  if ( !is_a_valid_formula($2) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"The formula after \"always\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom 
	       always $2
	       is an abbreviation for 
 	       false (if true) after -$2
	    */
	    
	    /* We create a new formula which is the negation of
	       the formula $2. */
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
	    temp_formula->connective = NEG;
	    temp_formula->right_formula = $2;
	    
	    temp_axiom = make_new_axiom(NULL, NULL, expand_formula(temp_formula));
	    
	    /* We create a new false formula corresponding to
	       the "bottom" in the first part of the axiom. */
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula->connective = FALSE;
	    temp_axiom->F = temp_formula;
	    
	    /* We create a new true formula corresponding to
	       the "top" in the middle part of the axiom. */
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula->connective = TRUE;
	    temp_axiom->G = temp_formula;
	  }
        }
      | RIGID term
        {
	  /* Such an axiom 
	       rigid $2
             is an abbreviation for
	       false if -$2=v after $2=v
             for all values v in the domain of $2. */

	  /* We should check that the term is a fluent constant. */
	  if ( $2 != NULL ) {
	    temp_constant_list_node = lookup_constant($2->LHS_term);
	    if ( temp_constant_list_node == NULL ) {
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
				  input_file_name, 
				  @2.first_line, 
				  "%s in \"rigid\" axiom was not declared as a constant.\n",
				  $2->LHS_term);
	    }
	    else if ( temp_constant_list_node->node_data != NULL 
		      && temp_constant_list_node->node_data->kind == ACTION_CONST ) {
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
				  input_file_name, 
				  @2.first_line, 
				  "%s in \"rigid\" was not declared as a fluent constant.\n",
				  $2->LHS_term);
	    }
	    else {
	      /* We should check that the term is validly formed. */
	      error_lineno = @2.first_line;
	      if (! is_a_valid_term($2->LHS_term, $2->LHS_arguments) ) {
		invalid_formula_seen_while_parsing_axiom = 1;
		print_parser_error_message(stderr,
				    input_file_name, 
				    @2.first_line, 
				    "The term after \"inertial\" is ill-formed.\n");
	      }

	      if ( invalid_formula_seen_while_parsing_axiom == 0 ) {	      
		temp_formula = $2;
		
		/* Since there is no value given and we need to assert that all
		   values are possible, we make a variable for the
		   sort of the values, and use that variable for the value. */
		
		/* We make a new variable that is different from the other identifiers. */
		
		temp_variable = make_new_variable_name(temp_constant_list_node->node_data->domain);
		/* Add this new variable to the list of identifiers and to the list of variables */
		insert_identifier(temp_variable);
		insert_variable(temp_variable);
		init_string_list(&temp_string_list2_first,
				 &temp_string_list2_last);
		insert_string_into_list(&temp_string_list2_first,
					&temp_string_list2_last,
					temp_variable);
		update_variables(temp_string_list2_first, temp_constant_list_node->node_data->domain);
		
		temp_formula->RHS_term = temp_variable;
		temp_formula->LHS_arguments = $2->LHS_arguments;
		
		/* Such an axiom is an abbreviation for
		   false if -H after H */                              
		temp_axiom = make_new_axiom(NULL, NULL, expand_formula(temp_formula));
		
		/* We create a new false formula corresponding to
		   the "bottom" in the first part of the axiom. */
		temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		temp_formula->connective = FALSE;
		temp_axiom->F = temp_formula;
		
		/* We create a new formula which is the negation of
		   the formula H. */
		temp_formula = make_new_formula(UNARY_CONNECTIVE);
		temp_formula->connective = NEG;
		temp_formula->right_formula = $2;
		temp_formula->right_formula->RHS_term = temp_variable;
		temp_axiom->G = expand_formula(temp_formula);
	      }
	    }
	  }
	  else { /* The term is NULL */
	    /* We should never encounter this situation but just in case we do... */
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@2.first_line, 
				"NULL term following \"rigid\".\n");
	  }
	}
      | formula MAY CAUSE formula possible_if_follower
        {
	  /* We should check that the formulas are validly formed. */
	  error_lineno = @1.first_line;
	  if ( !is_a_valid_formula($1) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@1.first_line, 
				"The formula before \"may cause\" is ill-formed.\n");
	  }
	  if ( find_formula_kind($1) != ACTION_FORMULA ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@1.first_line, 
				"The formula before \"may cause\" should be an action formula.\n");
	  }
	  error_lineno = @4.first_line;
	  if ( !is_a_valid_formula($4) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@4.first_line, 
				"The formula after \"may cause\" is ill-formed.\n");
	  }
	  error_lineno = @5.first_line;
	  if ( $5 != NULL && !is_a_valid_formula($5) ) {
	    invalid_formula_seen_while_parsing_axiom = 1;
	    print_parser_error_message(stderr,
				input_file_name, 
				@5.first_line, 
				"The formula after \"if\" is ill-formed.\n");
	  }

	  if ( invalid_formula_seen_while_parsing_axiom == 0 ) {
	    /* Such an axiom 
	       $1 may cause $4 if $5
	       is an abbreviation for 
	       
 	       $4 if $4 & $1 & $5        if $4 is an action formula
	       
               (same as: default $4 if $1 & $5)
	       or
	       
	       
	       $4 if $4 after $1 & $5    if $4 is a fluent formula
	       
	       (same as: default $4 after $1 & $5)
	       
	       depending on whether $4 exists. */

	    temp_formula_kind = find_formula_kind($4);
	    if ( temp_formula_kind == ACTION_FORMULA ) {
	      temp_axiom = make_new_axiom(expand_formula($4), NULL, NULL);
	      
	      /* First check if there was anything following the if */
	      if ( $5 != NULL ) {
		/* We need to construct the axiom's G part by conjoining three formulas */
		temp_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula->connective = AND;
		temp_formula->left_formula = $4;
		temp_formula->right_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula->right_formula->connective = AND;
		temp_formula->right_formula->left_formula = $1;
		temp_formula->right_formula->right_formula = $5;
		$5;
		temp_axiom->G = expand_formula(temp_formula);
	      }
	      else {
		/* We need to construct the axiom's G part by conjoining three formulas */
		temp_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula->connective = AND;
		temp_formula->left_formula = $4;
		temp_formula->right_formula = $1;
		
		temp_axiom->G = expand_formula(temp_formula);
	      }
	    }
	    else if ( temp_formula_kind != FORMULA ) {
	      temp_axiom = make_new_axiom(expand_formula($4), expand_formula($4), NULL);
	      
	      /* First check if there was anything following the if */
	      if ( $5 != NULL ) {
		/* We need to construct the axiom's H part by conjoining two formulas */
		temp_formula = make_new_formula(BINARY_CONNECTIVE);
		temp_formula-> connective = AND;
		temp_formula->left_formula = $1;
		temp_formula->right_formula = $5;
		temp_axiom->H = expand_formula(temp_formula);
	      }
	      else {
		temp_axiom->H = expand_formula($1);
	      }
	    }
	    else { /* It is not an action formula or a fluent formula */
	      invalid_formula_seen_while_parsing_axiom = 1;
	      print_parser_error_message(stderr,
					 input_file_name, 
					 @3.first_line, 
					 "The formula after \"may cause\" should be an action formula or a fluent formula.\n");
	      temp_axiom = NULL;
	    }
	  }
	}
      ;

// This possibly follows "F ..."
first_formula_follower : IF formula possible_after_formula
                       {
			 keyword_if_seen_in_first_formula_follower = 1;
			 /* first check if there was an after formula */
			 if ( temp_axiom == NULL ) {
			   temp_axiom = make_new_axiom(NULL, $2, NULL);
			 }
			 else {
			   temp_axiom->G = $2;
			 }
		       }
                       | possible_after_formula
                         {
			   /* temp_axiom is set in the action for possible_after_formula
			      so we don't need to do anything at all here. */
			 }
		       /* We don't need to have an empty possibility here since that
			  is covered by possible_after_formula. */
                       ;

// This possibly follows "F ..." or "F if G ..." or "default F ..." or "default F if G ..."
// or "constraint F ..."
possible_after_formula : AFTER formula
                       {
			 temp_axiom = make_new_axiom(NULL, NULL, $2);
		       }
                       | /* empty */
                       ;

// This possibly follows "F causes G ..." or "default F ..." or "exogenous c ..."
// or "inertial c ..." or "nonexecutable F ..."
possible_if_follower : IF formula
                       {
			 $$ = $2;
		       }
                     | /* empty */ { $$ = NULL; }
                     ;

formula : EXISTS ID formula     %prec QUANTIFIER_PRECEDENCE
          {
	    /* We should check that the ID is a variable, but this is done
	       when the function checking validity of a formula is called. */
	    temp_formula = make_new_formula(QUANTIFIER);
	    temp_formula->quantifier = EXISTS;
	    temp_formula->quantified_variable = (char *) malloc( sizeof(char) * strlen($2) + 1);
            strcpy(temp_formula->quantified_variable, $2);
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' EXISTS ID formula ')'    %prec QUANTIFIER_PRECEDENCE
          {
	    /* We should check that the ID is a variable, but this is done
	       when the function checking validity of a formula is called. */
	    temp_formula = make_new_formula(QUANTIFIER);
	    temp_formula->quantifier = EXISTS;
	    temp_formula->quantified_variable = (char *) malloc( sizeof(char) * strlen($3) + 1);
            strcpy(temp_formula->quantified_variable, $3);
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | FORALL ID formula     %prec QUANTIFIER_PRECEDENCE
          {
	    /* We should check that the ID is a variable, but this is done
	       when the function checking validity of a formula is called. */
	    temp_formula = make_new_formula(QUANTIFIER);
	    temp_formula->quantifier = FORALL;
	    temp_formula->quantified_variable = (char *) malloc( sizeof(char) * strlen($2) + 1);
            strcpy(temp_formula->quantified_variable, $2);
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' FORALL ID formula ')'    %prec QUANTIFIER_PRECEDENCE
          {
	    /* We should check that the ID is a variable, but this is done
	       when the function checking validity of a formula is called. */
	    temp_formula = make_new_formula(QUANTIFIER);
	    temp_formula->quantifier = FORALL;
	    temp_formula->quantified_variable = (char *) malloc( sizeof(char) * strlen($3) + 1);
            strcpy(temp_formula->quantified_variable, $3);
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | formula AND formula
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = AND;
	    temp_formula->left_formula = $1;
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' formula AND formula ')'
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = AND;
	    temp_formula->left_formula = $2;
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | formula OR formula
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = OR;
	    temp_formula->left_formula =  $1;
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' formula OR formula ')'
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = OR;
	    temp_formula->left_formula =  $2;
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | NEG formula
          {
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
            temp_formula->connective = NEG;
	    temp_formula->right_formula = $2;
            $$ = temp_formula;
          }
        | '(' NEG formula ')'
          {
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
            temp_formula->connective = NEG;
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | formula EQUIV formula
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = EQUIV;
	    temp_formula->left_formula =  $1;
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' formula EQUIV formula ')'
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = EQUIV;
	    temp_formula->left_formula =  $2;
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | formula IMPLIES formula
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = IMPLIES;
	    temp_formula->left_formula =  $1;
	    temp_formula->right_formula = $3;
            $$ = temp_formula;
          }
        | '(' formula IMPLIES formula ')'
          {
	    temp_formula = make_new_formula(BINARY_CONNECTIVE);
            temp_formula->connective = IMPLIES;
	    temp_formula->left_formula =  $2;
	    temp_formula->right_formula = $4;
            $$ = temp_formula;
          }
        | atomic_formula /* We cover zero-place connectives "true" and "false" in this case */
          {
            $$ = $1;
	  }
/* The two rules below are now covered by the rule "atomic_formula : term", since just "true" or
   just "false" may appear as a single term and we don't want conflicts.

        | TRUE
          {
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula->connective = TRUE;
            $$ = temp_formula;
          }          
        | FALSE
          {
	    temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
	    temp_formula->connective = FALSE;
            $$ = temp_formula;
          }          
*/
        | term NEQ term
          {
	    /* This is a case very similar to an atomic formula of the form "term = term"
	       However, we need to handle this separately, since it is an abbreviation
	       for the formula -(term = term). */
	    
	    /* We should check that any identifiers in the terms exist have been declared,
	       Also check that the variables, objects, constants are used in a manner consistent
               with their declaration.
	       However, since we may have an atom or a formula as part of a renaming
	       clause, we don't check this here.  We have a function to check the validity of a formula
	       and this is called later. */

	    /* We already made the atomic formula containing the LHS term, 
	       when parsing the LHS term.  Now the RHS term should be added
	       to that atomic formula, and the whole thing be should be negated. */

	    /* We negate the atomic formula. */
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
            temp_formula->connective = NEG;
	    temp_formula->right_formula = $1;
	    
	    /* For practical reasons, when a term is parsed, we pass it up as a formula whose
	       LHS is set to the term.  Now, we copy that second term to the RHS of the first term. */
	    temp_formula->right_formula->RHS_term = (char *) malloc( sizeof(char) * strlen( $3->LHS_term ) + 1);
	    strcpy( temp_formula->right_formula->RHS_term, $3->LHS_term);
	    temp_formula->right_formula->RHS_arguments = copy_arguments( $3->LHS_arguments );
	    
            $$ = temp_formula;
	  }
        | '(' term NEQ term ')'
          {
	    /* The case above with parentheses around it. */
	    temp_formula = make_new_formula(UNARY_CONNECTIVE);
            temp_formula->connective = NEG;
	    temp_formula->right_formula = $2;
	    
	    temp_formula->right_formula->RHS_term = (char *) malloc( sizeof(char) * strlen( $4->LHS_term ) + 1);
	    strcpy( temp_formula->right_formula->RHS_term, $4->LHS_term);
	    temp_formula->right_formula->RHS_arguments = copy_arguments( $4->LHS_arguments );
	    
            $$ = temp_formula;
	  }
        | inequality
          {
	    $$ = $1;
	  }
        | '(' inequality ')'
          {
	    $$ = $2;
	  }
        ;
atomic_formula : term '=' term
                 {
		   /* We should check that any identifiers in the terms exist have been declared,
		      Also check that the variables, objects, constants are used in a manner consistent
		      with their declaration.
		      However, since we may have an atom or a formula as part of a renaming
		      clause, we don't check this here.  We have a function to check the validity of a formula
		      and this is called later. */
		   
		   /* We already made the atomic formula containing the LHS term, 
                      when parsing the LHS term.  Now the RHS term should be added. */
		   temp_formula = $1;

		   /* For practical reasons, when a term is parsed, we pass it up as a formula whose
		      LHS is set to the term.  Now, we copy that second term to the RHS of the first term. */
		   temp_formula->RHS_term = (char *) malloc( sizeof(char) * strlen( $3->LHS_term ) + 1);
		   strcpy( temp_formula->RHS_term, $3->LHS_term);
		   temp_formula->RHS_arguments = copy_arguments( $3->LHS_arguments );
		   $$ = temp_formula;
		 }
               | '(' term '=' term ')'
                 {
		   /* The case above with parentheses around it. */

		   temp_formula = $2;

		   temp_formula->RHS_term = (char *) malloc( sizeof(char) * strlen( $4->LHS_term ) + 1);
		   strcpy( temp_formula->RHS_term, $4->LHS_term);
		   temp_formula->RHS_arguments = copy_arguments( $4->LHS_arguments );
		   $$ = temp_formula;
		 }
               | term
                 {
		   /* A single term.  There are two possibilities:
                      i) it is a zero-place connective "true" or "false"
                      ii) Corresponds to the atomic formula "term = true" */
		   
		   /* Check if this is actually a zero place connective. */
		   if ($1->LHS_term == "true") {
		     temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		     temp_formula->connective = TRUE;
		     $$ = temp_formula;
		   }
		   else if ($1->LHS_term == "false") {
		     temp_formula = make_new_formula(ZERO_PLACE_CONNECTIVE);
		     temp_formula->connective = FALSE;
		     $$ = temp_formula;
		   }
		   else { // It is not a zero place connective

		     /* !!!: We should check that any identifiers in the terms exist have been declared,
			Also check that the variables, objects, constants are used in a manner consistent
			with their declaration.
			However, since we may have an atom or a formula as part of a renaming
			clause, we don't check this here.  We have a function to check the validity of a formula
			and this is called later. */
		     
		     /* We already made the atomic formula containing the LHS term, 
			when parsing the LHS term.  Now the RHS term should be added. */
		     temp_formula = $1;
		     
		     /* Since there is no '=' in this atomic formula, the RHS_term is assumed to be "true" */
		     /* !!! But maybe this should be NULL and there should be a check to
			be performed, making sure that it is not an action or action
			variable before putting true.  Actually, we should check
			be careful since an action may be non-Boolean and what if there
			is an action variable. */

		     // temp_formula->RHS_term = "true";
		     temp_formula->RHS_term = NULL;
		     $$ = temp_formula;
		   }
		 }
               ;
inequality : term LESS_THAN term
             {
	       /* We should check that any identifiers in the terms exist have been declared,
		  Also check that the variables, objects, constants are used in a manner consistent
		  with their declaration.
		  However, since we may have an atom or a formula as part of a renaming
		  clause, we don't check this here.  We have a function to check the validity of a formula
		  and this is called later. */
	       
	       // For practical reasons, when a term is parsed, we pass it up
	       // as a formula whose is set to the term.  Therefore, at this
	       // point we already have one atomic formula for first term
	       // and one atomic formula for the second term.
	       // Now convert the first of those to an inequality formula,
	       // and move the contents of the second atomic formula
	       // into the RHS term of the inequality.
	       temp_formula = $1;
	       temp_formula->formula_kind = INEQUALITY;
	       temp_formula->inequality_kind = LESS_THAN;
	       
	       temp_formula->RHS_term 
		 = (char *) malloc( sizeof(char) * strlen( $3->LHS_term ) + 1);
	       strcpy( temp_formula->RHS_term, $3->LHS_term);
	       temp_formula->RHS_arguments = copy_arguments( $3->LHS_arguments );
	       
	       $$ = temp_formula;
	     }
           ;


/* We need the following first option below because Boolean(x) may appear as a
   "sort name formula". */ 
term : BOOLEAN '(' constant_arguments ')'
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen($1) + 1);
	 strcpy( temp_formula->LHS_term, $1);
	 // We won't set the value since we haven't parsed that yet but we'll add it later
	 temp_formula->LHS_arguments = $3;
	 $$ = temp_formula;
       }
/* We need the following because action(x) may appear as a "sort name formula". */ 
     | ACTION '(' constant_arguments ')'
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen($1) + 1);
	 strcpy( temp_formula->LHS_term, $1);
	 // We won't set the value since we haven't parsed that yet but we'll add it later
	 temp_formula->LHS_arguments = $3;
	 $$ = temp_formula;
       }
/*
       | BOOLEAN 
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen($1) + 1);
	 strcpy( temp_formula->LHS_term, $1);
	 // We won't set the value since we haven't parsed that yet but we'll add it later.
	 // There are no arguments so we don't need to update the arguments field of the formula.
	 $$ = temp_formula;
       }
     | ACTION
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen($1) + 1);
	 strcpy( temp_formula->LHS_term, $1);
	 // We won't set the value since we haven't parsed that yet but we'll add it later.
	 // There are no arguments so we don't need to update the arguments field of the formula.
	 $$ = temp_formula;
       }
*/
     | TRUE 
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = "true";
	 $$ = temp_formula;
       }
     | FALSE
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = "false";
	 $$ = temp_formula;
       }
/* The case below covers two possibilities: 
   i)  basic terms of the form ID, ID(args), INTEGER
   ii) arithmetic expressions built from arithmetic operators like +/*
*/
     | arithmetic_expression
       {
	 temp_formula = make_new_formula(ATOMIC_FORMULA);
	 temp_formula->LHS_term = (char *) malloc( sizeof(char) * strlen($1->name) + 1);
	 strcpy( temp_formula->LHS_term, $1->name);
	 
	 // if the term has arguments 
	 // or if it's an arithmetic operator (represented as "+(args)"
	 if ($1->arguments != NULL ) {
	   temp_formula->LHS_arguments = copy_arguments($1->arguments);
	 }

	 // We won't set the value since we haven't parsed that
	 // yet but we'll add it later.
	 // There are no arguments so we don't need to update the 
	 // arguments field of the formula.
	 $$ = temp_formula;
       }
     ;


arithmetic_expression : arithmetic_expression PLUS arithmetic_expression
                        {
			  // Make an argument with name="*" and two arguments
			  temp_argument = make_new_argument(ARGUMENTED_ARGUMENT, "+");
			  temp_argument->arguments = copy_arguments($1);
			  temp_argument->arguments->next_argument = copy_arguments($3);
			  $$ = temp_argument;
			}
                      | arithmetic_expression TIMES arithmetic_expression
                        {
			  // Make an argument with name="*" and two arguments
			  temp_argument = make_new_argument(ARGUMENTED_ARGUMENT, "*");
			  temp_argument->arguments = copy_arguments($1);
			  temp_argument->arguments->next_argument = copy_arguments($3);
			  $$ = temp_argument;
			}
                      | '(' arithmetic_expression ')'
		        {
			  $$ = $2;
		        }
                      | constant_argument_atom
		        {
			  $$ = $1;
			}
                      ;


/* The arguments for a constant may be of the form of atoms.  e.g. Actor(u, Go(u,p)=true) */
constant_arguments : constant_argument_atom constant_argument_breakup 
                     { 
		       /* check if there were additional arguments indeed */
		       if ( $2 != NULL ) {
			 if ( $1 != NULL ) {
			   $1->next_argument = $2;
			 }
			 else {
			   print_parser_error_message(stderr,
					       input_file_name, 
					       @1.first_line, 
					       "Internal error: an argument is NULL!\n");
			 }
		       }
		       $$ = $1;
		     }
                   | TRUE constant_argument_breakup
                     {
		       temp_argument = make_new_argument(PLAIN_ARGUMENT, $1);
		       temp_argument->next_argument = $2;
		       $$ = temp_argument;
		     }
                   | FALSE constant_argument_breakup
                     {
		       temp_argument = make_new_argument(PLAIN_ARGUMENT, $1);
		       temp_argument->next_argument = $2;
		       $$ = temp_argument;
		     }
                   ;
constant_argument_breakup : ',' constant_argument_atom constant_argument_breakup 
                            {
			      /* check if there were additional arguments indeed */
			      if ( $3 != NULL ) {
				$2->next_argument = $3;
			      }
			      $$ = $2;
                            }
                          | ',' TRUE constant_argument_breakup
                            { 
			      temp_argument = make_new_argument(PLAIN_ARGUMENT, $2);
			      temp_argument->next_argument = $3;
			      $$ = temp_argument;
			    }
                          | ',' FALSE constant_argument_breakup
                            { 
			      temp_argument = make_new_argument(PLAIN_ARGUMENT, $2);
			      temp_argument->next_argument = $3;
			      $$ = temp_argument;
			    }
                          | /* empty */
		          {
			    /* return NULL if there were no additional arguments */
			    $$ = NULL;
			  }
                          ;
/* The argument for a constant may be an atom but we want to treat it differently
   if it is an argument.  (An atom as an argument won't be a formula, though a single
   atom would be a valid formula if it's not an argument.) 

In order to accomodate non-Boolean actions as constant arguments, we used to allow
for a constant argument to have a value.  Now, we prohibit this, so the commented rule below,
the original form, was modified by dropping the first case (with a value).

constant_argument_atom : constant_argument_constant '=' value
                         {
			   temp_argument = $1;
			   temp_argument->has_a_value = 1;
			   temp_argument->value = (char *) malloc( sizeof(char) * strlen($3) + 1);
			   strcpy( temp_argument->value, $3);
			   $$ = temp_argument;
			 }
                       | constant_argument_constant
                         {
			   temp_argument = $1;
			   
			   // We should check that it is a Boolean constant, but we don't do
			   // this here.  Instead, the function which checks validity of a formula
			   // checks this.

			   // We don't "assume it's Boolean and set a value"
			   //temp_argument->value = "true";
			   $$ = temp_argument;
			 }
                       ;
*/
constant_argument_atom : constant_argument_constant
                         {
			   temp_argument = $1;
			   
			   /* We should check that it is a Boolean constant, but we don't do
			      this here.  Instead, the function which checks validity of a formula
			      checks this.  */

			   // We don't "assume it's Boolean and set a value"
			   //temp_argument->value = "true";
			   $$ = temp_argument;
			 }
                       ;
constant_argument_constant : ID '(' constant_arguments ')'
                             {
			       temp_argument = make_new_argument(ARGUMENTED_ARGUMENT, $1);
			       /* The arguments of the constant argument have been parsed so we 
				  update that field. */
			       temp_argument->arguments = $3;
			       $$ = temp_argument;
			     }
                           | ID 
                             {
			       $$ = make_new_argument(PLAIN_ARGUMENT, $1);
                             }
                           | INTEGER
                             {
                               $$ = make_new_argument(PLAIN_ARGUMENT, $1);
                             }
                           ;
/* In order to accomodate non-Boolean actions as constant arguments, 
   we used to allow for a constant argument to have a value.  
   Now we prohibit this, so the rules below are commented out.

value : ID
        {
	  $$ = $1;
        }
      | TRUE 
        {
          $$ = "true";
        }
      | FALSE
        {
          $$ = "false";
        }
      ;
*/


%%
/* Moved this to the top: #include "lex.yy.c" */

/* This is declared at the top, but defined here. */
/* For a GLR parser with the "locations" and "pure-parser" options on.  */
void yyerror (YYLTYPE *locp, char const *msg) {
   //fprintf (stderr, "%d: %s\n", locp->first_line, msg);
   print_parser_error_message(stderr, input_file_name, locp->first_line, "%s\n",msg);

};  
