package code;
import java.io.Externalizable;
import java.util.*;

import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;
import java.util.Arrays;
// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.serialization.SerializationHelper;

/** 
 *  AcceptVV: AN immutable VV for use as a generalized acceptstamp 
 *  in GeneralInv records 
 **/ 

public class AcceptVV extends VV implements Externalizable, Immutable, Comparable<AcceptVV>
{
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

  protected final Hashtable<NodeId, Long> stamps;
  public  final static long BEFORE_TIME_BEGAIN = -1;

  /** 
   *  empty Constructor for serialization 
   **/ 
  public AcceptVV(){
    stamps = new Hashtable<NodeId, Long>();
  }

  /** 
   *  Constructor 
   **/ 
  public AcceptVV(AcceptStamp[] stamps_){
    this.stamps = new Hashtable<NodeId, Long>(stamps_.length);
    for(int i = 0; i < stamps_.length; i++){
	assert(this.stamps.get(stamps_[i].getNodeId()) == null): Arrays.toString(stamps_);
      this.stamps.put(stamps_[i].getNodeId(),
          new Long(stamps_[i].getLocalClock()));
    }
  }

  /** 
   *  Constructor -- private b/c uses dangerous approach of including passed-in 
   *  hash table as private state. Would need to clone() it if we want to 
   *  be public. 
   **/ 
  public AcceptVV(Hashtable<NodeId, Long> newStamps){
    assert(this.verifyHashtable(newStamps));
    this.stamps = newStamps;
  }

  /** 
   *  Constructor -- private b/c uses dangerous approach of including passed-in 
   *  hash table as private state. Would need to clone() it if we want to 
   *  be public. 
   **/ 
  public AcceptVV(AcceptStamp as){
    this.stamps = new Hashtable<NodeId, Long>(1);
    stamps.put(as.getNodeId(), as.getLocalClock());
  }


  /** 
   *  Copy constructor 
   **/ 
  public
  AcceptVV(VV toCopyVV){
    VVIterator vvIter = null;
    Object token = null;
    NodeId nid = null;
    long timeStamp = 0;

    this.stamps = new Hashtable<NodeId, Long>(toCopyVV.getSize());
    for(vvIter = toCopyVV.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();
      nid = toCopyVV.getServerByIteratorToken(token);
      timeStamp = toCopyVV.getStampByIteratorToken(token);
      this.stamps.put(nid, new Long(timeStamp));
    }
  }

  /**
   * create a new AcceptVV by replacing all occurences of N1 by N2 
   * @param bm HashMap<N1, N2>
   * @return
   */
  public AcceptVV applyBranchMap(HashMap<NodeId, NodeId> bm){
    Hashtable<NodeId, Long> newStamps = new Hashtable<NodeId, Long>(stamps.size());
    for(NodeId n: stamps.keySet()){
      if(bm.containsKey(n)){
        if(!newStamps.containsKey(bm.get(n)) || (newStamps.get(bm.get(n)) < stamps.get(n))){
          newStamps.put(bm.get(n), stamps.get(n));
        }
      }else{
        if(!newStamps.containsKey(n) || (newStamps.get(n) < stamps.get(n))){
          newStamps.put(n, stamps.get(n));
        }
      }
    }
    return new AcceptVV(newStamps);
  }

  /** 
   *  Create new VV by copying all elements except the specified node 
   **/ 
  public AcceptVV
  makeVVexcludeNode(NodeId exclude){
    Hashtable<NodeId, Long> h = new Hashtable<NodeId, Long>(stamps.size());
    Enumeration keys = stamps.keys();
    while(keys.hasMoreElements()){
      NodeId nid = (NodeId)keys.nextElement();
      if(!nid.equals(exclude)){
        h.put(nid, stamps.get(nid));
      }
    }
    return new AcceptVV(h);
  }


  /** 
   *  Create new VV by copying all elements except the specified node 
   **/ 
  public AcceptVV
  project(VV select){
    Hashtable<NodeId, Long> h = new Hashtable<NodeId, Long>(stamps.size());
    Enumeration keys = stamps.keys();
    while(keys.hasMoreElements()){
      NodeId nid = (NodeId)keys.nextElement();
      if(select.containsNodeId(nid)){
        h.put(nid, stamps.get(nid));
      }
    }
    return new AcceptVV(h);
  }
  /** 
   *  Do sanity check for the custom object serialization 
   *  so that we don't forget updating the readObject and writeObject 
   *  when the object definition is changed 
   **/ 
  final private void doSanityCheck(){
    assert(this.getClass().getDeclaredFields().length == 7);

    //the fields we custom serialize remain the same
    Field[] f = new Field[1];  
    try{
      f[0] = this.getClass().getDeclaredField("stamps");
      assert (f[0].getType().getName().equals("java.util.Hashtable"));

    }catch(NoSuchFieldException ne){
      assert false;
    }
  }

  /** 
   *  Serialization -- optimization to reduce the pikles in the serialization  
   *                   and improve the performance -- removed from  
   *                   TaggedOutputStream to here 
   **/ 
  public void writeExternal(ObjectOutput out)
  throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("AcceptVV -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

//    out.writeInt(this.stamps.size());
//
//    Object token = null;
//    for(VVIterator vvIter = this.getIterator(); vvIter.hasMoreElements();){
//      token = vvIter.getNext();
//      if(SangminConfig.forkjoin){
//        out.writeObject(this.getServerByIteratorToken(token));
//      } else {
//        out.writeLong(this.getServerByIteratorToken(token).getIDint());
//      }
//      out.writeLong(this.getStampByIteratorToken(token));
//    }
    SerializationHelper.writeVV(out, this);
  }


  /** 
   *  Serialization --  optimization to reduce the pikles in the serialization  
   *                   and improve the performance --originated from  
   *                   TaggedInputStream 
   **/ 
  public void readExternal(ObjectInput in)
  throws IOException,
  ClassNotFoundException{

    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    AcceptVV th = SerializationHelper.readVV(in);
//    int avvSize = in.readInt();
//
//    //AcceptStamp[] vvTime = new AcceptStamp[avvSize];
//    Hashtable mystamps = new Hashtable();
//    for(int i=0; i<avvSize; i++){
//      NodeId nodeId;
//      if(SangminConfig.forkjoin){
//        nodeId = (BranchID)in.readObject();
//      } else {
//        nodeId = new NodeId(in.readLong());
//      }
//      mystamps.put(nodeId,  new Long(in.readLong()));
//    }

    Field[] f = this.getClass().getDeclaredFields();      
    try{

      f[0] = this.getClass().getDeclaredField("stamps");

    }catch(NoSuchFieldException ne){
      assert false;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, th.stamps);      
    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
  }

  public int
  getSize(){
    return this.stamps.size();
  }
  /** 
   *  Return true if this hash table is correct 
   **/ 
  private static final boolean verifyHashtable(Hashtable newStamps)
  {
    boolean result = true;
    Iterator it = null;
    Object o = null;
    Map.Entry entry = null;

    it = newStamps.entrySet().iterator();
    if(it.hasNext()){
      entry = (Map.Entry)it.next();
      result = (entry.getKey() instanceof NodeId) &&
      (entry.getValue() instanceof Long);
    }
    return(result);
  }

  /** 
   *  Clone -- since we are immutable, we can just return a pointer to this 
   **/ 
  public final Object clone()
  {
    assert(this instanceof Immutable);
    return this;
  }

  /** 
   *  Return a hash code for this vector 
   **/ 
  public int hashCode()
  {
    int result = 0;
    Enumeration entries = null;

    entries = this.stamps.keys();
    while(entries.hasMoreElements()){
      result = result ^ (entries.nextElement()).hashCode();
    }
    return(result);
  }

  /** 
   *  Return true if this vv is equal to the given vv when ignoring those 
   *  components with BEFORE_TIME_BEGAIN values 
   **/ 
  public final boolean 
  equalsIgnoreNegatives(Object vv) {

    return (vv instanceof VV) && (this.includes((VV)vv) && ((VV)vv).includes(this));
  }

  /** 
   *  Return true if this vv is equal to the given vv 
   **/ 
  public boolean
  equals(Object o){
    //return true if this *equals* vv
    //version vector A *equals* version vector B if
    //every {serverId, counterB} in B has a corresponding
    //{serverId, counterA} in A with counterA == counterB
    // |A| == |B|
    boolean result = true;
    NodeId s = null;
    VV vv = null;
    Long thisStampLong = null;
    VVIterator vvIter = null;
    int numElements = 0;
    Object token = null;

    if(o instanceof VV){
      assert(o instanceof AcceptVV);
      vv = (VV)o;
      vvIter = vv.getIterator();
      while(vvIter.hasMoreElements() && result){
        numElements++;
        token = vvIter.getNext();
        s = vv.getServerByIteratorToken(token);
        thisStampLong = (Long)this.stamps.get(s);
        if((thisStampLong == null) ||
            (vv.getStampByIteratorToken(token) !=
              thisStampLong.longValue())){
          result = false;
        }
      }
      result = result && (this.stamps.size() == numElements);
    }else{
      // o is not a VV
      result = false;
    }
    return(result);
  }

  /** 
   *  increment each clock with 1 except those with BEFORE_TIME_BEGIN. 
   *  
   *  post condition: 
   *  foreach entry e in time 
   *   this.stamps{e.server} = e.stamp + 1 
   *  
   **/ 
  static public AcceptVV 
  incrementAll(VV time){

    VVIterator timeIter = null;
    Long currentTimeStampLong = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;
    Hashtable newStamps = new Hashtable();

    for(timeIter = time.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = time.getServerByIteratorToken(token);
      timeTimeStamp = time.getStampByIteratorToken(token);
      if(timeTimeStamp != AcceptStamp.BEFORE_TIME_BEGAN){
        timeTimeStamp++;
      }
      newStamps.put(nodeId, new Long(timeTimeStamp));
    }

    return new AcceptVV(newStamps);
  }

  /** 
   *  decrement each clock with 1 except those with BEFORE_TIME_BEGIN. 
   *  
   *  post condition: 
   *  foreach entry e in time 
   *   this.stamps{e.server} = e.stamp -1 
   *  
   **/ 
  static public AcceptVV 
  decrementAll(VV time){

    VVIterator timeIter = null;
    Long currentTimeStampLong = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;
    Hashtable newStamps = new Hashtable();

    for(timeIter = time.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = time.getServerByIteratorToken(token);
      timeTimeStamp = time.getStampByIteratorToken(token);
      if(timeTimeStamp != AcceptStamp.BEFORE_TIME_BEGAN){
        timeTimeStamp--;
      }
      newStamps.put(nodeId, new Long(timeTimeStamp));
    }

    return new AcceptVV(newStamps);
  }

  /** 
   *   throw away any trivial component  
   *        whose stamp == AcceptStamp.BEFORE_TIME_BEGAN 
   **/ 
  public AcceptVV 
  dropNegatives(){

    VVIterator timeIter = null;
    Long currentTimeStampLong = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;
    Hashtable newStamps = new Hashtable();

    for(timeIter = this.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      timeTimeStamp = this.getStampByIteratorToken(token);
      if(timeTimeStamp != AcceptStamp.BEFORE_TIME_BEGAN){
        newStamps.put(nodeId, new Long(timeTimeStamp));
      }

    }

    return new AcceptVV(newStamps);
  }

  /** 
   *  Construct a new version vector that has for each nodeId the value that 
   *  is different from VV. E.g. 
   *    VV[n] != This[n] --> ret[n] = This[n]. 
   *    VV[n] not exist, This[n] exist  --> ret[n] = This[n]. 
   *    VV[n] exists, This[n] not exists --> ret[n] = -1; 
   **/ 
  public final AcceptVV getDiff(AcceptVV vv)
  {
    Object token = null;
    AcceptVV newVV = null;
    VVIterator vvIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;


    newStamps = new Hashtable();

    // Iterate through elements of vv to get the diff for each
    vvIter = vv.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = vv.getServerByIteratorToken(token);
      vvStamp = vv.getStampByIteratorToken(token);
      try{
        thisStamp = this.getStampByServer(nodeId);
        if(vvStamp != thisStamp){
          newStamps.put(nodeId, new Long(thisStamp));
        }
      }catch(NoSuchEntryException e){
        // VV has an element that this didn't. Add it to newStamps.
        if(vvStamp != AcceptStamp.BEFORE_TIME_BEGAN){
          newStamps.put(nodeId, new Long(AcceptStamp.BEFORE_TIME_BEGAN));

        }
      }
    }

    // Make sure we didn't miss any elements; iterate through
    // our own elements to check
    vvIter = this.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      thisStamp = this.getStampByIteratorToken(token);

      try{
        vvStamp = vv.getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        // I has an element that vv didn't. Add it to newStamps.
        if(thisStamp != AcceptStamp.BEFORE_TIME_BEGAN){
          newStamps.put(nodeId, new Long(thisStamp));

        }
      }
    }
    return(new AcceptVV(newStamps));
  }

  /*** Construct a new version vector that has for each nodeId the value that
  * is different from VV. E.g. <br>
  *   ret[n] = This[n] - VV[n]. <br>
  *   ret[n] not exist when This[n] not exist <br> 
  *   ret[n] = This[n] when VV[n] not exists <br>
  **/
  public final AcceptVV getRealDiff(VV vv)
  {
    NodeId token = null;
    AcceptVV newVV = null;
    VVIterator thisIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;


    newStamps = new Hashtable();

    //  Iterate through elements of vv to get the diff for each 
    thisIter = this.getIterator();
    while(thisIter.hasMoreElements()){

      nodeId = thisIter.getNext();
      try{
        if(vv.containsNodeId(nodeId)){
          vvStamp = this.getStampByServer(nodeId) - vv.getStampByServer(nodeId);
        }
        else{
          vvStamp = this.getStampByServer(nodeId);
        }
        if(thisStamp != AcceptStamp.BEFORE_TIME_BEGAN){
          newStamps.put(nodeId, new Long(vvStamp));

        }
      }catch(NoSuchEntryException e){
        assert false;
      }

    }
    return(new AcceptVV(newStamps));
  }

  /**
   * return compoenents for which mask >=0 
   * @param mask
   * @return
   */
  public final AcceptVV project(AcceptVV mask){
    NodeId token = null;
    AcceptVV newVV = null;
    VVIterator thisIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;


    newStamps = new Hashtable();

    //  Iterate through elements of vv to get the diff for each 
    thisIter = this.getIterator();
    while(thisIter.hasMoreElements()){

      nodeId = thisIter.getNext();
      try{
        if(mask.containsNodeId(nodeId) && mask.getStampByServer(nodeId) >= 0){
          newStamps.put(nodeId, new Long(this.getStampByIteratorToken(nodeId)));
        }
      }catch(NoSuchEntryException e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
    return(new AcceptVV(newStamps));
  }

  /** 
   * Construct a new version vector that has for each nodeId the minimum
   * of the corresponding stamp for this and for vv
   * Difference from cloneMinVV is that missing entries are treated as -1 
   **/ 
  public final AcceptVV cloneMinVVOmit(VV vv)
  {
    Object token = null;
    AcceptVV newVV = null;
    VVIterator vvIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;

    newStamps = new Hashtable();

    //  Iterate through elements of vv to get the minimum for each 
    vvIter = vv.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      if(stamps.containsKey(token)){
        nodeId = vv.getServerByIteratorToken(token);
        vvStamp = vv.getStampByIteratorToken(token);
        try{
          thisStamp = this.getStampByServer(nodeId);
          if(vvStamp < thisStamp){
            newStamps.put(nodeId, new Long(vvStamp));
          }else{
            newStamps.put(nodeId, new Long(thisStamp));
          }
        }catch(NoSuchEntryException e){
          // VV has an element that this didn't. Add it to newStamps. 
          newStamps.put(nodeId, new Long(vvStamp));
        }
      }
    }

    return(new AcceptVV(newStamps));
  }


  /** 
   * Construct a new version vector that has for each nodeId the minimum
   * of the corresponding stamp for this and for vv
   **/ 
  public final AcceptVV cloneMinVV(AcceptVV vv)
  {
    Object token = null;
    AcceptVV newVV = null;
    VVIterator vvIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;

    newStamps = new Hashtable();

    //  Iterate through elements of vv to get the minimum for each 
    vvIter = vv.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = vv.getServerByIteratorToken(token);
      vvStamp = vv.getStampByIteratorToken(token);
      try{
        thisStamp = this.getStampByServer(nodeId);
        if(vvStamp < thisStamp){
          newStamps.put(nodeId, new Long(vvStamp));
        }else{
          newStamps.put(nodeId, new Long(thisStamp));
        }
      }catch(NoSuchEntryException e){
        // VV has an element that this didn't. Add it to newStamps. 
        newStamps.put(nodeId, new Long(vvStamp));
      }
    }

    // Make sure we didn't miss any elements; iterate through 
    //  our own elements to check 
    vvIter = this.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      thisStamp = this.getStampByIteratorToken(token);
      if(newStamps.get(nodeId) == null){
        newStamps.put(nodeId, new Long(thisStamp));
      }
    }
    return(new AcceptVV(newStamps));
  }

  /** 
   * Construct a new version vector that has for each nodeId the maximum
   * of the corresponding stamp for this and for vv
   **/ 
  public final AcceptVV cloneMaxVV(VV vv)
  {
    Object token = null;
    AcceptVV newVV = null;
    VVIterator vvIter = null;
    NodeId nodeId = null;
    long vvStamp = 0;
    long thisStamp = 0;
    Hashtable newStamps = null;

    newStamps = new Hashtable();

    //  Iterate through elements of vv to get the maximum for each 
    vvIter = vv.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = vv.getServerByIteratorToken(token);
      vvStamp = vv.getStampByIteratorToken(token);
      try{
        thisStamp = this.getStampByServer(nodeId);
        if(vvStamp > thisStamp){
          newStamps.put(nodeId, new Long(vvStamp));
        }else{
          newStamps.put(nodeId, new Long(thisStamp));
        }
      }catch(NoSuchEntryException e){
        // VV has an element that this didn't. Add it to newStamps. 
        newStamps.put(nodeId, new Long(vvStamp));
      }
    }

    // Make sure we didn't miss any elements; iterate through 
    // our own elements to check 
    vvIter = this.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      thisStamp = this.getStampByIteratorToken(token);
      if(newStamps.get(nodeId) == null){
        newStamps.put(nodeId, new Long(thisStamp));
      }
    }
    return(new AcceptVV(newStamps));
  }

  /** 
   * Of all the <NodeId, TimeStamp> pairs in the version vector,
   * return the largest TimeStamp.
   **/ 
  public final long getMaxTimeStamp()
  {
    long largestTimeStamp = Long.MIN_VALUE;
    Object token = null;

    for(VVIterator i = this.getIterator(); i.hasMoreElements();){
      token = i.getNext();
      if(this.getStampByIteratorToken(token) > largestTimeStamp){
        largestTimeStamp = this.getStampByIteratorToken(token);
      }
    }
    return(largestTimeStamp);
  }

  /** 
   * Of all the <NodeId, TimeStamp> pairs in the version vector,
   * return the smallest TimeStamp.
   **/ 
  public final long getMinTimeStamp()
  {
    long smallestTimeStamp = Long.MAX_VALUE;
    Object token = null;

    for(VVIterator i = this.getIterator(); i.hasMoreElements();){
      token = i.getNext();
      if(this.getStampByIteratorToken(token) < smallestTimeStamp){
        smallestTimeStamp = this.getStampByIteratorToken(token);
      }
    }
    return(smallestTimeStamp);
  }

  /**  
   * Return true if this vv is a superset of the given vv
   *
   * Note: whenever change anything here, remember to do the same thing for
   *       method: includes(AcceptStampe as) and also the same methods in
   *       CounterVV.java 
   **/ 
  public final boolean includes(VV vv){
    // return true if this *includes* vv 
    // version vector A *includes* version vector B if 
    // every {serverId, counterB} in B has a corresponding 
    // {serverId, counterA} in A with counterA >= counterB 

    boolean result = true;
    NodeId s = null;
    Object token = null;
    VVIterator vvIter = null;


    for(vvIter = vv.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();

      if (vv.getStampByIteratorToken(token) == AcceptStamp.BEFORE_TIME_BEGAN){
        continue; // ignore the component if it's just a dummy value 
      }
      s = vv.getServerByIteratorToken(token);

      try{
        if(this.getStampByServer(s) < vv.getStampByIteratorToken(token)){
          result = false;
          break;
        }
      }catch(NoSuchEntryException e){
        result = false;
        break;
      }
    }

    return(result);
  }

  /** 
   * Return false if all components of vv are greater than the same in this
   **/ 
  public final boolean includesAnyPartOf(VV vv)
  {
    // return true if this *includesAnyPartOf* vv 

    // that is, if any event that happened before vv could also have 
    // happened before this 

    // that is, if there exists any server s.t. this.getStampByServer(s) 
    // >= vv.getStampByServer(s) 

    //  TEST: If includes is true, includesAnyPartOf must be true 
    //  NOTE: If there exists a server s that is not in vv but is in this 
    //        vector, return true. 

    NodeId s = null;
    boolean result = false;
    Enumeration e = null;

    for(e = this.stamps.keys(); (e.hasMoreElements() && (!result));){
      try{
        s = (NodeId)e.nextElement();
        if(this.getStampByServer(s)== AcceptStamp.BEFORE_TIME_BEGAN){
          continue;
        }
        if(this.getStampByServer(s) >= vv.getStampByServer(s)){
          result = true;
        }
      }catch(NoSuchEntryException excp){
        result = true;
      }
    }
    return(result);
  }

  /** 
   * Return true if any component of "this" is greater than the same in "vv"
   **/ 
  public final boolean isAnyPartGreaterThan(VV vv){
    // return true, only if there exists any server s s.t. 
    //    (1)s is a valid component in both this and vv  
    // && (2)this.getStampByServer(s) > vv.getStampByServer(s) 
    //  Note. the difference of this method and the above method 
    //     is > and >= 
    //     and if there's a s exists in "this", but not exist in "vv", 
    //            isAnyPartGreaterThan() will not return true 
    //        but includesAnyPartOf() will return true 
    NodeId s = null;
    boolean result = false;
    Enumeration e = null;

    for(e = this.stamps.keys(); (e.hasMoreElements() && (!result));){
      try{
        s = (NodeId)e.nextElement();
        if(this.getStampByServer(s) > vv.getStampByServer(s)){
          result = true;
        }
      }catch(NoSuchEntryException excp){
        //  result = true; 
      }
    }
    return(result);

  }
  /** 
   * Return true if this vv has a higher entry for the given node in ts
   *
   * Note: whenever change anything here, remember to do the same thing for
   *       method: includes(AcceptVV avv) and also the same methods in
   *       CounterVV.java
   **/ 
  public final boolean includes(AcceptStamp ts)
  {
    // return true if this has an entry for ts.nodeId and if 
    // that entry has at least as high a counter as ts.acceptStamp 

    NodeId s = null;
    Long thisClockLong = null;
    boolean result = false;

    assert ts != null;
    if(ts.getLocalClock()== AcceptStamp.BEFORE_TIME_BEGAN){
      return true;
    }
    s = ts.getNodeId();
    thisClockLong = (Long)this.stamps.get(s);
    if(thisClockLong != null){
      result = (thisClockLong.longValue() >= ts.getLocalClock());
    }else{
      assert ts.getLocalClock() != AcceptStamp.BEFORE_TIME_BEGAN;
      result = false;
    }
    return(result);
  }

  /**
   * Return the time stamp entry in this VV for the given node
   **/ 
  public final long getStampByServer(NodeId nodeId)
  throws NoSuchEntryException
  {
    Long tsLong = null;

    // return the clock value for the specified server  
    // or throw exception if no server 
    tsLong = (Long)this.stamps.get(nodeId);
    if(tsLong == null){
      throw(new NoSuchEntryException());
    }

    return(tsLong.longValue());
  }

  /** 
   * Return an iterator
   **/ 
  public final VVIterator getIterator()
  {
    //  getNext() of this iterator returns a token 
    VVIterator vvi = new VVIterator(this.stamps.keys());

    return(vvi);
  }

  public List<NodeId> getSortedNodeList(){
    List<NodeId> list = Collections.list(this.stamps.keys());
    Collections.sort(list);
    return list;
  }

  /** 
   * Return the server name stored at the token
   **/ 
  public NodeId getServerByIteratorToken(Object iterToken)
  {
    NodeId nodeId = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    assert(this.stamps.get(nodeId) != null);
    return(nodeId);
  }

  /**
   * Return the time stamp stored at the token
   **/ 
  public long getStampByIteratorToken(Object iterToken)
  {
    NodeId nodeId = null;
    Long tsLong = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    tsLong = (Long)this.stamps.get(nodeId);
    assert(tsLong != null);
    return(tsLong.longValue());
  }

  /** 
   * Return true if this vv contains NodeId n
   **/ 
  public boolean containsNodeId(NodeId n){
    return this.stamps.containsKey(n);
  }

  /**
   * Return true if this > vv (not if this == vv)
   **/ 
  public final boolean greaterThan(VV vv)
  {
    //  return true if this > vv, where A > B if (1) for all {node, stamp} 
    //  in B, A has a larger or equal stamp, and (2) at least one stamp 
    //  in A has a larger value than the corresponding value in B or A 
    //  has more entries than B 
    boolean seenLarger = false;
    boolean result = true;
    NodeId s = null;
    Object token = null;
    VVIterator vvIter = null;
    int numVVElements = 0;

    try{
      for(vvIter = vv.getIterator(); vvIter.hasMoreElements();){
        numVVElements++;
        token = vvIter.getNext();
        s = vv.getServerByIteratorToken(token);
        if(this.getStampByServer(s) <
            vv.getStampByIteratorToken(token)){
          result = false;
          break;
        }else if((!seenLarger) && (this.getStampByServer(s) >
        vv.getStampByIteratorToken(token))){
          seenLarger = true;
        }
      }
    }catch(NoSuchEntryException e){
      result = false;
    }
    if(result && (!seenLarger)){
      //  If we have more entries, even if the common entries match, 
      //  we are still larger 
      seenLarger = (this.stamps.size() > numVVElements);
    }
    return(result && seenLarger);
  }

  public Collection<AcceptStamp> getAllStamps(){
    if(stamps != null){
      ArrayList<AcceptStamp> stampList = new ArrayList<AcceptStamp>(stamps.size());
      for(NodeId nId: stamps.keySet()){
        stampList.add(new AcceptStamp(stamps.get(nId), nId));
      }
      return stampList;
    }else{
      return new Vector<AcceptStamp>();
    }

  }
  /** 
   * Return a string representation
   **/ 
  final public String toString()
  {
    String str = null;
    Object token = null;
    VVIterator vvIter = null;

    StringBuffer sb = new StringBuffer();
    sb.append("AcceptVV:(");

    for(vvIter = this.dropNegatives().getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();

      sb.append("<\"");
      sb.append(this.getServerByIteratorToken(token));
      sb.append("\", ");
      sb.append(this.getStampByIteratorToken(token));
      sb.append("> ");
    }
    // str += ")"; 
    sb.append(")");
    return(sb.toString());
  }
  /**
   * Return a string representation
   **/ 
  public final String toShortString()
  {
    String str = null;
    Object token = null;
    VVIterator vvIter = null;

    StringBuffer sb = new StringBuffer();
    sb.append("AcceptVV:(");

    for(vvIter = this.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();

      if(this.getStampByIteratorToken(token) != -1){
        sb.append("<\"");
        sb.append(this.getServerByIteratorToken(token));
        sb.append("\",");
        sb.append(this.getStampByIteratorToken(token));
        sb.append(">");
      }
    }
    // str += ")"; 
    sb.append(")");
    return(sb.toString());
  }

  /** 
   * Return a string representation
   **/ 
  public final int size()
  {
    return stamps.size();
  }

  /** 
   * Make a dummy startVV with negatives for all known nodes.
   **/ 
  public static AcceptVV
  makeVVAllNegatives(){
    Enumeration allNodes = Config.getKnownNodeIds();
    Vector v = new Vector();
    NodeId n;
    while(allNodes.hasMoreElements()){
      if(SangminConfig.forkjoin){
        n = new BranchID(((NodeId)allNodes.nextElement()).getIDint());
      } else {
        n = (NodeId)allNodes.nextElement();
      }
      AcceptStamp a = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, n);
      v.addElement(a);
    }
    AcceptStamp[] aa = (AcceptStamp[])v.toArray(new AcceptStamp[0]);
    return new AcceptVV(aa);
  }


  /**
   * Unit tests moved to AcceptVVUnit
   **/ 
  public static void main(String[] argv){

  }

  @Override
  public Collection<NodeId> getNodes(){
    // TODO Auto-generated method stub
    return this.stamps.keySet();
  }

  public int compareTo(AcceptVV o){
    boolean iInclO = this.includes(o);
    boolean oInclI = o.includes(this);
    if(iInclO && !oInclI){
      return 1;
    }else if(oInclI && !iInclO){
      return -1;
    }else{
      return 0;
    }
  }
}


/** 
/* $Log: AcceptVV.java,v $
/* Revision 1.41  2007/07/15 06:21:29  zjiandan
/* optimize bw for cp exchange
/*
/* Revision 1.40  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.39  2007/05/31 06:02:00  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.38  2007/05/30 20:30:19  dahlin
/* Added checkpoint to SubscribeBWUnit. Changed outgoingconnection checkpoint send to not send body by default. Told barrier to close sockets when done with them.
/*
/* Revision 1.37  2007/03/15 21:58:31  dahlin
/* Added experimetn to test BW for subscribe for SOSP paper
/*
/* Revision 1.36  2006/09/12 14:57:59  zjiandan
/* Move object custom serialization into individual classes from TaggedOutputStream
/* and TaggedInputStream. The new serialization code consumes a little bit more
/* bytes than the original approach because of java's default object information
/* overhead (I have excluded the field name and type information).
/* The major changes are using java reflection API to serialize "private final"
/* fields (which are very common among our objects that need to custom serialization).
/* Also add sanity check for each class so that we will remember to update
/* the serialization methods when we change the class definitions.
/*
/* Revision 1.35  2006/04/20 03:52:52  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.34  2005/10/13 00:24:23  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.33  2005/06/09 17:11:48  dahlin
/* Moving unit tests for RandomAccessState to junit
/*
/* Revision 1.32  2005/03/05 09:40:35  nayate
/* Added and tested a copy constructor
/*
/* Revision 1.31  2005/03/02 21:43:19  zjiandan
/* Removed some bugs
/*
/* Revision 1.30  2005/02/28 20:25:58  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.29  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.28  2004/10/26 20:28:05  zjiandan
/* fixed include(VV vv) methods to ignore dummy entries.
/*
/* Revision 1.27  2004/10/22 18:13:38  zjiandan
/* cleaned csn from UpdateLog, modified DeleteInv.
/* TBD:
/*    1. clean csn from all subclasses of GeneralInv
/*    2. fix *Inv::cloneIntersectInvaltargetChopStartEnd
/*
/* Revision 1.26  2004/10/12 20:40:13  zjiandan
/* add a new public method: boolean isAnyPartGreaterThan(VV vv)
/*
/* Revision 1.25  2004/10/07 22:20:54  dahlin
/* Fixed some warnings from Findbugs -- none appear to be significant
/*
/* Revision 1.24  2004/07/28 14:27:34  dahlin
/* Added sanity checks for immutable objects
/*
/* Revision 1.23  2004/05/20 02:12:39  dahlin
/* Fixing conflict with amol
/*
/* Revision 1.22  2004/05/20 01:47:19  nayate
/* Added a new method to init a VV with all values set to -1.
/*
/* Revision 1.21  2004/05/20 01:44:07  dahlin
/* Looks like spanning trees are getting set up with TrivialSpanningTreeDirectory (though more testing remains)
/*
/* Revision 1.20  2004/05/19 22:57:53  dahlin
/* Fixed make-all-zeros function
/*
/* Revision 1.19  2004/05/19 18:55:06  nayate
/* Added some verification
/*
/* Revision 1.18  2004/05/14 19:23:21  dahlin
/* Cleanup spanning tree logic; add update bodies
/*
/* Revision 1.17  2004/05/10 22:45:47  dahlin
/* All unit: target tests succeed
/*
/* Revision 1.16  2004/05/10 22:31:08  dahlin
/* Fixed bug in constructor assertion
/*
/* Revision 1.15  2004/05/10 20:48:19  dahlin
/* Clarified RMI exceptions; full version of (stub) DemandReadWorker
/*
/* Revision 1.14  2004/05/10 09:55:30  nayate
/* Changed test cases to use asserts instead of printing to screen
/*
/* Revision 1.13  2004/05/09 19:27:03  dahlin
/* Updated AcceptVV self tests
/*
/* Revision 1.12  2004/04/28 01:38:47  nayate
/* Added a "greaterThan" method
/*
/* Revision 1.11  2004/04/26 03:58:03  zjiandan
/* add containsNodeId() method
/*
/* Revision 1.10  2004/04/22 20:35:01  nayate
/* Copied the "equals" method from CounterVV.java to make it more generic
/* (CounterVV.equals allows comparison with any VV, not just CounterVV).
/*
/* Revision 1.9  2004/04/21 16:07:12  nayate
/* *** empty log message ***
/*
/* Revision 1.8  2004/04/20 20:00:46  nayate
/* Corrected a small bug in AcceptVV after adding a new test
/*
/* Revision 1.7  2004/04/20 19:41:45  nayate
/* Tested AcceptVV
/*
/* Revision 1.6  2004/04/20 18:58:06  nayate
/* Added a new getMaxTimeStamp() method to VV and AcceptVV
/*
/* Revision 1.5  2004/04/18 18:31:15  nayate
/* Added the "equals" method (still untested).
/*
/* Revision 1.4  2004/04/16 22:38:58  nayate
/* Finished (but untested) AcceptVV
/*
/* Revision 1.3  2004/04/16 16:05:16  nayate
/* Subset of PreciseInv implemented + removed some compiler errors.
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */

