package code;

 /** 
 *  Store a tree of PreciseSet objects 
 *  Note: every method is not synchronized. 
 *  assume that only synchronized methods (in DataStore) will invoke  
 *  the methods here. 
 **/ 
import java.util.Vector;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.util.Enumeration;
import java.io.Serializable;

public class AllPreciseSets implements Immutable, Serializable{

 /** 
 *  Private data 
 **/ 
  private PreciseSet rootPS;
  
 /** 
 *  Constructor 
 **/ 
  public
  AllPreciseSets(PreciseSet ps){
    this.rootPS = ps;
  }



 /** 
 *  Follow the tree starting with root until we find the best match for pathName 
 **/ 
  public PreciseSet findExactMatch(String pathName){
    FileNameTokenizer fnt = null;
    String token = null;
    boolean done = false;
    PreciseSet ps = null;
    PreciseSet psChild = null;
    Vector vec = null;
    
    ps = this.rootPS;
    if(pathName.equals("/")){
      return ps;
    }
    assert !pathName.endsWith("/*");
    fnt = new FileNameTokenizer(pathName);
    token = fnt.getNextToken();
    assert(token.equals(""));
    assert(this.rootPS.getFullName().equals(""));
    done = false;
    

    while(!done){
      token = fnt.getNextToken();
      if(token != null){
        psChild = ps.getChild(token);
        if(psChild == null){
          ps = null;
          done = true;
        }else{
          // Continue looking
          ps = psChild;
        }
      }else{
        done = true;
      }
    }
    return ps;
  }

  //-------------------------------------------------------------------------
  // Follow the tree starting with root until we find the exact match
  // or return null if not exact match
  //-------------------------------------------------------------------------
  public PreciseSet findBestMatch(String pathName){
    FileNameTokenizer fnt = null;
    String token = null;
    boolean done = false;
    PreciseSet ps = null;
    PreciseSet psChild = null;
    Vector vec = null;

    fnt = new FileNameTokenizer(pathName);
    token = fnt.getNextToken();
    assert(token.equals(""));
    assert(this.rootPS.getFullName().equals(""));
    done = false;
    ps = this.rootPS;

    while(!done){
      token = fnt.getNextToken();
      if(token != null){
        psChild = ps.getChild(token);
        if(psChild == null){
          done = true;
        }else{
          // Continue looking
          ps = psChild;
        }
      }else{
        done = true;
      }
    }
    return ps;
  }


  //------------------------------------------------------------------------
  // Get LpVV  -- called by read
  //------------------------------------------------------------------------
  public AcceptVV getLpVV(String objName){
    PreciseSet sps = this.findBestMatch(objName);
    assert sps != null;
    assert !objName.endsWith("/*");
    
    //System.out.println("sps.getFullName():"+sps.getFullName());
    //System.out.println("objName: " + objName);
    if (objName.equals("/")){
      objName = "";
    }

    if(sps.getFullName().equals(objName)){//match exactly
      return sps.getMyMaxLpVV();
    }else{
      return sps.getOtherChildrenMaxLpVV();
    }
  }
  
  //-------------------------------------------------------------------------
  // Split off a PreciseSet from a given other PreciseSet
  //-------------------------------------------------------------------------
  public  PreciseSet
  split(PreciseSet ps, String newName){
    PreciseSet newPS = null;

    newPS = ps.split(newName);
    return(newPS);
  }

  //-------------------------------------------------------------------------
  // Split off a PreciseSet from a given pathName matching Node
  //-------------------------------------------------------------------------
  public  PreciseSet
  split(String pathName, String newName)
  throws NoMatchPreciseSetException{
    PreciseSet newPS = null;
    PreciseSet matchPS = null;
    matchPS = this.findExactMatch(pathName);
    if (matchPS == null){
      throw new NoMatchPreciseSetException(pathName + " does not exist in the PreciseSet Tree" 
                                           + "can't be splitted further" );
    }
    newPS = matchPS.split(newName);
    
    return(newPS);
  }

  //-------------------------------------------------------------------------
  // join all this.children, remove this.children, update this.other.lpvv
  //-------------------------------------------------------------------------
  public boolean
  join(String pathName){
    PreciseSet matchPS = null;
    matchPS = this.findExactMatch(pathName);
    if(matchPS == null){
      return false;
    }
    matchPS.join();
    return true;
  }

  //-------------------------------------------------------------------------
  //     add connection pointer to the corresponding place in ISStatus tree
  //-------------------------------------------------------------------------  
  public void addConnection(SubscriptionSet ss,
                            ConnectionState ic){
    //Invariant for Ptreenode:
    // a live con is in a Ptreenode.liveconlist if and only if the 
    // connection's subset compeletely cover PTreeNode.name.

    // add conID into any node completely included by subset
    // => L1: might cause some "other" node to split so that one part is completely included by subset
    //        (hence add ConID into it) and only the remaining nonoverlapping part in "other"
    // or L2: ignore it without add conID into the overlapping subset.
    
    // the implementation here, take the L1 solution as normally the subscription set
    // will be stable.
    this.rootPS.split(ss);
    this.rootPS.addLiveCons(ss, ic);
  }

  private static boolean warnedRC = false;

  //-------------------------------------------------------------------------
  //  remove connection pointer from the corresponding place in ISStatus tree
  //-------------------------------------------------------------------------  
  public void removeConnection(SubscriptionSet ss,
                                            ConnectionState ic,
                                            VV newLpvv){
    //remove conID from those PTreeNodes that overlapping subset. It is possible
    //that the conID is not in some of the PTreeNodes because of L2. 
    // only update the lpvv of those nodes that completely covered by subset.
    if(!warnedRC){
      Env.performanceWarning("addConnection and removeConnection to ISStatus "
                             + " would imply a split traverse the tree first."
                             + " it might be a better choice to expose this " 
                             + " option to the controller to decide the depth"
                             + " and breadth of the ISStatus tree");
      Env.tbd(" every addConnection/removeConnection would split the ISStatus tree"
              + " need to add policies to jion subtrees, the join method is already there."); 
      warnedRC = true;
    }

    this.rootPS.split(ss);
    this.rootPS.removeLiveCons(ss, ic, newLpvv);
  }

  //-------------------------------------------------------------------------
  // update all corresponding PTreeNode's lpvv <pathName, lpvv>*
  //-------------------------------------------------------------------------  
  public void updateLpVV(Vector lpvvRecs){
    for( Enumeration e = lpvvRecs.elements(); e.hasMoreElements();){
      LPVVRecord nextRec = (LPVVRecord)e.nextElement();
      PreciseSet matchingNode = this.findExactMatch(nextRec.getPathName());
      if(matchingNode != null){//match exactly
        matchingNode.updateMyLpVV(nextRec.getLpVV());
        if(nextRec.getChildrenLpVV()!=null){
          matchingNode.updateAllChildLpVV(nextRec.getChildrenLpVV());
        }
      }else{
	//assert false: "can't find matchingPreciseSet for: "+ nextRec + " from : " + this;
        //(1)ignore  -- previous approach --> receiver won't be precise even if it gets all the invalidates 
        //(2)another option is to split and update
	this.rootPS.split(SubscriptionSet.makeSubscriptionSet(nextRec.getPathName()));
	matchingNode = this.findExactMatch(nextRec.getPathName());
	assert matchingNode != null;
	matchingNode.updateMyLpVV(nextRec.getLpVV());
        if(nextRec.getChildrenLpVV()!=null){
          matchingNode.updateAllChildLpVV(nextRec.getChildrenLpVV());
        }
      } 
    }
  }

  //-------------------------------------------------------------------
  // return pairs of (pathName, matching lpvv, other.lpvv) 
  // note: for pathName like /a/*, will return ("/a", a.myLpVV, a.childLPVV)
  //       instead of ("/a/*", null, a.childLPVV)
  //       e.g. will include the directory lpvv besides childMInlpVV
  //-------------------------------------------------------------------
  public Vector getMatchingLpvvs(SubscriptionSet ss, AcceptVV startVV){
    Vector ret = new Vector();
    boolean dbg = false;
    for(Enumeration e = ss.getFullPaths().elements(); e.hasMoreElements();){
      String pathName = (String)e.nextElement();
      if(dbg){
        System.out.print("pathName:" + pathName);
      }
      PreciseSet matchingNode = this.findBestMatch(pathName);
      assert matchingNode != null;
      String nodeFullName = pathName;
      boolean includesEveryChild = false;
      if (pathName.endsWith("/*")){
        includesEveryChild = true;
	//System.out.println("including all");
        nodeFullName = pathName.substring(0, pathName.length()-2);
      }
      AcceptVV mylp = matchingNode.getMyMaxLpVV();
      AcceptVV myclp = matchingNode.getChildMinLpVV();
      AcceptVV myAncestorOtherLP = matchingNode.getOtherChildrenMaxLpVV();
      if(nodeFullName.equals(matchingNode.getFullName())){//has exact matched node
        
	if(includesEveryChild){
          
	  if((!startVV.includes(mylp)) || ((myclp != null)&&(!startVV.includes(myclp)))){
	    LPVVRecord lr = new LPVVRecord(nodeFullName,
					   mylp,
					   myclp); 
	    if(dbg){
	      System.out.println("add " + lr + "\n");
	    }
	    ret.add(lr);
	  }
        }else{//no child
	  //System.out.println("no child");
          if(!startVV.includes(mylp)){
	    LPVVRecord lr = new LPVVRecord(nodeFullName,
					 mylp,
					 null); 
	  
	    if(dbg){
	      System.out.println("add " + lr + "\n");
	    }
	    ret.add(lr);
	  }
        }
      } else{//should be the ancestor's otherChildLpVV
	if( dbg ){
	  Env.dprintln(dbg, "no exactly match ptreeNode for " +pathName 
		     + " found bestmatch: " + nodeFullName);
	}
	if(!startVV.includes(myAncestorOtherLP)){
	  LPVVRecord lr = new LPVVRecord(nodeFullName,
					 myAncestorOtherLP,
					 myAncestorOtherLP); 
	  if(dbg){
	    System.out.println("add " + lr + "\n");
	  }
	     
	  ret.add(lr);
	}
      }
    }
    return ret;
  }


  //-------------------------------------------------------------------
  // return min of LpVVs of all matching nodes
  //-------------------------------------------------------------------
  public AcceptVV getMinLpVV(SubscriptionSet ss){
    CounterVV ret = null;
    // int i = 0;
    for(Enumeration e = ss.getFullPaths().elements(); e.hasMoreElements();){
      String pathName = (String)e.nextElement();
      //i++;
      //System.out.println("pathName["+ i + "]= " + pathName); 
      PreciseSet matchingNode = this.findBestMatch(pathName);
      assert matchingNode != null;
      String nodeFullName = pathName;
      boolean includesEveryChild = false;
      if (pathName.endsWith("/*")){
        includesEveryChild = true;
        nodeFullName = pathName.substring(0, pathName.length()-2);
      }

      if(nodeFullName.equals(matchingNode.getFullName())){//has exact matched node
        if(includesEveryChild){
          if(ret == null){
            ret = new CounterVV(matchingNode.getChildMinLpVV());
          }else{
            ret.setToMinVV(matchingNode.getChildMinLpVV());
          }
        }else{//no child
          if(ret == null){
            ret = new CounterVV(matchingNode.getMyMaxLpVV());
          }else{
            ret.setToMinVV(matchingNode.getMyMaxLpVV());
          }
        }
      } else{//should be the ancestor's otherChildLpVV
        if(ret == null){
          ret = new CounterVV(matchingNode.getOtherChildrenMaxLpVV());
        }else{
          ret.setToMinVV(matchingNode.getOtherChildrenMaxLpVV());
        }
      }
    }

    if (ret == null){
      System.out.println("WRONG: getMinLPVV(" + ss + ") in AllPreciseSets: " 
                         + this + "\n is null");
    }
    assert ret != null;
    return ret.cloneAcceptVV();
  }


  //-------------------------------------------------------------------------
  // Convert to a string
  //-------------------------------------------------------------------------
  public String
  toString(){
    return rootPS.toString();
  }

  //-------------------------------------------------------------------------
  // Convert lpvvRecs vector to a string
  //-------------------------------------------------------------------------
  public static String toStringLPVVRecordVector(Vector lpvvRecs){
    String str = "";
    int i = 0;
    for( Enumeration e = lpvvRecs.elements(); e.hasMoreElements();){
      Object next = e.nextElement();
      assert next instanceof LPVVRecord;
      LPVVRecord nextRec = (LPVVRecord)next;
      str += "v[" + i + "]:" + nextRec + "\n";
      i++;
    } 
    return str;
  }

  //-------------------------------------------------------------------------
  // Used for testing
  //-------------------------------------------------------------------------
  public static void
  main(String argv[]){
    //moved to AllPreciseSetsUnit.java
  }
}

//---------------------------------------------------------------------------
/* $Log: AllPreciseSets.java,v $
/* Revision 1.18  2007/07/15 06:21:29  zjiandan
/* optimize bw for cp exchange
/*
/* Revision 1.17  2007/07/12 17:02:32  zjiandan
/* *** empty log message ***
/*
/* Revision 1.16  2007/07/11 19:08:07  zjiandan
/* clean IncommingConnection
/*
/* Revision 1.15  2007/06/29 01:01:41  zjiandan
/* *** empty log message ***
/*
/* Revision 1.14  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.13  2007/05/31 06:02:00  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.12  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.11  2007/03/11 04:16:54  zjiandan
/* Add timeout into read interface and expose acceptstamp in LocalInterface.write.
/*
/* Revision 1.10  2007/03/06 18:25:50  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.9  2006/08/31 14:54:13  dahlin
/* DataStore restore from checkpoint seems to work now (including ISStatus and cVV)
/*
/* Revision 1.8  2006/08/25 20:31:45  dahlin
/* Added serialization to PreciseSets (plan is to use serialization rather than sync() for persistence). 
/* Moved PreciseSet unit tests to junit (PreciseSetUnit)
/*
/* Revision 1.7  2006/08/15 21:46:23  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.6  2006/04/20 21:41:59  zjiandan
/* add stub to make old code compilng happy.
/*
/* Revision 1.5  2006/04/20 03:52:52  zjiandan
/* Callbacks merged with runTime.
/**/
//---------------------------------------------------------------------------
