package code;

/* AttachJoinUnit.java
 *
 * Unit test for split join scenario:
 *  
 *   Four nodes, alpha, beta, gamma, delta
 *
 *   (1) alpha issues random writes to a, b, c
 *   
 *   (2) beta subscribes for a from alpha
 *       gamma subscribes for b from alpha
 *   
 *   (3) delta subscribes * from beta and gamma
 *       ==> delta reads /a, /b precise
 *           
 *  This class is delta.
 *  see papers/osdi2004/submitted.pdf splitjoin example for more details
 */


import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;
import code.security.SecureURANode;

import code.security.SangminConfig;
import code.security.holesync.filter.Filter;
import code.security.holesync.filter.SubscriptionSetFilter;
/**
 * TBD: Update class name
 */
public class AttachJoinUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals
  protected static boolean dbg = true;
  protected static boolean useSecure = (SangminConfig.securityLevel == SangminConfig.COMPLETE);
  private Process rmiregistry;
  private Process alpha;
  private Process beta;
  private Process gamma;

  private BarrierServer barrier;
  public static final int BARRIER_PORT = 9713;

  //
  // it turns out that if both nodes write another object /c.
  // /b might not be able to recover by subscribing delta itself
  // see self_subscribe_not_enough.txt
  //
  public static final boolean withThirdObj = false;

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public AttachJoinUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    if(verbose){
      System.out.println("AttachJoinUnit test...");
    }
    System.gc();
    if(vverbose){
      RandomAccessStateUnit.memoryReport();
    }
    /* 
     * TBD: Insert other fixture code here
     * e.g., this.objectUnderTest = new ClassToBeTested(...);
     */
    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    Thread.sleep(2000);

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    if(verbose){
      System.out.println("rmiregistry started");
    }
    Thread.sleep(2000);

    //
    // Start barrier server
    //
    barrier = new BarrierServer(BARRIER_PORT, 4, BarrierServer.LOOP_FOREVER);
    barrier.start();

    //
    // Set up config files
    //
    makePractiConfig(CONFIG_PATH);

    //
    // Set the connection style
    //
    OutgoingConnectionWorker.setDbgWithFileOutputStream(false);

    //
    // Start helper processes
    //
    alpha = startHelper(ALPHA_ID);
    beta = startHelper(BETA_ID);
    gamma = startHelper(GAMMA_ID);
  }


  /* 
   * Config info
   */
  public static final String CONFIG_PATH = "test" + File.separatorChar 
  + "tmp.AttachJoinUnit.config";
  public static final long ALPHA_ID = 10;
  public static final long BETA_ID = 20;
  public static final long GAMMA_ID = 30;
  public static final long DELTA_ID = 40;


  public static final String ALPHA_IS = "/a:/b:/c";
  public static final String BETA_IS = "/a";
  public static final String GAMMA_IS = "/c";
  public static final String DELTA_IS = "/a:/b:/c";

  public static final int TOTALWRITES = 1;

  /*
   * Just make a 4-node config file on localhost in the 
   * specified path
   */
  protected void  makePractiConfig(String path){

    Config.createEmptyConfig();
    Config.readKeys();
    long NODE_0_ID = ALPHA_ID;
    long NODE_1_ID = BETA_ID;
    long NODE_2_ID = GAMMA_ID;
    long NODE_3_ID = DELTA_ID;

    String NODE_0_IP = "localhost";
    String NODE_1_IP = "localhost";
    String NODE_2_IP = "localhost";
    String NODE_3_IP = "localhost";

    String NODE_0_IS = ALPHA_IS;
    String NODE_1_IS = BETA_IS;
    String NODE_2_IS = GAMMA_IS;
    String NODE_3_IS = DELTA_IS;

    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
        NODE_0_IP,
        1188,
        1189,
        1191,
        1192,
        1190,
        "test" + File.separatorChar + "local-" + 
        NODE_0_ID + ".db",
        NODE_0_IS,
        -1L,
        NODE_0_IP,
        1193,
        1194,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
        NODE_1_IP,
        1288,
        1289,
        1291,
        1292,
        1290,
        "test" + File.separatorChar + "local-" + 
        NODE_1_ID+".db",
        NODE_1_IS,
        -1L,
        NODE_1_IP,
        1293,
        1294,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_2_ID),
        NODE_2_IP,
        1388,
        1389,
        1391,
        1392,
        1390,
        "test" + File.separatorChar + "local-" + 
        NODE_2_ID+".db",
        NODE_2_IS,
        -1L,
        NODE_2_IP,
        9793,
        9794,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_3_ID),
        NODE_3_IP,
        1488,
        1489,
        1491,
        1492,
        1490,
        "test" + File.separatorChar + "local-" + 
        NODE_3_ID+".db",
        NODE_2_IS,
        -1L,
        NODE_3_IP,
        1493,
        1494,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.writeToFile(path);
  }



 /** 
 *  Start helper process. Set global writer to refer to helper process. 
 **/ 
  private Process startHelper(long myNodeId){
    Process helper = null;
    //
    // Start the helper PRACTI node
    // "make foo.unit" runs "java foo" with appropriate arguments
    //
    // Use "make" to figure out what arguments needed to
    // set up class paths, etc. Grab the command line
    // rather than just run make b/c we want a handle
    // on the helper process so we can kill it later.
    //
    try{
      helper = Runtime.getRuntime().exec("./runClass.sh code.AttachJoinUnitHelper " + myNodeId);
      if(verbose){
        System.out.println("helper-" + myNodeId + " started");
      }

      setUpHelperIO(helper, "helper-" + myNodeId);
    }
    catch(Exception ex){
      ex.printStackTrace();
      assert(false);
    }
    return helper;
  }

 /** 
 *  Wait for the helper process to create the files 
 *  I want to read. Spawn "/dev/null" threads to 
 *  consume itstheir output so it doesn't block. 
 **/ 
  private void setUpHelperIO(Process helper, String tag){
    InputStream is = helper.getInputStream();
    Thread dn = new DvNull(is, "AttachJoinUnitHelper " + tag + " stdout", 
        verbose, vverbose);
    dn.start();
    is = helper.getErrorStream();
    dn = new DvNull(is, "AttachJoinUnitHelper " + tag + " stderr",
        verbose, vverbose);
    dn.start();
  }



 /** 
 *  Kill helper process and kill rmiregistry 
 **/ 
  protected void tearDown() throws Exception{
    if(alpha != null){
      alpha.destroy();
    }
    if(verbose){
      System.out.println("helper alpha terminated");
    }

    if(beta != null){
      beta.destroy();
    }
    if(verbose){
      System.out.println("helper beta terminated");
    }

    if(gamma != null){
      gamma.destroy();
    }

    if(verbose){
      System.out.println("helper gamma terminated");
    }

    barrier.interrupt();


    rmiregistry.destroy();

    // 
    // the previous Unit Test might not completely kill
    // rmiregistry when the second unit test is about to run
    // 
    // Any unit test setup should clean all rmiregistry
    // before start rmiregistry
    // otherwise it is possible that someone else kills the
    // rmiregistry when this unit just starts it.
    //
    Thread.sleep(2000);
    if(verbose){
      System.out.println("rmiregistry terminated");
    }
    System.gc();
    super.tearDown();
  }






 /** 
 *  Test split join 
 *  
 *  this is for Delta 
 *  (1) wait for alpha, beta, gamma up and running 
 *   
 *  (2) wait for beta/gamma subscribe from alpha 
 *   
 *  (3) wait for alpha write and beta/gamma done 
 *   
 *  (4) subscribe for /a:/b:/c from beta, gemma 
 *   
 *  (5) read /a /c-> precise /b-->imprecise 
 *   
 *  (6) subscribe from myself 
 *  
 *  (7) wait Untill read /b Precise 
 *  
 **/ 
  public void testAttachJoin(){
    URANode node = null;
    BarrierClient bc = new BarrierClient("127.0.0.1", BARRIER_PORT, 0);
    NodeId myNodeId = new NodeId(DELTA_ID);
    NodeId alphaNodeId = new NodeId(ALPHA_ID);
    NodeId betaNodeId = new NodeId(BETA_ID);
    //NodeId gammaNodeId = new NodeId(GAMMA_ID);


    AcceptVV finalVV = null;
    AcceptStamp[] finalAS = new AcceptStamp[2];
    finalAS[0] = new AcceptStamp(TOTALWRITES-1, alphaNodeId);
    finalAS[1] = new AcceptStamp(TOTALWRITES-1, betaNodeId);
    finalVV = new AcceptVV(finalAS);

    SubscriptionSet ssa = SubscriptionSet.makeSubscriptionSet("/a");
    SubscriptionSet ssb = SubscriptionSet.makeSubscriptionSet("/b");
    if(vverbose){
      System.out.println("DELTA ID: " + myNodeId.toString());
    }


    //assert(myNodeId.equals(new NodeId(DELTA_ID)));
    if(useSecure){
      SubscriptionSetFilter ssf = new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*"));
      LinkedList<Filter> f = new LinkedList<Filter>();
      f.add(ssf);
	node = new SecureURANode(CONFIG_PATH, myNodeId,
				Controller.NULL_CONTROLLER,
				true,//cleanDB
				true,
				new code.security.liveness.TrustedServerTraceLivenessFilter(null), f);//noSynLog
    }else{
	node = new URANode(CONFIG_PATH, myNodeId,
			   Controller.NULL_CONTROLLER,
			   true,//cleanDB
			   true);//noSynLog
    }

    // make sure the accumulate time is long enough so that we don't
    // get PInvals when we are expecting ImpInval
    node.getCore().setMaxAccumulateTime(2000); 


    LocalInterface li = node.getLocalInterface();
    RMIClient ci = node.getRMIClientInterface();

    try{
      if(verbose){
        System.out.println("Hello from AttachUnit -- DELTA");
      }

      if(verbose){
        System.out.println("barrier 1 wait for everyone up and runing");
      }

      bc.sendBarrierRequest(0, 0); // 1 all nodes up and running
      if(verbose){
        System.out.println(" 1 All nodes up and running.");
      }



      //
      // me, delta, subscribes from Beta for "/b", from Alpha for "/a"
      //
      try{
        if(vverbose){
          System.out.println(" Delta subscribe frin beta for /a");
        }
        ci.subscribeInval(betaNodeId, myNodeId, 
            ssb,
            AcceptVV.makeVVAllNegatives(), false);
        if(vverbose){
          System.out.println(" Delta subscribe from alpha for /b");
        }
        ci.subscribeInval(alphaNodeId, myNodeId, 
            ssa,
            AcceptVV.makeVVAllNegatives(), false);
        if(vverbose){
          System.out.println(" Delta subscribe from alpha for /b done.");
        }
      }
      catch(Exception e){
        System.out.println("Exception on subscribe: " + e.toString());
        assert(false);
      }


      bc.sendBarrierRequest(0, 0); // 2 all subscription set up
      if(verbose){
        System.out.println(" 2 all subscription set up.");
      }


      //
      // wait syncCheck read /a|/b precise
      //
      syncCheck(finalVV, node);


      // waitUntil /a precise
      waitForPreciseInvalid(new ObjId("/a"), li, "DELTA");

      // make sure /b imprecise
      assert(!li.isPrecise(new ObjId("/b")));
      if(dbg){
        node.getCore().checkState();
      }
      // subscribe from myself
      try{

        ci.subscribeInval(myNodeId, myNodeId, 
            ssb,
            node.getCore().getLpVV(ssb), false);
      }
      catch(Exception e){
        System.out.println("Exception on subscribe: " + e.toString());
        assert(false);
      }

      if(vverbose){
        System.out.println("Delta::subscribe of subscribeId= " + ssb
            + " from self done.");
      }

      // waitUntil /b precise
      waitForPreciseInvalid(new ObjId("/b"), li, "DELTA");



      bc.sendBarrierRequest(0, 0); // 3 helpers can die now
      if(verbose){
        System.out.println(" 3 helpers can die now.");
      }
      try{
        //
        // wait for helpers to end.
        //
        alpha.waitFor();
        if(alpha.exitValue() != 0){
          fail("alpha exit abnormally");
        }
        beta.waitFor();
        if(beta.exitValue() != 0){
          fail("alpha exit abnormally");
        }
        gamma.waitFor();
        if(gamma.exitValue() != 0){
          fail("alpha exit abnormally");
        }
      }catch(InterruptedException ie){
        //ignore;
      }

      /*
      try{
	Thread.sleep(2000);
      }catch(InterruptedException ie){
	//ignore;
      }
       */
    } 
    finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      node.shutdown();
      node = null;
      System.gc();
      if(verbose){
        System.out.println("delta cleaningup.");
      }
    }

  }

 /** 
 *  syncCheck 
 **/ 
  public static void syncCheck(AcceptVV as, URANode unode){
    while(true){
      try{
        if(vverbose){
          System.out.println("syncCheck " + as);
        }
        unode.getCore().syncCheck(as);
        if(vverbose){
          System.out.println("syncCheck " + as + " done.");
        }
        break;
      }catch(InterruptedException ie){
        //ignore
      }
    } 
  }

 /** 
 *  syncCheck 
 **/ 
  public static void syncCheck(AcceptStamp as, URANode unode){
    while(true){
      try{
        if(vverbose){
          System.out.println("syncCheck " + as);
        }
        unode.getCore().syncCheck(as);
        if(vverbose){
          System.out.println("syncCheck " + as + " done.");
        }
        break;
      }catch(InterruptedException ie){
        //ignore
      }
    } 
  }

 /** 
 *  waitForPreciseInvalid 
 **/ 
  public static void waitForPreciseInvalid(ObjId checkObj, 
      LocalInterface checkLi, 
      String tag){
    int count=0;
    int maxtry = 100;
    if(verbose){
      System.out.println(tag + "waitForPreciseInvalid " + checkObj + " start");
    }
    while(true){
      count++;

      if(vverbose){
        System.out.println(tag + " waitForPreciseInvalid " + checkObj 
            + " tried " + count + " times");
      }
      if(count>=maxtry){

        fail(tag + " waitForPreciseInvalid("+checkObj 
            + ") reached maxtry and failed");
      }
      try{
        BodyMsg b = checkLi.read(checkObj, 0, 1, false, false);
        if(b == null){
          assert(false); // Timeout
        }
        assert false;
      }
      catch(ReadOfInvalidRangeException roire){
        if(checkLi.isPrecise(checkObj)){
          if(vverbose){
            System.out.println(tag + " waitForPreciseInvalid " + checkObj
                + " done.");
          }
          return;
        }else{
          if(vverbose){
            System.out.println(tag + " waitForPreciseInvalid " 
                + checkObj + " still imprecise");
          }
        }
      }
      catch(ReadOfHoleException rohe){
        assert(false);
      }
      catch(EOFException eof){
        assert(false);
      }
      catch(IOException iof){
        assert(false);
      }
      catch(ObjNotFoundException onfe){
        // This is OK -- we have not yet received the inval -- retry
      }
      try{
        Thread.sleep(200); // Wait before retry
      }
      catch(InterruptedException e){
      }
    }

  }

  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(AttachJoinUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   AttachJoinUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "AttachJoinUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new AttachJoinUnit("test" + s[ii]));
        }

      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }

}

//---------------------------------------------------------------------------
/*$Log: AttachJoinUnit.java,v $
/*Revision 1.3  2007/08/05 04:43:54  zjiandan
/*SocketServer shutdown quietly
/*
/*Revision 1.2  2007/07/04 21:47:48  zjiandan
/*fix unittests
/**/
//---------------------------------------------------------------------------


