package code;

 /** 
 *  Represents a message that contains the body of a prefetched/pushed object 
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

public class BodyMsg implements Externalizable{

  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

 /** 
 *  Data members 
 **/ 
  private final ObjInvalTarget obj;  
  private final AcceptStamp timestamp;
  private final ImmutableBytes body;
  private final double priority;
  private long requestId = EMPTY_RESULT;    // Token for matching requests and replies
  private boolean isDemand;  // distinguish between demand-fetch and prefetch
  
  public static final long NOT_DEMAND = 1;
  public static final long INVALID_REQUEST = 2;
  public static final long EMPTY_RESULT = 3;

  
  
 /** 
 *  Empty Constructor - for serialization 
 **/ 
  public
  BodyMsg(){
      
    this.obj = null;
    this.timestamp = null;
    this.body = null;
    this.isDemand = false;
    this.requestId = EMPTY_RESULT;;
    this.priority = Core.DEFAULT_PRIORITY;
  }

 /** 
 *  Constructor 
 **/ 
  public
  BodyMsg(ObjId objId,
          long offset,
          long len,
          AcceptStamp accept,
          ImmutableBytes data,
          boolean isDemand){
          
    this.obj = new ObjInvalTarget(objId, offset, len);
    this.timestamp = accept;
    this.body = data;
    this.isDemand = isDemand;
    this.priority = Core.DEFAULT_PRIORITY;
  }

 /** 
 *  Constructor 
 **/ 
  public
  BodyMsg(ObjId objId,
          long offset,
          long len,
          AcceptStamp accept,
          ImmutableBytes data,
          boolean isDemand,
          double newPriority){

    this.obj = new ObjInvalTarget(objId, offset, len);
    this.timestamp = accept;
    this.body = data;
    this.isDemand = isDemand;
    this.priority = newPriority;
  }

 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    if(this.getClass().getDeclaredFields().length != 15){
      System.out.println(this.getClass().getDeclaredFields().length);
    }
    assert(this.getClass().getDeclaredFields().length == 15);
     
    //the fields we custom serialize remain the same
    Field[] f = new Field[6];  
      try{
        f[5] = this.getClass().getDeclaredField("obj");
        f[0] = this.getClass().getDeclaredField("timestamp");
        f[1] = this.getClass().getDeclaredField("body");
        f[2] = this.getClass().getDeclaredField("priority");        
        f[3] = this.getClass().getDeclaredField("requestId");
        f[4] = this.getClass().getDeclaredField("isDemand");
        
        
        assert (f[5].getType().getName().equals("ObjInvalTarget"));
        assert (f[0].getType().getName().equals("AcceptStamp"));
        assert (f[1].getType().getName().equals("ImmutableBytes"));
        assert (f[2].getType().getName().equals("double"));
        assert (f[3].getType().getName().equals("long"));
        assert (f[4].getType().getName().equals("boolean"));
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("PreciseInv -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }
    
    //objInvalTarget
    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getObjInvalTarget().getOffset());
    out.writeLong(this.getObjInvalTarget().getLength());
    //acceptStamp
    out.writeLong(this.getAcceptStamp().getLocalClock());
    if(SangminConfig.forkjoin){
      out.writeObject(this.getAcceptStamp().getNodeId());
    } else {
      out.writeLong(this.getAcceptStamp().getNodeId().getIDint());
    }

    //body
    out.writeInt(this.getBody().getLength());
    out.write(this.getBody().dangerousGetReferenceToInternalByteArray());

    out.writeDouble(this.getPriority());
    out.writeLong(this.requestId);
    out.writeBoolean(this.isDemandReply());
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    String objStr = (String)(in.readObject());
    long offs = in.readLong();
    long len = in.readLong();
      
    long c = in.readLong();
    AcceptStamp as;
    if(SangminConfig.forkjoin){
      as = new AcceptStamp(c, (BranchID)in.readObject());
    } else {
      as = new AcceptStamp(c, new NodeId(in.readLong()));
    }
    

    int bodySize = in.readInt();
    byte[] data = new byte[bodySize];
    in.readFully(data);
    
    
    double mypriority = in.readDouble();
    this.requestId = in.readLong();
    this.isDemand = in.readBoolean();
    
    ObjInvalTarget myobj = new ObjInvalTarget(new ObjId(objStr),
                                              offs,
                                              len);
    //as
    //                    new ImmutableBytes(data),
    //                    isDemand,
    //                    priority);

    
    Field[] f = new Field[4];
      
      try{

        f[0] = this.getClass().getDeclaredField("obj");
        f[1] = this.getClass().getDeclaredField("timestamp");
        f[2] = this.getClass().getDeclaredField("body");
        f[3] = this.getClass().getDeclaredField("priority");
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        
        f[0].set(this, myobj);
        f[1].set(this, as);
        f[2].set(this, new ImmutableBytes(data));
        f[3].setDouble(this, mypriority);
        
      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }


 /** 
 *  Make this BodyMsg blank 
 **/ 
  public void
  setBlank() {
    requestId = EMPTY_RESULT;
  }
  
 /** 
 *  Return the inval target 
 **/ 
  public final ObjInvalTarget
  getObjInvalTarget(){
    assert(this.obj instanceof Immutable);
    return(this.obj);
  }

 /** 
 *  return the objId 
 **/ 
  public final ObjId
  getObjId(){
    ObjId id;
    id = this.obj.getObjId();
    assert(id instanceof Immutable);
    return id;
  }

 /** 
 *  return the priority 
 **/ 
  public final double
  getPriority(){
    return this.priority;
  }
    
 /** 
 *  Return the accept stamp 
 **/ 
  public final AcceptStamp
  getAcceptStamp(){
    assert(this.timestamp instanceof Immutable);
    return(this.timestamp);
  }

 /** 
 *  Return the object body 
 **/ 
  public final ImmutableBytes
  getBody(){
    assert(this.body instanceof Immutable);
    return(this.body);
  }

 /** 
 *  Return the length of the object body 
 **/ 
  public final long
  getLength(){
    return(this.body.getLength());
  }

 /** 
 *  Return the request ID 
 **/ 
  public final long
  getRequestId(){
    return requestId; // Primitive type --> copy passed out
  }

 /** 
 *  Set the request ID 
 **/ 
  public final void
  setRequestId(long r){
    this.requestId = r;
  }

 /** 
 *  Set isDemand 
 **/ 
  public final void
  setIsDemand(boolean isDemand){
    this.isDemand = isDemand;
  }

 /** 
 *  Return true if this message is a demand reply 
 **/ 
  public final boolean
  isDemandReply(){
    return isDemand;
  }

 /** 
 *  Convert this object to a String representation 
 **/ 
  public String
  toString(){
    return new String("BodyMsg: " + obj.getObjId().getPath() + " : [" + obj.getOffset() +
                      "," + obj.getLength() + "]" + " : [" + timestamp + "]" + 
                      " : " + this.priority + " reqId=" + requestId + 
                      " : <" + (body==null ? null:body.toString()) + ">");
  }

 /** 
 *  equals()-- 
 *            compares this object to the specified object. The result 
 *            is true if and only if the argument is not null and is  
 *            a BodyMsg object that contains the same members 
 *            as this object. 
 **/ 
  public boolean
  equals(Object obj){

    assert(this.getClass().getDeclaredFields().length == 13)
      :this.getClass().getDeclaredFields().length ;

    boolean result = false;
    BodyMsg bi = null;
    
    if(obj instanceof BodyMsg){
      bi = (BodyMsg)obj;
      result = (this.obj.equals(bi.obj) &&
                this.timestamp.equals(bi.timestamp) &&
                //this.getRTAcceptStamp().equals(bi.getRTAcceptStamp()) &&
                //(this.isEmbargoed() == bi.isEmbargoed()) &&
                this.body.equals(bi.body) &&
                (this.priority == bi.priority)&&
                (this.requestId == bi.requestId)&&
                (this.isDemand == bi.isDemand));
    }
    return(result);

  }
  
 /** 
 *  Test custom serialization 
 **/ 
  private static void
  test14(){
    ObjInvalTarget hit1 = null;
    
    BodyMsg ii1 = null;
    BodyMsg ii2 = null;
    AcceptStamp as1 = null;
    
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    ObjectOutputStream oos = null;
    ObjectInputStream ois = null;

    // Test 1
    // Try a simple invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      byte[] data = new byte[2];
      data[0] = 0;
      data[1] = 1;

      ii1 = new BodyMsg(new ObjId("/b"),
                        0,
                        2,
                        new AcceptStamp(1, new NodeId(10)),
                        new ImmutableBytes(data),
                        true);
      ii1.setRequestId(99);
      System.out.println("test " + ii1);
      oos.writeObject(ii1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        ii2 = (BodyMsg)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      System.out.println("get " + ii2);
      assert(ii1 != ii2);
      assert(ii2.equals(ii1));
      assert(ii1.equals(ii2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }

    // Test 2
    // Try a bodymsg with default requestId
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      byte[] data = new byte[2];
      data[0] = 0;
      data[1] = 1;

      ii1 = new BodyMsg(new ObjId("/b"),
                        0,
                        2,
                        new AcceptStamp(1, new NodeId(10)),
                        new ImmutableBytes(data),
                        false);
      //ii1.setRequestId(99);
      System.out.println("test " + ii1);
      oos.writeObject(ii1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        ii2 = (BodyMsg)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      System.out.println("get " + ii2);
      assert(ii1 != ii2);
      assert(ii2.equals(ii1));
      assert(ii1.equals(ii2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    test14();
  }
}

//---------------------------------------------------------------------------
/* $Log: BodyMsg.java,v $
/* Revision 1.27  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.26  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.25  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.24  2006/09/12 14:57:59  zjiandan
/* Move object custom serialization into individual classes from TaggedOutputStream
/* and TaggedInputStream. The new serialization code consumes a little bit more
/* bytes than the original approach because of java's default object information
/* overhead (I have excluded the field name and type information).
/* The major changes are using java reflection API to serialize "private final"
/* fields (which are very common among our objects that need to custom serialization).
/* Also add sanity check for each class so that we will remember to update
/* the serialization methods when we change the class definitions.
/*
/* Revision 1.23  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.22  2005/10/16 04:20:11  zjiandan
/* add "boolean withBody" parameter for Checkpoint exchange.
/*
/* Revision 1.21  2005/03/09 21:29:35  nayate
/* Added priority code
/*
/* Revision 1.20  2005/03/09 00:40:57  nayate
/* Added a priority field to BodyMsg
/*
/* Revision 1.19  2005/03/09 00:15:36  nayate
/* Added a priority to BoundInvals
/*
/* Revision 1.18  2005/03/01 10:40:34  nayate
/* First successful compilation
/*
/* Revision 1.17  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.16  2004/10/22 20:46:54  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.15  2004/09/15 22:59:53  dahlin
/* Test 6 RandomAccessState works; TBD: delete
/*
/* Revision 1.14  2004/08/24 22:21:55  dahlin
/* RandomAccessState self test 3 succeeds
/*
/* Revision 1.13  2004/07/28 20:18:25  dahlin
/* encapsulated byte[] bodies in ImmutableBytes for safety
/*
/* Revision 1.12  2004/07/26 20:03:38  dahlin
/* Fixed typos from windows checkin so it will compile under Linux
/*
/* Revision 1.11  2004/07/26 19:41:29  dahlin
/* No substantive changes; Added notes to guide cleanup of code as per mike's notes 2004.7.20b.txt
/*
/* Revision 1.10  2004/05/25 10:35:50  arun
/* Added support for chaining, bound invals and unbinding.
/*
/* Revision 1.9  2004/05/23 10:32:25  arun
/* added support for startVV based update priority queue
/*
/* Revision 1.8  2004/05/22 10:10:12  arun
/* minor changes to notifyUpq logic. Correction of store behaviour while handling bound invals.
/*
/* Revision 1.7  2004/05/15 00:23:16  lgao
/* Add flag to distingurish demand fetch from prefetch data.
/*
/* Revision 1.6  2004/05/14 08:41:11  arun
/* basic versions
/*
/* Revision 1.5  2004/04/26 20:05:13  nayate
/* Added a new getLength() message.
/*
/* Revision 1.4  2004/04/16 18:53:22  nayate
/* Stub implementations that compile + minor fixes
/*
/* Revision 1.3  2004/04/16 18:02:59  zjiandan
/* Add Timestamp.java VVIterator.java VVIteratorToken.java
/* Modified BodyMsg BoundInval InvalListItem so that get InvalListItem compiled
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
