package code;

 /** 
 *  Implementation of a worker thread that handles receiving a stream 
 *  of update messages 
 **/ 
import java.net.Socket;
import java.io.IOException;
import java.io.ObjectInputStream;

import code.security.SangminConfig;

public class BodyRecvWorker extends Thread implements SocketMagicConstant
{

 /** 
 *  Data members 
 **/ 
  private TaggedInputStream s;
  private Core core;
  private Controller controller;
  private Socket underlyingSocket;

  private boolean dbg = false;
 /** 
 *  Constructor 
 **/ 
  public
  BodyRecvWorker(TaggedInputStream s,
                 Core core,
                 Controller controller,
                 Socket underlyingSocket){
    Env.performanceWarning("BodyRecvWorker TBD: add a 'negotiation' " +
                           "exchange between SubscribeBodyWorker and " +
                           "BodyRecvWorker to avoid resending data that " +
                           "the receiver already has");
    this.s = s;
    this.core = core;
    this.controller = controller;
    this.underlyingSocket = underlyingSocket;
  }


 /** 
 *  Main thread method 
 **/ 
  public void
  run(){
    NodeId senderNodeId = null;

    //
    // Read sender nodeId from stream.
    //
    try{
      Long magic = (Long)s.readTaggedObject();
      if((magic.longValue() != INVAL_SOCKET_MAGIC)
         &&(magic.longValue() != INVAL_SOCKET_RECONNECT_MAGIC)){
        Env.remoteAssert(false, "descriptive error msg");
      }
      if ( dbg){
	Env.dprintln(dbg, "BodyRecvWorker: Received magic number");
      }
      senderNodeId = (NodeId)(s.readTaggedObject());   
      if ( dbg ){
	Env.dprintln(dbg, "BodyRecvWorker: Received senderNodeId");
      }

    }catch(Exception e){
      // 
      // Attempt to connect failed and we don't even know
      // who it was that was trying. Nothing to do but
      // have this thread die. If this was important,
      // then a controller should timeout and retry. 
      //
      System.err.println(" ATTEMPT TO CONNECT FAILED------------");
      this.closeStreamAndSocket(null, 
                                "BodyRecvWorker cannot handshake " +
                                "with sender " + e);
      return; // Thread exit.
    }
    
    controller.informBodyStreamInitiated(senderNodeId);
    if (dbg ){
      Env.dprintln(dbg, "BodyRecvWorker:Initiated stream from "+ senderNodeId);
    }

    while(true){
      try{
	if( dbg ){
	  Env.dprintln(dbg, "BodyRecvWorker: waiting for message");
        }

        Object next = s.readTaggedObject();

	if( dbg ){
	  Env.dprintln(dbg,"BodyRecvWorker:  Received message " + next);
	}

        if(next instanceof StartBodyConnMsg) {
          //ignore
        }
        else if(next instanceof BodySSStartMsg){
          BodySSStartMsg ssStart = (BodySSStartMsg) next;
          controller.informSubscribeBodySucceeded(senderNodeId, 
                                                   ssStart.getSS());
        }
        else if(next instanceof BodySSEndMsg) {
          BodySSEndMsg ssEnd = (BodySSEndMsg) next;
          controller.informSubscribeBodyRemoved(senderNodeId,
                                                 ssEnd.getSS());
        }
        else if(next instanceof TerminateBodyConnMsg) {
          TerminateBodyConnMsg terminateConn = (TerminateBodyConnMsg) next;
          controller.informSubscribeBodyRemoved(senderNodeId,
                                                terminateConn.getSS());
          this.closeStreamAndSocket(senderNodeId, "BodyRecvWorker received " +
                                    "terminate msg.");
          return;
        }

        else if(next instanceof BodyMsg) {
          BodyMsg msg = (BodyMsg)next;
          long offset = 0;
          try{
            if(Env.TBD){
              Env.tbd("BodyRecvWorker ends up applying bodies in FIFO order."
                      + " The fix is: all BodyRecvWorker threads should"
                      + " insert the BodyMsg's they pull off the network into "
                      + " a *shared* set of priority queues, one per NodeId, "
                      + " this set of priority queues should be notified whenever "
                      + " an element of currentVV is advanced "
                      + " e.g., pending.notify(nodeId, localClock), " 
                      + " and a worker thread(s) should pull causally legal "
                      + " events out of these shared queues and call "
                      + " core.applyBody with them. May want multiple worker "
                      + " threads since DataStore:applyBody can block.");
            }
	    if( dbg ){
	      Env.dprintln(dbg, "BodyRecvWorker: receive BodyMsg: " + msg);
	    }
	    offset = core.applyBody(msg);
	    if( dbg ){
	      Env.dprintln(dbg, "BodyRecvWorker: applied BodyMsg: " + msg);
	    }

            /*
              Env.sprintln("zjd ---- " + Config.getMyNodeId() +
              " applied message to core: " +
              " from " + senderNodeId + 
              msg.getObjInvalTarget().getObjId() +
              " " + msg.getObjInvalTarget().getOffset() +
              " " + msg.getObjInvalTarget().getLength() + "at" +
              System.currentTimeMillis());
            */
          }catch(InterruptedException e){
            System.err.println("" + e);
            assert(false);
          }catch(InvalidBodyException e){
        	assert(SangminConfig.securityLevel >= SangminConfig.SIGNATURE);
        	Env.tbd("Currently security related controller callbacks are" +
        			"implemented in a ugly way. we should come up with" +
        			"something like SecureController interface");
        	((LocalController)controller).informReceivedBodyUnmatched();
          }
          if(offset == RandomAccessState.NO_NEW_VALID_BYTES){
            Env.performanceWarning("BodyRecvWorker received " +
                                   msg.getLength() + " bytes, but " +
                                   "they were not useful. " +
                                   "TBD: add a negotiation between " +
                                   "sender and receiver to avoid " +
                                   "sending redundant stuff.");
          }
          if(msg.isDemandReply()){

            controller.informReceiveDemandReply(senderNodeId, msg);
            if(offset == RandomAccessState.NO_NEW_VALID_BYTES){
              ObjInvalTarget oit = null;

              // Tell the controller that it tried to apply a demand
              // reply that was too old to be applicable
              oit = msg.getObjInvalTarget();
              controller.informDemandReadMiss(oit.getObjId(),
                                              oit.getOffset(),
                                              oit.getLength());
            }
          }else{
            controller.informReceivePushBody(msg);
          }
          /*
            System.err.println("zjd -- " + Config.getMyNodeId() +
            "Wrote body from " + senderNodeId + " : " +
            msg.getObjInvalTarget().getObjId() +
            " " + msg.getObjInvalTarget().getOffset() +
            " " + msg.getObjInvalTarget().getLength());
          */
          // Env.dprintln(dbg, "BodyRecvWorker: finished body handling msg");
        } // finished handling bodyMsg
	if( dbg ){
	  Env.dprintln(dbg, "BodyRecvWorker: finished handling msg");
	}
      }catch(IOException e){
        // Includes ClosedByInterruptException
        //on exception:  call closeStreamAndSocket
        //and return (thread exit)
        this.closeStreamAndSocket(senderNodeId, 
                                  "BodyRecvWorker interrupted while " +
                                  "receiving data " + e);
        return;
      }catch(ClassNotFoundException e){
        this.closeStreamAndSocket(senderNodeId, 
                                  "BodyRecvWorker interrupted while " +
                                  "receiving data " + e);
        return;
      }catch(ClassCastException f){
        this.closeStreamAndSocket(senderNodeId, 
                                  "BodyRecvWorker interrupted while " +
                                  "receiving data " + f);
      }catch(Exception g){
        this.closeStreamAndSocket(senderNodeId, 
                                  "BodyRecvWorker interrupted while " +
                                  "receiving data " + g);
      }
    } // END OF WHILE LOOP
    
  }

 /** 
 *  Cleanly close the stream and socket 
 **/ 
  private void
  closeStreamAndSocket(NodeId node, String msg){
    try{
      s.close();
      underlyingSocket.close();
    }catch(Exception e){
      // Ignore errors when trying to close
    }
    if(node != null){
      controller.informBodyStreamTerminated(node);
    }
    if( dbg ){
      Env.dprintln(dbg, "BodyRecvWorker closing stream: " + msg);
    }
  }
}

//---------------------------------------------------------------------------
/* $Log: BodyRecvWorker.java,v $
/* Revision 1.36  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.35  2007/03/14 19:55:18  dahlin
/* Compiler errors fixed
/*
/* Revision 1.34  2007/03/06 18:25:50  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.33  2006/11/14 18:22:59  nalini
/* added OutgoingBodyConnectionUnit
/*
/* Revision 1.32  2006/10/31 21:43:32  nalini
/* added control msgs to body streams
/*
/* Revision 1.31  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.30  2006/08/15 21:46:23  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.29  2006/06/14 22:50:09  nalini
/* Changed nice sockets to normal sockets for outgoing body connections
/*
/* Revision 1.28  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
/* Revision 1.27  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.26.2.1  2006/06/02 22:18:12  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
/*
/* Revision 1.26  2006/04/23 05:18:44  nalini
/* Fixed Bug for Body Subscription
/*
/* Revision 1.25  2006/04/20 03:52:52  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.24  2005/10/11 21:00:00  nayate
/* Fixed a minor bug
/*
/* Revision 1.23  2005/03/20 13:17:11  nayate
/* Added a call to controller.informDemandReadMiss() when an object is too old to be applied
/*
/* Revision 1.22  2005/02/28 20:25:58  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.21  2005/01/18 22:49:42  zjiandan
/* Rewrited Class Serialization for network deliveringto reduce the overall bandwidth.
/*
/* Revision 1.20  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.19  2004/10/22 20:46:54  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.18  2004/05/26 16:19:28  nayate
/* *** empty log message ***
/*
/* Revision 1.17  2004/05/26 09:45:55  arun
/* *** empty log message ***
/*
/* Revision 1.16  2004/05/26 04:42:22  arun
/* *** empty log message ***
/*
/* Revision 1.15  2004/05/25 12:59:37  lgao
/* Add chain mechanism to filter the duplicated update forwarding.
/*
/* Revision 1.14  2004/05/25 08:16:41  lgao
/* *** empty log message ***
/*
/* Revision 1.13  2004/05/23 15:16:53  nayate
/* Minor changes
/*
/* Revision 1.12  2004/05/23 00:25:22  arun
/* only fresh writes get inserted in UPQ. Some debugging code related to SDIMS
/*
/* Revision 1.11  2004/05/22 03:05:13  nayate
/* Fixed blocked read bug
/*
/* Revision 1.10  2004/05/22 02:42:37  nayate
/* Changes in blocking demand reads
/*
/* Revision 1.9  2004/05/21 13:37:20  nayate
/* Minor changes (removed some debugging information)
/*
/* Revision 1.8  2004/05/21 13:11:38  nayate
/* Fixed applyWrite bug
/*
/* Revision 1.7  2004/05/15 20:47:39  arun
/* minor changes to local store classes
/*
/* Revision 1.6  2004/05/14 20:49:00  dahlin
/* SDIMSController logic for Body connections (incoming)
/*
/* Revision 1.5  2004/05/13 01:14:01  nayate
/* Added some debugging code
/*
/* Revision 1.4  2004/04/30 19:16:55  nayate
/* Added code to catch a previously uncaught exception
/*
/* Revision 1.3  2004/04/26 20:04:24  nayate
/* Finished but didn't test classes that receive invals, updates, and
/* sync messages.
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
