package code;

 /** 
 *  Represents an invalidate that is "bound" to data; i.e., that has 
 *  to be transmitted with its associated object 
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.security.SangminConfig;

public class BoundInval extends PreciseInv implements Externalizable
{
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

  private final ImmutableBytes body;
  private final double priority;
  private final long maxBoundHops;
  
 /** 
 *  empty Constructor for serialization 
 **/ 
  public BoundInval(){
    super();
    body = null;
    priority = 0.0;
    maxBoundHops = Long.MAX_VALUE;
  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             long maxBoundHops){    
    super(new ObjInvalTarget(objId, offset, len), accept);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = Core.DEFAULT_PRIORITY;
    assert(this.body.getLength() == len);
    this.maxBoundHops = maxBoundHops;
  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             AcceptStamp rtAcceptStamp_,
             long maxBoundHops,
             boolean embargoed_){
    super(new ObjInvalTarget(objId, offset, len),
          accept,
          rtAcceptStamp_,
          embargoed_);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = Core.DEFAULT_PRIORITY;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == len);

  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             long maxBoundHops){
    super(new ObjInvalTarget(objId, offset, len), accept);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = newPriority;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == len);
  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             AcceptStamp rtAcceptStamp,
             long maxBoundHops,
             boolean embargoed){
    super(new ObjInvalTarget(objId, offset, len),
          accept,
          rtAcceptStamp,
          embargoed);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = newPriority;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == len);
  }


 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer,
             long maxBoundHops){
    super(obj_, accept);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = Core.DEFAULT_PRIORITY;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == this.getObjInvalTarget().getLength());
  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             long maxBoundHops){
    super(obj_, accept);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = newPriority;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == this.getObjInvalTarget().getLength());
  }

 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             AcceptStamp rtAcceptStamp,
             long maxBoundHops,
             boolean embargoed){
    super(obj_, accept, rtAcceptStamp, embargoed);

    assert(buffer instanceof Immutable);
    this.body = buffer;
    this.priority = newPriority;
    this.maxBoundHops = maxBoundHops;
    assert(this.body.getLength() == this.getObjInvalTarget().getLength());
  }
 /** 
 *  Constructor 
 **/ 
  public
  BoundInval(BodyMsg msg){
    //
    // Constructor of BoundInval from a BodyMsg.
    // Note: the two classes look pretty much identical.
    // Although we could probably get away with getting
    // rid of the separate BoundInval class and having
    // MsgBody implement GeneralInv, I think that
    // this separate class makes the code more
    // "self documenting" -- the core behaves differently
    // when it is given a BoundInval than when it
    // is given a regular (prefetched or demand fetched) 
    // MsgBody.

    super(msg.getObjInvalTarget(), msg.getAcceptStamp());

    ImmutableBytes b = msg.getBody();
    assert(b instanceof Immutable);
    this.body = b;
    this.priority = msg.getPriority();
    this.maxBoundHops = 0;  // set the maxBoundHops to 0
    assert(this.body.getLength() == this.getObjInvalTarget().getLength());
  }


 /** 
 *  old interfaces -- to ensure backward compatibility with tests 
 **/ 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer) {    
    this(objId, offset, len, accept, buffer, Long.MAX_VALUE);
  }
 
  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority){
    this(objId, offset, len, accept, buffer, newPriority, Long.MAX_VALUE);
  }

  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             AcceptStamp rtAcceptStamp,
             boolean embargoed){
    this(obj_, accept, buffer, newPriority, rtAcceptStamp, Long.MAX_VALUE, embargoed);
  }

  public
  BoundInval(ObjId objId,
             long offset,
             long len,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority,
             AcceptStamp rtAcceptStamp,
             boolean embargoed) {
   this(objId, offset, len, accept, buffer, newPriority, rtAcceptStamp,
        Long.MAX_VALUE, embargoed);
  }

  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer) {
    this(obj_, accept, buffer, Long.MAX_VALUE);
  }

  public
  BoundInval(ObjInvalTarget obj_,
             AcceptStamp accept,
             ImmutableBytes buffer,
             double newPriority) {
    this(obj_, accept, buffer, newPriority, Long.MAX_VALUE);
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("PreciseInv -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    super.writeExternal(out);
    
    out.writeInt(this.body.getLength());
    out.write(this.body.dangerousGetReferenceToInternalByteArray());
    out.writeDouble(this.priority);
    out.writeLong(this.maxBoundHops);
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    super.readExternal(in);
    
    int strLength = in.readInt();
    byte[] b = new byte[strLength];
    in.readFully(b);
    
    Field[] f = new Field[3];
      
      try{

        f[0] = this.getClass().getDeclaredField("body");
        f[1] = this.getClass().getDeclaredField("priority");
        f[2] = this.getClass().getDeclaredField("maxBoundHops");
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        f[0].set(this, new ImmutableBytes(b));
        f[1].setDouble(this, in.readDouble());
        f[2].setLong(this, in.readLong());
        
      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }
  
 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    assert(this.getClass().getDeclaredFields().length == 9);
     
    //the fields we custom serialize remain the same
    Field[] f = new Field[3];  
      try{
        f[0] = this.getClass().getDeclaredField("body");
        f[1] = this.getClass().getDeclaredField("priority");
        f[2] = this.getClass().getDeclaredField("maxBoundHops");
                
        assert (f[0].getType().getName().equals("ImmutableBytes"));
        //System.out.println(f[1].getType().getName());
        assert (f[1].getType().getName().equals("double"));
        assert (f[2].getType().getName().equals("long"));
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Return a hash code for this object 
 *  NOTE: Do nothing for now 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Return true if "this" equals "obj" 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    BoundInval bi = null;
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }
    // 2 Java-provided fields + 2 declared fields
    // System.out.println(this.getClass().getDeclaredFields().length );
    
    if(obj instanceof BoundInval){
      bi = (BoundInval)obj;
      result = (this.getObjInvalTarget().equals(bi.getObjInvalTarget()) &&
                this.getAcceptStamp().equals(bi.getAcceptStamp()) &&
                this.getRTAcceptStamp().equals(bi.getRTAcceptStamp()) &&
                (this.isEmbargoed() == bi.isEmbargoed()) &&
                this.body.equals(bi.body) &&
                (this.priority == bi.priority) &&
                (this.maxBoundHops == bi.maxBoundHops));
    }
    return(result);
  }

 /** 
 *  Return the object body 
 **/ 
  public final ImmutableBytes
  getBody(){
    return(this.body);
  }

 /** 
 *  Return the object body 
 **/ 
  public final double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return the object body 
 **/ 
  public final long
  getMaxBoundHops(){
    return(this.maxBoundHops);
  }
 /** 
 *  Clone this object 
 *  
 *  If we *know* that this object and all objects we reference 
 *  are immutable, then just return this; otherwise perform 
 *      newOne = super.clone();  
 *      now replace any fields in newOne that point to mutable  
 *          objects with a clone of that field 
 *      return newOne 
 *  
 **/ 
  public Object
  clone(){
    assert(this instanceof Immutable);
    return this;
  }

 /** 
 *  Make a precise clone of this object 
 **/ 
  public PreciseInv
  clonePreciseInv(){
    PreciseInv pi = null;

    pi = new PreciseInv(this.getObjInvalTarget(),
                        this.getAcceptStamp(),
                        this.getRTAcceptStamp(),
                        this.isEmbargoed());
    return(pi);
  }

 /** 
 *  Make a BodyMsg clone of this object 
 **/ 
  public BodyMsg
  cloneBodyMsg(){
    BodyMsg bodyMsg = null;
    ObjInvalTarget oit = null;

    oit = this.getObjInvalTarget();
    bodyMsg = new BodyMsg(oit.getObjId(),
                          oit.getOffset(),
                          oit.getLength(),
                          this.getAcceptStamp(),
                          this.body,
                          false,
                          this.priority);
    return(bodyMsg);
  }

 /** 
 *  METHODS INHERITED FROM GENERALINV: -- Overrides PreciseInval 
 **/ 

  private static boolean warned2 = false;
 /** 
 *  Return a new invalidate with mutiple operations performed 
 **/ 
  public GeneralInv
  cloneIntersectInvaltargetChopStartEnd(GeneralInv inv){
    InvalTarget invTarget = null;
    GeneralInv intersectInv = null;

    if(!warned2){
      Env.tbd("Check code for cloneIntersectInvaltargetChopStartEndCSN()"
              + " for all subclasses of GeneralInv."
              + " I don't trust it (see TODO.txt)");
      warned2 = true;
    }
    // This call should be made for uncommitted invalidates only

    invTarget = this.getObjInvalTarget().getIntersection(inv.getInvalTarget());
    assert(invTarget != null);
    if(invTarget instanceof ObjInvalTarget){
      if(inv.isBound()){
        //this.embargoed = ((this.embargoed) &&(inv.isEmbargoed())); 
        intersectInv = this;
      }else {
	/* commented by zjd 2007 Feb
	   for TierStore: new semantic of BoundInval:
	   BoundInval will not be implicitly unbound because
	   of receiving other overlapping invalidates
           it will be unbound only when an explicit unbind msg is issued
           
        // NOTE: If we receive a message that overlaps with
        // this one, this invalidate MUST be unbound. We.
        assert(this.getAcceptStamp() instanceof Immutable);
        intersectInv = new PreciseInv((ObjInvalTarget)invTarget,
                                      this.getAcceptStamp(),
                                      this.getRTAcceptStamp(),
                                      (this.isEmbargoed()) &&
                                      (inv.isEmbargoed()));
       */
       intersectInv = this;
      }
    }else if(invTarget instanceof HierInvalTarget){
      //should never reached here. It does not make sense to have a BoundInval occupy a timestamp where 
      //there's another overlapping invalidate that covers the same timestamp but with different objInval.
      assert false: "this=" + this.toString() + " inv=" + inv.toString() + " intersection=" +invTarget.toString();//added by zjd
      assert(this.getAcceptStamp() instanceof Immutable);
      intersectInv = new SingleWriterImpreciseInv((HierInvalTarget)invTarget,
                                                  this.getAcceptStamp(),
                                                  this.getAcceptStamp(),
                                                  this.getRTAcceptStamp());
    }else if(invTarget instanceof MultiObjInvalTarget){
      System.err.println("Should not have received a MultiObjInvalTarget " +
                         "as the result of an intersection");
      assert(false);
    }else{
      System.err.println("Invalid inval target type");
      assert(false);
    }
    return(intersectInv);
  }

  private static boolean warned3 = false;
 /** 
 *  Return the portion of an invalidate for one writer whose 
 *  starting stamp is potentially chopped. 
 **/ 
  public GeneralInv
  getOneWriterSubsetFrom(NodeId node, long startStampInclusive){
    /*
      return an invalidation (possibly this) with 
      interestSet = this.interestSet
      nodeId = node // assert node was in this.VV
      startVV{node} == startStampInclusive
        // assert this.startStamp{node} >= startSTampInclusive
      endVV{node} == this.endVV{node}
      (Note: OK to just return this if we meet all of the other criteria)
    */
    GeneralInv subsetInv = null;

    if(!warned3){
      Env.tbd("Check code for getOneWriterSubsetFrom()"
              + "==  for startTime seems wrong!");
      warned3 = true;
    }

    assert(node.equals(this.getAcceptStamp().getNodeId()));
    if(this.getAcceptStamp().getLocalClock() == startStampInclusive){
      subsetInv = this;
    }
    return(subsetInv);
  }

 /** 
 *  Return true if this invalidate is bound (which it is!) 
 **/ 
  public boolean
  isBound(){
    return(true);
  }

 /** 
 *  METHODS INHERITED FROM SINGLEWRITERINVAL 
 **/ 

 /** 
 *  Convert to a string 
 **/ 
  public final String
  toString(){
    String str = null;

    str = "BoundInv: <" + super.toString() 
      + this.getPriority() + ", " + this.getMaxBoundHops() + ", *msg body*>";
    return(str);
  }

 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way +/- 1~7 bytes.  
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  whenever delete or add new members, recalculate the number of bytes 
 *  refering to universalReplication/designDoc/FootPrintSizeOfStandardObject.txt 
 **/ 
  public long memSize(){
    return 200 + body.getLength();
  }

  
  
  
}

//---------------------------------------------------------------------------
/* $Log: BoundInval.java,v $
/* Revision 1.32  2007/03/16 07:10:18  zjiandan
/* fixed OutgoingConnection unittest.
/*
/* Revision 1.31  2007/03/16 04:01:46  zjiandan
/* Fixed unit test.
/*
/* Revision 1.30  2007/03/02 09:43:46  zjiandan
/* Modified in-mem zip operation for BoundInval to adapt to the new semantics
/* of BoundInval.
/*
/* Revision 1.29  2007/03/01 06:39:43  nalini
/* added support for maxBoundHop at Local Interface
/*
/* Revision 1.28  2007/02/23 20:54:28  zjiandan
/* Fixed mem leak problems in NFS2Interface and some other bugs.
/* Andrew benchmark passed, performance still needs to tune.
/*
/* Revision 1.27  2006/09/12 14:57:59  zjiandan
/* Move object custom serialization into individual classes from TaggedOutputStream
/* and TaggedInputStream. The new serialization code consumes a little bit more
/* bytes than the original approach because of java's default object information
/* overhead (I have excluded the field name and type information).
/* The major changes are using java reflection API to serialize "private final"
/* fields (which are very common among our objects that need to custom serialization).
/* Also add sanity check for each class so that we will remember to update
/* the serialization methods when we change the class definitions.
/*
/* */
//---------------------------------------------------------------------------
