package code;

 /** 
 *  CPStartMsg -- 
 *    Msg marks the start of a Checkpoint stream. 
 **/ 
import java.io.Externalizable;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.IOException;
import java.util.Vector;
import java.util.Enumeration;


public class CPStartMsg implements Externalizable{

  private SubscriptionSet ss;
  private AcceptVV startVV;
  private AcceptVV senderCVV;
  private Vector lpvvRecs;

  private final boolean sanityCheck = true;
  private boolean warnPrinted = false;

  

 /** 
 *  Constructor 
 **/ 
  public CPStartMsg(SubscriptionSet ss, 
		    AcceptVV startVV,
		    AcceptVV senderCVV,
		    Vector lpvvRecs){
    this.ss = ss;
    this.startVV = startVV.dropNegatives();
    this.senderCVV = senderCVV.dropNegatives();
    this.lpvvRecs = lpvvRecs;
    if(sanityCheck){
      verifyLpvvRecs();
    }
  }

 /** 
 *  Constructor for externalizable 
 **/ 
  public CPStartMsg(){
    this.ss = null;
    this.startVV = null;
    this.senderCVV = null;
    this.lpvvRecs = null;
  }

  public final void verifyLpvvRecs(){
    for( Enumeration e = lpvvRecs.elements(); e.hasMoreElements();){
      Object next = e.nextElement();
      assert next instanceof LPVVRecord;
    } 
    if(!warnPrinted){
      Env.performanceWarning("CPStartMsg sanityCheck on.");
      Env.performanceWarning(" CPStartMsg tbd: get rid of senderCVV which is"
			     + " always the same as connection.prevVV"); 
      warnPrinted = true;
    }
  }

  public SubscriptionSet getSubscriptionSet(){
    return this.ss;
  }

  public AcceptVV getStartVV(){
    return this.startVV;
  }
  
  public AcceptVV getSenderCVV(){
    return this.senderCVV;
  }

  public Vector getLpvvRecs(){
    return this.lpvvRecs;
  }

 /** 
 *  equals()-- 
 *            compares this object to the specified object. The result 
 *            is true if and only if the argument is not null and is  
 *            a CPStartMsg object that contains the same members 
 *            as this object. 
 **/ 
  public boolean equals(Object obj){
      
      if(! (obj instanceof CPStartMsg)){
        return false;  
      }else{
        CPStartMsg pos = (CPStartMsg)obj;
        if((! this.ss.equals(pos.getSubscriptionSet()))
           ||(! this.startVV.equalsIgnoreNegatives(pos.getStartVV()))
	   ||(! this.senderCVV.equalsIgnoreNegatives(pos.getSenderCVV()))
	   ||(! this.lpvvRecs.equals(pos.getLpvvRecs()))){
          return false;
        }
        return true;
      }
    }


 /** 
 *  writeExternal -- externalization 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{
    long start = 0;
    
    //
    // note: to use the SubscriptionSet.write/readExternal directly
    //       is a little bit more expensive than
    //       copy the SubscriptionSet.write/readExternal here.
    //       which elliminate the SubscriptionSet class info
    //       and save 40 bytes per msg.
    //
    String ssStr = "";
    if(!ss.isEmpty()){
      ssStr = ss.toString();
    }
    out.writeUTF(ssStr);
    
    //send startVV
    /*
    out.writeInt(this.startVV.getSize());

    Object token = null;
    for(VVIterator vvIter = this.startVV.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();
      out.writeLong(this.startVV.getServerByIteratorToken(token).getIDint());
      out.writeLong(this.startVV.getStampByIteratorToken(token));
    }
    */
    this.startVV.writeExternal(out);

    //send senderCVV
    /*
    out.writeInt(this.senderCVV.getSize());

    Object token = null;
    for(VVIterator vvIter = this.senderCVV.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();
      out.writeLong(this.senderCVV.getServerByIteratorToken(token).getIDint());
      out.writeLong(this.senderCVV.getStampByIteratorToken(token));
    }
    */
    this.senderCVV.writeExternal(out);

    //send lpvvRecs
    out.writeInt(lpvvRecs.size());

    for( Enumeration e = lpvvRecs.elements(); e.hasMoreElements();){
      Object next = e.nextElement();
      assert next instanceof LPVVRecord;
      LPVVRecord lr = (LPVVRecord)next;
      String pname = lr.getPathName();
      AcceptVV lpvv = lr.getLpVV();
      AcceptVV clpvv = lr.getChildrenLpVV();
      
      out.writeUTF(pname);

      if(lpvv != null){
	lpvv = lpvv.getDiff(senderCVV);
      }
      lpvv.writeExternal(out);

      if(clpvv != null){
	clpvv = clpvv.getDiff(senderCVV);
      }
      out.writeObject(clpvv);
      
    } 
    
  }
 /** 
 *  readExternal  
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException, ClassNotFoundException{
    long start = 0;
    long middle = 0;
    String ssStr = (String)(in.readUTF());
    if(ssStr.equals("")){
      ss = SubscriptionSet.makeEmptySet();
    }else{
      ss = SubscriptionSet.makeSubscriptionSet(ssStr);
    }
    startVV = new AcceptVV();
    startVV.readExternal(in);
    senderCVV = new AcceptVV();
    senderCVV.readExternal(in);
    
    int vecSize = in.readInt();
    this.lpvvRecs = new Vector();
    for(int i = 0; i < vecSize; i++){
      String pname = (String)(in.readUTF());
      AcceptVV lpvv = new AcceptVV();
      lpvv.readExternal(in);
      if(lpvv != null){
	lpvv = senderCVV.cloneMinVV(lpvv);
      }

      AcceptVV clpvv = new AcceptVV();
      clpvv = (AcceptVV)in.readObject();
      if(clpvv != null){
	clpvv = senderCVV.cloneMinVV(clpvv);
      }
      LPVVRecord lr = new LPVVRecord(pname, 
				     lpvv,
				     clpvv);
      this.lpvvRecs.add(lr);
    }
    
    
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String toString(){
	return("CPStartMsg: (" + ss + ", " + startVV + ", " + senderCVV 
	       + ", " + lpvvRecs + " )" );
  }
}

//---------------------------------------------------------------------------
/* $Log: CPStartMsg.java,v $
/* Revision 1.4  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.3  2007/07/15 06:21:29  zjiandan
/* optimize bw for cp exchange
/*
/* Revision 1.2  2007/07/12 17:02:32  zjiandan
/* *** empty log message ***
/*
/* Revision 1.1  2006/04/20 03:52:52  zjiandan
/* Callbacks merged with runTime.
/* */
//---------------------------------------------------------------------------
