package code;

/* Core.java 
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */

import java.util.*;
import java.io.*;

import code.security.*;


public class Core
{

  public static final boolean useNewIterator = true; // use the enhanced iterator

  //
  // Locking and recovery:
  // We structure things carefully to not require
  // locking at this level. Recovery is done
  // by replaying the log into the DataStore on
  // reboot.
  //
  // The Log contains a redo log of all state in the 
  // DataStore. So, updates must go to the log before
  // the DataStore.
  //
  // Reads come from the DataStore.
  //
  // Writes to the log are normally asynchronous. 
  // We allow local DataStore reads of data that have not been 
  // sync'd to disk; but we don't send information
  // to other nodes until it is safely on disk.
  //

  //
  //replaced by exposing CP|LOG option to each subscribeInval request
  //    instead of set by Core for all.
  //
  // option for Outgoing connection
  //protected boolean setCP = false; 

  // settings for inval iterator

  protected static final boolean measureTime = false;

  protected long maxAccumulateTime = 1000;
  protected long maxSendDelay = 0;

  protected static final boolean PRINT_CP_INFO = false;
  final static long gcCycle = 1000;
  final static long gcThreshold = 10;
  public static final double DEFAULT_PRIORITY = 1.0;
  public static final long MAX_INVAL_DELAY_MS = 6000; // 60 seconds

  protected UpdateLog log;
  protected DataStore store;
  protected RMIClient rmiClient;
  protected String logPath;
  protected String storePath;
  //  protected String ufsDir = "/tmp/ufs/";
  protected NodeId myNodeId;
  protected InvalidateBuffer invalBuffer;
  protected StreamId localWriteStreamId;
  public InvalIteratorFilter iiFilter;

  protected CounterVV recoverStartVV = null;
  protected CounterVV recoverEndVV = null;
  protected boolean recovering;
  protected RecoveryStatus recoveryComplete;
  protected boolean closed = false;
  protected static final boolean verbose = false;
  public static final boolean dbgLog = false;
  protected static final boolean dbgMaxAcc = false;

  protected static final boolean dbgPerformance = false;

  protected Core(){
    recoveryComplete = new RecoveryStatus();

  }
  
  public
  Core(RMIClient rmiClient_, 
      boolean filterOn, 
      boolean cleanDb, 
      NodeId myId){
    this(rmiClient_, filterOn, cleanDb, myId, false);
  }

  public
  Core(RMIClient rmiClient_, 
      boolean filterOn, 
      boolean cleanDb, 
      NodeId myId,
      boolean noSyncLog){
    try{
      UpdateBuffer updateBuffer = null;
      if(verbose){
        Env.dprintln(verbose, "Core constructor cleanDb=" + cleanDb);
      }
      recoveryComplete = new RecoveryStatus();

      this.myNodeId = myId;
      logPath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".log";
      storePath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".store";
      // logPath = ufsDir +Config.getConfigFile() + this.myNodeId + ".log";
      // storePath = ufsDir +Config.getConfigFile() + this.myNodeId + ".store";
      if(cleanDb) cleanEnv();
      this.rmiClient = rmiClient_;
      String name = Config.getConfigFile();
      if(verbose){
        Env.dprintln(verbose, "Core constructor logPath=" + logPath);
      }
      log = new UpdateLog(logPath, myId, noSyncLog);
      recovering = false;
      recoverEndVV = new CounterVV(log.getCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverEndVV=" + recoverEndVV.toString());
      }
      this.store = new DataStore(storePath, myId);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
          UpdateBuffer.FAIL_ON_FULL);
      this.invalBuffer = new InvalidateBuffer(this.store, updateBuffer);
      /*
      new InvalidateBufferWorkerThread(this.invalBuffer,
                                       this.store,
                                       updateBuffer);
       */
      this.store.setInvalBuffer(this.invalBuffer);
      this.localWriteStreamId = StreamId.makeNewStreamId();
      /*
        recoverStartVV = 
        new CounterVV(store.getCurrentVV(new DirectoryInterestSet("/*")));

        this doesn't work, because the interestset doesn't exist
       */
      recoverStartVV = new CounterVV(this.store.getMaxCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverStartVV=" + recoverStartVV.toString());
      }
      Env.tbd("recoverStartVV should be initialized as store.getMinLPVV()"+
      "when the new isstatus is placed in");
      assert(recoverEndVV.includes(recoverStartVV));
      assert(recoverStartVV.includes(log.getDiskOmitVV()));
      if(!recoverStartVV.includes(recoverEndVV)){
        recovering = true;
      }
      //System.err.println("recoverStartVV: " + recoverStartVV);

      iiFilter = new InvalIteratorFilter(filterOn);
      log.startGCWorker(store);

    } catch (Exception e){
      e.printStackTrace();
      assert false;

      Env.printDebug("Failed to initiate the core.." + e);
      System.exit(-1);
    }
  }

 /** 
 *  Core construction with option to choose null RAS - for testing only 
 **/ 


  public
  Core(RMIClient rmiClient_, 
      boolean filterOn, 
      boolean cleanDb, 
      NodeId myId,
      boolean noSyncLog, 
      boolean nullRAS){
    try{
      UpdateBuffer updateBuffer = null;
      if(verbose){
        Env.dprintln(verbose, "Core constructor cleanDb=" + cleanDb);
      }
      recoveryComplete = new RecoveryStatus();

      this.myNodeId = myId;
      logPath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".log";
      storePath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".store";
      // logPath = ufsDir +Config.getConfigFile() + this.myNodeId + ".log";
      // storePath = ufsDir +Config.getConfigFile() + this.myNodeId + ".store";
      if(cleanDb) cleanEnv();
      this.rmiClient = rmiClient_;
      String name = Config.getConfigFile();
      if(verbose){
        Env.dprintln(verbose, "Core constructor logPath=" + logPath);
      }
      log = new UpdateLog(logPath, myId, noSyncLog);
      recovering = false;
      recoverEndVV = new CounterVV(log.getCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverEndVV=" + recoverEndVV.toString());
      }
      this.store = new DataStore(storePath, myId, nullRAS);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
          UpdateBuffer.FAIL_ON_FULL);
      this.invalBuffer = new InvalidateBuffer(this.store, updateBuffer);
      /*
      new InvalidateBufferWorkerThread(this.invalBuffer,
                                       this.store,
                                       updateBuffer);
       */
      this.store.setInvalBuffer(this.invalBuffer);
      this.localWriteStreamId = StreamId.makeNewStreamId();
      /*
        recoverStartVV = 
        new CounterVV(store.getCurrentVV(new DirectoryInterestSet("/*")));

        this doesn't work, because the interestset doesn't exist
       */
      recoverStartVV = new CounterVV(this.store.getMaxCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverStartVV=" + recoverStartVV.toString());
      }
      Env.tbd("recoverStartVV should be initialized as store.getMinLPVV()"+
      "when the new isstatus is placed in");
      assert(recoverEndVV.includes(recoverStartVV));
      assert(recoverStartVV.includes(log.getDiskOmitVV()));
      if(!recoverStartVV.includes(recoverEndVV)){
        recovering = true;
      }
      //System.err.println("recoverStartVV: " + recoverStartVV);

      iiFilter = new InvalIteratorFilter(filterOn);
      log.startGCWorker(store);



    } catch (Exception e){
      e.printStackTrace();
      assert false;

      Env.printDebug("Failed to initiate the core.." + e);
      System.exit(-1);
    }
  }


 /** 
 *  Core construction with given ISStatus (for testing only) 
 **/ 
  public
  Core(RMIClient rmiClient_, 
      boolean filterOn, 
      boolean cleanDb, 
      NodeId myId, 
      ISStatus isStatus,
      boolean noSyncLog){

    try{
      UpdateBuffer updateBuffer = null;
      if(verbose){
        Env.dprintln(verbose, "Core constructor cleanDb=" + cleanDb);
      }
      recoveryComplete = new RecoveryStatus();

      this.myNodeId = myId;
      logPath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".log";
      storePath = Config.getLocalStore(this.myNodeId) + this.myNodeId + ".store";
      // logPath = ufsDir +Config.getConfigFile() + this.myNodeId + ".log";
      // storePath = ufsDir +Config.getConfigFile() + this.myNodeId + ".store";
      if(cleanDb) cleanEnv();
      this.rmiClient = rmiClient_;
      String name = Config.getConfigFile();
      int index = name.indexOf('.');
      String prefix = name.substring(0,index);
      if(verbose){
        Env.dprintln(verbose, "Core constructor logPath=" + logPath);
      }
      
      log = new UpdateLog(logPath, myId, noSyncLog);
      recovering = false;
      recoverEndVV = new CounterVV(log.getCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverEndVV=" + recoverEndVV.toString());
      }
      this.store = new DataStore(storePath, isStatus, myId);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
          UpdateBuffer.FAIL_ON_FULL);
      this.invalBuffer = new InvalidateBuffer(this.store, updateBuffer);
      new InvalidateBufferWorkerThread(this.invalBuffer,
          this.store,
          updateBuffer);
      this.store.setInvalBuffer(this.invalBuffer);
      this.localWriteStreamId = StreamId.makeNewStreamId();
      /*
        recoverStartVV = 
        new CounterVV(store.getCurrentVV(new DirectoryInterestSet("/*")));

        this doesn't work, because the interestset doesn't exist
       */
      recoverStartVV = new CounterVV(this.store.getMaxCurrentVV());
      if(verbose){
        Env.dprintln(verbose, "Core constructor recoverStartVV=" + recoverStartVV.toString());
      }
      Env.tbd("recoverStartVV should be initialized as store.getMinLPVV()"+
      "when the new isstatus is placed in");
      assert(recoverEndVV.includes(recoverStartVV));
      assert(recoverStartVV.includes(log.getDiskOmitVV()));
      if(!recoverStartVV.includes(recoverEndVV)){
        recovering = true;
      }
      //System.err.println("recoverStartVV: " + recoverStartVV);

      iiFilter = new InvalIteratorFilter(filterOn);
      log.startGCWorker(store);


    } catch (Exception e){
      e.printStackTrace();
      assert false;

      Env.printDebug("Failed to initiate the core.." + e);
      System.exit(-1);
    }
  }

  /*
    replaced by exposing CP|LOG option to each subscribeInval request
    instead of set by Core for all.


  //---------------------------------------------------------------------------
  //for test of callbacks,  set CP for outgoingConnection's policy for catchup 
  //---------------------------------------------------------------------------
  public void setCatchupTypeToCP(boolean cp){
    setCP = cp;
  }   

  //---------------------------------------------------------------------------
  // get CP 
  //---------------------------------------------------------------------------
  public boolean getCP(){
    return setCP;
  }   

   */

 /** 
 *  set maxAccumulateTime -- sets the time in the outgoing connection to  
 *                           stop accumulating ImpInvals and send it out 
 **/ 
  public void setMaxAccumulateTime(long maxTime){
    maxAccumulateTime = maxTime;
  }   

 /** 
 *  get maxAccumulateTime 
 **/ 
  public long getMaxAccumulateTime(){
    return maxAccumulateTime;
  }   

 /** 
 *  get this node Id 
 **/ 
  public NodeId getMyNodeId(){
    return this.myNodeId;
  }

 /** 
 *  get UpdateLog,  
 **/ 
  public UpdateLog getLog(){
    return this.log;
  }

 /** 
 *  close() -- release all resources 
 **/ 
  public void close(){
    if(!closed){

      log.close();
      this.store.close();
      closed = true;
    }
  }

 /** 
 * Set recovering to true and call recoverLocalState 
 **/ 
  public synchronized void 
  forceRecoverLocalState(RMIServerImpl rmiServer){
    recovering = true;
    recoverLocalState(rmiServer);
  }

 /** 
 * Synchronize localstore with log(i.e., apply the inval after 
 * checkpoint. 
 * Note: at this time the rmiserver shouldn't accept any client 
 * request except the recovery subscribes. 
 * TBD: add new subscribe method with endVV in SubscribeInvalWorker.java 
 **/ 
  public synchronized void 
  recoverLocalState(RMIServerImpl rmiServer){
    if(recovering){
      long subscriptionDurationMS = 1000000;
      long maxDelayMS = 10;

      try{
        rmiServer.subscribeInval(this.myNodeId, 
            Config.getDNS(myNodeId), 
            Config.getPortInval(myNodeId), 
            SubscriptionSet.makeSubscriptionSet(":/*"),
            recoverStartVV.cloneAcceptVV(),//the startVV
            false, //catchupWithCP = false
            false/*, //cpWithBody = false
            false*/);

      } catch(Exception e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
        System.exit(-1);
      }
      //
      // how to know that the recovery stream is over?
      // right now just assume that the subscriptionDurationMS is
      // long enough so that all the invalidate is applied to local store
      //
      //rmiServer.requestSync(myNodeId, log.getCurrentVV());
      try{
        while (!recoverStartVV.includes(recoverEndVV)){
          wait();
        }
      }catch (InterruptedException ie){
        ie.printStackTrace();
        System.err.println("" + ie);
        assert(false);
        System.exit(-1);
      }
      try{
        rmiServer.removeConnection(this.myNodeId, 
            Config.getDNS(myNodeId), 
            Config.getPortInval(myNodeId)); 
      } catch (Exception re){

        //Env.inform("!!!!!!recover stream close fail: " + re.printStackTrace());
        //assert false;
      }

      recovering = false;
    }
    recoveryComplete.recoveryDone();
  }

  //-----------------------------------------------------------
  // Skip recovery from log. Used only for testing when you
  // don't want to create a RMIServer and you know you
  // don't need to recover anything.
  //-----------------------------------------------------------
  public synchronized void testSkipRecoverLocalState(){
    recoveryComplete.recoveryDone();
    recovering = false;
  }

  //-----------------------------------------------------------------------
  // Return the LpVV for this SubscriptionSet
  // We define the lpVV of a SubscriptionSet to be the component-wise
  // minimum of the lpVVs of the various PreciseSets that overlap "ss".
  //-----------------------------------------------------------------------
  public AcceptVV
  getLpVV(SubscriptionSet ss){
    return(this.store.getLpVV(ss));
  }

  ////////////////////////////////////////////////////
  // Local interface
  ////////////////////////////////////////////////////

  //----------------------------------------------------------------
  // checks if the object is valid
  //----------------------------------------------------------------  
  public boolean
  isValid(ObjId objId, long offset, long length)
  throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException {
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() 
    //after constructor
    return this.store.isValid(objId, offset, length, true);
  }

  //----------------------------------------------------------------
  // Checks if the object is precise
  //----------------------------------------------------------------  
  public boolean
  isPrecise(ObjId objId){
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() 
    //after constructor
    return this.store.isPrecise(objId);
  }


  //------------------------------------------------------------------------
  // read() -- read up to <length> bytes starting from <objId, offset> if
  // "exactOffset" is true, otherwise starting from <objId, invaloffset> where
  // "invaloffset'" is the offset of the invalidate which covers the read 
  // "offset".
  // 
  // Returns
  //   If the object exists and there is at least one valid byte at <objId, exactOffset:offset?invalOffset>
  //   we return between 1 and <length> bytes. Notice that we may return fewer
  //   bytes than requested.
  //
  //   If the object does not exist, throw ObjNotFoundException
  //
  //   If there is a low-level IO error or database error, throw IOException
  //
  //   Otherwise
  //
  //   If <blockInvalid> is true, then do not return until the specified range
  //   is valid. (If <blockInvalid> is false and the byte at <objId, offse>
  //   is invalid, throw ReadOfInvalidRangeException.)
  //
  //   If <blockImprecise> is true, then do not return until the specified
  //   object "is precise."
  //
  //   If offset is past the last byte of the file, throw EOFException
  //
  //   Notice that we never return null. 
  //
  //   Note that a read of a "hole" returns a 1-byte result with value 0.
  //   (A hole is created when an offset x is not written but some offset
  //   y>x is written.)
  //
  //------------------------------------------------------------------------

  //new interface provided by nalini for runtime -- added controller
  public BodyMsg
  read(ObjId objId,
      long offset,
      long length, 
      boolean blockInvalid,
      boolean blockImprecise,
      boolean exactOffset,
      Controller controller,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, 
  ReadOfInvalidRangeException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    return this.store.read(objId,
        offset,
        length,
        blockInvalid,
        blockImprecise, 
        exactOffset,
        controller,
        timeout);
  }


  public BodyMsg
  read(ObjId objId,
      long offset,
      long length, 
      boolean blockInvalid,
      boolean blockImprecise,
      boolean exactOffset,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, 
  ReadOfInvalidRangeException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    return this.store.read(objId,
        offset,
        length,
        blockInvalid,
        blockImprecise, 
        exactOffset,
        timeout);
  }

  //------------------------------------------------------------------------
  // same as above except that this one supports temporal error constraints
  //------------------------------------------------------------------------
  public BodyMsg
  read(ObjId objId,
      long offset,
      long length, 
      boolean blockInvalid,
      boolean blockImprecise,
      boolean exactOffset,
      long maxTE,
      Controller controller,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, 
  ReadOfInvalidRangeException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    return this.store.read(objId,
        offset,
        length,
        blockInvalid,
        blockImprecise,
        exactOffset,
        maxTE,
        controller,
        timeout);
  }

//the same old interface without controller
  public BodyMsg
  read(ObjId objId,
      long offset,
      long length, 
      boolean blockInvalid,
      boolean blockImprecise,
      boolean exactOffset,
      long maxTE,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, 
  ReadOfInvalidRangeException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    return this.store.read(objId,
        offset,
        length,
        blockInvalid,
        blockImprecise,
        exactOffset,
        maxTE,
        timeout);
  }

  //------------------------------------------------------------------------
  // read() -- read up to <length> bytes starting from <objId, offset>
  //
  // This version of read() always blocks if the range is invalid
  // so it never throws ReadOfInvalidRangeException.
  //------------------------------------------------------------------------
  public BodyMsg
  readBlockInvalid(ObjId objId,
      long offset,
      long length, 
      boolean blockImprecise,
      boolean exactOffset,
      Controller controller,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    try{
      //Env.sprintln("** ** Core:readBlocakInvalid:store:read going to be called");
      BodyMsg bm = null;
      bm = this.store.read(objId,
          offset,
          length,
          true, 
          blockImprecise,
          exactOffset,
          controller, timeout);
      //Env.sprintln("** ** Core:readBlocakInvalid:store:read returned");
      return bm;
    }catch(ReadOfInvalidRangeException e){
      assert(false);  // Called store.readBody with blockInvalid == true
      return null;
    }
  }

  //old
  public BodyMsg
  readBlockInvalid(ObjId objId,
      long offset,
      long length, 
      boolean blockImprecise,
      boolean exactOffset,
      long timeout)
  throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    try{
      //Env.sprintln("** ** Core:readBlocakInvalid:store:read going to be called");
      BodyMsg bm = null;
      bm = this.store.read(objId,
          offset,
          length,
          true, 
          blockImprecise,
          exactOffset, timeout);
      //Env.sprintln("** ** Core:readBlocakInvalid:store:read returned");
      return bm;
    }catch(ReadOfInvalidRangeException e){
      assert(false);  // Called store.readBody with blockInvalid == true
      return null;
    }
  }

  //-------------------------------------------------------------------------
  // readMeta() -- read the metadata for the specified offset of 
  //                the specified file
  //
  // returns
  //    PreciseInval on success
  //    throws NoSuchObjectException if objId does not exist (includes
  //         case where delete of objId is more recent than any write
  //         to objId)
  //    throws EOFException if offset is larger than any write to
  //         objId (since the last delete)
  //-------------------------------------------------------------------------
  public PreciseInv
  readMeta(ObjId objId, long offset)
  throws ObjNotFoundException, EOFException, IOException, ReadOfHoleException{
    recoveryComplete.waitRecovery();
    try{
      return this.store.readMeta(objId, offset);
    }catch(NoSuchEntryException e){
      throw new ObjNotFoundException(e.toString());
    }
  }

  //-------------------------------------------------------------------------
  // Bound parameter is there for debugging/testing -- to allow
  // us to insert an unbound write into log. Always return a 
  // bound inval summarizing the write.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  //-------------------------------------------------------------------------
  public BoundInval
  write(ObjId objId,
      long offset,
      long length,
      ImmutableBytes buffer,
      boolean bound,
      long maxBoundHops) 
  throws IOException{
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    return(this.write(objId, offset, length, DEFAULT_PRIORITY, buffer, bound, maxBoundHops));
  }

  //---------------------------------------------------------------
  // Bound parameter is there for debugging/testing -- to allow
  // us to insert an unbound write into log. Always return a 
  // bound inval summarizing the write.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  //---------------------------------------------------------------
  public BoundInval
  write(ObjId objId,
      long offset,
      long length,
      double priority,
      ImmutableBytes buffer,
      boolean bound,
      long maxBoundHops) 
  throws IOException{
    long ret = -1;
    AcceptStamp stamp;
    VVMap selfVVMap = null;
    //long deadlineMS = 0;

    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor

    AcceptStamp realStamp = new AcceptStamp(System.currentTimeMillis(), 
        this.myNodeId);
    //
    // Note: current implementation assume writes 
    // are bound by passing "true"
    //
    stamp = log.write(objId,
        offset,
        length,
        priority, 
        realStamp,
        buffer,
        bound,
        maxBoundHops,
        false);
    ObjInvalTarget target = new ObjInvalTarget(objId, offset, length);
    BoundInval bi = new BoundInval(target,
        stamp,
        buffer,
        priority,
        realStamp,
        maxBoundHops,
        false);

    //deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;

    //need to update All ISStatus.lpvv entry
    /*
    this.invalBuffer.applyInval(bi,
                                this.localWriteStreamId,
                                deadlineMS);
    replaced by calling the dataStore immediately
     */
    try{
      store.applyWrite(bi);
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 

    //System.out.println("*****after apply " + bi + ": isstatus= \n"+ this.getISStatusClone()); 
    return bi;
  }

 /** 
 *  Write multiple objects simultaneously 
 **/ 
  public MultiObjPreciseInv //removed synchronized by zjd
  writeMulti(MultiWriteEntry[] mwe, boolean embargoed) 
  throws IOException{
    long ret = -1;
    AcceptStamp stamp;
    VVMap selfVVMap = null;
    //    long deadlineMS = 0;
    MultiObjPreciseInv mopi = null;
    MultiObjPreciseInv boundMOPI = null;
    MultiWriteEntry[] boundMWE = null;

    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor

    AcceptStamp realStamp = new AcceptStamp(System.currentTimeMillis(),
        this.myNodeId);
    if(dbgLog){
      Env.dprintln(dbgLog, "Core::writeMulti calling log.writeMulti"); 
    }
    stamp = log.writeMulti(mwe, realStamp, embargoed);
    if(dbgLog){
      Env.dprintln(dbgLog, "Core::writeMulti calling log.writeMulti return "); 
    }
    mopi = new MultiObjPreciseInv(mwe, stamp, realStamp, embargoed);

    /*
    deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;
    this.invalBuffer.applyInval(mopi,
                                this.localWriteStreamId,
                                deadlineMS);
     */

    // Make a bound equivalent of the invalidate; make sure not to make
    // a bound delete
    boundMWE = new MultiWriteEntry[mwe.length];
    for(int i = 0; i < boundMWE.length; i++){
      boundMWE[i] = new MultiWriteEntry(mwe[i].getObjInvalTarget(),
          mwe[i].getPriority(),
          mwe[i].getImmutableBytes(),
          !mwe[i].getDelete(),
          mwe[i].getDelete());
    }
    boundMOPI = new MultiObjPreciseInv(boundMWE,
        stamp,
        realStamp,
        embargoed);

    try{
      if(dbgLog){
        Env.dprintln(dbgLog, "Core::writeMulti calling DataStore.writeMulti"); 
      }
      store.applyWrite(boundMOPI);
      if(dbgLog){
        Env.dprintln(dbgLog, "Core::writeMulti calling DataStore.writeMulti return"); 
      }
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 


    return boundMOPI;
  }

 /** 
 *  add Order Error Bound for local write 
 *  all the same as above method except add "targetOE" to log.write(); 
 *  
 *  WARNING -- if write is not bound, then body is not in redo  
 *  log --> data may be lost if we crash before next sync() 
 **/ 
  public BoundInval //removed synchronized by zjd
  write(ObjId objId,
      long offset,
      long length,
      double priority,
      ImmutableBytes buffer,
      boolean bound,
      long maxBoundHops,
      long targetOE, 
      boolean embargoed)
  throws IOException{

    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    long ret = -1;
    AcceptStamp stamp;
    VVMap selfVVMap = null;
    //long deadlineMS = 0;

    AcceptStamp realStamp = new AcceptStamp(System.currentTimeMillis(), 
        this.myNodeId);
    //
    // Note: current implementation assume writes 
    // are bound by passing "true"
    //
    stamp = log.write(objId, offset, length, 
        priority, realStamp, buffer, bound, 
        maxBoundHops, targetOE,
        embargoed);
    ObjInvalTarget target = new ObjInvalTarget(objId, offset, length);
    BoundInval bi = new BoundInval(target,
        stamp,
        buffer,
        priority,
        realStamp,
        maxBoundHops,
        embargoed);

    //deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;
    /*
    this.invalBuffer.applyInval(bi,
                                this.localWriteStreamId,
                                deadlineMS);
     */

    try{
      store.applyWrite(bi);
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 

    return bi;
  }

 /** 
 *  Write multiple objects simultaneously 
 **/ 
  public MultiObjPreciseInv//removed synchronized by zjd
  writeMulti(MultiWriteEntry[] mwe, long targetOE, boolean embargoed)
  throws Exception{
    long ret = -1;
    AcceptStamp stamp;
    VVMap selfVVMap = null;
//  long deadlineMS = 0;
    MultiObjPreciseInv mopi = null;

    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    AcceptStamp realStamp = new AcceptStamp(System.currentTimeMillis(),
        this.myNodeId);
    if(dbgLog){
      Env.dprintln(dbgLog, "Core::writeMulti calling log.writeMulti"); 
    }
    stamp = log.writeMulti(mwe, realStamp, targetOE, embargoed);
    if(dbgLog){
      Env.dprintln(dbgLog, "Core::writeMulti calling log.writeMulti return"); 
    }
    mopi = new MultiObjPreciseInv(mwe, stamp, realStamp, embargoed);

    /*
    deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;
    this.invalBuffer.applyInval(mopi,
                                this.localWriteStreamId,
                                deadlineMS);
     */

    try{
      if(dbgLog){
        Env.dprintln(dbgLog, "Core::writeMulti calling DataStore.writeMulti"); 
      }
      store.applyWrite(mopi);
      if(dbgLog){
        Env.dprintln(dbgLog, "Core::writeMulti calling DataStore.writeMulti return"); 
      }
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 

    return mopi;
  }

 /** 
 *  Currently, this interface just deletes the object. 
 *  Do we want to create an interface to atomically 
 *  update the directory file and delete the object? 
 **/ 
  public DeleteInv delete(ObjId obj)
  throws IOException{
    AcceptStamp stamp;
    VVMap selfVVMap = null;
//  long deadlineMS = 0;

    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    stamp = log.delete(obj);
    DeleteInv di = new DeleteInv(obj, stamp);

    /*  
      deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;

      this.invalBuffer.applyInval(di,
                                  this.localWriteStreamId,
                                  deadlineMS);
     */


    try{
      store.applyWrite(di);
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 
    return di;
  }

 /** 
 *  make the localstore and isstatus persistent 
 **/ 
  public void syncStateToDisk(){
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    try{
      log.syncToDisk();
      this.store.syncToDisk();
    }
    catch(IOException e){
      System.err.println("syncStateToDisk fails: " + e.toString());
      e.printStackTrace();
      System.exit(-1); // Something is very bad.
    }
  }

 /** 
 *  get the datastore checkpoint VV 
 **/ 
  public AcceptVV getCPVV(){
    return this.store.getCPVV();
  }

  ////////////////////////////////////////////////////
  // Interface to RMI-invoked workers to send
  // changes to other nodes.
  ////////////////////////////////////////////////////

 /** 
 *  check if my lpVV for all interest sets at or below subscribeSet are  
 *  at least minLpVV 
 **/ 
  public boolean
  checkMinLPVV(SubscriptionSet ss, VV cpMinLPVV){    
    return this.store.checkMinLPVV(ss, cpMinLPVV);
  }


 /** 
 *  asks the store to put updates for the subscription set in the upq 
 **/ 

  public void
  subscribeUpdates(UpdatePriorityQueue upq, SubscriptionSet is, VV startVV){
    this.store.subscribeUpdates(upq, is, startVV);
  }


 /** 
 *  informs the store so that it will not put updates for  
 *  a subscription set in the upq 
 **/ 
  public synchronized void
  removeSubscribeUpdates(UpdatePriorityQueue upq, SubscriptionSet is){
    this.store.removeSubscribeUpdates(upq, is);
  }

 /** 
 *  Remove a byte range that is scheduled to be prefetched from the 
 *  prefetch queue 
 **/ 
  public void
  cancelPrefetchRequest(ObjId id, long offset, long length, NodeId destId){
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    this.store.cancelPrefetchRequest(id, offset, length, destId);
  }

  public InMemLogIterator makeInMemLogIterator(AcceptVV excludedStartVV){
    return log.makeInMemLogIterator(excludedStartVV);
  }

  public void removeInMemLogIterator(InMemLogIterator iter){
    log.removeInMemLogIterator(iter);
  }
 /** 
 *  By default make a normal invalIterator 
 **/ 
  public InvalIterator makeInvalIterator(SubscriptionSet is, VV startVV, 
      NodeId nodeId)
  throws OmittedVVException
  {

    if(dbgMaxAcc) {
      System.out.println("making inval iterator with max acc time = " + maxAccumulateTime);
    }
    InvalIterator ret = log.makeInvalIterator(is, 
        startVV, 
        maxAccumulateTime,
        maxSendDelay);
    assert ret!= null;
    iiFilter.add(nodeId, ret);
    return ret;
  }


 /** 
 *     make a CatchupInvalIterator 
 **/ 
  public CatchupInvalIterator makeCatchupInvalIterator(SubscriptionSet is, VV startVV, 
      NodeId nodeId, 
      VV endVV)
  throws OmittedVVException
  {
    CatchupInvalIterator ret = log.makeCatchupInvalIterator(is, 
        startVV, 
        maxAccumulateTime,
        maxSendDelay, 
        endVV);
    assert ret!= null;
    iiFilter.add(nodeId, ret);
    return ret;
  }

  public void removeInvalIterator(NodeId nodeId, InvalIterator iter){
    //assert false;//refer to callback_psuedo_4_4.txt TBD section
    iiFilter.remove(nodeId, iter);
    log.removeInvalIterator(iter);
  }

  public void removeInvalIterator(NodeId nodeId, CatchupInvalIterator iter){
    //assert false;//refer to callback_psuedo_4_4.txt TBD section
    iiFilter.remove(nodeId, iter);
    log.removeInvalIterator(iter);
  }

  public OutgoingConnectionMailBox makeOutgoingConnectionMailBox(AcceptVV excludedStartVV){
    Iterator<SingleWriterInval> iter = log.makeInMemLogIterator(excludedStartVV);
    OutgoingConnectionMailBox ret = new OutgoingConnectionMailBox(iter);
    log.addMailBox(ret);
    return ret;
  }

  public void removeOutgoingConnectionMailBox(OutgoingConnectionMailBox mailBox){
    log.removeMailBox(mailBox);
  }

 /** 
 *  this method is called when this node receives an invalidate with endVV 
 *  of "vv" from "nodeId".  
 *  result: advance all the outgoing invaliterator(normal/catchup) for "nodeId" 
 *  so that we don't send this invalidate back to "nodeId" again. 
 **/ 
  public void notifyRecvInval(NodeId nodeId, VV vv){
    //assert false; //refer to callback_psuedo_4_4.txt TBD section
    // zjd 2007 Feb turn on again see mike's 2007.3.2.txt about pingpong 
    // take the first option and possible problem: some node might be starving
    // for getting some precise invalidates
    iiFilter.advanceVV(nodeId, vv);
  }

 /** 
 *  update recoverStartVV 
 **/ 
  protected synchronized void updateRecoverStartVV(VV vv){
    recoverStartVV.advanceTimestamps(vv);
    if(recoverStartVV.includes(recoverEndVV)){
      notifyAll();
    }
  }


  // Return number of "new" bytes that are made current by this store
  public long applyBody(BodyMsg body)
  throws InterruptedException, IOException, InvalidBodyException{
    try{
      //Env.sprintln(" -- -- Core::applyBody:log:wait going to be called");
      log.waitUntilTentativeApplied(body.getAcceptStamp());
      //Env.sprintln(" -- -- Core::applyBody:log:wait returned");
      //Env.sprintln(" -- -- Core::applyBody:store:applyBody going to be called");
      //long bytes = this.invalBuffer.applyBody(body, true);

           long bytes = store.applyBody(body);
      //Env.sprintln(" -- -- Core::applyBody:store.applyBody returned");
      return bytes;
    }
    catch(InterruptedException e){
      Env.warn("Core::interruptedException shouldn't happen"
          + " in current implementation, right?");
      e.printStackTrace();
      assert(false);
      throw e;
    }
    catch(IOException f){
      Env.warn("Core::applyBody got IOException " 
          + f.toString());
      f.printStackTrace();
      throw f;
    }
  }

 /** 
 *  when recovering, log ignore the unbind msg  
 **/ 
  public void applyUnbind(UnbindMsg unbind){
    //System.err.println("applying unbind " + unbind); 
    if(!recovering){
      log.applyUnbind(unbind);
    }
  }

  public void applyDebargo(DebargoMsg dm)
  throws IOException{
    log.applyDebargo(dm);
    this.store.applyDebargo(dm);
  }

  public void  syncCheck(AcceptStamp acceptStamp)
  throws InterruptedException{
    //
    // wait until the specified acceptStamp has been processed
    // and is reflected in the local state and then
    // return (this is used to allow a node that sends a bound
    // update to know when that update has reached a given node)
    // (If that msg has already been processed, return immediately)
    //
    if(acceptStamp.getLocalClock() <= AcceptStamp.BEFORE_TIME_BEGAN){
      return;
    }

    log.waitUntilTentativeApplied(acceptStamp);
  }

  public void  syncCheck(VV syncVV)
  throws InterruptedException{
    //
    // wait until the specified syncVV has been processed
    // and is reflected in the local state and then
    // return (this is used to allow a node that sends a bound
    // update to know when that update has reached a given node)
    // (If that msg has already been processed, return immediately)
    //
    log.waitUntilTentativeApplied(syncVV);
  }

  public void  logSyncCheck(AcceptStamp acceptStamp)
  throws InterruptedException{
    //
    // wait until the specified acceptStamp has been processed
    // and is reflected in the local state and then
    // return (this is used to allow a node that sends a bound
    // update to know when that update has reached a given node)
    // (If that msg has already been processed, return immediately)
    //
    log.waitUntilTentativeApplied(acceptStamp);
  }

  public void checkState(){
    Env.dprintln(true, "" + log);
  }

  public AcceptVV getDiskOmitVV(){
    return log.getDiskOmitVV();
  }

  public AcceptVV getInMemOmitVV(){
    return log.getInMemOmitVV();
  }

  /*  
  public VV getLpVV(PreciseSet ps){
    return ps.getISR().getLastPreciseVV();
  }
   */
 /** 
 *  Get the largest Stamp of the specific NodeId this Node knows 
 **/ 
  public AcceptStamp getAcceptStamp(NodeId nodeId)
  {
    try{
      return new AcceptStamp(log.getCurrentVV().getStampByServer(nodeId), nodeId);
    }catch (NoSuchEntryException e){
      return new AcceptStamp(-1, nodeId);
    }
  }

  public AcceptVV getCurrentVV(){
    return log.getCurrentVV();
  }

  public void startSyncBodyCheck(){
    (new SyncBodyCheckWorker(this)).start();
  }

  public boolean isAllUPQEmpty(){
    return this.store.isAllUPQEmpty();
  }

  public void waitAllUPQClear(){
    //this.store.waitAllUPQClear();
  }

  public void notifyEmptyAttempt(){
    this.store.notifyEmptyAttempt();
  }


 /** 
 *  ****** callback ********* start 
 **/ 

 /** 
 *  call datastore.sendCheckpoint: send CPStartMsg, cvv, PTreeNode list,  
 *                                      RAS records, RAS_SHIP_DONE 
 **/ 
  public boolean sendCheckpoint(TaggedOutputStream tos,
      SubscriptionSet ss,
      AcceptVV startVV,
      boolean withBody,
      AcceptVV prevVV)
  throws IOException{
    boolean ret = false;
    long start, end, totalTime;
    if(measureTime){
      start = System.currentTimeMillis();
    }
    if(!prevVV.includes(log.getCurrentVV())){//only send a checkpoint when the prevv is equals
      // to cvv
      if(measureTime||PRINT_CP_INFO){
        Env.dprintln(dbgPerformance||PRINT_CP_INFO, " sendCheckpoint for " + ss.toString() +
        " attempt failed");
      }
      return false;
    }

    ret = this.store.sendCheckpoint(tos, ss, startVV, withBody, prevVV);
    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-start);
      LatencyWatcher.put("DS.sendCP", 
          totalTime);

    }
    return ret;
  }

 /** 
 *  ****** callback ********* start 
 **/ 

 

 /** 
 *  call datastore.sendCheckpoint: send CPStartMsg, cvv, PTreeNode list,  
 *                                      RAS records, RAS_SHIP_DONE 
 *  difference from the previous sendCheckpoint: 
 *    doesn't need to wait until the cvv and streamCVV matches 
 *    just send the cp for streamSS from streamCVV to cvv if they doesn't match 
 **/ 
  public AcceptVV sendCheckpoint(TaggedOutputStream tos,
      SubscriptionSet ss,
      AcceptVV startVV,
      boolean withBody,
      SubscriptionSet streamSS,
      AcceptVV streamCVV/*, 
      boolean includeAll*/)
  throws IOException{

    long start, end, totalTime;
    if(measureTime){
      start = System.currentTimeMillis();
    }
    AcceptVV ret = null;
    ret = this.store.sendCheckpoint(tos, ss, startVV, withBody, 
        streamSS, streamCVV);
    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-start);
      LatencyWatcher.put("DS.sendCP", 
          totalTime);

    }
    return ret;
  }


 /** 
 *  call datastore.applyCheckpoint: receive PTreeNode list,  
 *                                      RAS records, RAS_SHIP_DONE 
 **/ 

  public void applyCheckpoint(TaggedInputStream s, CPStartMsg cpsm)
  throws IOException, ClassNotFoundException, CausalOrderException{

    //
    // now we are ready to update the checkpoint to the datastore
    // note: it's ok if gapInv apply succeed while the datastore update failed 
    //       as long as the datastore update is atomic
    //       If this is the case, we only leave the isstatus in a very
    //       imprecise state but still causal, i.e. safe.
    //
    this.store.applyCheckpoint(s, cpsm);
  }

 /** 
 *  add connection pointer to the corresponding place in ISStatus tree 
 **/ 
  public void addConnectionToISStatus(SubscriptionSet ss,
      ConnectionState ic){
    this.store.addConnectionToISStatus(ss, ic);
  }

 /** 
 *   remove connection pointer from the corresponding place in ISStatus tree 
 **/ 
  public void removeConnectionFromISStatus(SubscriptionSet ss,
      ConnectionState ic,
      VV newLpvv){
    this.store.removeConnectionFromISStatus(ss, ic, newLpvv);
  }

 /** 
 *  apply to log and hand it to the invalidate buffer 
 **/ 
  public synchronized void
  handGIToBuffer(GeneralInv gi, IncommingConnection ic){
    /* 
   long deadlineMS = (System.currentTimeMillis() +
                       this.MAX_INVAL_DELAY_MS);
    try{
      if(!recovering){

        log.applyInval(gi);

      } else {
        updateRecoverStartVV(gi.getEndVV());
      }

      this.invalBuffer.applyInval(gi,
                                  ic,
                                  deadlineMS);

    }catch(Exception e){
      e.printStackTrace();
    }
     */  
  }

 /** 
 *     -- apply to Datastore -- update cvv 
 *                              update per obj state if precise 
 **/ 
  public synchronized void applyInvalToDataStore(GeneralInv gi){
    // 
    // Note: The GeneralInv message can be a precise invalidation,
    // an imprecise invalidation, or a bound invalidation (e.g., a precise
    // invalidation *bound* to the message). 
    // In the third case, don't forget to mark the message
    // as "bound" in the log and to update the locally stored
    // copy. Also note -- no need to send update to subscribers
    // since it is bound, we will send the update with the inval
    // (though we might want to enqueue the update to subscribers
    // when we get an unbind message, though that risks sending
    // the same body twice, as well...perhaps track to whom
    // we have sent bound updates...
    try{
      this.store.applyInval(gi);
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
  }

  /* this method is splited into two methods to delay IncommingConnection update
   together with datastore update invoked by invalidateBuffer.
   the two methods: 1. handGIToBuffer : update log, hand(gi, ic) to invalidateBuffer
                    2. applyInvalToDataStore called by IncommingConnection.applyGI
                       after updating prevVV which is invoked by invalidateBufferThread
                       when the deadline arrives.
   */
 /** 
 *   Apply an invalidate 
 *     -- apply to log 
 *     -- apply to Datastore -- update cvv 
 *                              update per obj state if precise 
 **/ 
  public synchronized void applyInval(GeneralInv gi, StreamId streamId){
    // 
    // Note: The GeneralInv message can be a precise invalidation,
    // an imprecise invalidation, or a bound invalidation (e.g., a precise
    // invalidation *bound* to the message). 
    // In the third case, don't forget to mark the message
    // as "bound" in the log and to update the locally stored
    // copy. Also note -- no need to send update to subscribers
    // since it is bound, we will send the update with the inval
    // (though we might want to enqueue the update to subscribers
    // when we get an unbind message, though that risks sending
    // the same body twice, as well...perhaps track to whom
    // we have sent bound updates...

    // NOTE: Assume 0ms clock skew
    long start = System.currentTimeMillis();
    long end=start;
    long deadlineMS = (System.currentTimeMillis() +
        this.MAX_INVAL_DELAY_MS);
    try{
      if(!recovering){

        log.applyInval(gi);
        if(IncommingConnection.dbgPerformance){
          end = System.currentTimeMillis(); 
        }
      } else {
        updateRecoverStartVV(gi.getEndVV());
      }

      //  this.invalBuffer.applyInval(gi,
      //                            streamId,
      //                            deadlineMS);
      this.store.applyInval(gi);
      if(IncommingConnection.dbgPerformance){
        start = end;
        end = System.currentTimeMillis();
      }
    }catch(Exception e){
      e.printStackTrace();
    }

  }


 /** 
 *  notify imprecise read thread block on DataStore 
 **/ 
  public void notifyDatastore(SubscriptionSet ss, VV prevVV){
    store.notifyImpreciseRead(ss, prevVV);
  }

 /** 
 *  Split off a PreciseSet from a given pathName matching Node 
 **/ 
  public void
  splitISStatus(String pathName, String newName)
  throws NoMatchPreciseSetException{
    store.splitISStatus(pathName, newName);
  }

 /** 
 *  join all this.children, remove this.children, update this.other.lpvv 
 **/ 
  public boolean
  joinISStatus(String pathName){
    return store.joinISStatus(pathName);
  }


 /** 
 *  liesInPreciseIS() -- 
 *  Returns true if objId lies in precise interest sets only 
 **/ 
  public boolean 
  liesInPreciseIS(ObjId objId){
    return this.store.liesInPreciseIS(objId);
  }  
 /** 
 *  ****** callback ********* end 
 **/ 

 /** 
 *  clean all the environments: .log, .store, .stats(later on) 
 *  -- by deleting the corresponding files  
 **/ 
  protected void
  cleanEnv() throws IOException{
    File logDir = new File(logPath);
    File storeDir = new File(storePath);
    try{
      File[] logFiles = logDir.listFiles();
      File[] storeFiles = storeDir.listFiles();
      if (logFiles != null){
        for (int i = 0; i < logFiles.length; i ++){
          if(!logFiles[i].delete()){
        	  throw new IOException("CleanDB failed");
          }
        }
      }

      if (storeFiles != null){
        for ( int i = 0; i < storeFiles.length; i++){
          storeFiles[i].delete();
        }
      }
    } catch (SecurityException se){
      assert false; //should never be here.
    }
  }

 /** 
 *  For testing -- destroy the files in which the datastore state is  
 *  stored (this is used to test constuctor log replay). 
 **/ 
  protected void
  testDestroyDataStore(boolean verbose){
    close();
    File storeDir = new File(storePath);
    try{
      File[] storeFiles = storeDir.listFiles();
      if(storeFiles != null){
        for(int i = 0; i < storeFiles.length; i++){
          if(verbose){
            Env.dprintln(verbose, "Core::testDestroyDataStore deleting " 
                + storeFiles[i]);
          }
          storeFiles[i].delete();
        }
      }
      storeFiles = storeDir.listFiles();
      assert(storeFiles.length == 0);
    } 
    catch(SecurityException se){
      assert false; //should never be here.
    }
  }

 /** 
 *  return recoverStartVV -- for test 
 **/ 
  public VV getRecoverStartVV(){
    return recoverStartVV;
  }

 /** 
 *  return recoverEndVV -- for test 
 **/ 
  public VV getRecoverEndVV(){
    return recoverEndVV;
  }

 /** 
 *  Return the component-wise minimum LPVV of the LPVVs of all 
 *  PreciseSets that overlap this SubscriptionSet. We expose this 
 *  interface via Core because controllers and policies might 
 *  want to know about ISStatus; note that ISStatus has no lock 
 *  and replies on DataStore for lock, so we have to go through 
 *  there for synchronization. 
 **/ 
  public AcceptVV getISStatusMinLpVV(SubscriptionSet ss){
    return this.store.getISStatusMinLpVV(ss);
  }


 /** 
 *  applyFakeInval() -- apply fake invalidates for test only  
 **/ 
  public void applyFakeInval(GeneralInv gi){
    try{
      log.applyInval(gi);
      this.store.applyInval(gi);
      //this.invalBuffer.applyInval(gi,
      //                            this.localWriteStreamId,
      //                            System.currentTimeMillis());

    }catch(Exception e){
      assert false;
    }
  }



  public CommitInv commit(InvalTarget targetSet, AcceptStamp targetAS) throws IOException{
    assert(recoveryComplete.get()); // Require either recoverLocalState() or testSkipRecoverLocalState() after constructor
    long ret = -1;
    AcceptStamp stamp;

    AcceptStamp realStamp = new AcceptStamp(System.currentTimeMillis(), 
        this.myNodeId);
    //
    // Note: current implementation assume writes 
    // are bound by passing "true"
    //
    stamp = log.commit(targetSet, targetAS, realStamp);

    CommitInv bi = new CommitInv(targetSet,
        stamp,
        targetAS,
        realStamp);

    //deadlineMS = System.currentTimeMillis() + MAX_INVAL_DELAY_MS;
    /*
    this.invalBuffer.applyInval(bi,
                                this.localWriteStreamId,
                                deadlineMS);
     */

    try{
      store.applyWrite(bi);
    }catch(CausalOrderException e){
      System.exit(0);
      assert false;
    } 

    return bi;

  }




}



//---------------------------------------------------------------------------
/* $Log: Core.java,v $
/* Revision 1.128  2008/01/23 20:35:27  nalini
/* added null random access state
/*
/* Revision 1.127  2007/11/05 23:32:41  zjiandan
/* fix SubscribeBWunit.
/*
/* Revision 1.126  2007/10/07 04:47:04  zjiandan
/*  coda cooperative caching exp
/*
/* Revision 1.125  2007/09/10 23:52:22  zjiandan
/* upgrade to newest BerkeleyDB je version.
/*
/* Revision 1.124  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.123  2007/07/12 17:02:32  zjiandan
/* *** empty log message ***
/*
/* Revision 1.122  2007/07/11 19:08:07  zjiandan
/* clean IncommingConnection
/*
/* Revision 1.121  2007/07/03 07:57:54  zjiandan
/*  more unit tests
/*
/* Revision 1.120  2007/06/04 21:40:59  zjiandan
/* expose stream catchup type CP|LOG option to rmiClient.subscribeInval().
/*
/* Revision 1.119  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.118  2007/04/11 07:18:33  zjiandan
/* Fix SubscribeBWUnit test.
/*
/* Revision 1.117  2007/04/05 22:33:23  nalini
/* subscribeBWunit works at times
/*
/* Revision 1.116  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.115  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.114  2007/03/14 19:55:18  dahlin
/* Compiler errors fixed
/*
/* Revision 1.113  2007/03/11 04:16:54  zjiandan
/* Add timeout into read interface and expose acceptstamp in LocalInterface.write.
/*
/* Revision 1.112  2007/03/08 23:21:20  nalini
/* reverting to version 1.110
/*
/* Revision 1.110  2007/03/08 03:06:37  zjiandan
/* removed wrong assertion.
/*
/* Revision 1.109  2007/03/07 20:39:07  zjiandan
/* add InvalIterator Filter.
/*
/* Revision 1.108  2007/03/06 18:25:50  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.107  2007/03/01 06:39:44  nalini
/* added support for maxBoundHop at Local Interface
/*
/* Revision 1.106  2007/02/27 04:44:40  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.105  2007/02/23 20:54:28  zjiandan
/* Fixed mem leak problems in NFS2Interface and some other bugs.
/* Andrew benchmark passed, performance still needs to tune.
/*
/* Revision 1.104  2006/12/08 22:52:54  nalini
/* redid amol's work: added parameter to control log sync-ing
/*
/* Revision 1.99  2006/10/18 19:55:53  nalini
/* removed code for sending a sync to sender whenever a bound inval is received
/*
/* Revision 1.98  2006/10/16 05:44:26  zjiandan
/* Fixed DataStore::applyCheckpoint large lock problem (refer to mike's 2006.10.12.txt), moved IncommingConnection unit tests to junit.
/*
/* Revision 1.97  2006/10/12 22:28:56  nalini
/* updated footer to enable cvs log msg
/*
 */
//---------------------------------------------------------------------------
