package code;

 /** 
 *  Template for writing JUnit tests. To write the test for class
 
 *  Foo, copy this file to FooUnit.java and update as described
 
 *  below.
 
 **/ 

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class CoreUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals
  private Process rmiregistry;
  
  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public CoreUnit (final String s) {
    super (s);
  }


  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    if( verbose ){
      Env.dprintln(verbose, "rmiregistry started");
    }
    Thread.sleep(2000);
    

  }

  protected void tearDown() throws Exception{
    rmiregistry.destroy();
    Thread.sleep(2000);
    super.tearDown();
    
  }

  private static void makePractiConfig(long first, int total){
    Config.createEmptyConfig();

    long NODE_1_ID = first;
    int port = 9921;
    String NODE_1_IP = "localhost";
    
    for(int i = 0; i < total; i++){
      
      Config.addOneNodeConfig(new NodeId(NODE_1_ID++),
			      NODE_1_IP,
			    port++,
			    port++,
			    port++,
			    port++,
			    port++,
			    "test" + File.separatorChar + "local-" + 
			    NODE_1_ID + ".db",
			    "/*",
			    -1L,
			    NODE_1_IP,
			    port++,
			    port++,
			    -1,
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    }
        
  }

  //-----------------------------------------------------------------------
  // Test recovery of state across "reboots" of core.
  //-----------------------------------------------------------------------
  public static void testRecovery(){
    System.err.print("CoreUnit::testRecovery ... ");
    makePractiConfig(9, 10);
    //Config.readConfig("ufs.config");
    NodeId myId = new NodeId(10);
    NodeId myOtherId1 = new NodeId(11);
    NodeId myOtherId2 = new NodeId(12);
    doTestRecovery(myId, myOtherId1, myOtherId2, false, true);
    myOtherId1 = new NodeId(13);
    myOtherId2 = new NodeId(14);
    doTestRecovery(myId, myOtherId1, myOtherId2, true, true);
    myOtherId1 = new NodeId(15);
    myOtherId2 = new NodeId(16);
    doTestRecovery(myId, myOtherId1, myOtherId2, false, false);
    myOtherId1 = new NodeId(17);
    myOtherId2 = new NodeId(18);
    doTestRecovery(myId, myOtherId1, myOtherId2, true, false);
  }


  //-----------------------------------------------------------------------
  // recovery test
  // scenario:
  //    -- start from empty
  //       issue writes 
  //       if(doUnbind)
  //           apply unbind
  //       if(cleanShutdown)
  //          sync to disk
  //          destroy core and other objects
  //       else
  //          destroy core and other objects
  //          destroyDataStore
  //       recover core and other objects
  //       verify written ddata
  //       write more data -- verify correct acceptStamp
  //       verify written data
  //       
  //
  // A hack -- since we shutdown and restart core multiple times
  // from within the same process but we have not tried to get
  // clean shutdown to work (e.g., to tell all the worker threads
  // to cleanly shut down), we listen on different ports each time around
  // as specified in the port argument
  //-----------------------------------------------------------------------
  private static void
  doTestRecovery(NodeId myId, NodeId idToStealPorts1, NodeId idToStealPorts2,  
                 boolean doUnbind, boolean cleanShutdown){
    byte[] bdy = new byte[5];
    bdy[0] = 10;
    bdy[1] = 9;
    bdy[2] = 8;
    bdy[3] = 7;
    bdy[4] = 6;
    ImmutableBytes body = new ImmutableBytes(bdy);
    int ii;
    int nFiles1 = 100;
    int nFiles2 = 100;
    PreciseInv writeResult1[] = new PreciseInv[nFiles1];
    PreciseInv writeResult2[] = new PreciseInv[nFiles2];
    ObjId oid;
    RMIServerImpl rmiServer = null;
    BodyMsg bm = null;
    UnbindMsg ubm = null;


    if( verbose ){
      Env.dprintln(verbose, "test Core recovery (doUnbind="
                 + doUnbind + " cleanShutdown=" + cleanShutdown + " ...");
    }
    
    //Config.readConfig("ufs.config");
    RMIClient myRMIClient = new RMIClient();
    
    //create Core as URANode constructor did
    boolean filterOn = true;
    boolean cleanDb = true;  // start from empty
    if( verbose ){
      Env.dprintln(verbose, "WARNING: Hack Alert! "
                 + " Creating new core with ports stolen from nodeID " 
                 + idToStealPorts1.toString());
    }
    Config.setPortInval(myId, Config.getPortInval(idToStealPorts1));
    Config.setPortBody(myId, Config.getPortBody(idToStealPorts1));
    Config.setPortSync(myId, Config.getPortSync(idToStealPorts1));
    Config.setPortNice(myId, Config.getPortNice(idToStealPorts1));
    Config.setPortCP(myId, Config.getPortCP(idToStealPorts2));
    Core core = new Core(myRMIClient, filterOn, cleanDb, myId, false);
    Controller controller = new LocalController(core);
    try{
      rmiServer = new RMIServerImpl(core, controller);
    }
    catch(java.rmi.RemoteException e){
      e.printStackTrace();
      assert(false);
    }
    rmiServer.start();
    SocketServer socketServer = new SocketServer(core, controller);
    LocalInterface localInterface = new LocalInterface(controller, core);
    if( vverbose ){
      Env.dprintln(vverbose, "recoverStartVV = " + core.getRecoverStartVV().toString());
      Env.dprintln(vverbose, "recoverEndVV = " + core.getRecoverEndVV().toString());
    }
    assert (core.getRecoverStartVV().includes(core.getRecoverEndVV())); 
    core.recoverLocalState(rmiServer); //do nothing
    

    //
    // Write
    //
    for(ii = 0; ii < nFiles1; ii++){
      oid = new ObjId("/testRecovery/" + ii);
      try{
        writeResult1[ii] = core.write(oid, 0L, bdy.length, Core.DEFAULT_PRIORITY, 
                                      body, true, Long.MAX_VALUE);
      }
      catch(IOException ioe){
        ioe.printStackTrace();
        assert(false);
      }
    }
    
    //
    // Read, unbind, read
    //
    for(ii = 0; ii < nFiles1; ii++){
      oid = new ObjId("/testRecovery/" + ii);
      try{
        bm = core.read(oid, 0L, bdy.length, true, true, true, -1);
      }
      catch(Exception ex){
        ex.printStackTrace();
        assert(false);
      }
      assert bm != null;
      assert(bm.getAcceptStamp().equals(writeResult1[ii].getAcceptStamp()));
      if(doUnbind){
        ubm = new UnbindMsg(new ObjInvalTarget(oid, 0L, bdy.length), 
                                      bm.getAcceptStamp());
        core.applyUnbind(ubm);
      }
      try{
        bm = core.read(oid, 0L, bdy.length, true, true, true, -1);
      }
      catch(Exception ex2){
        ex2.printStackTrace();
        assert(false);
      }
      assert bm != null;
      assert(bm.getAcceptStamp().equals(writeResult1[ii].getAcceptStamp()));
    }


    socketServer.shutdown(); // This was added after hack alert... we may no longer need hack
    //
    // Shutdown. Optionally delete DataStore state (forcing
    // next instance to rely on log replay to recover state.)
    //
    core.close();
    if(!cleanShutdown){
      core.testDestroyDataStore(verbose);
    }
    


    localInterface = null;
    socketServer = null;
    rmiServer.close();
    rmiServer = null;
    controller = null;
    core.close();
    core = null;


    //
    // Reanimate Core. Since our old threads may not all be 
    // dead (we have not thought a lot about clean shutdown
    // within a process...), need to change to some ports
    // that are not currently in use. Steal ports from 
    // other nodeIds in the config file.
    //
    if( verbose ){
      Env.dprintln(verbose, "WARNING: Hack Alert! "
                 + " Creating new core with ports stolen from nodeID " 
                 + idToStealPorts2.toString());
    }
    Config.setPortInval(myId, Config.getPortInval(idToStealPorts2));
    Config.setPortBody(myId, Config.getPortBody(idToStealPorts2));
    Config.setPortSync(myId, Config.getPortSync(idToStealPorts2));
    Config.setPortNice(myId, Config.getPortNice(idToStealPorts2));
    Config.setPortCP(myId, Config.getPortCP(idToStealPorts2));
    core = new Core(myRMIClient, filterOn, false, myId, false);
   
    controller = new LocalController(core);
    try{
      rmiServer = new RMIServerImpl(core, controller);
    }
    catch(java.rmi.RemoteException rez){
      rez.printStackTrace();
      assert(false);
    }
    rmiServer.start();
    socketServer = new SocketServer(core, controller);
    localInterface = new LocalInterface(controller, core);

    //
    // Recover state from log that is not yet in datastore. Start
    // by checking to make sure that what we expect is in
    // log and datastore.
    //
    core.testSkipRecoverLocalState(); // We know what we're doing issuing reads before recovery
    if(cleanShutdown){
      //
      // Everything should already be there
      //
      assert(core.getRecoverStartVV().includes(core.getRecoverEndVV()));
      assert(core.getRecoverEndVV().includes(writeResult1[nFiles1-1].getAcceptStamp()));
      for(ii = 0; ii < nFiles1; ii++){
        oid = new ObjId("/testRecovery/" + ii);
        try{
          bm = core.read(oid, 0L, bdy.length, true, true, true, -1);
        }
        catch(Exception bme){
          bme.printStackTrace();
          assert(false);
        }
        assert bm != null;
        assert(bm.getAcceptStamp().equals(writeResult1[ii].getAcceptStamp()));
      }
    }
    else{
      //
      // Nothing should be there (but it should be in the recovery redo log)
      //
      if( vverbose ){
	Env.dprintln(vverbose, "recoverStartVV2 = " + core.getRecoverStartVV().toString());
	Env.dprintln(vverbose, "recoverEndVV2 = " + core.getRecoverEndVV().toString());
      }
      assert(!core.getRecoverStartVV().includes(core.getRecoverEndVV()));
      assert(!core.getRecoverStartVV().includes(writeResult1[nFiles1-1].getAcceptStamp()));
      assert(core.getRecoverEndVV().includes(writeResult1[nFiles1-1].getAcceptStamp()));
      for(ii = 0; ii < nFiles1; ii++){
        oid = new ObjId("/testRecovery/" + ii);
        try{
          bm = core.read(oid, 0L, bdy.length, true, true, true, -1);
          assert(false); // Experiment intended to rely on state in log...
        }
        catch(ObjNotFoundException onfe){
          // expected result
        }
        catch(Exception zef){
          // Not expected
          zef.printStackTrace();
          assert(false);
        }
      }
    }

    core.forceRecoverLocalState(rmiServer);

    //
    // Now everything should be there
    //
    for(ii = 0; ii < nFiles1; ii++){
      oid = new ObjId("/testRecovery/" + ii);
      try{
        bm = core.read(oid, 0L, bdy.length, false, true, true, -1);
        assert bm != null;
        assert(bm.getAcceptStamp().equals(writeResult1[ii].getAcceptStamp()));
      }
      catch(ReadOfInvalidRangeException roire){
        assert(doUnbind == true);
        try{
          PreciseInv pi = core.readMeta(oid, 0L);
          assert(pi != null);
          assert(pi.getAcceptStamp().equals(writeResult1[ii].getAcceptStamp()));
        }
        catch(Exception lsr){
          lsr.printStackTrace();
          assert(false);
        }
      }
      catch(Exception ler){
        ler.printStackTrace();
        assert(false);
      }
    }

    //
    // New writes should be given higher acceptstamp than old ones
    //
    for(ii = 0; ii < nFiles2; ii++){
      oid = new ObjId("/testRecovery/" + (nFiles1 + ii));
      try{
        writeResult2[ii] = core.write(oid, 0L, bdy.length, Core.DEFAULT_PRIORITY, 
                                      body, true, Long.MAX_VALUE);
      }
      catch(Exception lez){
        lez.printStackTrace();
        assert(false);
      }
      assert(writeResult2[ii].getAcceptStamp().gt(writeResult1[nFiles1-1].getAcceptStamp()));
    }
    for(ii = 0; ii < nFiles2; ii++){
      oid = new ObjId("/testRecovery/" + (nFiles1+ii));
      try{
        bm = core.read(oid, 0L, bdy.length, true, true, true, -1);
      }
      catch(Exception ldr){
        ldr.printStackTrace();
        assert(false);
      }
      assert bm != null;
      assert(bm.getAcceptStamp().equals(writeResult2[ii].getAcceptStamp()));
    }

    rmiServer.close();
    rmiServer = null;
    socketServer.shutdown();
    socketServer = null;
    
    core.close();
    if( verbose ){
      Env.dprintln(verbose, "recovery test succeeds");
    }

    
  }

  
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(CoreUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java CoreUnit foo" runs
   *   CoreUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "CoreUnit";
    System.err.print(name + " self test begins...");

    boolean useJunitFrameWork = true;
    
    if(!useJunitFrameWork){//call as normal java class
      CoreUnit cu = new CoreUnit("");
      try{
	cu.setUp();
	cu.testRecovery();
	cu.tearDown();
      }catch(Exception e){
	e.printStackTrace();
      }
    }else{
    
      //
      // Default: run all tests
      //
      TestSuite ste = new TestSuite();
      Test test;
      boolean doAllTests = true;
      
      if(s.length > 0){
	int ii;
	for(ii = 0; ii < s.length; ii++){
	  if(s[ii].equals("-verbose")){
	    verbose = true;
	  }
	  else if(s[ii].equals("-vverbose")){
	    verbose = true;
	  }
	  else{
	    doAllTests = false;
	    ste.addTest(new RandomAccessStateUnit("test" + s[ii]));
	  }
	  
	}
      }
      if(doAllTests){
	test = suite();
      }
      else{
	test = ste;
      }
      TestRunner tr = new TestRunner();
      tr.doRun(test);
      System.err.println(name + " self test succeeds");
      System.exit(0);
    }
  }
  
}
