package code;

 /** 
 * Implementation of a mutable vector that keeps track of currently seen 
 * updates 
 **/ 
import java.util.Collection;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.io.Serializable;

public class CounterVV extends VV implements Serializable{ // *mutable*
  /*
  Notice that the members will have to be either separate serverID,
  timestamp pairs or a new class; they can't be AcceptStamp's because
  AcceptStamps are immutable
   */


  private Hashtable stamps;

 /** 
 * Constructor 
 **/ 
  public 
  CounterVV(){
    this.stamps = new Hashtable();
  }

 /** 
 * Constructor 
 **/ 
  public 
  CounterVV(VV vv){
    Object token = null;
    VVIterator vvIter = null;
    NodeId nodeId = null;
    long timeStamp = 0;
    assert vv != null;
    this.stamps = new Hashtable();
    for(vvIter = vv.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();
      nodeId = vv.getServerByIteratorToken(token);
      timeStamp = vv.getStampByIteratorToken(token);
      assert(this.stamps.get(nodeId) == null);
      this.stamps.put(nodeId, new Long(timeStamp));
    }
  }

 /** 
 * Constructor 
 **/ 
  public 
  CounterVV(AcceptStamp[] stamps_){
    this.stamps = new Hashtable();
    for(int i = 0; i < stamps_.length; i++){
      assert(this.stamps.get(stamps_[i].getNodeId()) == null);
      this.stamps.put(stamps_[i].getNodeId(),
          new Long(stamps_[i].getLocalClock()));
    }
  }


 /** 
 * Clone -- not clonable 
 * since we are mutable, we cannot return a pointer to ourself. 
 **/ 
  public Object clone(){
    assert(false);// try to use the "cloneAcceptVV()" method
    return null;
  }

public void reset(){
  this.stamps.clear();
}

 /** 
 * since the CounterVV is not clonable because of its mutability 
 * we can create an immutable copy of the current version 
 **/ 
  public AcceptVV cloneAcceptVV(){
    VVIterator timeIter = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;

    int size = stamps.size();
    AcceptStamp[] asArr = new AcceptStamp[size];
    int count = 0;
    for(timeIter = this.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      timeTimeStamp = this.getStampByIteratorToken(token);
      asArr[count] = new AcceptStamp(timeTimeStamp, nodeId);
      count++;
    }
    assert size == count;
    return new AcceptVV(asArr);
  }

 /** 
 * Move timestamps forward 
 **/ 
  public void 
  advanceTimestamps(VV time){
    //
    // Post condition:
    // foreach entry e in time
    //  this.stamps{e.server} = max(prev value of
    //                           this.stamps{e.server}, e.stamp)
    //
    VVIterator timeIter = null;
    Long currentTimeStampLong = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;

    for(timeIter = time.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = time.getServerByIteratorToken(token);
      timeTimeStamp = time.getStampByIteratorToken(token);
      currentTimeStampLong = (Long)this.stamps.get(nodeId);
      if((currentTimeStampLong == null) ||
          (timeTimeStamp > currentTimeStampLong.longValue())){
        this.stamps.put(nodeId, new Long(timeTimeStamp));
      }
    }
  }


public void addMinVV(VV vv){
  // Set this vv to the component-wise minimum of "this" and "vv". If
  // vector A has an entry that vector B does not, the minimum of the
  // two components will be set to entry present.
  Object token = null;
  NodeId nid = null;
  long ts1 = 0;
  long ts2 = 0;
  VVIterator  i = null;

  i = this.getIterator();
  while(i.hasMoreElements()){
    token = i.getNext();
    nid = this.getServerByIteratorToken(token);
    ts1 = this.getStampByIteratorToken(token);
    if(vv.containsNodeId(nid)){
      try{
        ts2 = vv.getStampByServer(nid);
        this.setStampByServer(nid, ts1 < ts2 ? ts1 : ts2);
      }catch(NoSuchEntryException e){
       assert false;
      }
    }
  }
  //
  // In case vv has elements that this doesn't
  //
  i = vv.getIterator();
  while(i.hasMoreElements()){
    token = i.getNext();
    nid = vv.getServerByIteratorToken(token);
    if(!this.containsNodeId(nid)){
      ts2 = vv.getStampByIteratorToken(token);
      this.setStampByServer(nid, ts2);
    }
  }
}

public void addMinAS(AcceptStamp as){
    if(this.containsNodeId(as.getNodeId())){
      try{
        long ts = this.getStampByServer(as.getNodeId());
        this.setStampByServer(as.getNodeId(), ts < as.getLocalClock() ? ts : as.getLocalClock());
      }catch(NoSuchEntryException e){
       assert false;
      }
    }else{
      stamps.put(as.getNodeId(), as.getLocalClock());
    }
  }

public void addMaxAS(AcceptStamp as){
  advanceTimestamps(as);
}

public void addMaxVV(VV vv){
  advanceTimestamps(vv);
}
 /** 
 * Move timestamp forward 
 **/ 
public void 
advanceTimestamps(AcceptStamp as){
Long currentTimeStampLong = null;

 currentTimeStampLong = (Long)this.stamps.get(as.getNodeId());
 if((currentTimeStampLong == null) ||
    (as.getLocalClock() > currentTimeStampLong.longValue())){
   this.stamps.put(as.getNodeId(), as.getLocalClock());
 }
}
 /** 
 *  Set the current vector to the minimum of itself and "vv" 
 **/ 
  public void
  setToMinVV(VV vv){
    // Set this vv to the component-wise minimum of "this" and "vv". If
    // vector A has an entry that vector B does not, the minimum of the
    // two components will be set to -1.
    Object token = null;
    NodeId nid = null;
    long ts1 = 0;
    long ts2 = 0;
    VVIterator  i = null;

    i = this.getIterator();
    while(i.hasMoreElements()){
      token = i.getNext();
      nid = this.getServerByIteratorToken(token);
      ts1 = this.getStampByIteratorToken(token);
      try{
        ts2 = vv.getStampByServer(nid);
        this.setStampByServer(nid, ts1 < ts2 ? ts1 : ts2);
      }catch(NoSuchEntryException e){
        this.setStampByServer(nid, -1);
      }
    }
    //
    // In case vv has elements that this doesn't
    //
    i = vv.getIterator();
    while(i.hasMoreElements()){
      token = i.getNext();
      nid = vv.getServerByIteratorToken(token);
      ts2 = vv.getStampByIteratorToken(token);
      try{
        ts1 = this.getStampByServer(nid);
        assert(ts1 <= ts2);
      }catch(NoSuchEntryException e){
        this.setStampByServer(nid, -1);
      }
    }
  }


 /** 
 * Return true if this vv contains NodeId n 
 **/ 
  public boolean 
  containsNodeId(NodeId n){
    return this.stamps.containsKey(n);
  }

 /** 
 * Methods for VV 
 **/ 

 /** 
 * Return true if this vv is equal to the given vv when ignoring those 
 * components with BEFORE_TIME_BEGAIN values 
 **/ 
  public final boolean 
  equalsIgnoreNegatives(VV vv) {
    return (this.includes(vv) && vv.includes(this));
  }

 /** 
 * Return true if this vv is equal to the given vv 
 **/ 
  public final boolean 
  equals(Object o) {
    //
    // return true if this *equals* vv
    // version vector A *equals* version vector B if
    // every {serverId, counterB} in B has a corresponding
    // {serverId, counterA} in A with counterA == counterB
    // |A| == |B|
    boolean result = true;
    NodeId s = null;
    VV vv = null;
    Long thisStampLong = null;
    VVIterator vvIter = null;
    int numElements = 0;
    Object token = null;

    if(o instanceof VV){
      vv = (VV)o;
      vvIter = vv.getIterator();
      while(vvIter.hasMoreElements() && result){
        numElements++;
        token = vvIter.getNext();
        s = vv.getServerByIteratorToken(token);
        thisStampLong = (Long)this.stamps.get(s);
        if((thisStampLong == null) ||
            (vv.getStampByIteratorToken(token) !=
              thisStampLong.longValue())){
          result = false;
        }
      }
      result = result && (this.stamps.size() == numElements);
    }else{
      // o is not a VV
      result = false;
    }
    return(result);
  }

 /** 
 * Of all the <NodeId, TimeStamp> pairs in the version vector, 
 * return the largest TimeStamp. 
 **/ 
  public final long 
  getMaxTimeStamp(){
    long largestTimeStamp = Long.MIN_VALUE;
    Object token = null;

    for(VVIterator i = this.getIterator(); i.hasMoreElements();){
      token = i.getNext();
      if(this.getStampByIteratorToken(token) > largestTimeStamp){
        largestTimeStamp = this.getStampByIteratorToken(token);
      }
    }
    return(largestTimeStamp);
  }

 /** 
 * Of all the <NodeId, TimeStamp> pairs in the version vector, 
 * return the smallest TimeStamp. 
 **/ 
  public final long 
  getMinTimeStamp(){
    if(stamps.size() == 0){
      return AcceptStamp.BEFORE_TIME_BEGAN;
    }
    long smallestTimeStamp = Long.MAX_VALUE;
    Object token = null;

    for(VVIterator i = this.getIterator(); i.hasMoreElements();){
      token = i.getNext();
      if(this.getStampByIteratorToken(token) < smallestTimeStamp){
        smallestTimeStamp = this.getStampByIteratorToken(token);
      }
    }
    return(smallestTimeStamp);
  }

 /** 
 * Of all the <NodeId, TimeStamp> pairs in the version vector, 
 * return the smallest TimeStamp. 
 **/ 
  public final AcceptStamp 
  getMinStamp(){

    long smallestTimeStamp = Long.MAX_VALUE;
    Object token = null;

    for(VVIterator i = this.getIterator(); i.hasMoreElements();){
      token = i.getNext();
      if(this.getStampByIteratorToken(token) < smallestTimeStamp){
        smallestTimeStamp = this.getStampByIteratorToken(token);
      }
    }
    return new AcceptStamp(this.getStampByIteratorToken(token),
        this.getServerByIteratorToken(token));
  }

 /** 
 * Return true if this vv is a superset of the given vv 

 * Note: whenever change anything here, remember to do the same thing for 
 * method: includes(AcceptStampe as) and also the same methods in 
 * AcceptVV.java  
 **/ 
  public final boolean 
  includes(VV vv) {
    //return true if this *includes* vv
    //version vector A *includes* version vector B if
    //every {serverId, counterB} in B has a corresponding
    //{serverId, counterA} in A with counterA >= counterB

    boolean result = true;
    NodeId s = null;
    Object token = null;
    VVIterator vvIter = null;


    for(vvIter = vv.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();


      if (vv.getStampByIteratorToken(token) == AcceptStamp.BEFORE_TIME_BEGAN){
        continue; //ignore the component if it's just a dummy value
      }
      s = vv.getServerByIteratorToken(token);

      try{
        if(this.getStampByServer(s) < vv.getStampByIteratorToken(token)){
          result = false;

          break;
        }
      }catch(NoSuchEntryException e){
        result = false;

        break;
      }
    }

    return(result);
  }

 /** 
 * Returns component-wise max of self and vv. Assumes all VV's 
 * have the same number of components. 
 **/ 
  public final CounterVV 
  maxVV(CounterVV vv) {

    CounterVV max = new CounterVV(this);
    VVIterator  i = this.getIterator(); 
    while(i.hasMoreElements()) {
      Object token_i = i.getNext();	
      NodeId nid = this.getServerByIteratorToken(token_i);
      long ts1 = this.getStampByIteratorToken(token_i);
      long ts2;
      try {
        ts2 = vv.getStampByServer(nid);
      } catch (NoSuchEntryException e) {
        ts2 = ts1;
      }
      max.setStampByServer(nid, ts1 > ts2 ? ts1 : ts2);
    }
    //
    // In case vv has elements that this doesn't
    //
    i = vv.getIterator(); 
    while(i.hasMoreElements()) {
      Object token_i = i.getNext();	
      NodeId nid = vv.getServerByIteratorToken(token_i);
      long ts1 = vv.getStampByIteratorToken(token_i);
      long ts2;
      try {
        ts2 = this.getStampByServer(nid);
      } catch (NoSuchEntryException e) {
        ts2 = ts1;
      }
      max.setStampByServer(nid, ts1 > ts2 ? ts1 : ts2);
    }
    return max;
  }

 /** 
 * Return false if all components of vv are greater than the same in this 
 **/ 
  public final boolean 
  includesAnyPartOf(VV vv){
    //return true if this *includesAnyPartOf* vv

    //that is, if any event that happened before vv could also have
    //happened before this

    //that is, if there exists any server s.t. this.getStampByServer(s)
    //>= vv.getStampByServer(s)

    // TEST: If includes is true, includesAnyPartOf must be true
    // NOTE: If there exists a server s that is not in vv but is in this
    //       vector, return true.

    NodeId s = null;
    boolean result = false;
    Enumeration e = null;

    for(e = this.stamps.keys(); (e.hasMoreElements() && (!result));){
      try{
        s = (NodeId)e.nextElement();

        if(this.getStampByServer(s) == AcceptStamp.BEFORE_TIME_BEGAN){
          continue;
        }
        if(this.getStampByServer(s) >= vv.getStampByServer(s)){
          result = true;
        }
      }catch(NoSuchEntryException excp){
        result = true;
      }
    }
    return(result);
  }

 /** 
 *  Return true if any component of "this" is greater than the same in "vv" 
 **/ 
  public final boolean isAnyPartGreaterThan(VV vv){
    //return true, only if there exists any server s s.t.
    //   (1)s is a valid component in both this and vv 
    //&& (2)this.getStampByServer(s) > vv.getStampByServer(s)
    // Note. the difference of this method and the above method
    //    is > and >=
    //    and if there's a s exists in "this", but not exist in "vv",
    //           isAnyPartGreaterThan() will not return true
    //       but includesAnyPartOf() will return true
    NodeId s = null;
    boolean result = false;
    Enumeration e = null;

    for(e = this.stamps.keys(); (e.hasMoreElements() && (!result));){
      try{
        s = (NodeId)e.nextElement();
        if(this.getStampByServer(s) > vv.getStampByServer(s)){
          result = true;
        }
      }catch(NoSuchEntryException excp){
        // result = true;
      }
    }
    return(result);

  }

 /** 
 * Return true if this vv has a higher entry for the given node in ts 

 * Note: whenever change anything here, remember to do the same thing for 
 * method: includes(AcceptVV as) and also the same methods in 
 * AcceptVV.java 
 **/ 
  public final boolean 
  includes(AcceptStamp ts){
    //return true if this has an entry for ts.nodeId and if
    //that entry has at least as high a counter as ts.acceptStamp

    NodeId s = null;
    Long thisClockLong = null;
    boolean result = false;

    assert ts != null;
    if(ts.getLocalClock()== AcceptStamp.BEFORE_TIME_BEGAN){
      return true;
    }
    s = ts.getNodeId();
    thisClockLong = (Long)this.stamps.get(s);
    if(thisClockLong != null){
      result = (thisClockLong.longValue() >= ts.getLocalClock());
    }else{
      assert ts.getLocalClock() != AcceptStamp.BEFORE_TIME_BEGAN;
      result = false;
    }
    return(result);
  }

 /** 
 * Return the time stamp entry in this VV for the given node 
 **/ 
  public final long 
  getStampByServer(NodeId nodeId)
  throws NoSuchEntryException {
    Long tsLong = null;

    //return the clock value for the specified server 
    //or throw exception if no server
    tsLong = (Long)this.stamps.get(nodeId);
    if(tsLong == null){
      throw(new NoSuchEntryException());
    }

    return(tsLong.longValue());
  }

 /** 
 * Set the time stamp entry in this VV for the given node 
 **/ 
  public final long 
  setStampByServer(NodeId nodeId, long ts){
    Long tsLong = new Long(ts);
    Long oldVal = null;

    //return the clock value for the specified server 
    //or throw exception if no server
    oldVal =  (Long)this.stamps.put(nodeId, tsLong);

    if(oldVal != null) {
      return oldVal.longValue();
    }
    else {
      return -1;
    }
  }


 /** 
 * Set the time stamp entry in this VV for the accept stamp 
 **/ 
  public final void
  setStamp(AcceptStamp as){
    Long tsLong = new Long(as.getLocalClock());
    NodeId id = as.getNodeId();

    this.stamps.put(id, tsLong);

  }

 /** 
 * Return an iterator 
 **/ 
  public final VVIterator 
  getIterator(){
    // getNext() of this iterator returns a token
    VVIterator vvi = new VVIterator(this.stamps.keys());

    return(vvi);
  }

  public int
  getSize(){
    return this.stamps.size();
  }
 /** 
 * Return the server name stored at the token 
 **/ 
  public NodeId 
  getServerByIteratorToken(Object iterToken){
    NodeId nodeId = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    assert(this.stamps.get(nodeId) != null);
    return(nodeId);
  }

 /** 
 * Return the time stamp stored at the token 
 **/ 
  public long 
  getStampByIteratorToken(Object iterToken){
    NodeId nodeId = null;
    Long tsLong = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    tsLong = (Long)this.stamps.get(nodeId);
    assert(tsLong != null);
    return(tsLong.longValue());
  }

 /** 
 * Return the time stamp stored at the token 
 **/ 
  public void 
  setStampByIteratorToken(Object iterToken, long ts){
    NodeId nodeId = null;

    assert(iterToken instanceof NodeId);
    nodeId = (NodeId)iterToken;
    this.stamps.put(nodeId, new Long(ts));
  }

  
 /** 
 * Return the time stamp stored at the token 
 **/ 
  public void 
  setStampByNodeId(NodeId nodeId, long ts){
    this.stamps.put(nodeId, new Long(ts));
  }

 /** 
 * Return true if this > vv (not if this == vv) 
 **/ 
  public final boolean greaterThan(VV vv){
    // return true if this > vv, where A > B if (1) for all {node, stamp}
    // in B, A has a larger or equal stamp, and (2) at least one stamp
    // in A has a larger value than the corresponding value in B or A
    // has more entries than B (copied from AcceptVV)
    boolean seenLarger = false;
    boolean result = true;
    NodeId s = null;
    Object token = null;
    VVIterator vvIter = null;
    int numVVElements = 0;

    try{
      for(vvIter = vv.getIterator(); vvIter.hasMoreElements();){
        numVVElements++;
        token = vvIter.getNext();
        s = vv.getServerByIteratorToken(token);
        if(this.getStampByServer(s) <
            vv.getStampByIteratorToken(token)){
          result = false;
          break;
        }else if((!seenLarger) && (this.getStampByServer(s) >
        vv.getStampByIteratorToken(token))){
          seenLarger = true;
        }
      }
    }catch(NoSuchEntryException e){
      result = false;
    }
    if(result && (!seenLarger)){
      // If we have more entries, even if the common entries match,
      // we are still larger
      seenLarger = (this.stamps.size() > numVVElements);
    }
    return(result && seenLarger);
  }

 /** 
 *  for all i this[i]=this[i]-1 
 **/ 
  public void
  decrementAllByOne(){

    Object token = null;
    NodeId nid = null;
    long ts1 = 0;
    long ts2 = 0;
    VVIterator  i = null;

    i = this.getIterator();
    while(i.hasMoreElements()){
      token = i.getNext();
      nid = this.getServerByIteratorToken(token);
      ts1 = this.getStampByIteratorToken(token);

      ts1 --;
      this.setStampByServer(nid, ts1 < -1 ? -1 : ts1);
    }
  }

 /** 
 *   throw away any trivial component  
 *        whose stamp == AcceptStamp.BEFORE_TIME_BEGAN 
 **/ 
  public void 
  dropNegatives(){

    VVIterator timeIter = null;
    //Long currentTimeStampLong = null;
    NodeId nodeId = null;
    long timeTimeStamp = 0;
    Object token = null;
    //Hashtable newStamps = new Hashtable();

    for(timeIter = this.getIterator(); timeIter.hasMoreElements();){
      token = timeIter.getNext();
      nodeId = this.getServerByIteratorToken(token);
      timeTimeStamp = this.getStampByIteratorToken(token);
      if(timeTimeStamp == AcceptStamp.BEFORE_TIME_BEGAN){
        stamps.remove(nodeId);
      }

    }
    
  }
  
 /** 
 *   throw away component for node nId 
 **/ 
  public void 
  dropNode(NodeId nId){

    if(stamps.containsKey(nId)){
      this.stamps.remove(nId);
    }
  }

 /** 
 * Return a string representation 
 **/ 
  public final String 
  toString(){
    String str = null;
    Object token = null;
    VVIterator vvIter = null;

    str = "(";
    for(vvIter = this.getIterator(); vvIter.hasMoreElements();){
      token = vvIter.getNext();
      str += "<\"" + this.getServerByIteratorToken(token);
      str += "\", " + this.getStampByIteratorToken(token) + "> ";
    }
    str += ")";
    return(str);
  }

 /** 
 * Make a startVV with AcceptStamp.BEFORE_TIME_BEGAN for all known nodes. 
 **/ 
  public static CounterVV 
  makeVVAllNegatives(){
    Enumeration allNodes = Config.getKnownNodeIds();
    Vector v = new Vector();
    while(allNodes.hasMoreElements()){
      NodeId n = (NodeId)allNodes.nextElement();
      AcceptStamp a = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, n);
      v.addElement(a);
    }
    AcceptStamp[] aa = (AcceptStamp[])v.toArray(new AcceptStamp[0]);
    return new CounterVV(aa);
  }


 /** 
 * Used for testing 
 **/ 
  public static void 
  main(String[] argv){
    // Test new methods: New constructor, equals
    // NOTE: Because we copied almost all methods from AcceptVV,
    // we don't need to test most methods; only the new constructor
    // and the .equals methods
    AcceptVV acceptVV1 = null;
    AcceptVV acceptVV2 = null;
    AcceptStamp[] as1 = null;
    CounterVV counterVV1 = null;
    CounterVV counterVV2 = null;

    // Test CounterVV(VV vv)
    as1 = new AcceptStamp[3];
    as1[0] = new AcceptStamp(100, new NodeId(200));
    as1[1] = new AcceptStamp(200, new NodeId(300));
    as1[2] = new AcceptStamp(300, new NodeId(400));
    acceptVV1 = new AcceptVV(as1);
    counterVV1 = new CounterVV(acceptVV1);
    System.out.println("acceptVV1 = " + acceptVV1);
    System.out.println("new CounterVV(acceptVV1) = " + counterVV1);
    System.out.println("counterVV1.equals(acceptVV1) = " +
        counterVV1.equals(acceptVV1));

    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(300, new NodeId(200));
    as1[1] = new AcceptStamp(300, new NodeId(300));
    acceptVV2 = new AcceptVV(as1);
    counterVV2 = new CounterVV(acceptVV2);
    System.out.println("acceptVV1 = " + acceptVV2);
    System.out.println("new CounterVV(acceptVV2) = " + counterVV2);
    System.out.println("counterVV2.equals(acceptVV2) = " +
        counterVV2.equals(acceptVV2));
    System.out.println("counterVV1.equals(acceptVV2) = " +
        counterVV1.equals(acceptVV2));
    System.out.println("counterVV2.equals(acceptVV1) = " +
        counterVV2.equals(acceptVV1));
    System.out.println("counterVV2.equals(counterVV2) = " +
        counterVV2.equals(counterVV2));
    System.out.println("counterVV1.equals(counterVV1) = " +
        counterVV1.equals(counterVV1));

    CounterVV max = counterVV1.maxVV(counterVV2);

    System.out.println(counterVV1);
    System.out.println(counterVV2);
    System.out.println(max);

    //test includes
    as1 = new AcceptStamp[3];
    as1[0] = new AcceptStamp(300, new NodeId(200));
    as1[1] = new AcceptStamp(300, new NodeId(300));
    as1[2] = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, new NodeId(400));
    acceptVV1 = new AcceptVV(as1);
    counterVV1 = new CounterVV(acceptVV1);
    System.out.println("acceptVV1 = " + acceptVV1);
    System.out.println("new CounterVV(acceptVV1) = " + counterVV1);
    System.out.println("counterVV1.equals(acceptVV1) = " +
        counterVV1.equals(acceptVV1));

    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(300, new NodeId(200));
    as1[1] = new AcceptStamp(300, new NodeId(300));
    acceptVV2 = new AcceptVV(as1);
    counterVV2 = new CounterVV(acceptVV2);

    assert counterVV1.includes(counterVV2);
    assert counterVV2.includes(counterVV1);

    AcceptStamp ACCEPT_BEGINNING_OF_TIME = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, new NodeId(NodeId.WILDCARD_NODE_ID));

    AcceptStamp[] cps = new AcceptStamp[4];
    cps[0] = new AcceptStamp(99, new NodeId(0));
    cps[1] = new AcceptStamp(-1, new NodeId(1));
    cps[2] = new AcceptStamp(1, new NodeId(2));
    cps[3] = ACCEPT_BEGINNING_OF_TIME;
    CounterVV cpStartVV = new CounterVV(cps);


    assert cpStartVV.includes(ACCEPT_BEGINNING_OF_TIME);

    System.out.println("before increment:" + cpStartVV);
    AcceptVV newCP = AcceptVV.incrementAll(cpStartVV);
    System.out.println("after incrementAll:" + newCP);


    AcceptStamp[] newcps = new AcceptStamp[4];
    newcps[0] = new AcceptStamp(100, new NodeId(0));
    newcps[1] = new AcceptStamp(-1, new NodeId(1));
    newcps[2] = new AcceptStamp(2, new NodeId(2));
    newcps[3] = ACCEPT_BEGINNING_OF_TIME;
    CounterVV newcpStartVV = new CounterVV(newcps);


    assert newcpStartVV.equals(newCP);
  }

@Override
public Collection<NodeId> getNodes(){
  // TODO Auto-generated method stub
  return this.stamps.keySet();
}

}

//---------------------------------------------------------------------------
/* $Log: CounterVV.java,v $
/* Revision 1.30  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.29  2006/04/20 03:52:52  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.28  2005/10/13 00:24:23  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.27  2005/07/18 05:10:22  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.26  2005/03/15 21:15:34  zjiandan
/* Automatic GC checked in.
/*
/* Revision 1.25  2005/03/02 21:43:19  zjiandan
/* Removed some bugs
/*
/* Revision 1.24  2005/02/28 21:00:12  nayate
/* Added new method setToMinVV()
/*
/* Revision 1.23  2005/02/28 20:25:58  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.22  2004/10/26 20:28:05  zjiandan
/* fixed include(VV vv) methods to ignore dummy entries.
/*
/* Revision 1.21  2004/10/12 20:40:13  zjiandan
/* add a new public method: boolean isAnyPartGreaterThan(VV vv)
/*
/* Revision 1.20  2004/08/18 22:44:43  dahlin
/* Made BoundInval subclass of PreciseInval; RandomAccessState passes 2 self tests
/*
/* Revision 1.19  2004/07/29 14:24:34  dahlin
/* Fixed indentation as per standard; deleted incorrect (and unused) clone method
/*
/* Revision 1.18  2004/05/25 12:59:37  lgao
/* Add chain mechanism to filter the duplicated update forwarding.
/*
/* Revision 1.17  2004/05/23 10:29:22  nayate
/* Added a "makeVVAllNegatives" method
/*
/* Revision 1.16  2004/05/20 01:44:07  dahlin
/* Looks like spanning trees are getting set up with TrivialSpanningTreeDirectory (though more testing remains)
/*
/* Revision 1.15  2004/05/20 01:20:47  arun
/* maxVV support test
/*
/* Revision 1.14  2004/05/20 01:12:06  arun
/* *** empty log message ***
/*
/* Revision 1.13  2004/05/20 01:10:59  arun
/* *** empty log message ***
/*
/* Revision 1.12  2004/05/20 01:06:11  arun
/* added some code for recovery support in store. Added maxVV definition in COunterVV etc.
/*
/* Revision 1.11  2004/05/20 00:47:34  dahlin
/* Fixed several bugs to make inval log exchange work (biggest one: ISIterator handling case when an single-writer log exists but has no records in it; also added some debugging tools
/*
/* Revision 1.10  2004/05/10 20:48:19  dahlin
/* Clarified RMI exceptions; full version of (stub) DemandReadWorker
/*
/* Revision 1.9  2004/04/28 01:38:47  nayate
/* Added a "greaterThan" method
/*
/* Revision 1.8  2004/04/26 20:59:50  zjiandan
/* Finished UpdateLog and partially tested(InterestSet intersect not complete yet
/* waiting for the local interface to finish design of InterstSet).
/* TodoList: Test Parallel InvalIterators and Inserters.
/*
/* Revision 1.7  2004/04/25 20:34:54  zjiandan
/* Add containsNodeId public method
/*
/* Revision 1.6  2004/04/25 20:07:00  zjiandan
/* Add new constructer
/*
/* Revision 1.5  2004/04/22 20:33:56  nayate
/* Finished and tested CounterVV
/*
/* Revision 1.4  2004/04/21 17:37:41  zjiandan
/* Miner Change to make UpdateLog.java compile successfully.
/*
/* Revision 1.3  2004/04/18 03:53:12  zjiandan
/* Stub implementations that compile and minor fixes
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
