package code;

 /** 
 *  Used to store an interest set as a tree. Used by DirectoryInterestSet. 
 **/ 
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Set;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;
import java.io.File;
import java.io.Serializable;

public final class DISTreeNode implements Cloneable,Serializable, Immutable
{
    private final String name;
    private final boolean precise;
    private final HashMap children;

 /** 
 *  Constructor 
 **/ 
    private DISTreeNode(String name_, boolean precise_, HashMap children_)
    {
	this.name = name_;
	this.precise = precise_;
	this.children = children_;
    }


 /** 
 *  Return true if this node is empty 
 **/ 
    public final boolean isEmpty()
    {
	return(this.precise && this.name.equals(""));
    }

 /** 
 *  Make tree from a given path and return it 
 **/ 
    public final static DISTreeNode makeNode(String path)
    {
	LinkedList pathList = null;
	File file = null;
	boolean lastPrecise = false;
	String name = null;
	DISTreeNode node = null;
	HashMap newChildren = null;

	// Break the path into its components
	pathList = new LinkedList();
	file = new File(path);
	assert(path.startsWith("/"));
	while(file != null){
	    pathList.addFirst(file.getName());
	    file = file.getParentFile();
	}

	assert(pathList.size() > 0);
	if(pathList.getLast().equals("*")){
	    pathList.removeLast();
	    lastPrecise = false;
	}else{	    
	    lastPrecise = true;
	}
	name = (String)pathList.removeLast();
	node = new DISTreeNode(name, lastPrecise, new HashMap());
	while(!pathList.isEmpty()){
	    // Save the previous node as the child of this one
            newChildren = new HashMap(2); // Use small default initial capacity
	    newChildren.put(name, node);
	    name = (String)pathList.removeLast();
	    node = new DISTreeNode(name, true, newChildren);
	}
        
	return(node);
    }

 /** 
 *  Return an array of Strings representing the directories in this IS 
 **/ 
    public final void getDirStrings(String prefix, Vector dirs)
    {
	String str = null;
	String childName = null;
	DISTreeNode childNode = null;

	str = "";
	for(Iterator i = this.children.keySet().iterator(); i.hasNext();){
	    childName = (String)i.next();
	    childNode = (DISTreeNode)this.children.get(childName);
	    childNode.getDirStrings(prefix + this.name + "/", dirs);
	}
	if(this.children.isEmpty()){
	    str += prefix + this.name;
	    if(!this.precise){
		str += "/*";
	    }
	    dirs.add(str);
	}
    }

    //-----------------------------------------------------------------------
    // Clone this tree based but "chop" out regions that do not lie in isNode
    //-----------------------------------------------------------------------
    public final DISTreeNode cloneChopExcluded(DISTreeNode isNode)
    {
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode nodeChild = null;
	DISTreeNode newNode = null;
	DISTreeNode newChild = null;
	Map.Entry childEntry = null;
	HashMap newChildren = null;

	//assert(isNode.hasImpreciseLeaves());
	assert(this.name.equals(isNode.name));
	newChildren = new HashMap();
	if(isNode.children.isEmpty() && (isNode.precise)){
	    // isNode has reached a precise child; we must have as well.
	    //assert(this.children.isEmpty() && (this.precise));
	    //newNode = (DISTreeNode)this.clone();
	    newNode = new DISTreeNode(this.name, this.precise, new HashMap());
	}else{
	    assert((!isNode.children.isEmpty()) || (!isNode.precise));
	    Iterator i = null;
	    for(i = this.children.entrySet().iterator(); i.hasNext();){
		assert(this.precise);
		childEntry = (Map.Entry)i.next();
		childName = (String)childEntry.getKey();
		child = (DISTreeNode)childEntry.getValue();
		if(!isNode.precise){
		    // isNode is imprecise, so we can add this child directly
		    newChildren.put(childName, child);
		}else if(!isNode.children.containsKey(childName)){
		    // isNode does not have this child, so we only store an
		    // imprecise representation of the child
		    newChild = new DISTreeNode(childName,
					       false,
					       new HashMap());
		    newChildren.put(childName, newChild);
		}else{
		    nodeChild = (DISTreeNode)isNode.children.get(childName);
		    newChildren.put(childName,
				    child.cloneChopExcluded(nodeChild));
		}
	    }
	    newNode = new DISTreeNode(this.name, this.precise, newChildren);
	}
	return(newNode);
    }

    //-----------------------------------------------------------------------
    // Return true if this node refers to exactly one element
    //-----------------------------------------------------------------------
    public final boolean isSingleton()
    {
	return((this.precise) && (this.children.isEmpty()));
    }

    //-----------------------------------------------------------------------
    // Return the name stored in this entry
    //-----------------------------------------------------------------------
    public final String getName()
    {
	return(this.name);
    }

    //-----------------------------------------------------------------------
    // Return true if this InterestSet will overlap isNode
    //-----------------------------------------------------------------------
    public final boolean overlapsIS(DISTreeNode isNode)
    {
	/*
	  Trace until the leaves of isNode, and if we find a node
	  in this tree that hangs lower than isNode or a node in
	  isNode that hangs lower (or equal) than this, return true
	*/
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode isNodeChild = null;
	Map.Entry childEntry = null;
	boolean result = false;
	Iterator iter = null;

	assert(this.name.equals(isNode.name));
	if(this.children.isEmpty()){
	    // We reached a leaf node that is in the interest set
	    result = true;
	}else if(isNode.children.isEmpty()){
	    // We reached a leaf node in the interest set that is
	    // overlapped by "this"
	    if(!isNode.precise){
		// NOTE: Technically this if condition should always be
		// true, but for simplicity we allow the caller to call
		// is.overlapIS(inv) instead of the intended inv.overlapsIS(is)
		result = true;
	    }else{
		result = false;
	    }
	}else{
	    // Not at the leaf of either tree
	    assert(this.precise);
	    assert(isNode.precise);
	    result = false;
	    iter = this.children.entrySet().iterator(); 
	    while(iter.hasNext() && (!result)){
		childEntry = (Map.Entry)iter.next();
		childName = (String)childEntry.getKey();
		if(isNode.children.containsKey(childName)){
		    child = (DISTreeNode)childEntry.getValue();
		    isNodeChild = (DISTreeNode)isNode.children.get(childName);
		    result = result || child.overlapsIS(isNodeChild);
		}
	    }
	}
	return(result);
    }

 /** 
 *  Returned the union of this tree and another 
 **/ 
    public final DISTreeNode union(DISTreeNode b)
    {
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode bChild = null;
	Map.Entry childEntry = null;
	HashMap newChildren = null;
	DISTreeNode newNode = null;
	Iterator iter = null;

	assert(this.name.equals(b.name));
	newChildren = new HashMap();
	if(this.children.isEmpty()){
	    newNode = new DISTreeNode(this.name,
				      this.precise && b.precise,
				      b.children);
	}else if(b.children.isEmpty()){
	    newNode = new DISTreeNode(this.name,
				      this.precise && b.precise,
				      this.children);
	}else{
	    assert(this.precise && b.precise);
	    iter = this.children.entrySet().iterator();
	    while(iter.hasNext()){
		childEntry = (Map.Entry)iter.next();
		childName = (String)childEntry.getKey();
		child = (DISTreeNode)childEntry.getValue();
		if(b.children.containsKey(childName)){
		    bChild = (DISTreeNode)b.children.get(childName);
		    assert(bChild != null);
		    assert(bChild.name.equals(childName));
		    assert(!newChildren.containsKey(childName));
		    newChildren.put(childName, child.union(bChild));
		}else{
		    newChildren.put(childName, child.clone());
		}
	    }
	    iter = b.children.entrySet().iterator();
	    while(iter.hasNext()){
		childEntry = (Map.Entry)iter.next();
		childName = (String)childEntry.getKey();
		child = (DISTreeNode)childEntry.getValue();
		if(!newChildren.containsKey(childName)){
		    newChildren.put(childName, child.clone());
		}
	    }
	    newNode = new DISTreeNode(this.name,
				      this.precise && b.precise,
				      newChildren);
	}
	return(newNode);
    }

 /** 
 *  Return the intersection of this tree and another without enforcing 
 *  the "entails" property 
 **/ 
    public final DISTreeNode intersection(DISTreeNode b)
    {
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode bChild = null;
	DISTreeNode newChild = null;
	DISTreeNode result = null;
	Map.Entry childEntry = null;
	HashMap newChildren = null;
	Iterator iter = null;

	assert(this.name.equals(b.name));
	if(!b.precise){
	    // If b is imprecise, include all of this
	    assert(b.children.isEmpty());
	    result = this;
	}else if(!this.precise){
	    // If we are imprecise, include all of b
	    assert(this.children.isEmpty());
	    result = b;
	}else if(this.children.isEmpty() && b.children.isEmpty()){
	    // this and b both have reached a leaf node
	    assert(this.precise && b.precise);
	    result = this;
	}else{
	    // Look for common elements
	    assert(this.precise && b.precise);
	    newChildren = new HashMap();
	    iter = this.children.entrySet().iterator();
	    while(iter.hasNext()){
		childEntry = (Map.Entry)iter.next();
		childName = (String)childEntry.getKey();
		child = (DISTreeNode)childEntry.getValue();
		bChild = (DISTreeNode)b.children.get(childName);
		if(bChild != null){
		    newChild = child.intersection(bChild);
		    if(newChild != null){
			newChildren.put(childName, newChild);
		    }
		}
	    }
	    if(!newChildren.isEmpty()){
		result = new DISTreeNode(this.name,
					 this.precise || b.precise,
					 newChildren);
	    }else{
		// "this" and "b" share no common children
		result = null;
	    }
	}
	return(result);
    }

 /** 
 *  Return the intersection of this tree and another but enforce the 
 *  "entails" property for isNode 
 **/ 
    public final DISTreeNode intersectionIS(DISTreeNode isNode)
    {
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode nodeChild = null;
	DISTreeNode unionChild = null;
	DISTreeNode result = null;
	Map.Entry childEntry = null;
	HashMap newChildren = null;

	assert(this.name.equals(isNode.name));
	if(this.children.isEmpty()){
	    // This leaf node belongs in the interest set; copy it
	    if(isNode.children.isEmpty()){
		result = new DISTreeNode(this.name,
					 this.precise || isNode.precise,
					 new HashMap());
	    }else{
		result = this;
	    }
	}else if(isNode.children.isEmpty()){
	    assert(!this.children.isEmpty());
	    if(!isNode.precise){
		// Hit an imprecise leaf of isNode. We lie in the interest set.
		result = this;
	    }else{
		// Hit a precise isNode leaf, but we aren't a leaf. Hence,
		// we are an invalidate for an object not in the interest set.
		result = null;
	    }
	}else{
	    // Look for common elements
	    assert((!this.children.isEmpty()) && (!isNode.children.isEmpty()));
	    assert(this.precise && isNode.precise);
	    newChildren = new HashMap();
	    for(Iterator i = this.children.entrySet().iterator();
		i.hasNext();){
		childEntry = (Map.Entry)i.next();
		childName = (String)childEntry.getKey();
		child = (DISTreeNode)childEntry.getValue();
		nodeChild = (DISTreeNode)isNode.children.get(childName);
		if(nodeChild != null){
		    unionChild = child.intersectionIS(nodeChild);
		    if(unionChild != null){
			newChildren.put(childName, unionChild);
		    }
		}
	    }
	    if(!newChildren.isEmpty()){
		result = new DISTreeNode(this.name, true, newChildren);
	    }
	}
	return(result);
    }

 /** 
 *  Return true if this object is contained in our given tree 
 **/ 
    public final boolean contains(String path)
    {
	return(this.contains(DISTreeNode.makeNode(path)));
    }

 /** 
 *  Return true if this DISTreeNode has exactly one subdirectory 
 **/ 
    public final boolean containsExactlyOneSubdirectory()
    {
	boolean result = true;
	DISTreeNode current = null;
	Map.Entry childEntry = null;
	Iterator iter = null;

	current = this;
	while(result && !current.children.isEmpty()){
	    if(current.children.size() > 1){
		result = false;
	    }else{
		assert(current.children.size() == 1);
		iter = current.children.entrySet().iterator();
		childEntry = (Map.Entry)iter.next();
		current = (DISTreeNode)childEntry.getValue();
		assert(!iter.hasNext());
	    }
	}
	return(result);
    }

 /** 
 *  Return the parent of the last child in the tree 
 **/ 
    public final DISTreeNode cloneUntilLastParent()
    {
	Iterator iter = null;
	DISTreeNode child = null;
	DISTreeNode result = null;
	DISTreeNode childClone = null;
	Map.Entry childEntry = null;
	HashMap newChildren = null;

	if(this.children.isEmpty()){
	    // Return null if we have no children
	    result = null;
	}else{
	    newChildren = new HashMap();
	    iter = this.children.entrySet().iterator();
	    childEntry = (Map.Entry)iter.next();
	    // Ensure that we have exactly one child
	    assert(!iter.hasNext());
	    child = (DISTreeNode)childEntry.getValue();
	    childClone = child.cloneUntilLastParent();
	    if(childClone != null){
		newChildren.put(childClone.name, childClone);
	    }
	    result = new DISTreeNode(this.name, this.precise, newChildren);
	}
	return(result);
    }

 /** 
 *  Return true if node is contained in our given tree 
 **/ 
    public final boolean contains(DISTreeNode node)
    {
	boolean result = false;
	Iterator iter = null;
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode nodeChild = null;
	Map.Entry childEntry = null;

	if(!node.name.equals(this.name)){
	    result = false;
	}else if((!this.precise) || node.children.isEmpty()){
	    result = true;
	}else{
	    iter = node.children.entrySet().iterator();
	    assert(iter.hasNext());
	    assert(node.precise);
	    childEntry = (Map.Entry)iter.next();
	    childName = (String)childEntry.getKey();
	    nodeChild = (DISTreeNode)childEntry.getValue();
	    assert(!iter.hasNext());
	    if(!this.children.containsKey(childName)){
		result = false;
	    }else{
		child = (DISTreeNode)this.children.get(childName);
		assert(child != null);
		result = child.contains(nodeChild);
	    }
	}
	return(result);
    }

 /** 
 *  Return true if this tree is equal to "o" 
 **/ 
    public final boolean equals(Object o)
    {
	DISTreeNode disTree = null;
	boolean result = true;
	Iterator iter = null;
	String childName = null;
	DISTreeNode child = null;
	DISTreeNode disChild = null;
	Map.Entry childEntry = null;
	
	if(o instanceof DISTreeNode){
	    disTree = (DISTreeNode)o;
	    if((this.precise == disTree.precise) &&
	       (this.name.equals(disTree.name))){
		// Iterate through our children to make sure they match
		// those of disTree
		iter = this.children.entrySet().iterator();
		while(result && iter.hasNext()){
		    childEntry = (Map.Entry)iter.next();
		    childName = (String)childEntry.getKey();
		    child = (DISTreeNode)childEntry.getValue();
		    disChild = (DISTreeNode)disTree.children.get(childName);
		    result = (disChild != null) && child.equals(disChild);
		}
		// Iterate through the children of disTree to make
		// sure it doesn't have extra children
		iter = disTree.children.entrySet().iterator();
		while(result && iter.hasNext()){
		    childEntry = (Map.Entry)iter.next();
		    childName = (String)childEntry.getKey();
		    result = this.children.containsKey(childName);
		}
	    }else{
		result = false;
	    }
	}else{
	    result = false;
	}
	return(result);
    }

 /** 
 *  Return a string representation of this node 
 **/ 
    public final String toString()
    {
	return(this.makeString(""));
    }

 /** 
 *  Return a hashCode for this tree 
 **/ 
    public final int hashCode()
    {
	/*
	 * We calculate a hashcode by recursively taking the XOR of hash
	 * codes returned by all entries in the tree
	 */
	int childrenHashCode = 0;
	Iterator iter = null;
	DISTreeNode child = null;
	DISTreeNode disChild = null;
	Map.Entry childEntry = null;

	childrenHashCode = 0;
	iter = this.children.entrySet().iterator();
	while(iter.hasNext()){
	    childEntry = (Map.Entry)iter.next();
	    child = (DISTreeNode)childEntry.getValue();
	    childrenHashCode = childrenHashCode ^ child.hashCode();
	}
	return(this.name.hashCode() ^
	       (new Boolean(this.precise)).hashCode() ^
	       childrenHashCode);
    }

 /** 
 *  Return a string representation of this node 
 **/ 
    private final String makeString(String prefix)
    {
	String str = null;
	String childName = null;
	DISTreeNode childNode = null;

	str = "";
	for(Iterator i = this.children.keySet().iterator(); i.hasNext();){
	    childName = (String)i.next();
	    childNode = (DISTreeNode)this.children.get(childName);
	    str += childNode.makeString(prefix + this.name + "/");
	}
	if(this.children.isEmpty()){
	    str += prefix + this.name;
	    if(!this.precise){
		str += "/*";
	    }
	    str += ":";
	}
	return(str);
    }

    //-----------------------------------------------------------------------
    // Clone this node
    //-----------------------------------------------------------------------
    public final Object clone()
    {
	// NOTE: Because this is an immutable class, we don't have to copy
	// anything
        assert(this instanceof Immutable);
	return(this);
    }

    //-----------------------------------------------------------------------
    // Test this class
    //-----------------------------------------------------------------------
    public static void main(String[] argv)
    {
	DISTreeNode dtn1 = null;
	DISTreeNode dtn2 = null;
	DISTreeNode dtn3 = null;
	DISTreeNode dtn4 = null;
	DISTreeNode dtn5 = null;
	DISTreeNode dtn6 = null;
	DISTreeNode dtn7 = null;
	DISTreeNode dtn8 = null;
	DISTreeNode dtn9 = null;
	DISTreeNode dtn10 = null;
	DISTreeNode dtn11 = null;
	DISTreeNode dtn12 = null;
	DISTreeNode dtn13 = null;
	DISTreeNode dtn14 = null;
	DISTreeNode dtn15 = null;
	DISTreeNode dtn16 = null;
	DISTreeNode dtn17 = null;
	DISTreeNode dtn18 = null;
	DISTreeNode dtn19 = null;
	DISTreeNode dtn20 = null;
	DISTreeNode dtn21 = null;
	DISTreeNode dtn22 = null;
	DISTreeNode dtn23 = null;
	DISTreeNode dtn24 = null;
	DISTreeNode dtn25 = null;
	DISTreeNode dtn26 = null;
	DISTreeNode dtn27 = null;
	DISTreeNode dtn28 = null;
	DISTreeNode dtn29 = null;
	DISTreeNode dtn30 = null;
	DISTreeNode dtn31 = null;
	DISTreeNode dtn32 = null;
	DISTreeNode dtn33 = null;

	Env.verifyAssertEnabled();
	System.err.println("DISTreeNode self test...");

	dtn1 = DISTreeNode.makeNode("/a/b");
	dtn2 = DISTreeNode.makeNode("/c/d");
	assert(dtn1.intersection(dtn2) == null);

	dtn3 = DISTreeNode.makeNode("/a/e/f");
	dtn4 = DISTreeNode.makeNode("/a/*");
	dtn24 = dtn3.intersection(dtn4);
	assert(dtn24.equals(dtn4.intersection(dtn3)));
	assert(dtn24.equals(DISTreeNode.makeNode("/a/e/f")));

	// Test contains
	assert(dtn3.contains("/a/e"));
	assert(!dtn3.contains("/a/f"));
	assert(dtn4.contains("/a/f"));

	// Test cloneChopExcluded

	// Set dtn5 to "/a/h:/f/g:/b/d/e:/b/c/*:"
	dtn5 = DISTreeNode.makeNode("/b/d/e");
	dtn5 = dtn5.union(DISTreeNode.makeNode("/b/c/*"));
	dtn5 = dtn5.union(DISTreeNode.makeNode("/a/h"));
	dtn5 = dtn5.union(DISTreeNode.makeNode("/f/g"));
	// Set dtn6 to "/a/*:/b/c/*:"
	dtn6 = DISTreeNode.makeNode("/b/c/*");
	dtn6 = dtn6.union(DISTreeNode.makeNode("/a/*"));
	// Set dtn25 to the expected result: "/a/h:/f/*:/b/d/*:/b/c/*"
	dtn25 = DISTreeNode.makeNode("/a/h");
	dtn25 = dtn25.union(DISTreeNode.makeNode("/f/*"));
	dtn25 = dtn25.union(DISTreeNode.makeNode("/b/d/*"));
	dtn25 = dtn25.union(DISTreeNode.makeNode("/b/c/*"));
	assert(dtn25.equals(dtn5.cloneChopExcluded(dtn6)));
	// Set dtn7 to "/a:/c/d:/c/e:"
	dtn7 = DISTreeNode.makeNode("/a");
	dtn7 = dtn7.union(DISTreeNode.makeNode("/c/d"));
	dtn7 = dtn7.union(DISTreeNode.makeNode("/c/e"));
	// Set dtn8 to "/a/b/*"
	dtn8 = DISTreeNode.makeNode("/a/b/*");
	// Set dtn26 to the expected result: "/a:/c/*"
	dtn26 = DISTreeNode.makeNode("/a");
	dtn26 = dtn26.union(DISTreeNode.makeNode("/c/*"));
	assert(dtn26.equals(dtn7.cloneChopExcluded(dtn8)));

	// Test overlapsIS

	// Set dtn9 to "/a/c:/b"
	dtn9 = DISTreeNode.makeNode("/a/c");
	dtn9 = dtn9.union(DISTreeNode.makeNode("/b"));
	// Set dtn10 to "/a/d/*"
	dtn10 = DISTreeNode.makeNode("/a/d/*");
	assert(!dtn9.overlapsIS(dtn10));
	// Set dtn11 to "/a/*", dtn12 to "/a/d/*"
	dtn11 = DISTreeNode.makeNode("/a/*");
	dtn12 = DISTreeNode.makeNode("/a/d/*");
	assert(dtn11.overlapsIS(dtn12));
	assert(dtn12.overlapsIS(dtn11));
	assert(dtn7.overlapsIS(dtn8));

	// Test intersection and intersectionIS

	// Set dtn13 to "/a/*:/b/d:/c:/e"
	dtn13 = DISTreeNode.makeNode("/a/*");
	dtn13 = dtn13.union(DISTreeNode.makeNode("/b/d"));
	dtn13 = dtn13.union(DISTreeNode.makeNode("/c"));
	dtn13 = dtn13.union(DISTreeNode.makeNode("/e"));
	// Set dtn14 to "/a:/b:/c"
	dtn14 = DISTreeNode.makeNode("/a");
	dtn14 = dtn14.union(DISTreeNode.makeNode("/b"));
	dtn14 = dtn14.union(DISTreeNode.makeNode("/c"));
	// Set dtn27 to the expected result: "/a:/c:"
	dtn27 = DISTreeNode.makeNode("/a");
	dtn27 = dtn27.union(DISTreeNode.makeNode("/c"));
	assert(dtn27.equals(dtn13.intersection(dtn14)));
	assert(dtn27.equals(dtn14.intersection(dtn13)));
	// Set dtn28 to the expected result: "/a:/b:/c:"
	dtn28 = DISTreeNode.makeNode("/a");
	dtn28 = dtn28.union(DISTreeNode.makeNode("/b"));
	dtn28 = dtn28.union(DISTreeNode.makeNode("/c"));
	assert(dtn28.equals(dtn14.intersectionIS(dtn13)));
	// Set dtn29 to the expected result: "/a:/c"
	dtn29 = DISTreeNode.makeNode("/a");
	dtn29 = dtn29.union(DISTreeNode.makeNode("/c"));
	assert(dtn29.equals(dtn13.intersectionIS(dtn14)));
	// Set dtn15 and dtn16 to "/a/b" and "/c/d" respectively
	dtn15 = DISTreeNode.makeNode("/a/b");
	dtn16 = DISTreeNode.makeNode("/c/d");
	assert(dtn15.intersection(dtn16) == null);
	assert(dtn16.intersection(dtn15) == null);
	// Set dtn17 and dtn18 to "/*" and "/a/*" respectively
	dtn17 = DISTreeNode.makeNode("/*");
	dtn18 = DISTreeNode.makeNode("/a/*");
	// Set dtn30 to the expected result: "/a/*:"
	dtn30 = DISTreeNode.makeNode("/a/*");
	assert(dtn30.equals(dtn17.intersection(dtn18)));

	// Test equals
	dtn19 = DISTreeNode.makeNode("/a/b/c");
	dtn19 = dtn19.union(DISTreeNode.makeNode("/a/d/e"));
	assert(dtn19.equals(dtn19));

	dtn20 = DISTreeNode.makeNode("/a/b/c/x");
	dtn20 = dtn20.union(DISTreeNode.makeNode("/a/d/e"));
	assert(!dtn19.equals(dtn20));
	assert(!dtn20.equals(dtn19));

	dtn21 = DISTreeNode.makeNode("/a/d/e");
	dtn21 = dtn21.union(DISTreeNode.makeNode("/a/b/c"));
	assert(dtn21.equals(dtn19));
	assert(dtn19.equals(dtn21));

	// Test hashCode
	assert(dtn21.hashCode() == dtn19.hashCode());
	assert(dtn21.hashCode() == dtn21.hashCode());
	assert(dtn20.hashCode() == dtn20.hashCode());
	assert(DISTreeNode.makeNode("/a/b").hashCode() ==
	       DISTreeNode.makeNode("/a/b").hashCode());

	// Test containsExactlyOneSubdirectory
	dtn22 = DISTreeNode.makeNode("/a/b");
	assert(dtn22.containsExactlyOneSubdirectory());
	dtn23 = DISTreeNode.makeNode("/a/b");
	dtn23 = dtn23.union(DISTreeNode.makeNode("/c/d"));
	assert(!dtn23.containsExactlyOneSubdirectory());

	// Test cloneUntilLastParent

	// Set dtn31 to "/:"
	dtn31 = DISTreeNode.makeNode("/");
	assert(dtn31.cloneUntilLastParent() == null);

	// Set dtn32 to "/a/b/c:" and dtn33 to the result: "/a/b:"
	dtn32 = DISTreeNode.makeNode("/a/b/c");
	dtn33 = DISTreeNode.makeNode("/a/b");
	assert(dtn33.equals(dtn32.cloneUntilLastParent()));

	// Done
	System.err.println("DISTreeNode self test succeds");
	System.exit(0);
    }
}
