package code;

 /** 
 *  Unit test for DataStoreUnit
 
 **/ 

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class DataStoreUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
  // private Process rmiregistry;
  
  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public DataStoreUnit (final String s) {
    super (s);
  }


  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    /*
    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    Env.dprintln(verbose, "rmiregistry started");
    Thread.sleep(2000);
    */
  }

  protected void tearDown() throws Exception{
    /*
    rmiregistry.destroy();
    */
    super.tearDown();
    
  }



 /** 
 *  doTestReadNoSuchObj
 
 **/ 
  private void 
    doTestReadNoSuchObj(DataStore store, String path){
    ObjId id = new ObjId(path);
    BodyMsg b;
    try{
      b = store.read(id, 0, 1, false, false, true, -1);
      assert(false);
    }
    catch(ObjNotFoundException e){
      // Expected this
      return;
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  doTest read timeout because of invalid
 
 **/ 
  private void 
    doTestReadTimeoutBecauseOfInvalid(DataStore store, String path){
    long timeout = 100;
    ObjId id = new ObjId(path);
    BodyMsg b;
    try{
      b = store.read(id, 0, 1, true, false, true, timeout);
      assert(false);
    }
    catch(ReadTimeoutException e){
      if(vverbose){
	Env.dprintln(vverbose, "---------- read timeout " + e.toString());
      }
      // Expected this
      return;
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  doTest read timeout because of imprecise
 
 **/ 
  private void 
    doTestReadTimeoutBecauseOfImprecise(DataStore store, String path){
    long timeout = 100;
    ObjId id = new ObjId(path);
    BodyMsg b;
    try{
      b = store.read(id, 0, 1, false, true, true, timeout);
      assert(false);
    }
    catch(ReadTimeoutException e){
      if( vverbose ){
	Env.dprintln(vverbose, "---------- read timeout " + e.toString());
      }
      // Expected this
      return;
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  doTestReadInvalidRange
 
 **/ 
  private void 
  doTestReadInvalidRange(DataStore store, String path, long offset){
    ObjId id = new ObjId(path);
    try{
      store.read(id, offset, 1, false, false, true, -1);
      assert(false);
    }
    catch(ReadOfInvalidRangeException e){
      // Expected this
      return;
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  doTestReadEOF
 
 **/ 
  private void 
  doTestReadEOF(DataStore store, String path, long offset){
    ObjId id = new ObjId(path);
    try{
      store.read(id, offset, 1, false, false, true, -1);
      assert(false);
    }
    catch(EOFException e){
      // Expected this
      return;
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  doTestReadBody
 
 **/ 
  private void 
  doTestReadBody(DataStore store, String path, long offset, long len, byte val){
    ObjId id = new ObjId(path);
    BodyMsg bm;
    byte b[];
    int ii;
    try{
      bm = store.read(id, offset, len, false, false, true, -1);
      b = bm.getBody().dangerousGetReferenceToInternalByteArray();
      for(ii = 0; ii < len; ii++){
        assert(b[ii] == val);
      }
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  doTestWriteInval
 
 **/ 
  private void 
  doTestWriteInval(DataStore store, String path, long node, 
		   long localStamp, long offset, long len){
    ObjId id = new ObjId(path);
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    PreciseInv pi = new PreciseInv(oit, stamp);
    VVMap startVVs;

    try{
      store.applyInval(pi);
      store.removeConnectionFromISStatus(SubscriptionSet.makeSubscriptionSet(":/*"),
					 null,
					 pi.getEndVV());
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
  }

  //----------------------------------------------------------------------
  // doTestWriteEmbargoedBound
  //----------------------------------------------------------------------
  private void 
  doTestWriteEmbargoedBound(DataStore store, 
			    String path, long node, long localStamp, 
			    long offset, int len, byte val,
			    long realStamp, boolean embargoed){
    ObjId id = new ObjId(path);
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    AcceptStamp rStamp = new AcceptStamp(realStamp, nodeId);
    int ii;
    byte b[] = new byte[len];
    doTestSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    BoundInval bi = new BoundInval(oit, stamp, ib, Core.DEFAULT_PRIORITY,
                                   rStamp, embargoed);
    VVMap startVVs;

    try{
      store.applyInval(bi);
      store.removeConnectionFromISStatus(SubscriptionSet.makeSubscriptionSet(":/*"),
				   null,
				   bi.getEndVV());
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
  }

  //----------------------------------------------------------------------
  // doTestWriteBound
  //----------------------------------------------------------------------
  private void 
  doTestWriteBound(DataStore store, String path, long node, 
		   long localStamp, long offset, int len, byte val){
    ObjId id = new ObjId(path);
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    int ii;
    byte b[] = new byte[len];
    doTestSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    BoundInval bi = new BoundInval(oit, stamp, ib);
    VVMap startVVs;
    boolean dbg = false;
    try{
      
      store.applyInval(bi);
      store.removeConnectionFromISStatus(SubscriptionSet.makeSubscriptionSet(":/*"),
				   null,
				   bi.getEndVV());
      if(dbg){
	System.out.println("writeBound " + bi + " isPrecise " + store.isPrecise(id)); 
      }
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
  }

  //----------------------------------------------------------------------
  // doTestWriteImprecise
  //----------------------------------------------------------------------
  private void 
  doTestWriteImprecise(DataStore store, String path, 
		       long node, long localStart, long localEnd){
    
    HierInvalTarget hit = HierInvalTarget.makeHierInvalTarget(path);
    PreciseSet ps = null;
    boolean isPrecise = true;
    NodeId nodeId = new NodeId(node);
    AcceptStamp start = new AcceptStamp(localStart, nodeId);
    AcceptStamp end = new AcceptStamp(localEnd, nodeId);
    AcceptStamp startStamps[] = new AcceptStamp[1];
    startStamps[0] = start;
    AcceptStamp endStamps[] = new AcceptStamp[1];
    endStamps[0] = end;
    
    AcceptVV startVV = new AcceptVV(startStamps);
    AcceptVV endVV = new AcceptVV(endStamps);
    VVMap startVVs;
    
    ImpreciseInv ii = new ImpreciseInv(hit,
				       startVV,
				       endVV);
    
    try{
      
      store.applyInval(ii);
      //
      // Verify postcondition that interest set is imprecise
      //
      assert !store.isPrecise(SubscriptionSet.makeSubscriptionSet(path));
      
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
  }
 


  //----------------------------------------------------------------------
  // doTestUpdateBody
  //----------------------------------------------------------------------
  private void 
  doTestUpdateBody(DataStore store, String path, long node, 
		   long localStamp, long offset, int len, byte val){
    ObjId id = new ObjId(path);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    int ii;
    byte b[] = new byte[len];
    doTestSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    BodyMsg bm = new BodyMsg(id, offset, len, stamp, ib, false);

    try{
      store.applyBody(bm);
    }
    catch(IOException e){
      assert(false);
    }
  }

  //----------------------------------------------------------------------
  // doTestSetBytes
  //----------------------------------------------------------------------
  private void
  doTestSetBytes(byte b[], int len, byte value){
    int ii;
    for(ii = 0; ii < len; ii++){
      b[ii] = value;
    }
  }

  //----------------------------------------------------------------------
  // doTestUpdateBodyUnblocksInvalidRange -- test that the read of this invalid
  //    range blocks until a body write releases it
  //----------------------------------------------------------------------
  private void 
  doTestUpdateBodyUnblocksInvalidRange(DataStore store, String path, 
				       long node, long localStamp, 
				       long offset, int len, byte val){
    ObjId id = new ObjId(path);
    int ii;
    TestReadResultMailbox mb[] = new TestReadResultMailbox[len];
    // for a range of startOffsets from offset to offset+length
    //   Create a TestReadResultMailbox
    //   Fork off a read worker thread to read chunk
    for(ii = 0; ii < len; ii++){
      mb[ii] = new TestReadResultMailbox();
      ObjInvalTarget oit = new ObjInvalTarget(id, offset + ii, len - ii);
      TestReadWorker trw = new TestReadWorker(oit, mb[ii], store);
      trw.start();
    }
    // sleep
    try{
      Thread.sleep(1000);
    }
    catch(InterruptedException e){
      // we were just pausing for effect, anyhow
    }
    // for each test read result mailbox
    //   mailbox[ii].verifyNoResultYet
    for(ii = 0; ii < len; ii++){
      mb[ii].verifyNoResultYet();
    }
    // write offset..[offset+length/2]
    doTestUpdateBody(store, path, node, localStamp, offset, len/2, val);

    // for each mailbox that starts before end of write
    //   mailbox[ii].waitForAndConfirmReadResult
    for(ii = 0; ii < len/2; ii++){
      mb[ii].waitForAndConfirmReadResult(len/2 - ii, val);
    }


    // 
    // for each later mailbox
    //   mailbox[ii].verifyNoResultYet
    for(ii = len/2; ii < len; ii++){
      mb[ii].verifyNoResultYet();
    }

    // write the rest of the offsets
    doTestWriteBound(store, path, node, localStamp, offset, len, val);

    // for each mailbox
    //   mailbox[ii].waitForAndConfirmReadResult
    for(ii = len/2; ii < len; ii++){
      mb[ii].waitForAndConfirmReadResult(len - ii, val);
    }

  }

  //----------------------------------------------------------------------
  // doTestDebargoUnblocksEmbargoedRange -- test that the read of this embargoed
  //    range blocks until a debargo(debargomsg or debargoed inv) releases it
  //----------------------------------------------------------------------
  private void 
  doTestDebargoUnblocksEmbargoedRange(DataStore store){
    
    ObjId id = new ObjId("/test1/bar");
    long offset = 0;
    int maxLen = 999;
    long stamp = 1;
    long realStamp = 10000;
    NodeId nodeId = new NodeId(38);
    byte VAL1 = 25;
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, maxLen);
    AcceptStamp accept = new AcceptStamp(stamp, nodeId);
    AcceptStamp realAccept = new AcceptStamp(realStamp, nodeId);
    PreciseInv pi = new PreciseInv(oit, accept, realAccept, true);
    //test apply embargoed invalidate
    doTestWriteEmbargoedBound(store, 
			      "/test1/bar", 
			      38,
			      stamp,
			      offset,
			      maxLen,
			      VAL1, 
			      realStamp,
			      true);// expect ReadOfEmbargoedWriteException
    TestReadResultMailbox mb = new TestReadResultMailbox(); 
    TestReadWorker trw = new TestReadWorker(oit, mb, store);
    trw.start();
    // sleep
    try{
      Thread.sleep(1000);
    }
    catch(InterruptedException e){
      // we were just pausing for effect, anyhow
    }
    mb.verifyNoResultYet();
    
    try{
      store.applyDebargo(new DebargoMsg(oit, accept));
    }catch(IOException e){
      e.printStackTrace();
      assert false;
    }
    mb.waitForAndConfirmReadResult(maxLen, VAL1);
  }

  //----------------------------------------------------------------------
  // doTestNewInvUnblocksTemporalErrorRead -- test that the read of this 
  // file blocks because of Temporal Error constrains until a newer
  // invalidate releases it
  //----------------------------------------------------------------------
  private void 
  doTestNewInvUnblocksTemporalErrorRead(DataStore store){
    
    if(verbose){
      System.out.print("Test temporal error blocked read and unblock...");
    }
    ObjId id = new ObjId("/test1/bar");
    long offset = 0;
    int maxLen = 999;
    long stamp = 1;
    long realStamp = System.currentTimeMillis();
    NodeId nodeId = new NodeId(38);
    byte VAL1 = 25;
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, maxLen);
    AcceptStamp accept = new AcceptStamp(stamp, nodeId);
    AcceptStamp realAccept = new AcceptStamp(realStamp, nodeId);
    PreciseInv pi = new PreciseInv(oit, accept, realAccept, true);
    
    //test apply old writes
    doTestWriteEmbargoedBound(store, 
			      "/test1/bar", 
                            38,
                            stamp,
                            offset,
                            maxLen,
                            VAL1, 
                            realStamp,
                            false);
    if(verbose){
      System.out.println("realVV:" + store.getRealVV());
    }
    assert store.getRealVV().getMinTimeStamp()==realStamp;

    // sleep to make the read violate the temporal error
    try{
      Thread.sleep(1000);
    }
    catch(InterruptedException e){
      // we were just pausing for effect, anyhow
    }
    TestReadResultMailbox mb = new TestReadResultMailbox(); 
    TestTEReadWorker trw = new TestTEReadWorker(oit, mb, store, 100);
    trw.start();
    if(verbose){
      System.out.println("TestTEReadWorker starts");
    }
    mb.verifyNoResultYet();
    
    long newWriteTime = System.currentTimeMillis();
    //test apply old writes
    doTestWriteEmbargoedBound(store,"/test1/bar", 
                            38,
                            stamp,
                            offset,
                            maxLen,
                            VAL1, 
                            newWriteTime,
                            false);
    if(verbose){
      System.out.println("new realVV:" + store.getRealVV());
    }
    assert store.getRealVV().getMinTimeStamp()==newWriteTime;
    mb.waitForAndConfirmReadResult(maxLen, VAL1);
    
  }

  //----------------------------------------------------------------------
  // doTestInvalsMakeImprecisePrecise -- test that the read of this invalid
  //    range blocks until a series of invals releases it
  //----------------------------------------------------------------------
  private void 
    doTestInvalsMakeImprecisePrecise(DataStore store,
				     String path, long node, long localStart,
				     long localEnd, 
				     long offset, int len, byte val){


    ObjId id = new ObjId(path);
    int ii;
    long localStamp;
    if(verbose){
      System.out.println("doTestInvalsMakeImprecisePrecise start...");
    }
    TestReadResultMailbox mb[] = new TestReadResultMailbox[len];
    // for a range of startOffsets from offset to offset+length
    //   Create a TestReadResultMailbox
    //   Fork off a read worker thread to read chunk
    for(ii = 0; ii < len; ii++){
      mb[ii] = new TestReadResultMailbox();
      ObjInvalTarget oit = new ObjInvalTarget(id, offset + ii, len - ii);
      TestReadWorker trw = new TestReadWorker(oit, mb[ii], store);
      trw.start();
    }
    // sleep
    try{
      Thread.sleep(1000);
    }
    catch(InterruptedException e){
      // we were just pausing for effect, anyhow
    }
    // for each test read result mailbox
    //   mailbox[ii].verifyNoResultYet
    for(ii = 0; ii < len; ii++){
      mb[ii].verifyNoResultYet();
    }
    // do writes from localStart to localEnd-1
    for(localStamp = localStart; localStamp < localEnd; localStamp++){
      doTestWriteBound(store, path, node, localStamp, offset, len, val);
    }
    
    // 
    // for each mailbox
    //   mailbox[ii].verifyNoResultYet
    for(ii = 0; ii < len; ii++){
      mb[ii].verifyNoResultYet();
    }
    
    // write the rest of the timestamps (some of these are repeats; that's OK.
    for(localStamp = localStart; localStamp <= localEnd; localStamp++){
      doTestWriteBound(store, path, node, localStamp, offset, len, val);
    }
    
    if(verbose){
      System.out.println("all writes done. should be precise now");
    }
    
    // for each mailbox
    //   mailbox[ii].waitForAndConfirmReadResult
    for(ii = 0; ii < len; ii++){
      if(verbose){
	System.out.println("waitForAndConfirmReadResult("+ (len-ii) + ")");
      }
      mb[ii].waitForAndConfirmReadResult(len - ii, val);
      if(verbose){
	System.out.println("waitForAndConfirmReadResult("+ (len-ii) + ") return");
      }
    }
    
    if(verbose){
      System.out.println("doTestInvalsMakeImprecisePrecise done.");
    }
  }

  

  private void
  deleteFile(String fileName){
     if(fileName != null){
      try{
        File f = new File(fileName);
        if (f.exists()){
          f.delete();
        }
      }
      catch(Exception e){
        e.printStackTrace();
        return;
      }
    }
  }

  //---------------------------------------------------------------------------
  // test liesInPreciseIS
  //---------------------------------------------------------------------------
  private void
  doTestLiesInPreciseIS(DataStore store){
    if(verbose){
      System.out.println(" doTestLiesInPreciseIS start ... ");
    }
    AcceptStamp[] as1 = null;
    AcceptStamp[] as2 = null;
    AcceptVV vv1 = null;
    AcceptVV vv2 = null;

    PreciseSet ps = null;
    PreciseSet newPS = null;
    PreciseSet newPS2 = null;
    PreciseSet newChildPS = null;
    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(1, new NodeId(200));
    as1[1] = new AcceptStamp(1, new NodeId(300));

    as2 = new AcceptStamp[2];
    as2[0] = new AcceptStamp(2, new NodeId(200));
    as2[1] = new AcceptStamp(2, new NodeId(300));

    vv1 = new AcceptVV(as1);
    vv2 = new AcceptVV(as2);

    ps = new PreciseSet("", "", vv1, vv2);
    newPS = new PreciseSet("apple", "/apple", vv1, vv1);
    newPS2 = new PreciseSet("cherry", "/cherry", vv2, vv2);
    newChildPS = new PreciseSet("banana", "/apple/banana", vv2, vv1);
    
    ps.children.put("apple", newPS);
    ps.children.put("cherry", newPS2);
    newPS.children.put("banana", newChildPS);
      
    assert(ps.getChild("apple") == newPS);
    assert(ps.getChild("cherry") == newPS2);
    assert(ps.getChild("apple").getChild("banana") == newChildPS);

    AllPreciseSets aps = new AllPreciseSets(ps);
    ISStatus newISStatus = new ISStatus(aps, null);
    store.setISStatus(newISStatus);
    store.advanceCVV(vv2);
    assert store.liesInPreciseIS(new ObjId("/dog/dog"));
    assert !store.liesInPreciseIS(new ObjId(""));
    assert !store.liesInPreciseIS(new ObjId("/apple/dog"));
    assert store.liesInPreciseIS(new ObjId("/apple/banana"));
    assert store.liesInPreciseIS(new ObjId("/cherry"));
    assert store.liesInPreciseIS(new ObjId("/cherry/dog/dog/dog"));
    assert !store.liesInPreciseIS(new ObjId("/apple/banana/d"));
    if(verbose){
      System.out.println(" doTestLiesInPreciseIS done. ");
    }
    
  }  

  //----------------------------------------------------------------------
  // doTestSendCheckpoint()
  //----------------------------------------------------------------------
  private void
  doTestSendCheckpoint(DataStore store){
    if(verbose){
      System.out.println(" doTestSendCheckpoint start ... ");
    }
    
    //before entering the store should be cleaned up

    //test 1 -simple case prevVV is not equal to cvv
    AcceptStamp[] as0 = null;
    AcceptStamp[] as1 = null;
    AcceptStamp[] as2 = null;
    AcceptStamp[] as3 = null;
    AcceptVV vv0 = null;
    AcceptVV vv1 = null;
    AcceptVV vv2 = null;
    AcceptVV vv3 = null;

    PreciseSet ps = null;
    PreciseSet newPS = null;
    PreciseSet newPS2 = null;
    PreciseSet newChildPS = null;
    PreciseSet newChildPS2 = null;

    as0 = new AcceptStamp[2];
    as0[0] = new AcceptStamp(0, new NodeId(200));
    as0[1] = new AcceptStamp(0, new NodeId(300));

    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(1, new NodeId(200));
    as1[1] = new AcceptStamp(1, new NodeId(300));

    as2 = new AcceptStamp[2];
    as2[0] = new AcceptStamp(2, new NodeId(200));
    as2[1] = new AcceptStamp(2, new NodeId(300));

    as3 = new AcceptStamp[2];
    as3[0] = new AcceptStamp(3, new NodeId(200));
    as3[1] = new AcceptStamp(3, new NodeId(300));

    vv0 = new AcceptVV(as0);
    vv1 = new AcceptVV(as1);
    vv2 = new AcceptVV(as2);
    vv3 = new AcceptVV(as3);
    

    ps = new PreciseSet("", "", vv1, vv3);
    newPS = new PreciseSet("apple", "/apple", vv1, vv1);
    newPS2 = new PreciseSet("cherry", "/cherry", vv3, vv3);
    newChildPS = new PreciseSet("banana", "/apple/banana", vv3, vv1);
    newChildPS2 = new PreciseSet("dog", "/apple/dog", vv3, vv3);
    

    ps.children.put("apple", newPS);
    ps.children.put("cherry", newPS2);
    newPS.children.put("banana", newChildPS);
    newPS.children.put("dog", newChildPS2);  
    assert(ps.getChild("apple") == newPS);
    assert(ps.getChild("cherry") == newPS2);
    assert(ps.getChild("apple").getChild("banana") == newChildPS);
    assert(ps.getChild("apple").getChild("dog") == newChildPS2);
    AllPreciseSets aps = new AllPreciseSets(ps);
    
    ISStatus newISStatus = new ISStatus(aps, null); 
    store.setISStatus(newISStatus);
    store.advanceCVV(vv3);

    String outputFileName = "cpOutput.tmp";
    FileOutputStream fos = null;
    TaggedOutputStream tos = null;
    try{
      deleteFile(outputFileName);
      fos = new FileOutputStream(outputFileName);
      tos = new TaggedOutputStream(fos);
    }catch(Exception e){
      System.err.println("outputstream  error: "
                         + e.toString());
      e.printStackTrace();
      return;
    }
    
    SubscriptionSet ss1 = null;
    SubscriptionSet ss2 = null;
    
    Vector vec1 = null;
    Vector vec2 = null;

    ss1 = SubscriptionSet.makeSubscriptionSet(":/*");
    try{
      assert !store.sendCheckpoint(tos, 
				   ss1,
				   vv0,
				   false,
				   vv1);
      

      assert store.sendCheckpoint(tos,
                            ss1,
                            vv0,
                            false,
                            vv3);
      tos.close();
      fos.close();
    }catch(Exception eee){
      eee.printStackTrace();
      assert false;
    }
    int index = 0;
    vec2 = new Vector();
    vec2.add(new LPVVRecord("", vv1, null));
    vec2.add(new LPVVRecord("", vv1, vv1));

    vec1 = new Vector();
    vec1.add(index, new CPStartMsg(ss1, vv0, store.getMaxCurrentVV(), vec2));
    
    index++;
    vec1.add(index, new Long(RandomAccessState.RAS_SHIP_DONE));
    checkFile(outputFileName, vec1);
    
    BodyMsg bm = null;
    // /cherry valid and precise with stamp 3
    bm = fakeApplyBound(store, "/cherry", 200, 3, 0, 100, (byte)35);
    
    ///apple/banana, precise and invalid with stamp 3
    fakeApplyPI(store, "/apple/banana", 300, 3, 0, 100);
    
    // /apple/dog precise valid with stamp 1
    fakeApplyPI(store, "/apple/dog", 300, 1, 0, 100);
    
    // /cherry/water precise valid with stamp 3
    fakeApplyBound(store, "/cherry/water", 300, 3, 0, 100, (byte)36);
    
    try{
      deleteFile(outputFileName);
      fos = new FileOutputStream(outputFileName);
      tos = new TaggedOutputStream(fos);
    }catch(Exception e){
      System.err.println("outputstream  error: "
                         + e.toString());
      e.printStackTrace();
      return;
    }
    
    ss1 = SubscriptionSet.makeSubscriptionSet("/apple/*:/cherry");
    try{
      assert store.sendCheckpoint(tos,
                            ss1,
                            vv2,
                            true,
                            vv3);
      tos.close();
      fos.close();
    }catch(Exception eee){
      eee.printStackTrace();
      assert false;
    }
    index = 0;
    vec2 = new Vector();
    //vec2.add(new LPVVRecord("/apple", vv1, vv1));
    vec2.add(new LPVVRecord("/cherry", vv3, null));

    vec1 = new Vector();
    vec1.add(index, new CPStartMsg(ss1, vv2, store.getMaxCurrentVV(), vec2));
    
    index++;
    vec1.add(index, new PerRangeStateForSend(new ObjId("/apple/banana"),
                                             0,
                                             100,
                                             new AcceptStamp(3, new NodeId(300))));
    index++;
    vec1.add(index, bm);
    index++;
    vec1.add(index, new Long(RandomAccessState.RAS_SHIP_DONE));
    checkFile(outputFileName, vec1);
    if(verbose){
      System.out.println(" doTestSendCheckpoint done. ");
    }
    
  }

  //----------------------------------------------------------------------
  // doTestApplyCheckpoint() -- clean datastore and followed by the doTestsendCP
  //----------------------------------------------------------------------
  private void doTestApplyCheckpoint(DataStore store, String path, String configPath,
				     NodeId myId){
    
    //before entering the store should be cleaned up
    // Clean things up from prior run
    if(verbose){
      System.out.println(" doTestApplyCheckpoint start ... ");
    }
    
    //store.testDestroyAllStateCleanup(path, configPath);
    //store = null;

    AcceptStamp[] as1 = null;
    AcceptStamp[] as2 = null;
    AcceptStamp[] as3 = null;
    AcceptStamp[] as0 = null;

    AcceptVV vv1 = null;
    AcceptVV vv2 = null;
    AcceptVV vv3 = null;
    AcceptVV vv0 = null;

    PreciseSet ps = null;
    PreciseSet newPS = null;
    PreciseSet newPS2 = null;
    PreciseSet newChildPS = null;
    PreciseSet newChildPS2 = null;
    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(1, new NodeId(200));
    as1[1] = new AcceptStamp(1, new NodeId(300));

    as2 = new AcceptStamp[2];
    as2[0] = new AcceptStamp(2, new NodeId(200));
    as2[1] = new AcceptStamp(2, new NodeId(300));

    as3 = new AcceptStamp[2];
    as3[0] = new AcceptStamp(3, new NodeId(200));
    as3[1] = new AcceptStamp(3, new NodeId(300));

    as0 = new AcceptStamp[2];
    as0[0] = new AcceptStamp(0, new NodeId(200));
    as0[1] = new AcceptStamp(0, new NodeId(300));

    vv1 = new AcceptVV(as1);
    vv2 = new AcceptVV(as2);
    vv3 = new AcceptVV(as3);
    vv0 = new AcceptVV(as0);

    ps = new PreciseSet("", "", vv0, vv0);
    newPS = new PreciseSet("apple", "/apple", vv0, vv0);
    newPS2 = new PreciseSet("cherry", "/cherry", vv0, vv0);
    newChildPS = new PreciseSet("banana", "/apple/banana", vv0, vv0);
    newChildPS2 = new PreciseSet("dog", "/apple/dog", vv0, vv0);
    

    ps.children.put("apple", newPS);
    ps.children.put("cherry", newPS2);
    newPS.children.put("banana", newChildPS);
    newPS.children.put("dog", newChildPS2);  
    assert(ps.getChild("apple") == newPS);
    assert(ps.getChild("cherry") == newPS2);
    assert(ps.getChild("apple").getChild("banana") == newChildPS);
    assert(ps.getChild("apple").getChild("dog") == newChildPS2);
    AllPreciseSets aps = new AllPreciseSets(ps);
    ISStatus newISStatus = new ISStatus(aps, null);
    store.setISStatus(newISStatus);
    store.advanceCVV(vv3);

    FileInputStream fis = null;
    TaggedInputStream tis = null;
    Object nextInFile = null;
    
    try{
      fis = new FileInputStream("cpOutput.tmp");
      tis = new TaggedInputStream(fis);
    }catch(Exception e){
      assert false;
    }
    SubscriptionSet ss = null;
    try{
      nextInFile = tis.readTaggedObject();
      assert nextInFile instanceof CPStartMsg;
      
    }catch(Exception ee){
      ee.printStackTrace();
      assert false;
    }
    try{
      store.applyCheckpoint(tis, (CPStartMsg)nextInFile);
    }catch(Exception e3){
      e3.printStackTrace();
      assert false;
    }

    assert ps.getMyLpVV().equals(vv0);
    assert ps.getOtherChildrenLpVV().equals(vv0);
    assert newPS.getMyLpVV().equals(vv0):"newPS.getMyLpVV()=" + newPS.getMyLpVV();
    assert newPS.getOtherChildrenLpVV().equals(vv0);
    assert newChildPS.getMyLpVV().equals(vv0);
    assert newChildPS.getOtherChildrenLpVV().equals(vv0);
    assert newChildPS2.getMyLpVV().equals(vv0);
    assert newChildPS2.getOtherChildrenLpVV().equals(vv0);
    assert newPS2.getMyLpVV().equals(vv3);
    //System.out.println(newPS2.getOtherChildrenLpVV());
    assert newPS2.getOtherChildrenLpVV().equals(vv0);
    doTestReadNoSuchObj(store, "/apple/dog");
    doTestReadBody(store, "/cherry", 0, 100, (byte)35);
    assert !store.liesInPreciseIS(new ObjId("/apple/banana"));
    doTestReadInvalidRange(store, "/apple/banana", 0);
    
    if(verbose){
      System.out.println(" doTestApplyCheckpoint start done. ");
    }
  }

  //----------------------------------------------------------------------
  // applyInv to datastore but do not update ISStatus
  //----------------------------------------------------------------------
  private BodyMsg 
    fakeApplyBound(DataStore store, String path, long node, long localStamp, 
		 long offset, int len, byte val){
    ObjId id = new ObjId(path);
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    int ii;
    byte b[] = new byte[len];
    doTestSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    BoundInval bi = new BoundInval(oit, stamp, ib);
    BodyMsg bm = new BodyMsg(id, 
                             offset, len, 
                             stamp,
                             ib,
                             false);
    VVMap startVVs;

    try{
      
      store.applyInval(bi);
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
    return bm;
  }

  //----------------------------------------------------------------------
  // apply precise Inval to datastore but do nothing to ISStatus
  //----------------------------------------------------------------------
  private void 
    fakeApplyPI(DataStore store, String path, long node, 
		long localStamp, long offset, long len){
    ObjId id = new ObjId(path);
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    NodeId nodeId = new NodeId(node);
    AcceptStamp stamp = new AcceptStamp(localStamp, nodeId);
    PreciseInv pi = new PreciseInv(oit, stamp);
    VVMap startVVs;

    try{
      
      store.applyInval(pi);
      //removeConnectionFromISStatus(SubscriptionSet.makeSubscriptionSet(":/*"),
      //                             null
      //                             pi.getEndVV());
    }
    catch(CausalOrderException e){
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
  }


  //------------------------------------------------------------------------
  // check the result of a file with the object vector -- only for Checkpoint
  //------------------------------------------------------------------------
  private void checkFile(String fileName, Vector result){
    FileInputStream fis = null;
    TaggedInputStream tis = null;
    
    
    try{
      fis = new FileInputStream(fileName);
      tis = new TaggedInputStream(fis);
      
    }catch(Exception e){
      assert false;
    }
    
    Object nextInFile = null;
    Object nextInVec = null;
    for(Enumeration e = result.elements(); e.hasMoreElements();){
      nextInVec = e.nextElement();
      try{
        nextInFile = tis.readTaggedObject();
      }catch(Exception ee){
        ee.printStackTrace();
        assert false;
      }
      
      if(nextInVec instanceof Vector){
        Vector vec = (Vector)(nextInVec);
        assert nextInFile instanceof Vector;
        Vector vecInFile = (Vector)nextInFile;
        assert vec.size() == vecInFile.size():"vec.size()=" + vec.size() 
	  + "vecInFile.size()=" + vecInFile.size();
        for(Enumeration f = vec.elements(); f.hasMoreElements();){
          vecInFile.contains(f.nextElement());
        }
      }else{
        assert nextInVec.equals(nextInFile): "nextInVec=" + nextInVec
	  + " nextInFile=" + nextInFile;
      } 
    }
    try{
      tis.close();
      fis.close();
    }catch(Exception e){
      assert false;
    }
  }


  //----------------------------------------------------------------------
  // Check that everything survives shutdown/restart.
  //----------------------------------------------------------------------
  static void doTestRecovery(String path, 
			     String configPath, NodeId myId){
    if(verbose){
      System.out.println(" doTestRecovery starts ...");
    }
    
    final int MAX_WRITE = 100;

    //
    // Initialize db
    //
    DataStore store;
    store = new DataStore(path, myId);
    store.testDestroyAllStateCleanup(path, configPath);
    store = null;
    store = new DataStore(path, myId);

    //
    // Write stuff
    //
    int ii;
    ObjId id;
    for(ii = 0; ii < MAX_WRITE; ii++){
      id = new ObjId("/testRecovery/" + ii);
      ObjInvalTarget oit = new ObjInvalTarget(id, 0, 1);
      AcceptStamp stamp = new AcceptStamp(ii, myId);
      PreciseInv pi = new PreciseInv(oit, stamp);
      try{
        store.applyInval(pi);
      }
      catch(IOException ieo){
        ieo.printStackTrace();
        assert(false);
      }
      catch(CausalOrderException ieo){
        ieo.printStackTrace();
        assert(false);
      }
      //
      // We've cheated and written a bunch of stuff w/o
      // updating lpVV via connection state...
      //
      assert(!store.isPrecise(id));
    }
    id = new ObjId("/foo");
    assert(!store.isPrecise(id));

    //
    // Set up connections for ISStatus updates. Note that 
    // we update lpVV long after applying the writes...
    // Hope that doesn't break anything... (otherwise 
    // we will have to update the connection state
    // as we do each write...how tedious...
    //
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/testRecovery/*");
    AcceptStamp as[] = new AcceptStamp[1];
    as[0] = new AcceptStamp(MAX_WRITE-1, myId);
    AcceptVV vv = new AcceptVV(as);
    IncommingConnection ic = new IncommingConnection(StreamId.makeNewStreamId(),
                                                     ss,
                                                     vv);
    store.addConnectionToISStatus(ss, ic);
    //
    // Now that connection status updated, everything in /testRecovery/*
    // should be precise...
    //
    for(ii = 0; ii < MAX_WRITE; ii++){
      id = new ObjId("/testRecovery/" + ii);
      assert(store.isPrecise(id));
    }
    id = new ObjId("/foo");
    assert(!store.isPrecise(id));


    store.close();

    //
    // Re-open database. Everythign should be as we left it:
    // a bunch of invalid but precise objects in /testRecovery/*.
    // Everything else imprecise. VV is what we stored.
    //
    store = new DataStore(path, myId);
    for(ii = 0; ii < MAX_WRITE; ii++){
      id = new ObjId("/testRecovery/" + ii);
      assert(store.isPrecise(id));
      try{
        boolean valid = store.isValid(id, 0, 1, true);
        assert(!valid);
      }
      catch(ObjNotFoundException onfe){
        assert(false); // Should be there!
      }
      catch(Exception e){
        assert(false); // Something broken unexpectedly
      }
    }
    id = new ObjId("/foo");
    assert(!store.isPrecise(id));
    
    VV vv2 = store.getCPVV();
    assert(vv.includes(vv2));
    assert(vv2.includes(vv));
    if(verbose){
      System.out.println(" doTestRecovery done.");
    }
    
  }


  //----------------------------------------------------------------------
  // test basic functions
  //----------------------------------------------------------------------
  public void 
  testSimple() {
    if(verbose){
      System.out.println(" testSimple starts ...");
    }
    String path = "test" + File.separatorChar + "DataStoreTest.db";
    String configPath = "test" + File.separatorChar + "DataStoreTest.config";
    byte VAL1 = 25;
    byte VAL2 = 35;
    DataStore store = null;
    
    long nodeIdNum = 42;
    NodeId myId = new NodeId(nodeIdNum);
    UpdateBuffer updateBuffer = null;

    
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(42), "localhost", 5000, 5005, 5003, 5004, 5001, path, 
                            ":/*", -1, "42", 5002, 5006, 0, 1000000, 1000000000, 10000000); 
    // Clean things up from prior run
    store = new DataStore(path, myId);
    store.testDestroyAllStateCleanup(path, configPath);
    store = null;
    try{
      store = new DataStore(path, myId);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
                                      UpdateBuffer.FAIL_ON_FULL);
      InvalidateBuffer invalBuffer = new InvalidateBuffer(store,
                                                          updateBuffer);
      new InvalidateBufferWorkerThread(invalBuffer,
                                       store,
                                       updateBuffer);
      store.setInvalBuffer(invalBuffer);

      doTestLiesInPreciseIS(store);
      
      doTestNewInvUnblocksTemporalErrorRead(store);
      
      doTestDebargoUnblocksEmbargoedRange(store);
      
      doTestReadNoSuchObj(store, "/foo/bar");
      doTestWriteInval(store, "/foo/bar", 0, 0, 0, 100);
      doTestReadInvalidRange(store, "/foo/bar", 0);
      doTestReadInvalidRange(store, "/foo/bar", 25);
      doTestReadTimeoutBecauseOfInvalid(store, "/foo/bar");//because of invalid
      doTestReadEOF(store, "/foo/bar", 200);

      doTestUpdateBodyUnblocksInvalidRange(store, "/foo/bar", 0, 0, 0, 100, VAL1);


      doTestWriteBound(store, "/foo/bar", 1, 0, 0, 100, VAL2);
      doTestReadBody(store, "/foo/bar", 0, 100, VAL2);
      doTestWriteImprecise(store, "/baz/*", 1, 1, 2);
      doTestReadTimeoutBecauseOfImprecise(store, "/baz/bing");
      
      doTestInvalsMakeImprecisePrecise(store, "/baz/bing", 1L, 0L, 2L, 0L, (int)100, VAL1);
      

      doTestWriteImprecise(store, "/biz/*", 4, 1, 2);
      doTestWriteImprecise(store, "/biz/*", 5, 1, 2);
      doTestWriteInval(store, "/biz/bing", 4, 0, 0, 100);
      doTestWriteInval(store, "/biz/bang", 4, 1, 0, 100);
      doTestWriteInval(store, "/biz/bong", 4, 2, 0, 100);
      
      doTestInvalsMakeImprecisePrecise(store, "/baz/bing", 5L, 0L, 2L, 0L, (int)100, VAL2);
      
      store.testDestroyAllStateCleanup(path, configPath);
      
      store = null;
      
      store = new DataStore(path, myId);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
                                      UpdateBuffer.FAIL_ON_FULL);
      invalBuffer = new InvalidateBuffer(store, updateBuffer);
      new InvalidateBufferWorkerThread(invalBuffer, store, updateBuffer);
      store.setInvalBuffer(invalBuffer);
      
      doTestSendCheckpoint(store);

      
      store.testDestroyAllStateCleanup(path, configPath);
      store = null;
      
      store = new DataStore(path, myId);
      updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
                                      UpdateBuffer.FAIL_ON_FULL);
      invalBuffer = new InvalidateBuffer(store, updateBuffer);
      new InvalidateBufferWorkerThread(invalBuffer, store, updateBuffer);
      store.setInvalBuffer(invalBuffer);
      doTestApplyCheckpoint(store, path, configPath, myId);


      
      doTestRecovery(path, configPath, myId);
      
      
      System.out.println("DataStoreUnit selftest SUCCEEDS");
    }
    finally{
      if(store != null){
        store.testDestroyAllStateCleanup(path, configPath);
      }
      store = null;
      
    }
  }

  private static void makePractiConfig(){
    Config.createEmptyConfig();
    long NODE_0_ID = 0;
    long NODE_1_ID = 1;
    long NODE_2_ID = 2;
    long NODE_3_ID = 3;
    long NODE_4_ID = 4;
    String NODE_0_IP = "localhost";
    String NODE_1_IP = "localhost";
    String NODE_2_IP = "localhost";
    String NODE_3_IP = "localhost";
    String NODE_4_IP = "localhost";
    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
                            NODE_0_IP,
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "local-" + 
			    NODE_0_ID + ".db",
                            "/*",
                            -1L,
                            NODE_0_IP,
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(NODE_1_ID),
			   NODE_1_IP,
			   9888,
			   9889,
			   9891,
			   9892,
                            9890,
                           "test" + File.separatorChar + "local-" + 
			   NODE_1_ID+".db",
			   "/*",
			   -1L,
			   NODE_1_IP,
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
   Config.addOneNodeConfig(new NodeId(NODE_2_ID),
			   NODE_2_IP,
			   9888,
			   9889,
			   9891,
			   9892,
                            9890,
                           "test" + File.separatorChar + "local-" + 
			   NODE_2_ID+".db",
			   "/*",
			   -1L,
			   NODE_2_IP,
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);

   Config.addOneNodeConfig(new NodeId(NODE_3_ID),
			   NODE_3_IP,
			   9888,
			   9889,
			   9891,
			   9892,
                            9890,
                           "test" + File.separatorChar + "local-" + 
			   NODE_3_ID+".db",
			   "/*",
			   -1L,
			   NODE_3_IP,
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
    
   
   
   
   Config.addOneNodeConfig(new NodeId(NODE_4_ID),
			   NODE_4_IP,
			   9888,
			   9889,
			   9891,
			   9892,
                            9890,
                           "test" + File.separatorChar + "local-" + 
			   NODE_4_ID+".db",
			   "/*",
			   -1L,
			   NODE_4_IP,
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
   //Config.writeToFile(configPath);
  }

  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(DataStoreUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java DataStoreUnit foo" runs
   *   DataStoreUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "DataStoreUnit";
    System.err.print(name + " self test begins...");
    Env.verifyAssertEnabled();
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new DataStoreUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    //System.err.println(name + " self test succeeds");
    System.exit(0); 
  }

}



 /** 
 **/ 
 /** 
 * 
 
 *  TestReadResultMailbox
 
 * 
 
 **/ 
 /** 
 **/ 
  class TestReadResultMailbox{
    ImmutableBytes result;
    public 
    TestReadResultMailbox(){
      result = null;
    }

    public synchronized void
    verifyNoResultYet(){
      assert(result == null);
    }

    public synchronized void
    waitForAndConfirmReadResult(int len, byte value){
      byte b[];
      int ii;
      while(result == null){
        try{
	  if(DataStoreUnit.vverbose){
	    System.out.println("TestReadResultMailbox:wait");
	  }
          wait();
	   if(DataStoreUnit.vverbose){
	    System.out.println("TestReadResultMailbox:wait return");
	  }
        }
        catch(InterruptedException e){
          // continue to wait
        }
      }
      assert(result.getLength() == len);
      b = result.dangerousGetReferenceToInternalByteArray();
      for(ii = 0; ii < len; ii++){
        assert(b[ii] == value);
      }
    }

    public synchronized void
    applyResult(ImmutableBytes ib){
      assert(ib instanceof Immutable);
      result = ib;
      notifyAll();
      
      //System.out.println("TestReadResultMailbox:notifyAll called:" + ib);
      
    }

  }


 /** 
 **/ 
 /** 
 * 
 
 *  TestReadWorker -- fork off a thread that blocks on invalid or imprecise read
 
 *   When it finally succeeds, it puts the result in a TestReadResultMailbox
 
 * 
 
 **/ 
 /** 
 **/ 
  class TestReadWorker extends Thread{

    private ObjInvalTarget oit;
    private TestReadResultMailbox mb;
    private DataStore ds;
    private boolean dbg = false;
    public 
    TestReadWorker(ObjInvalTarget oit, TestReadResultMailbox mb, DataStore ds){
      this.ds = ds;
      this.mb = mb;
      this.oit = oit;
    }
  
    public void run(){
      try{
	if(dbg){
	  System.out.println(" TestReadWorker read: " + oit);
	}
        BodyMsg ret = ds.read(oit.getObjId(), oit.getOffset(), 
                              oit.getLength(), true, true, true, -1);
	if(dbg){
	  System.out.println(" TestReadWorker read return: " + oit + " =" +  ret);
        }
	mb.applyResult(ret.getBody());
      }
      catch(Exception e){
        e.printStackTrace();
        System.exit(-1);
        assert(false);
      }
    }

  }

 /** 
 **/ 
 /** 
 * 
 
 *  TestTEReadWorker -- fork off a thread that blocks on temporal error 
 
 * violationor invalid or imprecise or embargoed write
 
 * 
 
 *   When it finally succeeds, it puts the result in a TestReadResultMailbox
 
 * 
 
 **/ 
 /** 
 **/ 
  class TestTEReadWorker extends Thread{

    private ObjInvalTarget oit;
    private TestReadResultMailbox mb;
    private DataStore ds;
    private long temporalError;

    public 
    TestTEReadWorker(ObjInvalTarget oit, TestReadResultMailbox mb, DataStore ds, 
                     long temporalError){
      this.ds = ds;
      this.mb = mb;
      this.oit = oit;
      this.temporalError = temporalError;
    }
  
    public void run(){
      try{
        BodyMsg ret = ds.read(oit.getObjId(), oit.getOffset(), 
                              oit.getLength(), true, true, true, 
                              temporalError, -1);
	
	assert ret.getBody() != null;
        mb.applyResult(ret.getBody());
      }
      catch(Exception e){
        e.printStackTrace();
        System.exit(-1);
        assert(false);
      }
    }

  }